/**************************************************************************************
* @file         : OEMmenu.cpp
* @author       : Sriganesh
* @addtogroup   : AppHmi_System
* @brief        :
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/
#include "gui_std_if.h"
#include <string>
#include "OEMmenu.h"
#include "App/Core/AppLogic/CabinA/CabinASettingListhandler/CabinASettingListhandler.h"
#include "App/Core/Utils/XMLParser/XMLParser.h"
#include "App/Core/ClientHandler/SPM/SpmSrvClient.h"
#include <cstdlib>
#include "hmi_trace_if.h"
#include <vector>
#include <algorithm>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_SYSTEM_HALL
#define ETG_I_TRACE_CHANNEL TR_TTFIS_APPHMI_SYSTEM
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_SYSTEM_"
#define ETG_I_FILE_PREFIX                 App::Core::OEMmenu::
#include "trcGenProj/Header/OEMmenu.cpp.trc.h"
#define BACK_SPACE "0x232B"
#define MAX_VALID_PIN_TRIALS 3
#endif
#define Asterisk "*"
#define DEF_RESET_SUPPLIER_MODE 1
namespace App {
namespace Core {

OEMmenu* OEMmenu::m_poOEMmenu = NULL;

/************************************************************************
*FUNCTION: 		 OEMmenu - Constructor
*DESCRIPTION: 	 default constructor of the System setting Listhadler will be created
*PARAMETER:		 None
*RETURNVALUE: 	 None
************************************************************************/

OEMmenu::OEMmenu()
{
   ETG_TRACE_COMP(("OEMmenu:: Constructor is called"));
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SYSTEM_SO_OEM_MENU, this);
   listcreated = false;
   SystemInstallationActiveStatus = false;
   systemInstallationStatus = 1;
   _pintrials = 0;
   _userAuthentication.setUserPinSaved("2808");
   DisclaimerText = " ";
}


/************************************************************************
*FUNCTION: 		 OEMmenu - Destructor
*DESCRIPTION: 	 Allocated resources such as memory will be released here
*PARAMETER:		 None
*RETURNVALUE: 	 None
************************************************************************/

OEMmenu::~OEMmenu()
{
   ETG_TRACE_COMP(("OEMmenu:: Destructor is called"));
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SYSTEM_SO_OEM_MENU);
   listcreated = false;
}


/**********************************************************************************************
*FUNCTION: 		 OEMmenu - getListDataProvider
*DESCRIPTION: 	 Creates a list data provider which would contain unique list id
*PARAMETER:		 oMsg - Courier message received
*RETURNVALUE: 	 tSharedPtrDataProvider()
**********************************************************************************************/
tSharedPtrDataProvider OEMmenu::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_COMP(("OEMmenu:: getListDataProvider: List Id = %d", oMsg.GetListId()));
   switch (oMsg.GetListId())
   {
      case LIST_ID_SYSTEM_SO_OEM_MENU:
         return getMainViewListDataProvider();
   }
   return tSharedPtrDataProvider();
}


tSharedPtrDataProvider OEMmenu::getMainViewListDataProvider()
{
   ListDataProviderBuilder listBuilder(LIST_ID_SYSTEM_SO_OEM_MENU);
   listBuilder.AddItem(RESET_TO_FACTORY, 0UL, "Reset_to_Factory_Settings");
   listBuilder.AddItem(SYSTEM_INSTALLATION, 0UL, "System_Installation");
   listBuilder.AddItem(MONITOR_FLIP_CALIBRATION, 0UL, "MonitorFlipCalibration");
   listcreated = true;
   return listBuilder.CreateDataProvider();
}


bool OEMmenu::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   ListDataProviderBuilder modifiedListItems(LIST_ID_SYSTEM_SO_OEM_MENU);
   bool msgProcessed = false;
   unsigned int rowIdx = (unsigned int)oMsg.GetHdl(); //Check For the Row Clicked
   if (oMsg.GetListId() == LIST_ID_SYSTEM_SO_OEM_MENU)
   {
      if (oMsg.GetReaction() == enRelease)
      {
         ETG_TRACE_USR2(("OEMmenu::ButtonListItemUpdMsg:LIST_ID_SYSTEM_SO_OEM_MENU:reaction = %d, ROW VALUE =%d\n", oMsg.GetReaction(), oMsg.GetHdl()));
         switch (rowIdx)
         {
            case RESET_TO_FACTORY:
            {
               if (MIDWDiagnosis::poGetInstance())
               {
                  MIDWDiagnosis::poGetInstance()->SetResetToSupplierDefaultMode(DEF_RESET_SUPPLIER_MODE);
               }

               msgProcessed = true;
            }
            break;
            case SYSTEM_INSTALLATION:
            {
               (*_PopupButton).mButtonVisibility = false;
               (*_SystemInstallation).mWaitAnimationVisibility = true;
               (*_SystemInstallation).mTextbox1Visibility = true;
               (*_SystemInstallation).mTextbox2Visibility = true;
               (*_SystemInstallation).mTextbox3Visibility = true;
               (*_SystemInstallation).mTextbox4Visibility = false;
               (*_SystemInstallation).mTextbox5Visibility = false;
               _PopupButton.MarkAllItemsModified();
               _PopupButton.SendUpdate(true);
               _SystemInstallation.MarkAllItemsModified();
               _SystemInstallation.SendUpdate(true);
               POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_System#SoftwareDownload#Pfo_SystemInstallation"))));
               msgProcessed = true;
            }
            break;
            default:
            {
               // Do Nothing
            }
            break;
         }
      }
   }
   return msgProcessed;
}


bool OEMmenu::onCourierMessage(const PopupStatusMsg& oMsg)
{
   bool ret = false;
   uint32  ViewId = oMsg.GetActiveViewId();
   uint8 PopupState = oMsg.GetPopupStatus();
   ETG_TRACE_USR1(("OEMmenu::PopupStatusMsg: ViewId = %d, PopupState = %d", ViewId, PopupState));
   if ((ViewId == AppHmi_System_SoftwareDownload_Pfo_SystemInstallation) && (PopupState == true))
   {
      SystemInstallationActiveStatus = true;
      systemInstallationStatus = system("/opt/bosch/hmibase/hmi_doipcmcinstall.sh");
      ETG_TRACE_USR2(("OEMmenu::PopupStatusMsg: systemInstallationStatus = %d", systemInstallationStatus));
      if (!systemInstallationStatus)
      {
         (*_SystemInstallation).mWaitAnimationVisibility = false;
         (*_SystemInstallation).mTextbox1Visibility = false;
         (*_SystemInstallation).mTextbox2Visibility = false;
         (*_SystemInstallation).mTextbox3Visibility = false;
         (*_SystemInstallation).mTextbox4Visibility = true;
         (*_SystemInstallation).mTextbox5Visibility = false;
         _SystemInstallation.MarkAllItemsModified();
         _SystemInstallation.SendUpdate(true);
         POST_MSG((COURIER_MESSAGE_NEW(UpdatePopupTextReqMsg)(POPUP_SYS_INSTALLATION_SUCCESS)));
      }
      else
      {
         bool xmlstatus = XMLDocParser::GetInstance()->FetchSystemInstallResult();
         std::string _Status_Str = "";
         if (xmlstatus)
         {
            _Status_Str = XMLDocParser::GetInstance()->GetInstallationResultStatus();
         }
         else
         {
            _Status_Str = "XML file not found for Error log";
         }
         ETG_TRACE_USR3(("OEMmenu::PopupStatusMsg: xmlstatus = %d, _Status_Str = %s", xmlstatus, _Status_Str.c_str()));
         Candera::String Status_String = Candera::String(_Status_Str.c_str());
         (*_SystemInstallation).mWaitAnimationVisibility = false;
         (*_SystemInstallation).mTextbox1Visibility = false;
         (*_SystemInstallation).mTextbox2Visibility = false;
         (*_SystemInstallation).mTextbox3Visibility = false;
         (*_SystemInstallation).mTextbox4Visibility = true;
         (*_SystemInstallation).mTextbox5Visibility = true;
         (*_SystemInstallation).mFailureMessage = Status_String;
         POST_MSG((COURIER_MESSAGE_NEW(UpdatePopupTextReqMsg)(POPUP_SYS_INSTALLATION_FAILURE)));
         _SystemInstallation.MarkAllItemsModified();
         _SystemInstallation.SendUpdate(true);
      }
      ret = true;
   }
   else if (ViewId == AppHmi_System_SysSettingsScenes_SO_OEM_PIN)
   {
      if (PopupState == true)
      {
         if (_pintrials < MAX_VALID_PIN_TRIALS)
         {
            performPINspellerEnable(true);
         }
         ret = true;
      }
      else
      {
         performPINspellerEnable(false);
      }
      _userAuthentication.setPinInputIndex(0);
      _userAuthentication.resetUserPin();
   }
   else
   {
      /* do nothing */
   }
   return ret;
}


bool OEMmenu::getSystemInstallationActiveStatus()
{
   ETG_TRACE_COMP(("OEMmenu::getSystemInstallationActiveStatus: SystemInstallationActiveStatus= %d", SystemInstallationActiveStatus));
   return SystemInstallationActiveStatus;
}


bool OEMmenu::onCourierMessage(const SystemInstallationMsg&)
{
   SystemInstallationActiveStatus = false;
   ETG_TRACE_COMP(("OEMmenu::SystemInstallationMsg: systemInstallationStatus SystemInstallationActiveStatus= %d %d", systemInstallationStatus, SystemInstallationActiveStatus));
   if (!systemInstallationStatus)
   {
      SpmSrvClient::GetInstance()->TriggerIMXReboot();
   }
   else
   {
      /* do nothing */
   }
   return true;
}


bool OEMmenu::onCourierBindingItemChange_OEMPINKeyInfoItem(const Courier::Request& request)
{
   _oemPINKeyDataBinding.SetValue(request.ItemKey(), request.GetItemValue());
   string spellerKeyPressed = (*_oemPINKeyDataBinding).mPressedKey.GetCString();
   ETG_TRACE_USR1(("OEMmenu:onCourierBindingItemChange_OEMPINKeyInfoItem spellerKeyPressed is: %s", spellerKeyPressed.c_str()));
   if (_userAuthentication.isKeyDigitAllowed(spellerKeyPressed))
   {
      if (spellerKeyPressed == BACK_SPACE)
      {
         _userAuthentication.decrementPinInputIndex();
      }
      resetSpellerEditableAndCursor();
      setSpellerEditableAndCursor(_userAuthentication.getPinInputIndex(), spellerKeyPressed);
      setSpellerText(_userAuthentication.getPinInputIndex(), spellerKeyPressed);
      ETG_TRACE_USR2(("OEMmenu:onCourierBindingItemChange_OEMPINKeyInfoItem: isKeyDigitAllowed: keyIndex %d", _userAuthentication.getPinInputIndex()));
      if (spellerKeyPressed != BACK_SPACE)
      {
         _userAuthentication.incrementPinInputIndex();
      }
      DATA_BINDING_ALL_ITEM_UPDATE(_oemPINKeyDataBinding, rs)
   }
   return true;
}


bool OEMmenu::onCourierBindingItemChange_OEMPIN1TextInfoItem(const Courier::Request& request)
{
   _oemPIN1TextDataBinding.SetValue(request.ItemKey(), request.GetItemValue());
   ETG_TRACE_USR1(("OEMmenu::onCourierBindingItemChange_OEMPIN1TextInfoItem: PIN1TextInfo %s ", (*_oemPIN1TextDataBinding).mText.GetCString()));
   DATA_BINDING_ALL_ITEM_UPDATE(_oemPIN1TextDataBinding, rs)
   return true;
}


bool OEMmenu::onCourierBindingItemChange_OEMPIN2TextInfoItem(const Courier::Request& request)
{
   _oemPIN2TextDataBinding.SetValue(request.ItemKey(), request.GetItemValue());
   ETG_TRACE_USR1(("OEMmenu::onCourierBindingItemChange_OEMPIN2TextInfoItem: ProfilePIN2TextInfo %s ", (*_oemPIN2TextDataBinding).mText.GetCString()));
   DATA_BINDING_ALL_ITEM_UPDATE(_oemPIN2TextDataBinding, rs)
   return true;
}


bool OEMmenu::onCourierBindingItemChange_OEMPIN3TextInfoItem(const Courier::Request& request)
{
   _oemPIN3TextDataBinding.SetValue(request.ItemKey(), request.GetItemValue());
   ETG_TRACE_USR1(("OEMmenu: :onCourierBindingItemChange_OEMPIN3TextInfoItem: ProfilePIN3TextInfo %s ", (*_oemPIN3TextDataBinding).mText.GetCString()));
   DATA_BINDING_ALL_ITEM_UPDATE(_oemPIN3TextDataBinding, rs)
   return true;
}


bool OEMmenu::onCourierBindingItemChange_OEMPIN4TextInfoItem(const Courier::Request& request)
{
   _oemPIN4TextDataBinding.SetValue(request.ItemKey(), request.GetItemValue());
   if (_userAuthentication.isNotBackSpace((*_oemPIN4TextDataBinding).mText.GetCString()))
   {
      ETG_TRACE_USR1(("OEMmenu:: onCourierBindingItemChange_OEMPIN4TextInfoItem: ProfilePIN4TextInfo %s ", (*_oemPIN4TextDataBinding).mText.GetCString()));
      DATA_BINDING_ALL_ITEM_UPDATE(_oemPIN4TextDataBinding, rs)
   }
   return true;
}


bool OEMmenu::onCourierMessage(const OEMPinOkMsg&)
{
   ETG_TRACE_COMP(("OEMmenu:OEMPinOkMsg: _pintrials = %d", _pintrials));
   performAuthenticationCheckScreenValidation();
   performPINspellerEnable(true);
   if (_pintrials <= MAX_VALID_PIN_TRIALS)
   {
      ETG_TRACE_USR1(("OEMmenu:OEMPinOkMsg: getUserPin = %s", _userAuthentication.getUserPin().c_str()));
      if (strncmp((_userAuthentication.getUserPin()).c_str(), (_userAuthentication.getUserPinSaved()).c_str(), 4) == 0)
      {
         ETG_TRACE_USR2(("OEMmenu:OEMPinOkMsg PIN Validated"));
         POST_MSG(COURIER_MESSAGE_NEW(OEMMenuMsg)());
         _pintrials = 0;
         DisclaimerText = " ";
      }
      else
      {
         if (_pintrials == 0)
         {
            DisclaimerText = "Wrong PIN. Tries remaining:2";
         }
         else if (_pintrials == 1)
         {
            DisclaimerText = "Wrong PIN. Tries remaining:1";
         }
         else if (_pintrials == 2)
         {
            DisclaimerText = "Service Menu is locked. Please restart the system";
            performPINspellerEnable(false);//updates data binding to disable the keypad
         }
         else
         {
            //do nothing
         }
         _pintrials++;
         ETG_TRACE_USR2(("OEMmenu:OEMPinOkMsg: PIN incorrect increment _pintrials - %d", _pintrials));
      }
      (*_oemAuthenticationText).mText = Candera::String(DisclaimerText.c_str());
      DATA_BINDING_ALL_ITEM_UPDATE(_oemAuthenticationText, rs)
   }
   _userAuthentication.setPinInputIndex(0);
   _userAuthentication.resetUserPin();
   return true;
}


void OEMmenu::performAuthenticationCheckScreenValidation()
{
   ETG_TRACE_COMP(("OEMmenu::performAuthenticationCheckScreenValidation"));
   if (_userAuthentication.getUserPin() == "" && _userAuthentication.hasUserPin())
   {
      _userAuthentication.setUserPin();
   }
}


void OEMmenu::resetSpellerEditableAndCursor()
{
   ETG_TRACE_COMP(("OEMmenu::resetSpellerEditableAndCursor"));
   (*_oemPINEditableDataBinding).mEditableState1 = false;
   (*_oemPINEditableDataBinding).mEditableState2 = false;
   (*_oemPINEditableDataBinding).mEditableState3 = false;
   (*_oemPINEditableDataBinding).mEditableState4 = false;
   (*_oemPINCursorDataBinding).mState1 = false;
   (*_oemPINCursorDataBinding).mState2 = false;
   (*_oemPINCursorDataBinding).mState3 = false;
   (*_oemPINCursorDataBinding).mState4 = false;
   DATA_BINDING_ALL_ITEM_UPDATE(_oemPINCursorDataBinding, rs)
   DATA_BINDING_ALL_ITEM_UPDATE(_oemPINEditableDataBinding, rs1)
}


void OEMmenu::setDefaultPINCursorAndEditableDatabinding()
{
   ETG_TRACE_COMP(("OEMmenu::setDefaultPINCursorAndEditableDatabinding"));
   (*_oemPINEditableDataBinding).mEditableState1 = true;
   (*_oemPINEditableDataBinding).mEditableState2 = true;
   (*_oemPINEditableDataBinding).mEditableState3 = true;
   (*_oemPINEditableDataBinding).mEditableState4 = true;
   (*_oemPINCursorDataBinding).mState1 = true;
   DATA_BINDING_ALL_ITEM_UPDATE(_oemPINCursorDataBinding, rs)
   DATA_BINDING_ALL_ITEM_UPDATE(_oemPINEditableDataBinding, rs1)
   (*_oemPINkey1Databinding).mText = BACK_SPACE;
   (*_oemPINkey2Databinding).mText = BACK_SPACE;
   (*_oemPINkey3Databinding).mText = BACK_SPACE;
   (*_oemPINkey4Databinding).mText = BACK_SPACE;
   DATA_BINDING_ALL_ITEM_UPDATE(_oemPINkey1Databinding, rs2)
   DATA_BINDING_ALL_ITEM_UPDATE(_oemPINkey2Databinding, rs3)
   DATA_BINDING_ALL_ITEM_UPDATE(_oemPINkey3Databinding, rs4)
   DATA_BINDING_ALL_ITEM_UPDATE(_oemPINkey4Databinding, rs5)
}


void OEMmenu::setSpellerEditableAndCursor(uint8 keyIndex, string key)
{
   ETG_TRACE_COMP(("OEMmenu::setSpellerEditableAndCursor: keyIndex = %d, key = %s", keyIndex, key.c_str()));
   switch (keyIndex)
   {
      case PIN1:
      {
         (*_oemPINEditableDataBinding).mEditableState1 = true;
         if (key == BACK_SPACE)
         {
            (*_oemPINCursorDataBinding).mState1 = true;
         }
         else
         {
            (*_oemPINCursorDataBinding).mState2 = true;
         }
      }
      break;
      case PIN2:
      {
         (*_oemPINEditableDataBinding).mEditableState2 = true;
         if (key == BACK_SPACE)
         {
            (*_oemPINCursorDataBinding).mState2 = true;
         }
         else
         {
            (*_oemPINCursorDataBinding).mState3 = true;
         }
      }
      break;
      case PIN3:
      {
         (*_oemPINEditableDataBinding).mEditableState3 = true;
         if (key == BACK_SPACE)
         {
            (*_oemPINCursorDataBinding).mState3 = true;
         }
         else
         {
            (*_oemPINCursorDataBinding).mState4 = true;
         }
      }
      break;
      case PIN4:
      {
         (*_oemPINEditableDataBinding).mEditableState4 = true;
         if (key == BACK_SPACE)
         {
            (*_oemPINCursorDataBinding).mState4 = true;
         }
      }
      break;
      default:
      {} break;
   }
   DATA_BINDING_ALL_ITEM_UPDATE(_oemPINCursorDataBinding, r1)
   DATA_BINDING_ALL_ITEM_UPDATE(_oemPINEditableDataBinding, rs2)
}


void OEMmenu::setSpellerText(uint8 keyIndex, string key)
{
   ETG_TRACE_COMP(("OEMmenu::setSpellerText: keyIndex = %d, key = %s", keyIndex, key.c_str()));
   switch (keyIndex)
   {
      case PIN1:
      {
         (*_oemPINkey1Databinding).mText = (key != BACK_SPACE) ? "*" : key.c_str();
         DATA_BINDING_ALL_ITEM_UPDATE(_oemPINkey1Databinding, rs)
      }
      break;
      case PIN2:
      {
         (*_oemPINkey2Databinding).mText = (key != BACK_SPACE) ? "*" : key.c_str();
         DATA_BINDING_ALL_ITEM_UPDATE(_oemPINkey2Databinding, rs)
      }
      break;
      case PIN3:
      {
         (*_oemPINkey3Databinding).mText = (key != BACK_SPACE) ? "*" : key.c_str();
         DATA_BINDING_ALL_ITEM_UPDATE(_oemPINkey3Databinding, rs)
      }
      break;
      case PIN4:
      {
         (*_oemPINkey4Databinding).mText = (key != BACK_SPACE) ? "*" : key.c_str();
         DATA_BINDING_ALL_ITEM_UPDATE(_oemPINkey4Databinding, rs)
      }
      break;
      default:
      {} break;
   }
   _userAuthentication.setKeyPin(keyIndex, key);
}


void OEMmenu::performPINspellerEnable(bool status)
{
   ETG_TRACE_COMP(("OEMmenu:performPINspellerEnable  %d", status));
   (*_oemAuthenticationText).mSpellerEnable = status;
   DATA_BINDING_ALL_ITEM_UPDATE(_oemAuthenticationText, rs)
   if (status == true)
   {
      resetSpellerEditableAndCursor();
      setDefaultPINCursorAndEditableDatabinding();
   }
   else
   {
      setDefaultPINCursorAndEditableDatabinding();
      resetSpellerEditableAndCursor();
   }
}


void OEMmenu::UpdateResetToSupplierDefaultConfigStatus(bool status)
{
   ETG_TRACE_COMP(("OEMmenu:UpdateResetToSupplierDefaultConfigStatus  %d", status));
   if (status)
   {
      POST_MSG((COURIER_MESSAGE_NEW(UpdatePopupTextReqMsg)(POPUP_SYS_RESETSUPPLIERMODE_SUCCESS)));
   }
   else
   {
      POST_MSG((COURIER_MESSAGE_NEW(UpdatePopupTextReqMsg)(POPUP_SYS_RESETSUPPLIERMODE_FAILURE)));
   }
   POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_System#SoftwareDownload#Pfo_SysSetSupplierReset"))));
}


}
}
