/**************************************************************************************
* @file         : ClockSettings.cpp
* @author       : Sushmita Dikshit
* @addtogroup   : AppHmi_System
* @brief        : Updated changes for displaying date format.
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#include "gui_std_if.h"
#include "AppHmi_SystemStateMachine.h"
#include "hmi_trace_if.h"
#endif

#include <string>
//#include "util/StrUtf8.h"
#include "App/Core/ClientHandler/ClockSrvClient/ClockSrvClient.h"
#include "App/Core/Utils/ClockUtility.h"

#include "ClockSettings.h"
#include <cstdlib>

#define DP_S_IMPORT_INTERFACE_KDS_FI
#include "dp_kds_if.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_SYSTEM_HALL
#define ETG_I_TRACE_CHANNEL TR_TTFIS_APPHMI_SYSTEM
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_SYSTEM_"
#define ETG_I_FILE_PREFIX                 App::Core::ClockSettings::
#include "trcGenProj/Header/ClockSettings.cpp.trc.h"
#endif

namespace App {
namespace Core {

ClockSettings* ClockSettings::m_poClockSettings = NULL;
//12hr and 24hrs
Candera::String TimeFormatStrings[2]  = {  Candera::String(TextId_0x06E5), Candera::String(TextId_0x06E4)};
//Manuel and GPS
Candera::String ClockFormatStrings[2]  = { Candera::String(TextId_0x178C), Candera::String(TextId_0x06FA)};


static const char* const DATA_CONTEXT_TEXT_BUTTON_LIST_ITEM = "TextButton_ListItem";
static const char* const DATA_CONTEXT_TEXT_BUTTON_FIRST_LIST_ITEM = "FirstElement_Checkbox";
Candera::String ClockSettingText[5] = {Candera::String(TextId_0x06CF), Candera::String(TextId_0x06F1), Candera::String(TextId_0x06EF), Candera::String(TextId_0x06EB), Candera::String(TextId_0x06E9)};
unsigned long DateFormatString[6] = {TextId_0x06B9, TextId_0x13A0, TextId_0x06B7, TextId_0x06BA, TextId_0x13A1, TextId_0x06BB};
static const char* const DATA_CONTEXT_COMBOBOX_LIST_ITEM = "Timeformat_ListItem";
static const char* const DATA_CONTEXT_CLOCK_COMBOBOX_LIST_ITEM = "ClockSettingMode_ListItem";
const char* LISTCLOCKITEMTEMPLATE[1] = {"listTplate_BtnHourBar"};

/************************************************************************
*FUNCTION: 		 ClockSettings - Constructor
*DESCRIPTION: 	 default constructor of the System setting Listhadler will be created
*PARAMETER:		 None
*RETURNVALUE: 	 None
*
*HISTORY:
*27.3.2018:Sheena Mathew

*0.1 Initial Revision.
************************************************************************/

ClockSettings::ClockSettings()
{
   ETG_TRACE_COMP(("ClockSettings:: Constructor is called"));
   clocksettingmode = 2; //GPSVALUE
   listUpdateNeeded = false;
   updatedval = 0;
   type = COLUMN_BUTTON_MINUS;
   m_ListClockHourandMinuteVal[1] = CLOCK_C_AM; //to store AM/PM value
   m_ListClockHourandMinuteVal[2] = CLOCK_MODE12; //To store time format value
   m_ListClockHourandMinuteVal[3] = updatedval; // to store the hour value
   m_ListClockHourandMinuteVal[4] = updatedval; // to store the minute value
   vec_clocksettings.clear();
   vCreateMapCultureindextoClockString();
   _currentClockformatOptionStr = ClockFormatStrings[0];
   _currentTimeformatOptionStr = "12H";
   m_clockmode = 1;
   _currentTimeformatOption = twelvehour;
   _currentClockformatOption = GPS;
   u8RequestedHourValue = 0;
   rowIdx = 1;
   listcreated = false;
   _dateformat = "";
   _availableTimeZoneList.clear();
   _timeZoneListItemInfoMap.clear();
   _selectedTimeZone = 0;
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SYSTEM_CLOCK_TIMEFORMAT_LIST, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SYSTEM_CLOCK_CLOCKFORMAT_LIST, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SYSTEM_CLOCK_LIST, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SETTINGS_SYSTEM_CLOCK_TIMEZONE, this);
}


/************************************************************************
*FUNCTION: 		 ClockSettings - Destructor
*DESCRIPTION: 	 Allocated resources such as memory will be released here
*PARAMETER:		 None
*RETURNVALUE: 	 None
*
*HISTORY:
*27.3.2018:Sheena Mathew

*0.1 Initial Revision.
************************************************************************/

ClockSettings::~ClockSettings()
{
   ETG_TRACE_COMP(("ClockSettings:: Destructor is called"));
   vec_clocksettings.~vector<tp_clocksettings>();
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SYSTEM_CLOCK_TIMEFORMAT_LIST);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SYSTEM_CLOCK_CLOCKFORMAT_LIST);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SYSTEM_CLOCK_LIST);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SETTINGS_SYSTEM_CLOCK_TIMEZONE);
   listcreated = false;
}


void ClockSettings::vCreateMapCultureindextoClockString()
{
   ETG_TRACE_COMP(("ClockSettings::vCreateMapCultureindextoClockString"));
   vec_clocksettings.push_back(boost::make_tuple("GPS", 1, 1, 0));
   vec_clocksettings.push_back(boost::make_tuple("Manual", 2, 0, 0));
}


/**********************************************************************************************
*FUNCTION: 		 ClockSettings - getListDataProvider
*DESCRIPTION: 	 Creates a list data provider which would contain unique list id
*PARAMETER:		 oMsg - Courier message received
*RETURNVALUE: 	 tSharedPtrDataProvider()
*
*HISTORY:
*27.3.2018:Sheena Mathew

*0.1 Initial Revision.
**********************************************************************************************/

tSharedPtrDataProvider ClockSettings::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_COMP(("ClockSettings:: getListDataProvider: List Id = %d", oMsg.GetListId()));
   tSharedPtrDataProvider dataProvider = tSharedPtrDataProvider();
   switch (oMsg.GetListId())
   {
      case LIST_ID_SYSTEM_CLOCK_LIST:
         return getDropdownListDataProvider();

      case LIST_ID_SYSTEM_CLOCK_TIMEFORMAT_LIST:
         return getDropdownTimeFormatListDataProvider();

      case LIST_ID_SYSTEM_CLOCK_CLOCKFORMAT_LIST:
         return getDropdownClockSettingModeListDataProvider();
         break;
   }
   return tSharedPtrDataProvider();
}


tSharedPtrDataProvider ClockSettings::getTimeZoneClockListDataProvider()
{
   uint32 listSize = _timeZoneListItemInfoMap.size();
   ETG_TRACE_COMP(("ClockSettings:: getTimeZoneClockListDataProvider: list size =%d", listSize));
   tSharedPtrDataProvider dataProvider = tSharedPtrDataProvider();
   if (!(_timeZoneListItemInfoMap.empty()))
   {
      ListDataProviderBuilder listBuilder(LIST_ID_SETTINGS_SYSTEM_CLOCK_TIMEZONE);
      for (std::map< uint16, TimeZoneListItemInfo >::const_iterator iter = _timeZoneListItemInfoMap.begin(); iter != _timeZoneListItemInfoMap.end(); iter++)
      {
         if (iter->first == 0)
         {
            listBuilder.AddItem(iter->first, 0UL, DATA_CONTEXT_TEXT_BUTTON_FIRST_LIST_ITEM).AddData(iter->second.TimeZoneText).AddData(iter->second.enable);
            ETG_TRACE_USR3(("ClockSettings:: getTimeZoneClockListDataProvider: iter->first == 0, list items =%s", iter->second.TimeZoneText));
         }
         else if (iter->first > 0)
         {
            listBuilder.AddItem(iter->first, 0UL, DATA_CONTEXT_TEXT_BUTTON_LIST_ITEM).AddData(iter->second.TimeZoneText).AddData(iter->second.enable);
            ETG_TRACE_USR3(("ClockSettings:: getTimeZoneClockListDataProvider: iter->first == 1, list items =%s", iter->second.TimeZoneText));
         }
      }
      dataProvider = listBuilder.CreateDataProvider();
   }
   else
   {
   }
   return dataProvider;
}


tSharedPtrDataProvider ClockSettings::getDropdownListDataProvider()
{
   ListDataProviderBuilder listBuilder(LIST_ID_SYSTEM_CLOCK_LIST);
   listBuilder.AddItem(
      DateFormat,
      0UL,   //  - column
      DATA_CONTEXT_TEXT_BUTTON_LIST_ITEM)  //  - overwrites the default data context
   .AddData(ClockSettingText[0])
   .AddData(_dateformat);//1 - Text of the button, should be obtained from Candera.

   listBuilder.AddExpandableItem(static_cast<int>(TimeFormat),            //   - identifies the row
                                 LIST_ID_SYSTEM_CLOCK_TIMEFORMAT_LIST,                          //   - used to automatically toggle the visibility of the combobox dropdown list when the combox button is clicked
                                 DATA_CONTEXT_COMBOBOX_LIST_ITEM)         //   - overwrites the default data context
   .AddData(ClockSettingText[1])  //1 - Text of the combobox button.
   .AddData(LIST_ID_SYSTEM_CLOCK_TIMEFORMAT_LIST)				 //2 - ListId of the combobox dropdown list.
   .AddData(_currentTimeformatOptionStr);	//3 - Text of the label displayed in front of the combobox button, should be obtained

   listBuilder.AddExpandableItem(static_cast<int>(ClockSettingMode),            //   - identifies the row
                                 LIST_ID_SYSTEM_CLOCK_CLOCKFORMAT_LIST,                          //   - used to automatically toggle the visibility of the combobox dropdown list when the combox button is clicked
                                 DATA_CONTEXT_CLOCK_COMBOBOX_LIST_ITEM)         //   - overwrites the default data context
   .AddData(ClockSettingText[2])  //1 - Text of the combobox button.
   .AddData(LIST_ID_SYSTEM_CLOCK_CLOCKFORMAT_LIST)				 //2 - ListId of the combobox dropdown list.
   .AddData(_currentClockformatOptionStr);	//3 - Text of the label displayed in front of the combobox button, should be obtained

   for (int index = 3; index < 5; ++index)
   {
      updatedval = m_ListClockHourandMinuteVal.find(index)->second;
      onUpdateMenu(index, &updatedval);

      listBuilder.AddItem(index, 0u, LISTCLOCKITEMTEMPLATE[0])
      .AddData(ClockSettingText[index])
      .AddId(index, COLUMN_BUTTON_MINUS)     // -
      .AddId(index, COLUMN_BUTTON_PLUS)// +
      .AddDataBindingUpdater<HourValueUpdateItemDataBindingSource>(_HourValueUpdateItem);
   }
   listcreated = true;
   return listBuilder.CreateDataProvider();
}


//Drop Down List for 12Hrs and Manual Mode
tSharedPtrDataProvider ClockSettings::getDropdownClockSettingModeListDataProvider()
{
   const char* DATA_CONTEXT_DROPDOWN_LIST_ITEM_MODE = "ClockSettingDropdownListButton";
   ListDataProviderBuilder listBuilder(LIST_ID_SYSTEM_CLOCK_CLOCKFORMAT_LIST, DATA_CONTEXT_DROPDOWN_LIST_ITEM_MODE);
   listBuilder.AddItem(1UL).AddData(ClockFormatStrings[0]);
   listBuilder.AddItem(2UL).AddData(ClockFormatStrings[1]);
   return listBuilder.CreateDataProvider();
}


//Drop Down List for Gps and Manual Mode
tSharedPtrDataProvider ClockSettings::getDropdownTimeFormatListDataProvider()
{
   const char* DATA_CONTEXT_DROPDOWN_LIST_ITEM = "TimeformatDropdownListButton";
   ListDataProviderBuilder listBuilder(LIST_ID_SYSTEM_CLOCK_TIMEFORMAT_LIST, DATA_CONTEXT_DROPDOWN_LIST_ITEM);
   //enMenu currentValue =  (*_DropdownCombo1InfoData).mItem;
   listBuilder.AddItem(1UL).AddData(TimeFormatStrings[0]);
   listBuilder.AddItem(2UL).AddData(TimeFormatStrings[1]);
   listcreated = true;
   return listBuilder.CreateDataProvider();
}


//On Courier Message for GPS AND Manual
bool ClockSettings::onCourierMessage(const ChangeClockformatReqMsg& oMsg)
{
   int gpsEnableStatus, bpositionstatus;
   int selectedOption = oMsg.GetClockFormatOptions();
   ETG_TRACE_USR1(("ClockSettings::ChangeClockformatReqMsg: selectedOption=%d", selectedOption));
   ListDataProviderBuilder modifiedListItems(LIST_ID_SYSTEM_CLOCK_LIST);
   for (int itemIdx = 0 ; itemIdx < static_cast<int>(vec_clocksettings.size())  ; ++itemIdx)
   {
      if (boost::get<ClockMode>(vec_clocksettings[itemIdx]) == selectedOption)
      {
         gpsEnableStatus = boost::get<EnableStatus>(vec_clocksettings[itemIdx]);
         bpositionstatus = boost::get<PositionStatus>(vec_clocksettings[itemIdx]);
         ETG_TRACE_USR3(("ClockSettings::ChangeClockformatReqMsg boost::get<ClockMode>(vec_clocksettings[itemIdx]=%d, gpsEnableStatus=%d,bpositionstatus=%d", boost::get<ClockMode>(vec_clocksettings[itemIdx]), gpsEnableStatus, bpositionstatus));
      }
   }
   if (selectedOption != clocksettingmode)
   {
      POST_MSG((COURIER_MESSAGE_NEW(Displayclock)()));
   }
   //AUto GPS mode is required only for Navi variant
   if (isNaviVariant() == true)
   {
      if (selectedOption == 1)
      {
         bpositionstatus = true;
         //   POST_MSG((COURIER_MESSAGE_NEW(Displayclock)()));
      }
      else
      {
         bpositionstatus = false;
      }
   }
   else //Non-navi applicable for SMART and INF4CV turkey region
   {
      bpositionstatus =  false;
   }
   ETG_TRACE_USR1(("ClockSettings::ChangeClockformatReqMsg: gpsEnableStatus=%d,bpositionstatus=%d", gpsEnableStatus, bpositionstatus));
   ClockSrvClient::poGetInstance()->covSetAutoSyncPropertyToService(gpsEnableStatus, bpositionstatus);
   return true;
}


//On Courier Message for 12hr and 24hrs -ClockTimeFormat
bool ClockSettings::onCourierMessage(const ChangeClockTimeformatReqMsg& oMsg)
{
   unsigned int selectedTimeFormatOption = oMsg.GetTimeFormatOptions();
   ETG_TRACE_USR1(("SystemSettingsListHandler::ChangeClockTimeformatReqMsg selectedTimeFormatOption=%d", selectedTimeFormatOption));
   if (selectedTimeFormatOption <= 2)
   {
      // Subtract 1 to match the selection and enumeration index,
      // handle for first option in dropdown is 1. However, the enumeration value for same is 0
      _currentTimeformatOption = static_cast<enTimeformatOption>(selectedTimeFormatOption);
      _currentTimeformatOptionStr = TimeFormatStrings[selectedTimeFormatOption - 1U];
      m_ListClockHourandMinuteVal[2] = _currentTimeformatOption;
   }
   ETG_TRACE_USR1(("ClockSettings::ChangeClockformatReqMsg: _currentTimeformatOption=%d, _currentTimeformatOptionStr = %s", _currentTimeformatOption, _currentTimeformatOptionStr.GetCString()));
   ClockSrvClient::poGetInstance()->covsetTimeFormatProperty(_currentTimeformatOption);
   return true;
}


/*Clock Mode format update from Middleware to HMI ---- GPS OR Manual*/
void ClockSettings::bonsetsynmodetoPL(const int& gpsenablestatus, const int& positionstatus)
{
   ETG_TRACE_COMP(("ClockSettings::bonsetsynmodetoPL: gpsenablestatus = %d, positionstatus = %d", gpsenablestatus, positionstatus));
   if ((gpsenablestatus == 1) && (positionstatus == 1))
   {
      _currentClockformatOptionStr = ClockFormatStrings[0];
   }
   else
   {
      _currentClockformatOptionStr = ClockFormatStrings[1];
   }
   if (listcreated == true)
   {
      ListRegistry::s_getInstance().updateList(LIST_ID_SYSTEM_CLOCK_CLOCKFORMAT_LIST);
      ListRegistry::s_getInstance().updateList(LIST_ID_SYSTEM_CLOCK_LIST);
   }
}


/*Clock Time format update from Middleware to HMI --- 12 Hrs or 24 Hrs */
void ClockSettings::bonsetformatpropertyUpdateHMI(const int& timeFormat)
{
   ETG_TRACE_COMP(("bonsetformatpropertyUpdateHMI: TimeFormat:=%d", timeFormat));
   ClockUtility clockUtility;
   if (timeFormat == 2)
   {
      _currentTimeformatOptionStr = Candera::String(TextId_0x06E4);
      m_ListClockHourandMinuteVal[2] = static_cast<enTimeformatOption>(2);
      m_ListClockHourandMinuteVal[3] = clockUtility.convert12To24h(m_ListClockHourandMinuteVal[3], m_ListClockHourandMinuteVal[1]);
      ETG_TRACE_USR2(("ClockSettings::bonsetformatpropertyUpdateHMI inside 24hour modem_ListClockHourandMinuteVal[3] =%d", m_ListClockHourandMinuteVal[3]));
   }
   else
   {
      _currentTimeformatOptionStr = Candera::String(TextId_0x06E5);
      m_ListClockHourandMinuteVal[2] = static_cast<enTimeformatOption>(1);
      m_ListClockHourandMinuteVal[3] = clockUtility.convert24To12h(m_ListClockHourandMinuteVal[3]);
      ETG_TRACE_USR2(("ClockSettings::bonsetformatpropertyUpdateHMI inside 12hour modem_ListClockHourandMinuteVal[3] =%d", m_ListClockHourandMinuteVal[3]));
   }
   _currentTimeformatOption = static_cast<enTimeformatOption>(timeFormat);
   m_ListClockHourandMinuteVal[2] = _currentTimeformatOption;
   if (listcreated == true)
   {
      ListDataProviderBuilder modifiedListItems(LIST_ID_SYSTEM_CLOCK_LIST);
      ListRegistry::s_getInstance().updateList(LIST_ID_SYSTEM_CLOCK_TIMEFORMAT_LIST);
      ListRegistry::s_getInstance().updateList(LIST_ID_SYSTEM_CLOCK_LIST);
   }
}


void ClockSettings::callGps()
{
   ETG_TRACE_COMP(("ClockSettings::callGps() "));
   clockselectoption = 1;
}


void ClockSettings::callManual()
{
   ETG_TRACE_COMP(("ClockSettings::callManual() "));
   clockselectoption = 2;
}


void ClockSettings::onUpdateMenu(int index, int* updatedval)
{
   ETG_TRACE_COMP(("ClockSettings::onUpdateMenu: ROW VALUE =%d, updatedval = %d\n", index, *updatedval));
   ListDataProviderBuilder modifiedListItems(LIST_ID_SYSTEM_CLOCK_LIST);
   if (index >= 3 && index < 5)
   {
      *updatedval = m_ListClockHourandMinuteVal.find(index)->second;
      int val = *updatedval;
      stringstream bout;
      bout << *updatedval ;
      string m_ampmval;
      m_hourmin = bout.str();
      if (index == 3)
      {
         string m_hourvalupd;
         if (m_ListClockHourandMinuteVal[2] == CLOCK_MODE12)
         {
            if (m_ListClockHourandMinuteVal[1] == CLOCK_C_AM)
            {
               m_ampmval = " AM";
               m_hourvalupd = m_hourmin + m_ampmval;

               m_hourmin = "";
               ETG_TRACE_USR4(("ClockSettings::onUpdateMenu:12hrs am m_hourvalupd =%s\n", m_hourvalupd.c_str()));
            }
            else if (m_ListClockHourandMinuteVal[1] == CLOCK_C_PM)
            {
               m_ampmval = " PM";
               m_hourvalupd = m_hourmin + m_ampmval;

               m_hourmin = "";
               ETG_TRACE_USR4(("ClockSettings::onUpdateMenu:12hrs pm m_hourvalupd =%s\n", m_hourvalupd.c_str()));
            }
         }
         else if (m_ListClockHourandMinuteVal[2] == CLOCK_MODE24)
         {
            m_hourvalupd = m_hourmin;
            ETG_TRACE_USR4(("ClockSettings::onUpdateMenu:24 hours m_hourvalupd =%s\n", m_hourvalupd.c_str()));
         }

         (*_HourValueUpdateItem).mHourValueIndex = m_hourvalupd.c_str();
      }
      else if (index == 4)
      {
         ETG_TRACE_USR2(("ClockSettings::onUpdateMenu:Minutes m_hourmin =%s\n", m_hourmin.c_str()));
         (*_HourValueUpdateItem).mHourValueIndex = m_hourmin.c_str();
      }
      _HourValueUpdateItem.MarkAllItemsModified();
      _HourValueUpdateItem.SendUpdate(true);
      //	ListRegistry::s_getInstance().updateList(LIST_ID_SYSTEM_CLOCK_LIST);
      if (listcreated == true)
      {
         modifiedListItems.AddItem(index, 0u, LISTCLOCKITEMTEMPLATE[0], 0, index)
         .AddData(ClockSettingText[index])
         .AddId(index, COLUMN_BUTTON_MINUS)     // -
         .AddId(index, COLUMN_BUTTON_PLUS)// +
         .AddDataBindingUpdater<HourValueUpdateItemDataBindingSource>(_HourValueUpdateItem);

         // Create a ListDataProviderUpdater using ListDataProviderBuilder and post it with a ListDataProviderUpdMsg.
         tSharedPtrDataProviderUpdater dataProviderUpdater = modifiedListItems.CreateDataProviderUpdater();
         POST_MSG(COURIER_MESSAGE_NEW(ListDataProviderUpdMsg)(dataProviderUpdater));
      }
   }
}


bool ClockSettings::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   ListDataProviderBuilder modifiedListItems(LIST_ID_SYSTEM_CLOCK_LIST);
   bool msgProcessed = false;
   if (oMsg.GetListId() == LIST_ID_SYSTEM_CLOCK_LIST)
   {
      UpdateManualClkDropDownMsg();
      rowIdx = (unsigned int)oMsg.GetHdl();
      if (oMsg.GetReaction() == enRelease)
      {
         switch (oMsg.GetHdl())
         {
            case Hour:
            {
               if (oMsg.GetSubHdl() == COLUMN_BUTTON_MINUS)
               {
                  type = COLUMN_BUTTON_MINUS;
                  vClockSpecHourincrementdecrementvalue(type, m_ListClockHourandMinuteVal[3]);
               }
               else if (oMsg.GetSubHdl() == COLUMN_BUTTON_PLUS)
               {
                  type = COLUMN_BUTTON_PLUS;
                  vClockSpecHourincrementdecrementvalue(type, m_ListClockHourandMinuteVal[3]);
               }
               ETG_TRACE_USR3(("ClockSettings::ButtonListItemUpdMsg:LIST_ID_SYSTEM_CLOCK_LIST: row value = %d, column value = %d, m_ListClockHourandMinuteVal[3]=%d", oMsg.GetHdl(), oMsg.GetSubHdl(), m_ListClockHourandMinuteVal[3]));
               listUpdateNeeded = true;
            }
            break;
            case Minute:
            {
               if (oMsg.GetSubHdl() == COLUMN_BUTTON_MINUS)
               {
                  type = COLUMN_BUTTON_MINUS;
                  vClockSpecMinuteincrementdecrementvalue(type, m_ListClockHourandMinuteVal[4]);
               }
               else if (oMsg.GetSubHdl() == COLUMN_BUTTON_PLUS)
               {
                  type = COLUMN_BUTTON_PLUS;
                  vClockSpecMinuteincrementdecrementvalue(type, m_ListClockHourandMinuteVal[4]);
               }
               ETG_TRACE_USR3(("ClockSettings::ButtonListItemUpdMsg:LIST_ID_SYSTEM_CLOCK_LIST: row value = %d, column value = %d, m_ListtypetoAudSpecVolUpd[4]=%d", oMsg.GetHdl(), oMsg.GetSubHdl(), m_ListClockHourandMinuteVal[4]));
               listUpdateNeeded = true;
            }
            break;
            default:
               break;
         }
         msgProcessed = true;
      }
   }
   return msgProcessed;
}


void ClockSettings::UpdateManualClkDropDownMsg()
{
   ETG_TRACE_COMP(("ClockSettings::UpdateManualClkDropDownMsg: rowIdx = %d", rowIdx));
   Candera::DropdownChangeType clkmanddChange = Candera::DropdownChangeClose;
   if (((rowIdx == 1)))
   {
      clkmanddChange = Candera::DropdownChangeClose;
      DropdownChangeMsg* ddToggleMsg_ManualClockFormat = COURIER_MESSAGE_NEW(DropdownChangeMsg)(clkmanddChange, 112);
      ETG_TRACE_USR1(("ClockSettings::UpdateManualClkDropDownMsg: clkmanddChange = %d, ddToggleMsg_ManualClockFormat = %d", clkmanddChange, ddToggleMsg_ManualClockFormat));
      if (ddToggleMsg_ManualClockFormat != NULL)
      {
         ddToggleMsg_ManualClockFormat->Post();
      }
   }
   else if (((rowIdx == 2)))
   {
      clkmanddChange = Candera::DropdownChangeClose;
      DropdownChangeMsg* ddToggleMsg_ManualTimeFormat = COURIER_MESSAGE_NEW(DropdownChangeMsg)(clkmanddChange, 111);
      ETG_TRACE_USR1(("ClockSettings::UpdateManualClkDropDownMsg: clkmanddChange = %d, ddToggleMsg_ManualTimeFormat = %d", clkmanddChange, ddToggleMsg_ManualTimeFormat));
      if (ddToggleMsg_ManualTimeFormat != NULL)
      {
         ddToggleMsg_ManualTimeFormat->Post();
      }
   }
   else
   {
      DropdownChangeMsg* ddToggleMsg_ManualTimeFormat = COURIER_MESSAGE_NEW(DropdownChangeMsg)(clkmanddChange, 111);
      if (ddToggleMsg_ManualTimeFormat != NULL)
      {
         ddToggleMsg_ManualTimeFormat->Post();
      }
      DropdownChangeMsg* ddToggleMsg_ManualClockFormat = COURIER_MESSAGE_NEW(DropdownChangeMsg)(clkmanddChange, 112);
      if (ddToggleMsg_ManualClockFormat != NULL)
      {
         ddToggleMsg_ManualClockFormat->Post();
      }
   }
}


void ClockSettings::vClockSpecHourincrementdecrementvalue(int type, int volval)
{
   ETG_TRACE_COMP(("ClockSettings::vClockSpecHourincrementdecrementvalue: type = %d, val=%d", type, volval));
   u8RequestedHourValue = 0;
   if (type == COLUMN_BUTTON_PLUS)
   {
      u8RequestedHourValue =	++volval;
      calculateforplus();
   }
   else if (type == COLUMN_BUTTON_MINUS)
   {
      if (volval == 0)
      {
         u8RequestedHourValue = 23;
      }
      else
      {
         u8RequestedHourValue = --volval;
      }
      calculateforminus();
   }
   ETG_TRACE_USR1(("ClockSettings::vClockSpecHourincrementdecrementvalue: u8RequestedHourValue=%d ", u8RequestedHourValue));
   m_ListClockHourandMinuteVal[3] = u8RequestedHourValue;
   if (m_ListClockHourandMinuteVal[2] == CLOCK_MODE12)
   {
      ClockUtility clockUtility;
      updateClockHrMinValue(&m_ListClockHourandMinuteVal[3]);
      m_ListClockHourandMinuteVal[3] = clockUtility.convert12To24h(m_ListClockHourandMinuteVal[3], m_ListClockHourandMinuteVal[1]);
      ETG_TRACE_USR2(("ClockSettings::vClockSpecHourincrementdecrementvalue inside 12hour modem_ListClockHourandMinuteVal[3] =%d", m_ListClockHourandMinuteVal[3]));
      ClockSrvClient::poGetInstance()->covLocalTimesetMode(static_cast<uint8>(m_ListClockHourandMinuteVal[3]), static_cast<uint8>(m_ListClockHourandMinuteVal[4]));
   }
   else if (m_ListClockHourandMinuteVal[2] == CLOCK_MODE24)
   {
      ClockSrvClient::poGetInstance()->covLocalTimesetMode(static_cast<uint8>(m_ListClockHourandMinuteVal[3]), static_cast<uint8>(m_ListClockHourandMinuteVal[4]));
   }
}


void ClockSettings::vClockSpecMinuteincrementdecrementvalue(int type, int volval)
{
   int buttonType = type;
   int value = volval;
   ETG_TRACE_USR1(("ClockSettings::vClockSpecMinuteincrementdecrementvalue: buttonType = %d, value = %d", buttonType, value));
   ClockUtility clockUtility;
   tU8 u8RequestedMinuteValue = 0;
   u8RequestedMinuteValue = clockUtility.getMinutesValue(value, buttonType);
   if (m_ListClockHourandMinuteVal[2] == CLOCK_MODE12)
   {
      updateClockHrMinValue(&m_ListClockHourandMinuteVal[3]);
      m_ListClockHourandMinuteVal[3] = clockUtility.convert12To24h(m_ListClockHourandMinuteVal[3], m_ListClockHourandMinuteVal[1]);
   }
   m_ListClockHourandMinuteVal[4] = u8RequestedMinuteValue;
   ETG_TRACE_USR1(("ClockSettings::in vClockSpecMinuteincrementdecrementvalue: m_ListClockHourandMinuteVal[3] =%d, m_ListClockHourandMinuteVal[4] =%d\n", m_ListClockHourandMinuteVal[3], m_ListClockHourandMinuteVal[4]));
   ClockSrvClient::poGetInstance()->covLocalTimesetMode(static_cast<uint8>(m_ListClockHourandMinuteVal[3]), static_cast<uint8>(m_ListClockHourandMinuteVal[4]));
}


void ClockSettings::calculateforplus()
{
   ETG_TRACE_COMP(("ClockSettings::calculateforplus:u8RequestedHourValue = %d, m_ListClockHourandMinuteVal[2] = %d, m_ListClockHourandMinuteVal[1] = %d", u8RequestedHourValue, m_ListClockHourandMinuteVal[2], m_ListClockHourandMinuteVal[1]));
   if (u8RequestedHourValue == 24)
   {
      u8RequestedHourValue = 0;
   }
   if (m_ListClockHourandMinuteVal[2] == CLOCK_MODE12)
   {
      if (u8RequestedHourValue == 12)
      {
         if (m_ListClockHourandMinuteVal[1] == CLOCK_C_AM)
         {
            m_ListClockHourandMinuteVal[1] = CLOCK_C_PM ;
         }
         else  if (m_ListClockHourandMinuteVal[1] == CLOCK_C_PM)
         {
            m_ListClockHourandMinuteVal[1] = CLOCK_C_AM ;
         }
      }
      if ((u8RequestedHourValue == 13) && (m_ListClockHourandMinuteVal[1] == CLOCK_C_AM))
      {
         u8RequestedHourValue = u8RequestedHourValue - 12;
         // m_ListClockHourandMinuteVal[1] = CLOCK_C_PM ;
      }
   }
   ETG_TRACE_USR1(("ClockSettings::calculateforplus:m_ListClockHourandMinuteVal[1] = %d, u8RequestedHourValue = %d", m_ListClockHourandMinuteVal[1], u8RequestedHourValue));
}


void ClockSettings::calculateforminus()
{
   ETG_TRACE_COMP(("ClockSettings::calculateforminus:u8RequestedHourValue = %d, m_ListClockHourandMinuteVal[2] = %d, m_ListClockHourandMinuteVal[1] = %d", u8RequestedHourValue, m_ListClockHourandMinuteVal[2], m_ListClockHourandMinuteVal[1]));
   if (m_ListClockHourandMinuteVal[2] == CLOCK_MODE12)
   {
      if (u8RequestedHourValue == 11)
      {
         if (m_ListClockHourandMinuteVal[1] == CLOCK_C_AM)
         {
            m_ListClockHourandMinuteVal[1] = CLOCK_C_PM ;
         }
         else  if (m_ListClockHourandMinuteVal[1] == CLOCK_C_PM)
         {
            m_ListClockHourandMinuteVal[1] = CLOCK_C_AM ;
         }
      }
      if ((u8RequestedHourValue == 0) && (m_ListClockHourandMinuteVal[1] == CLOCK_C_PM))
      {
         u8RequestedHourValue = u8RequestedHourValue + 12;
      }
   }
   ETG_TRACE_USR1(("ClockSettings::calculateforminus: m_ListClockHourandMinuteVal[1] = %d, u8RequestedHourValue = %d", m_ListClockHourandMinuteVal[1], u8RequestedHourValue));
}


void ClockSettings::vUpdateMin_Hr_Value(int Hr_INC_DEC,  int Min_INC_DEC)
{
   /////// Storing the value of Current time
   m_ListClockHourandMinuteVal[3] = Hr_INC_DEC;
   m_ListClockHourandMinuteVal[4] = Min_INC_DEC;
   ETG_TRACE_USR1(("ClockSettings::vUpdateMin_Hr_Value: m_ListClockHourandMinuteVal[2] =%d, m_ListClockHourandMinuteVal[3] =%d, m_ListClockHourandMinuteVal[4] =%d", m_ListClockHourandMinuteVal[2], m_ListClockHourandMinuteVal[3], m_ListClockHourandMinuteVal[4]));
   if (m_ListClockHourandMinuteVal[2] == CLOCK_MODE12)
   {
      ClockUtility clockUtility;
      m_ListClockHourandMinuteVal[1] = clockUtility.getClockMode(Hr_INC_DEC);
      m_ListClockHourandMinuteVal[3] = clockUtility.convert24To12h(Hr_INC_DEC);
      ETG_TRACE_USR2(("ClockSettings::vUpdateMin_Hr_Value:CLOCK_MODE12: m_ListClockHourandMinuteVal[1] = %d, m_ListClockHourandMinuteVal[3] =%d", m_ListClockHourandMinuteVal[1], m_ListClockHourandMinuteVal[3]));
   }
   else if (m_ListClockHourandMinuteVal[2] == CLOCK_MODE24)
   {
      ETG_TRACE_USR2(("ClockSettings::vUpdateMin_Hr_Value:CLOCK_MODE24: m_ListClockHourandMinuteVal[3] =%d", m_ListClockHourandMinuteVal[3]));
   }
   onUpdateMenu(Hour, &m_ListClockHourandMinuteVal[3]);
   onUpdateMenu(Minute, &m_ListClockHourandMinuteVal[4]);
}


/*
 * ManualDateUpdateStatus - dateFormat String updated on call of every onNewDateFormatStatus from clock server client
 * @param[in] dateformat
 * @param[out] None
 * @return void
 * todo :check As of now after onNewDateFormatStatus is called the value of date format and update the string accordingly*/

void ClockSettings::ManualDateUpdateStatus(int dateformat)
{
   ETG_TRACE_USR1(("ClockSettings::ManualDateUpdateStatus received dateformat=%d", dateformat));
   if ((dateformat >= 1) && (dateformat <= 6))
   {
      ETG_TRACE_USR1(("ClockSettings::ManualDateUpdateStatus Valid Date format"));
      _dateformat = Candera::String(DateFormatString[dateformat - 1]);
   }
   else
   {
      ETG_TRACE_USR1(("ClockSettings::ManualDateUpdateStatus Invalid Date format"));
   }
   ETG_TRACE_USR1(("Updated value of dateformat-=%d, dateFormatString-= %s", dateformat, _dateformat.GetCString()));
   /*    oDateFmtUpdateItem.MarkAllItemsModified();
      oDateFmtUpdateItem.SendUpdate(true); */
   if (listcreated == true)
   {
      ListRegistry::s_getInstance().updateList(LIST_ID_SYSTEM_CLOCK_LIST);
   }
}


/**
 * onCourierMessage - To send the selected time zone to client
 * @param[in] ChangeClockTimeZoneReqMsg  oMsg
 * @param[out] none
 * @return bool
* */
bool ClockSettings::onCourierMessage(const ChangeClockTimeZoneReqMsg& oMsg)
{
   uint16 selectedIndex = oMsg.GetSelectedTimeZoneItem();
   ETG_TRACE_USR1(("ClockSettings:ChangeClockTimeZoneReqMsg  is received with time zone = %d", selectedIndex));
   std::map< uint16, TimeZoneListItemInfo >::iterator iter;
   for (iter = _timeZoneListItemInfoMap.begin(); iter != _timeZoneListItemInfoMap.end(); iter++)
   {
      (selectedIndex == iter->first) ? iter->second.enable = true : iter->second.enable = false;
   }
   //selectedIndex = selectedIndex + 1; //to align with MIDW values
   _selectedTimeZone = selectedIndex;
   ListRegistry::s_getInstance().updateList(LIST_ID_SETTINGS_SYSTEM_CLOCK_TIMEZONE);
   ClockSrvClient::poGetInstance()->setRegion_TimeZoneToService(selectedIndex);
   return true;
}


/**
 * updateTimezoneOnStartup - To update the timezone value in the list at startup
 * @param[in] index
 * @param[out] none
 * @return void
* */
void ClockSettings::updateTimezoneOnStartup(uint16 index)
{
   ETG_TRACE_COMP(("ClockSettings::updateTimezoneOnStartup: index = %d", index));
   std::map< uint16, TimeZoneListItemInfo >::iterator iter;
   for (iter = _timeZoneListItemInfoMap.begin(); iter != _timeZoneListItemInfoMap.end(); iter++)
   {
      (index == iter->first) ? iter->second.enable = true : iter->second.enable = false;
   }
   _selectedTimeZone = index;
   ListRegistry::s_getInstance().updateList(LIST_ID_SETTINGS_SYSTEM_CLOCK_TIMEZONE);
}


/**
 * handleAvailableTimezonePropertyUpdate - Get the available list of timezone values
 * @param[in] std::vector
 * @param[out] none
 * @return void
* */

void ClockSettings::handleAvailableTimezonePropertyUpdate(std::vector< TimeZoneList > availableTimeZoneList)
{
   /*This Implementation is applicable only for SMART*/
#ifdef VARIANT_S_FTR_ENABLE_AUX_SUPPORT
   uint16 size = static_cast<uint16>(availableTimeZoneList.size());
   ETG_TRACE_USR1(("ClockSettings:handleAvailableTimezonePropertyUpdate availableTimeZoneList size is = %d", size));
   uint16 index = 0;
   if (!availableTimeZoneList.empty())
   {
      for (std::vector< TimeZoneList >::iterator iter = availableTimeZoneList.begin(); iter != availableTimeZoneList.end(); iter++)
      {
         TimeZoneList stTimeZoneList = *iter;
         std::vector< std::string > timeZoneString = stTimeZoneList.TimeZoneTextList;
         ETG_TRACE_USR3(("ClockSettings:handleAvailableTimezonePropertyUpdate TimeZoneString size is = %d", timeZoneString.size()));
         for (std::vector< std::string >::iterator iter = timeZoneString.begin(); iter != timeZoneString.end(); iter++)
         {
            TimeZoneListItemInfo stTimeZoneListItemInfo;
            stTimeZoneListItemInfo.TimeZoneText = (sTimeZoneListId[index]);
            std::string timezoneMidwList = *iter;
            ETG_TRACE_USR4(("ClockSettings:handleAvailableTimezonePropertyUpdate TimeZoneString is = %s", timezoneMidwList.c_str()));
            _timeZoneListItemInfoMap.insert(std::pair<uint16, TimeZoneListItemInfo>(index, stTimeZoneListItemInfo));
            ++index;
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("ClockSettings:handleAvailableTimezonePropertyUpdate empty list"));
   }
#endif
}


/**
 * updateRegionTimeZoneResponse - Get the available list of timezone values
 * @param[in] uint8 status
 * @param[out] none
 * @return void
* */
void ClockSettings::updateRegionTimeZoneResponse(const uint8 status)
{
   ETG_TRACE_COMP(("ClockSettings::updateRegionTimeZoneResponse: status = %d", status));
   Candera::String selectedTimeZone = "";
   _selectedTimeZone = status;
   std::map< uint16, TimeZoneListItemInfo >::iterator iter = _timeZoneListItemInfoMap.find(_selectedTimeZone);
   if (iter != _timeZoneListItemInfoMap.end())
   {
      selectedTimeZone = iter->second.TimeZoneText;
   }
   updateTimezoneOnStartup(status);
   GpsClockListHandler::poGetInstance()->updatedSelectedTimeZone(selectedTimeZone);
}


uint16 ClockSettings::getVariantType()
{
   uint8 value = 0xFF;
   uint16 u16VehicleType = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("AIVIVariantCoding", "VariantSelection", &value, 1))
   {
      u16VehicleType = value;
   }
   ETG_TRACE_USR1(("ClockSettings::getVariantType:u16VehicleType = %d", u16VehicleType));
   return u16VehicleType;
}


bool ClockSettings::isNaviVariant()
{
   bool bRet = true;
   uint16 u16VehicleType = getVariantType();
   bRet = (u16VehicleType == 0x02) ? false : true;
   ETG_TRACE_USR1(("ClockSettings::isNaviVariant() = %d", bRet));
   return bRet;
}


/**
 * updateClockHrMinValue - To update hr/min value from the address
 * @param[in] address
 * @parm[out] None
 * @return int
 *
 */
void ClockSettings::updateClockHrMinValue(int* address)
{
   m_ListClockHourandMinuteVal[3] = *address;
   ETG_TRACE_USR1(("ClockSettings::updateClockHrMinValue: m_ListClockHourandMinuteVal[3] = %d", m_ListClockHourandMinuteVal[3]));
}


}
}
