/**
 * @file        :  SxmTabListModel.cpp
 * @addtogroup  :  AppHmi_Sxm
 * @brief       :  Contains functions required for Tabbed list handling
 * @copyright   :  (c) 2017-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */
#include "sys_std_if.h"
#include "App/Core/SxmUtils/SxmTabListModel.h"

/******************************************************************************
*
******************************************************************************/
AbstractListModel::AbstractListModel(unsigned int listId, DataItemContext listItemTemplate)
   : /*Base(),*/ _listId(listId), _listItemTemplate(listItemTemplate), _selectedItemIndex(0)
{
}


/******************************************************************************
*
******************************************************************************/
AbstractListModel::~AbstractListModel()
{
   _listItemTemplate = NULL;//for lint warning
}


/******************************************************************************
*
******************************************************************************/
unsigned int AbstractListModel::getListId() const
{
   return _listId;
}


/******************************************************************************
*
******************************************************************************/
size_t AbstractListModel::getSelectedItemIndex() const
{
   return _selectedItemIndex;
}


/******************************************************************************
*
******************************************************************************/
void AbstractListModel::setSelectedItemIndex(size_t index)
{
   if (index != _selectedItemIndex)
   {
      _selectedItemIndex = index;
      onSelectedItemChanged();
   }
}


/******************************************************************************
*
******************************************************************************/
void AbstractListModel::sendUpdate()
{
   ListDataInfo listDataInfo;
   listDataInfo.listId = _listId;
   listDataInfo.startIndex = 0;
   listDataInfo.windowSize = 16;
   ListRegistry::s_getInstance().updateList(50);
}


/******************************************************************************
*
******************************************************************************/
void AbstractListModel::onSelectedItemChanged()
{
   sendUpdate();
}


/******************************************************************************
*
******************************************************************************/
tSharedPtrDataProvider AbstractListModel::getListDataProvider(const ListDataInfo& listDataInfo)
{
   if (listDataInfo.listId != _listId)
   {
      return tSharedPtrDataProvider();
   }

   //use a list builder to fill the list data
   ListDataProviderBuilder listBuilder(_listId, _listItemTemplate);

   size_t lastItemIndex = listDataInfo.startIndex + listDataInfo.windowSize;
   if ((lastItemIndex == 0) || (lastItemIndex > getItemCount()))
   {
      lastItemIndex = getItemCount();
   }

   for (size_t index = listDataInfo.startIndex; index < lastItemIndex; ++index)
   {
      addListItem(listBuilder, index, _selectedItemIndex == index);
   }
   return listBuilder.CreateDataProvider(listDataInfo.startIndex, getItemCount());
}


/******************************************************************************
*
******************************************************************************/
bool AbstractListModel::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   //when an item selection is perfomed by pressing the list item button we also update the list content
   if ((oMsg.GetListId() == _listId) && (oMsg.GetReaction() == enRelease))
   {
      setSelectedItemIndex(oMsg.GetHdl());
      sendUpdate();

      return true;
   }
   return false;
}
