/**
 * @file        :  SxmHall.cpp
 * @addtogroup  :  AppHmi_Sxm
 * @brief       :  SXM HALL class which contains all the necessary business logic required for SXM HMI
 * @copyright   :  (c) 2019-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */
#include "hall_std_if.h"
#include "AppHmi_SxmStateMachine.h"
#include "App/Core/SxmHall.h"
#include "App/Core/SxmAudio/SxmAudioHall.h"
#include "App/Core/SxmUtils/SxmUtils.h"
#include "App/Core/SxmUtils/SxmUtilsListHandler.h"
#include "CgiExtensions/ImageLoader.h"
#include "App/Core/SxmCommon/SxmAdvisories.h"
#include "App/Core/SxmCommon/SxmHMIServiceStateHandler.h"
#include "CgiExtensions/DataBindingItem.hpp"
#include "CgiExtensions/CourierMessageMapper.h"
#include "Courier/Visualization/VisualizationMsgs.h"
#include "App/Core/ClientHandler/SourceSwitch/SXMSourceSwitch.h"
#include "App/Core/ClientHandler/ClockService/SXMClockService.h"
#include "App/Core/ClientHandler/HmiInfoService/SxmHmiInfoService.h"

//#define APPLICATION_SWITCH_PORT "applicationSwitchPort"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_SXM_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX                 App::Core::SxmHall::
#include "trcGenProj/Header/SxmHall.cpp.trc.h"
#endif

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_hmi_06_if.h"


namespace App {
namespace Core {

DEFINE_CLASS_LOGGER_AND_LEVEL("App/Core", SxmHall, Info);

/**
 * SxmHall class constructor, main HALL class for SXM HMI
 * @class : SxmHall
 */
SxmHall::SxmHall()
   : HallComponentBase("", "App.Core.AppHmi_Sxm", "/org/genivi/NodeStateManager/LifeCycleConsumer/AppHmi_Sxm")
{
   ETG_I_REGISTER_FILE();

   ETG_TRACE_USR4(("SxmHall() class constructor"));
   DefSetServiceBase::s_Intialize("AppHmiSxmServicePort");//port name needs to be same as port name given in cmc file
   DP_vCreateDatapool();

    _defSetServiceBase = DefSetServiceBase::GetInstance();
   _mPtrSxmAdvisories = SxmAdvisories::instance();
   _mPtrSxmUtilsListHandler = SxmUtilsListHandler::instance();
   _ptrSxmHMIServiceStateHandler = SxmHMIServiceStateHandler::instance();
   _mPtrSXMSourceSwitch = SXMSourceSwitch::instance();
   _mPtrSXMClockService = SXMClockService::instance();
   _pSxmHmiInfoService = SxmHmiInfoService::instance();

   SXM_UTILS_CREATE_OBJECT_WITH_0PARAM(_mPtrSxmAudioHall, SxmAudioHall);
   _applicationSwitchClientHandler = ApplicationSwitchClientHandler::instance();
   _applicationSwitchClientComponent = ApplicationSwitchClientComponent::poGetInstance();



   if (_applicationSwitchClientComponent != NULL)
   {
      _applicationSwitchClientComponent->setApplicationID(APPID_APPHMI_SXM);
   }
}


/**
 * SxmHall class destructor
 */
SxmHall::~SxmHall()
{
   ETG_TRACE_USR4(("SxmHall() class destructor"));

   if (_defSetServiceBase != NULL)
   {
      _defSetServiceBase->vUnRegisterforUpdate(this);
      //do not delete this instance as its not created by us.
      _defSetServiceBase = NULL;
   }
   _ptrSxmHMIServiceStateHandler = NULL;
   _applicationSwitchClientHandler = NULL;
   DefSetServiceBase::s_Destrory(); //to destroy DefSetServiceBase instance
   SXM_UTILS_RELEASE_OBJECT(_mPtrSxmAudioHall);
   SXM_UTILS_RELEASE_OBJECT(_mPtrSxmAdvisories);
   SXM_UTILS_RELEASE_OBJECT(_mPtrSxmUtilsListHandler);
   SXM_UTILS_RELEASE_OBJECT(_mPtrSXMSourceSwitch);
   SXM_UTILS_RELEASE_OBJECT(_mPtrSXMClockService);
   //SXM_UTILS_RELEASE_OBJECT(_applicationSwitchClientHandler);
   SXM_UTILS_RELEASE_OBJECT(_pSxmHmiInfoService);
   ETG_I_UNREGISTER_FILE();
}


/**
 * Courier Message Handler function
 * @param [in] Msg Courier Message posted from Courier Service Framework
 * @return True, if the message is consumed; False, if the message is not consumed.
 */
bool SxmHall::OnMessage(const Courier::Message& Msg)
{
   bool status = false; // variable used only to avoid coverity issue.
   if (Courier::StartupMsg::ID == Msg.GetId())
   {
      status = HallComponentBase::OnMessage(Msg);
   }
   else if (GuiStartupFinishedUpdMsg::ID == Msg.GetId())
   {
      ETG_TRACE_USR4(("SxmHall::GuiStartupFinishedUpdMsg"));
      Courier::Message* msg = COURIER_MESSAGE_NEW(::Courier::ViewReqMsg)(Courier::ViewAction::Create, ::Courier::ViewId("Sxm#Scenes#SXM_RADIO__MAIN"), true, true);
      if (msg != 0 && msg->Post())
      {
         ETG_TRACE_USR4(("ViewReqMsg  Message is  posted."));
      }
      else
      {
         ETG_TRACE_FATAL(("ViewReqMsg  Message is not posted."));
      }

      if (_ptrSxmHMIServiceStateHandler)
      {
         _ptrSxmHMIServiceStateHandler->OnMessage(Msg);
      }

      if (_defSetServiceBase != NULL)
      {
         _defSetServiceBase->vRegisterforUpdate(this);
      }
      vonGuiStartupFinished();
      status = HallComponentBase::OnMessage(Msg);
   }

   COURIER_MSG_DELEGATE_TO_OBJ(_mPtrSxmUtilsListHandler)
   COURIER_MSG_DELEGATE_TO_OBJ(_mPtrSxmAudioHall)
   COURIER_MSG_DELEGATE_TO_OBJ(_mPtrSXMSourceSwitch)
   COURIER_MSG_DELEGATE_TO_OBJ(_mPtrSxmAdvisories)
   COURIER_MSG_DELEGATE_TO_OBJ(_ptrSxmHMIServiceStateHandler)
   COURIER_MSG_DELEGATE_TO_OBJ(_applicationSwitchClientHandler)
   COURIER_MSG_DELEGATE_TO_CLASS(::languageHandler::LanguageHandlerAdapter)

   return false;
}


/**
 * vonGuiStartupFinished
 * @param [in] : Courier message
 */
void SxmHall::vonGuiStartupFinished()
{
   ETG_TRACE_USR4(("SxmHall::vonGuiStartupFinished"));
   if (_mPtrSxmAudioHall)
   {
      _mPtrSxmAudioHall->vonGuiStartupFinished();
   }
}


/**
 * Timer expiry handler function
 * @param [in] : timer - ASF core proxy class pointer
 * @param [in] : stateChange - Current service state of ASF core
 */
void SxmHall::onExpired(asf::core::Timer& /*timer*/,
                        boost::shared_ptr<asf::core::TimerPayload> /*data*/)
{
   ETG_TRACE_USR4(("SxmHall::onExpired()"));
}


/**
 * Function for executing restore factory setting
 */
void SxmHall::reqPrepareResponse(const PrepareResDiagMsgData& /*oPrepareResDiagMsgData*/)
{
   ETG_TRACE_USR4(("SxmHall reqPrepareResponse"));
   if (_defSetServiceBase != NULL)
   {
      _defSetServiceBase->sendPrepareResponse(0, this);
   }
}


/**
 * Function for preparing restore factory setting
 */
void SxmHall::reqExecuteResponse(const ExecuteResDiagMsgData& /*oExecuteResDiagMsgData*/)
{
   ETG_TRACE_USR4(("SxmHall reqExecuteResponse"));
   if (_defSetServiceBase != NULL)
   {
      _defSetServiceBase->sendExecuteResponse(static_cast<int>(bSxmRestoreFactorySetting()), this);
   }
}


/**
 * Function for finalizing restore factory setting
 */
void SxmHall::reqFinalizeResponse(const FinalizeResDiagMsgData& /*oFinalizeResDiagMsgData*/)
{
   ETG_TRACE_USR4(("SxmHall reqFinalizeResponse"));
   if (_defSetServiceBase != NULL)
   {
      _defSetServiceBase->sendFinalizeResponse(0, this);
   }
}


/**
 * Helper Function for implementing restore factory setting
 * @return : Restore result success or failure
 */
bool SxmHall::bSxmRestoreFactorySetting()
{
   bool bResult = true;//1 - Restore failed (for Diagnosis)
   bool bSxmAudioResult = false;//For our services, "false" is failed."true"-success
   ETG_TRACE_USR4(("SxmHall bSxmRestoreFactorySetting"));
   SXM_UTILS_DELEGATE_REQUEST_WITH_RETURN(_mPtrSxmAudioHall, bSxmAudioRestoreFactorySetting, bSxmAudioResult);

   ETG_TRACE_USR4(("Restore status of Audio:%u", bSxmAudioResult));
   //Only if all services have done the restore
   if (bSxmAudioResult)
   {
      bResult = false;//0 - Restore success (for Diagnosis)
   }
   return bResult;
}


} // namespace Core
} // namespace App
