/**************************************************************************************
 * @file        : SxmAdvisories.h
 * @addtogroup  : AppHmi_Sxm
 * @brief       : Common class to handle advisories.
 * @copyright   : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                The reproduction, distribution and utilization of this file as
 *                well as the communication of its contents to others without express
 *                authorization is prohibited. Offenders will be held liable for the
 *                payment of damages. All rights reserved in the event of the grant
 *                of a patent, utility model or design.
 **************************************************************************************/
#ifndef _SxmAdvisories_H_
#define _SxmAdvisories_H_

#include "App/Core/SxmUtils/SxmUtilsSingleton.h"
#include "ISxmCommon.h"
#include "sxm_audio_main_fiProxy.h"
#include "AppHmi_SxmMessages.h"
#include "bosch/cm/ai/nissan/hmi/hmidataservice/HmiDataProxy.h"
#include "AppBase/ServiceAvailableIF.h"

#ifndef HMI_DATA_SERVICE_NAMESPACE
#define HMI_DATA_SERVICE_NAMESPACE ::bosch::cm::ai::nissan::hmi::hmidataservice::HmiData
#endif


namespace App {
namespace Core {
/**
 * Common class to handle advisories.
 * @class : SxmAdvisories
 */
class SxmAdvisories
   : public SxmUtilsSingleton<SxmAdvisories>
   , public hmibase::ServiceAvailableIF
   , public StartupSync::PropertyRegistrationIF
   , public ::sxm_audio_main_fi::SxmDataServiceStatusCallbackIF
   , public ::sxm_audio_main_fi::SxmDataServiceAntennaStatusCallbackIF
   , public ::sxm_audio_main_fi::SystemMessageCallbackIF
   , public ::sxm_audio_main_fi::SxmDiagSignalQualityCallbackIF
{
   public:
      /**
       * SxmAdvisories Class constructor
       */
      SxmAdvisories();
      /**
       * SxmAdvisories Class destructor
       */
      virtual ~SxmAdvisories();
      /**
       * Utility function that registers for notifications updates from SXM Audio MIDW layer
       */
      void registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/);
      /**
       * Utility function that deregisters for notifications updates from SXM Audio MIDW layer
       */
      void deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/);
      /**
       * Error handling function for sxm data service status property
       */
      void onSxmDataServiceStatusError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/, const boost::shared_ptr< ::sxm_audio_main_fi::SxmDataServiceStatusError >& /*error*/);
      /**
       * Status handling function for sxm data service status property
       */
      void onSxmDataServiceStatusStatus(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/, const boost::shared_ptr< ::sxm_audio_main_fi::SxmDataServiceStatusStatus >& status);
      /**
       * Error handling function for sxm data service antenna status
       */
      void onSxmDataServiceAntennaStatusError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/, const boost::shared_ptr< ::sxm_audio_main_fi::SxmDataServiceAntennaStatusError >& /*error*/);
      /**
       * Status handling function for sxm data service antenna status
       */
      void onSxmDataServiceAntennaStatusStatus(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/, const boost::shared_ptr< ::sxm_audio_main_fi::SxmDataServiceAntennaStatusStatus >& /*status*/);
      /**
       * Error handling function for system messages property
       */
      void onSystemMessageError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
                                const boost::shared_ptr< ::sxm_audio_main_fi::SystemMessageError >& /*error*/);
      /**
       * Status handling function for system messages property
       */
      void onSystemMessageStatus(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
                                 const boost::shared_ptr< ::sxm_audio_main_fi::SystemMessageStatus >& status);
      /**
       * Error handling function for sxm diag signal quality property
       */
      void onSxmDiagSignalQualityError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& proxy,
                                       const ::boost::shared_ptr< ::sxm_audio_main_fi::SxmDiagSignalQualityError >& error);
      /**
       * Result handling function for sxm diag signal quality property
       */
      void onSxmDiagSignalQualityStatus(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& proxy,
                                        const ::boost::shared_ptr< ::sxm_audio_main_fi:: SxmDiagSignalQualityStatus >& status);
      /**
       * Register call back function to receive updates.
       */
      void vRegisterNotification(ISxmCommon* _obsptr, enSXMServiceType enServiceType, enSXMEventNotificationType enEventType);
      /**
       * De-Register call back function to receive updates.
       */
      void vDeregisterNotification(ISxmCommon* _obsptr, enSXMServiceType enServiceType, enSXMEventNotificationType enEventType);
      /**
       * Helper function to get current status of service.
       */
      enSXMDataServiceAdvisory getDataServiceStatus(enSXMServiceType enServiceType);
      void onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/);
      void onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/);

   private:
      friend class  SxmUtilsSingleton<SxmAdvisories>;

      //Member Variable
      ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy > _mPtrSxmAudioFiProxy;
      ::boost::shared_ptr<HMI_DATA_SERVICE_NAMESPACE::HmiDataProxy> _mPtrhmiDataProxyClient ;
      std::map<enSXMServiceType, enSXMDataServiceAdvisory> _dsAdvisoryStatus;
      std::map<enSXMServiceType, enSXMDataServiceAdvisory> _dsPrevAdvisoryStatus;
      std::map<enSXMServiceType, ISxmCommon*> _dsObsvrList;
      bool _dsAntennaStatus;
      bool _dsSignalStatus;
      bool _dsInitializing;

      //Member functions
      /**
       * Helper function to get current antenna status for data service service.
       */
      bool getDataServiceAntennaStatus(void) const;
      /**
       * Helper function to notify all registered clients.
       */
      void notifyDataServiceStatus();
      /**
       * Helper function to update Advisory status for SXM data services
       */
      void updateServiceStatus();
      /**
       * Helper function to update weather main status
       */
      enSXMDataServiceAdvisory getWeatherMainStatus();
      /**
      * Callback to handle Courier Message SxmSurfaceStateChngUpdtMsg
      */
      bool onCourierMessage(const SxmSurfaceStateChngUpdtMsg& msg);

      COURIER_MSG_MAP_BEGIN(TR_CLASS_APPHMI_SXM_COURIER_PAYLOAD_MODEL_COMP)
      ON_COURIER_MESSAGE(SxmSurfaceStateChngUpdtMsg)
      COURIER_MSG_MAP_DELEGATE_DEF_BEGIN()
      COURIER_MSG_MAP_DELEGATE_DEF_END()
      COURIER_MSG_MAP_DELEGATE_END()

   protected:
};


}  // namespace Core
}  // namespace App

#undef HMI_DATA_SERVICE_NAMESPACE

#endif
