/**************************************************************************************
 * @file        : SxmAdvisories.cpp
 * @addtogroup  : AppHmi_Sxm
 * @brief       : Utility class which contains all the necessary business logic required to handle Common Advisory.
 * @copyright   : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                The reproduction, distribution and utilization of this file as
 *                well as the communication of its contents to others without express
 *                authorization is prohibited. Offenders will be held liable for the
 *                payment of damages. All rights reserved in the event of the grant
 *                of a patent, utility model or design.
 **************************************************************************************/
#include "hmi_trace_if.h"
#include "CgiExtensions/CourierMessageMapper.h"
//#include "hall_std_if.h"  //when other headers also needed, include hall_std_if and remove above two
#include "SxmAdvisories.h"
#include "AppHmi_SxmStateMachineData.h"
#include "SxmHMIServiceStateHandler.h"
#include "App/Core/SxmUtils/SxmUtils.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXMCOMMON
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::SxmAdvisories::
#include "trcGenProj/Header/SxmAdvisories.cpp.trc.h"
#endif

namespace App {
namespace Core {
using namespace ::sxm_audio_main_fi;
using namespace ::sxm_main_fi_types;
using namespace ::sxm_audio_main_fi_types;
using namespace bosch::cm::ai::nissan::hmi::hmidataservice::HmiData;
/**
 * SxmAdvisories Class constructor
 * @param [in] : pSxmAudioFiProxy - SXM Audio FI proxy
 */
SxmAdvisories::SxmAdvisories()
   : _mPtrSxmAudioFiProxy(Sxm_audio_main_fiProxy::createProxy("SxmAudioFiPort", *this))
   , _dsAntennaStatus(true), _dsSignalStatus(true), _dsInitializing(false)
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("SxmAdvisories() Constructor"));
   SXM_UTILS_REGISTER_PROPERTY_CALLBACKS(StartupSync::getInstance(), this, _mPtrSxmAudioFiProxy);
   _dsObsvrList.clear();
   _dsAdvisoryStatus.clear();
   _dsPrevAdvisoryStatus.clear();
}


/**
 * SxmAdvisories Class destructor
 */
SxmAdvisories::~SxmAdvisories()
{
   ETG_TRACE_USR4(("~SxmAdvisories() Destructor"));
   ETG_I_UNREGISTER_FILE();

   _dsObsvrList.clear();
   _dsAdvisoryStatus.clear();
   _dsPrevAdvisoryStatus.clear();
   _mPtrSxmAudioFiProxy.reset();
}


/**
 * Utility function that registers for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAdvisories::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAdvisories::registerProperties"));
   if ((proxy == _mPtrSxmAudioFiProxy) && (_mPtrSxmAudioFiProxy))
   {
      _mPtrSxmAudioFiProxy->sendSystemMessageUpReg(*this);
      _mPtrSxmAudioFiProxy->sendSxmDataServiceAntennaStatusUpReg(*this);
      _mPtrSxmAudioFiProxy->sendSxmDataServiceStatusUpReg(*this);
      _mPtrSxmAudioFiProxy->sendSxmDiagSignalQualityUpReg(*this);
   }
}


/**
 * Utility function that deregisters for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAdvisories::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAdvisories::deregisterProperties"));
   if ((proxy == _mPtrSxmAudioFiProxy) && (_mPtrSxmAudioFiProxy))
   {
      _mPtrSxmAudioFiProxy->sendSystemMessageRelUpRegAll();
      _mPtrSxmAudioFiProxy->sendSxmDataServiceStatusRelUpRegAll();
      _mPtrSxmAudioFiProxy->sendSxmDataServiceAntennaStatusRelUpRegAll();
      _mPtrSxmAudioFiProxy->sendSxmDiagSignalQualityRelUpRegAll();
   }
   if ((proxy == _mPtrhmiDataProxyClient) && (_mPtrhmiDataProxyClient))
   {
      //_mPtrhmiDataProxyClient->sendVisibleApplicationModeDeregisterAll();
   }
}


/**
 * Utility function that registers for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAdvisories::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAdvisories::onAvailable"));
   if ((proxy == _mPtrSxmAudioFiProxy) && (_mPtrSxmAudioFiProxy))
   {
      _mPtrSxmAudioFiProxy->sendSystemMessageUpReg(*this);
      _mPtrSxmAudioFiProxy->sendSxmDataServiceAntennaStatusUpReg(*this);
      _mPtrSxmAudioFiProxy->sendSxmDataServiceStatusUpReg(*this);
      _mPtrSxmAudioFiProxy->sendSxmDiagSignalQualityUpReg(*this);
   }
}


/**
 * Utility function that deregisters for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAdvisories::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAdvisories::onUnavailable"));
   if ((proxy == _mPtrSxmAudioFiProxy) && (_mPtrSxmAudioFiProxy))
   {
      _mPtrSxmAudioFiProxy->sendSystemMessageRelUpRegAll();
      _mPtrSxmAudioFiProxy->sendSxmDataServiceStatusRelUpRegAll();
      _mPtrSxmAudioFiProxy->sendSxmDataServiceAntennaStatusRelUpRegAll();
      _mPtrSxmAudioFiProxy->sendSxmDiagSignalQualityRelUpRegAll();
   }
}


/**
 * Register call back function to receive updates.
 * @param [in] : _obsptr - object reference of observer.
 * @param [in] : enSXMServiceType - data Service type
 */
void SxmAdvisories::vRegisterNotification(ISxmCommon* _obsptr, enSXMServiceType enServiceType, enSXMEventNotificationType enEventType)
{
   if (enEventType == SXMDataServiceStatus)
   {
      if (_dsObsvrList.find(enServiceType) != _dsObsvrList.end())
      {
         ETG_TRACE_ERR(("Service is already registered"));
      }
      else
      {
         _dsObsvrList.insert(std::pair<enSXMServiceType, ISxmCommon*>(enServiceType, _obsptr));
      }
   }
}


/**
 * De-Register call back function to receive updates.
 * @param [in] : _obsptr - object reference of observer.
 * @param [in] : enSXMServiceType - data Service type
 */
void SxmAdvisories::vDeregisterNotification(ISxmCommon* /*_obsptr*/, enSXMServiceType enServiceType, enSXMEventNotificationType enEventType)
{
   if (enEventType == SXMDataServiceStatus)
   {
      if (_dsObsvrList.find(enServiceType) != _dsObsvrList.end())
      {
         _dsObsvrList.erase(enServiceType);
      }
      else
      {
         ETG_TRACE_ERR(("Service is not registered"));
      }
   }
}


/**
 *Error handling function for property updates SxmDataServiceStatusStatus
 * @param [in] : proxy - SXM Channel Art FI proxy class instance
 * @param [in] : error - result type
 */
void SxmAdvisories::onSxmDataServiceStatusError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/, const boost::shared_ptr< SxmDataServiceStatusError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmAdvisories::onSxmDataServiceStatusError"));
}


/**
 *Status handling function for property updates SxmDataServiceStatusStatus
 * @param [in] : proxy - SXM Channel Art FI proxy class instance
 * @param [in] : status
 */
void SxmAdvisories::onSxmDataServiceStatusStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/, const boost::shared_ptr< SxmDataServiceStatusStatus >& status)
{
   if (status) //Safety Check.
   {
      size_t listSize = status->getList().size();
      unsigned int lActiveService = SxmHMIServiceStateHandler::instance()->getActiveState();
      bool isSxmInForeground = SxmHMIServiceStateHandler::instance()->getApplicationState();
      ETG_TRACE_USR4(("onSxmDataServiceStatusStatus isSxmInForeground:%u, lActiveService:%u", isSxmInForeground, lActiveService));
      for (uint8 idx = 0; idx < listSize; ++idx)
      {
         unsigned int serviceStatus = status->getList()[idx].getStatus().getStatus();
         unsigned int serviceType = status->getList()[idx].getType();
         ETG_TRACE_USR4(("onSxmDataServiceStatusStatus serviceStatus %u, serviceType %u ", serviceStatus, serviceType));
         _dsAdvisoryStatus[static_cast<enSXMServiceType>(serviceType)] = static_cast<enSXMDataServiceAdvisory>(serviceStatus);
         //Display advisory.
         if ((serviceType == lActiveService) && (isSxmInForeground))
         {
            if (serviceStatus == static_cast<unsigned int>(T_e8_SxmDataServiceStatus__NOT_SUBSCRIBED))
            {
               POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(Sxm_SxmPopups_SXM__MPOP_SUBSCRIPTION_ADVISORY)));
            }
            else
            {
               //TODO post message only if pop up is active.
               POST_MSG((COURIER_MESSAGE_NEW(SXMDeActivatePopUpMsg)(Sxm_SxmPopups_SXM__MPOP_SUBSCRIPTION_ADVISORY)));
            }
         }
      }
      notifyDataServiceStatus();

      //To Display subscription advisory for weather main
      unsigned int weatherMain = static_cast <unsigned int>(SXMWeatherMain);
      if ((weatherMain == lActiveService) && (getWeatherMainStatus() == SXMDSNotSubscribed))
      {
         POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(Sxm_SxmPopups_SXM__MPOP_SUBSCRIPTION_ADVISORY)));
      }
      else if (weatherMain == lActiveService)
      {
         POST_MSG((COURIER_MESSAGE_NEW(SXMDeActivatePopUpMsg)(Sxm_SxmPopups_SXM__MPOP_SUBSCRIPTION_ADVISORY)));
      }
   }
}


/**
 * Error handling function for sxm data service antenna status property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAdvisories::onSxmDataServiceAntennaStatusError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/, const boost::shared_ptr< SxmDataServiceAntennaStatusError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAdvisories::onSxmDataServiceAntennaStatusError"));
}


/**
 * Status handling function for sxm data service antenna status property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - Property status info
 */
void SxmAdvisories::onSxmDataServiceAntennaStatusStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/, const boost::shared_ptr< SxmDataServiceAntennaStatusStatus >& status)
{
   ETG_TRACE_USR4(("SxmAdvisories::onSxmDataServiceAntennaStatusStatus"));
   _dsAntennaStatus = status->getIsAntennaConnected();
   updateServiceStatus();
   notifyDataServiceStatus();
}


/**
 * Error handling function for system messages property
 * @param [in] : proxy SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAdvisories::onSystemMessageError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/, const boost::shared_ptr< SystemMessageError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAdvisories::onSystemMessageError"));
}


/**
 * Status handling function for system messages property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - Property status info
 */
void SxmAdvisories::onSystemMessageStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/, const boost::shared_ptr< SystemMessageStatus >& status)
{
   ETG_TRACE_ERR(("SxmAdvisories::onSystemMessageStatus err:%d", status->getSystemErrorType()));
   bool currentStatus = (status->getSystemErrorType() == T_e8_SystemMessage__XMTUN_SYSTEM_MSG_SXM_INITIALIZING);
   if (currentStatus != _dsInitializing)
   {
      _dsInitializing = currentStatus;
      updateServiceStatus();
      notifyDataServiceStatus();
   }
}


/**
 * Error handling function for sxm diag signal quality property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAdvisories::onSxmDiagSignalQualityError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< SxmDiagSignalQualityError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmAudioDiagnosis::onSxmDiagSignalQualityError"));
}


/**
 * Status handling function for sxm diag signal quality property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - Property status info
 */
void SxmAdvisories::onSxmDiagSignalQualityStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< SxmDiagSignalQualityStatus >& status)
{
   ETG_TRACE_USR4(("SxmAdvisories::onSxmDiagSignalQualityStatus :%d", status->getSignalQualityStatus()));
   bool crntstatus = (status->getSignalQualityStatus() != T_e8_Sxm_SignalQuality__FC_SXM_DIAG_SQ_NO_SIGNAL);
   if (_dsSignalStatus != crntstatus)
   {
      _dsSignalStatus = crntstatus;
      updateServiceStatus();
      notifyDataServiceStatus();
   }
}


/**
 * Helper function to get current status of service.
 * @param [in] : enServiceType - data service type.
 * return : enSXMDataServiceAdvisory - data Service status
 */
enSXMDataServiceAdvisory SxmAdvisories::getDataServiceStatus(enSXMServiceType enServiceType)
{
   enSXMDataServiceAdvisory ldsadvisorystatus = SXMDSUnknown;
   if (_dsAdvisoryStatus.find(enServiceType) != _dsAdvisoryStatus.end())
   {
      ldsadvisorystatus =  _dsAdvisoryStatus[enServiceType];
   }
   else if (enServiceType == SXMWeatherMain)
   {
      ldsadvisorystatus = getWeatherMainStatus();
   }
   if (SXMChannelArt != enServiceType)
   {
      if (_dsInitializing)
      {
         ldsadvisorystatus = SXMDSUnknown;
      }
      else if (ldsadvisorystatus == SXMDSNotSubscribed)
      {
         ldsadvisorystatus = SXMDSNotSubscribed;
      }
      else if (!_dsAntennaStatus)
      {
         ldsadvisorystatus = SXMDSCheckAntenna;
      }
      else if (!_dsSignalStatus && ldsadvisorystatus == SXMDSUnknown)
      {
         ldsadvisorystatus = SXMDSNoSignal;
      }
      else if (!_dsSignalStatus && ldsadvisorystatus == SXMDSSubscribed)
      {
         ldsadvisorystatus = SXMDSReadyNoSignal;
      }
   }
   if ((enServiceType != SXMWeatherMain))
   {
      _dsAdvisoryStatus[enServiceType] = ldsadvisorystatus;
   }
   return ldsadvisorystatus;
}


/**
 * Helper function to get current antenna status of data service.
 * return : bool - antenna status
 */
bool SxmAdvisories::getDataServiceAntennaStatus(void) const
{
   return _dsAntennaStatus;
}


/**
 * Helper function to notify all registered clients.
 */
void SxmAdvisories::notifyDataServiceStatus()
{
   std::map<enSXMServiceType, ISxmCommon*>::const_iterator dsObsvrListitr;
   for (dsObsvrListitr = _dsObsvrList.begin(); dsObsvrListitr != _dsObsvrList.end(); ++dsObsvrListitr)
   {
      enSXMDataServiceAdvisory ldsadvisorystatus = getDataServiceStatus(dsObsvrListitr->first);
      if (_mPtrSxmAudioFiProxy && _mPtrSxmAudioFiProxy->hasSxmDataServiceStatus())
      {
         if ((_dsPrevAdvisoryStatus.find(dsObsvrListitr->first) != _dsPrevAdvisoryStatus.end()) &&
               (ldsadvisorystatus != _dsPrevAdvisoryStatus[dsObsvrListitr->first]))
         {
            dsObsvrListitr->second->vNotifyAdvisoryStatus(dsObsvrListitr->first, ldsadvisorystatus, SXMDataServiceStatus);
         }
      }
      else
      {
         dsObsvrListitr->second->vNotifyAdvisoryStatus(dsObsvrListitr->first, ldsadvisorystatus, SXMDataServiceStatus);
      }
   }
   _dsPrevAdvisoryStatus = _dsAdvisoryStatus;
}


/**
 * Helper function to update weather main status
 */
enSXMDataServiceAdvisory SxmAdvisories::getWeatherMainStatus()
{
   enSXMDataServiceAdvisory ldsadvisorystatus = SXMDSUnknown;
   std::map<enSXMServiceType, enSXMDataServiceAdvisory>::const_iterator itr;
   for (itr = _dsAdvisoryStatus.begin(); itr != _dsAdvisoryStatus.end(); ++itr)
   {
      if (SXMAGW == itr->first || SXMTabweather == itr->first || SXMWSAlerts == itr->first)
      {
         if (SXMDSNotSubscribed == itr->second)
         {
            ldsadvisorystatus = SXMDSNotSubscribed;
         }
         else
         {
            ldsadvisorystatus = (SXMDSSubscribed == itr->second) ? SXMDSSubscribed : ldsadvisorystatus;
            break;
         }
      }
   }
   return ldsadvisorystatus;
}


/**
* onCourierMessage - Courier message handler for SxmSurfaceStateChngUpdtMsg
* @param[in] : Message
* @return : TRUE if consumed
*/
bool SxmAdvisories::onCourierMessage(const SxmSurfaceStateChngUpdtMsg& /*msg*/)
{
   uint8 activeService = SxmHMIServiceStateHandler::instance()->getActiveState();
   ETG_TRACE_USR4(("SxmAdvisories::SxmSurfaceStateChngUpdtMsg activeService:%d", activeService));
   std::map<enSXMServiceType, enSXMDataServiceAdvisory>::const_iterator itr;
   bool isSxmInForeground = SxmHMIServiceStateHandler::instance()->getApplicationState();
   for (itr = _dsAdvisoryStatus.begin(); itr != _dsAdvisoryStatus.end(); ++itr)
   {
      if ((itr->first == static_cast<enSXMServiceType>(activeService)) && (itr->second == SXMDSNotSubscribed) && isSxmInForeground)
      {
         ETG_TRACE_USR4(("SxmSurfaceStateChngUpdtMsg Enable popup"));
         POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(Sxm_SxmPopups_SXM__MPOP_SUBSCRIPTION_ADVISORY)));
      }
   }
   if (SXMWeatherMain == activeService && getWeatherMainStatus() == SXMDSNotSubscribed)
   {
      POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(Sxm_SxmPopups_SXM__MPOP_SUBSCRIPTION_ADVISORY)));
   }
   return true;
}


/**
 * Helper function to update Advisory status for SXM data services
 */
void SxmAdvisories::updateServiceStatus()
{
   if (_mPtrSxmAudioFiProxy && _mPtrSxmAudioFiProxy->hasSxmDataServiceStatus())
   {
      size_t listSize = _mPtrSxmAudioFiProxy->getSxmDataServiceStatus().getList().size();
      for (uint8 idx = 0; idx < listSize; ++idx)
      {
         unsigned int serviceStatus = _mPtrSxmAudioFiProxy->getSxmDataServiceStatus().getList()[idx].getStatus().getStatus();
         unsigned int serviceType = _mPtrSxmAudioFiProxy->getSxmDataServiceStatus().getList()[idx].getType();
         _dsAdvisoryStatus[static_cast<enSXMServiceType>(serviceType)] = static_cast<enSXMDataServiceAdvisory>(serviceStatus);
      }
   }
}


}  // namespace Core
}  // namespace App


/**
 * Helper Function to get data service state.(Called from IAR)
 */
uint8 gacGetDataServiceStatus(uint8 servicetype)
{
   //TODO:Not thread safe revisit implementation.
   return ::App::Core::SxmAdvisories::instance()->getDataServiceStatus(static_cast<enSXMServiceType>(servicetype));
}
