/**************************************************************************************
 * @file         : SxmAudioTWNow.cpp
 * @addtogroup   : AppHmi_Sxm
 * @brief        : Functional class to handle business logic for Traffic and Weather Now.
 * @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 **************************************************************************************/
#include "hall_std_if.h"   // Including HALL framework code
#include "SxmAudioTWNow.h"
#include "SxmAudioListHelper.h"
#include "SxmAudioDataBindingUtils.h"
#include "App/Core/SxmCommon/SXMCommonInterface.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXMAUDIO
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::SxmAudioTWNow::
#include "trcGenProj/Header/SxmAudioTWNow.cpp.trc.h"
#endif


namespace App {
namespace Core {
using namespace ::sxm_audio_main_fi;
using namespace ::sxm_audio_main_fi_types;

/**
 * SxmAudioTWNow Class constructor
 */
SxmAudioTWNow::SxmAudioTWNow(::boost::shared_ptr<Sxm_audio_main_fiProxy > pSxmAudioFiProxy)
   : _sxmAudioFiProxy(pSxmAudioFiProxy)
   , _twnServiceStatus(T_e8_FlashServiceStatus__FLASH_SERVICE_ERROR)
   , _bulletinStatus(T_e8_TWNow_Bulletin_Status__TWNOW_NO_BULLETIN_AVAILABLE)
   , _bulletinId(1)
   , _buttonColor(0)
   , _isMarketActive(false)
   , _subStatus(SXM_AUDIO_SUBSCRIBED)
   , _playbackStatus(T_e8_FlashEventPlaybackStatus__FLASHEVENT_UNKNOWN)

{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("SxmAudioTWNow() Constructor"));
}


/**
 * SxmAudioTWNow Class destructor
 */
SxmAudioTWNow::~SxmAudioTWNow()
{
   ETG_TRACE_USR4(("~SxmAudioTWNow() Destructor"));
   ETG_I_UNREGISTER_FILE();
   _sxmAudioFiProxy.reset();
   _twnServiceStatus = T_e8_FlashServiceStatus__FLASH_SERVICE_ERROR;
   _bulletinStatus = T_e8_TWNow_Bulletin_Status__TWNOW_NO_BULLETIN_AVAILABLE;
   _bulletinId = 1;
   _buttonColor = 0;
   _isMarketActive = false;
   _subStatus = SXM_AUDIO_SUBSCRIBED;
   _playbackStatus = T_e8_FlashEventPlaybackStatus__FLASHEVENT_UNKNOWN;
}


/**
 * Utility function that registers for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioTWNow::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAudioTWNow::registerProperties"));
   if ((proxy == _sxmAudioFiProxy) && (_sxmAudioFiProxy))
   {
      _sxmAudioFiProxy->sendTWNowServiceStatusUpReg(*this);
      _sxmAudioFiProxy->sendTWNowBulletinStatusUpReg(*this);
      _sxmAudioFiProxy->sendTWNowPlaybackStatusUpReg(*this);
   }
}


/**
 * Utility function that deregisters for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioTWNow::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAudioTWNow::deregisterProperties"));
   if ((proxy == _sxmAudioFiProxy) && (_sxmAudioFiProxy))
   {
      _sxmAudioFiProxy->sendTWNowServiceStatusRelUpRegAll();
      _sxmAudioFiProxy->sendTWNowBulletinStatusRelUpRegAll();
      _sxmAudioFiProxy->sendTWNowPlaybackStatusRelUpRegAll();
   }
}


/**
 * Error handling function for get TWNowBulletinPlayback Method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error - Error type
 */
void SxmAudioTWNow::onTWNowBulletinPlaybackError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< TWNowBulletinPlaybackError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmAudioTWNow::onTWNowBulletinPlaybackError"));
}


/**
 * Result handling function for get TWNowBulletinPlayback Method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioTWNow::onTWNowBulletinPlaybackResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< TWNowBulletinPlaybackResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioTWNow::onTWNowBulletinPlaybackResult"));
   if (result && result->getStatus())
   {
      SxmAudioDataBindingUtils* pAudioUtilsInstance = SxmAudioDataBindingUtils::instance();
      if (T_e8_TWNow_Bulletin_PlayBackAction__TWNOW_BULLETIN_ACTION_PLAY == result->getAction())
      {
         pAudioUtilsInstance->setTWNButtonStatus(TWNRED, true);
         pAudioUtilsInstance->setSxmAudioPresetControlData(CONTROL);
         pAudioUtilsInstance->setSxmTWNPlayStatus(true);
         _playbackStatus = T_e8_FlashEventPlaybackStatus__FLASHEVENT_PLAYING;
      }
      else //if bulletin is not available -- to check
      {
         pAudioUtilsInstance->setSxmAudioPresetControlData(PREV_MODE);
         pAudioUtilsInstance->setTWNButtonStatus(TWNWHITE, true);
         pAudioUtilsInstance->setSxmTWNPlayStatus(false);
         _playbackStatus = T_e8_FlashEventPlaybackStatus__FLASHEVENT_UNKNOWN;
      }
   }
}


/**
* Error handling function for get TWNowBulletinStatus Property
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : error - Error type
*/
void SxmAudioTWNow::onTWNowBulletinStatusError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< TWNowBulletinStatusError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmAudioTWNow::onTWNowBulletinStatusError"));
}


/**
* Status handling function for get TWNowBulletinStatus Property
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : error - Error type
*/
void SxmAudioTWNow::onTWNowBulletinStatusStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< TWNowBulletinStatusStatus >& status)
{
   ETG_TRACE_USR4(("SxmAudioTWNow::onTWNowBulletinStatusStatus"));
   if (status && status->hasBulletinStatus())
   {
      ETG_TRACE_USR4(("SxmAudioTWNow::onTWNowBulletinStatusStatus _bulletinStatus:%d", status->getBulletinStatus()));
      _bulletinStatus = status->getBulletinStatus();
      _bulletinId = status->getBulletinID();
      updateTWNBulletinStatus();
   }
}


/**
* Error handling function for get TWNowPlaybackStatus Property
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : error - Error type
*/
void SxmAudioTWNow::onTWNowPlaybackStatusError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< TWNowPlaybackStatusError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmAudioTWNow::onTWNowPlaybackStatusError"));
}


/**
* Status handling function for get TWNowPlaybackStatus Property
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : error - Error type
*/
void SxmAudioTWNow::onTWNowPlaybackStatusStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< TWNowPlaybackStatusStatus >& status)
{
   if (status && status->hasFlashEventPlayState())
   {
      ETG_TRACE_USR4(("SxmAudioTWNow::onTWNowPlaybackStatusStatus %d", status->getFlashEventPlayState()));
      _playbackStatus = status->getFlashEventPlayState();
      if (T_e8_FlashEventPlaybackStatus__FLASHEVENT_PLAYING == _playbackStatus)
      {
         SxmAudioDataBindingUtils::instance()->setSxmTWNPlayStatus(true);
      }
      else
      {
         SxmAudioDataBindingUtils::instance()->setSxmTWNPlayStatus(false);
      }
      updateTWNBulletinStatus();
   }
}


/**
* Error handling function for get TWNowEnable Method
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : error - Error type
*/
void SxmAudioTWNow::onTWNowEnableError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
                                       const ::boost::shared_ptr< TWNowEnableError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmAudioTWNow::onTWNowEnableError"));
}


/**
* Result handling function for get TWNowEnable Method
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : result - Method result data
*/
void SxmAudioTWNow::onTWNowEnableResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
                                        const ::boost::shared_ptr< TWNowEnableResult >& /*result*/)
{
   ETG_TRACE_USR4(("SxmAudioTWNow::onTWNowEnableResult"));
}


/**
* Error handling function for get TWNowGetActiveMarket Method
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : error - Error type
*/
void SxmAudioTWNow::onTWNowGetActiveMarketError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >&  /*proxy*/,
      const ::boost::shared_ptr< TWNowGetActiveMarketError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmAudioTWNow::onTWNowGetActiveMarketError"));
}


/**
* Result handling function for get TWNowGetActiveMarket Method
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : result - Method result data
*/
void SxmAudioTWNow::onTWNowGetActiveMarketResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >&  /*proxy*/,
      const ::boost::shared_ptr< TWNowGetActiveMarketResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioTWNow::onTWNowGetActiveMarketResult"));
   if (result && result->hasMarket())
   {
      _isMarketActive = result->getMarket().getBIsActiveMarket();
   }
   else
   {
      _isMarketActive = false;
   }
}


/**
* Error handling function for get TWNowGetAvailableMarkets Method
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : error - Error type
*/
void SxmAudioTWNow::onTWNowGetAvailableMarketsError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< TWNowGetAvailableMarketsError >& error)
{
   ETG_TRACE_USR4(("SxmAudioTWNow::onTWNowGetAvailableMarketsError"));
   if (error)
   {
      //SxmAudioDataBindingUtils::instance()->setTWNButtonStatus(TWNGRAY, false);
   }
}


/**
* Result handling function for get TWNowGetAvailableMarkets Method
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : result - Method result data
*/
void SxmAudioTWNow::onTWNowGetAvailableMarketsResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >&  /*proxy*/,
      const ::boost::shared_ptr< TWNowGetAvailableMarketsResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioTWNow::onTWNowGetAvailableMarketsResult"));
   if (result)
   {
      SxmAudioListHelper::instance()->setMarketList(result->getMarkets());
      tSharedPtrDataProvider dataProvider = SxmAudioListHelper::instance()->getSxmTWNowListDataProvider();
      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
      SxmAudioDataBindingUtils::instance()->setTWNAdvisory(true);
   }
}


/**
* Error handling function for get TWNowServiceStatus Property
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : error - Error type
*/
void SxmAudioTWNow::onTWNowServiceStatusError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< TWNowServiceStatusError >&  /*error*/)
{
   ETG_TRACE_USR4(("SxmAudioTWNow::onTWNowServiceStatusError"));
}


/**
* Status handling function for get TWNowServiceStatus Property
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : error - Error type
*/
void SxmAudioTWNow::onTWNowServiceStatusStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< TWNowServiceStatusStatus >& status)
{
   if (status && _sxmAudioFiProxy)
   {
      _twnServiceStatus = status->getServiceStatus();
      ETG_TRACE_USR4(("SxmAudioTWNow::onTWNowServiceStatusStatus _twnServiceStatus:%d", _twnServiceStatus));
      if (T_e8_FlashServiceStatus__FLASH_SERVICE_STARTED == _twnServiceStatus ||
            T_e8_FlashServiceStatus__FLASH_SERVICE_ALREADY_STARTED == _twnServiceStatus)
      {
         SxmAudioDataBindingUtils::instance()->setTWNButtonStatus(TWNGRAY, true);//To check
         _sxmAudioFiProxy->sendTWNowGetActiveMarketStart(*this);
         if (SXMCommonInterface::getActiveViewName(Enum_SURFACEID_MAIN_SURFACE_SXM) == Sxm_Scenes_SXM_RADIO__TWN_SETTINGS)
         {
            _sxmAudioFiProxy->sendTWNowGetAvailableMarketsStart(*this);
         }
      }
      else
      {
         //disable twn button,if service is not ready.
         SxmAudioDataBindingUtils::instance()->setTWNButtonStatus(TWNGRAY, false);
      }
   }
}


/**
* Error handling function for get TWNowSetMarket Method
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : error - Error type
*/
void SxmAudioTWNow::onTWNowSetMarketError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< TWNowSetMarketError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmAudioTWNow::onTWNowSetMarketError"));
}


/**
* Result handling function for get TTWNowSetMarket Method
* @param [in] : proxy - SXM Audio FI proxy class instance
* @param [in] : result - Method result data
*/
void SxmAudioTWNow::onTWNowSetMarketResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< TWNowSetMarketResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioTWNow::onTWNowSetMarketResult"));
   if (result && result->getStatus() && _sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendTWNowGetAvailableMarketsStart(*this);
   }
}


/**
 *Error handling function for get TWNowGetBulletinID Method
 *@param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] :  error - Error type
 */
void SxmAudioTWNow::onTWNowGetBulletinIDError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< TWNowGetBulletinIDError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmAudioTWNow::onTWNowGetBulletinIDError"));
}


/**
 *Result handling function for get TWNowGetBulletinID Method
 *@param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioTWNow::onTWNowGetBulletinIDResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< TWNowGetBulletinIDResult >& /*result*/)
{
   ETG_TRACE_USR4(("SxmAudioTWNow::onTWNowGetBulletinIDResult"));
}


/**
 * Callback to handle Courier Message SxmTWNSettingsReqMsg Request
 * @param [in] : Reference of Courier::SxmTWNSettingsReqMsg
 * @return     : True-When Message is processed.
*/
bool SxmAudioTWNow:: onCourierMessage(const SxmTWNSettingsReqMsg& /*oMsg*/)
{
   if ((T_e8_FlashServiceStatus__FLASH_SERVICE_STARTED == _twnServiceStatus ||
         T_e8_FlashServiceStatus__FLASH_SERVICE_ALREADY_STARTED == _twnServiceStatus) && _sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendTWNowGetAvailableMarketsStart(*this);
   }
   else
   {
      //display loading.
      SxmAudioDataBindingUtils::instance()->setTWNAdvisory(false);
      POST_MSG((COURIER_MESSAGE_NEW(SxmSceneTransitionReqMsg)(Sxm_Scenes_SXM_RADIO__TWN_SETTINGS)));
   }
   return true;
};


/**
 * Callback to handle Courier Message SxmTWNButtonPressReqMsg Request
 * @param [in] : Reference of SxmTWNButtonPressReqMsg
 * @return     : bool true if message is consumed
 */
bool SxmAudioTWNow::onCourierMessage(const SxmTWNButtonPressReqMsg& oMsg)
{
   _buttonColor = oMsg.GetBtnColor();
   if (SXM_AUDIO_SUBSCRIBED != _subStatus)
   {
      SxmAudioDataBindingUtils::instance()->setTWNButtonStatus(TWNGRAY, true);
      POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__TPOP_FAILURE_CHANNEL_NOT_SUBSCRIBED)));
   }
   else if (!_isMarketActive)
   {
      //No active market set.
      POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__ICPOP_TWN_DEFINE_CITY)));
   }
   else if (T_e8_TWNow_Bulletin_Status__TWNOW_NO_BULLETIN_AVAILABLE == _bulletinStatus)
   {
      SxmAudioDataBindingUtils::instance()->setTWNTpopText();
      POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__TPOP_TWN_NO_INFO)));
   }
   else if ((TWNGREEN == _buttonColor) || (TWNWHITE == _buttonColor))
   {
      if (_sxmAudioFiProxy)
      {
         _sxmAudioFiProxy->sendTWNowBulletinPlaybackStart(*this, _bulletinId, T_e8_TWNow_Bulletin_PlayBackAction__TWNOW_BULLETIN_ACTION_PLAY);
      }
   }
   else if (TWNRED == _buttonColor)
   {
      if (_sxmAudioFiProxy)
      {
         _sxmAudioFiProxy->sendTWNowBulletinPlaybackStart(*this, _bulletinId, T_e8_TWNow_Bulletin_PlayBackAction__TWNOW_BULLETIN_ACTION_ABORT);
      }
   }
   return true;
}


/**
 * Callback to handle Courier Message SXMAudioSportsRefreshSFStatusReqMsg Request
 * @param [in] : Reference of SXMAudioSportsRefreshSFStatusReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioTWNow::onCourierMessage(const SXMAudioSportsRefreshSFStatusReqMsg& /*oMsg*/)
{
   //_subStatus = static_cast<SXMAudio_status>(oMsg.GetSubStatus());
   return true;
}


/**
 * Callback to handle Courier Message SxmTWNListSelectReqMsg Request
 * @param [in] : Reference of Courier::SxmTWNListSelectReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioTWNow::onCourierMessage(const SxmTWNListSelectReqMsg& oMsg)
{
   std::string marketname = "";
   bool isactive;
   if (_sxmAudioFiProxy && SxmAudioListHelper::instance()->getSelectedMarketInfo(oMsg.GetHdl(), marketname, isactive))
   {
      _isMarketActive = !isactive;
      _sxmAudioFiProxy->sendTWNowSetMarketStart(*this, _isMarketActive, marketname);
   }
   return true;
}


/**
 * Helper function to update bulletin status
 */
void SxmAudioTWNow::updateTWNBulletinStatus()
{
   uint8 u8Color = TWNGRAY;
   SxmAudioDataBindingUtils* pAudioUtilsInstance = SxmAudioDataBindingUtils::instance();
   if (T_e8_FlashServiceStatus__FLASH_SERVICE_STARTED == _twnServiceStatus ||
         T_e8_FlashServiceStatus__FLASH_SERVICE_ALREADY_STARTED == _twnServiceStatus)
   {
      if (SXM_AUDIO_SUBSCRIBED != _subStatus)
      {
         pAudioUtilsInstance->setTWNButtonStatus(TWNGRAY, true);
      }
      else if (T_e8_TWNow_Bulletin_Status__TWNOW_NO_BULLETIN_AVAILABLE == _bulletinStatus)
      {
         u8Color = TWNGRAY;
      }
      else
      {
         if (T_e8_TWNow_Bulletin_Status__TWNOW_NEW_BULLETIN_AVAILABLE == _bulletinStatus)
         {
            u8Color = TWNGREEN;
         }
         else if (T_e8_TWNow_Bulletin_Status__TWNOW_PLAYED_BULLETIN_AVAILABLE == _bulletinStatus)
         {
            u8Color = TWNWHITE;
         }
         if (T_e8_FlashEventPlaybackStatus__FLASHEVENT_PLAYING == _playbackStatus)
         {
            u8Color = TWNRED;
         }
         else
         {
            pAudioUtilsInstance->setDefaultSxmAudioPlayPauseData(T_e8_IRPlaybackStatus__SXM_AUDIO_IR_PLAY_STATE);
            if ((T_e8_FlashEventPlaybackStatus__FLASHEVENT_STOPPED == _playbackStatus) &&
                  ((!(pAudioUtilsInstance->getSxmTuneScanStatus() || pAudioUtilsInstance->getSxmSFPlayStatus()))))
            {
               pAudioUtilsInstance->setSxmAudioPresetControlData(PREV_MODE);
            }
            _playbackStatus = T_e8_FlashEventPlaybackStatus__FLASHEVENT_UNKNOWN;
         }
      }
      pAudioUtilsInstance->setTWNButtonStatus(u8Color, true);
   }
   else
   {
      pAudioUtilsInstance->setTWNButtonStatus(TWNGRAY, false);
   }
}


/**
 * Helper Function to restore to factory default settings.
 */
void SxmAudioTWNow::restoreAudioTWNInfo(void)
{
   ETG_TRACE_USR4(("SxmAudioTWNow::restoreAudioTWNInfo"));
   _isMarketActive = false;
   _bulletinStatus = T_e8_TWNow_Bulletin_Status__TWNOW_NO_BULLETIN_AVAILABLE;
   _playbackStatus = T_e8_FlashEventPlaybackStatus__FLASHEVENT_UNKNOWN;
   _bulletinId = 0;
   SxmAudioListHelper::instance()->setMarketList(::std::vector< T_MarketListEntry >());//clear list
   SxmAudioDataBindingUtils::instance()->setTWNButtonStatus(TWNGRAY, false);
}


}  // namespace Core
}  // namespace App
