/**************************************************************************************
 * @file         : SxmAudioSportsFlash.cpp
 * @addtogroup   : AppHmi_Sxm
 * @brief        : Functional class to handle business logic for SportsFlash.
 * @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 **************************************************************************************/
#include "hall_std_if.h"   // Including HALL framework code
#include "SxmAudioSportsFlash.h"
#include "SxmAudioListHelper.h"
#include "SxmAudioDataBindingUtils.h"
#include "App/Core/ClientHandler/DataCollector/SXMDataCollectorInterface.h"
#include "App/Core/SxmUtils/SxmUtils.h"
#include "App/Core/SxmCommon/SXMCommonInterface.h"
#include "App/Core/ClientHandler/SourceSwitch/SXMSourceSwitchInterface.h"
//#include "HmiTranslation_TextIds.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "App/Core/LanguageDefines.h"
#include "App/Core/SxmCommon/SxmHMIServiceStateHandler.h"
#include "App/Core/ClientHandler/ApplicationSwitchClientHandler/ApplicationSwitchClientHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXMAUDIO
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::SxmAudioSportsFlash::
#include "trcGenProj/Header/SxmAudioSportsFlash.cpp.trc.h"
#endif

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_hmi_06_if.h"

namespace App {
namespace Core {
using namespace ::sxm_audio_main_fi;
using namespace ::sxm_audio_main_fi_types;

/**
 * SxmAudioSportsFlash Class constructor
 * @param [in] : pSxmAudioFiProxy - SXM Audio FI proxy
 */
SxmAudioSportsFlash::SxmAudioSportsFlash(::boost::shared_ptr< Sxm_audio_main_fiProxy > pSxmAudioFiProxy,
      ISxmAudioChannelArtRequest* pSxmChannelart)
   : _sxmAudioFiProxy(pSxmAudioFiProxy)
   , _sxmChannelart(pSxmChannelart)
   , _selectedTeamId(0)
   , _selectedLeagueId(0)
   , _activeFlasheventId(0)
   , _selectedTeamName("")
   , _subStatus(SXM_AUDIO_SUBSCRIBED)
   , _replaceFavorite(false)
   , _isTeamAvailForMonitor(false)
   , _isActiveGamesAvailForMonitorTeam(false)
   , _isTeamAddedOnSFSelection(false)
   , _doTransition(false)
   , _isContextChangeRequested(false)
   , _isSFReturnSkPressed(false)
   , _isSourceSwitchRequested(false)
   , _isSFActivePopupVisible(false)
   , _allowSFAlert(true)
   , _popupCloseOnAppmodeChange(false)
   , _sfServiceStatus(T_e8_FlashServiceStatus__FLASH_SERVICE_ERROR)
   , _sportsType(T_e8_SportsFlash_Sport__SPORTS_UNKNOWN)
   , _playbackStatus(T_e8_FlashEventPlaybackStatus__FLASHEVENT_UNKNOWN)
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("SxmAudioSportsFlash() Constructor"));
//#include "Common/ListHandler/LockoutRestrictionData.h"
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_SPORTSFLASH_LEAGUELIST, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_SPORTSFLASH_TEAMLIST_LEAGUE, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_SPORTSFLASH_FAV_REPLACETEAM, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_SPORTSFLASH_ALERT_LIST, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_AUDIO_SPORTSFLASH_FAV_TEAMS, this);
   _favoriteTeams.clear();
   _activeSFEvents.clear();
   _SportsAlertList.clear();

   SXMCommonInterface::registerApplicationModeChangeNotication(this, SXMSportsFlash);

   //initialise values from datapool
   readFromSFDatapool();
   _noOfFavorites = _favoriteTeams.size();
   updateFavDatapool();
}


/**
 * SxmAudioSportsFlash Class destructor
 */
SxmAudioSportsFlash::~SxmAudioSportsFlash()
{
   ETG_TRACE_USR4(("~SxmAudioSportsFlash() Destructor"));
   ETG_I_UNREGISTER_FILE();
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_SPORTSFLASH_LEAGUELIST);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_SPORTSFLASH_TEAMLIST_LEAGUE);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_SPORTSFLASH_FAV_REPLACETEAM);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_SPORTSFLASH_ALERT_LIST);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_AUDIO_SPORTSFLASH_FAV_TEAMS);
   SXMCommonInterface::deregisterApplicationModeChangeNotication(this, SXMSportsFlash);
   _sxmAudioFiProxy.reset();
   _favoriteTeams.clear();
   _activeSFEvents.clear();
   _SportsAlertList.clear();
   _sxmChannelart = NULL;
   _selectedTeamId = 0;
   _selectedLeagueId = 0;
   _activeFlasheventId = 0;
   _selectedTeamName = "";
   _noOfFavorites = 0;
   _subStatus = SXM_AUDIO_SUBSCRIBED;
   _replaceFavorite = false;
   _isTeamAvailForMonitor = false;
   _isActiveGamesAvailForMonitorTeam = false;
   _isTeamAddedOnSFSelection = false;
   _doTransition = false;
   _isContextChangeRequested = false;
   _isSFReturnSkPressed = false;
   _isSourceSwitchRequested = false;
   _isSFActivePopupVisible = false;
   _allowSFAlert = true;
   _popupCloseOnAppmodeChange = false;
   _sfServiceStatus = T_e8_FlashServiceStatus__FLASH_SERVICE_ERROR;
   _sportsType = T_e8_SportsFlash_Sport__SPORTS_UNKNOWN;
   _playbackStatus = T_e8_FlashEventPlaybackStatus__FLASHEVENT_UNKNOWN;
}


/**
 * Utility function that registers for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioSportsFlash::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::registerProperties"));
   if (_sxmAudioFiProxy && (proxy == _sxmAudioFiProxy))
   {
      _sxmAudioFiProxy->sendSportsFlash_FlashEventNotificationUpReg(*this);
      _sxmAudioFiProxy->sendSportsFlash_ServiceStatusUpReg(*this);
      _sxmAudioFiProxy->sendSportsFlash_GameEventNotificationUpReg(*this);
      _sxmAudioFiProxy->sendSportsFlash_PlaybackStatusUpReg(*this);
   }
}


/**
 * Utility function that deregisters for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioSportsFlash::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::deregisterProperties"));
   if (_sxmAudioFiProxy && (proxy == _sxmAudioFiProxy))
   {
      _sxmAudioFiProxy->sendSportsFlash_FlashEventNotificationRelUpRegAll();
      _sxmAudioFiProxy->sendSportsFlash_ServiceStatusRelUpRegAll();
      _sxmAudioFiProxy->sendSportsFlash_GameEventNotificationRelUpRegAll();
      _sxmAudioFiProxy->sendSportsFlash_PlaybackStatusRelUpRegAll();
   }
}


/**
 * Error handling function for alert notification.
 */
void SxmAudioSportsFlash::onSportsFlash_FlashEventNotificationError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SportsFlash_FlashEventNotificationError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioSportsFlash::onSportsFlash_FlashEventNotificationError"));
}


/**
 * Status handling function for alert notification.
 */
void SxmAudioSportsFlash::onSportsFlash_FlashEventNotificationStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SportsFlash_FlashEventNotificationStatus >& status)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::onSportsFlash_FlashEventNotificationStatus"));
   if (status)
   {
      uint16 ChannelId = status->getGameInfo().getChannelID();
      ETG_TRACE_USR4(("SxmAudioSportsFlash::onSportsFlash_FlashEventNotificationStatus ChannelId %d", ChannelId));
      //Check to avoid status notification received for property registration.
      if ((ChannelId <= END_CHANNEL_ID) && (ChannelId >= START_CHANNEL_ID))
      {
         if (_sxmAudioFiProxy && SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__ALERT_SPORTSFLASHLIST))
         {
            _sxmAudioFiProxy->sendSportsFlashGetCurrentFlashEventsStart(*this);
         }
         handleSportFlashNotification(status);
      }
   }
}


/**
 * Error handling function for alert notification.
 */
void SxmAudioSportsFlash::onSportsFlash_PlaybackStatusError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SportsFlash_PlaybackStatusError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioSportsFlash::onSportsFlash_PlaybackStatusError"));
}


/**
 * Status handling function for alert notification.
 */
void SxmAudioSportsFlash::onSportsFlash_PlaybackStatusStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SportsFlash_PlaybackStatusStatus >& status)
{
   if (status && status->hasFlashEventPlayState())
   {
      ETG_TRACE_USR4(("SxmAudioSportsFlash::onSportsFlash_PlaybackStatusStatus %d, _isSourceSwitchRequested:%d, _isContextChangeRequested:%d, _noOfFavorites:%d",
                      status->getFlashEventPlayState(), _isSourceSwitchRequested, _isContextChangeRequested, _noOfFavorites));
      _playbackStatus = status->getFlashEventPlayState();
      SxmAudioDataBindingUtils* pAudioUtilsInstance = SxmAudioDataBindingUtils::instance();
      if (_noOfFavorites != 0)
      {
         if (T_e8_FlashEventPlaybackStatus__FLASHEVENT_STOPPED == _playbackStatus)
         {
            //When playback is stopped close pop up and switch source back.
            POST_MSG((COURIER_MESSAGE_NEW(SXMDeActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__ITPOP_SPORTSFLASH_RETURN)));
            if (!(pAudioUtilsInstance->getSxmTWNPlayStatus() || pAudioUtilsInstance->getSxmTuneScanStatus()))
            {
               pAudioUtilsInstance->setSxmAudioPresetControlData(PREV_MODE);
            }
            if (!_isSFReturnSkPressed)
            {
               //Deactivate SRC_SXM_NEWS
               ETG_TRACE_USR1(("SxmAudioSportsFlash::onSportsFlash_PlaybackStatusStatus"));
               SXMSourceSwitchInterface::handleSourceChangeRequest(SRC_SXM_NEWS, false, false, true);
            }
            pAudioUtilsInstance->setDefaultSxmAudioPlayPauseData(T_e8_IRPlaybackStatus__SXM_AUDIO_IR_PLAY_STATE);
            pAudioUtilsInstance->setSxmSFPlayStatus(false);
         }
         else if (T_e8_FlashEventPlaybackStatus__FLASHEVENT_PLAYING == _playbackStatus)
         {
            pAudioUtilsInstance->setSxmSFPlayStatus(true);
         }
         _isSFReturnSkPressed = false;
         updateSFIconStatus();
      }
   }
}


/**
 * Error handling function for league list.
 */
void SxmAudioSportsFlash::onSportsFlashEnabledLeaguesError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SportsFlashEnabledLeaguesError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioSportsFlash::onSportsFlashEnabledLeaguesError"));
}


/**
 * Result handling function for league list.
 */
void SxmAudioSportsFlash::onSportsFlashEnabledLeaguesResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SportsFlashEnabledLeaguesResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::onSportsFlashEnabledLeaguesResult"));
   if (result)
   {
      ETG_TRACE_USR4(("SxmAudioSportsFlash::onSportsFlashEnabledLeaguesResult size:%d _doTransition:%d", result->getLeagues().size(), _doTransition));
      SxmAudioDataBindingUtils::instance()->setSFAdvisory(true);//Remove loading state.
      SxmAudioListHelper::instance()->setSxmSportsFlashLeagueList(result->getLeagues());
      if (SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__SPORTSFLASH_SETTINGS_SPORTS))
      {
         //list is updated to remove loading state.
         updateListInfo(LIST_ID_SXM_SPORTSFLASH_LEAGUELIST);
      }
      else if (_doTransition)
      {
         POST_MSG((COURIER_MESSAGE_NEW(SxmSceneTransitionReqMsg)(Sxm_Scenes_SXM_RADIO__SPORTSFLASH_SETTINGS_SPORTS)));
         _doTransition = false;
      }
   }
}


/**
 * Error handling function for team list for a league.
 */
void SxmAudioSportsFlash::onSportsFlashGetTeamsError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SportsFlashGetTeamsError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioSportsFlash::onSportsFlashGetTeamsError"));
}


/**
 * Result handling function for team list for a league.
 */
void SxmAudioSportsFlash::onSportsFlashGetTeamsResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SportsFlashGetTeamsResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::onSportsFlashGetTeamsResult size:%d", result->getTeams().size()));
   if (result)
   {
      SxmAudioListHelper::instance()->setSxmSportsFlashTeamList(result->getTeams());
      if (SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__SPORTSFLASH_SETTINGS_TEAMS))
      {
         //list is updated when team is selected.
         updateListInfo(LIST_ID_SXM_SPORTSFLASH_TEAMLIST_LEAGUE);
      }
      else if (_doTransition)
      {
         POST_MSG((COURIER_MESSAGE_NEW(SxmSceneTransitionReqMsg)(Sxm_Scenes_SXM_RADIO__SPORTSFLASH_SETTINGS_TEAMS)));
         _doTransition = false;
      }
      if (_sxmAudioFiProxy)
      {
         //To get updated favorite list.
         _sxmAudioFiProxy->sendSportsFlashGetAllFavoritesStart(*this);
      }
   }
}


/**
 * Error handling function for getting favorites team list.
 */
void SxmAudioSportsFlash::onSportsFlashGetAllFavoritesError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SportsFlashGetAllFavoritesError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioSportsFlash::onSportsFlashGetAllFavoritesError"));
}


/**
 * Result handling function for getting favorites team list.
 */
void SxmAudioSportsFlash::onSportsFlashGetAllFavoritesResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SportsFlashGetAllFavoritesResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::onSportsFlashGetAllFavoritesResult listsize = %d", result->getFavouritesList().size()));
   if (result && result->hasFavouritesList())
   {
      _noOfFavorites = result->getFavouritesList().size();
      SxmAudioListHelper::instance()->setSxmSportsFlashFavoriteList(result->getFavouritesList());
      updateFavTeams();
      updateFavDatapool();
      updateAdvisoryForSF();
      if (_doTransition)
      {
         POST_MSG((COURIER_MESSAGE_NEW(SxmSceneTransitionReqMsg)(Sxm_Scenes_SXM_RADIO__EDIT_SPORTSFLASH_TEAMS)));
         _doTransition = false;
      }
      if (_isTeamAddedOnSFSelection && _noOfFavorites && _isTeamAvailForMonitor && _sxmAudioFiProxy)
      {
         //When user adds or starts monitoring a favorite on selection of SF button
         //from audio main screen, enable notification and display pop up.
         //postPopupMessageForSF(Sxm_SxmPopups_SXM_RADIO__TPOP_SPORTSFLASH_MONITORING);
         _sxmAudioFiProxy->sendSportsFlashEnableNotificationStart(*this, true);
         updateSFStatus(true);
         _isTeamAddedOnSFSelection = false;
      }
      updateSFIconStatus();
   }
}


/**
 * Error handling function for adding team to favorites team list.
 */
void SxmAudioSportsFlash::onSportsFlashAddFavoriteTeamError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SportsFlashAddFavoriteTeamError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioSportsFlash::onSportsFlashAddFavoriteTeamError"));
   --_noOfFavorites;
}


/**
 * Result handling function for adding team to favorites team list.
 */
void SxmAudioSportsFlash::onSportsFlashAddFavoriteTeamResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SportsFlashAddFavoriteTeamResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::onSportsFlashAddFavoriteTeamResult"));
   if (_sxmAudioFiProxy && result->getStatus())
   {
      _sxmAudioFiProxy->sendSportsFlashGetTeamsStart(*this, _sportsType, _selectedLeagueId);
      POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__TPOP_SPORTSFLASH_TEAM_SELECTED))); //To do:Use common message.
      SxmAudioDataBindingUtils::instance()->setsTeamNameForPopup(_selectedTeamName);//Map Text id.
   }
}


/**
 * Error handling function for getting alert list.
 */
void SxmAudioSportsFlash::onSportsFlashGetCurrentFlashEventsError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SportsFlashGetCurrentFlashEventsError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioSportsFlash::onSportsFlashGetCurrentFlashEventsError"));
}


/**
 * Result handling function for getting alert list.
 */
void SxmAudioSportsFlash::onSportsFlashGetCurrentFlashEventsResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SportsFlashGetCurrentFlashEventsResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::onSportsFlashGetCurrentFlashEventsResult %d", result->getFlashEvents().size()));
   if (result)
   {
      SxmAudioListHelper* pAudioListHelperUtilsInstance = SxmAudioListHelper::instance(); /* Get class instance */
      pAudioListHelperUtilsInstance->setSxmSportsFlashAlertList(result->getFlashEvents());
      if (result->getFlashEvents().size() == 0)
      {
         _SportsAlertList.clear();
         std::string strAdvisory = "";
         size_t SportsFlashList = pAudioListHelperUtilsInstance->getSxmSportsFlashFavoriteList().size();
         strAdvisory = getSxmSportsFlashAlertListAdvisory(!SportsFlashList);
         //To Check:Why common call bw content alerts.
         SxmAudioDataBindingUtils::instance()->setAlertsAdvisory(false, strAdvisory);
         if (_doTransition)
         {
            POST_MSG((COURIER_MESSAGE_NEW(SxmSceneTransitionReqMsg)(Sxm_Scenes_SXM_RADIO__ALERT_SPORTSFLASHLIST)));
            _doTransition = false;
         }
      }
      else
      {
         _SportsAlertList.clear();
         SxmAudioDataBindingUtils::instance()->setAlertsAdvisory(true, "");
         size_t flashEventsSize = result->getFlashEvents().size();
         for (uint32 index = 0; index < flashEventsSize; ++index)
         {
            _SportsAlertList.push_back(result->getFlashEvents()[index].getGameInfo().getServiceID());
         }
         if (_sxmChannelart)
         {
            _sxmChannelart->sendChannelArtListMethodRequest(this, _SportsAlertList, 1);//1:Only channel logo
         }
      }
   }
}


/**
 * Error handling function for removing team from favorites team list.
 */
void SxmAudioSportsFlash::onSportsFlashRemoveFavoriteTeamError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SportsFlashRemoveFavoriteTeamError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioSportsFlash::onSportsFlashRemoveFavoriteTeamError"));
}


/**
 * Result handling function for removing team from favorites team list.
 */
void SxmAudioSportsFlash::onSportsFlashRemoveFavoriteTeamResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SportsFlashRemoveFavoriteTeamResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::onSportsFlashRemoveFavoriteTeamResult _noOfFavorites:%d", _noOfFavorites));
   if (result && result->getStatus() && _sxmAudioFiProxy)
   {
      --_noOfFavorites;
      if (_replaceFavorite && _noOfFavorites < SXM_AUDIO_SPORTSFLASH_MAX_ENTRY)
      {
         _replaceFavorite = false;
         ++_noOfFavorites;
         _sxmAudioFiProxy->sendSportsFlashAddFavoriteTeamStart(*this, _sportsType, _selectedLeagueId, _selectedTeamId);
      }
      else
      {
         _sxmAudioFiProxy->sendSportsFlashGetAllFavoritesStart(*this);
         POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__TPOP_DELETE_FAVORITE_TEAM)));
      }
      ETG_TRACE_USR4(("SxmAudioSportsFlash::onSportsFlashRemoveFavoriteTeamResult _noOfFavorites:%d", _noOfFavorites));
   }
   else
   {
      ETG_TRACE_ERR(("Team not removed from Fav"));
   }
}


/**
 * Error handling function for enabling the notifications.
 */
void SxmAudioSportsFlash::onSportsFlashEnableNotificationError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SportsFlashEnableNotificationError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioSportsFlash::onSportsFlashEnableNotificationError"));
}


/**
 * Result handling function for enabling the notifications.
 */
void SxmAudioSportsFlash::onSportsFlashEnableNotificationResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SportsFlashEnableNotificationResult >& result)
{
   if (result->hasStatus())
   {
      ETG_TRACE_USR4(("SxmAudioSportsFlash::onSportsFlashEnableNotificationResult Status = %d", result->getStatus()));
      updateSFIconStatus();
   }
}


/**
 * Error handling function for abort channel method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error - Error type
 */
void SxmAudioSportsFlash::onSportsFlashAbortFlashEventError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy,*/,
      const ::boost::shared_ptr< SportsFlashAbortFlashEventError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioSportsFlash::onSportsFlashAbortFlashEventError"));
}


/**
 * Result handling function for abort channel method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioSportsFlash::onSportsFlashAbortFlashEventResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr<  SportsFlashAbortFlashEventResult >& result)
{
   ETG_TRACE_USR1(("SxmAudioList::onSportsFlashAbortFlashEventResult"));
   if (result && result->hasStatus() && result->getStatus())
   {
      SxmAudioDataBindingUtils* pAudioUtilsInstance = SxmAudioDataBindingUtils::instance();
      pAudioUtilsInstance->setSxmAudioPresetControlData(PREV_MODE);
      pAudioUtilsInstance->setSxmSFPlayStatus(false);
      _isSourceSwitchRequested = false;
      if (_isContextChangeRequested)//When SF in active region
      {
         ApplicationSwitchClientHandler::instance()->returnToPreviousSource();
      }
   }
}


/**
 * Error handling function for play Flash Event method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioSportsFlash::onSportsFlashPlayFlashEventError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< SportsFlashPlayFlashEventError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioSportsFlash::onSportsFlashPlayFlashEventError"));
}


/**
 * Result handling function for play Flash Event method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioSportsFlash::onSportsFlashPlayFlashEventResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< SportsFlashPlayFlashEventResult >& result)
{
   ETG_TRACE_USR1(("SxmAudioSportsFlash::onSportsFlashPlayFlashEventResult"));
   if (result && result->getStatus())
   {
      SxmAudioDataBindingUtils* pAudioUtilsInstance = SxmAudioDataBindingUtils::instance();
      pAudioUtilsInstance->setSxmAudioPresetControlData(CONTROL);
      pAudioUtilsInstance->setSxmSFPlayStatus(true);
      SXMSourceSwitchInterface::handleSourceChangeRequest(SRC_SXM_NEWS, true, false, true);
   }
}


/**
 * Error handling function for play Flash Monitor Teams method.
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioSportsFlash::onSportsFlashMonitorTeamsError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< SportsFlashMonitorTeamsError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioSportsFlash::onSportsFlashMonitorTeamsError"));
}


/**
 * Result handling function for play Flash  Monitor Teams method.
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioSportsFlash::onSportsFlashMonitorTeamsResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< SportsFlashMonitorTeamsResult >& result)
{
   ETG_TRACE_USR1(("SxmAudioSportsFlash::onSportsFlashMonitorTeamsResult"));
   if (result && result->getStatus() && _sxmAudioFiProxy)
   {
      //Get favorite list when monitoring status for team is changed.
      _sxmAudioFiProxy->sendSportsFlashGetAllFavoritesStart(*this);
   }
}


/**
 * Error handling function for Service Status property.
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioSportsFlash::onSportsFlash_ServiceStatusError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< SportsFlash_ServiceStatusError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioSportsFlash::onSportsFlash_ServiceStatusError"));
}


/**
 * Result handling function for play Service Status property.
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - Property result data
 */
void SxmAudioSportsFlash::onSportsFlash_ServiceStatusStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< SportsFlash_ServiceStatusStatus >& status)
{
   _sfServiceStatus = status->getServiceStatus();
   ETG_TRACE_USR1(("SxmAudioSportsFlash::onSportsFlash_ServiceStatusStatus %d", _sfServiceStatus));
   if ((T_e8_FlashServiceStatus__FLASH_SERVICE_STARTED == _sfServiceStatus) ||
         (T_e8_FlashServiceStatus__FLASH_SERVICE_ALREADY_STARTED == _sfServiceStatus))
   {
      SxmAudioDataBindingUtils::instance()->setSFButtonStatus(SFGREY, true);
      if (_sxmAudioFiProxy && _favoriteTeams.size())
      {
         _sxmAudioFiProxy->sendSportsFlashMonitorTeamsStart(*this, _favoriteTeams);
         _sxmAudioFiProxy->sendSportsFlashGetCurrentGamesStart(*this);
      }
      if (_sxmAudioFiProxy)
      {
         //Send Method start to enable/ disable sports flash notification
         _sxmAudioFiProxy->sendSportsFlashEnableNotificationStart(*this, getSFStatus());
         _sxmAudioFiProxy->sendSportsFlashGetAllFavoritesStart(*this);
      }
   }
   else if (T_e8_FlashServiceStatus__FLASH_SERVICE_ERROR == _sfServiceStatus)
   {
      handleBroadScopeChangesforSF();
   }
   else
   {
      //disable button.
      SxmAudioDataBindingUtils::instance()->setSFButtonStatus(SFGREY, false);
   }
}


/**
 * Error handling function for SportsFlashGetCurrent Method request.
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioSportsFlash::onSportsFlashGetCurrentGamesError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< SportsFlashGetCurrentGamesError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioSportsFlash::onSportsFlashGetCurrentGamesError"));
}


/*
* Result handling function for Current Games method
* If any of the favorite team is playing a game and it is not monitored, change the color to white.
* If any of monitor fav team is playing a game, change color to green.
*/
void SxmAudioSportsFlash::onSportsFlashGetCurrentGamesResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< SportsFlashGetCurrentGamesResult >& result)
{
   _isActiveGamesAvailForMonitorTeam = false;
   if (result->getGames().size())
   {
      ::std::vector< T_SportsFlash_Game>::const_iterator itr;
      for (itr = result->getGames().begin(); itr != result->getGames().end(); ++itr)
      {
         if (((itr->getAwayTeam().getIsFavorite()) && (itr->getAwayTeam().getIsMonitored())) ||
               ((itr->getHomeTeam().getIsFavorite()) && (itr->getHomeTeam().getIsMonitored())))
         {
            _isActiveGamesAvailForMonitorTeam = true;
            break;
         }
      }
   }
   ETG_TRACE_USR1(("SxmAudioSportsFlash::onSportsFlashGetCurrentGamesResult _isActiveGamesAvailForMonitorTeam:%d", _isActiveGamesAvailForMonitorTeam));
   updateSFIconStatus();
}


/**
 * Error handling function for SportsFlash_GameEventNotification Property.
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error - Failure message
 */
void SxmAudioSportsFlash::onSportsFlash_GameEventNotificationError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< SportsFlash_GameEventNotificationError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioSportsFlash::onSportsFlash_GameEventNotificationError"));
}


/**
 * Status handling function for SportsFlash_GameEventNotification Property.
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - status message
 */
void SxmAudioSportsFlash::onSportsFlash_GameEventNotificationStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< SportsFlash_GameEventNotificationStatus >& /*status*/)
{
   ETG_TRACE_USR1(("SxmAudioSportsFlash::onSportsFlash_GameEventNotificationStatus"));
   if (_sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendSportsFlashGetCurrentGamesStart(*this);
   }
}


/**
 * result handling function for channelart method
 * @param [in] : channelArtList - channel art list.
 */
void SxmAudioSportsFlash::onGetChannelArtListResult(const ::std::vector< ChannelGraphicsType >& channelArtList)
{
   ETG_TRACE_USR1(("SxmAudioSportsFlash::onGetChannelArtListResult"));
   SxmAudioListHelper::instance()->setLogoForSFAlertList(channelArtList);
   if (_doTransition)
   {
      POST_MSG((COURIER_MESSAGE_NEW(SxmSceneTransitionReqMsg)(Sxm_Scenes_SXM_RADIO__ALERT_SPORTSFLASHLIST)));
      _doTransition = false;
   }
   else
   {
      updateListInfo(LIST_ID_SXM_SPORTSFLASH_ALERT_LIST);
   }
}


/**
 * Helper Function to Restore datapool values
 */
void SxmAudioSportsFlash::restoreAudioSFInfo(bool isRestore)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash: restoreAudioSFInfo isRestore:%d", isRestore));
   _playbackStatus = T_e8_FlashEventPlaybackStatus__FLASHEVENT_UNKNOWN;
   _noOfFavorites = 0;
   _isTeamAvailForMonitor = false;
   _isActiveGamesAvailForMonitorTeam = false;
   _favoriteTeams.clear();
   _activeSFEvents.clear();
   updateFavDatapool();
   if (isRestore)
   {
      updateSFStatus(false);
   }
}


/**
 * Callback to handle Courier Message SXMAudioSportsLeagueListReqMsg Request
 */
bool SxmAudioSportsFlash::onCourierMessage(const SXMAudioSportsLeagueListReqMsg& /*oMsg*/)
{
   ETG_TRACE_USR1(("SXMAudioSportsLeagueListReqMsg _sfServiceStatus:%d", _sfServiceStatus));
   if ((T_e8_FlashServiceStatus__FLASH_SERVICE_STARTED == _sfServiceStatus ||
         T_e8_FlashServiceStatus__FLASH_SERVICE_ALREADY_STARTED == _sfServiceStatus) && _sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendSportsFlashEnabledLeaguesStart(*this);
      _doTransition = true;
   }
   else
   {
      //display loading.
      SxmAudioDataBindingUtils::instance()->setSFAdvisory(false);
      POST_MSG((COURIER_MESSAGE_NEW(SxmSceneTransitionReqMsg)(Sxm_Scenes_SXM_RADIO__SPORTSFLASH_SETTINGS_SPORTS)));
   }
   return true;
}


/**
 * Callback to handle Courier Message SXMAudioSportsTeamListForLeagueReqMsg Request
 */
bool SxmAudioSportsFlash::onCourierMessage(const SXMAudioSportsTeamListForLeagueReqMsg& oMsg)
{
   ::std::vector< T_SportsFlash_League > lSportsLeagueList =
      SxmAudioListHelper::instance()->getSxmSportsFlashLeagueList();
   uint32 rHdl = oMsg.GetHdl();
   if (rHdl < lSportsLeagueList.size())
   {
      _selectedLeagueId = lSportsLeagueList[rHdl].getLeagueID();
      _sportsType = lSportsLeagueList[rHdl].getSportType();
      ETG_TRACE_USR4(("SXMAudioSportsTeamListForLeagueReqMsg _sportsType:%d, _selectedLeagueId:%d", _sportsType, _selectedLeagueId));

      std::string lSportsTypeStr = SxmAudioListHelper::instance()->getSportsName(_sportsType);
      std::string lHeaderText = SxmUtils::conCatData("sss", 3, lSportsTypeStr.c_str(), " - ",
                                lSportsLeagueList[rHdl].getLeagueAbbrName().c_str());

      SXMDataCollectorInterface::vSetSXMDynamicData(lHeaderText);
      if (_sxmAudioFiProxy)
      {
         _sxmAudioFiProxy->sendSportsFlashGetTeamsStart(*this, _sportsType, _selectedLeagueId);
         _doTransition = true;
      }
   }
   return true;
}


/**
 * Callback to handle Courier Message SXMAudioSportsOnPressTeamNameReqMsg Request
 */
bool SxmAudioSportsFlash::onCourierMessage(const SXMAudioSportsOnPressTeamNameReqMsg& oMsg) //check for max fav and add a popup
{
   ::std::vector< T_SportsFlash_Team > lTeamId =
      SxmAudioListHelper::instance()->getSxmSportsFlashTeamList();

   //To do: Close Pop Up if already available.

   if (oMsg.GetHdl() >= lTeamId.size())
   {
      return true;
   }

   _selectedTeamId = lTeamId[oMsg.GetHdl()].getTeamID();
   _selectedTeamName = lTeamId[oMsg.GetHdl()].getTeamFullName();

   if (_noOfFavorites < SXM_AUDIO_SPORTSFLASH_MAX_ENTRY)
   {
      ETG_TRACE_USR4(("SXMAudioSportsOnPressTeamNameReqMsg _noOfFavorites:%d", _noOfFavorites));
      if (_sxmAudioFiProxy)
      {
         ++_noOfFavorites;
         _sxmAudioFiProxy->sendSportsFlashAddFavoriteTeamStart(*this, _sportsType, _selectedLeagueId, _selectedTeamId);
      }
   }
   else
   {
      //POST_MSG(COURIER_MESSAGE_NEW(SXMAudioSportsCreateOfMaxFavPopupMsg)());//Todo : Use common messge.
      POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__MPOP_SPORTSFLASH_MAX_FAV)));
   }
   return true;
}


/**
 * Callback to handle Courier Message SXMAudioSportsCloseOfMaxFavPopupReqMsg Request
 */
bool SxmAudioSportsFlash::onCourierMessage(const SXMAudioSportsCloseOfMaxFavPopupReqMsg& /*oMsg*/)
{
   //POST_MSG(COURIER_MESSAGE_NEW(SXMAudioSportsCloseMaxFavPopupMsg)());//Todo : Use common messge.
   POST_MSG((COURIER_MESSAGE_NEW(SxmSceneTransitionReqMsg)(Sxm_Scenes_SXM_RADIO__SPORTSFLASH_REPLACE_TEAM)));
   return true;
}


/**
 * Callback to handle Courier Message SXMAudioSportsAlertListReqMsg Request
 */
bool SxmAudioSportsFlash::onCourierMessage(const SXMAudioSportsAlertListReqMsg& /*oMsg*/)
{
   if (_sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendSportsFlashGetCurrentFlashEventsStart(*this);
      _doTransition = true;
   }
   return true;
}


/**
 * Callback to handle Courier Message SXMAudioSportsFavListReqMsg Request
 */
bool SxmAudioSportsFlash::onCourierMessage(const SXMAudioSportsFavListReqMsg& /*oMsg*/) //add wait sync
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash: SXMAudioSportsFavListReqMsg _sfServiceStatus:%d", _sfServiceStatus));
   if (T_e8_FlashServiceStatus__FLASH_SERVICE_STARTED == _sfServiceStatus ||
         T_e8_FlashServiceStatus__FLASH_SERVICE_ALREADY_STARTED == _sfServiceStatus)
   {
      if (_sxmAudioFiProxy)
      {
         _sxmAudioFiProxy->sendSportsFlashGetAllFavoritesStart(*this);
         _doTransition = true;
         _isTeamAddedOnSFSelection = false;
      }
   }
   else
   {
      updateAdvisoryForSF();
      POST_MSG((COURIER_MESSAGE_NEW(SxmSceneTransitionReqMsg)(Sxm_Scenes_SXM_RADIO__EDIT_SPORTSFLASH_TEAMS)));
   }
   return true;
}


/**
 * Callback to handle Courier Message SXMAudioSportsRefreshSFStatusReqMsg Request
 */
bool SxmAudioSportsFlash::onCourierMessage(const SXMAudioSportsRefreshSFStatusReqMsg& /*oMsg*/)
{
   //_subStatus = static_cast<SXMAudio_status>(oMsg.GetSubStatus());
   //updateSFIconStatus();
   return false;
}


/**
 * Callback to handle Courier Message SXMAudioSportsOnPressReplaceTeamNameReqMsg Request
 */
bool SxmAudioSportsFlash::onCourierMessage(const SXMAudioSportsOnPressReplaceTeamNameReqMsg& oMsg)
{
   ::std::vector<T_SportsFlash_FavoriteTeam > lFavList =
      SxmAudioListHelper::instance()->getSxmSportsFlashFavoriteList();
   uint32 rHdl = oMsg.GetHdl();
   ETG_TRACE_USR4(("SxmAudioSportsFlash: SXMAudioSportsOnPressReplaceTeamNameReqMsg rHdl:%d", rHdl));
   if (rHdl < lFavList.size())
   {
      T_e8_SportsFlash_Sport mFavSportsType = lFavList[rHdl].getSportType();
      uint16 mFavTeamId = lFavList[rHdl].getTeamID();
      uint8 mFavLeagueId = lFavList[rHdl].getLeagueID();
      if (_sxmAudioFiProxy)
      {
         _replaceFavorite = true;
         _sxmAudioFiProxy->sendSportsFlashRemoveFavoriteTeamStart(*this, mFavSportsType, mFavLeagueId, mFavTeamId);
      }
   }
   return true;
}


/**
 * Callback to handle Courier Message SXMAudioSportsflashTurnAlertAllOnReqMsg Request
 */
bool SxmAudioSportsFlash::onCourierMessage(const SXMAudioSportsflashTurnAlertAllOnReqMsg& oMsg)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash: SXMAudioSportsflashTurnAlertAllOnReqMsg"));
   updateAllTeamsOnOff(oMsg.GetSxm_SportsflashAlertStatus());
   if (_sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendSportsFlashMonitorTeamsStart(*this, _favoriteTeams);
   }
   return true;
}


/**
 * Callback to handle Courier Message SXMAudioSportsflashDeleteFavtTeamReqMsg Request
 * @param [in] : Reference of Courier::SXMAudioSportsflashDeleteFavtTeamReqMsg
 * @return     : bool
 */
bool SxmAudioSportsFlash::onCourierMessage(const SXMAudioSportsflashDeleteFavtTeamReqMsg& oMsg)
{
   uint32 lSubHdl = oMsg.GetSubHdl();
   uint32 lHdl = oMsg.GetHdl();
   ETG_TRACE_USR4(("SxmAudioSportsFlash: SXMAudioSportsflashDeleteFavtTeamReqMsg lHdl:%d lSubHdl:%d", lHdl, lSubHdl));
   ::std::vector<T_SportsFlash_FavoriteTeam > lFavList =
      SxmAudioListHelper::instance()->getSxmSportsFlashFavoriteList();
   if ((oMsg.GetHdl() < lFavList.size()) && (_favoriteTeams.size() == lFavList.size()))
   {
      if (SXM_AUDIO_SPORTSFLASH_DELETE == lSubHdl && _sxmAudioFiProxy)
      {
         _sxmAudioFiProxy->sendSportsFlashRemoveFavoriteTeamStart(*this, lFavList[lHdl].getSportType(),
               lFavList[lHdl].getLeagueID(),
               lFavList[lHdl].getTeamID());
      }
      else  if (lSubHdl == SXM_AUDIO_SPORTSFLASH_ALERT && _sxmAudioFiProxy)
      {
         _favoriteTeams[lHdl].setStatus(!(_favoriteTeams[lHdl].getStatus()));
         _sxmAudioFiProxy->sendSportsFlashMonitorTeamsStart(*this, _favoriteTeams);
      }
   }
   return true;
}


/**
 *Callback to handle Courier Message SxmSFButtonPressReqMsg Request
 */
bool SxmAudioSportsFlash::onCourierMessage(const SxmSFButtonPressReqMsg& oMsg)
{
   uint32 btnColor = oMsg.GetButtonColor();
   ETG_TRACE_USR4(("SxmAudioSportsFlash: SxmSFButtonPressReqMsg _noOfFavorites:%d,_isTeamAvailForMonitor:%d,getSFStatus:%d",
                   _noOfFavorites, _isTeamAvailForMonitor, getSFStatus()));

   if (SFGREY == btnColor)
   {
      if (SXM_AUDIO_SUBSCRIBED != _subStatus)
      {
         postPopupMessageForSF(Sxm_SxmPopups_SXM_RADIO__TPOP_FAILURE_CHANNEL_NOT_SUBSCRIBED);
      }
      else if ((!_noOfFavorites) || (!_isTeamAvailForMonitor)) // check this variables always upto date.
      {
         postPopupMessageForSF(Sxm_SxmPopups_SXM_RADIO__ICPOP_SPORTS_DEFINE_TEAM);
      }
      else if (!getSFStatus())
      {
         postPopupMessageForSF(Sxm_SxmPopups_SXM_RADIO__TPOP_SPORTSFLASH_MONITORING);
         if (_sxmAudioFiProxy)
         {
            _sxmAudioFiProxy->sendSportsFlashEnableNotificationStart(*this, true);
            updateSFStatus(true);
         }
      }
   }
   else if (SFWHITE == btnColor)
   {
      postPopupMessageForSF(Sxm_SxmPopups_SXM_RADIO__ICPOP_SF_NOGAME_FAVTEAM);
   }
   else if (SFGREEN == btnColor)
   {
      postPopupMessageForSF(Sxm_SxmPopups_SXM_RADIO__ICPOP_SF_FAVTEAM_GAME_AVAIL);
   }
   else if (SFRED == btnColor)
   {
      postPopupMessageForSF(Sxm_SxmPopups_SXM_RADIO__ICPOP_SF_REPLAY);
   }
   return true;
}


/**
 * Callback to handle Courier Message SxmSFYesForAddPopupReqMsg Request
 */
bool SxmAudioSportsFlash::onCourierMessage(const SxmSFDisableMonitorReqMsg& /*oMsg*/)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash: SxmSFDisableMonitorReqMsg"));
   if (_sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendSportsFlashEnableNotificationStart(*this, false);
      SxmAudioDataBindingUtils::instance()->setSFButtonStatus(SFGREY, true);
      updateSFStatus(false);
   }
   return true;
}


/**
 * Callback to handle Courier Message SxmSFYesForMonitorPopupReqMsg Request
 */
bool SxmAudioSportsFlash::onCourierMessage(const SxmSFYesForMonitorPopupReqMsg& /*oMsg*/)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash: SxmSFYesForMonitorPopupReqMsg"));
   if (_sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendSportsFlashGetAllFavoritesStart(*this);
      _isTeamAddedOnSFSelection = true;
      _doTransition = true;
   }
   return true;
}


/**
 * Callback to handle Courier Message SxmSFMonitorTeamPopupReqMsg Request
 */
bool SxmAudioSportsFlash::onCourierMessage(const SxmSFTuneChannelReqMsg& /*oMsg*/)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash: SxmSFTuneChannelReqMsg _activeFlasheventId:%d", _activeFlasheventId));
   _isSFActivePopupVisible = false;
   if (_sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendSportsFlashPlayFlashEventStart(*this, _activeFlasheventId);
   }
   return true;
}


/*
 * Callback to handle Courier Message SxmSFReturnChannelReqMsg Request
 */
bool SxmAudioSportsFlash::onCourierMessage(const SxmSFReturnChannelReqMsg& /*oMsg*/)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash: SxmSFReturnChannelReqMsg _isContextChangeRequested:%d", _isContextChangeRequested));
   if (_sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendSportsFlashAbortFlashEventStart(*this);
      _isSFReturnSkPressed = true;
   }
   SXMSourceSwitchInterface::handleSourceChangeRequest(SRC_SXM_NEWS, false, false, true);
   if (SXMSourceSwitchInterface::isSxmNewsSourceActive())
   {
      _isContextChangeRequested = true;
   }
   return true;
}


/*
 *  Callback to handle Courier Message SxmSFAlertCloseReqMsg
 */
bool SxmAudioSportsFlash::onCourierMessage(const SxmSFAlertCloseReqMsg& /*oMsg*/)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::SxmSFAlertCloseReqMsg _popupCloseOnAppmodeChange:%d", _popupCloseOnAppmodeChange));
   _isSFActivePopupVisible = false;
   if ((T_e8_FlashEventPlaybackStatus__FLASHEVENT_PLAYING != _playbackStatus) && !_popupCloseOnAppmodeChange)
   {
      updateAlertPopupInfo();
   }
   return true;
}


/*
 *  Helper function to update Alert details.
 */
void SxmAudioSportsFlash::updateAlertPopupInfo()
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash: updateAlertPopupInfo _activeSFEvents.size():%d", _activeSFEvents.size()));
   dp_tclAppHmi_SxmSXMAudioAlert dp_AudioAlert;
   HmiAppSxm::AlertSetting tSXMAudioAlert = dp_AudioAlert.tGetData();
   if (tSXMAudioAlert.OnOrOff[Item_2] && (isPopupAllowed()))
   {
      if (!_activeSFEvents.empty())
      {
         if (!_popupCloseOnAppmodeChange)
         {
            _activeSFEvents.erase(_activeSFEvents.begin());
         }
         if (!_activeSFEvents.empty() && _allowSFAlert)
         {
            setSportsFlashAlertinfo();
         }
      }
   }
   else
   {
      _activeSFEvents.clear();
   }
}


/**
 * Callback to handle Courier Message SXMTuneSelectedChannelItemReqMsg Request
 * @param [in] : Reference of Courier::SXMTuneSelectedChannelItemReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioSportsFlash::onCourierMessage(const SXMTuneSelectedChannelItemReqMsg& oMsg)
{
   uint32 u32selectedidx = oMsg.GetRowId();
   bool status = false;
   switch (oMsg.GetListId())
   {
      case LIST_ID_SXM_SPORTSFLASH_ALERT_LIST:
      {
         ::std::vector<T_SportsFlash_FlashEvent> lAlertList =  SxmAudioListHelper::instance()->getSxmSportsFlashAlertList();
         if (u32selectedidx < lAlertList.size())
         {
            //Play flash event.
            if (_sxmAudioFiProxy)
            {
               _sxmAudioFiProxy->sendSportsFlashPlayFlashEventStart(*this, lAlertList[u32selectedidx].getFlashEventID());
               POST_MSG((COURIER_MESSAGE_NEW(SXMAudioGotoAudioMainMsg)()));
            }
         }
         status = true;
         break;
      }
      default:
      {
         break;
      }
   }

   return status;
}


/**
 * Callback to handle Courier Message SXMSportsBroadScopeUpdMsg Request
 * @param [in] : Reference of Courier::SXMSportsBroadScopeUpdMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioSportsFlash::onCourierMessage(const SXMSportsBroadScopeUpdMsg& /*oMsg*/)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash: SXMSportsBroadScopeUpdMsg"));
   if (SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__EDIT_SPORTSFLASH_TEAMS) ||
         SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__SPORTSFLASH_SETTINGS_SPORTS) ||
         SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__SPORTSFLASH_SETTINGS_TEAMS) ||
         SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__SPORTSFLASH_REPLACE_TEAM))
   {
      ETG_TRACE_USR4(("SxmAudioSportsFlash: SXMSportsBroadScopeUpdMsg inside SF screens"));
      //only while in Audio sub screens.
      POST_MSG((COURIER_MESSAGE_NEW(SXMAudioGotoAudioMainMsg)()));
   }
   restoreAudioSFInfo();
   return true;
}


/**
 * Callback to handle Courier Message SXMSourceStatusUpdMsg Request
 * @param [in] : Reference of Courier::SXMSourceStatusUpdMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioSportsFlash::onCourierMessage(const SXMSourceStatusUpdMsg& oMsg)
{
   ETG_TRACE_USR1(("SxmAudioSportsFlash::SXMSourceStatusUpdMsg playbackStatus:%d", _playbackStatus));
   enConnectionState sxmCurrentConnectionState = static_cast<enConnectionState>(oMsg.GetSxmNewsConnectionStatus());
   enConnectionState sxmPrevConnectionState = static_cast<enConnectionState>(oMsg.GetSxmMainConnectionStatus());

   ETG_TRACE_USR2(("SxmAudioSportsFlash::SXMSourceStatusUpdMsg CurrentState:%d previousState:%d", sxmCurrentConnectionState, sxmPrevConnectionState));

   if (T_e8_FlashEventPlaybackStatus__FLASHEVENT_PLAYING == _playbackStatus /*&& _isSourceSwitchRequested*/)
   {
      handleSFPlaybackOnSourceChange(sxmCurrentConnectionState, sxmPrevConnectionState);
   }
   return true;
}


/**
 *  Helper function to handle sports flash playback on source change.
 */
void SxmAudioSportsFlash::handleSFPlaybackOnSourceChange(const enConnectionState& currentState, const enConnectionState& previousState)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::handleSFPlaybackOnSourceChange previousState:%d, currentState:%d _isContextChangeRequested:%d", previousState, currentState, _isContextChangeRequested));
   if (connected == previousState && disconnected == currentState)
   {
      if (_sxmAudioFiProxy)
      {
         _sxmAudioFiProxy->sendSportsFlashAbortFlashEventStart(*this);
         POST_MSG((COURIER_MESSAGE_NEW(SXMDeActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__ITPOP_SPORTSFLASH_RETURN)));
      }
   }
   else if (connected == previousState && suspended == currentState)
   {
      POST_MSG((COURIER_MESSAGE_NEW(SXMDeActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__ITPOP_SPORTSFLASH_RETURN)));
   }
   else if (suspended == previousState && connected == currentState)
   {
      POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__ITPOP_SPORTSFLASH_RETURN)));
   }
}


/*
* Helper Function to update SF icon status
*/
void SxmAudioSportsFlash::updateSFIconStatus()
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::updateSFIconStatus _playbackStatus:%d", _playbackStatus));
   enSxmSFBtnColor enChangeBtnColor = SFGREY;
   if (T_e8_FlashServiceStatus__FLASH_SERVICE_STARTED == _sfServiceStatus ||
         T_e8_FlashServiceStatus__FLASH_SERVICE_ALREADY_STARTED == _sfServiceStatus)
   {
      if ((SXM_AUDIO_SUBSCRIBED != _subStatus) || (!_noOfFavorites) || (!_isTeamAvailForMonitor) || !getSFStatus())
      {
         enChangeBtnColor = SFGREY;
      }
      else if (!_isActiveGamesAvailForMonitorTeam)
      {
         enChangeBtnColor = SFWHITE;
      }
      else
      {
         if (T_e8_FlashEventPlaybackStatus__FLASHEVENT_PLAYING == _playbackStatus)
         {
            enChangeBtnColor = SFRED;
            SxmAudioDataBindingUtils::instance()->setDefaultSxmAudioPlayPauseData(T_e8_IRPlaybackStatus__SXM_AUDIO_IR_PLAY_STATE);
         }
         else
         {
            enChangeBtnColor = SFGREEN;
         }
      }
      SxmAudioDataBindingUtils::instance()->setSFButtonStatus(enChangeBtnColor, true);
   }
}


/*
* Helper Function to update SF advisory
*/
void SxmAudioSportsFlash::updateAdvisoryForSF()
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::updateAdvisoryForSF _noOfFavorites:%d", _noOfFavorites));
   SxmAudioDataBindingUtils* pAudioDataBindingUtilsInstance = SxmAudioDataBindingUtils::instance();
   if (T_e8_FlashServiceStatus__FLASH_SERVICE_STARTED == _sfServiceStatus ||
         T_e8_FlashServiceStatus__FLASH_SERVICE_ALREADY_STARTED == _sfServiceStatus)
   {
      if (_noOfFavorites == 0)
      {
         std::string strAdvisory = "";
         strAdvisory.append(SxmUtils::safeGetCString(SXM__AUDIO_ADV_SFNOTEAMSADDEDL1)).append("\n\n").append(SxmUtils::safeGetCString(SXM__AUDIO_ADV_SFNOTEAMSADDEDL2));   //edit sports flash team
         pAudioDataBindingUtilsInstance->setAlertsAdvisory(false, strAdvisory);
         pAudioDataBindingUtilsInstance->switchSportsFlashDeleteAlertOnOff(false);
      }
      else
      {
         if (SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__EDIT_SPORTSFLASH_TEAMS))
         {
            //list is updated when team is selected.
            updateListInfo(LIST_ID_SXM_AUDIO_SPORTSFLASH_FAV_TEAMS);
         }
         pAudioDataBindingUtilsInstance->setAlertsAdvisory(true, "");
         pAudioDataBindingUtilsInstance->switchSportsFlashDeleteAlertOnOff(true);
      }
   }
   else
   {
      pAudioDataBindingUtilsInstance->setAlertsAdvisory(false, "Loading");
      pAudioDataBindingUtilsInstance->switchSportsFlashDeleteAlertOnOff(false);
   }
}


/*
* Helper Function to update SF teams
*/
void SxmAudioSportsFlash::updateFavTeams()
{
   ::std::vector<T_SportsFlash_FavoriteTeam> lFavList =
      SxmAudioListHelper::instance()->getSxmSportsFlashFavoriteList();
   ::std::vector< T_SportsFlash_FavoriteTeam>::const_iterator itr;
   _favoriteTeams.clear();
   for (itr = lFavList.begin(); itr != lFavList.end(); ++itr)
   {
      T_SportsFlash_MonitorTeam Team(itr->getLeagueID(), itr->getTeamID(), itr->getIsMonitored());
      _favoriteTeams.push_back(Team);
   }
}


/*
* Helper Function to update SF datapool
*/
void SxmAudioSportsFlash::updateFavDatapool()
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::updateFavDatapool"));
   dp_tclAppHmi_SxmSXMSportsFlashTeams dp_SportsFlash;
   HmiAppSxm::SXMSportsFlashFavTeams tSxmSportsFlash;
   tSxmSportsFlash = dp_SportsFlash.tGetData();
   _isTeamAvailForMonitor = false;
   bool lIsAllTeamsMonitored = true;
   for (uint8 u8index = 0; u8index < SXM_AUDIO_SPORTSFLASH_MAX_ENTRY; ++u8index)
   {
      if (u8index < _favoriteTeams.size())
      {
         tSxmSportsFlash.SFMonitorList[u8index].LeagueId = static_cast<uint16>(_favoriteTeams[u8index].getLeagueID());
         tSxmSportsFlash.SFMonitorList[u8index].TeamId = _favoriteTeams[u8index].getTeamID();
         tSxmSportsFlash.SFMonitorList[u8index].Status = static_cast<uint16>(_favoriteTeams[u8index].getStatus());
         if (tSxmSportsFlash.SFMonitorList[u8index].Status)
         {
            _isTeamAvailForMonitor = true;
            lIsAllTeamsMonitored = lIsAllTeamsMonitored && true;
         }
         else
         {
            lIsAllTeamsMonitored = lIsAllTeamsMonitored && false;
         }
      }
      else
      {
         tSxmSportsFlash.SFMonitorList[u8index].LeagueId = DEFAULT_SF_DP_INVALID;
         tSxmSportsFlash.SFMonitorList[u8index].TeamId = DEFAULT_SF_DP_INVALID;
         tSxmSportsFlash.SFMonitorList[u8index].Status = DEFAULT_SF_DP_INVALID;
      }
   }
   dp_SportsFlash.s32SetData(tSxmSportsFlash);
   SxmAudioDataBindingUtils::instance()->switchSportsFlashAlertOnOff(lIsAllTeamsMonitored);
}


/*
* Helper Function to Get SF status
*/
bool SxmAudioSportsFlash::getSFStatus() const
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::getSFStatus"));
   dp_tclAppHmi_SxmSxmSportsFlashNotification dp_SportsFlashNotify;
   return dp_SportsFlashNotify.tGetData();
}


/*
* Helper Function to set SF status
*/
void SxmAudioSportsFlash::updateSFStatus(bool Status) const
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::updateSFStatus"));
   dp_tclAppHmi_SxmSxmSportsFlashNotification dp_SportsFlashNotify;
   dp_SportsFlashNotify.s32SetData(Status);
}


/*
* Helper Function to read SF datapool
*/
void SxmAudioSportsFlash::readFromSFDatapool()
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::readFromSFDatapool"));
   _favoriteTeams.clear();
   dp_tclAppHmi_SxmSXMSportsFlashTeams dp_SportsFlash;
   HmiAppSxm::SXMSportsFlashFavTeams tSxmSportsFlash;
   tSxmSportsFlash = dp_SportsFlash.tGetData();
   for (uint8 u8index = 0; u8index < SXM_AUDIO_SPORTSFLASH_MAX_ENTRY; ++u8index)
   {
      if ((tSxmSportsFlash.SFMonitorList[u8index].LeagueId != DEFAULT_SF_DP_INVALID) &&
            (tSxmSportsFlash.SFMonitorList[u8index].TeamId != DEFAULT_SF_DP_INVALID) &&
            (tSxmSportsFlash.SFMonitorList[u8index].Status != DEFAULT_SF_DP_INVALID))
      {
         T_SportsFlash_MonitorTeam lSFMonitor;
         lSFMonitor.setLeagueID(tSxmSportsFlash.SFMonitorList[u8index].LeagueId);
         lSFMonitor.setTeamID(tSxmSportsFlash.SFMonitorList[u8index].TeamId);
         lSFMonitor.setStatus(tSxmSportsFlash.SFMonitorList[u8index].Status);
         _favoriteTeams.push_back(lSFMonitor);
      }
   }
}


/**
 * Helper Function to update all teams on / off status
 */
void SxmAudioSportsFlash::updateAllTeamsOnOff(bool bAlertStatus)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::updateAllTeamsOnOff"));
   size_t FavTeamSize = _favoriteTeams.size();
   for (uint8 u8index = 0; u8index < FavTeamSize; ++u8index)
   {
      _favoriteTeams[u8index].setStatus(!bAlertStatus);
   }
}


/**
 * getListDataProvider : Gets the ListDataProvider from the corresponding listdataprovider functions
 * based on the List ID.
 * @param [in] : listId
 * @return     : tSharedPtrDataProvider class pointer
 */
tSharedPtrDataProvider SxmAudioSportsFlash::getListDataProvider(const ListDateProviderReqMsg& msgListDataInfo)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::getListDataProvider()"));
   ListDataInfo listDataInfo;
   listDataInfo.listId = msgListDataInfo.GetListId();
   listDataInfo.startIndex = msgListDataInfo.GetStartIndex();
   listDataInfo.windowSize = msgListDataInfo.GetWindowElementSize();

   tSharedPtrDataProvider dataProvider = getSFListDataProvider(listDataInfo);
   return dataProvider;
}


/**
 * getSFListDataProvider : Gets the ListDataProvider for SportsFlash list
 * @param [in] : list data
 * @return     : tSharedPtrDataProvider class pointer
 */
tSharedPtrDataProvider SxmAudioSportsFlash::getSFListDataProvider(ListDataInfo listDataInfo)
{
   tSharedPtrDataProvider dataProvider = tSharedPtrDataProvider();
   switch (listDataInfo.listId)
   {
      case LIST_ID_SXM_SPORTSFLASH_LEAGUELIST:
      {
         dataProvider = SxmAudioListHelper::instance()->getSxmSportsFlashLeagueListDataProvider();
         break;
      }
      case LIST_ID_SXM_SPORTSFLASH_TEAMLIST_LEAGUE:
      {
         dataProvider = SxmAudioListHelper::instance()->getSxmSportsFlashTeamListDataProvider();
         break;
      }
      case LIST_ID_SXM_SPORTSFLASH_FAV_REPLACETEAM:
      {
         dataProvider = SxmAudioListHelper::instance()->getSxmSportsFlashReplaceListDataProvider();
         break;
      }
      case LIST_ID_SXM_AUDIO_SPORTSFLASH_FAV_TEAMS:
      {
         dataProvider = SxmAudioListHelper::instance()->getSxmSportsFlashFavoriteListDataProvider();
         break;
      }
      case LIST_ID_SXM_SPORTSFLASH_ALERT_LIST:
      {
         dataProvider = SxmAudioListHelper::instance()->getSxmSportsFlashAlertListDataProvider();
         break;
      }
      default:
      {
         ETG_TRACE_ERR(("getListDataProvider:ERROR:Invalid List"));
         break;
      }
   }
   return dataProvider;
}


/**
 * Update Lists on middleware data.
 * @param[in] : ListId - To update the lists.
 */
void SxmAudioSportsFlash::updateListInfo(unsigned int lListId)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::UpdateListInfo()"));
   ListDataInfo listDataInfo = {lListId,                       //Lint warning : listID
                                SXM_EMPTY,                       //StartIndex
                                SXM_NUM_ELE_PER_SCENE            //WindowSize
                               };
   tSharedPtrDataProvider dataProvider = getSFListDataProvider(listDataInfo);
   if (!dataProvider.PointsToNull())
   {
      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
   }
}


/**
 * Function to Set Sports Flash Alert
 */
void SxmAudioSportsFlash::setSportsFlashAlertinfo(void)
{
   ETG_TRACE_USR1(("SxmAudioSportsFlash::setSportsFlashAlertinfo()"));
   if (_activeSFEvents.size())
   {
      //As always we need to update the first item in the vector to show in popup, index - 0 is used
      _activeFlasheventId = _activeSFEvents[0].FlashEventID;
      uint16 ChannelId = _activeSFEvents[0].GameInfo.getChannelID();
      std::string hteam = _activeSFEvents[0].GameInfo.getHomeTeam().getTeamAbbrName();
      regionText regionText = SXMSourceSwitchInterface::getPopupText();
      std::string ateam = _activeSFEvents[0].GameInfo.getAwayTeam().getTeamAbbrName();
      std::string leagueName = _activeSFEvents[0].GameInfo.getHomeTeam().getLeagueAbbrName();
      ETG_TRACE_USR2(("SxmAudioSportsFlash::setSportsFlashAlertinfo, leagueName:%s", leagueName.c_str()));
      SxmAudioDataBindingUtils::instance()->setSFTunePopup(ChannelId, ateam, hteam, leagueName, regionText);
      POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__ITPOP_SPORTSFLASH_ACTIVE)));
      _isSFActivePopupVisible = true;
   }
}


/*
 *  Helper function to post popup message
 */
void SxmAudioSportsFlash::postPopupMessageForSF(uint32 u32Popup) const
{
   POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(u32Popup)));
}


/*
 *  Helper function to handle sports flash notification
 *  @param[in] : status - current flash event
 */
void SxmAudioSportsFlash::handleSportFlashNotification(const boost::shared_ptr< SportsFlash_FlashEventNotificationStatus >& status)
{
   dp_tclAppHmi_SxmSXMAudioAlert dp_AudioAlert;
   HmiAppSxm::AlertSetting tSXMAudioAlert = dp_AudioAlert.tGetData();
   T_e8_SportsFlash_FlashEventStatus flashEventStatus = status->getFlashEventStatus();
   uint32 flashEventID = status->getFlashEventID();
   T_SportsFlash_Game gameInfo = status->getGameInfo();
   ETG_TRACE_USR4(("SxmAudioSportsFlash::handleSportFlashNotification flashEventStatus:%d", flashEventStatus));
   ETG_TRACE_USR4(("SxmAudioSportsFlash::handleSportFlashNotification Item2:%d, Item3:%d", tSXMAudioAlert.OnOrOff[Item_2], tSXMAudioAlert.OnOrOff[Item_3]));

   if (tSXMAudioAlert.OnOrOff[Item_2] && (isPopupAllowed()))
   {
      if (T_e8_SportsFlash_FlashEventStatus__SPORTS_FLASH_EVENT_OCCURED == flashEventStatus)
      {
         T_FlashEventInfo newFlashEvent(flashEventID, gameInfo, flashEventStatus);
         _activeSFEvents.push_back(newFlashEvent);
         if (_activeSFEvents.size() == 1 && _allowSFAlert)//1:Check if only one flash event is received.
         {
            setSportsFlashAlertinfo();
         }
      }
      else if (T_e8_SportsFlash_FlashEventStatus__SPORTS_FLASH_EVENT_EXPIRED == flashEventStatus)
      {
         if (_isSFActivePopupVisible && (_activeFlasheventId == flashEventID))
         {
            //close SF Active popup
            POST_MSG((COURIER_MESSAGE_NEW(SXMDeActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__ITPOP_SPORTSFLASH_ACTIVE)));
            _isSFActivePopupVisible = false;
         }
         else
         {
            std::vector<T_FlashEventInfo>::iterator itr;
            for (itr = _activeSFEvents.begin(); itr != _activeSFEvents.end(); ++itr)
            {
               if (itr->FlashEventID == flashEventID)
               {
                  _activeSFEvents.erase(itr);
                  break;
               }
            }
         }
      }
      else
      {
         //To do:check handling of played state.
      }
   }
   else
   {
      _activeSFEvents.clear();
   }
}


/**
 * Helper function to getSxmSportsFlashAlertListAdvisory
 * @param[in] : bool value to indicate which advisory to display.
 * @return : List for  Advisory message
 */
const std::string SxmAudioSportsFlash::getSxmSportsFlashAlertListAdvisory(bool lAdvType) const
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::getSxmSportsFlashAlertListAdvisory lAdvType:%d", lAdvType));
   std::string strAdvisory = "";

   if (lAdvType == true)
   {
      strAdvisory = SxmUtils::safeGetCString(SXM__AUDIO_ADV_SFTEAMSNOTREGISTERED);
   }
   else if (lAdvType == false)
   {
      strAdvisory = SxmUtils::safeGetCString(SXM__AUDIO_ADV_SFTEAMSREGISTERED);
   }
   return strAdvisory;
}


/**
 * Helper Function to handle channel art result.
 */
void SxmAudioSportsFlash::onGetChannelArtListError()
{
   ETG_TRACE_ERR(("SxmAudioSportsFlash::onGetChannelArtListError"));
}


/**
 * Helper Function to handle channel art result.
 */
void SxmAudioSportsFlash::onGetChannelLogoError(const uint16& /*sid*/)
{
   ETG_TRACE_ERR(("SxmAudioSportsFlash::onGetChannelLogoError"));
}


/**
*  Call Back function for service status.
*
*  @param[in] : lEntryExitStatus - if service is entered or exited.
*/
void SxmAudioSportsFlash::vNotifyServiceStatus(unsigned short int lEntryExitStatus, unsigned int serviceType)
{
   ETG_TRACE_USR2(("SxmAudioSportsFlash::vNotifyServiceStatus %d", lEntryExitStatus));
}


/**
 *  Call Back function for system appmode status.
 */
void SxmAudioSportsFlash::vNotifyAppModeChange(unsigned short int lactiveAppMode)
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::vNotifySystemAppModeChange lactiveAppMode:%d", lactiveAppMode));

   //For RVC, SONAR - the interrupt management is handled through queuing concept.
   if ((lactiveAppMode == APP_MODE_PHONE_INCOMING_CALL) || (lactiveAppMode == APP_MODE_APPS_INCOMING_CALL) || (lactiveAppMode == APP_MODE_ECALL) ||
         (lactiveAppMode == APP_MODE_PHONE_INCOMING_SMS) || (lactiveAppMode == APP_MODE_OPERATOR) || (lactiveAppMode == APP_MODE_VR) ||
         (lactiveAppMode == APP_MODE_SPCXVR) /*|| (lactiveAppMode == APP_MODE_SONAR) || (lactiveAppMode == APP_MODE_AVM)*/)
   {
      //Based on these appmodes, popup is not allowed to create.
      ETG_TRACE_USR4(("SxmAudioSportsFlash::vNotifySystemAppModeChange _isSFActivePopupVisible:%d", _isSFActivePopupVisible));
      _allowSFAlert = false;
      if (_isSFActivePopupVisible)
      {
         _popupCloseOnAppmodeChange = true;
         //close SF Active popup
         POST_MSG((COURIER_MESSAGE_NEW(SXMDeActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__ITPOP_SPORTSFLASH_ACTIVE)));
      }
   }
   else
   {
      ETG_TRACE_USR4(("SxmAudioSportsFlash::vNotifySystemAppModeChange"));
      _allowSFAlert = true;
      if ((T_e8_FlashEventPlaybackStatus__FLASHEVENT_PLAYING != _playbackStatus) && _popupCloseOnAppmodeChange)
      {
         updateAlertPopupInfo();
         _popupCloseOnAppmodeChange = false;
      }
   }
}


/*
*  Helper function to handle broad scope changes.
*/
void SxmAudioSportsFlash::handleBroadScopeChangesforSF()
{
   ETG_TRACE_USR4(("SxmAudioSportsFlash::handleBroadScopeChangesforSF _isSFActivePopupVisible:%d _isContextChangeRequested:%d", _isSFActivePopupVisible, _isContextChangeRequested));
   SxmAudioDataBindingUtils::instance()->setSFButtonStatus(SFGREY, false);
   if (_isSFActivePopupVisible)
   {
      _isSFActivePopupVisible = false;
      //close SF Active popup
      POST_MSG((COURIER_MESSAGE_NEW(SXMDeActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__ITPOP_SPORTSFLASH_ACTIVE)));
   }
   restoreAudioSFInfo(false);
   if (SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__EDIT_SPORTSFLASH_TEAMS) ||
         SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__SPORTSFLASH_SETTINGS_SPORTS) ||
         SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__SPORTSFLASH_SETTINGS_TEAMS) ||
         SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__SPORTSFLASH_REPLACE_TEAM))
   {
      ETG_TRACE_USR4(("SxmAudioSportsFlash: SXMSportsBroadScopeUpdMsg inside SF screens"));
      //only while in Audio sub screens.
      POST_MSG((COURIER_MESSAGE_NEW(SXMAudioGotoAudioMainMsg)()));
   }
   if (SXMSourceSwitchInterface::isSxmNewsSourceActive())
   {
      SXMSourceSwitchInterface::handleSourceChangeRequest(SRC_SXM_NEWS, false, false, true);
   }
}


/**
 * Function to check whether popup allowed based on settings and source status
 */
bool SxmAudioSportsFlash::isPopupAllowed()
{
   bool status = false;
   std::bitset<6> alertSource;
   alertSource = std::bitset<6>(SXMSourceSwitchInterface::getAlertSourceStatus());
   //Either Alert to be allowed in any region (or) SXM to be active in any region
   /*
    const uint8 IS_ALERT_ALLOWED_COCKPIT = 0;
   const uint8 IS_ALERT_ALLOWED_CABINA = 1;
   const uint8 IS_ALERT_ALLOWED_CABINB = 2;
   const uint8 IS_SXM_IN_COCKPIT = 3;
   const uint8 IS_SXM_IN_CABINA = 4;
   const uint8 IS_SXM_IN_CABINB = 5;
    */
   status = alertSource.any();
   return status;
}


}  // namespace Core
}  // namespace App
