/**************************************************************************************
 * @file         : SxmAudioPlayback.cpp
 * @addtogroup   : AppHmi_Sxm
 * @brief        : Functional class to handle business logic for Instant Replay,Tune Scan.
 * @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 **************************************************************************************/
#include "hall_std_if.h"   // Including HALL framework code
#include "SxmAudioPlayback.h"
#include "SxmAudioListHelper.h"
#include "SxmAudioDataBindingUtils.h"
#include "SxmAudioTypes.h"
#include "App/Core/SxmUtils/SxmUtils.h"
#include "App/Core/ClientHandler/SourceSwitch/SXMSourceSwitchInterface.h"
#include "App/Core/SxmCommon/SxmHMIServiceStateHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXMAUDIO
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::SxmAudioPlayback::
#include "trcGenProj/Header/SxmAudioPlayback.cpp.trc.h"
#endif

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_hmi_06_if.h"

namespace App {
namespace Core {
using namespace ::sxm_audio_main_fi;
using namespace ::sxm_audio_main_fi_types;
/**
 * SxmAudioPlayback Class constructor
 * @param [in] : pSxmAudioFiProxy - SxmAudioFiProxy class instance
 */
SxmAudioPlayback::SxmAudioPlayback(::boost::shared_ptr< Sxm_audio_main_fiProxy > pSxmAudioFiProxy)
   : _sxmAudioFiProxy(pSxmAudioFiProxy)
   , _scanMode(T_e8_TuneScanMode__SXM_AUDIO_TUNE_SCAN_ABORT_MODE)
   , _previousPlaybackMode(SXM_AUDIO_REPLAY_PLAY)
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("SxmAudioPlayback() Constructor"));
}


/**
 * SxmAudioPlayback Class destructor
 */
SxmAudioPlayback::~SxmAudioPlayback()
{
   ETG_TRACE_USR4(("~SxmAudioPlayback() Destructor"));
   ETG_I_UNREGISTER_FILE();

   _sxmAudioFiProxy.reset();
   _scanMode = T_e8_TuneScanMode__SXM_AUDIO_TUNE_SCAN_ABORT_MODE;
   _previousPlaybackMode = SXM_AUDIO_REPLAY_PLAY;
}


/**
 * Utility function that registers for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioPlayback::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAudioPlayback::registerProperties"));
   if ((proxy == _sxmAudioFiProxy) && (_sxmAudioFiProxy))
   {
      ETG_TRACE_USR4(("_sxmAudioFiProxy valid, registering properties"));
      _sxmAudioFiProxy->sendInstantReplayStatusUpReg(*this);
      _sxmAudioFiProxy->sendInstantReplayModeUpReg(*this);
      _sxmAudioFiProxy->sendTuneScanContentStatusUpReg(*this);
   }
}


/**
 * Utility function that deregisters for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioPlayback::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAudioPlayback::deregisterProperties"));
   if ((proxy == _sxmAudioFiProxy) && (_sxmAudioFiProxy))
   {
      ETG_TRACE_USR4(("_sxmAudioFiProxy valid, deregistering properties"));
      _sxmAudioFiProxy->sendInstantReplayStatusRelUpRegAll();
      _sxmAudioFiProxy->sendInstantReplayModeRelUpRegAll();
      _sxmAudioFiProxy->sendTuneScanContentStatusRelUpRegAll();
   }
}


/**
 * Error handling function for instant replay control method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error - Error type
 */
void SxmAudioPlayback::onInstantReplayControlError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< InstantReplayControlError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioPlayback::onInstantReplayControlError"));
}


/**
 * Result handling function for instant replay control method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioPlayback::onInstantReplayControlResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< InstantReplayControlResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioPlayback::onInstantReplayControlResult"));
   SxmAudioDataBindingUtils::instance()->setSxmAudioPlayPauseData(result->getU8IRPlaybackMode());
}


/**
 * Error handling function for instant replay mode property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioPlayback::onInstantReplayModeError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< InstantReplayModeError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioPlayback::onInstantReplayModeError"));
}


/**
 * Status handling function for instant replay mode property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - Property status info
 */
void SxmAudioPlayback::onInstantReplayModeStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< InstantReplayModeStatus >& status)
{
   ETG_TRACE_USR4(("SxmAudioPlayback::onInstantReplayModeStatus status:%d", status->getIsIRModeActive()));
}


/**
 * Error handling function for instant replay status property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioPlayback::onInstantReplayStatusError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< InstantReplayStatusError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioPlayback::onInstantReplayStatusError"));
}


/**
 * Status handling function for instant replay status property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - Property status info
 */
void SxmAudioPlayback::onInstantReplayStatusStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& proxy,
      const boost::shared_ptr< InstantReplayStatusStatus >& status)
{
   SxmAudioDataBindingUtils* pInstance = SxmAudioDataBindingUtils::instance();
   if (proxy->hasInstantReplayStatus() && pInstance->getSxmChannelInfoSyncStatus())
   {
      uint32 u32TimeToLive =  status->getInstantReplayStatus().getTimeToLive();
      ETG_TRACE_USR4(("SxmAudioPlayback::onInstantReplayStatusStatus %d", u32TimeToLive));
      uint32 u32RecordDuration = status->getInstantReplayStatus().getRecordDuration();
      uint8 u8PlayPercentage = status->getInstantReplayStatus().getPlayPercentage();
      uint8 u8FillPercentage = status->getInstantReplayStatus().getFillPercentage();
      pInstance->setSxmAudioReplayContentData(u32TimeToLive, u32RecordDuration);
      pInstance->setSxmAudioProgressBarData(u8PlayPercentage, u8FillPercentage);
   }
}


/**
 * Error handling function for Tune Scan Content property
 */
void SxmAudioPlayback::onTuneScanContentStatusError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< ::sxm_audio_main_fi::TuneScanContentStatusError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioPlayback::onTuneScanContentStatusError"));
}


/**
 * Status handling function for Tune Scan Content property
 */
void SxmAudioPlayback::onTuneScanContentStatusStatus(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< ::sxm_audio_main_fi::TuneScanContentStatusStatus >& status)
{
   if (status->hasTuneScanStatus())
   {
      SxmAudioDataBindingUtils* pAudioUtilsInstance = SxmAudioDataBindingUtils::instance();
      T_e8_TuneScanStatus tuneStatus = status->getTuneScanStatus();
      ETG_TRACE_USR4(("SxmAudioPlayback::onTuneScanContentStatusStatus status:%d", tuneStatus));
      if (T_e8_TuneScanStatus__TUNE_SCAN_ABORTED == tuneStatus  ||
            T_e8_TuneScanStatus__TUNE_SCAN_STOPPED == tuneStatus)
      {
         pAudioUtilsInstance->updateForScanOn(tuneStatus, false);
         SXMSourceSwitchInterface::setSxmAudioScanStatus(false);
         ETG_TRACE_USR4(("SxmAudioPlayback::onTuneScanContentStatusStatus getSxmTWNPlayStatus:%d getSxmSFPlayStatus:%d", pAudioUtilsInstance->getSxmTWNPlayStatus(), pAudioUtilsInstance->getSxmSFPlayStatus()));
         if (!(pAudioUtilsInstance->getSxmTWNPlayStatus() || pAudioUtilsInstance->getSxmSFPlayStatus()))
         {
            pAudioUtilsInstance->setSxmAudioPresetControlData(PREV_MODE);
         }
      }
      else
      {
         SxmAudioListHelper* pAudioListHelperInstance = SxmAudioListHelper::instance();
         pAudioListHelperInstance->setTuneScanContentStatus(tuneStatus);
		 pAudioUtilsInstance->setTuneScanAvailableStatus(tuneStatus);
         pAudioListHelperInstance->getSxmAudioMenuListDataProvider();
         tSharedPtrDataProvider dataProvider = pAudioListHelperInstance->getSxmAudioMenuListDataProvider();;
         POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
         pAudioUtilsInstance->updateForScanOn(tuneStatus, false);
         SXMSourceSwitchInterface::setSxmAudioScanStatus(false);
      }
   }
}


/**
 * Error handling function for Tune Scan Control method
 */
void SxmAudioPlayback::onTuneScanControlError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< ::sxm_audio_main_fi::TuneScanControlError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmAudioPlayback::onTuneScanControlError"));
}


/**
 * Result handling function for Tune Scan Control method
 */
void SxmAudioPlayback::onTuneScanControlResult(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< ::sxm_audio_main_fi::TuneScanControlResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioPlayback::onTuneScanControlResult status:%d", result->getStatus()));
   const char* activeViewName = "";
   if (result->getStatus())
   {
      SxmAudioDataBindingUtils* pAudioUtilsInstance = SxmAudioDataBindingUtils::instance();
      if ((T_e8_TuneScanMode__SXM_AUDIO_TUNE_SCAN_ABORT_MODE == _scanMode) ||
            (T_e8_TuneScanMode__SXM_AUDIO_TUNE_SCAN_STOP_MODE == _scanMode))
      {
         pAudioUtilsInstance->updateForScanOn(true, false);
         pAudioUtilsInstance->setSxmAudioPresetControlData(PREV_MODE);
         SXMSourceSwitchInterface::setSxmAudioScanStatus(false);
      }
      else if (T_e8_TuneScanMode__SXM_AUDIO_TUNE_SCAN_START_MODE == _scanMode)
      {
         pAudioUtilsInstance->updateForScanOn(true, true);
         pAudioUtilsInstance->setSxmAudioPresetControlData(SCAN);
         SXMSourceSwitchInterface::setSxmAudioScanStatus(true);
         activeViewName = SxmHMIServiceStateHandler::instance()->getSxmSceneName();
         // POST_MSG((COURIER_MESSAGE_NEW(FocusReqMsg)(FOCUS_RESET, FOCUS_TIMER_STOP, Courier::ViewId(activeViewName), "")));
         if (SXM_AUDIO_REPLAY_PLAY == pAudioUtilsInstance->getSxmInstantReplayStatus())
         {
            pAudioUtilsInstance->setDefaultSxmAudioPlayPauseData(T_e8_IRPlaybackStatus__SXM_AUDIO_IR_PLAY_STATE);
         }
         POST_MSG((COURIER_MESSAGE_NEW(SXMAudioGotoAudioMainMsg)()));
      }
   }
}


/**
 * Error handling function for enable tune start method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error - Error type
 */
void SxmAudioPlayback::onEnableTuneStartError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< EnableTuneStartError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioList::onEnableTuneStartError"));
}


/**
 * Result handling function for enable tune start method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioPlayback::onEnableTuneStartResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< EnableTuneStartResult >& /*result*/)
{
   ETG_TRACE_USR4(("SxmAudioList::onEnableTuneStartResult"));
}


/**
 * Restore Dp value for Audio Playback
 */
void  SxmAudioPlayback::restoreAudioPlayback(void) const
{
   dp_tclAppHmi_SxmSxmAudioTuneStartStatus dp_TuneStartStatus;
   dp_TuneStartStatus.vSetData(true);
}


/**
 * Callback to handle Courier Message SXMAudioReplayControlReqMsg Request
 * @param [in] : Reference of Courier::SXMAudioReplayControlReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioPlayback::onCourierMessage(const SXMAudioReplayControlReqMsg& oMsg)
{
   enSxmIRPlaybackMode lMode = static_cast<enSxmIRPlaybackMode>(oMsg.GetPressedButton());
   if (SXM_AUDIO_REPLAY_PAUSE == _previousPlaybackMode &&
         (SXM_AUDIO_SOURCECHANGE_PAUSE == lMode || SXM_AUDIO_SOURCECHANGE_PLAY == lMode))
   {
      //User has paused playback,on source change do not handle.
      return true;
   }

   if (_sxmAudioFiProxy)
   {
      _previousPlaybackMode = lMode;
      T_e8_IRPlaybackMode requestedMode;
      if (SXM_AUDIO_SOURCECHANGE_PLAY == _previousPlaybackMode)
      {
         requestedMode = T_e8_IRPlaybackMode__SXM_AUDIO_IR_PLAY_MODE;
      }
      else if (SXM_AUDIO_SOURCECHANGE_PAUSE == _previousPlaybackMode)
      {
         requestedMode = T_e8_IRPlaybackMode__SXM_AUDIO_IR_PAUSE_MODE;
      }
      else
      {
         requestedMode = static_cast<T_e8_IRPlaybackMode>(oMsg.GetPressedButton());
      }
      _sxmAudioFiProxy->sendInstantReplayControlStart(*this, requestedMode);
   }

   if (SXM_AUDIO_REPLAY_FORWARD == oMsg.GetPressedButton()  || SXM_AUDIO_REPLAY_REWIND == oMsg.GetPressedButton())
   {
      POST_MSG((COURIER_MESSAGE_NEW(PlayBeepReqMsg)(hmibase::BEEPTYPE_CLICK)));
   }
   return true;
}


/**
 * Callback to handle Courier Message SxmAudioPresetControlReqMsg Request
 * @param [in] : Reference of Courier::SxmAudioPresetControlReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioPlayback::onCourierMessage(const SxmAudioPresetControlReqMsg& oMsg)
{
   //ETG_TRACE_USR4(("SxmAudioPlayback::SxmAudioPresetControlReqMsg "));
   uint8 u8ButtonPressed = oMsg.GetPressedButton();
   SxmAudioDataBindingUtils::instance()->setAudioActiveMode(u8ButtonPressed);
   SxmAudioDataBindingUtils::instance()->setSxmAudioPresetControlData(u8ButtonPressed);
   return true;
}


/**
* Callback to handle Courier Message SXMAudioScanOpertationReqMsg Request
* @param [in] : Reference of Courier::SXMAudioScanOpertationReqMsg
* @return     : True-When Message is processed.
*/
bool SxmAudioPlayback::onCourierMessage(const SXMAudioScanOpertationReqMsg& oMsg)
{
   _scanMode = static_cast<T_e8_TuneScanMode>(oMsg.GetOperationType());
   if (_scanMode <= T_e8_TuneScanMode__SXM_AUDIO_TUNE_SCAN_ABORT_MODE && _sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendTuneScanControlStart(*this, _scanMode);
   }
   return true;
}


/**
* Callback to handle Courier Message SXMAudioTuneStartReqMsg Request
* @param [in] : Reference of Courier::SXMAudioTuneStartReqMsg
* @return     : True-When Message is processed.
*/
bool SxmAudioPlayback::onCourierMessage(const SXMAudioTuneStartReqMsg& /*oMsg*/)
{
   ETG_TRACE_USR4(("SxmAudioPlayback::SXMAudioTuneStartReqMsg "));
   dp_tclAppHmi_SxmSxmAudioTuneStartStatus dp_TuneStartStatus;
   bool lIsTuneStartOn = !(dp_TuneStartStatus.tGetData());
   if (_sxmAudioFiProxy)
   {
      dp_TuneStartStatus.vSetData(lIsTuneStartOn);
      tSharedPtrDataProvider dataProvider =  SxmAudioListHelper::instance()->getSxmAudioMenuListDataProvider();
      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
      _sxmAudioFiProxy->sendEnableTuneStartStart(*this, lIsTuneStartOn);
   }
   return true;
}


/**
 * Send Midw Tune Start Status.
 */
void SxmAudioPlayback::sendTuneStartStatus()
{
   ETG_TRACE_USR4(("SxmAudioPlayback::sendTuneStartStatus "));
   dp_tclAppHmi_SxmSxmAudioTuneStartStatus dp_TuneStartStatus;
   if (_sxmAudioFiProxy)
   {
      ETG_TRACE_USR4(("SxmAudioPlayback::sendTuneStartStatus %d", dp_TuneStartStatus.tGetData()));
      _sxmAudioFiProxy->sendEnableTuneStartStart(*this, dp_TuneStartStatus.tGetData());
   }
}


/**
 * Function to set default value on service availability from suspended
 */
void SxmAudioPlayback::onServiceResumed(const ::boost::shared_ptr< asf::core::Proxy >& proxy)
{
   ETG_TRACE_USR4(("SxmAudioPlayback::onServiceResumed"));
   if ((proxy == _sxmAudioFiProxy) && (_sxmAudioFiProxy))
   {
      sendTuneStartStatus();
   }
}


}  // namespace Core
}  // namespace App
