/**************************************************************************************
 * @file         : SxmAudioListHelper.h
 * @addtogroup   : AppHmi_Sxm
 * @brief        : Utility class to handle all processing for List requests
 * @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 **************************************************************************************/
#ifndef _SxmAudioListHelper_H_
#define _SxmAudioListHelper_H_
#include "App/Core/SxmUtils/SxmUtilsSingleton.h"
#include "Widgets/2D/List/Data/ListDataProvider.h"
#include "SxmAudioTypes.h"
#include "sxm_audio_main_fi_typesConst.h"
#include "App/Core/SxmCommon/ISxmHMIServiceStateHandler.h"
#include "App/Core/SxmUtils/SxmUtilsListHandler.h"

//forward declaration
namespace sxm_audio_main_fi_types {
// forward declarations
class T_ChannelListEntry;
class T_CategoryListEntry;
class T_ChannelInformation;
class T_PresetListEntry;
class T_AlertInfo;
class T_RegisteredFavorite;
class T_SportsFlash_League;
class T_SportsFlash_Team;
class T_SportsFlash_FavoriteTeam;
class T_SportsFlash_FlashEvent;
class T_MarketListEntry;
class T_SportsFlash_MonitorTeam;
class T_SxmSmsDbInfo;
class T_SxmDiagTestModeData;
class T_SxmDataSubStatus;
class T_SxmVersionInfo;
}


namespace sxm_channelart_main_fi_types {
class T_ImageData;
}


namespace sxm_main_fi_types {
class T_SxmListElemDataSrvState;
}


namespace App {
namespace Core {

/**
 * Utility class to handle all processing for GUI requests
 * @class : SxmAudioListHelper
 */

class SxmAudioListHelper
   : public SxmUtilsSingleton<SxmAudioListHelper>
   , public ISxmHMIServiceStateHandler
{
   public:
      /**
       * SxmAudioListHelper Class destructor.
       */
      virtual ~SxmAudioListHelper();

      /*******************************************************Channel list & Info ********************************************************/
      /**
       * Helper function to set Active Channel Information.
       */
      void setSxmChannelInfoData(const ::sxm_audio_main_fi_types::T_ChannelInformation& channelInfo);
      /**
       * Helper function to handle processing of category list.
       */
      void setCategoryList(const ::std::vector< ::sxm_audio_main_fi_types::T_CategoryListEntry >& categoryList);
      /**
       * Helper function to get category list.
       */
      const ::std::vector< ::sxm_audio_main_fi_types::T_CategoryListEntry >& getCategoryList(void) const;
      /**
       * Helper function to handle processing of channel list.
       */
      void setSxmChannelList(const ::std::vector< ::sxm_audio_main_fi_types::T_ChannelListEntry>& channellist);
      /**
       * Helper function to get channel list.
       */
      const ::std::vector< ::sxm_audio_main_fi_types::T_ChannelListEntry >& getChannelList() const;
      /**
       * Helper function to get Sid for selected list item from channel list.
       */
      uint16 getSelectedChannelSID(uint32 selectedChannelIndex);
      /**
       * Helper function to set selected Category type.
       */
      void setSelectedCategoryType(uint16 catID);
      /**
      * Helper function to set tune scan content status
      */
      void setTuneScanContentStatus(bool status);
      /**
       * Helper function to get Active Category type.
       */
      uint16 getSelectedCategoryType(void)const;
      /**
       * Helper function to get Active channel SID.
       */
      uint16 getActiveChannelSID(void) const;
      /**
       * Helper function to get Active channel ID.
       */
      uint16 getActiveChannelCID(void) const;
      /**
       * Helper function to set CustomizedChannelList.
       */
      bool setSxmCustomizedChannelList(const ::std::vector< ::sxm_audio_main_fi_types::T_ChannelListEntry >& channellist);
      /**
       * Helper function to get customize channel list.
       */
      const ::std::vector< ::sxm_audio_main_fi_types::T_ChannelListEntry >& getCustomizedChannelList();
      /**
       * Helper function to get Service id for selected customized channel list entry.
       */
      uint16 getCustomizedListEntryServiceId(uint16 index);
      /**
       * Helper function to get Channel id for selected customized channel list entry.
       */
      uint16 getCustomizedListEntryChannelId(uint16 index);
      /**
       * Helper function to get Channel skipped status for selected customized channel list entry.
       */
      bool getCustomizedListEntryStatus(uint16 index);
      /*
       * Helper function to set Channellistlogo.
       */
      void setChannellistlogo(const ::std::vector< ChannelGraphicsType >& imageData, uint32 listID);
      /**
       * Helper function to get list handle for SxmAudioMenuList.
       */
      tSharedPtrDataProvider getSxmAudioMenuListDataProvider();
      /**
       * Helper function to get list handle for SxmChannelList.
       */
      tSharedPtrDataProvider getSxmChannelListDataProvider();
      /**
       * Helper function to get list handle for SxmCategorieList.
       */
      tSharedPtrDataProvider getSxmCategorieListDataProvider();
      /**
       * Helper function to get list handle for SxmCategorieList.
       */
      tSharedPtrDataProvider getSxmCustomizeChannelListDataProvider(uint16 startIndex, uint16 windowElemSize);

      /*******************************************************Preset********************************************************/

      /**
       * Helper function to set Active Preset Index for Auto Compare.
       */
      void setActivePresetIndex(uint8 presetIdx);
      /**
       * Helper function to get preset list.
       */
      const ::std::vector< ::sxm_audio_main_fi_types::T_PresetListEntry >& getPresetList() const;
      /**
       * Helper function to set preset list.
       */
      void setPresetList(const ::std::vector< ::sxm_audio_main_fi_types::T_PresetListEntry >& presetList);
      /**
       * Helper function to set Channellogo for preset list.
       */
      void setPresetCGList(const ::std::vector< ChannelGraphicsType >& presetCGList);
      /**
       * Helper function to SetPresetListElement.
       */
      void setPresetListElement(uint8 presetIdx, const ::sxm_audio_main_fi_types::T_PresetListEntry& presetElement);
      /**
       * Helper function to Preset Channel logo of a single channel
       */
      void setPresetCGListElement(uint8 presetIdx, uint16 sid, const std::vector<uint8>& imageData);
      /**
       * Helper function to get Preset list for list widget.
       * @return - tSharedPtrDataProvider for preset list.
       */
      tSharedPtrDataProvider getSxmPresetListDataProvider(bool btnstatus);
      /**
       * Helper Function to update autocompare status.
       */
      void updatePresetList(bool isEnable);
      /**
       * Helper Function to switch band.
       */
      void switchPresetBand();
      /**
       * Helper function to get preset band start index
       */
      uint8 getPresetBandStartIndex();
      /**
       * Helper function to update preset list element
       */
      bool updatePresetItem(uint8 presetIdx, const ::sxm_audio_main_fi_types::T_PresetListEntry& presetElement);
      /*******************************************************TWN********************************************************/
      /**
       * Helper function to set market list.
       */
      void setMarketList(const ::std::vector< ::sxm_audio_main_fi_types::T_MarketListEntry >& marketListEntry);
      /**
       * Helper function to get twn market info.
       */
      bool getSelectedMarketInfo(uint32 marketindex, std::string& marketname, bool& isactive);
      /**
       * Helper function to get TWN market list for list widget.
       */
      tSharedPtrDataProvider getSxmTWNowListDataProvider() ;
      /*******************************************************Content Alert********************************************************/

      /**
       * Helper function to save Favorite List.
       */
      void setFavoriteList(const ::std::vector< ::sxm_audio_main_fi_types::T_RegisteredFavorite >& mRegisteredFavoriteresult);
      /**
       * Helper function to get alert list.
       */
      const ::std::vector< ::sxm_audio_main_fi_types::T_AlertInfo >& getAlertInfoList() const;
      /**
       * Helper function to get favorite List.
       */
      const ::std::vector< ::sxm_audio_main_fi_types::T_RegisteredFavorite >& getRegisteredFavoriteList() const;
      /**
       * Helper function to save Alert Info List.
       */
      void setAlertInfoList(const ::std::vector< ::sxm_audio_main_fi_types::T_AlertInfo >& mAlertInfoList);
      /**
       * Helper function to get favorite artist list for list widget.
       */
      tSharedPtrDataProvider getSxmFavoriteArtistSongListDataProvider(uint32 listId, const char* dataItemContext) const;
      /**
       * Helper function to get replace favorite for list widget.
       */
      tSharedPtrDataProvider getSxmFavoriteReplaceListDataProvider() const;
      /**
        * Helper function to get artist/Song alert list for list widget.
        */
      tSharedPtrDataProvider getSxmAlertArtistSongListDataProvider(uint32 listId, const char* dataItemContext);

      /*******************************************************Sports Flash********************************************************/
      /**
       * Helper function to save Sports Flash logo data.
       */
      void setLogoForSFAlertList(const ::std::vector<ChannelGraphicsType>& imageData);
      /**
       * Helper function to save league list.
       */
      void setSxmSportsFlashLeagueList(const ::std::vector< ::sxm_audio_main_fi_types::T_SportsFlash_League >& sportsLeagueList);
      /**
       * Helper function to get league list.
       */
      const ::std::vector< ::sxm_audio_main_fi_types::T_SportsFlash_League >& getSxmSportsFlashLeagueList(void) const;
      /**
       * Helper function to save sports Flash team list.
       */
      void setSxmSportsFlashTeamList(const ::std::vector< ::sxm_audio_main_fi_types::T_SportsFlash_Team >& sportsTeamList);
      /**
       * Helper function to get sports Flash team list.
       */
      const ::std::vector< ::sxm_audio_main_fi_types::T_SportsFlash_Team >& getSxmSportsFlashTeamList(void) const;
      /**
       * Helper function to get Sport Name.
       */
      std::string getSportsName(uint32 lSportTypeEnum) const;
      /**
       * Helper function to save Sports Flash favorite list.
       */
      void setSxmSportsFlashFavoriteList(const ::std::vector< ::sxm_audio_main_fi_types::T_SportsFlash_FavoriteTeam >& sportsFavList);
      /**
       * Helper function to get Sports Flash favorite list.
       */
      const ::std::vector< ::sxm_audio_main_fi_types::T_SportsFlash_FavoriteTeam >& getSxmSportsFlashFavoriteList(void) const;
      /**
       * Helper function to save sports flash alert list.
       */
      void setSxmSportsFlashAlertList(const ::std::vector< ::sxm_audio_main_fi_types::T_SportsFlash_FlashEvent >& lSportsAlertList);
      /**
       * Helper function to get sports flash alert list.
       */
      const ::std::vector< ::sxm_audio_main_fi_types::T_SportsFlash_FlashEvent >& getSxmSportsFlashAlertList(void) const;
      /**
       * Helper function to get Sports Flash league list for list widget.
       */
      tSharedPtrDataProvider getSxmSportsFlashLeagueListDataProvider(void);
      /**
       * Helper function to get Sports Flash Team list for list widget.
       */
      tSharedPtrDataProvider getSxmSportsFlashTeamListDataProvider(void);
      /**
       * Helper function to get Replace Favorite list for list widget.
       */
      tSharedPtrDataProvider getSxmSportsFlashReplaceListDataProvider(void);
      /**
       * Helper function to get Sports Flash Favorite list for list widget.
       */
      tSharedPtrDataProvider getSxmSportsFlashFavoriteListDataProvider(void) const;
      /**
       * Helper function to get Replace Sports Flash Alert list for list widget.
       */
      tSharedPtrDataProvider getSxmSportsFlashAlertListDataProvider(void);

      /*******************************************************Diagnosis********************************************************/

      /**
       * Helper function to save radio id.
       */
      void setRadioData(std::string   radioId);
      /**
       * Helper function to save current antenna connection status.
       */
      void setDiagAntennaStatus(uint32 antennastatus);
      /**
       * Helper function to save current signal quality.
       */
      void setDiagSignalQuality(uint32 signalQuality);
      /**
       * Helper function to save audio subscription information.
       */
      void setDiagAudioSubInfo(uint32 audioSubInfo);
      /**
       * Helper function to save data service subscription.
       */
      void setDiagDataServiceStatus(const ::std::vector< ::sxm_main_fi_types::T_SxmListElemDataSrvState >& dataServiceStatus);
      /**
       * Helper function to save DTM diagnosis data.
       */
      void setDiagTestModeData(const ::sxm_audio_main_fi_types::T_SxmDiagTestModeData& testModeData);
      /**
       * Helper function to save sms database version.
       */
      void setDiagSmsDBVersion(const ::sxm_audio_main_fi_types::T_SxmSmsDbInfo& smsDBVersion);
      /**
       * Helper function to save module subscription information.
       */
      void setModuleSubInfo(const ::sxm_audio_main_fi_types::T_SxmDataSubStatus& moduleSubInfo);
      /**
       * Helper function to save HMI version information.
       */
      void setDiagSxmVersion();
      /**
       * Helper function to save diag status.
       */
      void setDiagStatus(bool diagState);
      /**
       * Helper function to save traffic status.
       */
      void setInfoTrafficStatus(bool trafficstatus);
      /**
       * Helper function to to get traffic status.
       */
      bool getInfoTrafficStatus();
      /*
       * Helper function to save travellink status.
       */
      void setInfoTravellinkStatus(bool travellinkstatus);
      /*
       * Helper function to save diag status.
       */
      bool getInfoTravellinkStatus();
      /**
       * Helper function to get DTM list elements for list widget.
       */
      tSharedPtrDataProvider getSxmDTMListDataProvider() const;
      /**
       * Helper function to get Monitor Menu list elements for list widget.
       */
      tSharedPtrDataProvider getSxmMonitorMenuListDataProvider();
      /**
       * Helper function to get Testmode monitor list elements for list widget.
       */
      tSharedPtrDataProvider getSxmTestModeMonitorSelection();
      /**
       * Helper function to get System Info list elements for list widget.
       */
      tSharedPtrDataProvider getSxmInfoListDataProvider();
      /**
       * Helper function to update Sxm lists in DTM.
       */
      void refreshSxmAudioList(unsigned int listId, unsigned int sceneId) const;
      /**
       * Helper function to set Travel link subscription.
       */
      void setSxmTravelLink();
      /**
       * Helper function to set Channel List Status
       */
      void setChannelListStatus(const ListDataInfo& listDataInfo);

      /*******************************************************Common********************************************************/

      /**
       * Helper function to get list handle for SxmAudioAlertSettingList.
       */
      tSharedPtrDataProvider getSxmAudioAlertSettingListDataProvider()const;
      /**
       * Helper function to clear member variables
       */
      void clearListVectors(void);
      /**
       * Helper function to get list of sids
       */
      template <typename ListType>
      const ::std::vector< uint16 > getListOfSid(ListType eventList, uint32 startindex, uint32 lastindex)
      {
         ::std::vector< uint16 > listOfSID;
         uint16 u16ChSize = eventList.size();
         for (uint16 index = startindex; (index < lastindex && index < u16ChSize); ++index)
         {
            listOfSID.push_back(eventList[index].getServiceID());
         }
         return listOfSID;
      }
      /**
      *  Call Back function for service status.
      */
      void vNotifyServiceStatus(unsigned short int lEntryExitStatus, unsigned int /*serviceType*/);
      /**
       *  Call Back function for appmode status.
       */
      void vNotifyAppModeChange(unsigned short int /*lactiveAppMode*/) {};
      void vNotifyAppState(unsigned short int appState);

   private:
      friend class  SxmUtilsSingleton<SxmAudioListHelper>;
      /**
       * SxmAudioListHelper Class private constructor
       */
      SxmAudioListHelper();
      /**
       * Helper function to get channel art.
       */
      void getChannelArt(uint16 dataSID, bool isChannelAvail, ::sxm_audio_main_fi_types::T_e8_Graphics graphicsAvailibility, uint32 cgIndex,
                         const ::std::vector< ChannelGraphicsType >& cgList, Candera::MemoryManagement::SharedPointer<Candera::Image2D>& img);
      void printPresetImagesize();
      /**
       * Helper function to initialize diagnosis map elements.
       */
      void initializeDiagMap();
      //Member Variable.
      ::std::vector< ::sxm_audio_main_fi_types::T_CategoryListEntry> _categoryList;
      ::std::vector< ::sxm_audio_main_fi_types::T_ChannelListEntry> _channelList;
      ::std::vector< ::sxm_audio_main_fi_types::T_ChannelListEntry> _customizedChannelList;
      ::std::vector< ::sxm_audio_main_fi_types::T_PresetListEntry > _presetList;
      ::std::vector< ::sxm_audio_main_fi_types::T_RegisteredFavorite > _registeredFavoriteresult;
      ::std::vector< ::sxm_audio_main_fi_types::T_AlertInfo > _alertInfoList;
      ::std::vector< ::sxm_audio_main_fi_types::T_SportsFlash_League>  _leagueList;
      ::std::vector< ::sxm_audio_main_fi_types::T_SportsFlash_Team>  _teamList;
      ::std::vector< ::sxm_audio_main_fi_types::T_SportsFlash_FavoriteTeam>  _sfFavoriteList;
      ::std::vector< ::sxm_audio_main_fi_types::T_SportsFlash_FlashEvent>  _sfAlertList;
      ::std::vector< ::sxm_audio_main_fi_types::T_MarketListEntry> _marketListEntry;
      ::std::vector< ::sxm_audio_main_fi_types::T_SportsFlash_MonitorTeam > _sportsFlashFavMonitorTeamsList;
      ::std::vector< ChannelGraphicsType > _chanllistlogo;
      ::std::vector< ChannelGraphicsType > _customChanllistlogo;
      ::std::vector< ChannelGraphicsType > _sportsFlashChannelLogo;
      ::std::vector< ChannelGraphicsType > _contentAlertlistlogo;
      ListDataInfo _channelListStatus;
      uint16 _selectedChannelSID;
      uint16 _selectedCatType;
      uint16 _activeChannelSID;
      uint16 _activeChannelID;
      uint16 _activeCategoryID;
      uint8 _activePresetIndex;
      bool _isExternalDiagmode;
      bool _isSystemInfoTraffic;
      bool _isSystemInfoTravellink;
      bool _isTuneScanContentAvail;
      std::string _radioID;
      std::string _sxmSubStatus;
      std::string _sxmSuspendDate;
      bool _isDAVariant;

      std::map<uint8, uint32> _diagnosisStmList;
      std::map<uint8, std::string> _diagnosisMonitorList;
      std::map<uint8, uint32> _diagnosisInfoList;
      std::map<uint16, Candera::MemoryManagement::SharedPointer<Candera::Image2D> > _presetBitmapImages;

      /**
       * Courier Message Handler
       */
      COURIER_MSG_MAP_BEGIN(TR_CLASS_APPHMI_SXM_COURIER_PAYLOAD_MODEL_COMP)
      COURIER_DUMMY_CASE(1)
      COURIER_MSG_MAP_DELEGATE_DEF_BEGIN()
      COURIER_MSG_MAP_DELEGATE_DEF_END()
      COURIER_MSG_MAP_DELEGATE_END()

   protected:
};


}  // namespace Core
}  // namespace App

#endif
