/**************************************************************************************
 * @file         : SxmAudioHall.cpp
 * @addtogroup   : AppHmi_Sxm
 * @brief        : SXM Audio HALL utility class which contains all the necessary business logic required for SXM Audio HMI.
 * @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 **************************************************************************************/
#include "hall_std_if.h"
#include "SxmAudioHall.h"
#include "SxmAudioList.h"
#include "SxmAudioChannelInformation.h"
#include "SxmAudioDiagnosis.h"
#include "SxmAudioPlayback.h"
#include "SxmAudioDiagnosis.h"
#include "SxmAudioContentAlert.h"
#include "SxmAudioSportsFlash.h"
#include "SxmAudioTWNow.h"
#include "SxmAudioChannelArt.h"
#include "SxmAudioPreset.h"
#include "App/Core/SxmUtils/SxmUtils.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXMAUDIO
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::SxmAudioHall::
#include "trcGenProj/Header/SxmAudioHall.cpp.trc.h"
#endif

namespace App {
namespace Core {
using namespace ::sxm_audio_main_fi;
using namespace ::sxm_channelart_main_fi;

/**
 * SxmAudioHall Class constructor
 */
SxmAudioHall::SxmAudioHall()
   : _sxmAudioFiProxy(Sxm_audio_main_fiProxy::createProxy("SxmAudioFiPort", *this))
   , _sxmChannelartFiProxy(Sxm_channelart_main_fiProxy::createProxy("SxmChannelArtFiPort", *this))
   , _onAvlStartup(false)
   , _onAvlAudio(false)
   , _onAvlChnArt(false)
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("SxmAudioHall() Constructor"));
   SXM_UTILS_CREATE_OBJECT_WITH_1PARAM(_sXMAudioChannelArt, SxmAudioChannelArt, _sxmChannelartFiProxy);
   SXM_UTILS_CREATE_OBJECT_WITH_1PARAM(_sxmAudioPlayback, SxmAudioPlayback, _sxmAudioFiProxy);
   SXM_UTILS_CREATE_OBJECT_WITH_2PARAM(_sxmAudioChannelInformation, SxmAudioChannelInformation, _sxmAudioFiProxy, _sXMAudioChannelArt);
   SXM_UTILS_CREATE_OBJECT_WITH_1PARAM(_sxmAudioDiagnosis, SxmAudioDiagnosis, _sxmAudioFiProxy);
   SXM_UTILS_CREATE_OBJECT_WITH_2PARAM(_sxmAudioList, SxmAudioList, _sxmAudioFiProxy, _sXMAudioChannelArt);
   SXM_UTILS_CREATE_OBJECT_WITH_2PARAM(_sxmAudioContentAlert, SxmAudioContentAlert, _sxmAudioFiProxy, _sXMAudioChannelArt);
   SXM_UTILS_CREATE_OBJECT_WITH_2PARAM(_sxmAudioSportsFlash, SxmAudioSportsFlash, _sxmAudioFiProxy, _sXMAudioChannelArt);
   SXM_UTILS_CREATE_OBJECT_WITH_1PARAM(_sxmAudioTWNow, SxmAudioTWNow, _sxmAudioFiProxy);
   SXM_UTILS_CREATE_OBJECT_WITH_2PARAM(_sXMAudioPreset, SxmAudioPreset, _sxmAudioFiProxy, _sXMAudioChannelArt);
   SXM_UTILS_REGISTER_PROPERTY_CALLBACKS(StartupSync::getInstance(), this, _sxmAudioFiProxy);
   SXM_UTILS_REGISTER_PROPERTY_CALLBACKS(StartupSync::getInstance(), this, _sxmChannelartFiProxy);
}


/**
 * SxmAudioHall Class destructor
 */
SxmAudioHall::~SxmAudioHall()
{
   ETG_TRACE_USR4(("~SxmAudioHall() Destructor"));
   SXM_UTILS_RELEASE_OBJECT(_sxmAudioPlayback);
   SXM_UTILS_RELEASE_OBJECT(_sXMAudioPreset);
   SXM_UTILS_RELEASE_OBJECT(_sxmAudioChannelInformation);
   SXM_UTILS_RELEASE_OBJECT(_sxmAudioDiagnosis);
   SXM_UTILS_RELEASE_OBJECT(_sxmAudioList);
   SXM_UTILS_RELEASE_OBJECT(_sxmAudioSportsFlash);
   SXM_UTILS_RELEASE_OBJECT(_sxmAudioContentAlert);
   SXM_UTILS_RELEASE_OBJECT(_sxmAudioTWNow);
   SXM_UTILS_RELEASE_OBJECT(_sXMAudioChannelArt);
   _sxmAudioFiProxy.reset();
   _sxmChannelartFiProxy.reset();
   _onAvlStartup = false;
   _onAvlAudio = false;
   _onAvlChnArt = false;
   ETG_I_UNREGISTER_FILE();
}


/**
 * Courier Message Handler function
 * @param [in]: Msg Courier Message posted from Courier Service Framework
 * @return : True, if the message is consumed; False, if the message is not consumed.
 */
bool SxmAudioHall::OnMessage(const Courier::Message& Msg)
{
   COURIER_MSG_DELEGATE_TO_OBJ(_sxmAudioPlayback);
   COURIER_MSG_DELEGATE_TO_OBJ(_sXMAudioPreset);
   COURIER_MSG_DELEGATE_TO_OBJ(_sxmAudioChannelInformation);
   COURIER_MSG_DELEGATE_TO_OBJ(_sxmAudioDiagnosis);
   COURIER_MSG_DELEGATE_TO_OBJ(_sxmAudioList);
   COURIER_MSG_DELEGATE_TO_OBJ(_sxmAudioContentAlert);
   COURIER_MSG_DELEGATE_TO_OBJ(_sxmAudioSportsFlash);
   COURIER_MSG_DELEGATE_TO_OBJ(_sxmAudioTWNow);
   COURIER_MSG_DELEGATE_TO_OBJ(_sXMAudioChannelArt);
   return false;
}


/**
 * Utility function that registers for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioHall::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/,
                                      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAudioHall::registerProperties"));
}


/**
 * Utility function that registers for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioHall::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
                               const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("SxmAudioHall::onAvailable"));
   if ((proxy == _sxmAudioFiProxy) && (_sxmAudioFiProxy))
   {
      SXM_UTILS_REGISTER_PROPERTY(_sxmAudioFiProxy, _sxmAudioPlayback, proxy, stateChange);
      SXM_UTILS_REGISTER_PROPERTY(_sxmAudioFiProxy, _sxmAudioChannelInformation, proxy, stateChange);
      SXM_UTILS_REGISTER_PROPERTY(_sxmAudioFiProxy, _sxmAudioDiagnosis, proxy, stateChange);
      SXM_UTILS_REGISTER_PROPERTY(_sxmAudioFiProxy, _sxmAudioList, proxy, stateChange);
      SXM_UTILS_REGISTER_PROPERTY(_sxmAudioFiProxy, _sxmAudioSportsFlash, proxy, stateChange);
      SXM_UTILS_REGISTER_PROPERTY(_sxmAudioFiProxy, _sxmAudioContentAlert, proxy, stateChange);
      SXM_UTILS_REGISTER_PROPERTY(_sxmAudioFiProxy, _sxmAudioTWNow, proxy, stateChange);
      SXM_UTILS_REGISTER_PROPERTY(_sxmAudioFiProxy, _sXMAudioPreset, proxy, stateChange);
      _onAvlAudio = true;
   }
   else if ((proxy == _sxmChannelartFiProxy) && (_sxmChannelartFiProxy))
   {
      SXM_UTILS_REGISTER_PROPERTY(_sxmChannelartFiProxy, _sXMAudioChannelArt, proxy, stateChange);
      _onAvlChnArt = true;
   }

   if (_sxmAudioFiProxy && proxy == _sxmAudioFiProxy)
   {
      ETG_TRACE_USR4(("Midw Audio service is available"));
      if (_sxmAudioPlayback)
      {
         _sxmAudioPlayback->sendTuneStartStatus();
      }
      if (_sXMAudioPreset)
      {
         _sXMAudioPreset->getPresetList();
      }
   }
   //Startup condition
   //_onAvlStartup - initally false, Once Audio proxy and ChnArt proxy is up, this is set true.
   //Service resumed condition, _onAvlStartup will be already true.
   //Based on proxy, audio resume functions or channel art resume function will be called
   if (_onAvlStartup)
   {
      if ((proxy == _sxmAudioFiProxy) && (_sxmAudioFiProxy))
      {
         SXM_UTILS_SERVICE_RESUMED(_sxmAudioFiProxy, _sXMAudioPreset, proxy);
         SXM_UTILS_SERVICE_RESUMED(_sxmAudioFiProxy, _sxmAudioPlayback, proxy);
         SXM_UTILS_SERVICE_RESUMED(_sxmAudioFiProxy, _sxmAudioList, proxy);
         SXM_UTILS_SERVICE_RESUMED(_sxmAudioFiProxy, _sxmAudioChannelInformation, proxy);
         SXM_UTILS_SERVICE_RESUMED(_sxmAudioFiProxy, _sxmAudioDiagnosis, proxy);
      }
      else if ((proxy == _sxmChannelartFiProxy) && (_sxmChannelartFiProxy))
      {
         SXM_UTILS_SERVICE_RESUMED(_sxmChannelartFiProxy, _sXMAudioChannelArt, proxy);
      }
   }
   if (_onAvlAudio && _onAvlChnArt)
   {
      _onAvlStartup = true;
   }
}


/**
 * Utility function that deregisters for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioHall::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy,
                                 const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("SxmAudioHall::onUnavailable"));
   if ((proxy == _sxmAudioFiProxy) && (_sxmAudioFiProxy))
   {
      SXM_UTILS_DEREGISTER_PROPERTY(_sxmAudioFiProxy, _sxmAudioPlayback, proxy, stateChange);
      SXM_UTILS_DEREGISTER_PROPERTY(_sxmAudioFiProxy, _sxmAudioChannelInformation, proxy, stateChange);
      SXM_UTILS_DEREGISTER_PROPERTY(_sxmAudioFiProxy, _sxmAudioDiagnosis, proxy, stateChange);
      SXM_UTILS_DEREGISTER_PROPERTY(_sxmAudioFiProxy, _sxmAudioList, proxy, stateChange);
      SXM_UTILS_DEREGISTER_PROPERTY(_sxmAudioFiProxy, _sxmAudioSportsFlash, proxy, stateChange);
      SXM_UTILS_DEREGISTER_PROPERTY(_sxmAudioFiProxy, _sxmAudioContentAlert, proxy, stateChange);
      SXM_UTILS_DEREGISTER_PROPERTY(_sxmAudioFiProxy, _sxmAudioTWNow, proxy, stateChange);
      SXM_UTILS_DEREGISTER_PROPERTY(_sxmAudioFiProxy, _sXMAudioPreset, proxy, stateChange);
      _onAvlAudio = false;
   }
   else if ((proxy == _sxmChannelartFiProxy) && (_sxmChannelartFiProxy))
   {
      SXM_UTILS_DEREGISTER_PROPERTY(_sxmChannelartFiProxy, _sXMAudioChannelArt, proxy, stateChange);
      _onAvlChnArt = false;
   }
}


/**
 * Utility function that deregisters for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioHall::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/,
                                        const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAudioHall::deregisterProperties"));
}


/*
* Function to perform restore factory setting for sxm audio
*/
bool SxmAudioHall::bSxmAudioRestoreFactorySetting()
{
   //todo: implementation for restore factory for sxm audio
   ETG_TRACE_USR4(("SxmAudioHall::bSxmAudioRestoreFactorySetting"));

   //Restore DP Value for Audio Alert
   SXM_UTILS_DELEGATE_REQUEST_WITH_0PARAM(_sxmAudioContentAlert, restoreAudioAlertDP);

   //Restore DP value for Audio Tune Start Status
   SXM_UTILS_DELEGATE_REQUEST_WITH_0PARAM(_sxmAudioPlayback, restoreAudioPlayback);

   //Restore DP value for Audio Active Preset Band
   SXM_UTILS_DELEGATE_REQUEST_WITH_0PARAM(_sXMAudioPreset, restoreAudioPreset);

   //Restore DP value for Audio Active Preset Band
   SXM_UTILS_DELEGATE_REQUEST_WITH_0PARAM(_sxmAudioChannelInformation, restoreChannelInfo);

   //Restore TWN info
   SXM_UTILS_DELEGATE_REQUEST_WITH_0PARAM(_sxmAudioTWNow, restoreAudioTWNInfo);

   //Restore SF info
   SXM_UTILS_DELEGATE_REQUEST_WITH_0PARAM(_sxmAudioSportsFlash, restoreAudioSFInfo);

   //Restore channel art info
   SXM_UTILS_DELEGATE_REQUEST_WITH_0PARAM(_sXMAudioChannelArt, restoreChannelArtInfo);

   return true;
}


/*
* Function to trigger handling for TTFis Command SxmAudio_GetChannelArt
*/
void SxmAudioHall::vGetChannelArt(uint8 testValue) const
{
   if (_sxmAudioList)
   {
      _sxmAudioList->vGetChannelArt(testValue);
   }
}


void SxmAudioHall::vonGuiStartupFinished()
{
   ETG_TRACE_USR4(("SxmAudioHall::vonGuiStartupFinished"));
   if (_sxmAudioChannelInformation)
   {
      _sxmAudioChannelInformation->onGuiStartupFinished();
   }
}


}  // namespace Core
}  // namespace App
