/**************************************************************************************
 * @file         : SxmAudioDiagnosis.cpp
 * @addtogroup   : AppHmi_Sxm
 * @brief        : Functional class to handle business logic for Sxm Diagnosis application.
 * @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 **************************************************************************************/
#include "hall_std_if.h"   // Including HALL framework code
#include "SxmAudioTypes.h"
#include "SxmAudioDiagnosis.h"
#include "SxmAudioListHelper.h"
#include "SxmAudioDataBindingUtils.h"
#include "App/Core/SxmUtils/SxmUtils.h"
#include "Common/DataModel/ListRegistry.h"
#include "HmiTranslation_TextIds.h"
#include "App/Core/LanguageDefines.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXMAUDIO
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::SxmAudioDiagnosis::
#include "trcGenProj/Header/SxmAudioDiagnosis.cpp.trc.h"
#endif

namespace App {
namespace Core {
using namespace ::sxm_audio_main_fi;
using namespace ::sxm_main_fi_types;
using namespace ::sxm_audio_main_fi_types;
/**
 * SxmAudioDiagnosis Class constructor
 * @param [in] : pSxmAudioFiProxy - SXM Audio FI proxy
 */
SxmAudioDiagnosis::SxmAudioDiagnosis(::boost::shared_ptr< Sxm_audio_main_fiProxy > pSxmAudioFiProxy)
   : _ptrSxmAudioFiProxy(pSxmAudioFiProxy)
   , _isExternalDiagmodeOn(true)
   , _isDataServiceInfoSubscribeOn(false)
   , _isAudioSubInfoSubscribeOn(false)
   , _isClearAllDBandData(false)


{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("SxmAudioDiagnosis() Constructor"));
}


/**
 * SxmAudioDiagnosis Class destructor
 */
SxmAudioDiagnosis::~SxmAudioDiagnosis()
{
   ETG_TRACE_USR4(("~SxmAudioDiagnosis() Destructor"));
   ETG_I_UNREGISTER_FILE();
   _ptrSxmAudioFiProxy.reset();
   _isExternalDiagmodeOn = true;
   _isDataServiceInfoSubscribeOn = false;
   _isAudioSubInfoSubscribeOn = false;
   _isClearAllDBandData = false;
}


/**
 * Utility function that registers for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioDiagnosis::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAudioDiagnosis::registerProperties"));
   if (_ptrSxmAudioFiProxy && (proxy == _ptrSxmAudioFiProxy))
   {
      //To Check : Register updates on entry to STM/DTM screens.
      _ptrSxmAudioFiProxy->sendSxmRadioIdUpReg(*this);
      _ptrSxmAudioFiProxy->sendSxmDiagAntennaStateUpReg(*this);
      _ptrSxmAudioFiProxy->sendSxmDiagSignalQualityUpReg(*this);
      _ptrSxmAudioFiProxy->sendSxmAudioSubInfoUpReg(*this);
      _ptrSxmAudioFiProxy->sendSxmDataServiceStatusUpReg(*this);
      _ptrSxmAudioFiProxy->sendDiagTestModeDataUpReg(*this);
      _ptrSxmAudioFiProxy->sendSxmSmsDbVersionsUpReg(*this);
      _ptrSxmAudioFiProxy->sendSxmModuleSubInfoUpReg(*this);
      _ptrSxmAudioFiProxy->sendSystemMessageUpReg(*this);
   }
}


/**
 * Utility function that deregisters for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioDiagnosis::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/) const
{
   ETG_TRACE_USR4(("SxmAudioDiagnosis::deregisterProperties"));
   if ((proxy == _ptrSxmAudioFiProxy) && (_ptrSxmAudioFiProxy))
   {
      //To Check : De-register updates on exit to STM/DTM screens.
      _ptrSxmAudioFiProxy->sendSxmRadioIdRelUpRegAll();
      _ptrSxmAudioFiProxy->sendSxmDiagAntennaStateRelUpRegAll();
      _ptrSxmAudioFiProxy->sendSxmDiagSignalQualityRelUpRegAll();
      _ptrSxmAudioFiProxy->sendSxmAudioSubInfoRelUpRegAll();
      _ptrSxmAudioFiProxy->sendSxmDataServiceStatusRelUpRegAll();
      _ptrSxmAudioFiProxy->sendDiagTestModeDataRelUpRegAll();
      _ptrSxmAudioFiProxy->sendSxmSmsDbVersionsRelUpRegAll();
      _ptrSxmAudioFiProxy->sendSxmModuleSubInfoRelUpRegAll();
      _ptrSxmAudioFiProxy->sendSystemMessageRelUpRegAll();
   }
}


/**
 * Error handling function for sxm Radio Id property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioDiagnosis::onSxmRadioIdError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::SxmRadioIdError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioDiagnosis::onSxmRadioIdError"));
}


/**
 * Status handling function for sxm Radio Id property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - Property status info
 */
void SxmAudioDiagnosis::onSxmRadioIdStatus(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::SxmRadioIdStatus >& status)
{
   ETG_TRACE_USR4(("SxmAudioDiagnosis::onSxmRadioIdStatus"));
   if (status && status->hasRadioId())
   {
      SxmAudioListHelper* pDiagnosisInstance = SxmAudioListHelper::instance();
      pDiagnosisInstance->setRadioData(status->getRadioId());
      pDiagnosisInstance->refreshSxmAudioList(LIST_ID_SXM_MONITOR, Sxm_Scenes_SERVICE__SXM_STM_MAIN);
      pDiagnosisInstance->refreshSxmAudioList(LIST_ID_SXM_AUDIO_SXM_INFOLIST, Sxm_Scenes_SETTINGS__RADIO_SXM);
   }
}


/**
 * Error handling function for  sxm diag antenna state
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioDiagnosis::onSxmDiagAntennaStateError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::SxmDiagAntennaStateError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioDiagnosis::onSxmDiagAntennaStateError"));
}


/**
 * Status handling function for  sxm diag antenna state
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - Property status info
 */
void SxmAudioDiagnosis::onSxmDiagAntennaStateStatus(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::SxmDiagAntennaStateStatus >& status)
{
   ETG_TRACE_USR4(("SxmAudioDiagnosis::onSxmDiagAntennaStateStatus"));
   if (status && status->hasAntennaStatus())
   {
      uint32 antennaStatus[Item_5] =
      {
         SXM__AUDIO_DIAG_WAIT,
         SXM__AUDIO_DIAG_OK,
         SXM__AUDIO_DIAG_DISCONNECTED,
         SXM__AUDIO_DIAG_SHORTED,
         SXM__AUDIO_DIAG_WAIT
      };
      SxmAudioListHelper* pDiagnosisInstance = SxmAudioListHelper::instance();
      pDiagnosisInstance->setDiagAntennaStatus(antennaStatus[status->getAntennaStatus()]);
      pDiagnosisInstance-> refreshSxmAudioList(LIST_ID_SXM_MONITOR, Sxm_Scenes_SERVICE__SXM_STM_MAIN);
   }
}


/**
 * Error handling function for sxm diag signal quality property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioDiagnosis::onSxmDiagSignalQualityError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi:: SxmDiagSignalQualityError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioDiagnosis::onSxmDiagSignalQualityError"));
}


/**
 * Status handling function for sxm diag signal quality property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - Property status info
 */
void SxmAudioDiagnosis::onSxmDiagSignalQualityStatus(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi:: SxmDiagSignalQualityStatus >& status)
{
   ETG_TRACE_USR4(("SxmAudioDiagnosis::onSxmDiagSignalQualityStatus"));
   if (status && status->hasSignalQualityStatus())
   {
      uint32 signalQuality[Item_5] =
      {
         SXM__AUDIO_DIAG_NOSIGNAL,
         SXM__AUDIO_DIAG_WEAK,
         SXM__AUDIO_DIAG_GOOD,
         SXM__AUDIO_DIAG_EXCELLENT,
         SXM__AUDIO_DIAG_WAIT
      };
      SxmAudioListHelper* pDiagnosisInstance = SxmAudioListHelper::instance();
      pDiagnosisInstance->setDiagSignalQuality(signalQuality[status->getSignalQualityStatus()]);
      pDiagnosisInstance-> refreshSxmAudioList(LIST_ID_SXM_MONITOR, Sxm_Scenes_SERVICE__SXM_STM_MAIN);
      pDiagnosisInstance-> refreshSxmAudioList(LIST_ID_SXM_AUDIO_SXM_INFOLIST, Sxm_Scenes_SETTINGS__RADIO_SXM);
	  pDiagnosisInstance-> refreshSxmAudioList(LIST_ID_SXM_AUDIO_MENU, Sxm_Scenes_SXM_RADIO__SETTING_OPTION);
   }
}


/**
 * Error handling function for sxm audio sub info property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioDiagnosis::onSxmAudioSubInfoError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::SxmAudioSubInfoError >& /* error*/)
{
   ETG_TRACE_ERR(("SxmAudioDiagnosis::onSxmAudioSubInfoError"));
}


/**
 * Status handling function for sxm audio sub info property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - Property status info
 */
void SxmAudioDiagnosis::onSxmAudioSubInfoStatus(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::SxmAudioSubInfoStatus >& status)
{
   if (status && status->hasData())
   {
      ETG_TRACE_USR4(("SxmAudioDiagnosis::onSxmAudioSubInfoStatus status:%d", status->getData().getSubStatus()));
      uint32 subStatus[Item_5] =
      {
         SXM__AUDIO_DIAG_NOTSUBSCRIBED,
         SXM__AUDIO_DIAG_SUBSCRIBED,
         SXM__AUDIO_DIAG_SUSPENDALERT,
         SXM__AUDIO_DIAG_SUSPENDED,
         SXM__AUDIO_DIAG_INVALID
      };
      SxmAudioListHelper* pDiagnosisInstance = SxmAudioListHelper::instance();
      pDiagnosisInstance->setDiagAudioSubInfo(subStatus[status->getData().getSubStatus()]);
      std::string StrStatus = SxmUtils::safeGetCString(subStatus[status->getData().getSubStatus()]);
      if (StrStatus == SxmUtils::safeGetCString(SXM__AUDIO_DIAG_AUDIONOTSUBSCRIBED))
      {
         _isAudioSubInfoSubscribeOn = true;
         // SxmAudioDataBindingUtils::instance()->setSxmInfoButtonEnable(_isInfoSubscribeOn);
      }
      else
      {
         _isAudioSubInfoSubscribeOn = false;
         // SxmAudioDataBindingUtils::instance()->setSxmInfoButtonEnable(_isInfoSubscribeOn);
      }
      setSubscribeButton();
      pDiagnosisInstance-> refreshSxmAudioList(LIST_ID_SXM_MONITOR, Sxm_Scenes_SERVICE__SXM_STM_MAIN);
      pDiagnosisInstance-> refreshSxmAudioList(LIST_ID_SXM_AUDIO_SXM_INFOLIST, Sxm_Scenes_SETTINGS__RADIO_SXM);
      POST_MSG((COURIER_MESSAGE_NEW(SXMAudioSportsRefreshSFStatusReqMsg)(status->getData().getSubStatus())));
   }
}


/**
 * Error handling function for Sxm Data Service Status
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioDiagnosis::onSxmDataServiceStatusError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::SxmDataServiceStatusError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioDiagnosis::onSxmDataServiceStatusError"));
}


/**
 * Status handling function for Sxm Data Service Status
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - Property status info
 */
void SxmAudioDiagnosis::onSxmDataServiceStatusStatus(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi:: SxmDataServiceStatusStatus >& status)
{
   ETG_TRACE_USR4(("SxmAudioDiagnosis::onSxmDataServiceStatusStatus"));
   if (status && status->hasList())
   {
      SxmAudioListHelper* pDiagnosisInstance = SxmAudioListHelper::instance();
      pDiagnosisInstance->setDiagDataServiceStatus(status->getList());

      _isDataServiceInfoSubscribeOn = SxmAudioListHelper::instance()->getInfoTrafficStatus() || SxmAudioListHelper::instance()->getInfoTravellinkStatus();
      //SxmAudioDataBindingUtils::instance()->setSxmInfoButtonEnable(_InfoButtonStatus);

      setSubscribeButton();
      pDiagnosisInstance-> refreshSxmAudioList(LIST_ID_SXM_MONITOR, Sxm_Scenes_SERVICE__SXM_STM_MAIN);
      pDiagnosisInstance-> refreshSxmAudioList(LIST_ID_SXM_AUDIO_SXM_INFOLIST, Sxm_Scenes_SETTINGS__RADIO_SXM);
   }
}


/**
 * Error handling function for system messages property
 */
void SxmAudioDiagnosis::onSystemMessageError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< ::sxm_audio_main_fi::SystemMessageError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioDiagnosis::onSystemMessageError"));
}


/**
 * Status handling function for system messages property
 */
void SxmAudioDiagnosis::onSystemMessageStatus(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< ::sxm_audio_main_fi::SystemMessageStatus >& status)
{
   ETG_TRACE_USR4(("SxmAudioDiagnosis::onSystemMessageStatus"));
   if (status)
   {
      T_e8_SystemMessage systemErrorType = status->getSystemErrorType();
      if (T_e8_SystemMessage__XMTUN_SYSTEM_MSG_ANTENNA_DISCONNECTED == systemErrorType || T_e8_SystemMessage__XMTUN_SYSTEM_MSG_NO_SIGNAL == systemErrorType
            || T_e8_SystemMessage__XMTUN_SYSTEM_MSG_NO_ERROR == systemErrorType)
      {
         SxmAudioListHelper* pDiagnosisInstance = SxmAudioListHelper::instance();
         pDiagnosisInstance->setSxmTravelLink();
         pDiagnosisInstance-> refreshSxmAudioList(LIST_ID_SXM_MONITOR, Sxm_Scenes_SERVICE__SXM_STM_MAIN);
         pDiagnosisInstance-> refreshSxmAudioList(LIST_ID_SXM_AUDIO_SXM_INFOLIST, Sxm_Scenes_SETTINGS__RADIO_SXM);
      }
   }
}


/**
 * Error handling function for diag test mode data property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioDiagnosis::onDiagTestModeDataError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::DiagTestModeDataError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioDiagnosis::onDiagTestModeDataError"));
}


/**
 * Status handling function for diag test mode data property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - Property status info
 */
void SxmAudioDiagnosis::onDiagTestModeDataStatus(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::DiagTestModeDataStatus >& status)
{
   ETG_TRACE_USR4(("SxmAudioDiagnosis::onDiagTestModeDataStatus"));
   if (status)
   {
      SxmAudioListHelper* pDiagnosisInstance = SxmAudioListHelper::instance();
      pDiagnosisInstance->setDiagTestModeData(status->getData());
      pDiagnosisInstance->setDiagSxmVersion();
      pDiagnosisInstance-> refreshSxmAudioList(LIST_ID_SXM_DTM_TESTMODE_MONITOR_SELECTION, Sxm_Scenes_SERVICE__SXM_DEV_TESTMODE_MONITOR);
      pDiagnosisInstance-> refreshSxmAudioList(LIST_ID_SXM_MONITOR, Sxm_Scenes_SERVICE__SXM_STM_MAIN);
   }
}


/**
 * Error handling function for sxm sms db version property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioDiagnosis::onSxmSmsDbVersionsError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::SxmSmsDbVersionsError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioDiagnosis::onSxmSmsDbVersionsError"));
}


/**
 * Status handling function for sxm sms db version property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - Property status info
 */
void SxmAudioDiagnosis::onSxmSmsDbVersionsStatus(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::SxmSmsDbVersionsStatus >& status)
{
   ETG_TRACE_USR4(("SxmAudioDiagnosis::onSxmSmsDbVersionsStatus"));
   if (status)
   {
      SxmAudioListHelper* pDiagnosisInstance = SxmAudioListHelper::instance();
      pDiagnosisInstance->setDiagSmsDBVersion(status->getData());
      pDiagnosisInstance-> refreshSxmAudioList(LIST_ID_SXM_DTM_TESTMODE_MONITOR_SELECTION, Sxm_Scenes_SERVICE__SXM_DEV_TESTMODE_MONITOR);
   }
}


/**
 * Error handling function for diag test mode enable method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioDiagnosis:: onDiagTestModeEnableError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::DiagTestModeEnableError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioDiagnosis::onDiagTestModeEnableError"));
}


/**
 * Result handling function for diag test mode enable method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioDiagnosis:: onDiagTestModeEnableResult(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::DiagTestModeEnableResult >& /*result*/)
{
   ETG_TRACE_USR4(("SxmAudioDiagnosis::onDiagTestModeEnableResult"));
}


/**
 * Error handling function for diag sxm reset method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioDiagnosis:: onDiagSxmResetError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi:: DiagSxmResetError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioDiagnosis::onDiagSxmResetError"));
}


/**
 * Result handling function for diag sxm reset method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioDiagnosis:: onDiagSxmResetResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< DiagSxmResetResult >& /*result*/)
{
   ETG_TRACE_USR4(("SxmAudioDiagnosis::onDiagSxmResetResult"));
   if (_isClearAllDBandData)
   {
      POST_MSG((COURIER_MESSAGE_NEW(SxmClearAllDBdataUpdtMsg)())); // to clear All datapool & User data
      _isClearAllDBandData = false;
   }
   POST_MSG((COURIER_MESSAGE_NEW(SXMDeActivatePopUpMsg)(Sxm_SxmPopups_SERVICE__SXM_DTM_POPUP_SYSIND)));
}


/**
 *  Error handling function for ExternalDiagnosisMode method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioDiagnosis::onExternalDiagnosisModeError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::ExternalDiagnosisModeError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioDiagnosis::onExternalDiagnosisModeError"));
}


/**
 * Result handling function for ExternalDiagnosisMode method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioDiagnosis:: onExternalDiagnosisModeResult(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::ExternalDiagnosisModeResult >& /*result*/)
{
   ETG_TRACE_USR4(("SxmAudioDiagnosis::onExternalDiagnosisModeResult"));
   POST_MSG((COURIER_MESSAGE_NEW(SXMDeActivatePopUpMsg)(Sxm_SxmPopups_SERVICE__SXM_DTM_POPUP_SYSIND)));
}


/**
 * Callback to handle Courier Message SxmDTMTestModeMonitorSelReqMsg Request
 * @param [in] : Reference of Courier::SxmDTMTestModeMonitorSelReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioDiagnosis:: onCourierMessage(const SxmDTMTestModeMonitorSelReqMsg& oMsg)
{
   ETG_TRACE_USR4(("SxmAudioDiagnosis::SxmDTMTestModeMonitorSelReqMsg"));
   if (_ptrSxmAudioFiProxy)
   {
      _ptrSxmAudioFiProxy->sendDiagTestModeEnableStart(*this, oMsg.GetStartMonitor());
   }
   return true;
}


/**
 *  Error handling function for ExternalDiagnosisMode method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioDiagnosis:: onSxmModuleSubInfoError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::SxmModuleSubInfoError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioDiagnosis::onSxmModuleSubInfoError"));
}


/**
 * Result handling function for ExternalDiagnosisMode method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioDiagnosis:: onSxmModuleSubInfoStatus(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::SxmModuleSubInfoStatus >& status)
{
   ETG_TRACE_USR4(("SxmAudioDiagnosis::onSxmModuleSubInfoStatus"));
   if (status)
   {
      SxmAudioListHelper* pDiagnosisInstance = SxmAudioListHelper::instance();
      pDiagnosisInstance->setModuleSubInfo(status->getData());
      pDiagnosisInstance-> refreshSxmAudioList(LIST_ID_SXM_DTM_TESTMODE_MONITOR_SELECTION, Sxm_Scenes_SERVICE__SXM_DEV_TESTMODE_MONITOR);
      pDiagnosisInstance-> refreshSxmAudioList(LIST_ID_SXM_MONITOR, Sxm_Scenes_SERVICE__SXM_STM_MAIN);
   }
}


/**
* Error handling function for ResetSXMModule method
*/
void SxmAudioDiagnosis::onResetSXMModuleError(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::ResetSXMModuleError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioDiagnosis::onResetSXMModuleError"));
}


/**
* Result handling function for ResetSXMModule method
*/
void SxmAudioDiagnosis::onResetSXMModuleResult(const ::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::sxm_audio_main_fi::ResetSXMModuleResult >& result)
{
   if (result)
   {
      ETG_TRACE_USR4(("SxmAudioDiagnosis::onResetSXMModuleResult Status:%d", result->getStatus()));
   }
}


/**
 * Callback to handle Courier Message SXMSettingsTestModeListReqMsg Request
 * @param [in] : Reference of Courier::SXMSettingsTestModeListReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioDiagnosis::onCourierMessage(const SXMSettingsTestModeListReqMsg& oMsg)
{
   ETG_TRACE_USR4(("SxmAudioDiagnosis::SXMSettingsTestModeListReqMsg"));

   if (oMsg.GetTestmodeListId() == LIST_ID_SXM_DTM_MAIN)
   {
      const char* serviceName[] =
      {
         "Clearing All DB & User Data. Please Wait...",
         "Clearing CGS. Please Wait...",
         "Resetting SiriusXM Module. Please Wait...",
         "Updating External Diag Mode. Please Wait...",
         "Clearing Tab Weather. Please Wait...",
         "Clearing WS Alerts. Please Wait...",
         "Clearing Fuel(US). Please Wait...",
         "Clearing Fuel(Canada). Please Wait...",
         "Clearing Movie. Please Wait...",
         "Clearing Parking. Please Wait...",
         "Clearing Sports. Please Wait...",
         "Clearing Stocks Data. Please Wait...",
      };

      uint8 testmodeService[] =
      {
         SXM_TESTMODE_COMPLETE_DEFAULT_SETTING,
         SXM_TESTMODE_RESET_CGS,
         SXM_TESTMODE_RESET_TABWEATHER,
         SXM_TESTMODE_WS_ALERTS,
         SXM_TESTMODE_RESET_FUEL_US,
         SXM_TESTMODE_RESET_FUEL_CANADA,
         SXM_TESTMODE_RESET_MOVIE,
         SXM_TESTMODE_PARKING,
         SXM_TESTMODE_SPORTS,
         SXM_TESTMODE_STOCKS,
         SXM_TESTMODE_EXTERNAL_SUBSTATUS
      };
      if (_ptrSxmAudioFiProxy)
      {
         uint8 idx = oMsg.GetHdl();
         if (Item_2 == idx)
         {
            _ptrSxmAudioFiProxy->sendResetSXMModuleStart(*this);
         }
         else if (Item_3 == idx)
         {
            externalDiagSwitch(oMsg.GetTestmodeListId(), Sxm_Scenes_SERVICE__SXM_DTM_MAIN);
            SxmAudioDataBindingUtils::instance()->updateSxmSettingTestmodeUpdatePopup(serviceName[idx]);
         }
         else
         {
            if (Item_0 == idx)
            {
               _isClearAllDBandData = true;
            }
            _ptrSxmAudioFiProxy->sendDiagSxmResetStart(*this, testmodeService[idx]);//used only for first 10 items
            SxmAudioDataBindingUtils::instance()->updateSxmSettingTestmodeUpdatePopup(serviceName[idx]);
         }
      }
   }
   return true;
}


/**
 * Helper function start External Diagnosis Mode
 * @param [in] : lstId
 * @param [in] : sceneid
 */
void SxmAudioDiagnosis::externalDiagSwitch(unsigned int lstId, unsigned int sceneId)
{
   ETG_TRACE_USR4(("SxmAudioDiagnosis::externalDiagSwitch"));
   SxmAudioListHelper* pDiagnosisInstance = SxmAudioListHelper::instance();
   _isExternalDiagmodeOn = !_isExternalDiagmodeOn;
   T_e8_SxmExtDiagMode diagState = (_isExternalDiagmodeOn) ? T_e8_SxmExtDiagMode__FC_SXM_EXTERNAL_DIAG_MODE_EXIT : T_e8_SxmExtDiagMode__FC_SXM_EXTERNAL_DIAG_MODE_START;
   pDiagnosisInstance->setDiagStatus(_isExternalDiagmodeOn);

   if (_ptrSxmAudioFiProxy)
   {
      _ptrSxmAudioFiProxy->sendExternalDiagnosisModeStart(*this, diagState);
   }

   pDiagnosisInstance-> refreshSxmAudioList(lstId, sceneId);
}


/**
     * Helper function to update the subscribe button.
     */
void SxmAudioDiagnosis::setSubscribeButton() const
{
   SxmAudioDataBindingUtils::instance()->setSxmInfoButtonEnable(_isAudioSubInfoSubscribeOn || _isDataServiceInfoSubscribeOn);
}


/**
 * Function to set default value on service availability from suspended
 */
void SxmAudioDiagnosis::onServiceResumed(const ::boost::shared_ptr< asf::core::Proxy >& proxy)
{
   ETG_TRACE_USR4(("SxmAudioDiagnosis::onServiceResumed"));
   if (_ptrSxmAudioFiProxy && (proxy == _ptrSxmAudioFiProxy))
   {
      POST_MSG((COURIER_MESSAGE_NEW(SXMDeActivatePopUpMsg)(Sxm_SxmPopups_SERVICE__MPOP_SXM_DTM_RESET_SXM_MODULE)));
   }
}


}  // namespace Core
}  // namespace App
