/**************************************************************************************
 * @file         : SxmAudioConetAlert.cpp
 * @addtogroup   : AppHmi_Sxm
 * @brief        : Functional class to handle business logic for Content Alerts.
 * @copyright    : (c) 2019-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 **************************************************************************************/
#include "hall_std_if.h" // Including HALL framework code
#include "SxmAudioContentAlert.h"
#include "SxmAudioDataBindingUtils.h"
#include "SxmAudioListHelper.h"
#include "App/Core/SxmUtils/SxmUtils.h"
#include "App/Core/SxmCommon/SXMCommonInterface.h"
#include "HmiTranslation_TextIds.h"
#include "App/Core/LanguageDefines.h"
#include "App/Core/SxmUtils/SxmUtilsListHandler.h"
#include "App/Core/ClientHandler/SourceSwitch/SXMSourceSwitchInterface.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXMAUDIO
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::SxmAudioContentAlert::
#include "trcGenProj/Header/SxmAudioContentAlert.cpp.trc.h"
#endif

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_hmi_06_if.h"

namespace App {
namespace Core {
using namespace ::sxm_audio_main_fi;
using namespace ::sxm_audio_main_fi_types;

/**
 *SxmAudioContentAlert class constructor
 *@param [in] : pSxmAudioFiProxy - SXM Audio FI proxy
 */
SxmAudioContentAlert::SxmAudioContentAlert(::boost::shared_ptr< ::sxm_audio_main_fi::Sxm_audio_main_fiProxy > pSxmAudioFiProxy
      , ISxmAudioChannelArtRequest* pSxmChannelart)
   : _sxmAudioFiProxy(pSxmAudioFiProxy)
   , _sxmChannelart(pSxmChannelart)
   , _favtype(T_e8_FavoriteType__FAVORITE_ARTIST)
   , _maxFavReached(false)
   , _isReplaceFav(false)
   , _isFavoriteAlertActive(false)
   , _transtionId(0)
   , _deleteAllFavType(0)
   , _cMaxfavCount(15)
   , _allowContentAlert(true)
   , _popupCloseOnAppmodeChange(false)
   , _artistFavCount(0)
   , _songFavCount(0)
   , _isAlertAllowed(true)
   , _isRegisterProperties(false)
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("SxmAudioContentAlert() Constructor"));
//#include "//Common/ListHandler/LockoutRestrictionData.h"
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_AUDIO_FAVORITE_ARTISTLIST, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_AUDIO_FAVORITE_SONGLIST, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_AUDIO_ARTISTLIST, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_AUDIO_SONGLIST, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_AUDIO_ALERTLIST, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SXM_AUDIO_FAVORITE_REPLACELIST, this);
   _contentAlertList.clear();

   SXMCommonInterface::registerApplicationModeChangeNotication(this, SXMContentAlert);

   dp_tclAppHmi_SxmSXMAudioAlert dp_AudioAlert;
   _alertSettingsStatus = dp_AudioAlert.tGetData();
}


/**
 * SxmAudioContentAlert class destructor
 */
SxmAudioContentAlert::~SxmAudioContentAlert()
{
   ETG_TRACE_USR4(("~SxmAudioContentAlert() Destructor"));
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_AUDIO_FAVORITE_ARTISTLIST);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_AUDIO_FAVORITE_SONGLIST);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_AUDIO_ARTISTLIST);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_AUDIO_SONGLIST);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_AUDIO_ALERTLIST);
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_SXM_AUDIO_FAVORITE_REPLACELIST);
   SXMCommonInterface::deregisterApplicationModeChangeNotication(this, SXMContentAlert);
   _sxmChannelart = NULL;
   _sxmAudioFiProxy.reset();
   _favtype = T_e8_FavoriteType__FAVORITE_ARTIST;
   _maxFavReached = false;
   _isReplaceFav = false;
   _isFavoriteAlertActive = false;
   _transtionId = 0;
   _deleteAllFavType = 0;
   _artistFavCount = 0;
   _songFavCount = 0;
   _isAlertAllowed = true;
   _isRegisterProperties = false;
   ETG_I_UNREGISTER_FILE();
}


/**
 * Utility function that registers for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioContentAlert::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAudioContentAlert::registerProperties"));

   if ((proxy == _sxmAudioFiProxy) && (_sxmAudioFiProxy))
   {
      _isRegisterProperties = true;
      //after restore factory, onAvailable received
      //upreg requested and alert from ASF queue is received and displayed.
      //so getting count and handling
      _favtype = T_e8_FavoriteType__FAVORITE_ARTIST;
      getFavList();
      //After getting the count, Upreg will be done
   }
}


/**
 * Utility function that deregisters for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioContentAlert::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAudioContentAlert::deregisterProperties"));
   if ((proxy == _sxmAudioFiProxy) && (_sxmAudioFiProxy))
   {
      _sxmAudioFiProxy->sendAlertNoticeListUpdateRelUpRegAll();
      _sxmAudioFiProxy->sendFavoriteContentAlertInfoRelUpRegAll();
      _artistFavCount = 0;
      _songFavCount = 0;
      _isAlertAllowed = false;//TODO: to  be done in restore
   }
}


/**
 *Error handling function for Alert Notice ListUpdate
 *@param [in] : proxy - SXM Audio FI proxy class instance
 *@param [in] : error - error TYPE info
 */
void SxmAudioContentAlert::onAlertNoticeListUpdateError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< AlertNoticeListUpdateError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmAudioContentAlert::onAlertNoticeListUpdateError"));
}


/**
 *Status handling function for Alert Notice ListUpdate
 *@param [in] : proxy - SXM Audio FI proxy class instance
 *@param [in] : status - Property status info
 */
void SxmAudioContentAlert::onAlertNoticeListUpdateStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< AlertNoticeListUpdateStatus >& status)
{
   ETG_TRACE_USR4(("SxmAudioContentAlert::onAlertNoticeListUpdateStatus"));
   if (status && status->getUpdateAvailable() && _sxmAudioFiProxy)
   {
      uint32 lSceneID = SXMCommonInterface::getActiveViewName(Enum_SURFACEID_MAIN_SURFACE_SXM);
      //Process updates only when respective Alert list is active.
      if (Sxm_Scenes_SXM_RADIO__ALERT_ARTISTLIST == lSceneID && T_e8_FavoriteType__FAVORITE_ARTIST == status->getFavoriteType())
      {
         _favtype = T_e8_FavoriteType__FAVORITE_ARTIST;
         _sxmAudioFiProxy->sendGetAlertNoticeListStart(*this, _favtype);
      }
      else if (Sxm_Scenes_SXM_RADIO__ALERT_SONGLIST == lSceneID && T_e8_FavoriteType__FAVORITE_SONG == status->getFavoriteType())
      {
         _favtype = T_e8_FavoriteType__FAVORITE_SONG;
         _sxmAudioFiProxy->sendGetAlertNoticeListStart(*this, _favtype);
      }
   }
}


/**
 *Error handling function for Favorite Content Alert Info
 *@param [in] : proxy - SXM Audio FI proxy class instance
 *@param [in] : error - error TYPE info
 */
void SxmAudioContentAlert::onFavoriteContentAlertInfoError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< FavoriteContentAlertInfoError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmAudioContentAlert::onFavoriteContentAlertInfoError"));
}


/**
 * Status handling function for Favorite Content Alert Info
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - Property status info
 */
void SxmAudioContentAlert::onFavoriteContentAlertInfoStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< FavoriteContentAlertInfoStatus >& status)
{
   ETG_TRACE_USR1(("SxmAudioContentAlert::onFavoriteContentAlertInfoStatus"));
   if (status)
   {
      ETG_TRACE_USR2(("SxmAudioContentAlert::onFavoriteContentAlertInfoStatus favtype:%d, IsExpired:%d, status->getChannelID: %d",
                      status->getAlertInfo().getFavoriteType(), status->getAlertInfo().getIsExpired(), status->getAlertInfo().getChannelID()));
      uint16 lchnid = status->getAlertInfo().getChannelID();
      if ((lchnid <= END_CHANNEL_ID) && (lchnid >= START_CHANNEL_ID))
      {
         ETG_TRACE_USR3(("SxmAudioContentAlert::onFavoriteContentAlertInfoStatus artistCount:%u, songCount:%u, AlertAllowed:%u",
                         _artistFavCount, _songFavCount, _isAlertAllowed));
         bool isPopupAllowed = (((_artistFavCount != 0) || (_songFavCount != 0)) && (_isAlertAllowed));
         if (!_isAlertAllowed)
         {
            _isAlertAllowed = true;
         }
         if (!(SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__ALERT_ARTISTLIST) &&
               T_e8_FavoriteType__FAVORITE_ARTIST == status->getAlertInfo().getFavoriteType())  &&
               !(SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__ALERT_SONGLIST) &&
                 T_e8_FavoriteType__FAVORITE_SONG == status->getAlertInfo().getFavoriteType()) &&
               !(status->getAlertInfo().getIsExpired()) && isPopupAllowed)
         {
            //Queue alerts only if channel is valid & tunable.
            enQueueContentAlerts(status->getAlertInfo());
         }
         else if (status->getAlertInfo().getIsExpired())
         {
            std::vector<T_AlertInfo>::iterator itrAlert;
            for (itrAlert = _contentAlertList.begin(); itrAlert != _contentAlertList.end(); ++itrAlert)
            {
               ETG_TRACE_USR4(("SxmAudioContentAlert onFavoriteContentAlertInfoStatus itrAlert->getFavoriteType():%d", itrAlert->getFavoriteType()));
               if (itrAlert->getFavoriteType() == status->getAlertInfo().getFavoriteType())
               {
                  if (((T_e8_FavoriteType__FAVORITE_ARTIST == itrAlert->getFavoriteType()) && (itrAlert->getArtistName() == status->getAlertInfo().getArtistName())) ||
                        ((T_e8_FavoriteType__FAVORITE_SONG == itrAlert->getFavoriteType()) && (itrAlert->getSongName() == status->getAlertInfo().getSongName())))
                  {
                     ETG_TRACE_USR4(("Erased expired alert for FavType: %d", itrAlert->getFavoriteType()));
                     _contentAlertList.erase(itrAlert);
                     break;
                  }
               }
            }
         }
      }
   }
}


/**
 *Error handling function for Get Alert Notice List
 *@param [in] : proxy - SXM Audio FI proxy class instance
 *@param [in] : error - error TYPE info
 */
void SxmAudioContentAlert::onGetAlertNoticeListError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< GetAlertNoticeListError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmAudioContentAlert::onGetAlertNoticeListError"));
}


/**
 * Result handling function for Get Alert Notice List
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioContentAlert::onGetAlertNoticeListResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /* proxy*/,
      const ::boost::shared_ptr< GetAlertNoticeListResult >& result)
{
   if (result)
   {
      ETG_TRACE_USR4(("SxmAudioContentAlert::onGetAlertNoticeListResult alert size:%d", result->getAlertInfoList().size()));
      SxmAudioListHelper* pAudioUtilsInstance = SxmAudioListHelper::instance();

      //Set alert info for tunable channels.
      pAudioUtilsInstance->setAlertInfoList(result->getAlertInfoList());

      //to switch screen on channel art result.
      _transtionId = (_favtype == T_e8_FavoriteType__FAVORITE_ARTIST) ?
                     Sxm_Scenes_SXM_RADIO__ALERT_ARTISTLIST : Sxm_Scenes_SXM_RADIO__ALERT_SONGLIST;

      if (_sxmChannelart && pAudioUtilsInstance->getAlertInfoList().size())
      {
         _sxmChannelart->sendChannelArtListMethodRequest(this,
               pAudioUtilsInstance->getListOfSid(pAudioUtilsInstance->getAlertInfoList(), 0, pAudioUtilsInstance->getAlertInfoList().size()), 1); //1:Only channel logo.
         //Clear advisory
         SxmAudioDataBindingUtils::instance()->setAlertsAdvisory(true, "");
      }
      else
      {
         SxmAudioDataBindingUtils::instance()->setAlertsAdvisory(false, getAdvisoryText());
         POST_MSG((COURIER_MESSAGE_NEW(SxmSceneTransitionReqMsg)(_transtionId)));
         _transtionId = 0; //Clear after transition is handled.
      }
   }
}


/**
 * Error handling function for Get Favorite List
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error - error TYPE info
 */
void SxmAudioContentAlert::onGetFavoriteListError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /* proxy*/,
      const ::boost::shared_ptr< GetFavoriteListError >& /* error*/)
{
   ETG_TRACE_USR4(("SxmAudioContentAlert::onGetFavoriteListError"));
   //Disable buttons.
   SxmAudioDataBindingUtils::instance()->setSxmAudioFavButtonEnable(false);
   //To Check: advisory handling in error cases.
}


/**
 *Result handling function for Get Favorite List
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioContentAlert::onGetFavoriteListResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< GetFavoriteListResult >& result)
{
   ETG_TRACE_USR1(("SxmAudioContentAlert::onGetFavoriteListResult"));
   if (result)
   {
      SxmAudioListHelper::instance()->setFavoriteList(result->getRegisteredFavoriteList());
      uint8 favsize = result->getRegisteredFavoriteList().size();
      ETG_TRACE_USR2(("SxmAudioContentAlert::onGetFavoriteListResult favsize = %d", favsize));
      storeFavoriteStatus(favsize);
      if (_favtype == T_e8_FavoriteType__FAVORITE_ARTIST)
      {
         _artistFavCount = favsize;
         if (_isRegisterProperties)
         {
            _favtype = T_e8_FavoriteType__FAVORITE_SONG;
            getFavList();
         }
      }
      else if (_favtype == T_e8_FavoriteType__FAVORITE_SONG)
      {
         _songFavCount = favsize;
         if (_isRegisterProperties && _sxmAudioFiProxy)
         {
            _sxmAudioFiProxy->sendAlertNoticeListUpdateUpReg(*this);
            _sxmAudioFiProxy->sendFavoriteContentAlertInfoUpReg(*this);
            _isRegisterProperties = false;
         }
      }
      ETG_TRACE_USR3(("SxmAudioContentAlert::onGetFavoriteListResult artistCount:%u, songCount:%u", _artistFavCount, _songFavCount));

      if (!favsize)
      {
         //To disable "Turn All Artist/Song" & "Delete All Artist" buttons.
         SxmAudioDataBindingUtils::instance()->setSxmAudioFavButtonEnable(false);
      }
      else
      {
         //To toggle "Turn All Artist/Song on/OFF button
         uint8 buttonStatus = DISABLE;
         for (uint8 favidx = 0; favidx < favsize; ++favidx)
         {
            if (result->getRegisteredFavoriteList()[favidx].getIsEnabled() == false)
            {
               buttonStatus = ENABLE;
               break;
            }
         }
         SxmAudioDataBindingUtils::instance()->updateOnOffButtonStatus((uint8)_favtype, buttonStatus);
         SxmAudioDataBindingUtils::instance()->setSxmAudioFavButtonEnable(true);
      }
      updateFavoriteList();
   }
}


/**
 *Error handling function for Modify Favorite
 *@param [in] : proxy - SXM Audio FI proxy class instance
 *@param [in] : error - error TYPE info
 */
void SxmAudioContentAlert::onModifyFavoriteError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ModifyFavoriteError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmAudioContentAlert::onModifyFavoriteError"));
}


/**
 *Result handling function for Modify Favorite
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioContentAlert::onModifyFavoriteResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ModifyFavoriteResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioContentAlert::onModifyFavoriteResult"));
   if (result)
   {
      _favtype = result->getFavoriteType();
      handleModifyFavoriteList(result->getAction());
   }
}


/**
 *Error handling function for Save Favorite
 *@param [in] : proxy - SXM Audio FI proxy class instance
 *@param [in] : error - error TYPE info
 */
void SxmAudioContentAlert::onSaveFavoriteError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< SaveFavoriteError >& /*error*/)
{
   ETG_TRACE_USR4(("SxmAudioContentAlert::onSaveFavoriteError"));
}


/**
 *Result handling function for Save Favorite
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioContentAlert::onSaveFavoriteResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< SaveFavoriteResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioContentAlert::onSaveFavoriteResult"));
   if (result)
   {
      if (T_e8_FavoriteStatus__CONTENT_MAX_FAVORITES_REACHED == result->getStatus())
      {
         _maxFavReached = true;//To be removed
         //POST_MSG((COURIER_MESSAGE_NEW(SxmAudioFavMaxReachedPopupReqMsg)()));
         POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__POPUP_ALERT_ARTIST_SONG_SYSDLG)));
      }
      else
      {
         setSaveResult(result->getStatus(), result->getFavoriteCount(), result->getContentName());
         if (result->getFavoriteType() == T_e8_FavoriteType__FAVORITE_ARTIST)
         {
            _artistFavCount = result->getFavoriteCount();
         }
         else if (result->getFavoriteType() == T_e8_FavoriteType__FAVORITE_SONG)
         {
            _songFavCount = result->getFavoriteCount();
         }
      }
   }
}


/**
 *Helper Function to Get Save Favorite Result
 *@param [in] : uStatus Method result data
 */
void SxmAudioContentAlert::setSaveResult(uint8 status, uint8 favcount, std::string contentName)
{
   bool isTwoLine = false;
   std::string PopupStatus = "";
   std::ostringstream Favoritecount;
   std::ostringstream Emptycount;
   Favoritecount << (0 + favcount);
   Emptycount << (_cMaxfavCount  - favcount);
   switch (status)
   {
      case T_e8_FavoriteStatus__CONTENT_REGISTERED:
      {
         isTwoLine = true;
         if (T_e8_FavoriteType__FAVORITE_ARTIST == _favtype)
         {
            PopupStatus = SxmUtils::safeGetCString(SXM__AUDIO_POPUP_ARTISTADDED);
            PopupStatus = SxmUtils::replaceInTranslationTextID(PopupStatus, contentName, "Artist");
            PopupStatus = SxmUtils::replaceInTranslationTextID(PopupStatus, Favoritecount.str(), "%1");
            PopupStatus = SxmUtils::replaceInTranslationTextID(PopupStatus, Emptycount.str(), "%2");
         }
         else if (T_e8_FavoriteType__FAVORITE_SONG == _favtype)
         {
            PopupStatus = SxmUtils::safeGetCString(SXM__AUDIO_POPUP_SONGADDED);
            PopupStatus = SxmUtils::replaceInTranslationTextID(PopupStatus, contentName, "Song");
            PopupStatus = SxmUtils::replaceInTranslationTextID(PopupStatus, Favoritecount.str(), "%1");
            PopupStatus = SxmUtils::replaceInTranslationTextID(PopupStatus, Emptycount.str(), "%2");
         }
         if (_isReplaceFav)
         {
            //getFavList();
            _isReplaceFav = false;
         }
         storeFavoriteStatus(favcount);
         POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__POPUP_ALL_ARTISTSONG_USERCONF)));
         break;
      }
      case T_e8_FavoriteStatus__CONTENT_ALREADY_REGISTERED:
      {
         isTwoLine = false;
         PopupStatus = SxmUtils::safeGetCString(SXM__AUDIO_POPUP_FAVTALREDAYSAVED);
         PopupStatus = SxmUtils::replaceInTranslationTextID(PopupStatus, contentName, "Artist");
         POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__POPUP_ALL_ARTISTSONG_USERCONF)));
         break;
      }
      default :
         break;
   }
   SxmAudioDataBindingUtils::instance()->setConentAlertAdivisorypopup(PopupStatus, isTwoLine);
   uint32 viewid = SXMCommonInterface::getActiveViewName(Enum_SURFACEID_MAIN_SURFACE_SXM);
   if (viewid == Sxm_Scenes_SXM_RADIO__ALERT_ARTISTLIST_EDIT || viewid == Sxm_Scenes_SXM_RADIO__ALERT_SONGLIST_EDIT)
   {
      getFavList();
   }
}


/**
 * Helper Function to update Favorite List
 * @param [in] : u8Action Modify Method Action
 */
void SxmAudioContentAlert::handleModifyFavoriteList(uint8 action)
{
   ETG_TRACE_USR4(("SxmAudioContentAlert::UpdateFavoriteList"));
   std::string PopupStatus;
   switch (action)
   {
      case T_e8_Action__ENABLE:
      case T_e8_Action__DISABLE:
      {
         getFavList();
         break;
      }
      case T_e8_Action__DELETE:
      {
         if (!_isReplaceFav)
         {
            PopupStatus = SxmUtils::safeGetCString(SXM__AUDIO_POPUP_FAVDELETED);
            SxmAudioDataBindingUtils::instance()->setDeleteFavAdivisorypopup(PopupStatus);
            POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__POPUP_REMOVE_ARTISTS_USERCONF)));
            getFavList();
         }
         else
         {
            uint16 ServiceId = SxmAudioListHelper::instance()->getActiveChannelSID();
            uint16 ChannelId = SxmAudioListHelper::instance()->getActiveChannelCID();
            _sxmAudioFiProxy->sendSaveFavoriteStart(*this, ServiceId, ChannelId, _favtype, _cMaxfavCount);
         }
         break;
      }
      case T_e8_Action__ENABLE_ALL:
      {
         getFavList();
         SxmAudioDataBindingUtils::instance()->updateOnOffButtonStatus((uint8)_favtype, DISABLE);
         break;
      }
      case T_e8_Action__DISABLE_ALL:
      {
         getFavList();
         SxmAudioDataBindingUtils::instance()->updateOnOffButtonStatus((uint8)_favtype, ENABLE);
         break;
      }
      case T_e8_Action__DELETE_ALL:
      {
         PopupStatus = SxmUtils::safeGetCString(SXM__AUDIO_POPUP_ALLFAVDELETED);
         SxmAudioDataBindingUtils::instance()->setDeleteFavAdivisorypopup(PopupStatus);
         getFavList();
         break;
      }
      default:
         break;
   }
}


/**
 * Helper Function to Get Updated Favorite List
 */
void SxmAudioContentAlert::getFavList()
{
   if (_sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendGetFavoriteListStart(*this, _favtype);
   }
}


/**
 * Helper Function to update Favorite List Artist and Song
 */
void SxmAudioContentAlert::updateFavoriteList()
{
   ETG_TRACE_USR4(("SxmAudioContentAlert::updateFavoriteList _transtionId = %d, ", _transtionId));
   if (_transtionId != Sxm_Scenes_SXM_RADIO__FAVORITE_REPLACELIST)
   {
      uint32 lSceneId = (_favtype == T_e8_FavoriteType__FAVORITE_ARTIST) ?
                        Sxm_Scenes_SXM_RADIO__ALERT_ARTISTLIST_EDIT : Sxm_Scenes_SXM_RADIO__ALERT_SONGLIST_EDIT;
      if (SxmAudioListHelper::instance()->getRegisteredFavoriteList().size() == 0)
      {
         SxmAudioDataBindingUtils::instance()->setAlertsAdvisory(false, getAdvisoryTextForFavorites());
         POST_MSG((COURIER_MESSAGE_NEW(SxmSceneTransitionReqMsg)(lSceneId)));
      }
      else
      {
         uint32 viewid = SXMCommonInterface::getActiveViewName(Enum_SURFACEID_MAIN_SURFACE_SXM);
         SxmAudioDataBindingUtils::instance()->setAlertsAdvisory(true, "");
         if (viewid == Sxm_Scenes_SXM_RADIO__ALERT_ARTISTLIST_EDIT || viewid == Sxm_Scenes_SXM_RADIO__ALERT_SONGLIST_EDIT)
         {
            tSharedPtrDataProvider dataProvider;
            SxmAudioListHelper* pAudioHelperInstance = SxmAudioListHelper::instance();
            const char* dataitem;
            uint32 listId = 0;

            if (viewid == Sxm_Scenes_SXM_RADIO__ALERT_ARTISTLIST_EDIT)
            {
               listId = LIST_ID_SXM_AUDIO_FAVORITE_ARTISTLIST;
               dataitem = "SXMAudioFavtArtist";
            }
            else
            {
               listId = LIST_ID_SXM_AUDIO_FAVORITE_SONGLIST;
               dataitem = "SXMAudioFavtSong";
            }
            dataProvider = pAudioHelperInstance->getSxmFavoriteArtistSongListDataProvider(listId, dataitem);
            POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
         }
         else
         {
            POST_MSG((COURIER_MESSAGE_NEW(SxmSceneTransitionReqMsg)(lSceneId)));
         }
      }
   }
}


/**
 * Helper Function to update Alert Info and Notify.
 */
void SxmAudioContentAlert::setSxmAudioContentAlertinfo()
{
   if (_contentAlertList.size())
   {
      ETG_TRACE_USR4(("SxmAudioContentAlert::setSxmAudioContentAlertinfo favtype:%d getChannelID:%d",
                      _contentAlertList[Item_0].getFavoriteType(), _contentAlertList[Item_0].getChannelID()));
   }
   unsigned int alertType;
   std::string sArtistSongname;
   uint16 u16channelnum;
   std::string sAlertImg;

   if (T_e8_FavoriteType__FAVORITE_ARTIST == _contentAlertList[Item_0].getFavoriteType())
   {
      sArtistSongname = _contentAlertList[Item_0].getArtistName();
      alertType = T_e8_FavoriteType__FAVORITE_ARTIST;
      u16channelnum = _contentAlertList[Item_0].getChannelID();
      sAlertImg = "Sxm#Images#Landing#Icon_TabArtist_active";
   }
   else
   {
      sArtistSongname = _contentAlertList[Item_0].getSongName();
      alertType = T_e8_FavoriteType__FAVORITE_SONG;
      u16channelnum = _contentAlertList[Item_0].getChannelID();
      sAlertImg = "Sxm#Images#Landing#Icon_Tag_Song_active";
   }
   regionText regText = SXMSourceSwitchInterface::getPopupText();
   SxmAudioDataBindingUtils::instance()->setSxmAudioContentAlertinfo(u16channelnum, sArtistSongname, alertType, sAlertImg, regText);
   POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__ITPOP_ALERT_NOTIFICATION)));
   _isFavoriteAlertActive = true;
}


/**
 *Callback to handle Courier Message SxmAudioSaveFavReqMsg Request
 * @param [in] : Reference of Courier::SxmAudioSaveFavReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioContentAlert::onCourierMessage(const SxmAudioSaveFavReqMsg& oMsg)
{
   uint16 ServiceId = SxmAudioListHelper::instance()->getActiveChannelSID();
   uint16 ChannelId = SxmAudioListHelper::instance()->getActiveChannelCID();
   _favtype = static_cast<T_e8_FavoriteType>(oMsg.GetFavoriteType());
   if (_sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendSaveFavoriteStart(*this, ServiceId, ChannelId, _favtype, _cMaxfavCount);
   }
   return true;
}


/**
 * Callback to handle Courier Message SXMAudioFavtArtistListReqMsg Request
 * @param [in] : Reference of Courier::SXMAudioFavtArtistListReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioContentAlert::onCourierMessage(const SxmAudioContentAlertListReqMsg& oMsg)
{
   _favtype = static_cast<T_e8_FavoriteType>(oMsg.GetFavType());
   if (_sxmAudioFiProxy)
   {
      switch (oMsg.GetHdl())
      {
         case SXM_AUDIO_FAVORITE_ARTISTS:
         {
            SxmUtilsListHandler::instance()->clearFooterInfo(LIST_ID_SXM_AUDIO_FAVORITE_ARTISTLIST, false);
            _transtionId = Sxm_Scenes_SXM_RADIO__ALERT_ARTISTLIST_EDIT;
            _sxmAudioFiProxy->sendGetFavoriteListStart(*this, _favtype);
            break;
         }
         case SXM_AUDIO_FAVORITE_SONGS:
         {
            SxmUtilsListHandler::instance()->clearFooterInfo(LIST_ID_SXM_AUDIO_FAVORITE_SONGLIST, false);
            _transtionId = Sxm_Scenes_SXM_RADIO__ALERT_SONGLIST_EDIT;
            _sxmAudioFiProxy->sendGetFavoriteListStart(*this, _favtype);
            break;
         }
         case SXM_AUDIO_ARTISTS_ALERT:
         case SXM_AUDIO_SONGS_ALERT:
         {
            deQueueContentAlerts(_favtype);
            _sxmAudioFiProxy->sendGetAlertNoticeListStart(*this, _favtype);
            break;
         }
         default:
            break;
      }
   }
   return true;
}


/**
 *Helper Fucntion to send Modify favorite request to middleware
 */
void SxmAudioContentAlert::sendModifyFavoriteStart(uint32 fav_type, uint8 action, std::string contentname)
{
   if (_sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendModifyFavoriteStart(*this, static_cast<T_e8_FavoriteType>(fav_type), static_cast<T_e8_Action>(action), contentname);
   }
}


/**
 * Callback to handle Courier Message SxmAudioAlertSettingOnorOffReqMsg Request
 * @param [in] : Reference of Courier::SxmAudioAlertSettingOnorOffReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioContentAlert::onCourierMessage(const SxmAudioAlertSettingOnOffReqMsg& oMsg)
{
   SxmAudioDataBindingUtils::instance()->setSxmAudioAlertOnorOff(oMsg.GetHdl());
   SXMSourceSwitchInterface::setAlertSourceStatus();
   dp_tclAppHmi_SxmSXMAudioAlert dp_AudioAlert;
   _alertSettingsStatus = dp_AudioAlert.tGetData();
   tSharedPtrDataProvider dataProvider = SxmAudioListHelper::instance()->getSxmAudioAlertSettingListDataProvider();
   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
   return true;
}


/**
 * Helper Function to restore to factory default settings.
 */
void SxmAudioContentAlert::restoreAudioAlertDP(void)
{
   dp_tclAppHmi_SxmSXMAudioAlert dp_AudioAlert;
   HmiAppSxm::AlertSetting tSXMAudioAlert;
   for (uint8 index = 0; index < MAX_ALERTSETTINGS_ENTRIES; ++index)
   {
      if (index < Item_4)
      {
         tSXMAudioAlert.OnOrOff[index] = 1; //Set to Alert on.
      }
      else
      {
         tSXMAudioAlert.OnOrOff[index] = 0;
      }
   }
   dp_AudioAlert.s32SetData(tSXMAudioAlert);
   _alertSettingsStatus = dp_AudioAlert.tGetData();
   SXMSourceSwitchInterface::setAlertSourceStatus();
}


/**
*Callback to handle Courier Message SXMAudioEnableDeleteFavReqMsg Request
* @param [in] : Reference of Courier::SXMAudioEnableDeleteFavReqMsg
* @return     : True-When Message is processed.
*/
bool SxmAudioContentAlert::onCourierMessage(const SXMAudioEnableDeleteFavReqMsg& oMsg)
{
   ::std::vector< T_RegisteredFavorite > favoriteList =
      SxmAudioListHelper::instance()->getRegisteredFavoriteList();
   uint32 hdl = oMsg.GetHdl();
   if (hdl < favoriteList.size())
   {
      if (oMsg.GetSubHdl() == 0)//List Item button.
      {
         sendModifyFavoriteStart(oMsg.GetFavoriteType(), favoriteList[hdl].getIsEnabled(), favoriteList[hdl].getContentName());
      }
      else
      {
         sendModifyFavoriteStart(oMsg.GetFavoriteType(), T_e8_Action__DELETE, favoriteList[hdl].getContentName());
      }
   }
   return true;
}


/**
*Callback to handle Courier Message SXMAudioAlertsTurnOnOffReqMsg Request
*@param [in] : Reference of Courier::SXMAudioAlertsTurnOnOffReqMsg
* @return     : True-When Message is processed.
*/
bool SxmAudioContentAlert::onCourierMessage(const SXMAudioAlertsTurnOnOffReqMsg& oMsg)
{
   sendModifyFavoriteStart(oMsg.GetFavoriteType(), oMsg.GetOnOffStatus(), "");
   return true;
}


/**
 * Callback to handle Courier Message SxmAduioDeleteAllFavReqMsg Request
 * @param [in] : Reference of Courier::SxmAduioDeleteAllFavReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioContentAlert::onCourierMessage(const SxmAudioDeleteAllFavReqMsg& oMsg)
{
   ETG_TRACE_USR4(("SxmAudioContentAlert::SxmAduioDeleteAllFavReqMsg"));
   _deleteAllFavType = oMsg.GetFavoriteType();
   //sendModifyFavoriteStart(oMsg.GetFavoriteType(), T_e8_Action__DELETE_ALL, "");
   return true;
}


/**
 * Callback to handle Courier Message SxmAudioDeleteAllConfirmReqMsg Request
 * @param [in] : Reference of Courier::SxmAudioDeleteAllConfirmReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioContentAlert::onCourierMessage(const SxmAudioDeleteAllConfirmReqMsg& /*oMsg*/)
{
   ETG_TRACE_USR4(("SxmAudioContentAlert::SxmAudioDeleteAllConfirmReqMsg"));
   sendModifyFavoriteStart(_deleteAllFavType, T_e8_Action__DELETE_ALL, "");
   return true;
}


/**
 *Callback to handle Courier Message SxmAudioCloseFavMaxReachedPopupReqMsg Request
 * @param [in] : Reference of Courier::SxmAudioCloseFavMaxReachedPopupReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioContentAlert::onCourierMessage(const SxmAudioCloseFavMaxReachedPopupReqMsg& /*oMsg*/)
{
   if (_sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendGetFavoriteListStart(*this, _favtype);
   }
   //POST_MSG((COURIER_MESSAGE_NEW(SxmAudioCreateFavReplacescreenReqMsg)()));
   SxmUtilsListHandler::instance()->clearFooterInfo(LIST_ID_SXM_AUDIO_FAVORITE_REPLACELIST, false);
   POST_MSG((COURIER_MESSAGE_NEW(SxmSceneTransitionReqMsg)(Sxm_Scenes_SXM_RADIO__FAVORITE_REPLACELIST)));
   _transtionId = Sxm_Scenes_SXM_RADIO__FAVORITE_REPLACELIST;
   return true;
}


/**
 * Callback to handle Courier Message SxmAudioReplaceSelectedFavReqMsg Request
 * @param [in] : Reference of Courier::SxmAudioReplaceSelectedFavReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioContentAlert::onCourierMessage(const SxmAudioReplaceSelectedFavReqMsg& oMsg)
{
   ::std::vector<T_RegisteredFavorite > favoriteList  =
      SxmAudioListHelper::instance()->getRegisteredFavoriteList();
   _isReplaceFav = true;
   if (oMsg.GetRowId() < favoriteList.size())
   {
      sendModifyFavoriteStart(_favtype, T_e8_Action__DELETE, favoriteList[oMsg.GetRowId()].getContentName());
   }
   return true;
}


/**
 * Callback to handle Courier Message SxmAudioCloseContentAlertPopupReqMsg Request
 * @param [in] : Reference of Courier::SxmAudioCloseContentAlertPopupReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioContentAlert::onCourierMessage(const SxmAudioCloseContentAlertPopupReqMsg& /*oMsg*/)
{
   _isFavoriteAlertActive = false;
   SxmAudioDataBindingUtils* pInstance = SxmAudioDataBindingUtils::instance();
   pInstance->setSxmContentAlertStatus(false);
   if (!_contentAlertList.empty())
   {
      updateAlertPopup();
   }
   if (!pInstance->getSxmTransitionDoneStatus())
   {
      if (SXMCommonInterface::isSxmInForeground() &&  Enum_SXMAudioUtility == SXMCommonInterface::getActiveServiceState())
      {
         pInstance->switchToPresetMode();
         POST_MSG((COURIER_MESSAGE_NEW(SXMAudioGotoAudioMainMsg)()));
      }
      pInstance->setSxmTransitionDoneStatus(true);
   }
   return true;
}


/**
 * Helper function to handle display of Alert pop up.
 */
void SxmAudioContentAlert::updateAlertPopup()
{
   ETG_TRACE_USR4(("SxmAudioContentAlert::updateAlertPopup _contentAlertList size : %d", _contentAlertList.size()));
   if (!_contentAlertList.empty())
   {
      if (((T_e8_FavoriteType__FAVORITE_ARTIST == _contentAlertList[Item_0].getFavoriteType() && _alertSettingsStatus.OnOrOff[Item_0]) ||
            (T_e8_FavoriteType__FAVORITE_SONG == _contentAlertList[Item_0].getFavoriteType() && _alertSettingsStatus.OnOrOff[Item_1])) &&
            (isPopupAllowed()))
      {
         if (!_popupCloseOnAppmodeChange)
         {
            _contentAlertList.erase(_contentAlertList.begin());
         }
         if (!_contentAlertList.empty() && _allowContentAlert)
         {
            //To Check: Only tunable alerts are queued,why check is needed?
            if (_contentAlertList[Item_0].getChannelStatus().getCHANNEL_STATUS_TUNABLE())
            {
               setSxmAudioContentAlertinfo();
            }
            else
            {
               _contentAlertList.erase(_contentAlertList.begin());
               updateAlertPopup();
            }
         }
      }
      else
      {
         deQueueContentAlerts(_contentAlertList[Item_0].getFavoriteType());
      }
   }
}


/**
 * Callback to handle Courier Message SxmAudioTunetoContentAlertReqMsg Request
 * @param [in] : Reference of Courier::SxmAudioTunetoContentAlertReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioContentAlert::onCourierMessage(const SxmAudioTunetoContentAlertReqMsg& /*oMsg*/)
{
   POST_MSG((COURIER_MESSAGE_NEW(SXMTuneToChannelReqMsg)(SXM_TUN_SELECT_CID, _contentAlertList[0].getChannelID(), 1)));//1:Step size.
   SxmAudioDataBindingUtils::instance()->setSxmContentAlertStatus(true);
   return true;
}


/**
 * getListDataProvider : Gets the ListDataProvider from the corresponding listdataprovider functions
 * based on the List ID.
 * @param [in] : listId
 * @return     : tSharedPtrDataProvider class pointer
 */
tSharedPtrDataProvider SxmAudioContentAlert::getListDataProvider(const ListDateProviderReqMsg& listDataInfo)
{
   ETG_TRACE_USR4(("SxmAudioContentAlert::getListDataProvider"));
   tSharedPtrDataProvider dataProvider = tSharedPtrDataProvider();
   SxmAudioListHelper* pAudioHelperInstance = SxmAudioListHelper::instance();
   uint32 listId = listDataInfo.GetListId();
   switch (listId)
   {
      case LIST_ID_SXM_AUDIO_FAVORITE_ARTISTLIST:
      {
         dataProvider =  pAudioHelperInstance->getSxmFavoriteArtistSongListDataProvider(listId, "SXMAudioFavtArtist");
         break;
      }
      case LIST_ID_SXM_AUDIO_FAVORITE_SONGLIST:
      {
         dataProvider =  pAudioHelperInstance->getSxmFavoriteArtistSongListDataProvider(listId, "SXMAudioFavtSong");
         break;
      }
      case LIST_ID_SXM_AUDIO_ARTISTLIST:
      {
         dataProvider =  pAudioHelperInstance->getSxmAlertArtistSongListDataProvider(listId, "SXMAudioAlertArtistList");
         break;
      }
      case LIST_ID_SXM_AUDIO_SONGLIST:
      {
         dataProvider =  pAudioHelperInstance->getSxmAlertArtistSongListDataProvider(listId, "SXMAudioAlertSongList");
         break;
      }
      case LIST_ID_SXM_AUDIO_ALERTLIST:
      {
         dataProvider = pAudioHelperInstance->getSxmAudioAlertSettingListDataProvider();
         break;
      }
      case LIST_ID_SXM_AUDIO_FAVORITE_REPLACELIST:
      {
         dataProvider = pAudioHelperInstance->getSxmFavoriteReplaceListDataProvider();
         break;
      }
      default:
      {
         ETG_TRACE_USR4(("getListDataProvider:ERROR:Invalid List"));
         break;
      }
   }
   return dataProvider;
}


/**
 * Helper Function to get advisory text.
 */
const std::string SxmAudioContentAlert::getAdvisoryText() const
{
   std::string msg = "";
   if (_favtype == T_e8_FavoriteType__FAVORITE_ARTIST)
   {
      dp_tclAppHmi_SxmSxmAudioArtistFavoriteStatus dp_ArtistFavoriteStatus;
      if (dp_ArtistFavoriteStatus.tGetData())
      {
         msg.append(SxmUtils::safeGetCString(SXM__AUDIO_ADV_ARTISTREGISTEREDL1)).append("\n\n").append(SxmUtils::safeGetCString(SXM__AUDIO_ADV_ARTISTREGISTEREDL2));
      }
      else
      {
         msg.append(SxmUtils::safeGetCString(SXM__AUDIO_ADV_ARTISTNOTREGISTEREDL1)).append("\n\n").append(SxmUtils::safeGetCString(SXM__AUDIO_ADV_ARTISTNOTREGISTEREDL2));
      }
   }
   else
   {
      dp_tclAppHmi_SxmSxmAudioSongFavoriteStatus dp_SongFavoriteStatus;
      if (dp_SongFavoriteStatus.tGetData())
      {
         msg.append(SxmUtils::safeGetCString(SXM__AUDIO_ADV_SONGREGISTEREDL1)).append("\n\n").append(SxmUtils::safeGetCString(SXM__AUDIO_ADV_SONGREGISTEREDL2));
      }
      else
      {
         msg.append(SxmUtils::safeGetCString(SXM__AUDIO_ADV_SONGNOTREGISTEREDL1)).append("\n\n").append(SxmUtils::safeGetCString(SXM__AUDIO_ADV_SONGNOTREGISTEREDL2));
      }
   }
   return msg;
}


/**
 * Helper Function to get advisory text.
 */
const std::string SxmAudioContentAlert::getAdvisoryTextForFavorites() const
{
   std::string msg = "";
   if (T_e8_FavoriteType__FAVORITE_ARTIST == _favtype)
   {
      msg.append(SxmUtils::safeGetCString(SXM__AUDIO_ADV_EDITARTISTL1)).append("\n\n").append(SxmUtils::safeGetCString(SXM__AUDIO_ADV_EDITARTISTL2));
   }
   else
   {
      msg.append(SxmUtils::safeGetCString(SXM__AUDIO_ADV_EDITSONGL1)).append("\n\n").append(SxmUtils::safeGetCString(SXM__AUDIO_ADV_EDITSONGL2));
   }
   return msg;
}


/**
 * Helper Function to store favorite availability.
 */
void SxmAudioContentAlert::storeFavoriteStatus(bool status) const
{
   if (T_e8_FavoriteType__FAVORITE_ARTIST == _favtype)
   {
      dp_tclAppHmi_SxmSxmAudioArtistFavoriteStatus dp_ArtistFavoriteStatus;
      dp_ArtistFavoriteStatus.s32SetData(status);
   }
   else
   {
      dp_tclAppHmi_SxmSxmAudioSongFavoriteStatus dp_SongFavoriteStatus;
      dp_SongFavoriteStatus.s32SetData(status);
   }
}


/**
 * Helper Function to queue all alerts notified by middleware.
 */
void SxmAudioContentAlert::enQueueContentAlerts(const T_AlertInfo& lAlertInfo)
{
   ETG_TRACE_USR4(("SxmAudioContentAlert::enQueueContentAlerts _allowContentAlert:%d", _allowContentAlert));
   if (((T_e8_FavoriteType__FAVORITE_ARTIST == lAlertInfo.getFavoriteType() && _alertSettingsStatus.OnOrOff[Item_0]) ||
         (T_e8_FavoriteType__FAVORITE_SONG == lAlertInfo.getFavoriteType() && _alertSettingsStatus.OnOrOff[Item_1])) &&
         (isPopupAllowed()))
   {
      _contentAlertList.push_back(lAlertInfo);
      ETG_TRACE_USR4(("SxmAudioContentAlert::enQueueContentAlerts _contentAlertList size :%d", _contentAlertList.size()));
      if ((!_isFavoriteAlertActive) && _allowContentAlert) //Only if previous alert is closed.
      {
         setSxmAudioContentAlertinfo();
      }
   }
   else
   {
      deQueueContentAlerts(lAlertInfo.getFavoriteType());
   }
}


/**
 * Helper Function to clear all alerts when alert settings are changed.
 */
void SxmAudioContentAlert::deQueueContentAlerts(const T_e8_FavoriteType& lfavtype)
{
   std::vector<T_AlertInfo>::iterator itrAlert;
   ETG_TRACE_USR4(("list %d", _contentAlertList.size()));
   for (itrAlert = _contentAlertList.begin(); itrAlert != _contentAlertList.end();)
   {
      if (itrAlert->getFavoriteType() == lfavtype)
      {
         ETG_TRACE_USR4(("SxmAudioContentAlert::deQueueContentAlerts"));
         //clear only requested favorite entries.
         itrAlert = _contentAlertList.erase(itrAlert);
      }
      else
      {
         ++itrAlert;
      }
   }
}


/**
 * Function to check whether popup allowed based on settings and source status
 */
bool SxmAudioContentAlert::isPopupAllowed()
{
   bool status = false;
   std::bitset<6> alertSource;
   alertSource = std::bitset<6>(SXMSourceSwitchInterface::getAlertSourceStatus());
   //Either Alert to be allowed in any region (or) SXM to be active in any region
   /*
    const uint8 IS_ALERT_ALLOWED_COCKPIT = 0;
   const uint8 IS_ALERT_ALLOWED_CABINA = 1;
   const uint8 IS_ALERT_ALLOWED_CABINB = 2;
   const uint8 IS_SXM_IN_COCKPIT = 3;
   const uint8 IS_SXM_IN_CABINA = 4;
   const uint8 IS_SXM_IN_CABINB = 5;
    */
   status = alertSource.any();
   return status;
}


/**
 * Helper Function to handle channel art result.
 */
void SxmAudioContentAlert::onGetChannelArtListResult(const ::std::vector< ChannelGraphicsType >& channelArtList)
{
   uint32 viewid = SXMCommonInterface::getActiveViewName(Enum_SURFACEID_MAIN_SURFACE_SXM);
   if (Sxm_Scenes_SXM_RADIO__ALERT_ARTISTLIST == viewid || Sxm_Scenes_SXM_RADIO__ALERT_SONGLIST == viewid)
   {
      //To handle dynamic updates in alert history screen.
      SxmAudioListHelper::instance()->setChannellistlogo(channelArtList, viewid);
      tSharedPtrDataProvider dataProvider;
      SxmAudioListHelper* pAudioHelperInstance = SxmAudioListHelper::instance();
      const char* dataitem;
      uint32 listId = 0;

      if (viewid == Sxm_Scenes_SXM_RADIO__ALERT_ARTISTLIST)
      {
         listId = LIST_ID_SXM_AUDIO_ARTISTLIST;
         dataitem = "SXMAudioAlertArtistList";
      }
      else
      {
         listId = LIST_ID_SXM_AUDIO_SONGLIST;
         dataitem = "SXMAudioAlertSongList";
      }

      dataProvider =  pAudioHelperInstance->getSxmAlertArtistSongListDataProvider(listId, dataitem);

      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
   }
   else if (_transtionId)
   {
      //To handle transition to alert history screen.
      SxmAudioListHelper::instance()->setChannellistlogo(channelArtList, _transtionId);
      POST_MSG((COURIER_MESSAGE_NEW(SxmSceneTransitionReqMsg)(_transtionId)));
      _transtionId = 0;//Clear after transition request is handled.
   }
}


/**
 * Helper Function to handle channel art Error.
 */
void SxmAudioContentAlert::onGetChannelArtListError()
{
   ETG_TRACE_ERR(("SxmAudioContentAlert::onGetChannelArtListError"));
}


/**
 * Helper Function to handle channel art Error.
 */
void SxmAudioContentAlert::onGetChannelLogoError(const uint16& /*sid*/)
{
   ETG_TRACE_ERR(("SxmAudioContentAlert::onGetChannelLogoError"));
}


/**
*  Call Back function for service status.
*
*  @param[in] : lEntryExitStatus - if service is entered or exited.
*/
void SxmAudioContentAlert::vNotifyServiceStatus(unsigned short int lEntryExitStatus, unsigned int serviceType)
{
   ETG_TRACE_USR2(("SxmAudioContentAlert::vNotifyServiceStatus %d", lEntryExitStatus));
}


/**
 *  Call Back function for system appmode status.
 */
void SxmAudioContentAlert::vNotifyAppModeChange(unsigned short int lactiveAppMode)
{
   ETG_TRACE_USR4(("SxmAudioContentAlert::vNotifySystemAppModeChange lactiveAppMode:%d", lactiveAppMode));

   //For RVC, SONAR - the interrupt management is handled through queuing concept.
   if ((lactiveAppMode == APP_MODE_PHONE_INCOMING_CALL) || (lactiveAppMode == APP_MODE_APPS_INCOMING_CALL) || (lactiveAppMode == APP_MODE_ECALL) ||
         (lactiveAppMode == APP_MODE_PHONE_INCOMING_SMS) || (lactiveAppMode == APP_MODE_OPERATOR) || (lactiveAppMode == APP_MODE_VR) ||
         (lactiveAppMode == APP_MODE_SPCXVR))
   {
      //Based on these appmodes, popup is not allowed to create.
      ETG_TRACE_USR4(("SxmAudioContentAlert::vNotifySystemAppModeChange _isFavoriteAlertActive:%d", _isFavoriteAlertActive));
      _allowContentAlert = false;
      if (_isFavoriteAlertActive)
      {
         _popupCloseOnAppmodeChange = true;
         //close Content alert popup
         POST_MSG((COURIER_MESSAGE_NEW(SXMDeActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__ITPOP_ALERT_NOTIFICATION)));
      }
   }
   else
   {
      ETG_TRACE_USR4(("SxmAudioContentAlert::vNotifySystemAppModeChange"));
      _allowContentAlert = true;
      if (_popupCloseOnAppmodeChange)
      {
         updateAlertPopup();
         _popupCloseOnAppmodeChange = false;
      }
   }
}


} //namespace Core
} //namespace App
