/**************************************************************************************
 * @file         : SxmAudioChannelInformation.cpp
 * @addtogroup   : AppHmi_Sxm
 * @brief        : Functional class to handle business logic for Channel browsing.
 * @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 **************************************************************************************/
#include "hall_std_if.h"   // Including HALL framework code
#include "SxmAudioChannelInformation.h"
#include "SxmAudioListHelper.h"
#include "SxmAudioDataBindingUtils.h"
#include "App/Core/ClientHandler/SourceSwitch/SXMSourceSwitchInterface.h"
#include "App/Core/ClientHandler/ClockService/SXMClockServiceInterface.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "App/Core/SxmUtils/SxmUtils.h"
#include "HmiTranslation_TextIds.h"
#include "sxm_channelart_main_fi.h"
#include "App/Core/LanguageDefines.h"
#include "App/Core/SxmCommon/SxmHMIServiceStateHandler.h"
#include "App/Core/SxmUtils/SxmUtilsListHandler.h"
#include "App/Core/ClientHandler/Vehicle/SXMVehicleServiceInterface.h"
#include "App/Core/SxmUtils/SxmCommonTypes.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXMAUDIO
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::SxmAudioChannelInformation::
#include "trcGenProj/Header/SxmAudioChannelInformation.cpp.trc.h"
#endif

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_hmi_06_if.h"

namespace App {
namespace Core {
using namespace ::sxm_audio_main_fi;
using namespace ::sxm_audio_main_fi_types;
using namespace ::sxm_channelart_main_fi;
using namespace ::sxm_channelart_main_fi_types;
using namespace Util;
/**
 * SxmAudioChannelInformation Class constructor
 * @param [in] : pSxmAudioFiProxy - SXM Audio FI proxy
 * @param [in] : pSxmChannelartFiProxy - SXM Channel Art FI proxy
 */
SxmAudioChannelInformation::SxmAudioChannelInformation(::boost::shared_ptr< Sxm_audio_main_fiProxy > pSxmAudioFiProxy,
      ISxmAudioChannelArtRequest* pSxmChannelart)
   : _sxmAudioFiProxy(pSxmAudioFiProxy)
   , _sxmChannelart(pSxmChannelart)
   , _goToSXMMainScreen(false)
   , _noSystemMessageError(false)
   , _isRapidChannelModeActive(false)
   , _displayTpopAudio(false)
   , _isDstOn(false)
   , _dwellsteps(0)
   , _activeAdvisoryId(0)
   , _currentTimeZone("")
   , _cStepSizeOne(1)
   , _cDefaultCategoryType(0)
   , _dwellTimer(NULL)
   , _isAudioUtility(false)
   , _isEncodertuned(false)

{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("SxmAudioChannelInformation() Constructor"));
   SXMCommonInterface::registerServiceEntryExitNotification(this, SXMAudioUtility);
   SXMCommonInterface::registerApplicationModeChangeNotication(this, SXMAudioUtility);
   _channelInfoStatus.clear();
}


/**
 * SxmAudioChannelInformation Class destructor
 */
SxmAudioChannelInformation::~SxmAudioChannelInformation()
{
   ETG_TRACE_USR4(("~SxmAudioChannelInformation() Destructor"));
   ETG_I_UNREGISTER_FILE();
   SXMCommonInterface::deregisterServiceEntryExitNotification(this, SXMAudioUtility);
   SXMCommonInterface::deregisterApplicationModeChangeNotication(this, SXMAudioUtility);
   _goToSXMMainScreen = false;
   _noSystemMessageError = false;
   _isRapidChannelModeActive = false;
   _displayTpopAudio = false;
   _isDstOn = false;
   _dwellsteps = 0;
   _activeAdvisoryId = 0;
   _currentTimeZone = "";
   _channelInfoStatus.clear();
   _sxmAudioFiProxy.reset();
   SXM_UTILS_RELEASE_OBJECT(_dwellTimer);
   _sxmChannelart = NULL;//Interface pointer.
   _isAudioUtility = false;
   _isEncodertuned = false;
}


/**
 * Utility function that registers for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioChannelInformation::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAudioChannelInformation::registerProperties"));
   if (_sxmAudioFiProxy && (proxy == _sxmAudioFiProxy))
   {
      _sxmAudioFiProxy->sendSystemMessageUpReg(*this);
      _sxmAudioFiProxy->sendChannelInformationUpReg(*this);
      setClockParams();
   }
}


/**
 * Utility function that deregisters for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioChannelInformation::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAudioChannelInformation::deregisterProperties"));
   if ((proxy == _sxmAudioFiProxy) && (_sxmAudioFiProxy))
   {
      _sxmAudioFiProxy->sendChannelInformationRelUpRegAll();
      _sxmAudioFiProxy->sendSystemMessageRelUpRegAll();
   }
}


/**
 * Error handling function for channel information property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioChannelInformation::onChannelInformationError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< ChannelInformationError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioChannelInformation::onChannelInformationError"));
}


/**
 * Status handling function for channel information property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - Property status info
 */
void SxmAudioChannelInformation::onChannelInformationStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< ChannelInformationStatus >& status)
{
   ETG_TRACE_USR4(("SxmAudioChannelInformation::onChannelInformationStatus"));

   uint16 chSid = SxmAudioListHelper::instance()->getActiveChannelSID();
   uint16 lchnid = status->getChannelInformation().getChannelNumber();
   if (_sxmChannelart && (Timer::Stopped == _sxmChannelart->getChannelArtTimerStatus()))
   {
      _sxmChannelart->startLoadingTimer();
   }
   if (_noSystemMessageError && !_isRapidChannelModeActive && (lchnid <= END_CHANNEL_ID))
   {
      _channelInfoStatus = status->getChannelInformation();
      //Send Information to update channel information.
      updateChannelInfo();

      //Update the list information based on channel change
      updateListOnChannelInfo(chSid, status->getChannelInformation().getSID());

      //Request to middleware for channel art and album art.
      requestSxmGraphics(_channelInfoStatus.getChnArtAvail(), _channelInfoStatus.getAlbumArtAvail(), _channelInfoStatus.getSID());

      std::string ChannelNumber = SxmUtils::conCatData("i", 1, _channelInfoStatus.getChannelNumber());
      SXMVehicleServiceInterface::sendXMInfoData(SXM_CHANNEL_NUMBER, ChannelNumber);
      SXMVehicleServiceInterface::sendXMInfoData(SXM_CATEGORY, _channelInfoStatus.getCategoryName());
   }
}


/**
 * Error handling function for select channel method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error - Error type
 */
void SxmAudioChannelInformation::onSelectChannelError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SelectChannelError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioChannelInformation::onSelectChannelError"));
}


/**
 * Result handling function for select channel method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioChannelInformation::onSelectChannelResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SelectChannelResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioChannelInformation::onSelectChannelResult"));
   if (result->getStatus())
   {
      //When New channel is selected then toggle to Pause state.
      SxmAudioDataBindingUtils::instance()->setDefaultSxmAudioPlayPauseData(T_e8_IRPlaybackStatus__SXM_AUDIO_IR_PLAY_STATE);
      postTpopAudioMessage();
   }
   else
   {
      _goToSXMMainScreen = false;
   }
}


/**
 * Error handling function for select channel with dwell time method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error - Error type
 */
void SxmAudioChannelInformation::onSelectChannelWithDwellTimeError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SelectChannelWithDwellTimeError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioChannelInformation::onSelectChannelWithDwellTimeError"));
}


/**
 * Result handling function for select channel with dwell time method
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : result - Method result data
 */
void SxmAudioChannelInformation::onSelectChannelWithDwellTimeResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SelectChannelWithDwellTimeResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioChannelInformation::onSelectChannelWithDwellTimeResult"));
   if (result->getStatus())
   {
      //When New channel is selected then toggle to Pause state.
      SxmAudioDataBindingUtils::instance()->setDefaultSxmAudioPlayPauseData(T_e8_IRPlaybackStatus__SXM_AUDIO_IR_PLAY_STATE);
   }
}


/**
 * Error handling function for system messages property
 * @param [in] : proxy SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioChannelInformation::onSystemMessageError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SystemMessageError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioChannelInformation::onSystemMessageError"));
}


/**
 * Status handling function for system messages property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : status - Property status info
 */
void SxmAudioChannelInformation::onSystemMessageStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< SystemMessageStatus >& status)
{
   const T_e8_SystemMessage&  systemErrorType = status->getSystemErrorType();
   const uint16& cid = status->getCID();
   ETG_TRACE_USR4(("SxmAudioChannelInformation::onSystemMessageStatus u8ErrorType:%d,""channelid:%d", systemErrorType, cid));

   //Post message to controller based on advisory message.
   _noSystemMessageError = (T_e8_SystemMessage__XMTUN_SYSTEM_MSG_NO_ERROR == systemErrorType);
   SxmAudioDataBindingUtils::instance()->setSystemMessageStatus(systemErrorType);
   postPopupMessageForAudio(systemErrorType, cid);
   postPopupMessageForNonAudioMain(systemErrorType, cid);
   if ((T_e8_SystemMessage__XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE == systemErrorType ||
         T_e8_SystemMessage__XMTUN_SYSTEM_MSG_CHANNEL_UNAUTHORIZED == systemErrorType ||
         T_e8_SystemMessage__XMTUN_SYSTEM_MSG_CHANNEL_MATURE == systemErrorType ||
         T_e8_SystemMessage__XMTUN_SYSTEM_MSG_CHANNEL_LOCKED == systemErrorType)
         &&      _channelInfoStatus.getChannelNumber() == cid)
   {
      //Reset Play/Pause state when active channel is unavailable & preview channel is triggered.
      SxmAudioDataBindingUtils::instance()->setDefaultSxmAudioPlayPauseData(T_e8_IRPlaybackStatus__SXM_AUDIO_IR_PLAY_STATE);
   }
}


/**
 * Error handling function for ClockParams property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : error Error type
 */
void SxmAudioChannelInformation::onSetClockParamsError(const ::boost::shared_ptr<Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr<SetClockParamsError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmAudioChannelInformation::onSetClockParametersError"));
}


/**
 * status handling function for ClockParams property
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] :  status - Property status info
 */
void SxmAudioChannelInformation::onSetClockParamsStatus(const ::boost::shared_ptr<Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr<SetClockParamsStatus >& /*status*/)
{
   ETG_TRACE_ERR(("SxmAudioChannelInformation::onSetClockParamsStatus"));
}


/**
 * Error handling function for ClockParams property
 */
void SxmAudioChannelInformation::onRapidChannelBrowseInfoError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< RapidChannelBrowseInfoError >& /*error*/)
{
}


/**
 * Status handling function for RapidChannelBrowseInfo property
 */
void SxmAudioChannelInformation::onRapidChannelBrowseInfoStatus(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< RapidChannelBrowseInfoStatus >& status)
{
   ETG_TRACE_USR4(("SxmAudioChannelInformation::onRapidChannelBrowseInfoStatus"));
   //Send Information to update channel information.
   uint16 lchnid = status->getChannelInfo().getChannelNumber();
   if (lchnid <= END_CHANNEL_ID)
   {
      SxmAudioDataBindingUtils::instance()->setSxmChannelInfoData(status->getChannelInfo());
      //Request to middleware for channel art and album art.
      if ((SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__MAIN) && SXMCommonInterface::isSxmInForeground()) ||
            ((SXMCommonInterface::getGadgetVisibility(SXMGadgetAudioLarge)) || (SXMCommonInterface::getGadgetVisibility(SXMGadgetAudioSmall))))
      {
         requestSxmGraphics(status->getChannelInfo().getChnArtAvail(),
                            status->getChannelInfo().getAlbumArtAvail(), status->getChannelInfo().getSID());
         if ((SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__MAIN) && SXMCommonInterface::isSxmInForeground()))
         {
            _displayTpopAudio = false;
         }
         else
         {
            _displayTpopAudio = true;
            postTpopAudioMessage();
         }
      }
      else
      {
         SxmAudioDataBindingUtils::instance()->updateAllChannelInformation();
         _displayTpopAudio = true;
         postTpopAudioMessage();
      }
   }
}


/**
 * Error handling function for RapidChannelBrowseStart Method
 */
void SxmAudioChannelInformation::onRapidChannelBrowseStartError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< RapidChannelBrowseStartError >& /*error*/)
{
}


/**
 * Result handling function for RapidChannelBrowseStart Method
 */
void SxmAudioChannelInformation::onRapidChannelBrowseStartResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< RapidChannelBrowseStartResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioChannelInformation::onRapidChannelBrowseStartResult"));
   _isRapidChannelModeActive = result->getStatus();
}


/**
 * Error handling function for RapidChannelBrowseStop Method
 */
void SxmAudioChannelInformation::onRapidChannelBrowseStopError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< RapidChannelBrowseStopError >& /*error*/)
{
}


/**
 * Result handling function for RapidChannelBrowseStop Method
 */
void SxmAudioChannelInformation::onRapidChannelBrowseStopResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< RapidChannelBrowseStopResult >& result)
{
   _isRapidChannelModeActive = !(result->getStatus());
}


/**
 * Result handling function for GetChannelArt method.
 */
void SxmAudioChannelInformation::onGetChannelLogoResult(const uint16& /*sid*/, const uint8& graphicsType, const std::vector<uint8>& u8ImageData,
      const std::vector<uint8>& albumImageData)
{
   ETG_TRACE_USR4(("SxmAudioChannelInformation::onGetChannelLogoResult"));
   SxmAudioDataBindingUtils* pInstance = SxmAudioDataBindingUtils::instance();
   //Set binding data based on result.
   T_e8_ChnGraphicsTypes lChannelGraphicsType = static_cast<T_e8_ChnGraphicsTypes>(graphicsType);
   if ((T_e8_ChnGraphicsTypes__CHANNEL_ART_AVAILABLE_IMAGE_ALBUM |
         T_e8_ChnGraphicsTypes__CHANNEL_ART_AVAILABLE_IMAGE_LOGO) == lChannelGraphicsType)
   {
      pInstance->setSxmChannelLogoData(u8ImageData, u8ImageData.size());
      pInstance->setSxmAlbumartData(albumImageData, albumImageData.size());
   }
   else if (T_e8_ChnGraphicsTypes__CHANNEL_ART_AVAILABLE_IMAGE_ALBUM == lChannelGraphicsType)
   {
      pInstance->setSxmAlbumartData(albumImageData, albumImageData.size());
   }
   else if (T_e8_ChnGraphicsTypes__CHANNEL_ART_AVAILABLE_IMAGE_NONE == lChannelGraphicsType ||
            T_e8_ChnGraphicsTypes__CHANNEL_ART_AVAILABLE_IMAGE_LOGO == lChannelGraphicsType)
   {
      pInstance->setSxmChannelLogoData(u8ImageData, u8ImageData.size());
   }
   //Synchronize channel information and preset list.
   synchronizeAllChannelInfo();
   if (!pInstance->getSxmContentAlertStatus())
   {
      //Once All data is available after tune request do transition,
      if (_goToSXMMainScreen)
      {
         //post msg to go to audio main & Switch to Preset Mode if scan is active.
         pInstance->switchToPresetMode();
         POST_MSG((COURIER_MESSAGE_NEW(SXMAudioGotoAudioMainMsg)()));
         _goToSXMMainScreen = false;
      }
      pInstance->setSxmTransitionDoneStatus(true);
   }
   else
   {
      //wait
      _goToSXMMainScreen = false;
      pInstance->setSxmTransitionDoneStatus(false);
   }
}


/**
 * Callback to handle Courier Message SXMAudioDirectTuneOkReqMsg Request
 * @param [in] : Reference of Courier::SXMAudioDirectTuneOkReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioChannelInformation::onCourierMessage(const SXMAudioDirectTuneOkReqMsg& /*oMsg*/)
{
   _displayTpopAudio = false;
   _goToSXMMainScreen = true;
   sendSelectChannel(T_e8_SelectChannelMode__XMTUN_SELECT_CID, SxmAudioDataBindingUtils::instance()->getEditFeildStringOfSpeller(),
                     _cDefaultCategoryType, _cStepSizeOne);
   return true;
}


/**
 * Callback to handle Courier Message SXMAudioDirectTuneClearSpellerReqMsg Request
 * @param [in] : Reference of Courier::SXMAudioDirectTuneClearSpellerReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioChannelInformation::onCourierMessage(const SXMAudioDirectTuneClearSpellerReqMsg& /*oMsg*/) const
{
   SxmAudioDataBindingUtils::instance()->clearSpellerEditField();
   return true;
}


/**
 * Callback to handle Courier Message SXMAudioDirectTuneDeletePosReqMsg Request
 * @param [in] : Reference of Courier::SXMAudioDirectTuneDeletePosReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioChannelInformation::onCourierMessage(const SXMAudioDirectTuneDeletePosReqMsg& /*oMsg*/)
{
   SxmAudioDataBindingUtils::instance()->enableDirecttuneDeletePos();
   return true;
}


/**
 * Callback to handle Courier Message SXMAudioPostTimeZoneChangeReqMsg Request
 * @param [in] : Reference of Courier::SXMAudioPostTimeZoneChangeReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioChannelInformation::onCourierMessage(const SXMAudioPostTimeZoneChangeReqMsg& oMsg)
{
   _currentTimeZone = oMsg.GetTimeZone().GetCString();
   _isDstOn = oMsg.GetDst();
   setClockParams();
   return true;
}


/**
 *  Call Back function for service status.
 */
void SxmAudioChannelInformation::vNotifyServiceStatus(unsigned short int lEntryExitStatus, unsigned int serviceType)
{
   ETG_TRACE_USR2(("SxmAudioChannelInformation::vNotifyServiceStatus %d", lEntryExitStatus));
   //lEntryExitStatus ->Entry = 0, Exit = 1
   _isAudioUtility = !(static_cast<bool>(lEntryExitStatus));    //Entered audio utility.
}


/**
*  Call Back function for appmode status.
*
*  @param[in] : lactiveAppMode - current active app mode.
*/
void SxmAudioChannelInformation::vNotifyAppModeChange(unsigned short int /*lactiveAppMode*/)
{
   ETG_TRACE_USR2(("SxmAudioChannelInformation::vNotifyAppModeChange"));
}


/**
 * Callback to handle Courier Message SXMTuneToChannelReqMsg Request
 * @param [in] : Reference of Courier::SXMTuneToChannelReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioChannelInformation::onCourierMessage(const SXMTuneToChannelReqMsg& oMsg)
{
   bool status = true;
   uint8 u8stepsize = abs(oMsg.GetSteps());
   uint8 lmode = oMsg.GetSelectionMode();
   _displayTpopAudio = true;
   switch (lmode)
   {
      case Enum_SXM_TUN_SELECT_CHANNEL_ACCELERATE_UP:
      case Enum_SXM_TUN_SELECT_CHANNEL_ACCELERATE_DOWN:
      {
         startRapidChannelBrowse(lmode);
         break;
      }
      case Enum_SXM_TUN_SELECT_CHANNEL_ACCELERATE_RELEASE:
      {
         stopRapidChannelBrowse();
         break;
      }
      case Enum_SXM_TUN_SELECT_CHANNEL_UP_ENCODER:
      case Enum_SXM_TUN_SELECT_CHANNEL_DOWN_ENCODER:
      {
         _isEncodertuned = true;
         channelBrowseWithDwellTime(oMsg);
         break;
      }
      case Enum_SXM_TUN_SELECT_CID:
      case Enum_SXM_TUN_SELECT_SID:
      {
         sendSelectChannel(static_cast<T_e8_SelectChannelMode>(lmode), oMsg.GetCidOrSidNumber(), _cDefaultCategoryType, u8stepsize);
         _goToSXMMainScreen = _isAudioUtility;
         ETG_TRACE_USR2(("_goToSXMMainScreen = %d", _goToSXMMainScreen));
         if (!SXMSourceSwitchInterface::isSxmSourceActive())
         {
            _displayTpopAudio = false;
         }
         SXMSourceSwitchInterface::handleSourceChangeRequest(SRC_TUNER_XM, true, false, true);
         break;
      }
      case Enum_SXM_TUN_SELECT_CHANNEL_UP:
      case Enum_SXM_TUN_SELECT_CHANNEL_DOWN:
      {
         sendSelectChannel(static_cast<T_e8_SelectChannelMode>(lmode), oMsg.GetCidOrSidNumber(), _cDefaultCategoryType, u8stepsize);
         break;
      }
      case Enum_SXM_TUN_SELECT_CHANNEL_CONTEXT_CHANGE:
      {
         sendSelectChannel(T_e8_SelectChannelMode__XMTUN_SELECT_CID, oMsg.GetCidOrSidNumber(), _cDefaultCategoryType, u8stepsize);
         _displayTpopAudio = false;
         break;
      }
      default:
      {
         status = false;
         break;
      }
   }
   return status;
}


/**
* Helper Function to start rapid channel browse.
*/
void SxmAudioChannelInformation::startRapidChannelBrowse(uint8 lmode)
{
   T_e8_BrowseDirection browseMode = (lmode == Enum_SXM_TUN_SELECT_CHANNEL_ACCELERATE_UP) ?
                                     T_e8_BrowseDirection__BROWSE_UP : T_e8_BrowseDirection__BROWSE_DOWN;
   if (_sxmAudioFiProxy)
   {
      ::std::vector< T_BrowseRateConfiguration > lBrowseRate;
      lBrowseRate.push_back(T_BrowseRateConfiguration(2, 400));// First step 2secs: time duration 400ms:time in between the jump.
      lBrowseRate.push_back(T_BrowseRateConfiguration(2, 200));// Second step 2secs: time duration 200ms:time in between the jump.
      lBrowseRate.push_back(T_BrowseRateConfiguration(0, 100));// Third step 0secs: time duration 100ms:time in between the jump.
      T_BrowseChannelConfig rapidBrowseConfigutation(lBrowseRate, T_e8_BrowseChannelLoop__LOOP_AROUND, T_e8_BrowseChannelType__BROWSETYPE_ALL_CHANNELS);
      _sxmAudioFiProxy->sendRapidChannelBrowseStartStart(*this, browseMode, rapidBrowseConfigutation, 0);//0: category iD for all channels.
      _sxmAudioFiProxy->sendRapidChannelBrowseInfoUpReg(*this);
   }
}


/**
* Helper Function to stop rapid channel browse.
*/
void SxmAudioChannelInformation::stopRapidChannelBrowse()
{
   if (_sxmAudioFiProxy && _sxmAudioFiProxy->hasRapidChannelBrowseInfo())
   {
      ETG_TRACE_USR4(("stopRapidChannelBrowse"));
      _sxmAudioFiProxy->sendRapidChannelBrowseStopStart(*this, _sxmAudioFiProxy->getRapidChannelBrowseInfo().getChannelInfo().getChannelNumber());
      _sxmAudioFiProxy->sendRapidChannelBrowseInfoRelUpRegAll();
   }
}


/**
* Helper Function to start channel browse.
*/
void SxmAudioChannelInformation::channelBrowseWithDwellTime(const SXMTuneToChannelReqMsg& oMsg)
{
   uint8 u8stepsize = abs(oMsg.GetSteps());
   uint8 lmode = oMsg.GetSelectionMode();
   if (_sxmAudioFiProxy && (!_dwellTimer || (_dwellTimer->getStatus() != Timer::Running)))
   {
      T_e8_SelectChannelMode browseMode = (lmode == Enum_SXM_TUN_SELECT_CHANNEL_UP_ENCODER) ?
                                          T_e8_SelectChannelMode__XMTUN_SELECT_CHANNEL_UP : T_e8_SelectChannelMode__XMTUN_SELECT_CHANNEL_DOWN;
      _sxmAudioFiProxy->sendSelectChannelWithDwellTimeStart(*this, browseMode, oMsg.GetCidOrSidNumber(),
            _cDefaultCategoryType, u8stepsize, 1000);//1000:wait time before tune completion.
      startDwellTimer();//To avoid queuing of channel & Album art.
      _dwellsteps = 0;
   }
   else
   {
      _dwellsteps += oMsg.GetSteps();
   }
}


/**
* Helper Function to set clock parameters to midw.
*/
void SxmAudioChannelInformation::setClockParams()
{
   if (_sxmAudioFiProxy && ::asf::core::ServiceState__Available == _sxmAudioFiProxy->getServiceState())
   {
      // LocalDateTimeType localtime = SXMClockServiceInterface::GetLocalTimeDate();
      // T_LocalTimeDate ltime(localtime.s16Year, localtime.u8Month, localtime.u8Day, localtime.u8Hours,
      // localtime.u8Minutes, localtime.u8Seconds, localtime.u8Weekday);
      // _sxmAudioFiProxy->sendSetClockParamsSet(*this, _currentTimeZone, _isDstOn, ltime);
   }
}


/**
 * Helper function to handle dwell time for channel browsing using rotary encoder,
 * Timer is used to avoid queing.
 */
void SxmAudioChannelInformation::startDwellTimer()
{
   if (_dwellTimer == NULL)
   {
      //Timer Initialize
      _dwellTimer = new Timer(150);//100 ms: timeout between two dwell request.
   }
   if (_dwellTimer != NULL)
   {
      if (Timer::Stopped == _dwellTimer->getStatus())
      {
         _dwellTimer->start();
      }
      else if (Timer::Expired == _dwellTimer->getStatus())
      {
         _dwellTimer->restart();
      }
   }
}


/**
 * Callback to handle Courier Message TimerExpiredMsg Request
 * @param [in] : Reference of Courier::TimerExpiredMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioChannelInformation::onCourierMessage(const TimerExpiredMsg& oMsg)
{
   bool status = false;
   if (oMsg.GetTimer() == _dwellTimer)
   {
      T_e8_SelectChannelMode browseMode = (_dwellsteps > 0) ? T_e8_SelectChannelMode__XMTUN_SELECT_CHANNEL_UP :
                                          T_e8_SelectChannelMode__XMTUN_SELECT_CHANNEL_DOWN;
      _sxmAudioFiProxy->sendSelectChannelWithDwellTimeStart(*this, browseMode, 0,
            _cDefaultCategoryType, abs(_dwellsteps), 1000);//1000:wait time before tune completion.
      _dwellsteps = 0;
      status = true;
   }
   return status;
}


/**
 * Callback to handle binding update of data item SxmAudioDirectTuneTextInfo.
 * @param [in] : Reference of Courier::Request
 * @return     : True-When Message is processed.
 */
bool SxmAudioChannelInformation::onCourierBindingItemChange_SxmAudioDirectTuneTextInfoItem(Courier::Request& request)
{
   SxmAudioDataBindingUtils::instance()->setSxmAudioDirectTuneTextInfo(request);
   return true;
}


/**
 * Callback to handle binding update of data item SxmAudioDirectTuneKeyPressInfo.
 * @param [in] : Reference of Courier::Request
 * @return     : True-When Message is processed.
 */
bool SxmAudioChannelInformation::onCourierBindingItemChange_SxmAudioDirectTuneKeyPressInfoItem(Courier::Request& request)
{
   SxmAudioDataBindingUtils::instance()->setSxmAudioDirectTuneKeyPressInfo(request);
   return true;
}


/**
 * Callback to handle Courier Message SXMPopUpStateUpdMsg Request
 * @param [in] : Reference of Courier::SXMPopUpStateUpdMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioChannelInformation::onCourierMessage(const SXMPopUpStateUpdMsg& oMsg) const
{
   SxmAudioDataBindingUtils::instance()->setPopUpStatus(oMsg.GetVisible());
   return true;
}


/**
 * Callback to handle Courier Message SXMTuneSelectedChannelItemReqMsg Request
 * @param [in] : Reference of Courier::SXMTuneSelectedChannelItemReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioChannelInformation::onCourierMessage(const SXMTuneSelectedChannelItemReqMsg& oMsg)
{
   T_e8_SelectChannelMode u8SelectChannelMode = T_e8_SelectChannelMode__XMTUN_SELECT_SID;
   uint16 u16SID = 0;
   uint16 u16CategoryType = 0;
   uint32 u32selectedidx = oMsg.GetRowId();
   bool status = true;
   const char* activeViewName = "";

   switch (oMsg.GetListId())
   {
      case LIST_ID_SXM_AUDIO_CHANNELLIST:
      {
         u16SID = SxmAudioListHelper::instance()->getSelectedChannelSID(u32selectedidx);
         u16CategoryType = SxmAudioListHelper::instance()->getSelectedCategoryType();
         activeViewName = SxmHMIServiceStateHandler::instance()->getSxmSceneName();
         break;
      }
      case LIST_ID_SXM_AUDIO_ARTISTLIST:
      case LIST_ID_SXM_AUDIO_SONGLIST:
      {
         ::std::vector<T_AlertInfo> lAlertList =  SxmAudioListHelper::instance()->getAlertInfoList();
         if (u32selectedidx < lAlertList.size())
         {
            u16SID = SxmAudioListHelper::instance()->getAlertInfoList()[u32selectedidx].getServiceID();
         }
         activeViewName = SxmHMIServiceStateHandler::instance()->getSxmSceneName();
         break;
      }
      default:
      {
         status = false;
         break;
      }
   }
   if (status)
   {
      //send tune request
      _goToSXMMainScreen = true;
      _displayTpopAudio = false;
      sendSelectChannel(u8SelectChannelMode, u16SID, u16CategoryType, _cStepSizeOne);
   }
   return status;
}


/**
 * Callback to handle Courier Message SxmUpdatePresetlistReqMsg Request
 * @param [in] : Reference of Courier::SxmUpdatePresetlistReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioChannelInformation::onCourierMessage(const SxmUpdatePresetlistReqMsg& /*oMsg*/)
{
   _displayTpopAudio = true;
   postTpopAudioMessage();
   return false;
}


/**
 * Callback to handle Courier Message SXMAudioPostLocalTimeChangeReqMsg Request
 * @param [in] : Reference of Courier::SXMAudioPostLocalTimeChangeReqMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioChannelInformation::onCourierMessage(const SXMAudioPostLocalTimeChangeReqMsg& oMsg)
{
   _currentTimeZone = oMsg.GetTimeZone().GetCString();
   _isDstOn = oMsg.GetDst();
   setClockParams();
   return false;
}


/**
 * Callback to handle Courier Message SxmKeyRegisterStatusMsg Request
 * @param [in] : Reference of Courier::SxmKeyRegisterStatusMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioChannelInformation::onCourierMessage(const SxmKeyRegisterStatusMsg& oMsg)
{
   ETG_TRACE_USR4(("SxmKeyRegisterStatusMsg isRegistered:%d", oMsg.GetIsRegistered()));
   if (!(oMsg.GetIsRegistered()) && _isRapidChannelModeActive)
   {
      stopRapidChannelBrowse();
   }
   return true;
}


/**
 * Callback to handle Courier Message SxmKeyRegisterStatusMsg Request
 * @param [in] : Reference of Courier::SxmKeyRegisterStatusMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioChannelInformation::onCourierMessage(const SxmAudioFocusResetReqMsg& oMsg)
{
   ETG_TRACE_USR4(("SxmAudioFocusResetReqMsg"));
   uint32 viewId = oMsg.GetViewId();
   if (viewId == Sxm_Scenes_SXM_RADIO__CHANNELLIST_3)
   {
      // POST_MSG((COURIER_MESSAGE_NEW(FocusReqMsg)(FOCUS_RESET, FOCUS_TIMER_STOP, Courier::ViewId("Sxm#Scenes#SXM_RADIO__CHANNELLIST_3"), "")));
   }
   else if (viewId == Sxm_Scenes_SXM_RADIO__CATEGORYLIST)
   {
      //  POST_MSG((COURIER_MESSAGE_NEW(FocusReqMsg)(FOCUS_RESET, FOCUS_TIMER_STOP, Courier::ViewId("Sxm#Scenes#SXM_RADIO__CATEGORYLIST"), "")));
   }
   else if (viewId == Sxm_Scenes_SXM_RADIO__ALERT_ARTISTLIST)
   {
      //   POST_MSG((COURIER_MESSAGE_NEW(FocusReqMsg)(FOCUS_RESET, FOCUS_TIMER_STOP, Courier::ViewId("Sxm#Scenes#SXM_RADIO__ALERT_ARTISTLIST"), "")));
   }
   else if (viewId == Sxm_Scenes_SXM_RADIO__ALERT_SONGLIST)
   {
      //  POST_MSG((COURIER_MESSAGE_NEW(FocusReqMsg)(FOCUS_RESET, FOCUS_TIMER_STOP, Courier::ViewId("Sxm#Scenes#SXM_RADIO__ALERT_SONGLIST"), "")));
   }
   return true;
}


/**
 * Helper function to Posts Toast Pop up Audio,if context is not in Main Screen.
 */
void SxmAudioChannelInformation::postTpopAudioMessage(void)
{
   ETG_TRACE_USR4(("SxmAudioChannelInformation::postTpopAudioMessage _isContextIsInAudioMain:%d,_goToSXMMainScreen:%d,_isSxmInForeground:%d,_displayTpopAudio:%d",
                   SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__MAIN), _goToSXMMainScreen, SXMCommonInterface::isSxmInForeground(), _displayTpopAudio));

   if (((!SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__MAIN) && !_goToSXMMainScreen) ||
         !SXMCommonInterface::isSxmInForeground()) && _displayTpopAudio)
   {
      //Post Top Pop up only when active context is not audio main screen and SXM app is in foreground.
      POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__TPOP_AUDIO_WITH_STATIONNAME)));
   }
}


/**
* Helper function post PopUp Message on channel selection failure
*/
void SxmAudioChannelInformation::postPopupMessageForNonAudioMain(::sxm_audio_main_fi_types::T_e8_SystemMessage errorType, uint16 cid)
{
   ETG_TRACE_USR4(("postPopupMessageForNonAudioMain u8ErrorType:%d", errorType));
   const char* activeViewName = "";
   SxmAudioDataBindingUtils* pAudioUtilsInstance = SxmAudioDataBindingUtils::instance();

   if ((!SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__MAIN) && SXMCommonInterface::isSxmInForeground()) ||
         ((!SXMCommonInterface::isSxmInForeground()) && SXMSourceSwitchInterface::isSxmSourceActive()))
   {
      switch (errorType)
      {
         case T_e8_SystemMessage__XMTUN_SYSTEM_MSG_ANTENNA_DISCONNECTED:
         {
            _activeAdvisoryId = Sxm_SxmPopups_SXM_RADIO__TPOP_FAILURE_CHECK_ANTENNA;
            break;
         }
         case T_e8_SystemMessage__XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE:
         {
            pAudioUtilsInstance->updateChannelNotAvailablePopup(cid);
            _activeAdvisoryId = Sxm_SxmPopups_SXM_RADIO__TPOP_FAILURE_CHANNEL_NA;
            break;
         }
         case T_e8_SystemMessage__XMTUN_SYSTEM_MSG_CHANNEL_UNAUTHORIZED:
         {
            pAudioUtilsInstance->updateChannelNotSubscribedPopup(cid);
            _activeAdvisoryId = Sxm_SxmPopups_SXM_RADIO__TPOP_FAILURE_CHANNEL_NOT_SUBSCRIBED;
            break;
         }
         case T_e8_SystemMessage__XMTUN_SYSTEM_MSG_CHANNEL_LOCKED:
         case T_e8_SystemMessage__XMTUN_SYSTEM_MSG_CHANNEL_MATURE:
         {
            pAudioUtilsInstance->setAdvisoryForSkippedChannel(cid);
            _activeAdvisoryId = Sxm_SxmPopups_SXM_RADIO__TPOP_CHANNEL_SKIPPED;
            break;
         }
         case T_e8_SystemMessage__XMTUN_SYSTEM_MSG_NO_ERROR:
         {
            if (_activeAdvisoryId)
            {
               POST_MSG((COURIER_MESSAGE_NEW(SXMDeActivatePopUpMsg)(_activeAdvisoryId)));
               _activeAdvisoryId = 0;
            }
            break;
         }
         case T_e8_SystemMessage__XMTUN_SYSTEM_MSG_SXM_INITIALIZING:
         {
            if (Enum_SXMAudioUtility == SXMCommonInterface::getActiveServiceState())
            {
               //only while in Audio sub screens.
               activeViewName = SxmHMIServiceStateHandler::instance()->getSxmSceneName();
               //      POST_MSG((COURIER_MESSAGE_NEW(FocusReqMsg)(FOCUS_RESET, FOCUS_TIMER_STOP, Courier::ViewId(activeViewName), "")));
               POST_MSG((COURIER_MESSAGE_NEW(SXMAudioGotoAudioMainMsg)()));
            }
            break;
         }
         default:
         {
            break;
         }
      }

      if (_activeAdvisoryId && ((Enum_SXMAudioUtility == SXMCommonInterface::getActiveServiceState()
                                 && _activeAdvisoryId == Sxm_SxmPopups_SXM_RADIO__TPOP_FAILURE_CHECK_ANTENNA)
                                || (_activeAdvisoryId != Sxm_SxmPopups_SXM_RADIO__TPOP_FAILURE_CHECK_ANTENNA)))
      {
         POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(_activeAdvisoryId)));
      }
   }
}


/**
 * Posts audio advisory messages to controller.
 * @param [in] : u8ErrorType- current advisory type.
 */
void SxmAudioChannelInformation::postPopupMessageForAudio(uint8 errorType, uint16 cid)
{
   ETG_TRACE_USR4(("postPopupMessageForAudio u8ErrorType:%d,u16Cid:%d", errorType, cid));
   SxmAudioDataBindingUtils* pAudioUtilsInstance = SxmAudioDataBindingUtils::instance();
   TChannelInformation channelInfo = {0, ""};
   switch (errorType)
   {
      case SXMAUDIO_CHECKANTENNA:
      {
         pAudioUtilsInstance->clearSxmChannelInfoData();
         pAudioUtilsInstance->setAdvisoryText(Candera::String(SXM__AUDIO_ADV_CHECKANTENNA));
         pAudioUtilsInstance->setGadgetAdvisoryText(Candera::String(SXM__AUDIO_GADGET_CHECKANTENNA));
         channelInfo.channelName = SxmUtils::safeGetCString(SXM__AUDIO_ADV_CHECKANTENNA);
         SXMCommonInterface::updateMapStatusBar(channelInfo);
         synchronizeAllChannelInfo();
         break;
      }
      case SXMAUDIO_NOSIGNAL:
      {
         pAudioUtilsInstance->clearSxmChannelInfoData();
         pAudioUtilsInstance->setAdvisoryText(Candera::String(SXM__AUDIO_ADV_NOSIGNAL));
         pAudioUtilsInstance->setGadgetAdvisoryText(Candera::String(SXM__AUDIO_ADV_NOSIGNAL));
         channelInfo.channelName = SxmUtils::safeGetCString(SXM__AUDIO_ADV_NOSIGNAL);
         SXMCommonInterface::updateMapStatusBar(channelInfo);
         synchronizeAllChannelInfo();
         break;
      }
      case SXMAUDIO_CHANNELUNAVAILABLE:
      {
         pAudioUtilsInstance->clearSxmChannelInfoData();
         std::string ChannelNum = (cid <= END_CHANNEL_ID) ? SxmUtils::conCatData("i", 1, cid) : "";
         std::string ChannelNotAvailable = SxmUtils::safeGetCString(SXM__AUDIO_ADV_CHANNELNOTAVAILABLE); // TextId_IT_00252  //channel not available
         ChannelNotAvailable = SxmUtils::replaceInTranslationTextID(ChannelNotAvailable, ChannelNum, "%1");
         pAudioUtilsInstance->setAdvisoryText(ChannelNotAvailable);
         pAudioUtilsInstance->setGadgetAdvisoryText(ChannelNotAvailable);
         channelInfo.channelName = ChannelNotAvailable;
         SXMCommonInterface::updateMapStatusBar(channelInfo);
         synchronizeAllChannelInfo();
         break;
      }
      case SXMAUDIO_CHANNELUNAUTHORIZED:
      {
         pAudioUtilsInstance->clearSxmChannelInfoData();
         std::string ChannelNum = (cid <= END_CHANNEL_ID) ? SxmUtils::conCatData("i", 1, cid) : "";
         std::string ChannelNotSubscribed = SxmUtils::replaceInTranslationTextID(SxmUtils::safeGetCString(SXM__AUDIO_ADV_CHANNELNOTSUBSCRIBED), ChannelNum, "%1");
         pAudioUtilsInstance->setAdvisoryText(ChannelNotSubscribed);
         pAudioUtilsInstance->setGadgetAdvisoryText(ChannelNotSubscribed);
         channelInfo.channelName = ChannelNotSubscribed;
         SXMCommonInterface::updateMapStatusBar(channelInfo);
         synchronizeAllChannelInfo();
         break;
      }
      case SXMAUDIO_SUBSCRIPTIONUPDATED:
      {
         POST_MSG((COURIER_MESSAGE_NEW(SXMActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__TPOP_SUBSCRIPTION_UPDATED)));
         break;
      }
      case SXMAUDIO_INITIALIZING:
      {
         pAudioUtilsInstance->clearSxmChannelInfoData();
         pAudioUtilsInstance->clearSXMIRData();
         SxmAudioListHelper::instance()->clearListVectors();
         pAudioUtilsInstance->enablePresetListButton(LEFTANDRIGHT, 0);//disable page up/Down buttons.
         std::string Advisory = SxmUtils::safeGetCString(SXM__AUDIO_ADV_LOADING);
         pAudioUtilsInstance->setAdvisoryText(Advisory);
         pAudioUtilsInstance->displayLoading();
         pAudioUtilsInstance->setGadgetAdvisoryText(Candera::String(SXM__AUDIO_GADGET_LOADING));
         pAudioUtilsInstance->setSxmAllChannelInfoAvailbility(false);
         SxmAudioListHelper::instance()->switchPresetBand();
         channelInfo.channelName = Advisory;
         SXMCommonInterface::updateMapStatusBar(channelInfo);
         if (_sxmChannelart)
         {
            _sxmChannelart->stopLoadingTimer();
            _sxmChannelart->clearAllChannelArtReq();
         }
         break;
      }
      case SXMAUDIO_CHANNELLOCKED:
      case SXMAUDIO_CHANNELMATURE:
      {
         pAudioUtilsInstance->clearSxmChannelInfoData();
         std::string ChannelNum = (cid <= END_CHANNEL_ID) ? SxmUtils::conCatData("i", 1, cid) : "";
         std::string ChannelLocked = SxmUtils::replaceInTranslationTextID(SxmUtils::safeGetCString(SXM__AUDIO_ADV_CHANNELSKIPPED), ChannelNum, "%1");
         pAudioUtilsInstance->setAdvisoryText(ChannelLocked);
         pAudioUtilsInstance->setGadgetAdvisoryText(ChannelLocked);
         channelInfo.channelName = ChannelLocked;
         SXMCommonInterface::updateMapStatusBar(channelInfo);
         synchronizeAllChannelInfo();

         break;
      }
      case SXMAUDIO_NOERROR:
      {
         pAudioUtilsInstance->updateReplayModeStatus();
         break;
      }
      default:
      {
         break;
      }
   }
   if ((errorType != SXMAUDIO_NOERROR) && (!channelInfo.channelName.empty()))
   {
      SXMVehicleServiceInterface::sendXMInfoData(SXM_ADVISORY, channelInfo.channelName);
   }
}


/**
* Helper function to send tune request to middleware.
*/
void SxmAudioChannelInformation::sendSelectChannel(T_e8_SelectChannelMode selectionMode, uint16 cidOrSid, uint16 categoryType, uint8 steps)
{
   if (_sxmAudioFiProxy)
   {
      _sxmAudioFiProxy->sendSelectChannelStart(*this, selectionMode, cidOrSid, categoryType, steps);
   }
}


/**
* Request Middleware to get channel art or Album art for selected service id.
* @param[in] : Service Id
*/
void SxmAudioChannelInformation::sendChannelArtReq(uint16 u16ServiceId, uint8 eGraphicsType)
{
   ETG_TRACE_USR4(("SxmAudioChannelInformation::sendChannelArtReq()"));
   SxmAudioDataBindingUtils* pinstance = SxmAudioDataBindingUtils::instance();
   if (_sxmChannelart)
   {
      _sxmChannelart->sendChannelLogoMethodRequest(this, u16ServiceId, eGraphicsType);
   }
   else
   {
      //To Check
      if (T_e8_ChnGraphicsTypes__CHANNEL_ART_AVAILABLE_IMAGE_LOGO == eGraphicsType)
      {
         pinstance->setSxmDefaultChannelLogoData();
      }
      else if (T_e8_ChnGraphicsTypes__CHANNEL_ART_AVAILABLE_IMAGE_ALBUM == eGraphicsType)
      {
         pinstance->setSxmDefaultAlbumartData();
      }
      else
      {
         pinstance->setSxmDefaultAlbumartData();
         pinstance->setSxmDefaultChannelLogoData();
      }
      synchronizeAllChannelInfo();
   }
}


/**
* Helper Function to Request for Channel Logo or Album art.
*/
void SxmAudioChannelInformation::requestSxmGraphics(T_e8_Graphics chnArtAvail, T_e8_Graphics albumArtAvail, uint16 sid)
{
   ETG_TRACE_USR3(("requestSxmGraphics ChnArt status:%d,AlbmArt status:%d", chnArtAvail, albumArtAvail));

   //On startup from channelinformation status always has channel art availability has default,
   //so wait until channel art service to be ready to get valid data.
   if (_sxmChannelart && SXMDSUnknown == _sxmChannelart->getChannelArtSubscriptionState() &&
         (chnArtAvail == T_e8_Graphics__GRAPHICS_NEEDS_DEFAULT || albumArtAvail == T_e8_Graphics__GRAPHICS_NEEDS_DEFAULT))
   {
      sendChannelArtReq(sid, (T_e8_ChnGraphicsTypes__CHANNEL_ART_AVAILABLE_IMAGE_LOGO |
                              T_e8_ChnGraphicsTypes__CHANNEL_ART_AVAILABLE_IMAGE_ALBUM));
      return;
   }

   if (chnArtAvail == albumArtAvail)
   {
      switch (chnArtAvail)
      {
         case T_e8_Graphics__GRAPHICS_NEEDS_REFRESH:
         {
            ETG_TRACE_USR3(("Channel Logo & Album art is available"));
            sendChannelArtReq(sid, (T_e8_ChnGraphicsTypes__CHANNEL_ART_AVAILABLE_IMAGE_LOGO |
                                    T_e8_ChnGraphicsTypes__CHANNEL_ART_AVAILABLE_IMAGE_ALBUM));
            break;
         }
         case T_e8_Graphics__GRAPHICS_NEEDS_DEFAULT:
         {
            //Load default for both.
            SxmAudioDataBindingUtils::instance()->setSxmDefaultAlbumartData();
            SxmAudioDataBindingUtils::instance()->setSxmDefaultChannelLogoData();
            synchronizeAllChannelInfo();
            break;
         }
         default:
         {
            synchronizeAllChannelInfo();
            break;
         }
      }
   }
   else
   {
      //Request for channel
      handleRequestChannelLogo(chnArtAvail);
      handleRequestAlbumart(albumArtAvail);
   }
}


/**
* Helper Function to Request for Channel Logo.
*/
void SxmAudioChannelInformation::handleRequestChannelLogo(T_e8_Graphics chnArtAvail)
{
   switch (chnArtAvail)
   {
      case T_e8_Graphics__GRAPHICS_NEEDS_REFRESH:
      {
         sendChannelArtReq(_channelInfoStatus.getSID(), T_e8_ChnGraphicsTypes__CHANNEL_ART_AVAILABLE_IMAGE_LOGO);
         break;
      }
      case T_e8_Graphics__GRAPHICS_NEEDS_DEFAULT:
      {
         SxmAudioDataBindingUtils::instance()->setSxmDefaultChannelLogoData();
         break;
      }
      default:
      {
         //Do Nothing
         break;
      }
   }
}


/**
* Helper Function to Request for Album art.
*/
void SxmAudioChannelInformation::handleRequestAlbumart(T_e8_Graphics albumArtAvail)
{
   switch (albumArtAvail)
   {
      case T_e8_Graphics__GRAPHICS_NEEDS_REFRESH:
      {
         sendChannelArtReq(_channelInfoStatus.getSID(), T_e8_ChnGraphicsTypes__CHANNEL_ART_AVAILABLE_IMAGE_ALBUM);
         break;
      }
      case T_e8_Graphics__GRAPHICS_NEEDS_DEFAULT:
      {
         SxmAudioDataBindingUtils::instance()->setSxmDefaultAlbumartData();
         break;
      }
      default:
      {
         //Do Nothing
         break;
      }
   }
}


/**
 * Helper Function to handle channel art result.
 */
void SxmAudioChannelInformation::onGetChannelArtListError()
{
   ETG_TRACE_ERR(("SxmAudioChannelInformation::onGetChannelArtListError"));
}


/**
 * Helper Function to handle channel art result.
 */
void SxmAudioChannelInformation::onGetChannelLogoError(const uint16& /*sid*/)
{
   ETG_TRACE_ERR(("SxmAudioChannelInformation::onGetChannelLogoError"));
   SxmAudioDataBindingUtils::instance()->setSxmDefaultAlbumartData();
   SxmAudioDataBindingUtils::instance()->setSxmDefaultChannelLogoData();
   synchronizeAllChannelInfo();
   //Once All data is available to do transition.
   if (_goToSXMMainScreen)
   {
      //post msg to go to audio main & Switch to Preset Mode if scan is active.
      SxmAudioDataBindingUtils::instance()->switchToPresetMode();
      POST_MSG((COURIER_MESSAGE_NEW(SXMAudioGotoAudioMainMsg)()));
      _goToSXMMainScreen = false;
   }
}


/**
 * Function to handle Gui startup
 */
void SxmAudioChannelInformation::onGuiStartupFinished()
{
   ETG_TRACE_USR4(("SxmAudioChannelInformation::onGuiStartupFinished"));
   initializeChannelInfo();
   SxmAudioListHelper* pAudioListHelper = SxmAudioListHelper::instance();
   pAudioListHelper->setPresetList(::std::vector< T_PresetListEntry >());
   pAudioListHelper->updatePresetList(false);
   dp_tclAppHmi_SxmSxmAudioActivePresetBand dp_ActivePresetBand;
   uint8 presetBand = dp_ActivePresetBand.tGetData();
   SxmAudioDataBindingUtils::instance()->updatePresetPageIndicator(presetBand);
   SxmAudioDataBindingUtils::instance()->updateAudioReplayContentData();
   SXMSourceSwitchInterface::setAlertSourceStatus();
}


/**
 * Helper function to synchronize display of all channel information.
 */
void SxmAudioChannelInformation::synchronizeAllChannelInfo()
{
   SxmAudioDataBindingUtils* pInstance = SxmAudioDataBindingUtils::instance();

   ETG_TRACE_USR4(("SxmAudioChannelInformation::synchronizeAllChannelInfo getSxmChannelInfoSyncStatus:%d, pInstance->getSxmPresetListAvailbility():%d", pInstance->getSxmChannelInfoSyncStatus(), pInstance->getSxmPresetListAvailbility()));
   //Synchronize channel information and preset list.
   if (!pInstance->getSxmChannelInfoSyncStatus() && pInstance->getSxmPresetListAvailbility())
   {
      //To handle dynamic updates in preset list.
      tSharedPtrDataProvider dataProvider = SxmAudioListHelper::instance()->getSxmPresetListDataProvider(true);
      pInstance->setSxmAllChannelInfoAvailbility(true);
      if (_noSystemMessageError)
      {
         pInstance->updateAllChannelInformation();
      }
      else
      {
         pInstance->updateAdvisoryMessage();
      }
      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
      pInstance->enablePresetListButton(LEFTANDRIGHT, 1);//enable page up/Down buttons.
      SxmAudioListHelper::instance()->switchPresetBand();
      dp_tclAppHmi_SxmSxmAudioActivePresetBand dp_ActivePresetBand;
      SxmAudioDataBindingUtils::instance()->updatePresetPageIndicator(dp_ActivePresetBand.tGetData());
   }
   else if (pInstance->getSxmChannelInfoSyncStatus())
   {
      if (_noSystemMessageError)
      {
         pInstance->updateAllChannelInformation();
      }
      else
      {
         pInstance->updateAdvisoryMessage();
      }
   }
   else
   {
      pInstance->setSxmAllChannelInfoAvailbility(true);
   }
}


/**
 * initialize channel information
 */
void SxmAudioChannelInformation::initializeChannelInfo(void)
{
   ETG_TRACE_USR4(("SxmAudioChannelInformation::initializeChannelInfo"));
   SxmAudioDataBindingUtils* pAudioUtilsInstance = SxmAudioDataBindingUtils::instance();
   pAudioUtilsInstance->clearSxmChannelInfoData();
   pAudioUtilsInstance->enablePresetListButton(LEFTANDRIGHT, 0);//disable page up/Down buttons.
   pAudioUtilsInstance->displayLoading();
   pAudioUtilsInstance->setGadgetAdvisoryText(Candera::String(SXM__AUDIO_GADGET_LOADING));
}


/**
 * update channel information
 */
void SxmAudioChannelInformation::updateChannelInfo()
{
   //Send Information to update channel information.
   SxmAudioDataBindingUtils::instance()->setSxmChannelInfoData(_channelInfoStatus);
   SxmAudioListHelper::instance()->setSxmChannelInfoData(_channelInfoStatus);

   TChannelInformation channelInfo = {_channelInfoStatus.getChannelNumber(), _channelInfoStatus.getChannelName()};
   SXMCommonInterface::updateMapStatusBar(channelInfo);
}


/**
 * update lists on channel information status
 */
void SxmAudioChannelInformation::updateListOnChannelInfo(uint16 activeChannel, uint16 newChannel)
{
   //uint16 currentActiveMode = SXMCommonInterface::getActiveAppMode();
   SxmAudioDataBindingUtils* pAudioUtilsInstance = SxmAudioDataBindingUtils::instance();
   if (activeChannel != newChannel)
   {
#if 0
      if ((APP_MODE_MAP == currentActiveMode || APP_MODE_HOME == currentActiveMode || APP_MODE_SCROLLMAP == currentActiveMode) && _isEncodertuned)
      {
         postTpopAudioMessage();
         _isEncodertuned = false;
      }
#endif
      if (SXM_AUDIO_REPLAY_PLAY == pAudioUtilsInstance->getSxmInstantReplayStatus())
      {
         pAudioUtilsInstance->setDefaultSxmAudioPlayPauseData(T_e8_IRPlaybackStatus__SXM_AUDIO_IR_PLAY_STATE);
      }
      //Update list screens.
      if (SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__CATEGORYLIST))
      {
         tSharedPtrDataProvider dataProvider = SxmAudioListHelper::instance()->getSxmCategorieListDataProvider();
         POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
      }
      if (SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__CHANNELLIST_3) && !_goToSXMMainScreen)
      {
         tSharedPtrDataProvider dataProvider = SxmAudioListHelper::instance()->getSxmChannelListDataProvider();
         POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
      }
   }
}


/**
 * Restore Dp value for Audio channel information
 */
void SxmAudioChannelInformation::restoreChannelInfo(void)
{
   ETG_TRACE_USR4(("SxmAudioChannelInformation::restoreChannelInfo"));
   initializeChannelInfo();
   SxmAudioDataBindingUtils* pAudioUtilsInstance = SxmAudioDataBindingUtils::instance();
   pAudioUtilsInstance->setSxmAllChannelInfoAvailbility(false);
   if (_sxmChannelart)
   {
      _sxmChannelart->stopLoadingTimer();
      _sxmChannelart->clearAllChannelArtReq();
   }
}


/**
 * Function to set default value on service availability from suspended
 */
void SxmAudioChannelInformation::onServiceResumed(const ::boost::shared_ptr< asf::core::Proxy >& proxy)
{
   ETG_TRACE_USR4(("SxmAudioChannelInformation::onServiceResumed"));
   if (_sxmAudioFiProxy && (proxy == _sxmAudioFiProxy))
   {
      setClockParams();
   }
}


}// namespace Core
}  // namespace App
