/**************************************************************************************
 * @file         : SxmAudioChannelArt.cpp
 * @addtogroup   : AppHmi_Sxm
 * @brief        : Utility class which contains all the necessary business logic required to handle channel art.
 * @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 **************************************************************************************/
#include "hall_std_if.h"   // Including HALL framework code
#include "SxmAudioChannelArt.h"
#include "SxmAudioTypes.h"
#include "App/Core/SxmUtils/SxmUtils.h"
#include "App/Core/SxmCommon/SXMCommonInterface.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXMAUDIO
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::SxmAudioChannelArt::
#include "trcGenProj/Header/SxmAudioChannelArt.cpp.trc.h"
#endif

namespace App {
namespace Core {
using namespace ::sxm_channelart_main_fi;
using namespace ::sxm_channelart_main_fi_types;
using namespace Util;
/**
 * SxmAudioChannelArt Class constructor
 * @param [in] : pSxmAudioFiProxy - SxmAudioFiProxy class instance
 */
SxmAudioChannelArt::SxmAudioChannelArt(::boost::shared_ptr<Sxm_channelart_main_fiProxy> pSxmChannelartFiProxy)
   : _sxmChannelartFiProxy(pSxmChannelartFiProxy), _isPreviousRequestProcessed(true)
   , _actIdForChannelArt(0)
   , _channelartSubStatus(SXMDSUnknown)
   , _loadingTimer(NULL)
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("SxmAudioChannelArt() Constructor"));
   _clientCb.clear();
   //Register for subscription update
   SXMCommonInterface::vRegisterNotification(this, SXMChannelArt, SXMDataServiceStatus);
}


/**
 * SxmAudioChannelArt Class destructor
 */
SxmAudioChannelArt::~SxmAudioChannelArt()
{
   ETG_TRACE_USR4(("~SxmAudioChannelArt() Destructor"));
   _sxmChannelartFiProxy.reset();
   _clientCb.clear();
   _isPreviousRequestProcessed = false;
   _actIdForChannelArt = 0;
   _channelartSubStatus = SXMDSUnknown;
   SXM_UTILS_RELEASE_OBJECT(_loadingTimer);
   //De-Register for subscription update
   SXMCommonInterface::vDeregisterNotification(this, SXMChannelArt, SXMDataServiceStatus);
}


/**
 * Utility function that registers for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioChannelArt::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAudioChannelArt::registerProperties"));
   //Once service is available process channel art updates.
   getChannelart();
}


/**
 * Utility function that deregisters for notifications updates from SXM Audio MIDW layer
 * @param [in] : proxy - SXM Audio FI proxy class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmAudioChannelArt::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("SxmAudioChannelArt::deregisterProperties"));
}


/**
 * Error handling function for get ChannelArtList Method
 * @param [in] : proxy - SXM Channel Art FI proxy proxy class instance
 * @param [in] : error - Error type
 */
void SxmAudioChannelArt::onGetChannelArtListError(const ::boost::shared_ptr< Sxm_channelart_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr<GetChannelArtListError >& error)
{
   ETG_TRACE_ERR(("SxmAudioChannelArt::onGetChannelArtListError"));
   if (!_clientCb.empty() && (FID_GetChannelArtList == _clientCb[0].fID) && (_actIdForChannelArt == error->getAct()))
   {
      _isPreviousRequestProcessed = true;
      _clientCb[0].client->onGetChannelArtListError();
      _clientCb.erase(_clientCb.begin());//erase first element.
      getChannelart();
   }
}


/**
 * Result handling function for get ChannelArtList Method
 * @param [in] : proxy - SXM Channel Art FI proxy proxy class instance
 * @param [in] : error - result type
 */
void SxmAudioChannelArt::onGetChannelArtListResult(const ::boost::shared_ptr<Sxm_channelart_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr<GetChannelArtListResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioChannelArt::onGetChannelArtListResult _actIdForChannelArtList = %u , result->getAct()= %u", _actIdForChannelArt, result->getAct()));
   if (!_clientCb.empty() && (FID_GetChannelArtList == _clientCb[0].fID) && (_actIdForChannelArt == result->getAct()))
   {
      ETG_TRACE_USR4(("SxmAudioChannelArt::onGetChannelArtListResult"));
      _isPreviousRequestProcessed = true;
      ::std::vector< ChannelGraphicsType > lArtList;
      uint16 size = result->getImageDataList().size();
      for (uint16 idx = 0; idx < size; ++idx)
      {
         const T_ImageData& imgdata = result->getImageDataList()[idx];
         lArtList.push_back(ChannelGraphicsType(imgdata.getSID(), imgdata.getImageData(), imgdata.getImageDataSize()));
      }
      _clientCb[0].client->onGetChannelArtListResult(lArtList);
      _clientCb.erase(_clientCb.begin());//erase first element.
      getChannelart();
   }
}


/**
 * Error handling function for get ChannelArt Method
 * @param [in] : proxy - SXM Channel Art FI proxy proxy class instance
 * @param [in] : error - Error type
 */
void SxmAudioChannelArt::onGetChannelArtError(const ::boost::shared_ptr< Sxm_channelart_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< GetChannelArtError >& error)
{
   ETG_TRACE_ERR(("SxmAudioChannelArt::onGetChannelArtError"));
   if (!_clientCb.empty() && (FID_GetChannelArt == _clientCb[0].fID) && (_actIdForChannelArt == error->getAct()))
   {
      _isPreviousRequestProcessed = true;
      _clientCb[0].client->onGetChannelLogoError(_clientCb[0].listOfSid[0]);
      _clientCb.erase(_clientCb.begin());//erase first element.
      getChannelart();
   }
}


/**
 * Result handling function for get ChannelArt Method
 * @param [in] : proxy - SXM Channel Art FI proxy proxy class instance
 * @param [in] : error - result type
 */
void SxmAudioChannelArt::onGetChannelArtResult(const ::boost::shared_ptr< Sxm_channelart_main_fiProxy >& /*proxy*/,
      const boost::shared_ptr< GetChannelArtResult >& result)
{
   ETG_TRACE_USR4(("SxmAudioChannelArt::onGetChannelArtResult _actIdForChannelArt = %u result->getAct()= %u", _actIdForChannelArt, result->getAct()));
   if (!_clientCb.empty() && (FID_GetChannelArt == _clientCb[0].fID) && (_actIdForChannelArt == result->getAct()))
   {
      ETG_TRACE_USR4(("SxmAudioChannelArt::onGetChannelArtResult"));
      _isPreviousRequestProcessed = true;
      _clientCb[0].client->onGetChannelLogoResult(result->getSID(), result->getChannelGraphicsType(), result->getU8ImageData(), result->getAlbumImageData());
      _clientCb.erase(_clientCb.begin());//erase first element.
      getChannelart();
   }
}


/**
 * Function handles method request for channel art list.
 * @param [in] : client - instance of clients.
 * @param [in] : listOfSid - list of service id.
 * @param [in] : channelGraphicsType - graphics type requested(Ex : Channellogo,Album art).
 */
void SxmAudioChannelArt::sendChannelArtListMethodRequest(ISxmAudioChannelArtResponse* client, const ::std::vector< uint16 >& listOfSid,
      const uint16& channelGraphicsType)
{
   ETG_TRACE_USR4(("SxmAudioChannelArt::sendChannelArtListMethodRequest graphictype:%d", channelGraphicsType));
   if ((_loadingTimer != NULL) && (_loadingTimer->getStatus() == Timer::Expired) && (SXMDSUnknown == _channelartSubStatus))
   {
      client->onGetChannelArtListResult(::std::vector< ChannelGraphicsType >());
   }
   else
   {
      _clientCb.push_back(ChannelArtCallback(FID_GetChannelArtList, client, listOfSid, channelGraphicsType));
      getChannelart();
   }
}


/**
 * Function handles method request for channel logo.
 * @param [in] : client - instance of clients.
 * @param [in] : sid - service id.
 * @param [in] : channelGraphicsType - graphics type requested(Ex : Channellogo,Album art).
 */
void SxmAudioChannelArt::sendChannelLogoMethodRequest(ISxmAudioChannelArtResponse* client, const uint16& sid,
      const uint16& channelGraphicsType)
{
   ETG_TRACE_USR4(("SxmAudioChannelArt::sendChannelLogoMethodRequest sid:%d,graphictype:%d", sid, channelGraphicsType));
   std::vector<uint16> tmp;
   tmp.push_back(sid);
   if ((_loadingTimer != NULL) && (_loadingTimer->getStatus() == Timer::Expired) && (SXMDSUnknown == _channelartSubStatus))
   {
      client->onGetChannelLogoResult(sid, channelGraphicsType, std::vector<uint8>(), std::vector<uint8>());
   }
   else
   {
      _clientCb.push_back(ChannelArtCallback(FID_GetChannelArt, client, tmp, channelGraphicsType));
      getChannelart();
   }
}


/**
 * Helper function to synchronize channel art requests.
 */
void SxmAudioChannelArt::getChannelart()
{
   ETG_TRACE_USR4(("SxmAudioChannelArt::getChannelart _clientCb.size:%d", !_clientCb.empty()));
   ETG_TRACE_USR4(("SxmAudioChannelArt::getChannelart _isPreviousRequestProcessed:%d,_channelartSubStatus:%d", _isPreviousRequestProcessed, _channelartSubStatus));
   if (_sxmChannelartFiProxy && !_clientCb.empty() && _isPreviousRequestProcessed && (SXMDSUnknown != _channelartSubStatus))
   {
      T_e8_ChnGraphicsTypes enGraphicsType = static_cast<T_e8_ChnGraphicsTypes>(_clientCb[0].channelGraphicsType);//0:First entry in queue
      ETG_TRACE_USR4(("SxmAudioChannelArt::getChannelart _clientCb[0].fID:%d", _clientCb[0].fID));
      if (_clientCb[0].fID == FID_GetChannelArt)
      {
         _actIdForChannelArt = _sxmChannelartFiProxy->sendGetChannelArtStart(*this, _clientCb[0].listOfSid[0], enGraphicsType);
      }
      else
      {
         _actIdForChannelArt = _sxmChannelartFiProxy->sendGetChannelArtListStart(*this, _clientCb[0].listOfSid, enGraphicsType);
      }
      _isPreviousRequestProcessed = false;
   }
}


/**
 * Helper function to synchronize channel art requests.
 */
void SxmAudioChannelArt::sendDefaultChannelart()
{
   ETG_TRACE_USR4(("SxmAudioChannelArt::getDefaultChannelart _clientCb.size:%d", !_clientCb.empty()));
   for (std::vector<ChannelArtCallback>::iterator itr = _clientCb.begin() ; itr != _clientCb.end(); ++itr)
   {
      ETG_TRACE_USR4(("SxmAudioChannelArt::getDefaultChannelart _clientCb[0].size:%d", _clientCb.size()));
      ETG_TRACE_USR4(("SxmAudioChannelArt::getDefaultChannelart _clientCb[0].fID:%d", _clientCb[0].fID));
      T_e8_ChnGraphicsTypes enGraphicsType = static_cast<T_e8_ChnGraphicsTypes>(_clientCb[0].channelGraphicsType);
      if ((FID_GetChannelArt == itr->fID))
      {
         itr->client->onGetChannelLogoResult(itr->listOfSid[0], enGraphicsType, std::vector<uint8>(), std::vector<uint8>());
      }
      else
      {
         itr->client->onGetChannelArtListResult(::std::vector< ChannelGraphicsType >());
      }
      //_clientCb.erase(itr);//erase first element.
   }
   _clientCb.clear();
}


/**
 * Helper function to know service state.
 */
bool SxmAudioChannelArt::isChannelArtServiceAvailable()
{
   return (_sxmChannelartFiProxy && ::asf::core::ServiceState__Available == _sxmChannelartFiProxy->getServiceState());
}


/**
 * Helper function to get subscription state.
 */
uint8 SxmAudioChannelArt::getChannelArtSubscriptionState()
{
   return _channelartSubStatus;
}


/**
 * Helper function to start loading timer.
 */
void SxmAudioChannelArt::startLoadingTimer()
{
   ETG_TRACE_USR4(("SxmAudioChannelArt::startLoadingTimer"));
   if (_loadingTimer == NULL)
   {
      //Timer Initialize
      _loadingTimer = new Timer(10000);//10000 ms:
   }
   if (_loadingTimer != NULL)
   {
      if (Timer::Stopped == _loadingTimer->getStatus())
      {
         _loadingTimer->start();
      }
      else if (Timer::Expired == _loadingTimer->getStatus())
      {
         _loadingTimer->restart();
      }
   }
}


/**
 * Helper function to stop loading timer.
 */
void SxmAudioChannelArt::stopLoadingTimer()
{
   ETG_TRACE_USR4(("SxmAudioChannelArt::stopLoadingTimer"));
   if ((_loadingTimer != NULL) && (Timer::Stopped != _loadingTimer->getStatus()))
   {
      _loadingTimer->stop();
   }
}


/**
 * Helper function to clear all channel art request
 */
void SxmAudioChannelArt::clearAllChannelArtReq()
{
   ETG_TRACE_USR4(("SxmAudioChannelArt::clearAllChannelArtReq _clientCb.size:%d", _clientCb.size()));
   if (!_clientCb.empty())
   {
      _clientCb.clear();
      _isPreviousRequestProcessed = true;
   }
}


/**
 * Helper function to get all channel art timer expiry
 */
uint8 SxmAudioChannelArt::getChannelArtTimerStatus()
{
   uint8 timerStatus = static_cast<uint8>(Timer::Stopped);
   if (_loadingTimer != NULL)
   {
      timerStatus = _loadingTimer->getStatus();
   }
   ETG_TRACE_USR4(("SxmAudioChannelArt::getChannelArtTimerStatus timerStatus:%d", timerStatus));
   return timerStatus;
}


/**
 * Callback to handle Courier Message TimerExpiredMsg Request
 * @param [in] : Reference of Courier::TimerExpiredMsg
 * @return     : True-When Message is processed.
 */
bool SxmAudioChannelArt::onCourierMessage(const TimerExpiredMsg& oMsg)
{
   bool status = false;
   if (oMsg.GetTimer() == _loadingTimer)
   {
      ETG_TRACE_USR4(("SxmAudioChannelArt::TimerExpiredMsg _channelartSubStatus:%d", _channelartSubStatus));
      if (SXMDSUnknown == _channelartSubStatus)
      {
         sendDefaultChannelart();
      }
      status = true;
   }
   return status;
}


/**
 * Call Back function for advisory status.
 * @param[in] : enServiceState - Status
 * @param[in] : enEventType  - DataServiceStatus/Antenna Status
 */
void SxmAudioChannelArt::vNotifyAdvisoryStatus(enSXMServiceType enServiceType, enSXMDataServiceAdvisory enServiceState , enSXMEventNotificationType enEventType)
{
   ETG_TRACE_USR4(("SxmAudioChannelArt::vNotify enServiceState:%d,enEventType:%d", enServiceState, enEventType));
   if ((SXMChannelArt == enServiceType) && (_channelartSubStatus != enServiceState))
   {
      _channelartSubStatus = enServiceState;
      getChannelart();
   }
}


/*
* restore channel art info on restore factory settings
*/
void SxmAudioChannelArt::restoreChannelArtInfo()
{
   clearAllChannelArtReq();
}


/**
 * Function to set default value on service availability from suspended
 */
void SxmAudioChannelArt::onServiceResumed(const ::boost::shared_ptr< asf::core::Proxy >& proxy)
{
   ETG_TRACE_USR4(("SxmAudioChannelArt::onServiceResumed"));
   if ((proxy == _sxmChannelartFiProxy) && (_sxmChannelartFiProxy))
   {
      getChannelart();
   }
}


}  // namespace Core
}  // namespace App
