/**************************************************************************************
 * @file        : SXMSourceSwitch.h
 * @addtogroup  : AppHmi_Sxm
 * @brief       : SXM SourceSwitch utility class required to handle source change requests.
 * @copyright   : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                The reproduction, distribution and utilization of this file as
 *                well as the communication of its contents to others without express
 *                authorization is prohibited. Offenders will be held liable for the
 *                payment of damages. All rights reserved in the event of the grant
 *                of a patent, utility model or design.
 **************************************************************************************/
#ifndef _SXMSourceSwitch_H_
#define _SXMSourceSwitch_H_

#include "AppBase/ServiceAvailableIF.h"
#include "bosch/cm/ai/hmi/masteraudioservice/AudioSourceChangeProxy.h"
#include "bosch/cm/ai/hmi/masteraudioservice/SoundPropertiesProxy.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "App/Core/SxmUtils/SxmUtilsSingleton.h"
#include "App/Core/SxmUtils/SxmCommonTypes.h"
#include "AppHmi_SxmMessages.h"
#include "App/Core/SxmCommon/ISxmHMIServiceStateHandler.h"
#include "Common/ApplicationSwitchClientComponent/ApplicationSwitchClientComponent.h"
#include "App/Core/ClientHandler/DiagnosticsClient/DiagnosticsClient.h"

#ifndef AUDIO_SRC_CHNG_NAMESPACE
#define AUDIO_SRC_CHNG_NAMESPACE ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange
#endif

#ifndef AUDIO_SOUND_PROP_NAMESPACE
#define AUDIO_SOUND_PROP_NAMESPACE ::bosch::cm::ai::hmi::masteraudioservice::SoundProperties
#endif

namespace App {
namespace Core {
/**
 * SXM SourceSwitch utility class required to handle source change requests.
 * @class : SXMSourceSwitch
 */
class SXMSourceSwitch
   : public SxmUtilsSingleton<SXMSourceSwitch>
   , public hmibase::ServiceAvailableIF
   , public StartupSync::PropertyRegistrationIF
   , public AUDIO_SRC_CHNG_NAMESPACE::ActivateSourceCallbackIF
   , public AUDIO_SRC_CHNG_NAMESPACE::ActiveSourceCallbackIF
   , public AUDIO_SRC_CHNG_NAMESPACE::ActiveSourceListCallbackIF
   , public AUDIO_SRC_CHNG_NAMESPACE::DeactivateSourceCallbackIF
   , public AUDIO_SOUND_PROP_NAMESPACE::MuteStateMapCallbackIF
   , public ISxmHMIServiceStateHandler
{
   public:

      /**
       * SXMSourceSwitch Class constructor
       */
      SXMSourceSwitch();
      /**
       * SXMSourceSwitch Class destructor
       */
      virtual ~SXMSourceSwitch();
      /**
       * Utility function that registers for notifications updates from AudioSourceChange.
       */
      void registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange);
      /**
       * Utility function that deregisters for notifications updates from AudioSourceChange.
       */
      void deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange);
      /**
       * Utility function that deregisters for notifications updates from AudioSourceChange.
       */
      void onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/);
      /**
       * Utility function that registers for notifications updates from AudioSourceChange.
       */
      void onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/);
      /**
       * Virtual function implemented to get update of ActivateSource Error
       */
      virtual void onActivateSourceError(const ::boost::shared_ptr< AUDIO_SRC_CHNG_NAMESPACE::AudioSourceChangeProxy >& /*proxy*/,
                                         const ::boost::shared_ptr< AUDIO_SRC_CHNG_NAMESPACE::ActivateSourceError >& /*error*/);
      /**
       * Virtual function implemented to get update of ActivateSource Response
       */
      virtual void onActivateSourceResponse(const ::boost::shared_ptr< AUDIO_SRC_CHNG_NAMESPACE::AudioSourceChangeProxy >& proxy,
                                            const ::boost::shared_ptr< AUDIO_SRC_CHNG_NAMESPACE::ActivateSourceResponse >& /*response*/);
      /**
       * Virtual function implemented to get update of ActiveSource Response
       */
      virtual void onActiveSourceError(const ::boost::shared_ptr< AUDIO_SRC_CHNG_NAMESPACE::AudioSourceChangeProxy >& proxy,
                                       const ::boost::shared_ptr< AUDIO_SRC_CHNG_NAMESPACE::ActiveSourceError >& error);
      /**
       * Virtual function implemented to get update of ActiveSource Response
       */
      virtual void onActiveSourceUpdate(const ::boost::shared_ptr< AUDIO_SRC_CHNG_NAMESPACE::AudioSourceChangeProxy >& proxy,
                                        const ::boost::shared_ptr< AUDIO_SRC_CHNG_NAMESPACE::ActiveSourceUpdate >& update);

      virtual void onActiveSourceListError(const ::boost::shared_ptr< AUDIO_SRC_CHNG_NAMESPACE::AudioSourceChangeProxy >& proxy,
                                           const ::boost::shared_ptr< AUDIO_SRC_CHNG_NAMESPACE::ActiveSourceListError >& error);

      virtual void onActiveSourceListUpdate(const ::boost::shared_ptr< AUDIO_SRC_CHNG_NAMESPACE::AudioSourceChangeProxy >& /*proxy*/,
                                            const ::boost::shared_ptr< AUDIO_SRC_CHNG_NAMESPACE::ActiveSourceListUpdate >& update);

      virtual void onDeactivateSourceError(const ::boost::shared_ptr< AUDIO_SRC_CHNG_NAMESPACE::AudioSourceChangeProxy >& proxy,
                                           const ::boost::shared_ptr< AUDIO_SRC_CHNG_NAMESPACE::DeactivateSourceError >& error);

      virtual void onDeactivateSourceResponse(const ::boost::shared_ptr< AUDIO_SRC_CHNG_NAMESPACE::AudioSourceChangeProxy >& proxy,
                                              const ::boost::shared_ptr< AUDIO_SRC_CHNG_NAMESPACE::DeactivateSourceResponse >& response);

      virtual void onMuteStateMapError(const ::boost::shared_ptr< AUDIO_SOUND_PROP_NAMESPACE::SoundPropertiesProxy >& /*proxy*/,
                                       const ::boost::shared_ptr< AUDIO_SOUND_PROP_NAMESPACE::MuteStateMapError >& /*update*/);

      virtual void onMuteStateMapUpdate(const ::boost::shared_ptr< AUDIO_SOUND_PROP_NAMESPACE::SoundPropertiesProxy >& /*proxy*/,
                                        const ::boost::shared_ptr< AUDIO_SOUND_PROP_NAMESPACE::MuteStateMapUpdate >& /*update*/);
      /**
       * Helper function to request for source switch to sxm.
       */
      bool handleSourceChangeRequest(uint32 sourceID, bool activate, bool forceactive, bool isAlert);
      /**
       * Helper function to get Sxm Source connection Status.
       */
      uint8 getSxmSourceStatus(void) ;
      uint8 getSxmNewsSourceStatus();
      /**
       * Helper function to get Sxm source active status.
       */
      bool isSxmSourceActive(void) const;
      /**
       * Helper function to get Sxm SF News source active status.
       */
      bool isSxmNewsSourceActive(void) const;
      /**
      * Helper function to set Sxm tune scan status.
      */
      void setSxmAudioScanStatus(bool isActive);
      /**
      * Helper function to get Sxm tune scan status.
      */
      bool getSxmAudioScanStatus();
      /**
      * Helper function to get Sxm Current Region Mute Status.
      */
      bool getCurrentRegionMuteStatus();
      /**
        *  Call Back function for system appmode status.
        */
      void vNotifyAppModeChange(unsigned short int lactiveAppMode);
      /**
      *  Function to update connection state
      */
      void setConnectionState(uint8 activeRegion);
      /**
       *  Function to update Mute state
      */
      void setMuteStatus();
      /**
       *  Call Back function for service status.
       */
      void vNotifyServiceStatus(unsigned short int lEntryExitStatus, unsigned int serviceType);
      /**
        *  Function to request Sxm Cabin Source Deactivation
       */
      void requestSxmCabinSourceDeactivation(int srcid, int connectionState);
      /**
        *  Function to request Sxm Cabin Source Activation
        */
      void requestSxmCabinSourceActivation(int srcid);
      void vNotifyAppState(unsigned short int appState);
      regionText getPopupText();
      void setAlertSourceStatus();
      uint8 getAlertSourceStatus();
      int32 getPrevSource();
      void handleSxmNewsSourceDeactivation(uint32 srcid, bool activate, uint sinkId);

      /**
       * Function to get the sink source status
       */
      void getSinkSourceStatus(uint16 sinkId, int32& srcId, uint32& connStatus);

   private:
      friend class  SxmUtilsSingleton<SXMSourceSwitch>;

      //Member Variables
      ::boost::shared_ptr< AUDIO_SRC_CHNG_NAMESPACE::AudioSourceChangeProxy > _audioSrcChgProxy;
      ::boost::shared_ptr< AUDIO_SOUND_PROP_NAMESPACE::SoundPropertiesProxy > _soundPropertiesProxy;
      enConnectionState _cockpitSxmConnectionState;
      enConnectionState _cockpitSxmNewsConnectionState;
      enConnectionState _cabinSxmConnectionState;
      enConnectionState _cabinSxmNewsConnectionState;
      const int INITIALISE_SUBSRC_ID;
      bool _isScanActive;
      int32 _currSourceID;
      int32 _prevSourceID;
      std::map <uint16, bool> _muteStatusForRegion;
      std::map<Region, _Sink> _regionSink;
      std::map<uint16, std::pair<int32, uint32> > _sinkSrcStatus;
      std::map<uint16, std::pair<int32, uint32> > _prevSinkSrcStatus;
      std::map<uint8, regionText> _popupText;
      std::bitset<6> _alertSettingSxmSrcStatus;
      std::bitset<9> _sinksPendingSrcChng;//9 - _Sink enum value is till 8
      bool _activate;
      bool _deactivate;
      bool _isSxmSelfContextRequired;
      uint32 _pendingActivationSourceID;
      bool _isSxmConnected;
      //Member Functions
      /**
       * Helper function to check sxm source status.
       */
      void updateSxmSourceStatus(const ::std::vector< AUDIO_SRC_CHNG_NAMESPACE::sourceData >& sourceList);
      void setSourceStatus(int32 srcId, uint16 sinkId, uint32 connStatus);
      void IsSrcChgReqdInOtherSinks(uint32 reqSourceID, bool activate);
      void initRegionAndStatus();
      void initAlertSourceText();
      void handlePendingDeactivation(const ::std::vector< AUDIO_SRC_CHNG_NAMESPACE::sourceData >& sourceList);
      bool isPauseAllowed();
      void updateConnectionState(const ::std::vector< AUDIO_SRC_CHNG_NAMESPACE::sourceData >& sourceList);
      void RunTimeSrcSXM();
      void setSXMSourceConnectionStatus(const int32& srcId);
      /**
       * Helper function to check if sink is already connected, if yes then don't send Source change req
       */
      void checkSinkConnectionState();
      /**
        * Callback to handle Courier Message SXMSourceChangeReqMsg Request
        */
      bool onCourierMessage(const SXMSourceChangeReqMsg& /*oMsg*/);

      COURIER_MSG_MAP_BEGIN(TR_CLASS_APPHMI_SXM_COURIER_PAYLOAD_MODEL_COMP)
      ON_COURIER_MESSAGE(SXMSourceChangeReqMsg)
      COURIER_MSG_MAP_DELEGATE_DEFAULT_BEGIN()
      COURIER_MSG_MAP_DELEGATE_END()

   protected:
};


}  // namespace Core
}  // namespace App

#undef AUDIO_SRC_CHNG_NAMESPACE
#undef AUDIO_SOUND_PROP_NAMESPACE
#undef APP_SWITCH_NAMESPACE

#endif
