/**************************************************************************************
 * @file        : SXMSourceSwitch.cpp
 * @addtogroup  : AppHmi_Sxm
 * @brief       : SXM utility class required to interact with AudioSourceChange.
 * @copyright   : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                The reproduction, distribution and utilization of this file as
 *                well as the communication of its contents to others without express
 *                authorization is prohibited. Offenders will be held liable for the
 *                payment of damages. All rights reserved in the event of the grant
 *                of a patent, utility model or design.
 **************************************************************************************/
#include "hmi_trace_if.h"
#include "CgiExtensions/CourierMessageMapper.h"
#include "App/Core/SxmUtils/SxmUtils.h"
#include "App/Core/SxmCommon/SXMCommonInterface.h"
#include "App/Core/ClientHandler/ApplicationSwitchClientHandler/ApplicationSwitchClientHandler.h"
#include "App/Core/SxmAudio/SxmAudioDataBindingUtils.h"
#include "App/Core/SxmAudio/SxmAudioTypes.h"
#include "SXMSourceSwitch.h"
#include "App/Core/ClientHandler/DiagnosticsClient/DiagnosticsClientInterface.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXM_HALL
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::SXMSourceSwitch::
#include "trcGenProj/Header/SXMSourceSwitch.cpp.trc.h"
#endif

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_hmi_06_if.h"

namespace App {
namespace Core {
using namespace ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange;
using namespace ::bosch::cm::ai::hmi::masteraudioservice::SoundProperties;

/**
 * SXMSourceSwitch Class constructor
 */
SXMSourceSwitch::SXMSourceSwitch()
   : _audioSrcChgProxy(AudioSourceChangeProxy::createProxy("audioSourceChangePort", *this))
   , _soundPropertiesProxy(SoundPropertiesProxy::createProxy("soundPropertiesPort", *this))
   , _cockpitSxmConnectionState(disconnected)
   , _cockpitSxmNewsConnectionState(disconnected)
   , _cabinSxmConnectionState(disconnected)
   , _cabinSxmNewsConnectionState(disconnected)
   , INITIALISE_SUBSRC_ID(-1)
   , _isScanActive(false)
   , _currSourceID(SRC_INVALID)
   , _prevSourceID(SRC_INVALID)
   , _activate(false)
   , _deactivate(false)
   , _isSxmSelfContextRequired(false)
   , _pendingActivationSourceID(SRC_INVALID)
   , _isSxmConnected(false)
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("SXMSourceSwitch() Constructor"));
   SXM_UTILS_REGISTER_PROPERTY_CALLBACKS(StartupSync::getInstance(), this, _audioSrcChgProxy);
   SXM_UTILS_REGISTER_PROPERTY_CALLBACKS(StartupSync::getInstance(), this, _soundPropertiesProxy);
   initRegionAndStatus();
   initAlertSourceText();
   SXMCommonInterface::registerServiceEntryExitNotification(this, SXMAudio);
   SXMCommonInterface::registerApplicationStateNotication(this, SXMAudio);
   _sinkSrcStatus.clear();
   _prevSinkSrcStatus.clear();
   _alertSettingSxmSrcStatus.reset();
   _sinksPendingSrcChng.reset();
   _muteStatusForRegion.clear();
}


/**
 * SXMSourceSwitch Class destructor
 */
SXMSourceSwitch::~SXMSourceSwitch()
{
   ETG_TRACE_USR4(("~SXMSourceSwitch() Destructor"));
   ETG_I_UNREGISTER_FILE();
   _audioSrcChgProxy.reset();
   _soundPropertiesProxy.reset();
   _cockpitSxmConnectionState = disconnected;
   _cockpitSxmNewsConnectionState = disconnected;
   _cabinSxmConnectionState = disconnected;
   _cabinSxmNewsConnectionState = disconnected;
   _isScanActive = false;
   _currSourceID = SRC_INVALID;
   _prevSourceID = SRC_INVALID;
   _regionSink.clear();
   _sinkSrcStatus.clear();
   _prevSinkSrcStatus.clear();
   _popupText.clear();
   _alertSettingSxmSrcStatus.reset();
   _sinksPendingSrcChng.reset();
   _activate = false;
   _deactivate = false;
   SXMCommonInterface::deregisterServiceEntryExitNotification(this, SXMAudio);
   SXMCommonInterface::deregisterApplicationStateNotication(this, SXMAudio);
   _muteStatusForRegion.clear();
   _isSxmSelfContextRequired = false;
   _pendingActivationSourceID = SRC_INVALID;
   _isSxmConnected = false;
}


/**
*  Initialize the region Sink-ID's
*/
void SXMSourceSwitch::initRegionAndStatus()
{
   _regionSink.clear();
   _regionSink[Region__REG_COCKPIT] = COCKPIT_SINK_ID;
   _regionSink[Region__REG_A] = CABIN_A_SINK_ID;
   _regionSink[Region__REG_B] = CABIN_B_SINK_ID;

   _sinkSrcStatus[COCKPIT_SINK_ID] = std::make_pair(SRC_INVALID, disconnected);
   _sinkSrcStatus[CABIN_A_SINK_ID] = std::make_pair(SRC_INVALID, disconnected);
   _sinkSrcStatus[CABIN_B_SINK_ID] = std::make_pair(SRC_INVALID, disconnected);

   _prevSinkSrcStatus[COCKPIT_SINK_ID] = std::make_pair(SRC_SXM_NEWS, disconnected);//used only for SXM NEWS
   _prevSinkSrcStatus[CABIN_A_SINK_ID] = std::make_pair(SRC_SXM_NEWS, disconnected);
   _prevSinkSrcStatus[CABIN_B_SINK_ID] = std::make_pair(SRC_SXM_NEWS, disconnected);
}


/**
 * Utility function that registers for notifications updates from AudioSourceChange.
 * @param [in] : proxy - AudioSourceChange class instance
 * @param [in] : stateChange - Current ASF state
 */
void SXMSourceSwitch::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR1(("SXMSourceSwitch registerProperties"));
}


/**
 * Utility function that deregisters for notifications updates from AudioSourceChange.
 * @param [in] : proxy - AudioSourceChange class instance
 * @param [in] : stateChange - Current ASF state
 */
void SXMSourceSwitch::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR1(("SXMSourceSwitch deregisterProperties"));
}


/**
 * Utility function that registers for notifications updates from AudioSourceChange.
 * @param [in] : proxy - AudioSourceChange class instance
 * @param [in] : stateChange - Current ASF state
 */
void SXMSourceSwitch::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("SXMSourceSwitch::onAvailable"));
   if ((_audioSrcChgProxy) && (proxy == _audioSrcChgProxy))
   {
      _audioSrcChgProxy->sendActiveSourceGet(*this);
      _audioSrcChgProxy->sendActiveSourceRegister(*this);
      _audioSrcChgProxy->sendActiveSourceListRegister(*this);
   }
   if ((_soundPropertiesProxy) && (proxy == _soundPropertiesProxy))
   {
      _soundPropertiesProxy->sendMuteStateMapRegister(*this);
   }
}


/**
 * Utility function that deregisters for notifications updates from AudioSourceChange.
 * @param [in] : proxy - AudioSourceChange class instance
 * @param [in] : stateChange - Current ASF state
 */
void SXMSourceSwitch::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("SXMSourceSwitch::onUnavailable"));
   if ((_audioSrcChgProxy) && (proxy == _audioSrcChgProxy))
   {
      //_audioSrcChgProxy->sendSourceActivatedDeregisterAll();
      _audioSrcChgProxy->sendActiveSourceDeregisterAll();
      _audioSrcChgProxy->sendActiveSourceListDeregisterAll();
   }
   if ((_soundPropertiesProxy) && (proxy == _soundPropertiesProxy))
   {
      _soundPropertiesProxy->sendMuteStateMapDeregisterAll();
   }
}


/**
 * Virtual function implemented to get update of ActivateSource Error
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'AudioSourceChangeProxy'
 * @param[in]      : error : The error message of 'ActivateSource'
 */
void SXMSourceSwitch::onActivateSourceError(const ::boost::shared_ptr< AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< ActivateSourceError >& /*error*/)
{
   ETG_TRACE_ERR(("onActivateSourceError received"));
}


/**
 * Virtual function implemented to get update of ActivateSource Response
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'AudioSourceChangeProxy'
 * @param[in]      : response : The response message of 'ActivateSource'
 */
void SXMSourceSwitch::onActivateSourceResponse(const ::boost::shared_ptr< AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< ActivateSourceResponse >& /*response*/)
{
   ETG_TRACE_COMP(("SXMSourceSwitch::onActivateSourceResponse"));
}


/**
 * Virtual function implemented to get update of ActivateSource Error
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'AudioSourceChangeProxy'
 * @param[in]      : error : The error message of 'ActivateSource'
 */
void SXMSourceSwitch::onDeactivateSourceError(const ::boost::shared_ptr< AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< DeactivateSourceError >& /*error*/)
{
   ETG_TRACE_ERR(("onDeactivateSourceError received"));
}


/**
 * Virtual function implemented to get update of ActivateSource Response
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'AudioSourceChangeProxy'
 * @param[in]      : response : The response message of 'ActivateSource'
 */
void SXMSourceSwitch::onDeactivateSourceResponse(const ::boost::shared_ptr< AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< DeactivateSourceResponse >& /*response*/)
{
   ETG_TRACE_COMP(("SXMSourceSwitch::onDeactivateSourceResponse"));
}


/**
 * Virtual function implemented to get update of ActiveSource Response
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'AudioSourceChangeProxy'
 * @param[in]      : response : The response message of 'ActiveSource'
 */
void SXMSourceSwitch::onActiveSourceError(const ::boost::shared_ptr< AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< ActiveSourceError >& /*error*/)
{
   ETG_TRACE_ERR(("onActiveSourceError received"));
}


/**
 * Virtual function implemented to get update of ActiveSource Response
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'AudioSourceChangeProxy'
 * @param[in]      : Update : The response message of 'ActiveSource'
 */
void SXMSourceSwitch::onActiveSourceUpdate(const ::boost::shared_ptr< AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< ActiveSourceUpdate >& update)
{
   ETG_TRACE_COMP(("SXMSourceSwitch::onActiveSourceUpdate"));
   if (update && update->hasActiveSource())
   {
      sourceData source = update->getActiveSource();
      int32 activeSource = source.getSrcId();
      if ((activeSource == SRC_MIC2_ANNOUNCEMENT) ||
            (activeSource == SRC_MIC3_ANNOUNCEMENT) ||
            (activeSource == SRC_NAVI_SPEECH))
      {
         return;
      }
      uint16 sinkID = source.getSinkId();
      uint32 connStatus = update->getActiveSource().getConnectionState();
      enConnectionState state = static_cast<enConnectionState>(connStatus);
      uint8 activeRegion = ApplicationSwitchClientHandler::instance()->getActiveRegion();
      ETG_TRACE_USR1(("SXMSourceSwitch::onActiveSourceUpdate SrcId: %d  SinkId: %d  connStatus: %d",
                      activeSource, sinkID, connStatus));
      ETG_TRACE_COMP(("SXMSourceSwitch::onActiveSourceUpdate sinksPending : %d", _sinksPendingSrcChng.count()));
      setSXMSourceConnectionStatus(activeSource);
      if ((sinkID == COCKPIT_SINK_ID) || (sinkID == CABIN_A_SINK_ID) || (sinkID == CABIN_B_SINK_ID))
      {
         setSourceStatus(activeSource, sinkID, connStatus);
         if ((activeSource == SRC_TUNER_XM) || (activeSource == SRC_SXM_NEWS))
         {
            /*
              if (!_sinksPendingSrcChng.any() && !_deactivate && state == connected && activeSource == SRC_TUNER_XM)//No pending deactivation
              {
                 //if SXM_NEWS is playing in one region and SXM gets connected in other region then SXM_NEWS should be deactivated
                 if (_sinkSrcStatus[COCKPIT_SINK_ID].first == SRC_TUNER_XM && _sinkSrcStatus[COCKPIT_SINK_ID].second == connected)
                 {
                    if (_sinkSrcStatus[CABIN_A_SINK_ID].first == SRC_SXM_NEWS || _sinkSrcStatus[CABIN_B_SINK_ID].first == SRC_SXM_NEWS)
                    {
                       POST_MSG((COURIER_MESSAGE_NEW(SXMDeActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__ITPOP_SPORTSFLASH_RETURN)));
                       //will abort SF and give source change request as deactivate, wherever alert is set
                       POST_MSG((COURIER_MESSAGE_NEW(SxmSFReturnChannelReqMsg)()));
                    }
                 }
                 else if (_sinkSrcStatus[CABIN_A_SINK_ID].first == SRC_TUNER_XM && _sinkSrcStatus[CABIN_A_SINK_ID].second == connected)
                 {
                    if (_sinkSrcStatus[COCKPIT_SINK_ID].first == SRC_SXM_NEWS)
                    {
                       POST_MSG((COURIER_MESSAGE_NEW(SXMDeActivatePopUpMsg)(Sxm_SxmPopups_SXM_RADIO__ITPOP_SPORTSFLASH_RETURN)));
                       //will abort SF and give source change request as deactivate, wherever alert is set
                       POST_MSG((COURIER_MESSAGE_NEW(SxmSFReturnChannelReqMsg)()));
                    }
                 }
              } */
            if (_sinksPendingSrcChng.any() && _activate)
            {
               sourceData srcData;
               if (_sinksPendingSrcChng.test(COCKPIT_SINK_ID))
               {
                  srcData.setSrcId(_pendingActivationSourceID);
                  srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
                  srcData.setConnectionState(STATE_UNKNOWN);
                  srcData.setSinkId(COCKPIT_SINK_ID);
                  ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
                  _sinksPendingSrcChng.reset(COCKPIT_SINK_ID);
               }
               else if (_sinksPendingSrcChng.test(CABIN_A_SINK_ID))
               {
                  srcData.setSrcId(_pendingActivationSourceID);
                  srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
                  srcData.setConnectionState(STATE_UNKNOWN);
                  srcData.setSinkId(CABIN_A_SINK_ID);
                  ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
                  _sinksPendingSrcChng.reset(CABIN_A_SINK_ID);
               }
               if (!_sinksPendingSrcChng.any())
               {
                  _activate = false;
                  _pendingActivationSourceID = SRC_INVALID;
               }
            }
            if (_regionSink[static_cast<Region>(activeRegion)] == sinkID)
            {
               if (state == connected
                     && (_prevSourceID != SRC_INVALID))
               {
                  if (true == _isSxmSelfContextRequired)
                  {
                     //Use case : On Mute/Unmute and toggle gadget Button, handling not to self context to sxm radio Main screen.
                     ApplicationSwitchClientHandler::instance()->requestSxmSelfContext();
                  }
                  _isSxmSelfContextRequired = false;
               }
               if (activeSource == SRC_TUNER_XM)
               {
                  if (COCKPIT_SINK_ID == sinkID)
                  {
                     _cockpitSxmConnectionState = state;
                  }
                  else if (CABIN_A_SINK_ID == sinkID)
                  {
                     _cabinSxmConnectionState = state;
                  }
               }
               else if (activeSource == SRC_SXM_NEWS)
               {
                  if (COCKPIT_SINK_ID == sinkID)
                  {
                     _cockpitSxmNewsConnectionState = state;
                  }
                  else if (CABIN_A_SINK_ID == sinkID)
                  {
                     _cabinSxmNewsConnectionState = state;
                  }
               }
            }
         }
         if (_regionSink[static_cast<Region>(activeRegion)] == sinkID && (state == connected))
         {
            _prevSourceID = _currSourceID;
            _currSourceID = activeSource;
         }
      }
   }
}


/**
 * Helper function to set timer when Source is SXM
 * @param [in] : SourceID
*/

void SXMSourceSwitch::setSXMSourceConnectionStatus(const int32& srcId)
{
   if ((srcId == SRC_TUNER_XM) && (_isSxmConnected == false))
   {
      _isSxmConnected = true;
      RunTimeSrcSXM();
   }
   else
   {
      if (_isSxmConnected == true)
      {
         _isSxmConnected = false;
         RunTimeSrcSXM();
      }
   }
}


/**
 * Helper function to calculate and send timer info when SXM is connected
 */
void SXMSourceSwitch::RunTimeSrcSXM()
{
   static uint32_t sxmStartTime = 0;
   static uint32_t sxmStopTime = 0;
   if (true == _isSxmConnected)
   {
      sxmStartTime = OSAL_ClockGetElapsedTime();
      ETG_TRACE_USR4(("SXMSourceSwitch::RunTimeSrcSXM, SXMStartTime start time : %d ms", sxmStartTime));
   }
   else
   {
      sxmStopTime = OSAL_ClockGetElapsedTime();
      ETG_TRACE_USR4(("SrcChangeHandler::RunTimeSrcFM(), fmStopTime stop time : %d ms", sxmStopTime));
      uint32_t duration = (sxmStopTime - sxmStartTime) / 1000;   // Convert milliseconds to seconds
      sxmStartTime = 0;
      sxmStopTime = 0;
      DiagnosticsClientInterface::updateSXMRuntimeInfo(duration);
   }
}


/**
 * Virtual function implemented to get update of ActiveSource Response
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'AudioSourceChangeProxy'
 * @param[in]      : response : The response message of 'ActiveSource'
 */
void SXMSourceSwitch::onActiveSourceListError(const ::boost::shared_ptr<AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr<ActiveSourceListError >& /*error*/)
{
   ETG_TRACE_ERR(("onActiveSourceListError received"));
}


/**
 * Virtual function implemented to get update of ActiveSource Response
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'AudioSourceChangeProxy'
 * @param[in]      : Update : The response message of 'ActiveSource'
 */
void SXMSourceSwitch::onActiveSourceListUpdate(const ::boost::shared_ptr<AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< ActiveSourceListUpdate >& update)
{
   ETG_TRACE_COMP(("SXMSourceSwitch::onActiveSourceListUpdate pendingsrcchng count %d, _deactivate=%d", _sinksPendingSrcChng.count(), _deactivate));
   if (update && update->hasActiveSourceList())
   {
      const ::std::vector<sourceData> sourceList = update->getActiveSourceList();
      handlePendingDeactivation(sourceList);
      updateConnectionState(sourceList);
      updateSxmSourceStatus(sourceList);
   }
}


/**
*  Function to check source change required in other sinks
*  @return : sinkID's value
*/
void SXMSourceSwitch::IsSrcChgReqdInOtherSinks(uint32 reqSourceID, bool activate)
{
   ETG_TRACE_USR1(("SXMSourceSwitch::IsSrcChgReqdInOtherSinks reqSourceID=%d, activate =%d", reqSourceID, activate));
   dp_tclAppHmi_SxmSXMAudioAlert dp_AudioAlert;
   HmiAppSxm::AlertSetting alertSettingsStatus = dp_AudioAlert.tGetData();
   //Based on Notification settings
   if (alertSettingsStatus.OnOrOff[Item_3])//Cockpit setting
   {
      _sinksPendingSrcChng.set(COCKPIT_SINK_ID);
   }
   if (SRC_TUNER_XM == reqSourceID && activate)
   {
      checkSinkConnectionState();
   }
   //SporstFlash alert, when setting is not enabled and SXM source is playing
   if (reqSourceID == SRC_SXM_NEWS)
   {
      if (activate)
      {
         if (_sinkSrcStatus[COCKPIT_SINK_ID].first == SRC_TUNER_XM)
         {
            _sinksPendingSrcChng.set(COCKPIT_SINK_ID);
         }
         if (_sinkSrcStatus[CABIN_A_SINK_ID].first == SRC_TUNER_XM)
         {
            _sinksPendingSrcChng.set(CABIN_A_SINK_ID);
         }
      }
      else//deactivate
      {
         if (_sinkSrcStatus[COCKPIT_SINK_ID].first == SRC_SXM_NEWS)
         {
            _sinksPendingSrcChng.set(COCKPIT_SINK_ID);
         }
         if (_sinkSrcStatus[CABIN_A_SINK_ID].first == SRC_SXM_NEWS)
         {
            _sinksPendingSrcChng.set(CABIN_A_SINK_ID);
         }
      }
   }
   ETG_TRACE_USR1(("SXMSourceSwitch::IsSrcChgReqdInOtherSinks count: %d", _sinksPendingSrcChng.count()));
}


/**
 * Helper function to check sxm source status.
 * @param[in]  : sourceList: list of source in stack.
 */
void SXMSourceSwitch::updateSxmSourceStatus(const ::std::vector<sourceData>& sourceList)
{
   ETG_TRACE_USR1(("SXMSourceSwitch::updateSxmSourceStatus _isScanActive=%d", _isScanActive));
//Cockpit
   std::vector<sourceData>::const_iterator iter = sourceList.begin(); // stack top is list bottom
   bool found = false;
   for (; iter != sourceList.end(); ++iter)
   {
      //if sxm_news is there
      uint16 sinkId = iter->getSinkId();
      if ((COCKPIT_SINK_ID == sinkId))
      {
         if (SRC_SXM_NEWS == iter->getSrcId())
         {
            found = true;
            if (_prevSinkSrcStatus[sinkId].second != iter->getConnectionState())
            {
               //post msg (prev, itr->connectionstate);
               ETG_TRACE_USR1(("SXMSourceSwitch::updateSxmSourceStatus previousConnectionstatus=%d, connection status=%d", _prevSinkSrcStatus[sinkId].second, iter->getConnectionState()));
               POST_MSG((COURIER_MESSAGE_NEW(SXMSourceStatusUpdMsg)(_prevSinkSrcStatus[sinkId].second, iter->getConnectionState())));
               _prevSinkSrcStatus[sinkId].second = iter->getConnectionState();
            }
            break;
         }
      }
   }
   if (!found)
   {
      if (_prevSinkSrcStatus[COCKPIT_SINK_ID].second != disconnected)
      {
         //post msg(prev, disconnected)
         ETG_TRACE_USR1(("SXMSourceSwitch::updateSxmSourceStatus previousConnectionstatus=%d", _prevSinkSrcStatus[COCKPIT_SINK_ID].second));
         POST_MSG((COURIER_MESSAGE_NEW(SXMSourceStatusUpdMsg)(_prevSinkSrcStatus[COCKPIT_SINK_ID].second, disconnected)));
         _prevSinkSrcStatus[COCKPIT_SINK_ID].second = disconnected;
      }
   }
   found = false;
   std::vector<sourceData>::const_iterator itr = sourceList.begin(); // stack top is list bottom
//Cabin
   for (; itr != sourceList.end(); ++itr)
   {
      //if sxm_news is there
      uint16 sinkId = itr->getSinkId();
      if (CABIN_A_SINK_ID == sinkId)
      {
         if (SRC_SXM_NEWS == itr->getSrcId())
         {
            found = true;
            if (_prevSinkSrcStatus[sinkId].second != itr->getConnectionState())
            {
               //post msg (prev, itr->connectionstate);
               ETG_TRACE_USR1(("SXMSourceSwitch::updateSxmSourceStatus previousConnectionstatus=%d, connection status=%d", _prevSinkSrcStatus[sinkId].second, itr->getConnectionState()));
               POST_MSG((COURIER_MESSAGE_NEW(SXMSourceStatusUpdMsg)(_prevSinkSrcStatus[sinkId].second, itr->getConnectionState())));
               _prevSinkSrcStatus[sinkId].second = itr->getConnectionState();
            }
            break;
         }
      }
   }
   if (!found)
   {
      if (_prevSinkSrcStatus[CABIN_A_SINK_ID].second != disconnected)
      {
         //post msg(prev, disconnected)
         POST_MSG((COURIER_MESSAGE_NEW(SXMSourceStatusUpdMsg)(_prevSinkSrcStatus[CABIN_A_SINK_ID].second, disconnected)));
         _prevSinkSrcStatus[CABIN_A_SINK_ID].second = disconnected;
      }
   }
   std::vector<sourceData>::const_iterator constIter = sourceList.begin(); // stack top is list bottom
   bool isSXMInCockPit = false;
   bool isSXMinCabin = false;
   for (; constIter != sourceList.end(); ++constIter)
   {
      Region reg = static_cast<Region>(ApplicationSwitchClientHandler::instance()->getActiveRegion());
      uint16 activeSinkId = _regionSink[reg];
      if (activeSinkId == constIter->getSinkId())
      {
         if (_isScanActive && (SRC_TUNER_XM == constIter->getSrcId() && constIter->getConnectionState() != connected))
         {
            POST_MSG((COURIER_MESSAGE_NEW(SXMAudioScanOpertationReqMsg)(SCAN_ABORT)));  //Abort scan if scan active
         }
      }
      if (constIter->getSinkId() == COCKPIT_SINK_ID && SRC_TUNER_XM == constIter->getSrcId())
      {
         isSXMInCockPit = true;
      }
      else if (constIter->getSinkId() == CABIN_A_SINK_ID && SRC_TUNER_XM == constIter->getSrcId())
      {
         isSXMinCabin = true;
      }
   }
   _alertSettingSxmSrcStatus[IS_SXM_IN_COCKPIT] = isSXMInCockPit;
   _alertSettingSxmSrcStatus[IS_SXM_IN_CABINA] = isSXMinCabin;
   std::string alertStr = _alertSettingSxmSrcStatus.to_string();
   ETG_TRACE_USR1(("SXMSourceSwitch::updateSxmSourceStatus: alertSettingSxmSrcStatus:%s", alertStr.c_str()));
   ETG_TRACE_USR1(("SXMSourceSwitch::updateSxmSourceStatus: isSXMInCockPit:%d, isSXMinCabin=%d", isSXMInCockPit, isSXMinCabin));
}


/**
 * Function to set the source status
 * @param[in]      : srcId: active sourceID
 * @param[in]      : sinkId: active sinkId
 * @param[in]      : connStatus: Connection Status.
 */
void SXMSourceSwitch::setSourceStatus(int32 srcId, uint16 sinkId, uint32 connStatus)
{
   if (connStatus == connected)
   {
      _sinkSrcStatus[sinkId] = std::make_pair(srcId, connStatus);
      if (_sinkSrcStatus[sinkId].first == SRC_TUNER_XM)
      {
         POST_MSG((COURIER_MESSAGE_NEW(SXMAudioReplayControlReqMsg)(SXM_AUDIO_SOURCECHANGE_PLAY)));
      }
   }
}


/**
 *  Function to get the sink source status
 * @param[in]      : sinkId: active sinkId
 * @param[in]      : srcId: active sourceID
 * @param[in]      : connStatus: Connection Status.
 */

void SXMSourceSwitch::getSinkSourceStatus(uint16 sinkId, int32& srcId, uint32& connStatus)
{
   if ((sinkId == COCKPIT_SINK_ID)  || (sinkId == CABIN_A_SINK_ID))
   {
      srcId = _sinkSrcStatus[sinkId].first;
      connStatus = _sinkSrcStatus[sinkId].second;
   }
}


/**
*  Function to handle pending source deativation
* @param[in]  : sourceList: list of source in stack.
*/
void SXMSourceSwitch::handlePendingDeactivation(const ::std::vector<sourceData>& sourceList)
{
   ETG_TRACE_USR1(("SXMSourceSwitch::handlePendingDeactivation: _sinksPendingSrcChng count=%d", _sinksPendingSrcChng.count()));
   std::vector<sourceData>::const_iterator iter = sourceList.begin(); // stack top is list bottom
   int32 srcId;
   for (; iter != sourceList.end(); ++iter)
   {
      srcId = iter->getSrcId();
      if ((srcId == SRC_SXM_NEWS))
      {
         if (_sinksPendingSrcChng.any() && _deactivate)
         {
            sourceData srcData;
            srcData.setSrcId(srcId);
            srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
            srcData.setConnectionState(STATE_UNKNOWN);
            if (_sinksPendingSrcChng.test(COCKPIT_SINK_ID))
            {
               srcData.setSinkId(COCKPIT_SINK_ID);
               ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendDeactivateSourceRequest(*this, srcData)));
               _sinksPendingSrcChng.reset(COCKPIT_SINK_ID);
            }
            else if (_sinksPendingSrcChng.test(CABIN_A_SINK_ID))
            {
               srcData.setSinkId(CABIN_A_SINK_ID);
               ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendDeactivateSourceRequest(*this, srcData)));
               _sinksPendingSrcChng.reset(CABIN_A_SINK_ID);
            }
            if (!_sinksPendingSrcChng.any())
            {
               _deactivate = false;
            }
         }
      }
   }
}


/**
*  Function to update sxm source connection states in both Cockpit and cabin region
* @param[in]  : sourceList: list of source in stack.
*/
void SXMSourceSwitch::updateConnectionState(const ::std::vector< sourceData >& sourceList)
{
   ETG_TRACE_USR1(("SXMSourceSwitch::updateConnectionState: "));
   std::vector<sourceData>::const_iterator iter = sourceList.begin(); // stack top is list bottom
   _cockpitSxmConnectionState = disconnected;
   _cabinSxmConnectionState = disconnected;
   _cockpitSxmNewsConnectionState =  disconnected;
   _cabinSxmNewsConnectionState = disconnected;
   for (; iter != sourceList.end(); ++iter)
   {
      int32 srcId = iter->getSrcId();
      Region reg = static_cast<Region>(ApplicationSwitchClientHandler::instance()->getActiveRegion());
      uint16 activeSinkId = _regionSink[reg];
      if (srcId == SRC_TUNER_XM)
      {
         if (COCKPIT_SINK_ID == iter->getSinkId())
         {
            _cockpitSxmConnectionState = static_cast<enConnectionState>(iter->getConnectionState());
         }
         else if (CABIN_A_SINK_ID == iter->getSinkId())
         {
            _cabinSxmConnectionState = static_cast<enConnectionState>(iter->getConnectionState());
         }
      }
      else if (srcId == SRC_SXM_NEWS)
      {
         if (COCKPIT_SINK_ID == iter->getSinkId())
         {
            _cockpitSxmNewsConnectionState = static_cast<enConnectionState>(iter->getConnectionState());
         }
         else if (CABIN_A_SINK_ID == iter->getSinkId())
         {
            _cabinSxmNewsConnectionState = static_cast<enConnectionState>(iter->getConnectionState());
         }
      }
   }
   ETG_TRACE_USR1(("SXMSourceSwitch::updateConnectionState: _cockpitSxmConnectionState=%d, _cockpitSxmNewsConnectionState=%d", _cockpitSxmConnectionState, _cockpitSxmNewsConnectionState));
   ETG_TRACE_USR1(("SXMSourceSwitch::updateConnectionState: _ cabinSxmConnectionState =%d, _ cabinSxmNewsConnectionState =%d", _cabinSxmConnectionState, _cabinSxmNewsConnectionState));
}


/**
 * Helper function to handle Sxm source activation or deactivation.
 * @param[in]      : srcType:  source type.
 * @param[in]      : activate : status to connect or disconnect source.
 */
bool SXMSourceSwitch::handleSourceChangeRequest(uint32 sourceID, bool activate, bool forceactive, bool isAlert)
{
   bool isReqSent = false;
   uint8 activeRegion = ApplicationSwitchClientHandler::instance()->getActiveRegion();
   ETG_TRACE_USR1(("SXMSourceSwitch::handleSourceChangeRequest SourceID:%d CurrentSource:%d ActiveRegion:%d isAlert:%d", sourceID, _currSourceID, activeRegion, isAlert));
   uint sinkId = _regionSink[static_cast<Region>(activeRegion)];
   handleSxmNewsSourceDeactivation(sourceID, activate, sinkId);
   if (isAlert)
   {
      IsSrcChgReqdInOtherSinks(sourceID, activate);
      if (activate && ((_currSourceID != SRC_TUNER_XM) || ((_currSourceID == SRC_TUNER_XM) &&
                       (!(SXMCommonInterface::isViewVisible(Sxm_Scenes_SXM_RADIO__MAIN) && SXMCommonInterface::isSxmInForeground())))))
      {
         if (_sinksPendingSrcChng.test(sinkId))
         {
            ETG_TRACE_USR2(("SXMSourceSwitch::handleSourceChangeRequest - _isSxmSelfContextRequired"));
            _isSxmSelfContextRequired = true;
         }
      }
   }
   else
   {
      _sinksPendingSrcChng.reset();
      _sinksPendingSrcChng.set(sinkId);
   }

   sourceData srcData;
   srcData.setSrcId(sourceID);
   srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
   srcData.setConnectionState(STATE_UNKNOWN);

   if (activate)
   {
      if (_sinksPendingSrcChng.any())
      {
         _activate = true;
         _pendingActivationSourceID = sourceID;
         if (_sinksPendingSrcChng.test(COCKPIT_SINK_ID))
         {
            srcData.setSinkId(COCKPIT_SINK_ID);
            ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
            _sinksPendingSrcChng.reset(COCKPIT_SINK_ID);
            isReqSent = true;
         }
         else if (_sinksPendingSrcChng.test(CABIN_A_SINK_ID))
         {
            srcData.setSinkId(CABIN_A_SINK_ID);
            ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true)));
            _sinksPendingSrcChng.reset(CABIN_A_SINK_ID);
            isReqSent = true;
         }
         if (!_sinksPendingSrcChng.any())
         {
            _activate = false;
            _pendingActivationSourceID = SRC_INVALID;
         }
      }
      else
      {
         ETG_TRACE_USR2(("SXMSourceSwitch::handleSourceChangeRequest - sink not required"));
      }
   }
   else//Deactivate
   {
      if (_sinksPendingSrcChng.any() && sourceID == SRC_SXM_NEWS)
      {
         _deactivate = true;
         if (_sinksPendingSrcChng.test(COCKPIT_SINK_ID))
         {
            srcData.setSinkId(COCKPIT_SINK_ID);
            ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendDeactivateSourceRequest(*this, srcData)));
            _sinksPendingSrcChng.reset(COCKPIT_SINK_ID);
            isReqSent = true;
         }
         else if (_sinksPendingSrcChng.test(CABIN_A_SINK_ID))
         {
            srcData.setSinkId(CABIN_A_SINK_ID);
            ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendDeactivateSourceRequest(*this, srcData)));
            _sinksPendingSrcChng.reset(CABIN_A_SINK_ID);
            isReqSent = true;
         }
         if (!_sinksPendingSrcChng.any())
         {
            _deactivate = false;
         }
      }
      else if (sourceID == SRC_TUNER_XM && _sinksPendingSrcChng.any())
      {
         _sinksPendingSrcChng.reset();
         ETG_TRACE_USR2(("SXMSourceSwitch::handleSourceChangeRequest -SRC id=%d", SRC_TUNER_XM));
      }
      else
      {
         ETG_TRACE_USR2(("SXMSourceSwitch::handleSourceChangeRequest - sink not required"));
      }
   }
   ETG_TRACE_USR3(("SXMSourceSwitch::handleSourceChangeRequest Sinks pendingSrcChg: %d", _sinksPendingSrcChng.count()));
   return isReqSent;
}


/**
 * Virtual function implemented to get update of MuteState Error
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'SoundPropertiesProxy'
 * @param[in]      : error : The error message
 */
void SXMSourceSwitch::onMuteStateMapError(const ::boost::shared_ptr< SoundPropertiesProxy >& /*proxy*/,
      const ::boost::shared_ptr< MuteStateMapError >& /*error*/)
{
   ETG_TRACE_USR4(("SXMSourceSwitch:: onMuteStateMapError()"));
}


/**
 * Virtual function implemented to get update of MuteState Update
 * @param[in]      : proxy: the client side representation of the CCA Functional Interface 'SoundPropertiesProxy'
 * @param[in]      : update : The update message of 'MuteState'
 */
void SXMSourceSwitch::onMuteStateMapUpdate(const ::boost::shared_ptr< SoundPropertiesProxy >& /*proxy*/,
      const ::boost::shared_ptr< MuteStateMapUpdate >& update)
{
   if (update && update->hasMuteStateMap())
   {
      _muteStatusForRegion = update->getMuteStateMap();
      setMuteStatus();
   }
}


/**
 * Helper function to get Sxm Source Status.
 */
uint8 SXMSourceSwitch::getSxmSourceStatus(void)
{
   Region reg = static_cast<Region>(ApplicationSwitchClientHandler::instance()->getActiveRegion());
   if (_regionSink[reg] == COCKPIT_SINK_ID)
   {
      return _cockpitSxmConnectionState;
   }
   else
   {
      return _cabinSxmConnectionState;
   }
}


/**
 * Helper function to get Sxm News Source Status.
 */
uint8 SXMSourceSwitch::getSxmNewsSourceStatus()
{
   Region reg = static_cast<Region>(ApplicationSwitchClientHandler::instance()->getActiveRegion());
   if (_regionSink[reg] == COCKPIT_SINK_ID)
   {
      return _cockpitSxmNewsConnectionState;
   }
   else
   {
      return _cabinSxmNewsConnectionState;
   }
}


/**
 * Helper function to get Sxm source active status
 */
bool SXMSourceSwitch::isSxmSourceActive(void) const
{
   return (connected == _cockpitSxmConnectionState || connected == _cabinSxmConnectionState);
}


/**
 * Helper function to get Sxm News source active status
 */
bool SXMSourceSwitch::isSxmNewsSourceActive(void) const
{
   return (connected == _cockpitSxmNewsConnectionState || connected == _cabinSxmNewsConnectionState);
}


/**
 * Callback to handle Courier Message SXMSourceChangeReqMsg Request
 * @param [in] : Reference of Courier::SXMSourceChangeReqMsg
 * @return     : True-When Message is processed.
 */
bool SXMSourceSwitch::onCourierMessage(const SXMSourceChangeReqMsg& /*oMsg*/)
{
   return true;
}


/**
* Helper function to set Sxm tune scan status.
* @param [in] : tune scan active status
*/
void SXMSourceSwitch::setSxmAudioScanStatus(bool isActive)
{
   _isScanActive = isActive;
}


/**
* Helper function to get Sxm tune scan status.
*/
bool SXMSourceSwitch::getSxmAudioScanStatus()
{
   return _isScanActive;
}


/**
* Helper function to get mute status of Current Region.
*/
bool SXMSourceSwitch::getCurrentRegionMuteStatus()
{
   ETG_TRACE_USR4(("SXMSourceSwitch::getCurrentRegionMuteStatus entered "));
   bool bMuteStatus = false;
   uint8 u8activeRegion = ApplicationSwitchClientHandler::instance()->getActiveRegion();
   Region reg = static_cast<Region>(u8activeRegion);
   uint16 RegsinkId = _regionSink[reg];
   ETG_TRACE_USR4(("SXMSourceSwitch::getCurrentRegionMuteStatus u8activeRegion %d ", u8activeRegion));
   ETG_TRACE_USR4(("SXMSourceSwitch::getCurrentRegionMuteStatus RegsinkId %d ", RegsinkId));
   std::map<uint16, bool>::iterator itr;
   for (itr = _muteStatusForRegion.begin(); itr != _muteStatusForRegion.end(); ++itr)
   {
      uint16 iSinkId = itr->first;
      if (RegsinkId == iSinkId)
      {
         bMuteStatus = itr->second;
         break;
      }
   }
   ETG_TRACE_USR4(("SXMSourceSwitch::getCurrentRegionMuteStatus bMuteStatus %d", bMuteStatus));
   return bMuteStatus;
}


/**
*  Call Back function for service status.
*  @param[in] : Entry Exit status
*  @param[in] : service type
*/
void SXMSourceSwitch::vNotifyServiceStatus(unsigned short int lEntryExitStatus, unsigned int serviceType)
{
   ETG_TRACE_USR2(("SXMSourceSwitch::vNotifyServiceStatus %d", lEntryExitStatus));
   if (SXMServiceStateEntry == lEntryExitStatus && SXMAudio == serviceType)
   {
      ApplicationSwitchClientHandler::instance()->requestSendContext(APPID_APPHMI_SXM, enActivityIDs__eActivityID_SXM_MAIN);
   }
   else if (SXMServiceStateEntry == lEntryExitStatus && SXMAudioUtility == serviceType)
   {
      ApplicationSwitchClientHandler::instance()->requestSendContext(APPID_APPHMI_SXM, enActivityIDs__eActivityID_SXM_AUDIO_UTILITY);
   }
}


/**
*  Call Back function for system appmode status.
*  @param[in] : App state
*/
void SXMSourceSwitch::vNotifyAppState(unsigned short int appState)
{
   ETG_TRACE_USR2(("SXMSourceSwitch::vNotifyAppState %d", appState));
   if (appState == hmibase::IN_BACKGROUND && _isScanActive)
   {
      POST_MSG((COURIER_MESSAGE_NEW(SXMAudioScanOpertationReqMsg)(SCAN_ABORT)));
   }
}


/**
*  Call Back function for system appmode status.
*  @param[in] : lactiveAppMode - App mode
*/
void SXMSourceSwitch::vNotifyAppModeChange(unsigned short int lactiveAppMode)
{
   ETG_TRACE_USR4(("SXMSourceSwitch, ActiveAppMode %d", lactiveAppMode));
}


/**
* Function to intialize the popup text based on alert setting and source status
*/
void SXMSourceSwitch::initAlertSourceText()
{
   /*
   Alert setting  Src(SXM/No) Num  Text
   bit-5	4	3	2	1	0
   Coc	CaA	CaB	Coc	CaA	CaB
   0	0	0	0	0	0	0	""
   0	0	0	0	0	1	1	for CabinB
   0	0	0	0	1	0	2	for CabinA
   0	0	0	0	1	1	3	for CabinA & CabinB
   0	0	0	1	0	0	4	for Cockpit
   0	0	0	1	0	1	5	for Cockpit & CabinB
   0	0	0	1	1	0	6	for Cockpit & CabinA
   0	0	0	1	1	1	7	""
   0	0	1	0	0	0	8	for CabinB
   0	0	1	0	0	1	9	for CabinB
   0	0	1	0	1	0	10	for CabinA & CabinB
   0	0	1	0	1	1	11	for CabinA & CabinB
   0	0	1	1	0	0	12	for Cockpit & CabinB
   0	0	1	1	0	1	13	for Cockpit & CabinB
   0	0	1	1	1	0	14	for Cockpit, CabinA & CabinB
   0	0	1	1	1	1	15	""
   0	1	0	0	0	0	16	for CabinA
   0	1	0	0	0	1	17	for CabinA & CabinB
   0	1	0	0	1	0	18	for CabinA
   0	1	0	0	1	1	19	for CabinA & CabinB
   0	1	0	1	0	0	20	for Cockpit & CabinA
   0	1	0	1	0	1	21	for Cockpit, CabinA & CabinB
   0	1	0	1	1	0	22	for Cockpit & CabinA
   0	1	0	1	1	1	23	""
   0	1	1	0	0	0	24	for CabinA & CabinB
   0	1	1	0	0	1	25	for CabinA & CabinB
   0	1	1	0	1	0	26	for CabinA & CabinB
   0	1	1	0	1	1	27	for CabinA & CabinB
   0	1	1	1	0	0	28	for Cockpit, CabinA & CabinB
   0	1	1	1	0	1	29	for Cockpit, CabinA & CabinB
   0	1	1	1	1	0	30	for Cockpit, CabinA & CabinB
   0	1	1	1	1	1	31	""
   1	0	0	0	0	0	32	for Cockpit
   1	0	0	0	0	1	33	for Cockpit & CabinB
   1	0	0	0	1	0	34	for Cockpit & CabinA
   1	0	0	0	1	1	35	for Cockpit, CabinA & CabinB
   1	0	0	1	0	0	36	for Cockpit
   1	0	0	1	0	1	37	for Cockpit & CabinB
   1	0	0	1	1	0	38	for Cockpit & CabinA
   1	0	0	1	1	1	39	""
   1	0	1	0	0	0	40	for Cockpit & CabinB
   1	0	1	0	0	1	41	for Cockpit & CabinB
   1	0	1	0	1	0	42	for Cockpit, CabinA & CabinB
   1	0	1	0	1	1	43	for Cockpit, CabinA & CabinB
   1	0	1	1	0	0	44	for Cockpit & CabinB
   1	0	1	1	0	1	45	for Cockpit & CabinB
   1	0	1	1	1	0	46	for Cockpit, CabinA & CabinB
   1	0	1	1	1	1	47	""
   1	1	0	0	0	0	48	for Cockpit & CabinA
   1	1	0	0	0	1	49	for Cockpit, CabinA & CabinB
   1	1	0	0	1	0	50	for Cockpit & CabinA
   1	1	0	0	1	1	51	for Cockpit, CabinA & CabinB
   1	1	0	1	0	0	52	for Cockpit & CabinA
   1	1	0	1	0	1	53	for Cockpit, CabinA & CabinB
   1	1	0	1	1	0	54	for Cockpit & CabinA
   1	1	0	1	1	1	55	""
   1	1	1	0	0	0	56	for Cockpit, CabinA & CabinB
   1	1	1	0	0	1	57	for Cockpit, CabinA & CabinB
   1	1	1	0	1	0	58	for Cockpit, CabinA & CabinB
   1	1	1	0	1	1	59	for Cockpit, CabinA & CabinB
   1	1	1	1	0	0	60	for Cockpit, CabinA & CabinB
   1	1	1	1	0	1	61	for Cockpit, CabinA & CabinB
   1	1	1	1	1	0	62	for Cockpit, CabinA & CabinB
   1	1	1	1	1	1	63	""
    */
   _popupText.clear();
   _popupText[0] = regionText("", "");
   _popupText[1] = regionText("CabinB", "");
   _popupText[2] = regionText("CabinA", "");
   _popupText[3] = regionText("CabinA", "CabinB");
   _popupText[4] = regionText("Cockpit", "");
   _popupText[5] = regionText("Cockpit", "CabinB");
   _popupText[6] = regionText("Cockpit", "CabinA");
   _popupText[7] = regionText("", "");
   _popupText[8] = regionText("CabinB", "");
   _popupText[9] = regionText("CabinB", "");
   _popupText[10] = regionText("CabinA", "CabinB");
   _popupText[11] = regionText("CabinA", "CabinB");
   _popupText[12] = regionText("Cockpit", "CabinB");
   _popupText[13] = regionText("Cockpit", "CabinB");
   _popupText[14] = regionText("", "");
   _popupText[15] = regionText("", "");
   _popupText[16] = regionText("CabinA", "");
   _popupText[17] = regionText("CabinA", "CabinB");
   _popupText[18] = regionText("CabinA", "");
   _popupText[19] = regionText("CabinA", "CabinB");
   _popupText[20] = regionText("Cockpit", "CabinA");
   _popupText[21] = regionText("", "");
   _popupText[22] = regionText("Cockpit", "CabinA");
   _popupText[23] = regionText("", "");
   _popupText[24] = regionText("CabinA", "CabinB");
   _popupText[25] = regionText("CabinA", "CabinB");
   _popupText[26] = regionText("CabinA", "CabinB");
   _popupText[27] = regionText("CabinA", "CabinB");
   _popupText[28] = regionText("", "");
   _popupText[29] = regionText("", "");
   _popupText[30] = regionText("", "");
   _popupText[31] = regionText("", "");
   _popupText[32] = regionText("Cockpit", "");
   _popupText[33] = regionText("Cockpit", "CabinB");
   _popupText[34] = regionText("Cockpit", "CabinA");
   _popupText[35] = regionText("", "");
   _popupText[36] = regionText("Cockpit", "");
   _popupText[37] = regionText("Cockpit", "CabinB");
   _popupText[38] = regionText("Cockpit", "CabinA");
   _popupText[39] = regionText("", "");
   _popupText[40] = regionText("Cockpit", "CabinB");
   _popupText[41] = regionText("Cockpit", "CabinB");
   _popupText[42] = regionText("", "");
   _popupText[43] = regionText("", "");
   _popupText[44] = regionText("Cockpit", "CabinB");
   _popupText[45] = regionText("Cockpit", "CabinB");
   _popupText[46] = regionText("", "");
   _popupText[47] = regionText("", "");
   _popupText[48] = regionText("Cockpit", "CabinA");
   _popupText[49] = regionText("", "");
   _popupText[50] = regionText("Cockpit", "CabinA");
   _popupText[51] = regionText("", "");
   _popupText[52] = regionText("Cockpit", "CabinA");
   _popupText[53] = regionText("", "");
   _popupText[54] = regionText("Cockpit", "CabinA");
   _popupText[55] = regionText("", "");
   _popupText[56] = regionText("", "");
   _popupText[57] = regionText("", "");
   _popupText[58] = regionText("", "");
   _popupText[59] = regionText("", "");
   _popupText[60] = regionText("", "");
   _popupText[61] = regionText("", "");
   _popupText[62] = regionText("", "");
   _popupText[63] = regionText("", "");
}


/**
*  Function to get Popup text based on settings and source status
*  @return : Popup text to be appended
*/
regionText SXMSourceSwitch::getPopupText()
{
   unsigned int value = _alertSettingSxmSrcStatus.to_ulong();
   ETG_TRACE_USR1(("SXMSourceSwitch::getPopupText: index =%d", value));
   return _popupText[value];
}


/**
*  Function to set the Alert setting
*/
void SXMSourceSwitch::setAlertSourceStatus()
{
   dp_tclAppHmi_SxmSXMAudioAlert dp_AudioAlert;
   HmiAppSxm::AlertSetting alertSettingsStatus = dp_AudioAlert.tGetData();
   _alertSettingSxmSrcStatus[IS_ALERT_ALLOWED_COCKPIT] = alertSettingsStatus.OnOrOff[3];
   _alertSettingSxmSrcStatus[IS_ALERT_ALLOWED_CABINA] = 0;
   _alertSettingSxmSrcStatus[IS_ALERT_ALLOWED_CABINB] = 0;

   std::string alertStr = _alertSettingSxmSrcStatus.to_string();
   ETG_TRACE_USR1(("SXMSourceSwitch::setAlertSourceStatus: alertSettingSxmSrcStatus:%s", alertStr.c_str()));
}


/**
*  Function to get Alert settings and source status
*  @return : Alert and Source status
*/
uint8 SXMSourceSwitch::getAlertSourceStatus()
{
   return _alertSettingSxmSrcStatus.to_ulong();
}


/**
*  Function to get Pause allowed condition
*  @return : Pause allowed or not
*/
bool SXMSourceSwitch::isPauseAllowed()
{
   /*(SXM/Not)Coc CabA CabB Value  Action
    (state)
   Suspended	0	0	0		0	No pause
   Suspended	0	0	1		1	Pause
   Suspended	0	1	0		2	Pause
   Suspended	0	1	1		3	Pause
   Suspended	1	0	0		4	Pause
   Suspended	1	0	1		5	No Pause
   Suspended	1	1	0		6	No Pause
   Suspended	1	1	1		7	No Pause
   */
   bool status = false;
   std::bitset<3> srcStatus;
   srcStatus[IS_SXM_IN_COCKPIT] = _alertSettingSxmSrcStatus[IS_SXM_IN_COCKPIT];
   srcStatus[IS_SXM_IN_CABINA] = _alertSettingSxmSrcStatus[IS_SXM_IN_CABINA];
   srcStatus[IS_SXM_IN_CABINB] = 0;
   unsigned int value = srcStatus.to_ulong();
   if (value == 1 || value == 2 || value == 4)//refer above comments
   {
      status = true;
   }
   return status;
}


/**
*  Function to get Previous source ID
*  @return : previous Source
*/
int32 SXMSourceSwitch::getPrevSource()
{
   return _prevSourceID;
}


/**
*  Function to update connection state when active region updated
*  @param[in] : Active region
*/
void SXMSourceSwitch::setConnectionState(uint8 activeRegion)
{
   ETG_TRACE_USR4(("SXMSourceSwitch::setConnectionState entered"));
   Region reg = static_cast<Region>(activeRegion);
   uint16 sinkId = _regionSink[reg];
   _currSourceID = _sinkSrcStatus[sinkId].first;

   setMuteStatus();
}


/**
*  Function to update Mute status
*/
void SXMSourceSwitch::setMuteStatus()
{
   ETG_TRACE_USR4(("SXMSourceSwitch::setMuteStatus entered"));
   bool bMuteStatus = getCurrentRegionMuteStatus();
   SxmAudioDataBindingUtils::instance()->setMuteStatus(bMuteStatus);
}


/**
 * Helper function to check if sink is already connected, if yes then don't send Source change req
 */
void SXMSourceSwitch::checkSinkConnectionState()
{
   ETG_TRACE_USR4(("SXMSourceSwitch::checkSinkConnectionState"));
   if (_sinkSrcStatus[COCKPIT_SINK_ID].first == SRC_TUNER_XM && _sinkSrcStatus[COCKPIT_SINK_ID].second == connected &&
         _sinksPendingSrcChng.test(COCKPIT_SINK_ID))
   {
      _sinksPendingSrcChng.reset(COCKPIT_SINK_ID);
   }
   if ((_sinkSrcStatus[CABIN_A_SINK_ID].first == SRC_TUNER_XM && _sinkSrcStatus[CABIN_A_SINK_ID].second == connected) &&
         (_sinksPendingSrcChng.test(CABIN_A_SINK_ID)))
   {
      _sinksPendingSrcChng.reset(CABIN_A_SINK_ID);
   }
}


/**
 * Send Source Deactivation request to Audio
 * @param [in] : SourceID, Connection State
*/
void SXMSourceSwitch::requestSxmCabinSourceDeactivation(int srcid, int connectionState)
{
   ETG_TRACE_USR4(("SXMSourceSwitch::requestSxmCabinSourceDeactivation"));
   if ((srcid == SRC_TUNER_XM) && ((connectionState == connected) || (connectionState == suspended)))
   {
      sourceData srcData;
      srcData.setSrcId(srcid);
      srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
      srcData.setConnectionState(STATE_UNKNOWN);
      srcData.setSinkId(CABIN_A_SINK_ID);
      if (NULL != _audioSrcChgProxy)
      {
         _audioSrcChgProxy->sendDeactivateSourceRequest(*this, srcData);
      }
   }
   else if (srcid == SRC_SXM_NEWS)
   {
      handleSourceChangeRequest(SRC_SXM_NEWS, false, false, true);
   }
}


/**
 * Send Source Activation request to Audio
 * @param [in] : SourceID
*/
void SXMSourceSwitch::requestSxmCabinSourceActivation(int srcid)
{
   ETG_TRACE_USR4(("SXMSourceSwitch::requestSxmCabinSourceActivation"));
   if (srcid == SRC_TUNER_XM)
   {
      sourceData srcData;
      srcData.setSrcId(srcid);
      srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
      srcData.setConnectionState(STATE_UNKNOWN);
      srcData.setSinkId(CABIN_A_SINK_ID);
      if (NULL != _audioSrcChgProxy)
      {
         _audioSrcChgProxy->sendActivateSourceRequest(*this, srcData, true);
      }
   }
   else if (srcid == SRC_SXM_NEWS)
   {
      handleSourceChangeRequest(SRC_SXM_NEWS, true, false, true);
   }
}


/**
 * Send Source Deactivation request to Audio
 * @param [in] : SourceID, activate, sinkId
*/
void SXMSourceSwitch::handleSxmNewsSourceDeactivation(uint32 srcid, bool activate, uint sinkId)
{
   ETG_TRACE_USR4(("SXMSourceSwitch::handleSxmNewsSourceDeactivation"));
   if ((srcid == SRC_TUNER_XM) && (activate == true))
   {
      sourceData srcData;
      srcData.setSrcId(SRC_SXM_NEWS);
      srcData.setSubSrcId(INITIALISE_SUBSRC_ID);
      srcData.setConnectionState(STATE_UNKNOWN);
      if ((_sinkSrcStatus[COCKPIT_SINK_ID].first == SRC_SXM_NEWS && _sinkSrcStatus[COCKPIT_SINK_ID].second == connected))
      {
         srcData.setSinkId(COCKPIT_SINK_ID);
         ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendDeactivateSourceRequest(*this, srcData)));
      }
      if ((_sinkSrcStatus[CABIN_A_SINK_ID].first == SRC_SXM_NEWS && _sinkSrcStatus[CABIN_A_SINK_ID].second == connected))
      {
         srcData.setSinkId(CABIN_A_SINK_ID);
         ((_audioSrcChgProxy.get()) && (_audioSrcChgProxy->sendDeactivateSourceRequest(*this, srcData)));
      }
   }
}


}  // namespace Core
}  // namespace App
