/**************************************************************************************
 * @file        : SxmHmiInfoService.cpp
 * @addtogroup  : AppHmi_Sxm
 * @brief       : Client for HMI infor service
 * @copyright   : (c) 2019-2020 Robert Bosch Car Multimedia GmbH
 *                The reproduction, distribution and utilization of this file as
 *                well as the communication of its contents to others without express
 *                authorization is prohibited. Offenders will be held liable for the
 *                payment of damages. All rights reserved in the event of the grant
 *                of a patent, utility model or design.
 **************************************************************************************/
#include "hmi_trace_if.h"
#include "App/Core/SxmUtils/SxmUtils.h"
#include "SxmHmiInfoService.h"
#include "App/Core/SxmAudio/SxmAudioDataBindingUtils.h"
#include "spm_core_fi_typesConst.h"
#include "App/Core/ClientHandler/SourceSwitch/SXMSourceSwitchInterface.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS       TR_CLASS_APPHMI_SXMCOMMON
#define ETG_I_TRACE_CHANNEL           TR_TTFIS_APPHMI_SXM
#define ETG_I_TTFIS_CMD_PREFIX        "APPHMI_SXM_"
#define ETG_I_FILE_PREFIX             App::Core::SxmHmiInfoService::
#include "trcGenProj/Header/SxmHmiInfoService.cpp.trc.h"
#endif

using namespace ::bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService;
using namespace ::spm_core_fi_types;

namespace App {
namespace Core {
/*
 * Constructor
 */
SxmHmiInfoService::SxmHmiInfoService()
   : _sxmHmiInfoServiceProxy(HmiInfoServiceProxy::createProxy("hmiinfoservicePort", *this))
   , _disconnect_src(false)
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR1(("SxmHmiInfoService Constructor"));
   SXM_UTILS_REGISTER_PROPERTY_CALLBACKS(StartupSync::getInstance(), this, _sxmHmiInfoServiceProxy);
}


/*
 * Destructor
 */
SxmHmiInfoService::~SxmHmiInfoService()
{
   _sxmHmiInfoServiceProxy.reset();
   ETG_TRACE_USR1(("~SxmHmiInfoService Destructor"));
   _disconnect_src = false;
   ETG_I_UNREGISTER_FILE();
}


/**
 * Utility function that registers for notifications updates from clock service.
 * @param [in] : proxy - clock service class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmHmiInfoService::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR1(("SxmHmiInfoService registerProperties"));
}


/**
 * Utility function that deregisters for notifications updates from clock service.
 * @param [in] : proxy - clock service class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmHmiInfoService::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& /*proxy*/,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR1(("SxmHmiInfoService deregisterProperties"));
}


/**
 * Utility function that registers for notifications updates from clock service.
 * @param [in] : proxy - clock service class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmHmiInfoService::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy,
                                    const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR1(("SxmHmiInfoService::onAvailable"));
   if (_sxmHmiInfoServiceProxy && (_sxmHmiInfoServiceProxy == proxy))
   {
      _sxmHmiInfoServiceProxy->sendLanguageStatusInfoRegister(*this);
      //to get update at starting
      _sxmHmiInfoServiceProxy->sendLanguageStatusInfoGet(*this);
      _sxmHmiInfoServiceProxy->sendToggleGadgetAudioSourceRequestRegister(*this);
      _sxmHmiInfoServiceProxy->sendCabinsSourceRestorationInfoRegister(*this);
      _sxmHmiInfoServiceProxy->sendCabinsSystemsInfoRegister(*this);
   }
}


/**
 * Utility function that deregisters for notifications updates from clock service.
 * @param [in] : proxy - clock service class instance
 * @param [in] : stateChange - Current ASF state
 */
void SxmHmiInfoService::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy,
                                      const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR1(("SxmHmiInfoService::onUnavailable"));
   if (_sxmHmiInfoServiceProxy && _sxmHmiInfoServiceProxy == proxy)
   {
      _sxmHmiInfoServiceProxy->sendLanguageStatusInfoDeregisterAll();
      _sxmHmiInfoServiceProxy->sendToggleGadgetAudioSourceRequestDeregisterAll();
      _sxmHmiInfoServiceProxy->sendCabinsSourceRestorationInfoDeregisterAll();
      _sxmHmiInfoServiceProxy->sendCabinsSystemsInfoDeregisterAll();
   }
}


/**
 * Error handling function for onLanguageStatusInfo
 * @param [in] : proxy - HmiInfoServiceProxy class instance
 * @param [in] : error - Response message
*/
void SxmHmiInfoService::onLanguageStatusInfoError(const ::boost::shared_ptr< HmiInfoServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< LanguageStatusInfoError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmHmiInfoService::onLanguageStatusInfoError"));
}


/**
 * Update handling function for onLanguageStatusInfo
 * @param [in] : proxy - HmiInfoServiceProxy class instance
 * @param [in] : update - language id
*/
void SxmHmiInfoService::onLanguageStatusInfoUpdate(const ::boost::shared_ptr< HmiInfoServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< LanguageStatusInfoUpdate >& update)
{
   if (update && update->hasLanguageStatusInfo())
   {
      ETG_TRACE_USR1(("SxmHmiInfoService::onLanguageStatusInfoUpdate language:%u", update->getLanguageStatusInfo()));
      SxmAudioDataBindingUtils::instance()->updateAudioReplayContentData();
   }
}


/**
 * Error handling function for onToggleGadgetAudioSource
 * @param [in] : proxy - HmiInfoServiceProxy class instance
 * @param [in] : error - Response message
*/
void SxmHmiInfoService::onToggleGadgetAudioSourceError(const ::boost::shared_ptr< HmiInfoServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< ToggleGadgetAudioSourceError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmHmiInfoService::onToggleGadgetAudioSourceError"));
}


/**
 * Response handling function of ToggleGadgetAudioSource
 * @param[in]: proxy: the client side representation of the CCA Functional Interface 'HmiInfoServiceProxy'
 * @param[in]: response : The response message of 'ToggleGadgetAudioSource'
 */
void SxmHmiInfoService::onToggleGadgetAudioSourceResponse(const ::boost::shared_ptr< HmiInfoServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< ToggleGadgetAudioSourceResponse >& /*response*/)
{
   ETG_TRACE_USR1(("SxmHmiInfoService::onToggleGadgetAudioSourceResponse"));
}


/**
 * Error handling function for onToggleGadgetAudioSourceRequest
 * @param [in] : proxy - HmiInfoServiceProxy class instance
 * @param [in] : error - Response message
*/
void SxmHmiInfoService::onToggleGadgetAudioSourceRequestError(const ::boost::shared_ptr<HmiInfoServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr<ToggleGadgetAudioSourceRequestError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmHmiInfoService::onToggleGadgetAudioSourceRequestError"));
}


/**
 * Signal handling function of onToggleGadgetAudioSourceRequest
 * @param[in]: proxy: the client side representation of the CCA Functional Interface 'HmiInfoServiceProxy'
 * @param[in]: response : The response message of 'ToggleGadgetAudioSource'
 */
void SxmHmiInfoService::onToggleGadgetAudioSourceRequestSignal(const ::boost::shared_ptr<HmiInfoServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr<ToggleGadgetAudioSourceRequestSignal >& /*signal*/)
{
   ETG_TRACE_USR1(("SxmHmiInfoService::onToggleGadgetAudioSourceRequestSignal"));
}


/**
 * Send Toggle request for Audio source
 * @param [in] : Active region
*/
void SxmHmiInfoService::sendToggleRequest(uint8 activeRegion)
{
   if (_sxmHmiInfoServiceProxy)
   {
      //SourceAppID-Tuner, Destination App ID - Media
      _sxmHmiInfoServiceProxy->sendToggleGadgetAudioSourceRequest(*this, activeRegion, APPID_APPHMI_TUNER, APPID_APPHMI_MEDIA);
   }
}


/**
 * Error handling function for onCabinsSystemsInfoError
 * @param [in] : proxy - HmiInfoServiceProxy class instance
 * @param [in] : error - Response message
 */
void SxmHmiInfoService::onCabinsSystemsInfoError(const ::boost::shared_ptr< HmiInfoServiceProxy >& /*proxy*/, const ::boost::shared_ptr< CabinsSystemsInfoError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmHmiInfoService::onCabinsSystemsInfoError"));
}


/**
 * Error handling function for onCabinsSourceRestorationInfoError
 * @param [in] : proxy - HmiInfoServiceProxy class instance
 * @param [in] : error - Response message
 */
void SxmHmiInfoService::onCabinsSourceRestorationInfoError(const ::boost::shared_ptr< HmiInfoServiceProxy >& /*proxy*/, const ::boost::shared_ptr< CabinsSourceRestorationInfoError >& /*error*/)
{
   ETG_TRACE_ERR(("SxmHmiInfoService::onCabinsSourceRestorationInfoError"));
}


/**
 * Signal handling function of onCabinsSystemsInfoUpdate
 * @param[in]: proxy: the client side representation of the CCA Functional Interface 'HmiInfoServiceProxy'
 * @param[in]: response : The response message of 'onCabinsSystemsInfoUpdate'
 */
void  SxmHmiInfoService::onCabinsSystemsInfoUpdate(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr< CabinsSystemsInfoUpdate >& update)
{
   ETG_TRACE_COMP(("SxmHmiInfoService::onCabinsSystemsInfoUpdate"));
   if (NULL != update && update->hasCabinsSystemsInfo())
   {
      ::std::vector< SystemInfo > systemInfoVector = update->getCabinsSystemsInfo();
      if (!systemInfoVector.empty())
      {
         ::std::vector< SystemInfo >::const_iterator systemInfo_iter = systemInfoVector.begin();
         for (; systemInfo_iter != systemInfoVector.end(); ++systemInfo_iter)
         {
            uint8 systemId = systemInfo_iter->getSystemId();
            uint8 systemPosition = systemInfo_iter->getSystemPosition();
            uint8 systemGroup = systemInfo_iter->getSystemGroup();
            uint8 systemStatus = systemInfo_iter->getSystemStatus();
            uint8 systemCategory = systemInfo_iter->getSystemCategory();
            ETG_TRACE_USR4(("SxmHmiInfoService::onCabinsSystemsInfoUpdate systemId %d, systemPosition %d, systemGroup %d, systemStatus %d, systemCategory %d", systemId, systemPosition, systemGroup, systemStatus, systemCategory));
            if (SYSTEM_CATEGORY_CMA == systemCategory)
            {
               if ((SYSTEM_STATUS_DISCONNECTED == systemStatus) ||
                     (SYSTEM_STATUS_INACTIVE_DISCONNECTED == systemStatus) ||
                     (SYSTEM_STATUS_PNM_DISCONNECTED == systemStatus) ||
                     (SYSTEM_STATUS_USER_OFF_DISCONNECTED == systemStatus) ||
                     (SYSTEM_STATUS_HOURLOGIC_DISCONNECTED == systemStatus) ||
                     (SYSTEM_STATUS_HMI_OFF_DISCONNECTED == systemStatus) ||
                     (SYSTEM_STATUS_COCKPIT_DISCONNECTED == systemStatus) ||
                     (SYSTEM_STATUS_GOODBYE_DISCONNECTED == systemStatus))
               {
                  int32 sourceId = SRC_INVALID;
                  uint32 sourceState = disconnected;
                  SXMSourceSwitchInterface::getSinkSourceStatus(CABIN_A_SINK_ID, sourceId, sourceState);
                  if ((sourceState == connected) && ((sourceId == SRC_TUNER_XM) || (sourceId == SRC_SXM_NEWS)) && (_disconnect_src == false))
                  {
                     _disconnect_src = true;
                     ETG_TRACE_USR4(("SrcChangeHandler::onCabinsSystemsInfoUpdate Disconnected: systemStatus %d,sourceId %d", systemStatus, sourceId));
                     SXMSourceSwitchInterface::requestSxmCabinSourceDeactivation(sourceId, sourceState);
                     break;
                  }
               }
               else if (SYSTEM_STATUS_CONNECTED == systemStatus)
               {
                  _disconnect_src = false;
               }
            }
         }
      }
   }
}


/**
 * Signal handling function of onCabinsSourceRestorationInfoUpdate
 * @param[in]: proxy: the client side representation of the CCA Functional Interface 'HmiInfoServiceProxy'
 * @param[in]: response : The response message of 'onCabinsSourceRestorationInfoUpdate'
 */
void  SxmHmiInfoService::onCabinsSourceRestorationInfoUpdate(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr<CabinsSourceRestorationInfoUpdate >& update)
{
   ETG_TRACE_COMP(("SxmHmiInfoService::onCabinsSourceRestorationInfoUpdate"));
   if (NULL != update && update->hasCabinsSourceRestorationInfo())
   {
      ::std::vector< CabinSourceRestorationInfo > restoreVector = update->getCabinsSourceRestorationInfo();
      if (!restoreVector.empty())
      {
         ::std::vector< CabinSourceRestorationInfo >::const_iterator cabinsSourceRestore_iter = restoreVector.begin();
         for (; cabinsSourceRestore_iter != restoreVector.end(); ++cabinsSourceRestore_iter)
         {
            uint8 cabinId = cabinsSourceRestore_iter->getCabinId();
            ETG_TRACE_USR4(("SxmHmiInfoService::onCabinsSourceRestorationInfoUpdate cabinId : %d ", cabinId));
            ::std::vector<SourceInfoHolder> SourcesInfoArrayData = cabinsSourceRestore_iter->getLastActiveSourcesInfo();
            ::std::vector<SourceInfoHolder>::const_iterator sourcesInfo_iter = SourcesInfoArrayData.begin();
            for (; sourcesInfo_iter != SourcesInfoArrayData.end(); ++sourcesInfo_iter)
            {
               int32 sourceId = sourcesInfo_iter->getSourceId();
               uint16 sinkId = sourcesInfo_iter->getSinkId();
               uint8 status = sourcesInfo_iter->getStatus();
               ETG_TRACE_USR4(("SxmHmiInfoService::onCabinsSourceRestorationInfoUpdate sinkId = %d, sourceId = %d, status = %d", sinkId, sourceId, status));
               if (((status == CABIN_RESTORE_STATUS_RESTORE)
                     && (sinkId == CABIN_A_SINK_ID)) && ((sourceId == SRC_TUNER_XM) || (sourceId == SRC_SXM_NEWS)))
               {
                  _disconnect_src = false;
                  SXMSourceSwitchInterface::requestSxmCabinSourceActivation(sourceId);
               }
            }
         }
      }
   }
}


}//Core
}//App
