/**************************************************************************//**
* \file       SdsAdapterProvider.cpp
*
* This file is part of the SDS HMI Application.
*
* \copyright  (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*             The reproduction, distribution and utilization of this file
*             as well as the communication of its contents to others without
*             express authorization is prohibited. Offenders will be held
*             liable for the payment of damages. All rights reserved in the
*             event of the grant of a patent, utility model or design.
*****************************************************************************/

#include "App/Core/SdsAdapter/Proxy/provider/SdsAdapterProvider.h"
#include "hmi_trace_if.h"
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#include "gui_std_if.h"
#include "App/Core/IHMIModelComponent.h"
#include "App/Core/SdsAdapter/GuiUpdater.h"
#include "App/Core/SdsAdapter/Proxy/requestor/SdsAdapterRequestor.h"
#include "App/Core/SdsDefines.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_SDS_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_SDS
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_SDS_"
#define ETG_I_FILE_PREFIX                 App::Core::SdsAdapterProvider::
#include "trcGenProj/Header/SdsAdapterProvider.cpp.trc.h"
#endif
#endif

using namespace ::App::Core;

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
using namespace sds_gui_fi::SdsGuiService;
using namespace sds_gui_fi::PopUpService;
using namespace sds_gui_fi::SettingsService;
using namespace sds_gui_fi::SdsPhoneService;

using namespace ::bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService;
#endif

#include <iostream>
using namespace std;

namespace App {
namespace Core {

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
SdsAdapterProvider::SdsAdapterProvider(
   GuiUpdater* pGuiUpdater,
   ListHandler* pListHandler,
   SdsAdapterRequestor* sdsAdapterRequestor
)
   : _pGuiUpdater(pGuiUpdater)
   , _pListhandler(pListHandler)
   , _pSdsAdapterRequestor(sdsAdapterRequestor)
   , _state(SdsAdapterProvider::INACTIVE)
   , Current_LayoutType(SDS_LAYOUT_INVALID)
   , Previous_LayoutType(SDS_LAYOUT_INVALID)
   , _talkingHeadState(0)
   , SRC_INVALID(-1)
   , mStatus(0)
   , mPageNumber(0)
   , mNumberOfPages(0)
   , _sdsStatus(sds_gui_fi::PopUpService::SpeechInputState__UNKNOWN)
   , mPause(false)
#else
SdsAdapterProvider::SdsAdapterProvider()
#endif
{
   ETG_TRACE_USR4(("SdsAdapterProvider::Constructor"));
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST

   _sdsGuiServiceProxy = sds_gui_fi::SdsGuiService::SdsGuiServiceProxy::createProxy("sdsGuiAppPort", *this);
   _popUpServiceProxy = sds_gui_fi::PopUpService::PopUpServiceProxy::createProxy("sdsPopUpPort", *this);

   _hmiInfoProxyClient = HmiInfoServiceProxy::createProxy("hmiinfoservicePort", *this);
   _SPMSetproxy        = SPM_CORE_FI::SPM_CORE_FIProxy::createProxy("spmFiPort", *this);
   _ApplicationSwitchProxy = bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::ApplicationSwitchProxy::createProxy("applicationSwitchServerPort", *this);
   ///////////////////Default Service Initializations////////////////////////////////////////////////////////////////////////////////////////////////////
   StartupSync::getInstance().registerPropertyRegistrationIF(this, _sdsGuiServiceProxy->getPortName());
   StartupSync::getInstance().registerPropertyRegistrationIF(this, _popUpServiceProxy->getPortName());
   StartupSync::getInstance().registerPropertyRegistrationIF(this, _ApplicationSwitchProxy->getPortName()); // change this
   StartupSync::getInstance().registerPropertyRegistrationIF(this, _hmiInfoProxyClient->getPortName());
   StartupSync::getInstance().registerPropertyRegistrationIF(this, _SPMSetproxy->getPortName());
   _applicationSwitchClientComponent = ApplicationSwitchClientComponent::poGetInstance();
#endif
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
void SdsAdapterProvider::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange */)
{
   ETG_TRACE_USR1(("SdsAdapterProvider::onAvailable"));

   if ((proxy == _ApplicationSwitchProxy) && (_ApplicationSwitchProxy.get()))
   {
      _ApplicationSwitchProxy->sendSig_activate_deactivate_SpeechSessionRegister(*this);
      ETG_TRACE_USR1(("SdsAdapterProvider::Sig_activate_deactivate_SpeechSessionRegister sucesfull"));
   }
   if ((proxy == _popUpServiceProxy) && (_popUpServiceProxy.get()))
   {
      _popUpServiceProxy->sendPopupRequestCloseRegister(*this);
      _popUpServiceProxy->sendPopupRequestRegister(*this);
      _popUpServiceProxy->sendSdsStatusRegister(*this);
   }
   /////////  Context Switch Concept ////////////////////////////////////////////
   if ((proxy == _sdsGuiServiceProxy) && (_sdsGuiServiceProxy.get()))
   {
      _sdsGuiServiceProxy ->sendActivateAudioDeviceTagRegister(*this);
      _sdsGuiServiceProxy->sendEventRegister(*this);
      _sdsGuiServiceProxy ->sendActivateAudioSourceRegister(*this);
   }
   if (proxy == _hmiInfoProxyClient)
   {
      ETG_TRACE_USR4(("SdsAdapterProvider::Registered with sendRVCSignalStatusRegister()"));
      _hmiInfoProxyClient->sendRVCSignalStatusRegister(*this);
   }
   if ((proxy == _SPMSetproxy) && (_SPMSetproxy.get()))
   {
      ETG_TRACE_USR4(("SdsAdapterProvider::Registered with SPMStatusRegister()"));
      _SPMSetproxy->sendHmiStateUpReg(*this);
   }
}


void SdsAdapterProvider::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange */)
{
   ETG_TRACE_USR1(("SdsAdapterProvider::onUnavailable "));

   if ((proxy == _ApplicationSwitchProxy) && (_ApplicationSwitchProxy.get()))
   {
      _ApplicationSwitchProxy->sendSig_activate_deactivate_SpeechSessionDeregisterAll();
   }
   if ((proxy == _popUpServiceProxy) && (_popUpServiceProxy.get()))
   {
      _popUpServiceProxy->sendPopupRequestCloseDeregisterAll();
      _popUpServiceProxy->sendPopupRequestDeregisterAll();
      _popUpServiceProxy->sendSdsStatusDeregisterAll();
   }
   ////////  Context Switch concept ////////////////
   if ((proxy == _sdsGuiServiceProxy) && (_sdsGuiServiceProxy.get()))
   {
      _sdsGuiServiceProxy ->sendActivateAudioDeviceTagDeregisterAll();
      _sdsGuiServiceProxy ->sendActivateAudioSourceDeregisterAll();
      _sdsGuiServiceProxy->sendEventDeregisterAll();
   }
   if (proxy == _hmiInfoProxyClient)
   {
      ETG_TRACE_USR4(("SdsAdapterProvider::sendRVCSignalStatusDeregisterAll()"));
      _hmiInfoProxyClient->sendRVCSignalStatusDeregisterAll();
   }
   if ((proxy == _SPMSetproxy) && (_SPMSetproxy.get()))
   {
      ETG_TRACE_USR4(("SdsAdapterProvider::Deregistered with SPMStatus"));
      _SPMSetproxy->sendHmiStateRelUpRegAll();
   }
}


void SdsAdapterProvider::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
}


/************************************************************************
*FUNCTION		: deregisterProperties
*DESCRIPTION	: Trigger property registration to mediaplayer,  called from MediaHall class
*PARAMETER		: proxy
*PARAMETER		: stateChange - state change service for corrosponding  proxy
*RETURNVALUE	: void
************************************************************************/
void SdsAdapterProvider::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
}


/************************************************************************
*FUNCTION		: onCourierMessage
*DESCRIPTION	: Startup message
*PARAMETER		: Courier::StartupMsg& oMsg
*PARAMETER		:
*RETURNVALUE	: bool
************************************************************************/
bool SdsAdapterProvider::onCourierMessage(const Courier::StartupMsg& /*oMsg*/)
{
   return true;
}


void SdsAdapterProvider::onPopupRequestSignal(const ::boost::shared_ptr< sds_gui_fi::PopUpService::PopUpServiceProxy >& /*proxy*/, const ::boost::shared_ptr< sds_gui_fi::PopUpService::PopupRequestSignal >& signal)
{
   ETG_TRACE_USR4(("----------SdsAdapterProvider::onPopupRequestSignal -------------"));
   ETG_TRACE_USR4(("Layout= %s", signal->getLayout().c_str()));//Remove
   ETG_TRACE_USR4(("getNumberOfPages= %d", signal->getNumberOfPages()));//Page count for list.
   //int32_t Listsize = 0;
   ::std::vector< sds_gui_fi::PopUpService::TextField > textFieldVector;

   textFieldVector = signal->getTextFields();
   mPageNumber     = signal->getPageNumber();
   mNumberOfPages  = signal->getNumberOfPages();
   _pListhandler->setListsize(mPageNumber);// page num status
   processHeaderSpeakingIcon(signal); // Icon Status
   ETG_TRACE_USR4(("textFieldVector.size() = %d", textFieldVector.size()));
   for (::std::vector< TextField >::iterator iter = textFieldVector.begin(); iter != textFieldVector.end(); ++iter)
   {
      ETG_TRACE_USR4(("iter.getString() = %s", iter->getString().c_str()));
   }
   if (_popupRequest.getTextFields() != signal->getTextFields())
   {
      ETG_TRACE_USR4(("SdsAdapterProvider:: Previous Text not equal TO Current Text"));
      processListData(signal);
   }

   if (_popupRequest.getTextFields() != signal->getTextFields()
         || _popupRequest.getLayout() != signal->getLayout())
   {
      ETG_TRACE_USR4(("SdsAdapterProvider:: processLayout"));
      processLayout(signal);
   }
   ETG_TRACE_USR4(("SdsAdapterProvider::processHeaderData getHeader() = %s", signal->getHeader().c_str()));
   if (_popupRequest.getHeader() != signal->getHeader())
   {
      ETG_TRACE_USR4(("SdsAdapterProvider:: processHeaderData"));
      processHeaderData(signal);
      /*  //For future reference
         ETG_TRACE_USR4(("SdsAdapterProvider:: processtextfieldData"));
        processTextfieldData(signal,requestedLayoutType); */
   }

   processGetCurrentPageNum(signal);
   /// Slider Data //
   processSliderData();
   if (signal.get())
   {
      _popupRequest = *signal.get();
   }
   if (_ApplicationSwitchProxy)//try
   {
      //for future reference
      //_ApplicationSwitchProxy->sendSpeechSession_activated_deactivatedRequest(*this, bool status true);
      ETG_TRACE_USR4(("sendSpeechSession_activated_deactivatedRequest %d", mStatus));
      _ApplicationSwitchProxy->sendSpeechSession_activated_deactivatedRequest(*this, mStatus);
   }
   ETG_TRACE_USR4(("SdsAdapterProvider:: END Request PopUp"));
}


std::string SdsAdapterProvider::getLayout(const ::boost::shared_ptr< sds_gui_fi::PopUpService::PopupRequestSignal >& signal) const
{
   std::string layoutType = signal->getLayout();
   return layoutType;
}


void SdsAdapterProvider::processHeaderSpeakingIcon(const ::boost::shared_ptr< sds_gui_fi::PopUpService::PopupRequestSignal >& signal)
{
   ETG_TRACE_USR4(("SdsAdapterProvider::processHeaderSpeakingIcon Listening State = %d ", signal->getSpeechInputState()));
   if (_pGuiUpdater)
   {
      _talkingHeadState = convertToTalkingHeadState(signal->getSpeechInputState());
      ETG_TRACE_USR4(("_talkingHeadState: speechInputState = %d", _talkingHeadState));

      _pGuiUpdater->updateHeaderSpeakingIconIndex(_talkingHeadState);
      if (_talkingHeadState == ((unsigned int)TALKINGHEAD_STATE_LISTENING))
      {
         _pGuiUpdater->bargeInEnableStatus(false);
      }
      else
      {
         _pGuiUpdater->bargeInEnableStatus(true);
      }
   }
}


unsigned int SdsAdapterProvider::convertToTalkingHeadState(sds_gui_fi::PopUpService::SpeechInputState speechInputState)
{
   ETG_TRACE_USR4(("SdsAdapterProvider :: convertToTalkingHeadState "));
   switch (speechInputState)
   {
      case sds_gui_fi::PopUpService::SpeechInputState__LISTENING:
         return ((unsigned int)TALKINGHEAD_STATE_LISTENING);

      case sds_gui_fi::PopUpService::SpeechInputState__ACTIVE:
         return ((unsigned int)TALKINGHEAD_STATE_ACTIVE);

      default:
         return ((unsigned int)TALKINGHEAD_STATE_IDLE);
   }
}


void SdsAdapterProvider::processHeaderData(const ::boost::shared_ptr< sds_gui_fi::PopUpService::PopupRequestSignal >& signal)
{
   ETG_TRACE_USR4(("SdsAdapterProvider::processHeaderData getHeader() = %s", signal->getHeader().c_str()));
   if (_pGuiUpdater)
   {
      _pGuiUpdater->updateHeaderTextData(signal->getHeader().c_str());
   }
   std::string layoutType = getLayout(signal);
   enSdsScreenLayoutType requestedLayoutType = getLayoutEnumValue(layoutType);
   if (requestedLayoutType == LAYOUT_CV_E_PHO /* && signal->getHeader()!= "" *//* future reference */)
   {
      if (_pGuiUpdater)
      {
         ETG_TRACE_USR4(("SdsAdapterProvider::updatePopupNumberfieldData"));
         _pGuiUpdater->updatePopupNumberfieldData(signal/* processNumberfieldvector *//* future reference */); //working one
      }
   }
}


/************************************************************************
*FUNCTION		: listSliderDisplay
*DESCRIPTION	: To display the slider in list screen.
*PARAMETER		: bool
************************************************************************/
void SdsAdapterProvider::listSliderDisplay(bool status)
{
   ETG_TRACE_USR4(("----------Entered SdsAdapterProvider::listScrollbarDisplay()--------------"));
   if (_pGuiUpdater)
   {
      _pGuiUpdater->listSliderDisplay(status);
   }
}


/************************************************************************
*FUNCTION		: processGetCurrentPageNum
*DESCRIPTION	: To display the scroll bar in list screen.
*PARAMETER		: bool
************************************************************************/
void SdsAdapterProvider::processGetCurrentPageNum(const ::boost::shared_ptr< sds_gui_fi::PopUpService::PopupRequestSignal >& signal)
{
   ETG_TRACE_USR4(("----------Entered SdsAdapterProvider::signal->getPageNumber()()--------------"));
   ETG_TRACE_USR4(("signal->getPageNumber() = %d", signal->getPageNumber()));
   ETG_TRACE_USR4(("signal->getNumberOfPages() = %d", signal->getNumberOfPages()));

   if (_pGuiUpdater)
   {
      if (signal->getPageNumber() == 0)
      {
         _pGuiUpdater->NextButtonEnableStatus(true);
         _pGuiUpdater->PrevButtonEnableStatus(false);
      }
      else
      {
         _pGuiUpdater->PrevButtonEnableStatus(true);
         _pGuiUpdater->NextButtonEnableStatus(false);
      }

      if (signal->getNumberOfPages() > 1)
      {
         std::string pageIndicator;
         pageIndicator = _pGuiUpdater->toString(signal->getPageNumber()) + "/" + _pGuiUpdater->toString(signal->getNumberOfPages());
         _pGuiUpdater->updatePageNumber(pageIndicator);
      }
      else
      {
         _pGuiUpdater->updatePageNumber("");
      }
   }
}


/************************************************************************
*FUNCTION		: processSliderData
*DESCRIPTION	: To display the slider and update contents of Slider in list screen.
*PARAMETER		: bool
************************************************************************/
void SdsAdapterProvider::processSliderData()
{
   if (mNumberOfPages > 1)
   {
      ETG_TRACE_USR4(("-----SdsAdapterProvider::calling listScrollbarDisplay() true-------"));
      listSliderDisplay(true);
      if (mPageNumber >= 1)
      {
         float SliderValue = 100 / (mNumberOfPages - 1);
         ETG_TRACE_USR4(("SdsAdapterProvider::SliderValue= %d", SliderValue));
         _pGuiUpdater->TimedStepMovementSlider((mNumberOfPages - 1), (SliderValue * mPageNumber));
         _pListhandler->setCurrentSliderValue(SliderValue * mPageNumber);
      }
      else
      {
         _pGuiUpdater->TimedStepMovementSlider((mNumberOfPages - 1), 0);
      }
   }
   else
   {
      listSliderDisplay(false);
   }
}


void SdsAdapterProvider::processTextfieldData(const ::boost::shared_ptr< sds_gui_fi::PopUpService::PopupRequestSignal >& signal, enSdsScreenLayoutType requestedLayoutType)
{
   if (_popupRequest.getTextFields() != signal->getTextFields())
   {
      ::std::vector< sds_gui_fi::PopUpService::TextField > processTextfieldvector;
      processTextfieldvector = signal->getTextFields();
      ETG_TRACE_USR4(("Current Layout SdsAdapterProvider::processTextfieldData = %d", requestedLayoutType));
      for (int i = 0; i < processTextfieldvector.size(); i++)
      {
         ETG_TRACE_USR4(("SdsAdapterProvider:: List Data: %d", i));
         ETG_TRACE_USR4(("SdsAdapterProvider:: List Data: %s", processTextfieldvector.at(i).getString().c_str()));
      }

      switch (requestedLayoutType)
      {
         case LAYOUT_CV_E_PHO:
            if (_pGuiUpdater)
            {
               ETG_TRACE_USR4(("SdsAdapterProvider::updatePopupNumberfieldData"));
               _pGuiUpdater->updatePopupNumberfieldData(signal/* processTextfieldvector *//* future reference */); //working one
            }
            break;
         case LAYOUT_CV_AUD_T1:
         case LAYOUT_CV_PHO_T1:
         case LAYOUT_CV_NAV_T1:
         case LAYOUT_CV_NAV_CONF:
         case LAYOUT_CV_AUD_CONF:
            if (_pGuiUpdater)
            {
               _pGuiUpdater->updatePopupTextfieldData(processTextfieldvector);
            }
            break;
         default:
            break;
      }
   }
}


void SdsAdapterProvider::processListData(const ::boost::shared_ptr< sds_gui_fi::PopUpService::PopupRequestSignal >& signal)
{
   ETG_TRACE_USR4(("SdsAdapterProvider::processListData"));

   std::string layoutType = getLayout(signal);
   ETG_TRACE_USR4(("layoutType= %s", layoutType)); //NDW5kor
   ETG_TRACE_USR4(("SdsAdapterProvider::processListData layoutType:%s", layoutType.c_str()));

   SelectableListItem selectableItem;

   ::std::vector< sds_gui_fi::PopUpService::TextField > textFieldVector;
   textFieldVector = signal->getTextFields();

   ETG_TRACE_USR4(("textFieldVector.size() = %d", textFieldVector.size()));
   _pListhandler->getVectCmdString().clear();

   for (::std::vector< TextField >::iterator iter = textFieldVector.begin(); iter != textFieldVector.end(); ++iter)
   {
      ETG_TRACE_USR4(("iter.getTagName() = %s", iter->getTagName().c_str()));
      ETG_TRACE_USR4(("iter.getString() = %s", iter->getString().c_str()));

      if (iter->getTagName().substr(0, TAGNAME_CMD_SIZE) == TAGNAME_CMD)
      {
         selectableItem.text = iter->getString();

         ETG_TRACE_USR4(("SdsAdapterProvider :: selectableItem = %s", selectableItem.text.c_str()));

         switch (iter->getAttrib())
         {
            case TextAttribute__NORMAL:
               selectableItem.isSelectable = true;
               break;
            case TextAttribute__COMMAND:
               selectableItem.isSelectable = true;
               break;
            case TextAttribute__GREYED_OUT:
               ETG_TRACE_USR4(("case TextAttribute__GREYED_OUT"));
               selectableItem.isSelectable = false;
               break;
            default:
               selectableItem.isSelectable = true;
               break;
         }
         ETG_TRACE_USR4(("SdsAdapterProvider :: isSelectable= %d", selectableItem.isSelectable));
         if (_pListhandler && (layoutType != "CV_S_PHO") && (layoutType != "CV_S_AUD") && (layoutType != "CV_S_NAV"))
         {
            _pListhandler->getVectCmdString().push_back(selectableItem);
         }
      }
      else if (_pListhandler && (iter->getTagName().substr(0, TAGNAME_DESC_SIZE) == TAGNAME_DESC) && ((layoutType == "CV_S_PHO") || (layoutType == "CV_S_AUD") || (layoutType == "CV_S_NAV")))
      {
         selectableItem.text += (" " + iter->getString());
         selectableItem.isSelectable = true;
         _pListhandler->getVectCmdString().push_back(selectableItem);
      }
   }
}


void SdsAdapterProvider::clearScreenData(std::string layoutType)
{
   if (_pListhandler && _pGuiUpdater)
   {
      _pListhandler->getVectCmdString().clear();
   }
}


void SdsAdapterProvider::processLayout(const ::boost::shared_ptr< sds_gui_fi::PopUpService::PopupRequestSignal >& signal)
{
   std::string layoutType = getLayout(signal);

   ETG_TRACE_USR4(("SdsAdapterProvider::processLayout layoutType:%s", layoutType.c_str()));
   Previous_LayoutType = Current_LayoutType;
   Extern_PreviousLayout_SM = Previous_LayoutType;

   ETG_TRACE_USR4(("Previous Layout = %d", Previous_LayoutType));
   ETG_TRACE_USR4(("Current Layout = %d", Current_LayoutType));

   ETG_TRACE_USR4(("Extern_PreviousLayout_SM = %d", Extern_PreviousLayout_SM));

   enSdsScreenLayoutType requestedLayoutType = getLayoutEnumValue(layoutType);

   Current_LayoutType = requestedLayoutType;

   ETG_TRACE_USR4(("Previous Layout = %d", Previous_LayoutType));
   ETG_TRACE_USR4(("Current Layout = %d", Current_LayoutType));

   if (requestedLayoutType != Previous_LayoutType)
   {
      _pListhandler->setShowScreenLayout(requestedLayoutType, Previous_LayoutType);
   }
   else
   {
      unsigned int listType = _pListhandler->getCurrentListType();
      _pListhandler->updateDynamicList(listType);
   }
   processTextfieldData(signal, requestedLayoutType); //applicable for cv_t1 layout
}


#endif


enSdsScreenLayoutType SdsAdapterProvider::getLayoutEnumNavValue(std::string layout) const
{
   if (layout == "CV_NAV_T4")
   {
      return LAYOUT_CV_NAV_T4;
   }
   else if (layout == "CV_S_NAV") //Layout yet to be finalised.
   {
      return LAYOUT_CV_S_NAV;
   }
   else if (layout == "CV_NAV_T2")
   {
      return LAYOUT_CV_NAV_T2;
   }
   else if (layout == "CV_NAV_T5")
   {
      return LAYOUT_CV_NAV_T5;
   }
   else if (layout == "CV_NAV_T1")
   {
      return LAYOUT_CV_NAV_T1;
   }
   else if (layout == "CV_NAV_CONF")
   {
      return LAYOUT_CV_NAV_CONF;
   }
   else
   {
      return SDS_LAYOUT_INVALID;
   }
}


enSdsScreenLayoutType SdsAdapterProvider::getLayoutEnumAudioValue(std::string layout) const
{
   if (layout == "CV_AUD") // Added as per SR_AUD_Main layout concept.
   {
      return LAYOUT_CV_AUD;
   }
   else if (layout == "CV_AUD_T1")
   {
      return LAYOUT_CV_AUD_T1;
   }
   else if (layout == "CV_S_AUD")
   {
      return LAYOUT_CV_S_AUD;
   }
   else if (layout == "CV_TUNER") // Added as per CV_TUNER layout concept.
   {
      return LAYOUT_CV_TUNER;
   }
   else if (layout == "CV_AUD_CONF")
   {
      return LAYOUT_CV_AUD_CONF;
   }
   else
   {
      return SDS_LAYOUT_INVALID;
   }
}


enSdsScreenLayoutType SdsAdapterProvider::getLayoutEnumPhoneValue(std::string layout) const
{
   if (layout == "CV_S_PHO")// Added as per SR_PHO_SelectPhoneNumber layout concept.
   {
      return LAYOUT_CV_S_PHO;
   }
   else if (layout == "CV_PHO_T1")
   {
      return LAYOUT_CV_PHO_T1;
   }
   else if (layout == "CV_PHO")
   {
      return LAYOUT_CV_PHO;
   }
   else if (layout == "CV_E_PHO")
   {
      return LAYOUT_CV_E_PHO;
   }
   else
   {
      return SDS_LAYOUT_INVALID;
   }
}


enSdsScreenLayoutType SdsAdapterProvider::getLayoutEnumValue(std::string layout) const
{
   ETG_TRACE_USR4((" SdsAdapterProvider::getLayoutEnumValue layout = %s", layout.c_str()));
   if (layout == "CV_GLO") // GLO_MAIN
   {
      return LAYOUT_CV_GLO;
   }
   else if ((layout == "CV_NAV_T4") || (layout == "CV_S_NAV") || (layout == "CV_NAV_T2") || (layout == "CV_NAV_T5") || (layout == "CV_NAV_T1") || (layout == "CV_NAV_CONF"))
   {
      return getLayoutEnumNavValue(layout);
   }
   else if ((layout == "CV_AUD") || (layout == "CV_AUD_T1") || (layout == "CV_S_AUD") || (layout == "CV_TUNER") || (layout == "CV_AUD_CONF"))
   {
      return getLayoutEnumAudioValue(layout);
   }
   else if ((layout == "CV_S_PHO") || (layout == "CV_PHO_T1") || (layout == "CV_PHO") || (layout == "CV_E_PHO"))
   {
      return getLayoutEnumPhoneValue(layout);
   }
   else
   {
      return SDS_LAYOUT_INVALID;
   }
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
void SdsAdapterProvider::onEventError(const ::boost::shared_ptr< SdsGuiServiceProxy >& /*proxy*/,
                                      const ::boost::shared_ptr< EventError >& /*error*/)
{
}


void SdsAdapterProvider::onEventSignal(const ::boost::shared_ptr< SdsGuiServiceProxy >& /*proxy*/,
                                       const ::boost::shared_ptr< EventSignal >& signal)
{
   switch (signal->getGuiEvent())
   {
      case Event__SPEECH_DIALOG_START_NAVIHOMEDESTINATION:
      case Event__SPEECH_DIALOG_SHOW_NAVI_SOURCECHANGE_ACTIVE:
      {
         ETG_TRACE_USR4(("Event__SPEECH_DIALOG_SHOW_NAVI_SOURCECHANGE_ACTIVE"));
         _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_SPEECH,
               enActivityIDs__eActivityID_SPEECH, APPID_APPHMI_NAVIGATION,
               (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)
               enActivityIDs__eActivityID_NAVI_MAP /* enActivityIDs__eActivityID_NAVI_ACTIVERG */);
      }
      break;
      case Event__SPEECH_DIALOG_SHOW_POPUP_LANGUAGE_NOT_SUPPORTED:
      {
         ETG_TRACE_USR4(("Event__SPEECH_DIALOG_SHOW_POPUP_LANGUAGE_NOT_SUPPORTED"));
         POST_MSG((COURIER_MESSAGE_NEW(SDSErrorResp)()));
         ETG_TRACE_USR4(("Message posted successfull for SDSErrorResp"));
      }
      break;
      default:
      {
         ETG_TRACE_USR4(("Gui event is %d", signal->getGuiEvent()));
      }
   }
}


void SdsAdapterProvider::onPopupRequestCloseError(
   const ::boost::shared_ptr< sds_gui_fi::PopUpService::PopUpServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< sds_gui_fi::PopUpService::PopupRequestCloseError >& /* error*/)
{
}


void SdsAdapterProvider::onPopupRequestCloseSignal(
   const ::boost::shared_ptr< sds_gui_fi::PopUpService::PopUpServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< sds_gui_fi::PopUpService::PopupRequestCloseSignal >& /*signal*/)
{
   Extern_CurrentLayout_SM = Current_LayoutType;
   ETG_TRACE_USR4(("SdsAdapterProvider::onPopupRequestCloseSignal"));
   ETG_TRACE_USR4(("Previous Layout = %d", Previous_LayoutType));
   ETG_TRACE_USR4(("Current Layout = %d", Current_LayoutType));
   ETG_TRACE_USR4(("Extern_CurrentLayout_SM= %d", Extern_CurrentLayout_SM));

   _pListhandler->closeVRScreen(Previous_LayoutType);

   Courier::Message* cpfoMsg = COURIER_MESSAGE_NEW(HideCurrentPopup)();

   if (cpfoMsg)
   {
      bool bRetVal = cpfoMsg->Post();
      ETG_TRACE_USR4(("SdsAdapterProvider::HideCurrentPopup Post"));
      if (false == bRetVal)
      {
         ETG_TRACE_USR4(("SdsAdapterProvider::onPopupRequestCloseSignal::Sending  Resp' = message failed"));
      }
      Current_LayoutType = SDS_LAYOUT_INVALID;
   }
   if (_pGuiUpdater)
   {
      _pGuiUpdater->ClearHeaderTextData();
   }
   _popupRequest.clear(); //Remove
   if (_ApplicationSwitchProxy)//try
   {
      ETG_TRACE_USR4(("onPopupRequestCloseSignal %d", mStatus));
      _ApplicationSwitchProxy->sendSpeechSession_activated_deactivatedRequest(*this, false);
   }
}


void SdsAdapterProvider::onPopupRequestError(
   const ::boost::shared_ptr< sds_gui_fi::PopUpService::PopUpServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< sds_gui_fi::PopUpService::PopupRequestError >& /* error*/)
{
}


///////////   Context Switch Concept ////////////////////////////

void SdsAdapterProvider::onActivateAudioDeviceTagError(
   const ::boost::shared_ptr< sds_gui_fi::SdsGuiService::SdsGuiServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< sds_gui_fi::SdsGuiService::ActivateAudioDeviceTagError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void SdsAdapterProvider::onActivateAudioDeviceTagSignal(
   const ::boost::shared_ptr< sds_gui_fi::SdsGuiService::SdsGuiServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< sds_gui_fi::SdsGuiService::ActivateAudioDeviceTagSignal >& signal)
{
   ETG_TRACE_USR4(("SdsAdapterProvider: onActivateAudioDeviceTagSignal"));
   switch (signal->getAudioSource())
   {
      case AudioSource__USB1:
      {
         ETG_TRACE_USR4(("AudioSource__USB : enActivityIDs__eActivityID_MEDIA_MCP_USB1"));
         _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_SPEECH,
               enActivityIDs__eActivityID_SPEECH, APPID_APPHMI_MEDIA,
               (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)
               enActivityIDs__eActivityID_MEDIA_MCP_USB1);
      }
      break;
      case AudioSource__USB2:
      {
         ETG_TRACE_USR4(("AudioSource__USB : enActivityIDs__eActivityID_MEDIA_MCP_USB2"));
         _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_SPEECH,
               enActivityIDs__eActivityID_SPEECH, APPID_APPHMI_MEDIA,
               (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)
               enActivityIDs__eActivityID_MEDIA_MCP_USB2);
      }
      break;
      case AudioSource__IPOD:
      {
         ETG_TRACE_USR4(("AudioSource__IPOD : enActivityIDs__eActivityID_MEDIA_MCP_IPOD"));
         _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_SPEECH,
               enActivityIDs__eActivityID_SPEECH, APPID_APPHMI_MEDIA,
               (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)
               enActivityIDs__eActivityID_MEDIA_MCP_IPOD);
      }
      break;
      default:
         ETG_TRACE_USR4(("missing getAudioSource"));
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void SdsAdapterProvider::onActivateAudioSourceError(
   const ::boost::shared_ptr< sds_gui_fi::SdsGuiService::SdsGuiServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< sds_gui_fi::SdsGuiService::ActivateAudioSourceError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void SdsAdapterProvider::onActivateAudioSourceSignal(
   const ::boost::shared_ptr< sds_gui_fi::SdsGuiService::SdsGuiServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< sds_gui_fi::SdsGuiService::ActivateAudioSourceSignal >& signal)
{
   int32 src = sdsSourceToHmiSource(signal->getAudioSource());
   ETG_TRACE_USR4(("SdsAdapterProvider : AudioSource src = %d", src));
}


/**************************************************************************//**
*
 ******************************************************************************/
unsigned int SdsAdapterProvider::sdsSourceToHmiSource(sds_gui_fi::SdsGuiService::AudioSource audioSource)
{
   ETG_TRACE_USR4(("SdsAdapterProvider: sdsSourceToHmiSource"));

   switch (audioSource)
   {
      case AudioSource__FM:
      {
         ETG_TRACE_USR4(("AudioSource__FM : enActivityIDs__eActivityID_TUNER_FM1"));

         _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_SPEECH,
               enActivityIDs__eActivityID_SPEECH, APPID_APPHMI_TUNER,
               (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)
               enActivityIDs__eActivityID_TUNER_FM1);

         return enActivityIDs__eActivityID_TUNER_FM1;
      }
      case AudioSource__FM_TA:
         return enActivityIDs__eActivityID_TUNER_FM_AST; // No Implementation
      case AudioSource__AM:
      {
         ETG_TRACE_USR4(("AudioSource__AM : enActivityIDs__eActivityID_TUNER_AM"));

         _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_SPEECH,
               enActivityIDs__eActivityID_SPEECH, APPID_APPHMI_TUNER,
               (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)
               enActivityIDs__eActivityID_TUNER_AM);
         return enActivityIDs__eActivityID_TUNER_AM;
      }
      case AudioSource__AM_TI:
         return enActivityIDs__eActivityID_TUNER_AM_AST; // No Implementation
      case AudioSource__DAB:
      {
         ETG_TRACE_USR4(("AudioSource__DAB : enActivityIDs__eActivityID_TUNER_DAB"));

         _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_SPEECH,
               enActivityIDs__eActivityID_SPEECH, APPID_APPHMI_TUNER,
               (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)
               enActivityIDs__eActivityID_TUNER_DAB);
         return enActivityIDs__eActivityID_TUNER_DAB;
      }
      case AudioSource__BT_AUDIO:
      {
         ETG_TRACE_USR4(("AudioSource__BT_AUDIO : enActivityIDs__eActivityID_MEDIA_MCP_BTSTREAMING"));
         _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_SPEECH,
               enActivityIDs__eActivityID_SPEECH, APPID_APPHMI_MEDIA,
               (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)
               enActivityIDs__eActivityID_MEDIA_MCP_BTSTREAMING);
         return enActivityIDs__eActivityID_MEDIA_MCP_BTSTREAMING;
      }
      case AudioSource__XM:
      {
         ETG_TRACE_USR4(("AudioSource__XM : enActivityIDs___eActivityID_SXM_MAIN"));

         _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_SPEECH,
               enActivityIDs__eActivityID_SPEECH, APPID_APPHMI_SXM,
               (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)
               enActivityIDs__eActivityID_SXM_MAIN);
         return enActivityIDs__eActivityID_SXM_MAIN;
      }
      default:
         ETG_TRACE_USR4(("missing source mapping"));
         return SRC_INVALID;
   }
}


///////////////      HMIState     //////////
void SdsAdapterProvider::onHmiStateError(const ::boost::shared_ptr< SPM_CORE_FI::SPM_CORE_FIProxy >& proxy,
      const boost::shared_ptr< ::SPM_CORE_FI::HmiStateError >& error)
{
   ETG_TRACE_ERR(("SdsAdapterProvider:: onHmiStateError"));
}


void SdsAdapterProvider::onHmiStateStatus(const ::boost::shared_ptr< ::SPM_CORE_FI::SPM_CORE_FIProxy >& proxy,
      const boost::shared_ptr< ::SPM_CORE_FI::HmiStateStatus >& status)
{
   ETG_TRACE_USR4(("SdsAdapterProvider::onHmiStateStatus() status->getHmiState() = %d", status->getHmiState()));

   /*
   	=============================== HMI State Mapping ===============================
   	*BACKGROUND / STANDBY / PREPARE_SHUTDOWN : T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_OFF
   	*IGNITION(User OFF)						 : T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_STANDBY
   	*ON										 : T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_HMI
   	*STATE_PROFILE (Hour Logic )			 : T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_HMI_HOUR_MODE
   	=============================== HMI State Mapping ===============================
   */

   switch (status->getHmiState())
   {
      case ::spm_core_fi_types::T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_OFF :
      {
         ETG_TRACE_USR4(("SdsAdapterProvider::onHmiStateStatus() HMISTATE_SHOW_OFF"));
         if (_pSdsAdapterRequestor && isSDSActive())
         {
            _pSdsAdapterRequestor->sendStopSession();
         }

         break;
      }
      default :
         ETG_TRACE_USR4(("SdsAdapterProvider::onHmiStateStatus() default"));
   }
}


////////////// PTT Signal Responce from ADapter////////////////////////////////
void SdsAdapterProvider::sendPttShortPress()
{
   ETG_TRACE_USR1(("SdsAdapterProvider::sendPttShortPress"));
   if (_sdsGuiServiceProxy != NULL)
   {
      _sdsGuiServiceProxy->sendPttPressRequest(*this, KeyState__KEY_HK_MFL_PTT_SHORT);
   }
}


void SdsAdapterProvider::onPttPressError(
   const ::boost::shared_ptr< SdsGuiServiceProxy >& proxy,
   const ::boost::shared_ptr< PttPressError >& error)
{
}


void SdsAdapterProvider::onPttPressResponse(
   const ::boost::shared_ptr< SdsGuiServiceProxy >& proxy,
   const ::boost::shared_ptr< PttPressResponse >& response)
{
}


// Functions for PTT Triger //////////////////////////
void SdsAdapterProvider::onSig_activate_deactivate_SpeechSessionError(const ::boost::shared_ptr< bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::ApplicationSwitchProxy >& proxy,
      const ::boost::shared_ptr< bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::Sig_activate_deactivate_SpeechSessionError >& error)
{
   ETG_TRACE_USR1(("SdsAdapterProvider::onSig_activate_deactivate_SpeechSessionError"));
}


void SdsAdapterProvider::onSig_activate_deactivate_SpeechSessionSignal(const ::boost::shared_ptr< bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::ApplicationSwitchProxy >& proxy,
      const ::boost::shared_ptr< bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::Sig_activate_deactivate_SpeechSessionSignal >& signal)
{
   mStatus = signal->getStatus();
   ETG_TRACE_USR1(("SdsAdapterProvider::onSig_activate_deactivate_SpeechSessionSignal %d", mStatus));
   if (mStatus)
   {
      sendPttShortPress();
   }
   else
   {
      if (_pSdsAdapterRequestor)
      {
         if (isSDSActive())
         {
            _pSdsAdapterRequestor->sendStopSession();
         }
      }
   }
}


void SdsAdapterProvider::onSpeechSession_activated_deactivatedError(const ::boost::shared_ptr< bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::ApplicationSwitchProxy >& proxy,
      const ::boost::shared_ptr< bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::SpeechSession_activated_deactivatedError >& error)
{
   ETG_TRACE_USR1(("SdsAdapterProvider::onSpeechSession_activated_deactivatedError"));
}


void SdsAdapterProvider::onSpeechSession_activated_deactivatedResponse(const ::boost::shared_ptr< bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::ApplicationSwitchProxy >& proxy,
      const ::boost::shared_ptr< bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::SpeechSession_activated_deactivatedResponse >& response)
{
   ETG_TRACE_USR1(("SdsAdapterProvider::onSpeechSession_activated_deactivatedResponse"));
}


#endif


// Functions for PTT Triger //////////////////////////
SdsAdapterProvider::~SdsAdapterProvider()
{
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   PURGE(_pGuiUpdater);
   PURGE(_pListhandler);
   //PURGE(_pHMIModelComponent);
#endif
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
// TODO : find a more generic way for this conversion
static std::string toString(unsigned int number)
{
   char acBuffer[40];
   sprintf(acBuffer, "%u", number);
   return acBuffer;
}


void SdsAdapterProvider::onScreenIDError(
   const ::boost::shared_ptr< PopUpServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< ScreenIDError >& /*error*/)
{
}


void SdsAdapterProvider::onScreenIDUpdate(
   const ::boost::shared_ptr< PopUpServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< ScreenIDUpdate >& update)
{
}


void SdsAdapterProvider::onRVCSignalStatusError(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy,
      const ::boost::shared_ptr< RVCSignalStatusError >& error)
{
   ETG_TRACE_USR4(("SdsAdapterProvider::onRVCSignalStatusError()"));
}


void SdsAdapterProvider::onRVCSignalStatusUpdate(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy,
      const ::boost::shared_ptr< RVCSignalStatusUpdate >& update)
{
   ETG_TRACE_USR4(("SdsAdapterProvider::onRVCSignalStatusUpdate():%d", update->getRVCSignalStatus()));
   if (update->getRVCSignalStatus())
   {
      if (_pSdsAdapterRequestor)
      {
         if (isSDSActive())
         {
            _pSdsAdapterRequestor->sendStopSession();
         }
      }
   }
}


void SdsAdapterProvider::onSdsStatusError(const ::boost::shared_ptr< sds_gui_fi::PopUpService::PopUpServiceProxy >& /* proxy */,
      const ::boost::shared_ptr< sds_gui_fi::PopUpService::SdsStatusError >& /* error */)
{
}


bool SdsAdapterProvider::isSDSActive()
{
   bool sdsActiveStatus = false;
   if ((_sdsStatus == sds_gui_fi::PopUpService::SpeechInputState__LISTENING) || (_sdsStatus == sds_gui_fi::PopUpService::SpeechInputState__ACTIVE) || (_sdsStatus == sds_gui_fi::PopUpService::SpeechInputState__DIALOGOPEN))
   {
      sdsActiveStatus = true;
   }
   ETG_TRACE_USR4(("isSDSActive %d", sdsActiveStatus));
   return sdsActiveStatus;
}


void SdsAdapterProvider::onSdsStatusUpdate(const ::boost::shared_ptr< sds_gui_fi::PopUpService::PopUpServiceProxy >& /* proxy */,
      const ::boost::shared_ptr< sds_gui_fi::PopUpService::SdsStatusUpdate >& update)
{
   _sdsStatus = update->getSdsStatus();
   ETG_TRACE_USR4(("sdsStatus %d", _sdsStatus));
   if (_sdsStatus == sds_gui_fi::PopUpService::SpeechInputState__PAUSE)
   {
      POST_MSG((COURIER_MESSAGE_NEW(SDSPause)()));
      mPause = true;
   }

   else
   {
      if (mPause && isSDSActive())
      {
         mPause = false;
         _pListhandler->setShowScreenLayout(Current_LayoutType, SDS_LAYOUT_INVALID);
      }
   }
}


bool SdsAdapterProvider::onCourierMessage(const HKStatusChangedNotifyMsg& oMsg)
{
   ETG_TRACE_USR4(("SdsAdapterProvider::onCourierMessage(HKStatusChangedNotifyMsg)"));
   if (_pSdsAdapterRequestor && isSDSActive())
   {
      _pSdsAdapterRequestor->sendStopSession();
   }

   return true;
}


bool SdsAdapterProvider::onCourierMessage(const StartVoiceRecognition& /* oMsg */)
{
   ETG_TRACE_USR4(("SdsAdapterProvider::onCourierMessage(StartVoiceRecognition)"));
   if (!isSDSActive())
   {
      sendPttShortPress();
   }

   return true;
}


#endif
} // Namespace Core
} // Namespace App
