/**************************************************************************************
 * @file         : DisplaySourceClientHandler.cpp
 * @author       : Bruce Mario Savio Netto
 * @addtogroup   : AppHmi_SPI
 * @brief        :
 * @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 *
 * 17.04.19        0.1 Bruce Netto (RBEI/ECH2)
 *                 New Client Handler to handle Display Resource Management in SPI
 **************************************************************************************/
#include "hall_std_if.h"
#include "Core/DisplaySourceClientHandler/DisplaySourceClientHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_SPI_HALL
#define ETG_I_TRACE_CHANNEL TR_TTFIS_APPHMI_SPI
#define ETG_I_TTFIS_CMD_PREFIX "APPHMI_SPI_"
#define ETG_I_FILE_PREFIX App::Core::DisplaySourceClientHandler::
#include "trcGenProj/Header/DisplaySourceClientHandler.cpp.trc.h"
#endif

using namespace ::asf::core;
using namespace ::bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService;
using namespace ::midw_smartphoneint_fi_types;

namespace App {
namespace Core {
/*****************************************************************
| static data implementation
|----------------------------------------------------------------*/
DisplaySourceClientHandler* DisplaySourceClientHandler::_displaySourceClientHandler = NULL;

/************************************************************************
*FUNCTION:		DisplaySourceClientHandler
*DESCRIPTION:	Constructor
*PARAMETER:     None
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     17.04.19      Bruce Netto       Initial Version
************************************************************************/
DisplaySourceClientHandler::DisplaySourceClientHandler(ConnectionClientHandler* pConnectionClientHandler)
   : _connectionClientHandler(pConnectionClientHandler)
   , _hmiInfoProxyClient(HmiInfoServiceProxy::createProxy("hmiinfoservicePort", *this))
{
   ETG_TRACE_USR4(("DisplaySourceClientHandler::DisplaySourceClientHandler()"));
   if (_hmiInfoProxyClient.get())
   {
      ETG_TRACE_USR4(("DisplaySourceClientHandler::DisplaySourceClientHandler StartupSync _hmiInfoProxyClient"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _hmiInfoProxyClient->getPortName());
   }
   ETG_I_REGISTER_FILE();
   m_displayFlag = true;
   m_RVCState = false;
   m_tempRVC = m_RVCState;
   m_valDisplayContext = ::midw_smartphoneint_fi_types::T_e8_DisplayContext__DISPLAY_CONTEXT_UNKNOWN;
   m_appID = APPID_APPHMI_MASTER;
   m_PrevAppID = APPID_APPHMI_UNKNOWN;
   currentAppID = APPID_APPHMI_MASTER;
}


/************************************************************************
*FUNCTION:		~DisplaySourceClientHandler
*DESCRIPTION:	Destructor
*PARAMETER:     None
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     17.04.19      Bruce Netto       Initial Version
************************************************************************/
DisplaySourceClientHandler::~DisplaySourceClientHandler()
{
   ETG_TRACE_USR4(("DisplaySourceClientHandler::~DisplaySourceClientHandler()"));
   if (_displaySourceClientHandler)
   {
      delete _displaySourceClientHandler;
   }
   _displaySourceClientHandler = NULL;
   ETG_I_UNREGISTER_FILE();
}


/************************************************************************
*FUNCTION:		onCourierMessage
*DESCRIPTION:	Function to process courier message
*PARAMETER:     StartupMsg
*RETURNVALUE:	bool
*SWFL:          NA
*HISTORY:
*0.1     12.02.19      Bruce Netto       Initial Version
************************************************************************/
bool DisplaySourceClientHandler::onCourierMessage(const Courier::StartupMsg& oMsg)
{
   return true;
}


/************************************************************************
*FUNCTION:		deregisterProperties
*DESCRIPTION:	Trigger property deregistration
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     17.04.19      Bruce Netto       Initial Version
************************************************************************/
void DisplaySourceClientHandler::deregisterProperties(const ::boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("DisplaySourceClientHandler::deregisterProperties()"));
}


/************************************************************************
*FUNCTION:		registerProperties
*DESCRIPTION:	Trigger property registration
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     17.04.19      Bruce Netto       Initial Version
************************************************************************/
void DisplaySourceClientHandler::registerProperties(const ::boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("DisplaySourceClientHandler::registerProperties()"));
}


/************************************************************************
*FUNCTION:		onUnavailable
*DESCRIPTION:	Method called when SPI service is unavailable
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     17.04.19      Bruce Netto       Initial Version
************************************************************************/
void DisplaySourceClientHandler::onUnavailable(const ::boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("DisplaySourceClientHandler::onUnavailable()"));
   if ((proxy == _hmiInfoProxyClient) && (_hmiInfoProxyClient.get()))
   {
      _hmiInfoProxyClient->sendCurrentForegroundApplicationIdDeregisterAll();
      ETG_TRACE_USR4(("DisplaySourceClientHandler::Registered with sendCurrentForegroundApplicationIdDeregisterAll()"));
      _hmiInfoProxyClient->sendRVCSignalStatusDeregisterAll();
      ETG_TRACE_USR4(("DisplaySourceClientHandler::Registered with sendRVCSignalStatusDeregisterAll()"));
   }
}


/************************************************************************
*FUNCTION:		onAvailable
*DESCRIPTION:	Method called when SPI service is available
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     17.04.19      Bruce Netto       Initial Version
************************************************************************/
void DisplaySourceClientHandler::onAvailable(const ::boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("DisplaySourceClientHandler::onAvailable()"));
   if ((proxy == _hmiInfoProxyClient) && (_hmiInfoProxyClient.get()))
   {
      _hmiInfoProxyClient->sendCurrentForegroundApplicationIdRegister(*this);
      ETG_TRACE_USR4(("DisplaySourceClientHandler::Registered with sendCurrentForegroundApplicationIdRegister()"));
      _hmiInfoProxyClient->sendRVCSignalStatusRegister(*this);
      ETG_TRACE_USR4(("DisplaySourceClientHandler::Registered with sendRVCSignalStatusRegister()"));
   }
}


/************************************************************************
*FUNCTION:		onCurrentForegroundApplicationIdError
*DESCRIPTION:	Method called when SPI service is available
*PARAMETER:    proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     22.04.19      Bruce Netto       Initial Version
************************************************************************/
void DisplaySourceClientHandler::onCurrentForegroundApplicationIdError(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy,
      const ::boost::shared_ptr< CurrentForegroundApplicationIdError >& error)
{
   ETG_TRACE_USR4(("DisplaySourceClientHandler::onCurrentForegroundApplicationIdError()"));
}


/************************************************************************
*FUNCTION:		onCurrentForegroundApplicationIdUpdate
*DESCRIPTION:	Method called when SPI service is available
*PARAMETER:    proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     22.04.19      Bruce Netto       Initial Version
************************************************************************/
void DisplaySourceClientHandler::onCurrentForegroundApplicationIdUpdate(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy,
      const ::boost::shared_ptr< CurrentForegroundApplicationIdUpdate >& update)
{
   ETG_TRACE_USR4(("DisplaySourceClientHandler::onCurrentForegroundApplicationIdUpdate()"));
   m_PrevAppID = m_appID;
   m_appID = update->getCurrentForegroundApplicationId();
   static bool lastStateisRVC = false;

   ETG_TRACE_USR4(("DisplaySourceClientHandler::onCurrentForegroundApplicationIdUpdate() Current App ID: %d", m_appID));
   ETG_TRACE_USR4(("DisplaySourceClientHandler::onCurrentForegroundApplicationIdUpdate() Previous App ID: %d", m_PrevAppID));
   currentAppID = m_appID;
   if (m_appID != m_PrevAppID && m_RVCState != true)
   {
      if (lastStateisRVC != true)
      {
         vsetAccessoryDisplayContext(m_appID);
      }
   }
   lastStateisRVC = m_RVCState;
}


/************************************************************************
*FUNCTION:		vCheckAccessoryDisplayContext
*DESCRIPTION:	Method to set the display context based on the APP ID received
*PARAMETER:    proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     22.04.19      Bruce Netto       Initial Version
************************************************************************/
void DisplaySourceClientHandler::vsetAccessoryDisplayContext(int m_appID)
{
   ETG_TRACE_USR4(("DisplaySourceClientHandler::vCheckAccessoryDisplayContext()"));
   // if (m_appID == APPID_APPHMI_SPI)
   // {
   // ETG_TRACE_USR4(("DisplaySourceClientHandler::vsetAccessoryDisplayContext return "));
   // m_displayFlag = false;
   // return;
   // }
   m_displayFlag = true;

   switch (m_appID)
   {
      case APPID_APPHMI_UNKNOWN:
      {
         ETG_TRACE_USR4(("DisplaySourceClientHandler::Display Context set as T_e8_DisplayContext__DISPLAY_CONTEXT_UNKNOWN"));
         m_valDisplayContext = T_e8_DisplayContext__DISPLAY_CONTEXT_UNKNOWN;
         break;
      }

      case APPID_APPHMI_PHONE:
      {
         ETG_TRACE_USR4(("DisplaySourceClientHandler::Display Context set as T_e8_DisplayContext__DISPLAY_CONTEXT_PHONE"));
         m_valDisplayContext = T_e8_DisplayContext__DISPLAY_CONTEXT_PHONE;
         break;
      }

      case APPID_APPHMI_NAVIGATION:
      {
         ETG_TRACE_USR4(("DisplaySourceClientHandler::Display Context set as T_e8_DisplayContext__DISPLAY_CONTEXT_NAVIGATION"));
         m_valDisplayContext = T_e8_DisplayContext__DISPLAY_CONTEXT_NAVIGATION;
         break;
      }

      case APPID_APPHMI_MEDIA:
      case APPID_APPHMI_TUNER:
      {
         ETG_TRACE_USR4(("DisplaySourceClientHandler::Display Context set as T_e8_DisplayContext__DISPLAY_CONTEXT_MEDIA"));
         m_valDisplayContext = T_e8_DisplayContext__DISPLAY_CONTEXT_MEDIA;
         break;
      }

      // case APPID_APPHMI_CAMERA:
      // {
      //    ETG_TRACE_USR4(("DisplaySourceClientHandler::Display Context set as T_e8_DisplayContext__DISPLAY_CONTEXT_EMERGENCY"));
      //    m_valDisplayContext = T_e8_DisplayContext__DISPLAY_CONTEXT_EMERGENCY;
      //    break;
      // }

      case APPID_APPHMI_SPEECH:
      {
         ETG_TRACE_USR4(("DisplaySourceClientHandler::Display Context set as T_e8_DisplayContext__DISPLAY_CONTEXT_SPEAKING"));
         m_valDisplayContext = T_e8_DisplayContext__DISPLAY_CONTEXT_SPEAKING;
         break;
      }

      case APPID_APPHMI_MASTER:
      {
         ETG_TRACE_USR4(("DisplaySourceClientHandler::Display Context set as T_e8_DisplayContext__DISPLAY_CONTEXT_NATIVE"));
         m_valDisplayContext = T_e8_DisplayContext__DISPLAY_CONTEXT_NATIVE;
         break;
      }

      case APPID_APPHMI_SPI:
      {
         ETG_TRACE_USR4(("DisplaySourceClientHandler::Display Context set as T_e8_DisplayContext__DISPLAY_CONTEXT_PROJECTION"));
         m_valDisplayContext = T_e8_DisplayContext__DISPLAY_CONTEXT_PROJECTION;
         ETG_TRACE_USR4(("DisplaySourceClientHandler::vsetAccessoryDisplayContext return "));
         m_displayFlag = false;
         return;
         break;
      }

      default:
         break;
   }
   vsendCurrentAppID(m_valDisplayContext, m_displayFlag);
}


int DisplaySourceClientHandler::getCurrentAppID()
{
   ETG_TRACE_USR4(("DisplaySourceClientHandler::getCurrentAppID() = %d", currentAppID));
   return currentAppID;
}


/************************************************************************
*FUNCTION:		vsendCurrentAppID
*DESCRIPTION:	Method to inform the SPI Middle ware about the current
*              change in the display AppID
*PARAMETER:    proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     22.04.19      Bruce Netto       Initial Version
************************************************************************/
void DisplaySourceClientHandler::vsendCurrentAppID(midw_smartphoneint_fi_types::T_e8_DisplayContext valDisplayContext, bool m_displayFlag)
{
   ETG_TRACE_USR4(("DisplaySourceClientHandler::vsendCurrentAppID()"));


//  ETG_TRACE_USR4(("DisplaySourceClientHandler::vsendCurrentAppID() this->m_displayFlag: %d", this->m_displayFlag));

   SetAccessoryDisplayContextDetails l_SetAccessoryDisplayContextDetails;

   midw_smartphoneint_fi_types::T_DisplayConstraint l_DisplayConstraint(
      /*Transfer Type*/ midw_smartphoneint_fi_types::T_e8_ResourceTransferType__TRANSFER_TYPE_TAKE,
      /*Transfer Priority*/ midw_smartphoneint_fi_types::T_e8_ResourceTransferPriority__TRANSFER_PRIORITY_USERINITIATED,
      /*Take Constraint*/ midw_smartphoneint_fi_types::T_e8_ResourceTransferConstraint__TRANSFER_CONSTRAINT_USERINITIATED,
      /*Borrow Constraint*/ midw_smartphoneint_fi_types::T_e8_ResourceTransferConstraint__TRANSFER_CONSTRAINT_NA, "");

   midw_smartphoneint_fi_types::T_AccessoryDisplayContext l_AccessoryDisplayContext(
      /*DisplayContext*/ valDisplayContext,
      /*DisplayFlag*/ m_displayFlag //Stop SPI Display Projection if FALSE 0 is for SPI, 1 is for HEadunit
   );

   l_SetAccessoryDisplayContextDetails.tDeviceHandle =  _connectionClientHandler->tGetActiveDeviceHandle();
   l_SetAccessoryDisplayContextDetails.enDisplayContextInfo = l_AccessoryDisplayContext;
   l_SetAccessoryDisplayContextDetails.enDisplayConstraintInfo = l_DisplayConstraint;
   //l_SetAccessoryDisplayContextDetails.enContextResponsibility = midw_smartphoneint_fi_types::T_e8_DisplayInfo__DISPLAY_CONSTRAINT;
   l_SetAccessoryDisplayContextDetails.enContextResponsibility = midw_smartphoneint_fi_types::T_e8_DisplayInfo__DISPLAY_CONTEXT; // SPI Component is responsible for display context mapping
   ETG_TRACE_USR4(("DisplaySourceClientHandler::vsendCurrentAppID  DeviceHandle: %d", _connectionClientHandler->tGetActiveDeviceHandle()));
   ETG_TRACE_USR4(("DisplaySourceClientHandler::vsendCurrentAppID() Current Context Sent: %d", valDisplayContext));
   ETG_TRACE_USR4(("DisplaySourceClientHandler::vsendCurrentAppID() m_displayFlag: %d", m_displayFlag));

   _connectionClientHandler->vSetAccessoryDisplayContext(l_SetAccessoryDisplayContextDetails);
   ETG_TRACE_USR4(("DisplaySourceClientHandler::vsendCurrentAppID() Current App ID Sent: %d", m_appID));
}


/************************************************************************
*FUNCTION:		onRVCSignalStatusError
*DESCRIPTION:	Method called when SPI service is available
*PARAMETER:    proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     22.04.19      Bruce Netto       Initial Version
************************************************************************/
void DisplaySourceClientHandler::onRVCSignalStatusError(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy,
      const ::boost::shared_ptr< RVCSignalStatusError >& error)
{
   ETG_TRACE_USR4(("DisplaySourceClientHandler::onRVCSignalStatusError()"));
}


/************************************************************************
*FUNCTION:		onRVCSignalStatusUpdate
*DESCRIPTION:	Method called when SPI service is available
*PARAMETER:    proxy, stateChange
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     22.04.19      Bruce Netto       Initial Version
************************************************************************/
void DisplaySourceClientHandler::onRVCSignalStatusUpdate(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy,
      const ::boost::shared_ptr< RVCSignalStatusUpdate >& update)
{
   ETG_TRACE_USR4(("DisplaySourceClientHandler::onRVCSignalStatusUpdate()"));
   m_RVCState = update->getRVCSignalStatus();
   ETG_TRACE_USR4(("DisplaySourceClientHandler::onRVCSignalStatusUpdate() m_RVCState = %d m_tempRVC = %d ", m_RVCState, m_tempRVC));
   if (m_RVCState != m_tempRVC)
   {
      vsetRVCInfo(m_RVCState);
   }
   else
   {
      ETG_TRACE_USR4(("DisplaySourceClientHandler::FALSE RVC CASE()"));
   }
}


/************************************************************************
*FUNCTION:		vsetRVCInfo
*DESCRIPTION:	Method called to check RVC info to set
*PARAMETER:    proxy, m_RVCState
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     29.05.19      Bruce Netto       Initial Version
************************************************************************/
void DisplaySourceClientHandler::vsetRVCInfo(bool m_RVCState)
{
   ETG_TRACE_USR4(("DisplaySourceClientHandler::vsetRVCInfo()"));
   ETG_TRACE_USR4(("DisplaySourceClientHandler::vsetRVCInfo() Previous App ID: %d", m_PrevAppID));
   if (m_RVCState == true)
   {
      ETG_TRACE_USR4(("DisplaySourceClientHandler::vsetRVCInfo() ENGAGED RVC"));
      m_valDisplayContext = T_e8_DisplayContext__DISPLAY_CONTEXT_EMERGENCY;
      m_displayFlag = true;
      vsendCurrentAppID(m_valDisplayContext, m_displayFlag);
   }
   else
   {
      ETG_TRACE_USR4(("DisplaySourceClientHandler::vsetRVCInfo() DISENGAGED RVC"));
      m_valDisplayContext = T_e8_DisplayContext__DISPLAY_CONTEXT_EMERGENCY;
      m_displayFlag = false;
      vsendCurrentAppID(m_valDisplayContext, m_displayFlag);
      //vsetAccessoryDisplayContext(m_PrevAppID);
   }
   m_tempRVC = m_RVCState;
}


}//namespace Core
}//namespace App
