/**************************************************************************************
 * @file         : ConnectionClientHandler.cpp
 * @author       : Athira K Sreekumar
 * @addtogroup   : AppHmi_SPI
 * @brief        :
 * @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 *20.04.18 Rev. 0.1 Athira K Sreekumar (RBEI/ECG2)
 * 		  			Initial Revision.
 *18.12.18 Rev. 0.2 Bruce Mario Savio Netto (RBEI/ECH2)
 *                  Bt Pairing and Spi Retry Popup implementation
 **************************************************************************************/

#include "hall_std_if.h"
#include "hmi_trace_if.h"
#include "Core/ConnectionClientHandler/ConnectionClientHandler.h"
#include "Core/VehicleDataSpiClientHandler/VehicleDataSpiClientHandler.h"
#include "Core/SystemPowerModeClientHandler/SystemPowerModeClientHandler.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_SPI_HALL
#define ETG_I_TRACE_CHANNEL TR_TTFIS_APPHMI_SPI
#define ETG_I_TTFIS_CMD_PREFIX "APPHMI_SPI_"
#define ETG_I_FILE_PREFIX App::Core::ConnectionClientHandler::
#include "trcGenProj/Header/ConnectionClientHandler.cpp.trc.h"
#endif

/************************************************************************
*FUNCTION:		isKthBitSet
*DESCRIPTION:	Check if k-th bit of a given number is set or not
*PARAMETER:     n - Given number, k - bit to be checked
*RETURNVALUE:	bool - true if bit is set. false if bit is not set
*SWFL:          NA
*HISTORY:		NA
*0.1     30.04.19      Sundeep Parvathikar       Initial Version
************************************************************************/
bool isKthBitSet(const uint8 n, const uint8 k)
{
   if (n & (1 << k))
   {
      return true;   //k-th bit inside number n is set
   }
   else
   {
      return false;   //k-th bit inside number n is not set
   }
}


using namespace ::midw_smartphoneint_fi;
namespace App {
namespace Core {

/*****************************************************************
| static data implementation
|----------------------------------------------------------------*/
ConnectionClientHandler* ConnectionClientHandler::_connectionClientHandler = NULL;

/************************************************************************
*FUNCTION:		ConnectionClientHandler
*DESCRIPTION:	Constructor
*PARAMETER:     None
*RETURNVALUE:	NA
*SWFL:          NA
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
************************************************************************/
ConnectionClientHandler::ConnectionClientHandler()
   : _smartphoneFIProxy(Midw_smartphoneint_fiProxy::createProxy("midwspiPort", *this))

{
   ETG_TRACE_USR4(("ConnectionClientHandler::ConnectionClientHandler()"));
   readKdsForTechnology();
   getKDSVehicleType();
   if (_smartphoneFIProxy.get())
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::ConnectionClientHandler StartupSync _smartphoneFIProxy"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _smartphoneFIProxy->getPortName());
   }

   ETG_I_REGISTER_FILE();
   vInitialize();
}


/************************************************************************
*FUNCTION:		~ConnectionClientHandler
*DESCRIPTION:	Destructor
*PARAMETER:     None
*RETURNVALUE:	NA
*SWFL:          NA
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
************************************************************************/
ConnectionClientHandler::~ConnectionClientHandler()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::~ConnectionClientHandler()"));
   if (_connectionClientHandler)
   {
      delete _connectionClientHandler;
   }
   _connectionClientHandler = NULL;
   ETG_I_UNREGISTER_FILE();
}


/************************************************************************
*FUNCTION:		deregisterProperties
*DESCRIPTION:	Trigger property deregistration
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
*SWFL:          NA
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::deregisterProperties(const ::boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::deregisterProperties()"));
}


/************************************************************************
*FUNCTION:		registerProperties
*DESCRIPTION:	Trigger property registration
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
*SWFL:          NA
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::registerProperties(const ::boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::registerProperties()"));
}


/************************************************************************
*FUNCTION:		vInitialize
*DESCRIPTION:	Initialization of member variables
*PARAMETER:    None
*RETURNVALUE:	NA
*SWFL:         SWFL-2445
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vInitialize()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vInitialize"));
   PairingStatus = UNKNOWN_INFO;
   m_bIsDeviceAdded = false;
   m_DeviceHandle = UNDEFINED;
   m_DeviceConnectionType = midw_smartphoneint_fi_types::T_e8_DeviceConnectionType__UNKNOWN_CONNECTION;
   m_DeviceStatus = midw_smartphoneint_fi_types::T_e8_DeviceStatusInfo__NOT_KNOWN;
   m_IsActiveDeviceHandleValid = false;
   m_ActiveDeviceCategory = ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_UNKNOWN;
   m_DeviceConnectionStatus = ::midw_smartphoneint_fi_types::T_e8_DeviceConnectionStatus__DEV_NOT_CONNECTED;
   m_HandSetInteraction = ::midw_smartphoneint_fi_types::T_e8_HandsetInteraction__HANDSET_INTERACTION_UNKNOWN;
   m_speech = ::midw_smartphoneint_fi_types::T_e8_SpeechAppState__SPI_APP_STATE_SPEECH_UNKNOWN;
   m_phonestate =  ::midw_smartphoneint_fi_types::T_e8_PhoneAppState__SPI_APP_STATE_PHONE_UNKNOWN;
   //m_SessionStatus = ::midw_smartphoneint_fi_types::T_e8_SessionStatus__SESSION_UNKNOWN;
   m_DeviceAlreadyConnected = false;
   m_bPairingReqd = true;
   m_SPIBTPopupShow = false;
   m_retryFlag = false;
   prevDeviceHandle = 0;
   mSleepContextFlag = false;
   m_status = ::midw_smartphoneint_fi_types::T_e8_DisplayContextReason__DISPLAY_CONTEXT_REASON_UNKNOWN;
   mParkBrakeInfo = ::midw_smartphoneint_fi_types::T_e8_ParkBrake__PARK_BRAKE_INFO_UNAVAILABLE;
   mVehicleParkState = ::midw_smartphoneint_fi_types::T_e8_ParkBrake__PARK_BRAKE_INFO_UNAVAILABLE;
   //Bruce
   //Databindings Initialization
   (*_SPIGadgetInfo).mDeviceConnection = DTR_NO_CONNECTION;
   _SPIGadgetInfo.MarkAllItemsModified();
   _SPIGadgetInfo.SendUpdate();
}


/************************************************************************
*FUNCTION:		onAvailable
*DESCRIPTION:	Method called when SPI service is available
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
*SWFL:          NA
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onAvailable(const ::boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onAvailable()"));

   if ((proxy == _smartphoneFIProxy) && (_smartphoneFIProxy.get()))
   {
      _smartphoneFIProxy->sendDeviceStatusInfoUpReg(*this);
      _smartphoneFIProxy->sendDeviceDisplayContextUpReg(*this);
      _smartphoneFIProxy->sendSessionStatusInfoUpReg(*this);
      _smartphoneFIProxy->sendProjectionDeviceAuthorizationUpReg(*this);
      _smartphoneFIProxy->sendBluetoothDeviceStatusUpReg(*this);
      vSetMySPINDisplayAttributesOnStartUp();
      vSetAndroidDeviceUsagePreference(ENABLE_INFO); // Bug = 575381
      vSetAppleDeviceUsagePreference(ENABLE_INFO);  // Bug = 575381
      //vSetDisplayAttributes();
      //vSetDeviceUsagePreference(ENABLE_INFO); //Not used for time being as middle ware takes care of this for all technologies
      // vSetTechnologyPreference();
   }
}


/************************************************************************
*FUNCTION:		onUnavailable
*DESCRIPTION:	Method called when SPI service is unavailable
*PARAMETER:     proxy, stateChange
*RETURNVALUE:	NA
*SWFL:          NA
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onUnavailable()"));
   _smartphoneFIProxy->sendDeviceStatusInfoRelUpRegAll();
   _smartphoneFIProxy->sendDeviceDisplayContextRelUpRegAll();
   _smartphoneFIProxy->sendSessionStatusInfoRelUpRegAll();
   _smartphoneFIProxy->sendProjectionDeviceAuthorizationRelUpRegAll();
   _smartphoneFIProxy->sendBluetoothDeviceStatusRelUpRegAll();
}


/************************************************************************
*FUNCTION:		onCourierMessage
*DESCRIPTION:	Function to process courier message
*PARAMETER:     StartupMsg
*RETURNVALUE:	bool
*SWFL:          NA
*HISTORY:
*0.1     04.05.18      Athira Sreekumar       Initial Version
************************************************************************/
bool ConnectionClientHandler::onCourierMessage(const Courier::StartupMsg& oMsg)
{
   return true;
}


/*********************************************************************************************
                                GETTERS AND SETTERS
*********************************************************************************************/
/************************************************************************
*FUNCTION:		vSetActiveDeviceCategory
*DESCRIPTION:	Set Active Device Category
*PARAMETER:     category`
*RETURNVALUE:	NA
*SWFL:          NA
*HISTORY:
*0.1     02.05.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vSetActiveDeviceCategory(::midw_smartphoneint_fi_types::T_e8_DeviceCategory category)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetDeviceCategory()"));
   m_ActiveDeviceCategory = category;
   //vSetDisplayAttributes();
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetDeviceCategory DeviceCategory = %d", m_ActiveDeviceCategory));
}


/************************************************************************
*FUNCTION:		tGetActiveDeviceCategory
*DESCRIPTION:	Get Active Device Category
*PARAMETER:     None
*RETURNVALUE:	category
*SWFL:          NA
*HISTORY:
*0.1     02.05.18      Athira Sreekumar       Initial Version
************************************************************************/
uint32 ConnectionClientHandler::tGetActiveDeviceCategory()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::tGetActiveDeviceCategory()"));
   switch (m_ActiveDeviceCategory)
   {
      case 0:
         ETG_TRACE_USR4(("ConnectionClientHandler::tGetActiveDeviceCategory Unknown"));
         break;
      case 4:
         ETG_TRACE_USR4(("ConnectionClientHandler::tGetActiveDeviceCategory MYSPIN"));
         break;
      case 1:
         ETG_TRACE_USR4(("ConnectionClientHandler::tGetActiveDeviceCategory CarPlay"));
         break;
      case 3:
         ETG_TRACE_USR4(("ConnectionClientHandler::tGetActiveDeviceCategory Android Auto"));
         break;
      default:
         ETG_TRACE_USR4(("ConnectionClientHandler::tGetActiveDeviceCategory Unknown"));
         break;
   }
   return m_ActiveDeviceCategory;
}


/************************************************************************
*FUNCTION:		vSetActiveDeviceHandleValidity
*DESCRIPTION:	set the device validity
*PARAMETER:     validity
*RETURNVALUE:	NA
*SWFL:          NA
*HISTORY:
*0.1     02.05.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vSetActiveDeviceHandleValidity(bool validity)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetActiveDeviceHandleValidity()"));
   m_IsActiveDeviceHandleValid = validity;
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetActiveDeviceHandleValidity Validity = %d", m_IsActiveDeviceHandleValid));
}


/************************************************************************
*FUNCTION:		bGetActiveDeviceHandleValidity
*DESCRIPTION:	get the device validity
*PARAMETER:     None
*RETURNVALUE:	deviceValidity
*SWFL:          NA
*HISTORY:
*0.1     02.05.18      Athira Sreekumar       Initial Version
************************************************************************/
bool ConnectionClientHandler::bGetActiveDeviceHandleValidity()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::bGetActiveDeviceHandleValidity()"));
   ETG_TRACE_USR4(("ConnectionClientHandler::bGetActiveDeviceHandleValidity Validity = %d", m_IsActiveDeviceHandleValid));
   return m_IsActiveDeviceHandleValid;
}


/************************************************************************
*FUNCTION:		vSetActiveDeviceHandle
*DESCRIPTION:	set the active device
*PARAMETER:     deviceHandle
*RETURNVALUE:	NA
*SWFL:          NA
*HISTORY:
*0.1     02.05.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vSetActiveDeviceHandle(uint32 deviceHandle)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetActiveDeviceHandle()"));
   m_DeviceHandle = deviceHandle;
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetActiveDeviceHandle DeviceHandle = %d", m_DeviceHandle));
}


/************************************************************************
*FUNCTION:		tGetActiveDeviceHandle
*DESCRIPTION:	get the active device
*PARAMETER:     None
*RETURNVALUE:	deviceHandle
*SWFL:          NA
*HISTORY:
*0.1     02.05.18      Athira Sreekumar       Initial Version
************************************************************************/
uint32 ConnectionClientHandler::tGetActiveDeviceHandle()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::tGetActiveDeviceHandle()"));
   ETG_TRACE_USR4(("ConnectionClientHandler::tGetActiveDeviceHandle DeviceHandle = %d", m_DeviceHandle));
   return m_DeviceHandle;
}


/************************************************************************
*FUNCTION:		vSetDeviceConnectionStatus
*DESCRIPTION:	set connection status
*PARAMETER:     None
*RETURNVALUE:	connectionStatus
*SWFL:          NA
*HISTORY:
*0.1     02.05.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vSetDeviceConnectionStatus(::midw_smartphoneint_fi_types::T_e8_DeviceConnectionStatus connectionStatus)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetDeviceConnectionStatus()"));
   m_DeviceConnectionStatus = connectionStatus;
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetDeviceConnectionStatus DeviceConnectionStatus = %d", m_DeviceConnectionStatus));
}


/************************************************************************
*FUNCTION:		tGetDeviceConnectionStatus
*DESCRIPTION:	get device connection status
*PARAMETER:     None
*RETURNVALUE:	deviceHandle
*SWFL:          NA
*HISTORY:
*0.1     02.05.18      Athira Sreekumar       Initial Version
************************************************************************/
uint32 ConnectionClientHandler::tGetDeviceConnectionStatus()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::tGetDeviceConnectionStatus()"));
   ETG_TRACE_USR4(("ConnectionClientHandler::tGetDeviceConnectionStatus DeviceConnectionStatus = %d", m_DeviceConnectionStatus));
   return m_DeviceConnectionStatus;
}


/************************************************************************
*FUNCTION:		vSetSessionStatus
*DESCRIPTION:	set session status
*PARAMETER:     None
*RETURNVALUE:	sessionStatus
*SWFL:          NA
*HISTORY:
*0.1     02.05.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vSetSessionStatus(::midw_smartphoneint_fi_types::T_e8_SessionStatus sessionStatus)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetSessionStatus()"));
   m_SessionStatus = sessionStatus;
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetSessionStatus SessionStatus = %d", m_SessionStatus));
}


/************************************************************************
*FUNCTION:		tGetSessionStatus
*DESCRIPTION:	get session status
*PARAMETER:     None
*RETURNVALUE:	sessionStatus
*SWFL:          NA
*HISTORY:
*0.1     02.05.18      Athira Sreekumar       Initial Version
************************************************************************/
midw_smartphoneint_fi_types::T_e8_SessionStatus ConnectionClientHandler::tGetSessionStatus()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::tGetSessionStatus()"));
   ETG_TRACE_USR4(("ConnectionClientHandler::tGetSessionStatus SessionStatus = %d", m_SessionStatus));
   return (::midw_smartphoneint_fi_types::T_e8_SessionStatus) m_SessionStatus;
}


/************************************************************************
*FUNCTION:		tGetDiPOAppPhoneStateinfo
*DESCRIPTION:	get DiPOAppPhoneState status
*PARAMETER:     None
*RETURNVALUE:	AppStatePhone
*SWFL:          NA
*HISTORY:
*0.1     02.05.18      Athira Sreekumar       Initial Version
************************************************************************/
int ConnectionClientHandler::tGetDiPOAppPhoneStateinfo()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::tGetDiPOAppPhoneStateinfo()"));
   ETG_TRACE_USR4(("ConnectionClientHandler::tGetDiPOAppStatusInfo DiPOAppPhoneState = %d", m_phonestate));
   return (::midw_smartphoneint_fi_types::T_e8_PhoneAppState) m_phonestate;
}


/************************************************************************
*FUNCTION:		tGetDiPOAppSpeechinfo
*DESCRIPTION:	get DiPOAppSpeechInfo status
*PARAMETER:     None
*RETURNVALUE:	AppStateSpeech
*SWFL:          NA
*HISTORY:
*0.1     02.05.18      Athira Sreekumar       Initial Version
************************************************************************/
int ConnectionClientHandler::tGetDiPOAppSpeechinfo()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::tGetDiPOAppSpeechinfo()"));
   ETG_TRACE_USR4(("ConnectionClientHandler::tGetDiPOAppSpeechinfo DiPOAppSpeechInfo = %d", m_speech));
   return (::midw_smartphoneint_fi_types::T_e8_SpeechAppState) m_speech;
}


/*********************************************************************************************
                                Property: DeviceStatusInfo
*********************************************************************************************/
/************************************************************************
*FUNCTION:		onDeviceStatusInfoStatus
*DESCRIPTION:	Update of device connection status
*PARAMETER:     proxy, status
*RETURNVALUE:	NA
*SWFL:          1993, 1994, 2581, 2584, 2604
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
*0.2     18.12.18      Bruce Mario Savio Netto  vOnDeviceRemovedPopupHandle();
************************************************************************/
void ConnectionClientHandler::onDeviceStatusInfoStatus(const ::boost::shared_ptr<Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::DeviceStatusInfoStatus>& status)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onDeviceStatusInfoStatus()"));
   uint32 l_DeviceHandle;
   ::midw_smartphoneint_fi_types::T_e8_DeviceConnectionType l_DeviceConnectionType;
   ::midw_smartphoneint_fi_types::T_e8_DeviceStatusInfo l_DeviceStatus;

   l_DeviceHandle = status->getDeviceHandle();
   l_DeviceConnectionType = status->getDeviceConnectionType();
   l_DeviceStatus = status->getDeviceStatus();
   m_DeviceStatus = l_DeviceStatus;

   ETG_TRACE_USR4(("ConnectionClientHandler::onDeviceStatusInfoStatus DeviceHandle = %d", l_DeviceHandle));
   ETG_TRACE_USR4(("ConnectionClientHandler::onDeviceStatusInfoStatus DeviceConnectionType = %d", l_DeviceConnectionType));
   ETG_TRACE_USR4(("ConnectionClientHandler::onDeviceStatusInfoStatus DeviceStatus = %d", l_DeviceStatus));

   if (midw_smartphoneint_fi_types::T_e8_DeviceStatusInfo__DEVICE_ADDED == l_DeviceStatus)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::onDeviceStatusInfoStatus Device Added"));
      vOnDeviceInfoStatusDeviceAdded(l_DeviceHandle, l_DeviceConnectionType);
   }

   else if (midw_smartphoneint_fi_types::T_e8_DeviceStatusInfo__DEVICE_REMOVED == l_DeviceStatus)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::onDeviceStatusInfoStatus Device Removed"));
      POST_MSG((COURIER_MESSAGE_NEW(SPI_SPI_ParkedCondition_Popup_Close)()));//USB Disconnect
      vOnDeviceInfoStatusDeviceRemoved(l_DeviceHandle);
      vOnDeviceRemovedPopupHandle();
   }

   else if (midw_smartphoneint_fi_types::T_e8_DeviceStatusInfo__DEVICE_CHANGED == l_DeviceStatus)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::onDeviceStatusInfoStatus Device Changed"));
      vOnDeviceInfoStatusDeviceChanged(l_DeviceHandle, l_DeviceConnectionType);
   }

   else if (midw_smartphoneint_fi_types::T_e8_DeviceStatusInfo__NOT_KNOWN == l_DeviceStatus)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::onDeviceStatusInfoStatus Device not known"));
   }
}


/************************************************************************
*FUNCTION:		onDeviceStatusInfoError
*DESCRIPTION:	DeviceStatusInfo updated with error
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          1993, 1994, 2581, 2584, 2604
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onDeviceStatusInfoError(const ::boost::shared_ptr<Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::DeviceStatusInfoError>& error)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onDeviceStatusInfoError()"));
}


/************************************************************************
*FUNCTION:		vOnDeviceInfoStatusDeviceAdded
*DESCRIPTION:	Update on device added
*PARAMETER:     None
*RETURNVALUE:	NA
*SWFL:          1993, 1994, 2581, 2604
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vOnDeviceInfoStatusDeviceAdded(uint32 deviceHandle,
      ::midw_smartphoneint_fi_types::T_e8_DeviceConnectionType deviceConnectionType)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vOnDeviceInfoStatusDeviceAdded()"));

   if (!bGetActiveDeviceHandleValidity()) //no device previously added
   {
      if (midw_smartphoneint_fi_types::T_e8_DeviceConnectionType__USB_CONNECTED == deviceConnectionType)
      {
         ETG_TRACE_USR4(("ConnectionClientHandler::vOnDeviceInfoStatusDeviceAdded USB Connected"));
         ETG_TRACE_USR4(("ConnectionClientHandler::vOnDeviceInfoStatusDeviceAdded deviceHandle = %d", deviceHandle));
         ETG_TRACE_USR4(("ConnectionClientHandler::vOnDeviceInfoStatusDeviceAdded deviceConnectionType = %d", deviceConnectionType));
         m_DeviceConnectionType = deviceConnectionType;
         m_bIsDeviceAdded = true;

         ETG_TRACE_USR4(("ConnectionClientHandler::vOnDeviceInfoStatusDeviceAdded sendGetDeviceInfoListStart"));
         _smartphoneFIProxy->sendGetDeviceInfoListStart(*this);
      }

      else
      {
         ETG_TRACE_USR4(("ConnectionClientHandler::vOnDeviceInfoStatusDeviceAdded Not connected via USB"));
      }
   }
}


/************************************************************************
*FUNCTION:		vOnDeviceInfoStatusDeviceRemoved
*DESCRIPTION:	Update on device removed
*PARAMETER:     deviceHandle
*RETURNVALUE:	NA
*SWFL:          2584
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vOnDeviceInfoStatusDeviceRemoved(uint32 deviceHandle)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vOnDeviceInfoStatusDeviceRemoved()"));
   if (bGetActiveDeviceHandleValidity() && (deviceHandle == tGetActiveDeviceHandle()))
   {
      m_bIsDeviceAdded = false;
      vSetActiveDeviceHandleValidity(false);
      vSetActiveDeviceHandle(UNDEFINED);
      vSetActiveDeviceCategory(midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_UNKNOWN);
      vSetDeviceConnectionStatus(midw_smartphoneint_fi_types::T_e8_DeviceConnectionStatus__DEV_NOT_CONNECTED);
      ETG_TRACE_USR4(("ConnectionClientHandler::vOnDeviceInfoStatusDeviceRemoved() set the display context as projection"));
      DisplaySourceClientHandler::getInstance(ConnectionClientHandler::getInstance())->vsendCurrentAppID(::midw_smartphoneint_fi_types::T_e8_DisplayContext__DISPLAY_CONTEXT_PROJECTION, true);  //Certification requirement

      vDisableSPILayers();

      if (m_CurrentView == VIEW_MYSPIN_APPVIEW || m_CurrentView == VIEW_S_FULLSCREEN /*ADD VIEW_S_FULLSCREEN for AAP and CP*/)//Bruce
      {
         POST_MSG((COURIER_MESSAGE_NEW(SPIView_Phone_not_connected)()));
      }
   }
}


/************************************************************************
*FUNCTION:		vOnDeviceInfoStatusDeviceChanged
*DESCRIPTION:	Update on device changed
*PARAMETER:     deviceHandle, deviceConnectionType
*RETURNVALUE:	NA
*SWFL:          1993, 1994, 2581, 2604
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vOnDeviceInfoStatusDeviceChanged(uint32 deviceHandle,
      ::midw_smartphoneint_fi_types::T_e8_DeviceConnectionType deviceConnectionType)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vOnDeviceInfoStatusDeviceChanged()"));
   ETG_TRACE_USR4(("ConnectionClientHandler::vOnDeviceInfoStatusDeviceChanged deviceHandle = %d", deviceHandle));
   ETG_TRACE_USR4(("ConnectionClientHandler::vOnDeviceInfoStatusDeviceChanged deviceConnectionType = %d", deviceConnectionType));

   if (midw_smartphoneint_fi_types::T_e8_DeviceConnectionType__USB_CONNECTED == deviceConnectionType)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::sendGetDeviceInfoListStart"));
      _smartphoneFIProxy->sendGetDeviceInfoListStart(*this);
   }
}


/*********************************************************************************************
                                Method: GetDeviceInfoList
*********************************************************************************************/
/************************************************************************S
*FUNCTION:		onGetDeviceInfoListResult
*DESCRIPTION:	Function to get DeviceInfoList
*PARAMETER:     proxy, result
*RETURNVALUE:	NA
*SWFL:          1993, 1994, 2581, 2604
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onGetDeviceInfoListResult(const ::boost::shared_ptr<Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::GetDeviceInfoListResult>& result)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onGetDeviceInfoListResult()"));

   uint16 l_numDevices = result->getNumDevices();

   std::vector<midw_smartphoneint_fi_types::T_DeviceDetails> l_DeviceInfoList = result->getDeviceInfoList();
   std::vector<midw_smartphoneint_fi_types::T_DeviceDetails>::const_iterator l_DeviceInfoIterator = l_DeviceInfoList.begin();

   ETG_TRACE_USR4(("ConnectionClientHandler::onGetDeviceInfoListResult numDevices = %d", l_numDevices));

   uint16 l_DeviceCount = 1;
   for (; l_DeviceInfoIterator != l_DeviceInfoList.end(); ++l_DeviceInfoIterator, ++l_DeviceCount)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::onGetDeviceInfoListResult  Device %d DeviceName: %s", l_DeviceCount, (l_DeviceInfoIterator->getSzDeviceName())));
      ETG_TRACE_USR4(("ConnectionClientHandler::onGetDeviceInfoListResult  Device %d DeviceHandle: %d", l_DeviceCount, (l_DeviceInfoIterator->getU32DeviceHandle())));

      if (::midw_smartphoneint_fi_types::T_e8_DeviceConnectionStatus__DEV_CONNECTED == l_DeviceInfoIterator->getEnDeviceConnectionStatus())
      {
         vHandleDeviceDetails(*l_DeviceInfoIterator);
      }
   }
}


/************************************************************************
*FUNCTION:		onGetDeviceInfoListError
*DESCRIPTION:	DeviceInfoList updated with error
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          1993, 1994, 2581, 2604
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onGetDeviceInfoListError(const ::boost::shared_ptr<Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::GetDeviceInfoListError>& error)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onGetDeviceInfoListError()"));
}


/************************************************************************
*FUNCTION:		vHandleDeviceDetails
*DESCRIPTION:	Check and handle device details
*PARAMETER:     deviceDetails
*RETURNVALUE:	NA
*SWFL:          1993, 1994, 2581, 2604
*HISTORY:
*0.1     02.05.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vHandleDeviceDetails(const ::midw_smartphoneint_fi_types::T_DeviceDetails& deviceDetails)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vHandleDeviceDetails()"));

   if (bDeviceDetailsValidity(deviceDetails))
   {
      vSetActiveDeviceHandleValidity(true);
      vSetActiveDeviceHandle(deviceDetails.getU32DeviceHandle());
      vSetActiveDeviceCategory(midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_UNKNOWN);
      vSetDeviceConnectionStatus(midw_smartphoneint_fi_types::T_e8_DeviceConnectionStatus__DEV_NOT_CONNECTED);

      if (deviceDetails.getRProjectionCapability().getEnDeviceType() == midw_smartphoneint_fi_types::T_e8_DeviceType__ANDROID_DEVICE)
      {
         ETG_TRACE_USR4(("ConnectionClientHandler::vHandleDeviceDetails Android device connected"));

         vSetActiveDeviceCategory(deviceDetails.getEnDeviceCategory());
         ETG_TRACE_USR4(("ConnectionClientHandler::vHandleDeviceDetails DeviceCategory set as %d", deviceDetails.getEnDeviceCategory()));

         vSetDeviceConnectionStatus(deviceDetails.getEnDeviceConnectionStatus());
         ETG_TRACE_USR4(("ConnectionClientHandler::vHandleDeviceDetails DeviceConnectionStatus set as %d", deviceDetails.getEnDeviceConnectionStatus()));
         vSetAndroidDisplayAttributes();
         //vSetAndroidDeviceUsagePreference(ENABLE_INFO);
         vSetTechnologyAndroidPreference();
         //vSetVehicleMovementParkBrake(); command to set the parked condition
      }
      //Bruce
      else if (deviceDetails.getRProjectionCapability().getEnDeviceType() == midw_smartphoneint_fi_types::T_e8_DeviceType__APPLE_DEVICE)
      {
         ETG_TRACE_USR4(("ConnectionClientHandler::vHandleDeviceDetails Apple  device connected"));

         vSetActiveDeviceCategory(deviceDetails.getEnDeviceCategory());
         ETG_TRACE_USR4(("ConnectionClientHandler::vHandleDeviceDetails DeviceCategory set as %d", deviceDetails.getEnDeviceCategory()));

         vSetDeviceConnectionStatus(deviceDetails.getEnDeviceConnectionStatus());
         ETG_TRACE_USR4(("ConnectionClientHandler::vHandleDeviceDetails DeviceConnectionStatus set as %d", deviceDetails.getEnDeviceConnectionStatus()));
         vSetAppleDisplayAttributes();
         //vSetAppleDeviceUsagePreference(ENABLE_INFO);
         vSetTechnologyApplePreference();
      }
   }

   else
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::vHandleDeviceDetails() Validity failed"));
   }
   vCheckForeGroundNavi();
}


/************************************************************************
*FUNCTION:		bDeviceDetailsValidity
*DESCRIPTION:	Check validity of the device
*PARAMETER:     deviceDetails
*RETURNVALUE:	NA
*SWFL:          1993, 1994, 2581, 2604
*HISTORY:
*0.1     02.05.18      Athira Sreekumar       Initial Version
************************************************************************/
bool ConnectionClientHandler::bDeviceDetailsValidity(const ::midw_smartphoneint_fi_types::T_DeviceDetails& deviceDetails)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::bDeviceDetailsValidity()"));
   if (((deviceDetails.getRProjectionCapability().getEnDeviceType() == midw_smartphoneint_fi_types::T_e8_DeviceType__ANDROID_DEVICE) ||
         (deviceDetails.getRProjectionCapability().getEnDeviceType() == midw_smartphoneint_fi_types::T_e8_DeviceType__APPLE_DEVICE)) &&
         (deviceDetails.getEnDeviceConnectionType() == midw_smartphoneint_fi_types::T_e8_DeviceConnectionType__USB_CONNECTED) &&
         //(deviceDetails.getRProjectionCapability().getEnUSBPortType() == midw_smartphoneint_fi_types::T_e8_USBPortType__PORT_TYPE_OTG) && //Middleware will take care
         (deviceDetails.getEnDeviceConnectionStatus() == midw_smartphoneint_fi_types::T_e8_DeviceConnectionStatus__DEV_CONNECTED))
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::bDeviceDetailsValidity Device Eligible"));
      return true;
   }

   else
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::bDeviceDetailsValidity Device Not Eligible"));
      return false;
   }
}


/*********************************************************************************************
                                Method: SetDisplayAttributes
*********************************************************************************************/
/************************************************************************
*FUNCTION:		onSetDisplayAttributesResult
*DESCRIPTION:	Function to get result of SetDisplayAttributes
*PARAMETER:     proxy, result
*RETURNVALUE:	NA
*SWFL:          1993, 2581, 2604
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onSetDisplayAttributesResult(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::SetDisplayAttributesResult>& result)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onSetDisplayAttributesResult()"));
   ETG_TRACE_USR4(("ConnectionClientHandler::onSetDisplayAttributesResult ResponseCode = %d", result->getResponseCode()));
}


/************************************************************************
*FUNCTION:		onSetDisplayAttributesError
*DESCRIPTION:	SetDisplayAttributes with error
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          1993, 2581, 2604
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onSetDisplayAttributesError(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::SetDisplayAttributesError>& error)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onSetDisplayAttributesError()"));
}


/************************************************************************
*FUNCTION:		vSetDisplayAttributes
*DESCRIPTION:	Set Display Attributes
*PARAMETER:     NA
*RETURNVALUE:	NA
*SWFL:          1993, 2581, 2604
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
************************************************************************/
// void ConnectionClientHandler::vSetDisplayAttributes()
// {
//    ETG_TRACE_USR4(("ConnectionClientHandler::vSetDisplayAttributes()"));
//    ETG_TRACE_USR4(("ConnectionClientHandler::vSetDisplayAttributes DeviceCategory = %d", m_ActiveDeviceCategory));

//    if (_smartphoneFIProxy)
//    {
//       if(ismySPINEnabled())//KDS CHECK
//       {
//          ETG_TRACE_USR4(("ConnectionClientHandler::vSetDisplayAttributes MY SPIN DeviceCategory = %d", m_ActiveDeviceCategory));
//          std::vector<midw_smartphoneint_fi_types::T_DisplayLayerAttributes> l_DisplayLayers;

//          midw_smartphoneint_fi_types::T_DisplayLayerAttributes l_DisplayLayerAttr_MYSPIN(
//             /*DeviceCategory*/      midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN,
//             /*TouchLayerID*/        3501,
//             /*TouchSurfaceID*/      80,
//             /*VideoLayerID*/        3500,
//             /*VideoSurfaceID*/      59,
//             /*LayerWidth*/          702,
//             /*LayerHeight*/         472,
//             /*LayerWidthinMM*/      137.2059,
//             /*LayerHeightinMM*/     80.712,
//             /*PixelAspectRatio*/    "1.69",
//             /*DPI*/                 160,
//             /*DisplayTouchType*/    midw_smartphoneint_fi_types::T_e8_DisplayTouchType__e8MULTI_TOUCH,
//             /*PriorityIndex*/       0);

//          l_DisplayLayers.push_back(l_DisplayLayerAttr_MYSPIN);

//          midw_smartphoneint_fi_types::T_DisplayAttributes l_DisplayAttributes(
//             /*ScreenHeight*/            480,
//             /*ScreenWidth*/             700,
//             /*ScreenHeightMm*/          186,
//             /*ScreenWidthMm*/           127,
//             /*DisplayType*/             midw_smartphoneint_fi_types::T_e8_DisplayType__e8CAPACITIVE,
//             /*DisplayLayerAttributes*/  l_DisplayLayers);

//          //Sending Required Screen Configuration to be displayed
//          _smartphoneFIProxy->sendSetDisplayAttributesStart(*this, l_DisplayAttributes);
//       }
//    }
// }

/************************************************************************
*FUNCTION:		vSetAndroidDisplayAttributes
*DESCRIPTION:	Set Display Attributes
*PARAMETER:     NA
*RETURNVALUE:	NA
*SWFL:          1993, 2581, 2604
*HISTORY:
*0.1     26.06.19      Bruce Netto      Initial Version
************************************************************************/
void ConnectionClientHandler::vSetAndroidDisplayAttributes()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetAndroidDisplayAttributes()"));
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetAndroidDisplayAttributes DeviceCategory = %d", m_ActiveDeviceCategory));

   if (_smartphoneFIProxy)
   {
      if (ismySPINEnabled()) //KDS CHECK
      {
         ETG_TRACE_USR4(("ConnectionClientHandler::vSetAndroidDisplayAttributes MY SPIN DeviceCategory = %d", m_ActiveDeviceCategory));
         std::vector<midw_smartphoneint_fi_types::T_DisplayLayerAttributes> l_DisplayLayers;

         midw_smartphoneint_fi_types::T_DisplayLayerAttributes l_DisplayLayerAttr_MYSPIN(
            /*DeviceCategory*/      midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN,
            /*TouchLayerID*/        3501,
            /*TouchSurfaceID*/      80,
            /*VideoLayerID*/        3500,
            /*VideoSurfaceID*/      59,
            /*LayerWidth*/          706,
            /*LayerHeight*/         480,
            /*LayerWidthinMM*/      137.2059,
            /*LayerHeightinMM*/     80.712,
            /*PixelAspectRatio*/    "1.69",
            /*DPI*/                 160,
            /*DisplayTouchType*/    midw_smartphoneint_fi_types::T_e8_DisplayTouchType__e8MULTI_TOUCH,
            /*PriorityIndex*/       0);

         l_DisplayLayers.push_back(l_DisplayLayerAttr_MYSPIN);

         midw_smartphoneint_fi_types::T_DisplayAttributes l_DisplayAttributes(
            /*ScreenHeight*/            480,
            /*ScreenWidth*/             700,
            /*ScreenHeightMm*/          186,
            /*ScreenWidthMm*/           127,
            /*DisplayType*/             midw_smartphoneint_fi_types::T_e8_DisplayType__e8CAPACITIVE,
            /*DisplayLayerAttributes*/  l_DisplayLayers);

         //Sending Required Screen Configuration to be displayed
         _smartphoneFIProxy->sendSetDisplayAttributesStart(*this, l_DisplayAttributes);
      }
      else
      {
         std::vector<midw_smartphoneint_fi_types::T_DisplayLayerAttributes> l_DisplayLayers;
         //FOR APPLE DEVICE: HINT Whats the config for android?
         midw_smartphoneint_fi_types::T_DisplayLayerAttributes l_DisplayLayerAttr_APPLE_DEVICE(
            /*DeviceCategory*/      midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO,
            /*TouchLayerID*/        3503,
            /*TouchSurfaceID*/      80,
            /*VideoLayerID*/        3502,
            /*VideoSurfaceID*/      59,
            /*LayerWidth*/          800,
            /*LayerHeight*/         480,
            /*LayerWidthinMM*/      138,
            /*LayerHeightinMM*/     82,
            /*PixelAspectRatio*/    "1.69",
            /*DPI*/                 160,
            /*DisplayTouchType*/    midw_smartphoneint_fi_types::T_e8_DisplayTouchType__e8MULTI_TOUCH,
            /*PriorityIndex*/       0);

         l_DisplayLayers.push_back(l_DisplayLayerAttr_APPLE_DEVICE);
         ETG_TRACE_USR4(("ConnectionClientHandler::vSetAndroidDisplayAttributes ANDROID DeviceCategory = %d", m_ActiveDeviceCategory));

         midw_smartphoneint_fi_types::T_DisplayAttributes l_DisplayAttributes(
            /*ScreenHeight*/            480,
            /*ScreenWidth*/             800,
            /*ScreenHeightMm*/          186,
            /*ScreenWidthMm*/           127,
            /*DisplayType*/             midw_smartphoneint_fi_types::T_e8_DisplayType__e8CAPACITIVE,
            /*DisplayLayerAttributes*/  l_DisplayLayers);

         //Sending Required Screen Configuration to be displayed
         _smartphoneFIProxy->sendSetDisplayAttributesStart(*this, l_DisplayAttributes);
      }
   }
}


/************************************************************************
*FUNCTION:		vSetAppleDisplayAttributes
*DESCRIPTION:	Set Display Attributes
*PARAMETER:     NA
*RETURNVALUE:	NA
*SWFL:          1993, 2581, 2604
*HISTORY:
*0.1     26.06.19      Bruce Netto      Initial Version
************************************************************************/
void ConnectionClientHandler::vSetAppleDisplayAttributes()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetAppleDisplayAttributes()"));
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetAppleDisplayAttributes DeviceCategory = %d", m_ActiveDeviceCategory));

   if (_smartphoneFIProxy)
   {
      if (ismySPINEnabled()) //KDS CHECK
      {
         ETG_TRACE_USR4(("ConnectionClientHandler::vSetAppleDisplayAttributes MY SPIN DeviceCategory = %d", m_ActiveDeviceCategory));
         std::vector<midw_smartphoneint_fi_types::T_DisplayLayerAttributes> l_DisplayLayers;

         midw_smartphoneint_fi_types::T_DisplayLayerAttributes l_DisplayLayerAttr_MYSPIN(
            /*DeviceCategory*/      midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN,
            /*TouchLayerID*/        3501,
            /*TouchSurfaceID*/      80,
            /*VideoLayerID*/        3500,
            /*VideoSurfaceID*/      59,
            /*LayerWidth*/          706,
            /*LayerHeight*/         480,
            /*LayerWidthinMM*/      137.2059,
            /*LayerHeightinMM*/     80.712,
            /*PixelAspectRatio*/    "1.69",
            /*DPI*/                 160,
            /*DisplayTouchType*/    midw_smartphoneint_fi_types::T_e8_DisplayTouchType__e8MULTI_TOUCH,
            /*PriorityIndex*/       0);

         l_DisplayLayers.push_back(l_DisplayLayerAttr_MYSPIN);

         midw_smartphoneint_fi_types::T_DisplayAttributes l_DisplayAttributes(
            /*ScreenHeight*/            480,
            /*ScreenWidth*/             700,
            /*ScreenHeightMm*/          186,
            /*ScreenWidthMm*/           127,
            /*DisplayType*/             midw_smartphoneint_fi_types::T_e8_DisplayType__e8CAPACITIVE,
            /*DisplayLayerAttributes*/  l_DisplayLayers);

         //Sending Required Screen Configuration to be displayed
         _smartphoneFIProxy->sendSetDisplayAttributesStart(*this, l_DisplayAttributes);
      }
      else
      {
         std::vector<midw_smartphoneint_fi_types::T_DisplayLayerAttributes> l_DisplayLayers;
         //FOR APPLE DEVICE: HINT Whats the config for android?
         midw_smartphoneint_fi_types::T_DisplayLayerAttributes l_DisplayLayerAttr_APPLE_DEVICE(
            /*DeviceCategory*/      midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO,
            /*TouchLayerID*/        3503,
            /*TouchSurfaceID*/      80,
            /*VideoLayerID*/        3502,
            /*VideoSurfaceID*/      59,
            /*LayerWidth*/          800,
            /*LayerHeight*/         480,
            /*LayerWidthinMM*/      188.3,
            /*LayerHeightinMM*/     117.9,
            /*PixelAspectRatio*/    "1.69",
            /*DPI*/                 160,
            /*DisplayTouchType*/    midw_smartphoneint_fi_types::T_e8_DisplayTouchType__e8MULTI_TOUCH,
            /*PriorityIndex*/       0);

         l_DisplayLayers.push_back(l_DisplayLayerAttr_APPLE_DEVICE);
         ETG_TRACE_USR4(("ConnectionClientHandler::vSetAppleDisplayAttributes APPLE DeviceCategory = %d", m_ActiveDeviceCategory));

         midw_smartphoneint_fi_types::T_DisplayAttributes l_DisplayAttributes(
            /*ScreenHeight*/            480,
            /*ScreenWidth*/             800,
            /*ScreenHeightMm*/          186,
            /*ScreenWidthMm*/           127,
            /*DisplayType*/             midw_smartphoneint_fi_types::T_e8_DisplayType__e8CAPACITIVE,
            /*DisplayLayerAttributes*/  l_DisplayLayers);

         //Sending Required Screen Configuration to be displayed
         _smartphoneFIProxy->sendSetDisplayAttributesStart(*this, l_DisplayAttributes);
      }
   }
}


/*********************************************************************************************
                                Property: DeviceDisplayContext
*********************************************************************************************/
/************************************************************************
*FUNCTION:		onDeviceDisplayContextStatus
*DESCRIPTION:	Function to get status of DeviceDisplayContext
*PARAMETER:     proxy, status
*RETURNVALUE:	NA
*SWFL:          2581, 2604
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onDeviceDisplayContextStatus(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::DeviceDisplayContextStatus>& status)
{
   int l_DisplayFlag = 0;
   l_DisplayFlag = status->getDisplayFlag();
   m_status = status->getDisplayContextReason();
   ETG_TRACE_USR4(("ConnectionClientHandler::onDeviceDisplayContextStatus()"));
   ETG_TRACE_USR4(("ConnectionClientHandler::onDeviceDisplayContextStatus DisplayFlag = %d", l_DisplayFlag));
   ETG_TRACE_USR4(("ConnectionClientHandler::onDeviceDisplayContextStatus hasDisplayFlag = %d", status->hasDisplayFlag()));
   ETG_TRACE_USR4(("ConnectionClientHandler::onDeviceDisplayContextStatus DisplayContext = %d", status->getDisplayContextReason()));

   if (l_DisplayFlag)
   {
      vHandleDisplayFlagTRUE();
   }

   else
   {
      vHandleDisplayFlagFALSE();
   }
}


/************************************************************************
*FUNCTION:		vHandleDisplayFlagTRUE
*DESCRIPTION:	Function to handle display context if display flag is
*               set to 1
*PARAMETER:     NA
*RETURNVALUE:	NA
*SWFL:          1993, 2581, 2604
*HISTORY:
*0.1     02.05.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vHandleDisplayFlagTRUE()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vHandleDisplayFlagTRUE()"));
   if (tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::vHandleDisplayFlagTRUE MySPIN"));
      vEnableSPILayers();
   }
   else if (tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::vHandleDisplayFlagTRUE AndroidAuto"));
      vEnableSPILayers();
   }
   else if (tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::vHandleDisplayFlagTRUE Carplay "));
      vEnableSPILayers();
   }
}


/************************************************************************
*FUNCTION:		vEnableSPILayers
*DESCRIPTION:	Function to enable SPI layers
*PARAMETER:     NA
*RETURNVALUE:	NA
*SWFL:          1993, 2581, 2604
*HISTORY:
*0.1     02.05.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vEnableSPILayers()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vEnableSPILayers"));
   // ilmErrorTypes l_error = ilm_layerSetVisibility(3500, 1);
   // l_error = ilm_layerSetVisibility(3501, 1);
   // l_error = ilm_surfaceSetVisibility(59, 1);
   // l_error = ilm_surfaceSetVisibility(80, 1);

   // if (ILM_SUCCESS == l_error)
   // {
   // l_error = ilm_commitChanges();
   // if (ILM_SUCCESS == l_error)
   // {
   // ETG_TRACE_USR4(("ConnectionClientHandler::vEnableSPILayers Layers and surfaces enabled"));
   // }

   // else
   // {
   // ETG_TRACE_USR4(("ConnectionClientHandler::vEnableSPILayers Error in enabling layers and surfaces"));
   // }
   // }

   // else
   // {
   // ETG_TRACE_USR4(("ConnectionClientHandler::vEnableSPILayers Error in enabling layers and surfaces"));
   // }
   if (tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN)
   {
      ilmErrorTypes l_error = ilm_layerSetVisibility(3500, 1);
      l_error = ilm_layerSetVisibility(3501, 1);
      l_error = ilm_surfaceSetVisibility(59, 1);
      l_error = ilm_surfaceSetVisibility(80, 1);
      if (ILM_SUCCESS == l_error)
      {
         l_error = ilm_commitChanges();
         if (ILM_SUCCESS == l_error)
         {
            ETG_TRACE_USR4(("ConnectionClientHandler::vEnableSPILayers Layers and surfaces enabled"));
         }
         else
         {
            ETG_TRACE_USR4(("ConnectionClientHandler::vEnableSPILayers Error in enabling layers and surfaces"));
         }
      }
   }
   else if (tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO ||
            tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO)
   {
      ilmErrorTypes l_error1 = ilm_layerSetVisibility(3502, 1);
      l_error1 = ilm_layerSetVisibility(3503, 1);
      l_error1 = ilm_surfaceSetVisibility(59, 1);
      l_error1 = ilm_surfaceSetVisibility(80, 1);
      if (ILM_SUCCESS == l_error1)
      {
         l_error1 = ilm_commitChanges();
         if (ILM_SUCCESS == l_error1)
         {
            ETG_TRACE_USR4(("ConnectionClientHandler::vEnableSPILayers Layers and surfaces enabled"));
         }
         else
         {
            ETG_TRACE_USR4(("ConnectionClientHandler::vEnableSPILayers Error in enabling layers and surfaces"));
         }
      }
   }
}


/************************************************************************
*FUNCTION:		vHandleDisplayFlagFALSE
**DESCRIPTION:	Function to handle display context if display flag is
*               set to 0
*PARAMETER:     NA
*RETURNVALUE:	NA
*SWFL:          2581, 2604
*HISTORY:
*0.1     02.05.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vHandleDisplayFlagFALSE()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vHandleDisplayFlagFALSE()"));
   mSleepContextFlag = SystemPowerModeClientHandler::getInstance()->getSleepContextFlag();
   if (tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::vHandleDisplayFlagFALSE MySPIN"));
      vDisableSPILayers();
   }
   else  if (tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::vHandleDisplayFlagFALSE AndroidAuto"));
      vDisableSPILayers();
      //durng sleep mode context switch should not be sent as display flag is false when system enters to sleep mode
      if ((m_status == ::midw_smartphoneint_fi_types::T_e8_DisplayContextReason__DISPLAY_CONTEXT_REASON_UNKNOWN) && (mSleepContextFlag == false)) //Bug = 559968
      {
         ETG_TRACE_USR4(("ConnectionClientHandler::vHandleDisplayFlagFALSE DisplayContextReason"));
         ContextSwitchClientHandler::poGetInstance()->requestContextSwitchApplication((uint8)2
               , (uint8)APPID_APPHMI_SPI
               , (APPLICATIONSWITCH_TYPES::enActivityIDs)ContextSwitchClientHandler::poGetInstance()->getActivityIDforCurrentView()
               , (uint8)APPID_APPHMI_MASTER
               , (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_HOME);
      }
   }
   else if (tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::vHandleDisplayFlagFALSE DIPO"));
      vDisableSPILayers();
   }

   ETG_TRACE_USR4(("ConnectionClientHandler::vHandleDisplayFlagFALSE HMI has taken the Video Focus"));
}


/************************************************************************
*FUNCTION:		vDisableSPILayers
*DESCRIPTION:	Function to disable SPI layers
*PARAMETER:     NA
*RETURNVALUE:	NA
*SWFL:          2581, 2583, 2584, 2604
*HISTORY:
*0.1     02.05.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vDisableSPILayers()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vDisableSPILayers"));
   // ilmErrorTypes l_error;
   // l_error = ilm_layerSetVisibility(3500, 0);
   // l_error = ilm_layerSetVisibility(3501, 0);
   // l_error = ilm_surfaceSetVisibility(59, 0);
   // l_error = ilm_surfaceSetVisibility(80, 0);

   // if (ILM_SUCCESS == l_error)
   // {
   // l_error = ilm_commitChanges();
   // if (ILM_SUCCESS == l_error)
   // {
   // ETG_TRACE_USR4(("ConnectionClientHandler::vDisableSPILayers Layers and surfaces disabled"));
   // }

   // else
   // {
   // ETG_TRACE_USR4(("ConnectionClientHandler::vDisableSPILayers Error in disabling layers and surfaces"));
   // }
   // }

   // else
   // {
   // ETG_TRACE_USR4(("ConnectionClientHandler::vDisableSPILayers Error in disabling layers and surfaces"));
   // }
   if (tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN)
   {
      ilmErrorTypes l_error = ilm_layerSetVisibility(3500, 0);
      l_error = ilm_layerSetVisibility(3501, 0);
      l_error = ilm_surfaceSetVisibility(59, 0);
      l_error = ilm_surfaceSetVisibility(80, 0);
      if (ILM_SUCCESS == l_error)
      {
         l_error = ilm_commitChanges();
         if (ILM_SUCCESS == l_error)
         {
            ETG_TRACE_USR4(("ConnectionClientHandler::vDisableSPILayers Layers and surfaces disabled"));
         }
         else
         {
            ETG_TRACE_USR4(("ConnectionClientHandler::vDisableSPILayers Error in disabled layers and surfaces"));
         }
      }
   }
   else if (tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO ||
            tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO)
   {
      ilmErrorTypes l_error1 = ilm_layerSetVisibility(3502, 0);
      l_error1 = ilm_layerSetVisibility(3503, 0);
      l_error1 = ilm_surfaceSetVisibility(59, 0);
      l_error1 = ilm_surfaceSetVisibility(80, 0);
      if (ILM_SUCCESS == l_error1)
      {
         l_error1 = ilm_commitChanges();
         if (ILM_SUCCESS == l_error1)
         {
            ETG_TRACE_USR4(("ConnectionClientHandler::vDisableSPILayers Layers and surfaces disabled"));
         }
         else
         {
            ETG_TRACE_USR4(("ConnectionClientHandler::vDisableSPILayers Error in disabled layers and surfaces"));
         }
      }
   }
}


/************************************************************************
*FUNCTION:		onDeviceDisplayContextError
*DESCRIPTION:	DeviceDisplayContext with error
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          2581, 2604
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onDeviceDisplayContextError(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::DeviceDisplayContextError>& error)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onDeviceDisplayContextError()"));
}


/*********************************************************************************************
                                Property: SessionStatusInfo
*********************************************************************************************/
/*****************************************************************************
*FUNCTION:		onSessionStatusInfoStatus
*DESCRIPTION:	Function to get status of SessionStatusInfo
*PARAMETER:     proxy, status
*RETURNVALUE:	NA
*SWFL:          2581, 2604, 2608, 2609, 2607, 2611, //
*               2445, 2506, 2505, 2503, 2502, 2501
*HISTORY:
*0.1     20.04.18      Athira Sreekumar         Initial Version
*0.2     18.12.18      Bruce Mario Savio Netto  SPI BT Pairing & SPI Retry Popup
*0.3     26.06.19      Bruce Netto      DTR Changes for All three trchnologies added
********************************************************************************/
void ConnectionClientHandler::onSessionStatusInfoStatus(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::SessionStatusInfoStatus>& status)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus()"));
   m_HandSetInteraction = status->getHandsetInteractionStatus();

   ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus() %d", m_HandSetInteraction));
   //vParkedConditionPopup();

   int deviceHandle = status->getDeviceHandle();
   if (_smartphoneFIProxy)
   {
      ::midw_smartphoneint_fi_types::T_e8_SessionStatus l_SessionStatus = status->getSessionStatus();
      ::midw_smartphoneint_fi_types::T_e8_DeviceCategory l_DeviceCategory = status->getDeviceCategory();

      ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus l_SessionStatus = %d", l_SessionStatus));
      vSetSessionStatus(l_SessionStatus);

      switch (l_SessionStatus)
      {
         case ::midw_smartphoneint_fi_types::T_e8_SessionStatus__SESSION_INACTIVE:
         {
            ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus SESSION_INACTIVE"));
            //TODO: Send session details to MySPIN
            //bruce
            if ((midw_smartphoneint_fi_types::T_e8_DeviceStatusInfo__DEVICE_REMOVED == m_DeviceStatus) ||
                  (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_UNKNOWN) ||
                  (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN) ||
                  (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO) ||
                  (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO) ||
                  (m_DeviceConnectionStatus == ::midw_smartphoneint_fi_types::T_e8_DeviceConnectionStatus__DEV_NOT_CONNECTED))
            {
               ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus SESSION_INACTIVE Databindings Update Start"));
               (*_SPIGadgetInfo).mDeviceConnection = DTR_NO_CONNECTION;
               _SPIGadgetInfo.MarkAllItemsModified();
               _SPIGadgetInfo.SendUpdate();
               ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus SESSION_INACTIVE Databindings Update End"));
            }
            break;
         }

         case ::midw_smartphoneint_fi_types::T_e8_SessionStatus__SESSION_ACTIVE:
         {
            ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus SESSION_ACTIVE"));
            //TODO: Send session details to MySPIN
            //TODO: BT popup here

            POST_MSG((COURIER_MESSAGE_NEW(SPI_Retry_Popup_Close)()));
            ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus() SPI_Retry_Popup_Close sent"));

            if (m_CurrentView == VIEW_PHONE_NOT_CONNECTED)
            {
               //Bruce
               if (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN) /* OR Based on SPI KDS*/
               {
                  POST_MSG((COURIER_MESSAGE_NEW(SPIView_mySPIN_AppView)()));
               }
               else
               {
                  POST_MSG((COURIER_MESSAGE_NEW(SPIView_AAPorCP_AppView)()));
                  ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus() phone not connected scene"));
                  DisplaySourceClientHandler::getInstance(ConnectionClientHandler::getInstance())->vsendCurrentAppID(::midw_smartphoneint_fi_types::T_e8_DisplayContext__DISPLAY_CONTEXT_PROJECTION, true);	 //Certification requirement
               }
            }
            //Bug 644654
            vCheckForeGroundNavi();

            //Bruce
            if (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN)
            {
               ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus SESSION_ACTIVE Databindings Update Start"));
               (*_SPIGadgetInfo).mDeviceConnection = DTR_MYSPIN_CONNECTED;
               _SPIGadgetInfo.MarkAllItemsModified();
               _SPIGadgetInfo.SendUpdate();
               ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus SESSION_ACTIVE Databindings Update End"));
            }
            else if (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO)
            {
               //vParkedConditionPopup();
               ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus Android SESSION_ACTIVE Databindings Update Start"));
               (*_SPIGadgetInfo).mDeviceConnection = DTR_ANDROIDAUTO_CONNECTED;
               _SPIGadgetInfo.MarkAllItemsModified();
               _SPIGadgetInfo.SendUpdate();
               ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus Android SESSION_ACTIVE Databindings Update End"));
            }
            else if (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO)
            {
               ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus APPLE SESSION_ACTIVE Databindings Update Start"));
               (*_SPIGadgetInfo).mDeviceConnection = DTR_CARPLAY_CONNECTED;
               _SPIGadgetInfo.MarkAllItemsModified();
               _SPIGadgetInfo.SendUpdate();
               VehicleDataSpiClientHandler::getInstance()->setFeatureRestrictions_Carplay();
               ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus APPLE SESSION_ACTIVE Databindings Update End"));
            }
            ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus SESSION_ACTIVE Function Call"));
            //VehicleDataSpiClientHandler::getInstance()->setFeatureRestrictionsAAP();
            break;
         }

         case ::midw_smartphoneint_fi_types::T_e8_SessionStatus__SESSION_SUSPENDED:
         {
            ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus SESSION_SUSPENDED"));
            //TODO: Send session details to MySPIN
            break;
         }

         case ::midw_smartphoneint_fi_types::T_e8_SessionStatus__SESSION_ERROR:
         {
            //The session status is not in scope of MySPIN
            ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus SESSION_ERROR"));
            break;
         }

         case ::midw_smartphoneint_fi_types::T_e8_SessionStatus__SESSION_SUSPENDED_APP_TRANSITION:
         {
            ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus SESSION_SUSPENDED_APP_TRANSITION"));
            //Bruce
            if (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN)
            {
               POST_MSG((COURIER_MESSAGE_NEW(SPI_Retry_Popup)()));
            }
            //TODO: Send session details to MySPIN
            break;
         }

         default:
            break;
      }
   }
   ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus should enter? %d", m_DeviceAlreadyConnected));
   if ((m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO) && ((m_DeviceAlreadyConnected == false) || (prevDeviceHandle !=  m_DeviceHandle)) && (m_HandSetInteraction == ::midw_smartphoneint_fi_types::T_e8_HandsetInteraction__HANDSET_INTERACTION_REQUIRED))
   {
      prevDeviceHandle = deviceHandle;
      ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus Popup Show "));
      m_DeviceAlreadyConnected = true;
      //POST_MSG((COURIER_MESSAGE_NEW(SPI_FirstDeviceConnection)()));
   }
}


/************************************************************************
*FUNCTION:		vCheckForeGroundNavi
*DESCRIPTION:	Check if the fore ground application is navigation or not.
				When display attributes for mySPIN was set at startup,
				it caused overlap of myspin projection view on top of
				navigation map view. Thus, accessory display context is send to
				MW again to resolve the issue.
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*Bug : 			644654
*HISTORY:
*0.1     12.02.2020      Shivani Priya       Initial Version
************************************************************************/
void ConnectionClientHandler::vCheckForeGroundNavi()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vCheckForeGroundNavi()"));
   if ((tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN) &&
         DisplaySourceClientHandler::getInstance(ConnectionClientHandler::getInstance())->getCurrentAppID() == APPID_APPHMI_NAVIGATION)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoStatus SESSION_ACTIVE and Foreground App is Navigation"));
      DisplaySourceClientHandler::getInstance(ConnectionClientHandler::getInstance())->vsetAccessoryDisplayContext(APPID_APPHMI_NAVIGATION);
   }
}


/************************************************************************
*FUNCTION:		vParkedConditionPopup
*DESCRIPTION:	SessionStatusInfo with error
*PARAMETER:     NA
*RETURNVALUE:	Void
************************************************************************/
void ConnectionClientHandler::vParkedConditionPopup()
{
   mParkBrakeInfo = VehicleDataSpiClientHandler::getInstance()->getParkBrakeInfo();
   //VehicleDataSpiClientHandler::getInstance()->getParkBrakeInfo();
   if ((m_HandSetInteraction == ::midw_smartphoneint_fi_types::T_e8_HandsetInteraction__HANDSET_INTERACTION_REQUIRED) && (mParkBrakeInfo == midw_smartphoneint_fi_types::T_e8_ParkBrake__PARK_BRAKE_NOT_ENGAGED))
   {
      POST_MSG((COURIER_MESSAGE_NEW(SPI_ParkedCondition_Popup)()));
   }
}


/************************************************************************
*FUNCTION:		onSessionStatusInfoError
*DESCRIPTION:	SessionStatusInfo with error
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          2581, 2604
*HISTORY:
*0.1     20.04.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onSessionStatusInfoError(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::SessionStatusInfoError>& error)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onSessionStatusInfoError()"));
}


/*********************************************************************************************
                                Method: LaunchApp
*********************************************************************************************/
/************************************************************************
*FUNCTION:		onLaunchAppResult
*DESCRIPTION:	Function to get result of LaunchApp
*PARAMETER:     proxy, result
*RETURNVALUE:	NA
*SWFL:          2581, 2604
*HISTORY:
*0.1     04.05.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onLaunchAppResult(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy, const ::boost::shared_ptr<midw_smartphoneint_fi::LaunchAppResult>& result)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onLaunchAppResult()"));
   ETG_TRACE_USR4(("ConnectionClientHandler::onLaunchAppResult() DeviceHandle = %d", result->getDeviceHandle()));
   ETG_TRACE_USR4(("ConnectionClientHandler::onLaunchAppResult() AppHandle = %d", result->getAppHandle()));
   ETG_TRACE_USR4(("ConnectionClientHandler::onLaunchAppResult() DiPOAppType = %d", result->getDiPOAppType()));
   ETG_TRACE_USR4(("ConnectionClientHandler::onLaunchAppResult() ResponseCode = %d", result->getResponseCode()));
   ETG_TRACE_USR4(("ConnectionClientHandler::onLaunchAppResult() ErrorCode = %d", result->getErrorCode()));
}


/************************************************************************
*FUNCTION:		onLaunchAppError
*DESCRIPTION:	Function to get result of LaunchApp
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          2581, 2604
*HISTORY:
*0.1     04.05.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onLaunchAppError(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy, const ::boost::shared_ptr<midw_smartphoneint_fi::LaunchAppError>& error)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onLaunchAppError()"));
}


/************************************************************************
*FUNCTION:		vLaunchApp
*DESCRIPTION:	Function to Launch App
*PARAMETER:     proxy, result
*RETURNVALUE:	NA
*SWFL:          2581, 2604
*HISTORY:
*0.1     04.05.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vLaunchApp(uint32 deviceHandle,
      ::midw_smartphoneint_fi_types::T_e8_DeviceCategory deviceCategory,
      ::midw_smartphoneint_fi_types::T_e8_DiPOAppType DiPOAppType)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vLaunchApp()"));

   ETG_TRACE_USR4(("ConnectionClientHandler::vLaunchApp DeviceHandle = %d", deviceHandle));
   ETG_TRACE_USR4(("ConnectionClientHandler::vLaunchApp DeviceCategory = %d", deviceCategory));
   ETG_TRACE_USR4(("ConnectionClientHandler::vLaunchApp AppHandle = %d", UNDEFINED));
   ETG_TRACE_USR4(("ConnectionClientHandler::vLaunchApp DiPOAppType = %d", DiPOAppType));
   ETG_TRACE_USR4(("ConnectionClientHandler::vLaunchApp TelephoneNumber = %s", ""));
   ETG_TRACE_USR4(("ConnectionClientHandler::vLaunchApp EcnrSetting = %d", ::midw_smartphoneint_fi_types::T_e8_EcnrSetting__ECNR_NOCHANGE));

   _smartphoneFIProxy->sendLaunchAppStart(*this,
                                          deviceHandle,
                                          deviceCategory,
                                          UNDEFINED,
                                          DiPOAppType,
                                          "",
                                          ::midw_smartphoneint_fi_types::T_e8_EcnrSetting__ECNR_NOCHANGE);
}


/************************************************************************
*FUNCTION:		onCourierMessage
*DESCRIPTION:	Function to Launch mySPIN App
*PARAMETER:     LaunchMySpinAppMsg
*RETURNVALUE:	bool
*SWFL:          2581, 2604
*HISTORY:
*0.1     04.05.18      Athira Sreekumar       Initial Version
************************************************************************/
bool ConnectionClientHandler::onCourierMessage(const LaunchMySpinAppMsg& oMsg)
{
   ETG_TRACE_USR4(("Application launched"));

   ETG_TRACE_USR4(("ConnectionClientHandler::LaunchMySpinAppMsg()"));

   if (bGetActiveDeviceHandleValidity())
   {
      vLaunchApp(tGetActiveDeviceHandle(),
                 ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN,
                 ::midw_smartphoneint_fi_types::T_e8_DiPOAppType__DIPO_NOT_USED);
      return true;
   }
   return false;
}


/************************************************************************
*FUNCTION:		onCourierMessage
*DESCRIPTION:	Function to Launch AAP or CP App from
*              State machine S_Fullscreen_view
*PARAMETER:     LaunchMySpinAppMsg
*RETURNVALUE:	bool
*SWFL:          2581, 2604
*HISTORY:
*0.1     19.06.19      Bruce Netto       Initial Version
************************************************************************/
//Bruce
bool ConnectionClientHandler::onCourierMessage(const LaunchAAPorCPAppMsg& oMsg)
{
   ETG_TRACE_USR4(("Application launched via LaunchAAPorCPAppMsg"));

   ETG_TRACE_USR4(("ConnectionClientHandler::LaunchAAPorCPAppMsg()"));
   ::midw_smartphoneint_fi_types::T_e8_DeviceCategory category = (::midw_smartphoneint_fi_types::T_e8_DeviceCategory) tGetActiveDeviceCategory();

   if (bGetActiveDeviceHandleValidity() &&
         (category == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO ||
          category == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO))
   {
      ETG_TRACE_USR4(("LaunchAAPorCPAppMsg: Launching"));
      vLaunchApp(tGetActiveDeviceHandle(),
                 category,
                 ::midw_smartphoneint_fi_types::T_e8_DiPOAppType__DIPO_NOT_USED);
      return true;
   }
   return false;
}


/*********************************************************************************************
                                Method: SetDeviceUsagePreference
*********************************************************************************************/
/************************************************************************
*FUNCTION:		onSetDeviceUsagePreferenceResult
*DESCRIPTION:	Function to get result of SetDeviceUsagePreference
*PARAMETER:     proxy, result
*RETURNVALUE:	void
*SWFL:          1993, 1994, 2581, 2604
*HISTORY:
*0.1     30.05.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onSetDeviceUsagePreferenceResult(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy, const ::boost::shared_ptr<midw_smartphoneint_fi::SetDeviceUsagePreferenceResult>& result)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onSetDeviceUsagePreferenceResult"));
   if (proxy == _smartphoneFIProxy)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::onSetDeviceUsagePreferenceResult DeviceHandle = %d", result->getDeviceHandle()));
      ETG_TRACE_USR4(("ConnectionClientHandler::onSetDeviceUsagePreferenceResult ResponseCode = %d", result->getResponseCode()));
      ETG_TRACE_USR4(("ConnectionClientHandler::onSetDeviceUsagePreferenceResult ErrorCode = %d", result->getErrorCode()));
   }
}


/************************************************************************
*FUNCTION:		onSetDeviceUsagePreferenceError
*DESCRIPTION:	Function to get result of LaunchApp
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          1993, 1994, 2581, 2604
*HISTORY:
*0.1     30.05.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onSetDeviceUsagePreferenceError(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy, const ::boost::shared_ptr<midw_smartphoneint_fi::SetDeviceUsagePreferenceError>& error)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onSetDeviceUsagePreferenceError"));
}


/************************************************************************
*FUNCTION:		vSetDeviceUsagePreference
*DESCRIPTION:	Function to set device usage preference
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          1993, 1994, 2581, 2604
*HISTORY:
*0.1     30.05.18      Athira Sreekumar       Initial Version
************************************************************************/
// void ConnectionClientHandler::vSetDeviceUsagePreference(bool enableFlag /*, midw_smartphoneint_fi_types::T_e8_DeviceCategory deviceCategory*/)
// {
//    ETG_TRACE_USR4(("ConnectionClientHandler::vSetDeviceUsagePreference"));
//    if (_smartphoneFIProxy)
//    {
//       if (ismySPINEnabled())
//       {
//          if (enableFlag)
//          {
//             _smartphoneFIProxy->sendSetDeviceUsagePreferenceStart(*this,
//                0xFFFFFFFF,
//                ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_UNKNOWN,
//                ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_ENABLED);
//                ETG_TRACE_USR4(("ConnectionClientHandler::sendSetDeviceUsagePreferenceStart MySPIN enableFlag = %d", enableFlag));
//          }
//          else
//          {
//             _smartphoneFIProxy->sendSetDeviceUsagePreferenceStart(*this,
//                0xFFFFFFFF,
//                ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_UNKNOWN,
//                ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_DISABLED);
//                ETG_TRACE_USR4(("ConnectionClientHandler::sendSetDeviceUsagePreferenceStart MYSPIN enableFlag = %d", enableFlag));
//          }
//       }
//       //Bruce
//       //Receiving the Category of any technology and passing it
//       else if (isAndroidAutoEnabled() || isCarPlayEnabled())
//       {
//          if(isAndroidAutoEnabled())
//          {
//             if (enableFlag)
//             {
//                _smartphoneFIProxy->sendSetDeviceUsagePreferenceStart(*this,
//                      0xFFFFFFFF,
//                      ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO,
//                      ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_ENABLED);

//                ETG_TRACE_USR4(("ConnectionClientHandler::sendSetDeviceUsagePreferenceStart AAP enableFlag = %d", enableFlag));
//             }
//             else
//             {
//                _smartphoneFIProxy->sendSetDeviceUsagePreferenceStart(*this,
//                      0xFFFFFFFF,
//                      ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO,
//                      ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_DISABLED);

//                ETG_TRACE_USR4(("ConnectionClientHandler::sendSetDeviceUsagePreferenceStart AAP enableFlag = %d", enableFlag));
//                ETG_TRACE_USR4(("ConnectionClientHandler::sendSetDeviceUsagePreferenceStart Device Usage Preference Unknown"));
//             }
//          }
//          else if(isCarPlayEnabled())
//          {
//             if (enableFlag)
//             {
//                _smartphoneFIProxy->sendSetDeviceUsagePreferenceStart(*this,
//                   0xFFFFFFFF,
//                   ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO,
//                   ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_ENABLED);

//                ETG_TRACE_USR4(("ConnectionClientHandler::sendSetDeviceUsagePreferenceStart CP enableFlag = %d", enableFlag));
//             }
//             else
//             {
//                _smartphoneFIProxy->sendSetDeviceUsagePreferenceStart(*this,
//                   0xFFFFFFFF,
//                   ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO,
//                   ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_DISABLED);

//                ETG_TRACE_USR4(("ConnectionClientHandler::sendSetDeviceUsagePreferenceStart CP enableFlag = %d", enableFlag));
//                ETG_TRACE_USR4(("ConnectionClientHandler::sendSetDeviceUsagePreferenceStart Device Usage Preference Unknown"));
//             }
//          }
//       }
//    }
// }

/************************************************************************
*FUNCTION:		vSetAndroidDeviceUsagePreference
*DESCRIPTION:	Function to set device usage preference
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          1993, 1994, 2581, 2604
*HISTORY:
*0.1     26.06.19      Bruce Netto      Initial Version
************************************************************************/
void ConnectionClientHandler::vSetAndroidDeviceUsagePreference(bool enableFlag /*, midw_smartphoneint_fi_types::T_e8_DeviceCategory deviceCategory*/)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetAndroidDeviceUsagePreference"));
   if (_smartphoneFIProxy)
   {
      if (ismySPINEnabled())
      {
         if (enableFlag)
         {
            _smartphoneFIProxy->sendSetDeviceUsagePreferenceStart(*this,
                  0xFFFFFFFF,
                  ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_UNKNOWN,
                  ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_ENABLED);
            ETG_TRACE_USR4(("ConnectionClientHandler::sendSetAndroidDeviceUsagePreferenceStart MySPIN enableFlag = %d", enableFlag));
         }
         else
         {
            _smartphoneFIProxy->sendSetDeviceUsagePreferenceStart(*this,
                  0xFFFFFFFF,
                  ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_UNKNOWN,
                  ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_DISABLED);
            ETG_TRACE_USR4(("ConnectionClientHandler::sendSetAndroidDeviceUsagePreferenceStart MYSPIN enableFlag = %d", enableFlag));
         }
      }
      //Bruce
      //Receiving the Category of any technology and passing it
      else
      {
         if (enableFlag)
         {
            _smartphoneFIProxy->sendSetDeviceUsagePreferenceStart(*this,
                  0xFFFFFFFF,
                  ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO,
                  ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_ENABLED);

            ETG_TRACE_USR4(("ConnectionClientHandler::sendSetAndroidDeviceUsagePreferenceStart AAP enableFlag = %d", enableFlag));
         }
         else
         {
            _smartphoneFIProxy->sendSetDeviceUsagePreferenceStart(*this,
                  0xFFFFFFFF,
                  ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO,
                  ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_DISABLED);

            ETG_TRACE_USR4(("ConnectionClientHandler::sendSetAndroidDeviceUsagePreferenceStart AAP enableFlag = %d", enableFlag));
            ETG_TRACE_USR4(("ConnectionClientHandler::sendSetAndroidDeviceUsagePreferenceStart Device Usage Preference Unknown"));
         }
      }
   }
}


/************************************************************************
*FUNCTION:		vSetAppleDeviceUsagePreference
*DESCRIPTION:	Function to set device usage preference
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          1993, 1994, 2581, 2604
*HISTORY:
*0.1     26.06.19      Bruce Netto      Initial Version
************************************************************************/
void ConnectionClientHandler::vSetAppleDeviceUsagePreference(bool enableFlag /*, midw_smartphoneint_fi_types::T_e8_DeviceCategory deviceCategory*/)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetAppleDeviceUsagePreference"));
   if (_smartphoneFIProxy)
   {
      if (ismySPINEnabled())
      {
         if (enableFlag)
         {
            _smartphoneFIProxy->sendSetDeviceUsagePreferenceStart(*this,
                  0xFFFFFFFF,
                  ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_UNKNOWN,
                  ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_ENABLED);
            ETG_TRACE_USR4(("ConnectionClientHandler::sendSetAppleDeviceUsagePreferenceStart MySPIN enableFlag = %d", enableFlag));
         }
         else
         {
            _smartphoneFIProxy->sendSetDeviceUsagePreferenceStart(*this,
                  0xFFFFFFFF,
                  ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_UNKNOWN,
                  ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_DISABLED);
            ETG_TRACE_USR4(("ConnectionClientHandler::sendSetAppleDeviceUsagePreferenceStart MYSPIN enableFlag = %d", enableFlag));
         }
      }
      //Bruce
      //Receiving the Category of any technology and passing it
      else
      {
         if (enableFlag)
         {
            _smartphoneFIProxy->sendSetDeviceUsagePreferenceStart(*this,
                  0xFFFFFFFF,
                  ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO,
                  ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_ENABLED);
            ETG_TRACE_USR4(("ConnectionClientHandler::sendSetAppleDeviceUsagePreferenceStart CP enableFlag = %d", enableFlag));
         }
         else
         {
            _smartphoneFIProxy->sendSetDeviceUsagePreferenceStart(*this,
                  0xFFFFFFFF,
                  ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO,
                  ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_DISABLED);

            ETG_TRACE_USR4(("ConnectionClientHandler::sendSetAppleDeviceUsagePreferenceStart CP enableFlag = %d", enableFlag));
            ETG_TRACE_USR4(("ConnectionClientHandler::sendSetAppleDeviceUsagePreferenceStart Device Usage Preference Unknown"));
         }
      }
   }
}


/*********************************************************************************************
                                Method: GetDeviceUsagePreference
*********************************************************************************************/
/************************************************************************
*FUNCTION:		onGetDeviceUsagePreferenceResult
*DESCRIPTION:	Function to get result of GetDeviceUsagePreference
*PARAMETER:     proxy, result
*RETURNVALUE:	void
*SWFL:          NA
*HISTORY:
*0.1     30.05.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onGetDeviceUsagePreferenceResult(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::GetDeviceUsagePreferenceResult>& result)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onGetDeviceUsagePreferenceResult"));
   ETG_TRACE_USR4(("ConnectionClientHandler::onGetDeviceUsagePreferenceResult DeviceHandle = %d", result->getDeviceHandle()));
   ETG_TRACE_USR4(("ConnectionClientHandler::onGetDeviceUsagePreferenceResult DeviceCategory = %d", result->getDeviceCategory()));
   ETG_TRACE_USR4(("ConnectionClientHandler::onGetDeviceUsagePreferenceResult EnabledInfo = %d", result->getEnabledInfo()));

   if (_smartphoneFIProxy == proxy)
   {
      //TODO: Logic not added
      switch (result->getEnabledInfo())
      {
         case ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_DISABLED:
         {
            ETG_TRACE_USR4(("ConnectionClientHandler::onGetDeviceUsagePreferenceResult EnabledInfo__USAGE_DISABLED"));
         }
         break;

         case ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_ENABLED:
         {
            ETG_TRACE_USR4(("ConnectionClientHandler::onGetDeviceUsagePreferenceResult EnabledInfo__USAGE_ENABLED"));
         }
         break;

         case ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_CONF_REQD:
         {
            ETG_TRACE_USR4(("ConnectionClientHandler::onGetDeviceUsagePreferenceResult EnabledInfo__USAGE_CONF_REQD"));
         }
         break;

         default:
            break;
      }
   }
}


/************************************************************************
*FUNCTION:		onGetDeviceUsagePreferenceError
*DESCRIPTION:	Function to get result of GetDeviceUsagePreference
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          NA
*HISTORY:
*0.1     30.05.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onGetDeviceUsagePreferenceError(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::GetDeviceUsagePreferenceError>& error)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onGetDeviceUsagePreferenceError()"));
}


/************************************************************************
*FUNCTION:		vGetDeviceUsagePreference
*DESCRIPTION:	Function to get device usage preference
*PARAMETER:     NA
*RETURNVALUE:	NA
*SWFL:          NA
*HISTORY:
*0.1     30.05.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vGetDeviceUsagePreference()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vGetDeviceUsagePreference"));
   if (_smartphoneFIProxy)
   {
      _smartphoneFIProxy->sendGetDeviceUsagePreferenceStart(*this,
            0xFFFFFFFF,
            (::midw_smartphoneint_fi_types::T_e8_DeviceCategory)tGetActiveDeviceCategory());
      ETG_TRACE_USR4(("ConnectionClientHandler::sendGetDeviceUsagePreferenceStart()"));
   }
}


/*********************************************************************************************
                                Method: SetTechnologyPreference
*********************************************************************************************/
/************************************************************************
*FUNCTION:		onSetTechnologyPreferenceResult
*DESCRIPTION:	Function to get result of SetTechnologyPreference
*PARAMETER:     proxy, result
*RETURNVALUE:	void
*SWFL:          1993, 1994, 2581, 2604
*HISTORY:
*0.1     26.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onSetTechnologyPreferenceResult(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::SetTechnologyPreferenceResult>& result)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onSetTechnologyPreferenceResult"));
   ETG_TRACE_USR4(("ConnectionClientHandler::onSetTechnologyPreferenceResult ResponseCode = %d", result->getResponseCode()));
}


/************************************************************************
*FUNCTION:		onSetTechnologyPreferenceError
*DESCRIPTION:	Function to get error from SetTechnologyPreference
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          1993, 1994, 2581, 2604
*HISTORY:
*0.1     26.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onSetTechnologyPreferenceError(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::SetTechnologyPreferenceError>& error)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onSetTechnologyPreferenceError"));
}


/************************************************************************
*FUNCTION:		vSetTechnologyPreference
*DESCRIPTION:	Function to set technology preference
*PARAMETER:     NA
*RETURNVALUE:	NA
*SWFL:          1993, 1994, 2581, 2604
*HISTORY:
*0.1     26.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vSetTechnologyPreference()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetTechnologyPreference"));
   //Bruce
   // if(ismySPINEnabled())/*KDS CHECK*/
   // {
   //    ETG_TRACE_USR4(("ConnectionClientHandler::vSetTechnologyPreference MYSPIN"));
   //    ::std::vector< ::midw_smartphoneint_fi_types::T_e8_DeviceCategory > l_PreferenceOrderList;
   //    ::midw_smartphoneint_fi_types::T_e8_DeviceCategory l_DeviceCategory = ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN;
   //    l_PreferenceOrderList.push_back(l_DeviceCategory);

   //    uint32 l_PreferenceOrderListSize = 1;

   //    ::midw_smartphoneint_fi_types::T_e8_DeviceType l_DeviceType = ::midw_smartphoneint_fi_types::T_e8_DeviceType__ANDROID_DEVICE;

   //    _smartphoneFIProxy->sendSetTechnologyPreferenceStart(*this,
   //          0xFFFFFFFF,
   //          l_PreferenceOrderListSize,
   //          l_PreferenceOrderList,
   //          l_DeviceType);
   // }
   // else if (isCarPlayEnabled())/*ANDROID KDS TECHNOLOGY NEEDED *//*APPLE KDS TECHNOLOGY NEEDED */
   // {
   //    ETG_TRACE_USR4(("ConnectionClientHandler::vSetTechnologyPreference AAP OR CP"));

   //    if(deviceDetails.getRProjectionCapability().getEnDeviceType() == midw_smartphoneint_fi_types::T_e8_DeviceType__APPLE_DEVICE)
   //    {
   //       ETG_TRACE_USR4(("ConnectionClientHandler::vSetTechnologyPreference vSetTechnologyApplePreference"));
   //       vSetTechnologyApplePreference();
   //    }
   // }
   // else if(isAndroidAutoEnabled())
   // {
   //    if (deviceDetails.getRProjectionCapability().getEnDeviceType() == midw_smartphoneint_fi_types::T_e8_DeviceType__ANDROID_DEVICE)
   //    {
   //       ETG_TRACE_USR4(("ConnectionClientHandler::vSetTechnologyPreference vSetTechnologyAndroidPreference"));
   //       vSetTechnologyAndroidPreference();
   //    }
   // }
}


/************************************************************************
*FUNCTION:		vSetTechnologyApplePreference
*DESCRIPTION:	Function to set technology preference
*PARAMETER:     NA
*RETURNVALUE:	NA
*SWFL:          1993, 1994, 2581, 2604
*HISTORY:
*0.1     26.06.19      Bruce Netto      Initial Version
************************************************************************/
void ConnectionClientHandler::vSetTechnologyApplePreference()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetTechnologyApplePreference"));

   if (isCarPlayEnabled())
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::vSetTechnologyApplePreference CP"));
      ::std::vector< ::midw_smartphoneint_fi_types::T_e8_DeviceCategory > l_PreferenceOrderList;
      ::midw_smartphoneint_fi_types::T_e8_DeviceCategory l_DeviceCategory = ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_DIPO;
      l_PreferenceOrderList.push_back(l_DeviceCategory);

      uint32 l_PreferenceOrderListSize = 1;

      ::midw_smartphoneint_fi_types::T_e8_DeviceType l_DeviceType = ::midw_smartphoneint_fi_types::T_e8_DeviceType__APPLE_DEVICE;

      _smartphoneFIProxy->sendSetTechnologyPreferenceStart(*this,
            0xFFFFFFFF,
            l_PreferenceOrderListSize,
            l_PreferenceOrderList,
            l_DeviceType);
   }
   else if (ismySPINEnabled()) /*KDS CHECK*/
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::vSetTechnologyApplePreference MYSPIN"));
      ::std::vector< ::midw_smartphoneint_fi_types::T_e8_DeviceCategory > l_PreferenceOrderList;
      ::midw_smartphoneint_fi_types::T_e8_DeviceCategory l_DeviceCategory = ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN;
      l_PreferenceOrderList.push_back(l_DeviceCategory);

      uint32 l_PreferenceOrderListSize = 1;

      ::midw_smartphoneint_fi_types::T_e8_DeviceType l_DeviceType = ::midw_smartphoneint_fi_types::T_e8_DeviceType__APPLE_DEVICE;

      _smartphoneFIProxy->sendSetTechnologyPreferenceStart(*this,
            0xFFFFFFFF,
            l_PreferenceOrderListSize,
            l_PreferenceOrderList,
            l_DeviceType);
   }
}


/************************************************************************
*FUNCTION:		vSetTechnologyAndroidPreference
*DESCRIPTION:	Function to set technology preference
*PARAMETER:     NA
*RETURNVALUE:	NA
*SWFL:          1993, 1994, 2581, 2604
*HISTORY:
*0.1     26.06.19      Bruce Netto      Initial Version
************************************************************************/
void ConnectionClientHandler::vSetTechnologyAndroidPreference()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetTechnologyAndroidPreference"));

   if (isAndroidAutoEnabled())
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::vSetTechnologyAndroidPreference AAP"));
      ::std::vector< ::midw_smartphoneint_fi_types::T_e8_DeviceCategory > l_PreferenceOrderList;
      ::midw_smartphoneint_fi_types::T_e8_DeviceCategory l_DeviceCategory = ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO;
      l_PreferenceOrderList.push_back(l_DeviceCategory);

      uint32 l_PreferenceOrderListSize = 1;

      ::midw_smartphoneint_fi_types::T_e8_DeviceType l_DeviceType = ::midw_smartphoneint_fi_types::T_e8_DeviceType__ANDROID_DEVICE;

      _smartphoneFIProxy->sendSetTechnologyPreferenceStart(*this,
            0xFFFFFFFF,
            l_PreferenceOrderListSize,
            l_PreferenceOrderList,
            l_DeviceType);
   }
   else if (ismySPINEnabled()) /*KDS CHECK*/
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::vSetTechnologyAndroidPreference MYSPIN"));
      ::std::vector< ::midw_smartphoneint_fi_types::T_e8_DeviceCategory > l_PreferenceOrderList;
      ::midw_smartphoneint_fi_types::T_e8_DeviceCategory l_DeviceCategory = ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN;
      l_PreferenceOrderList.push_back(l_DeviceCategory);

      uint32 l_PreferenceOrderListSize = 1;

      ::midw_smartphoneint_fi_types::T_e8_DeviceType l_DeviceType = ::midw_smartphoneint_fi_types::T_e8_DeviceType__ANDROID_DEVICE;

      _smartphoneFIProxy->sendSetTechnologyPreferenceStart(*this,
            0xFFFFFFFF,
            l_PreferenceOrderListSize,
            l_PreferenceOrderList,
            l_DeviceType);
   }
}


/************************************************************************
*FUNCTION:		vSetVehicleMovementParkBrake
*DESCRIPTION:	Function to set vechicle park brake condition
*PARAMETER:     NA
*RETURNVALUE:	NA
*SWFL:          1993, 1994, 2581, 2604
*HISTORY:
*0.1     26.06.19      Bruce Netto      Initial Version
************************************************************************/
void ConnectionClientHandler::vSetVehicleMovementParkBrake()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetVehicleMovementParkBrake"));
   mVehicleParkState = VehicleDataSpiClientHandler::getInstance()->getParkBrakeInfo();
   //if(mVehicleParkState == T_e8_ParkBrake__PARK_BRAKE_ENGAGED)
   //{
   //SPI_SET_VEHICLE_MOVEMENT_STATE PARK_BRAKE_APPLIED AAP_GEAR_PARK VEH_MOV_STATE_PARKED
   //}
}


/*********************************************************************************************
                                Method: GetTechnologyPreference
*********************************************************************************************/
/************************************************************************
*FUNCTION:		onGetTechnologyPreferenceResult
*DESCRIPTION:	Function to get result of SetTechnologyPreference
*PARAMETER:     proxy, result
*RETURNVALUE:	void
*SWFL:          NA
*HISTORY:
*0.1     26.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onGetTechnologyPreferenceResult(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::GetTechnologyPreferenceResult>& result)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onGetTechnologyPreferenceResult"));

   uint16 l_NumOfPreferences = result->getNumofPreferences();
   ETG_TRACE_USR4(("ConnectionClientHandler::onGetTechnologyPreferenceResult NumofPreferences = %d", l_NumOfPreferences));

   std::vector<midw_smartphoneint_fi_types::T_e8_DeviceCategory> l_TechnologyPreferenceList = result->getPreferenceOrderList();
   std::vector<midw_smartphoneint_fi_types::T_e8_DeviceCategory>::iterator l_TechnologyPreferenceListItr = l_TechnologyPreferenceList.begin();

   uint16 l_PreferenceOrder = 1;
   for (; l_TechnologyPreferenceListItr != l_TechnologyPreferenceList.end(); ++l_TechnologyPreferenceListItr, ++l_PreferenceOrder)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::onGetTechnologyPreferenceResult %d. DeviceCategory = %d", l_PreferenceOrder, (*l_TechnologyPreferenceListItr)));
   }
}


/************************************************************************
*FUNCTION:		onGetTechnologyPreferenceError
*DESCRIPTION:	Function to get error from SetTechnologyPreference
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          NA
*HISTORY:
*0.1     26.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onGetTechnologyPreferenceError(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::GetTechnologyPreferenceError>& error)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onGetTechnologyPreferenceError"));
}


/************************************************************************
*FUNCTION:		vGetTechnologyPreference
*DESCRIPTION:	Function to get technology preference
*PARAMETER:     deviceHandle, deviceType
*RETURNVALUE:	NA
*SWFL:          NA
*HISTORY:
*0.1     26.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vGetTechnologyPreference(uint32 deviceHandle, ::midw_smartphoneint_fi_types::T_e8_DeviceType deviceType)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vGetTechnologyPreference"));
   ETG_TRACE_USR4(("ConnectionClientHandler::vGetTechnologyPreference deviceHandle = %d", deviceHandle));
   ETG_TRACE_USR4(("ConnectionClientHandler::vGetTechnologyPreference deviceType = %d", deviceType));
   _smartphoneFIProxy->sendGetTechnologyPreferenceStart(*this,
         deviceHandle,
         deviceType);
}


/*********************************************************************************************
                                Property: ProjectionDeviceAuthorization
*********************************************************************************************/
/************************************************************************
*FUNCTION:		onProjectionDeviceAuthorizationStatus
*DESCRIPTION:	Function to get status of device authorization
*PARAMETER:     proxy, status
*RETURNVALUE:	NA
*SWFL:          NA
*HISTORY:
*0.1     26.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onProjectionDeviceAuthorizationStatus(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::ProjectionDeviceAuthorizationStatus>& status)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onProjectionDeviceAuthorizationStatus"));
   uint16 l_numDevices = status->getNumDevices();
   ETG_TRACE_USR4(("ConnectionClientHandler::onProjectionDeviceAuthorizationStatus Number of devices= %d" , l_numDevices));
   UserAuthorizationDetails l_UserAuthorizationDetails = {0,
                                                          ::midw_smartphoneint_fi_types::T_e8_DeviceType__DEVICE_TYPE_UNKNOWN,
                                                          ::midw_smartphoneint_fi_types::T_e8_UserAuthorizationStatus__USER_AUTH_UNKNOWN
                                                         };

   std::vector<midw_smartphoneint_fi_types::T_DeviceAuthInfo> l_DeviceAuthInfo = status->getDeviceAuthInfoList();
   std::vector<midw_smartphoneint_fi_types::T_DeviceAuthInfo>::const_iterator l_DeviceAuthInfoIterator = l_DeviceAuthInfo.begin();

   uint16 l_DeviceCount = 1;
   for (; l_DeviceAuthInfoIterator != l_DeviceAuthInfo.end(); ++l_DeviceAuthInfoIterator, ++l_DeviceCount)
   {
      l_UserAuthorizationDetails.tDeviceHandle = l_DeviceAuthInfoIterator->getDeviceHandle();
      l_UserAuthorizationDetails.enDeviceType = l_DeviceAuthInfoIterator->getDeviceType();
      l_UserAuthorizationDetails.enUserAuthorizationStatus = l_DeviceAuthInfoIterator->getUserAuthorizationStatus();

      ETG_TRACE_USR4(("ConnectionClientHandler::onProjectionDeviceAuthorizationStatus  Device %d DeviceName: %d", l_DeviceCount, (l_DeviceAuthInfoIterator->getDeviceHandle())));
      ETG_TRACE_USR4(("ConnectionClientHandler::onProjectionDeviceAuthorizationStatus  Device %d DeviceHandle: %d", l_DeviceCount, (l_DeviceAuthInfoIterator->getDeviceType())));
      ETG_TRACE_USR4(("ConnectionClientHandler::onProjectionDeviceAuthorizationStatus  Device %d AuthorizaationStatus: %d", l_DeviceCount, (l_DeviceAuthInfoIterator->getUserAuthorizationStatus())));
      //Taken for AAP and CP
      if (((l_UserAuthorizationDetails.enDeviceType  == ::midw_smartphoneint_fi_types::T_e8_DeviceType__APPLE_DEVICE) || (l_UserAuthorizationDetails.enDeviceType  == ::midw_smartphoneint_fi_types::T_e8_DeviceType__ANDROID_DEVICE)) &&
            (l_UserAuthorizationDetails.enUserAuthorizationStatus == ::midw_smartphoneint_fi_types::T_e8_UserAuthorizationStatus__USER_AUTH_UNKNOWN))
      {
         //  uint32 l_deviceHandle = itr->getDeviceHandle();
         //setActiveDeviceHandle(l_deviceHandle);
         ETG_TRACE_USR4(("onProjectionDeviceAuthorizationStatus UserAuthorizationStatus UNKNOWN"));
         //  AccessoryAppStateHandler::getInstance()->updateAccessoryDisplayContext(true, DISPLAY_CONTEXT_PROJECTION);
         // SpiSourceHandling::getInstance()->vUpdateAudioAccessoryContext(true, AudioContext__SPI_AUDIO_PROJECTION);
         DisplaySourceClientHandler::getInstance(ConnectionClientHandler::getInstance())->vsendCurrentAppID(::midw_smartphoneint_fi_types::T_e8_DisplayContext__DISPLAY_CONTEXT_PROJECTION, true); //Certification requirement
         _smartphoneFIProxy->sendProjectionDeviceAuthorizationSet(*this, l_UserAuthorizationDetails.tDeviceHandle , ::midw_smartphoneint_fi_types::T_e8_UserAuthorizationStatus__USER_AUTH_AUTHORIZED);
      }
   }
}


/************************************************************************
*FUNCTION:		onProjectionDeviceAuthorizationError
*DESCRIPTION:	Function to get errors from authorizing device
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          NA
*HISTORY:
*0.1     26.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onProjectionDeviceAuthorizationError(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::ProjectionDeviceAuthorizationError>& error)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onProjectionDeviceAuthorizationError"));
}


/************************************************************************
*FUNCTION:		vSetProjectionDeviceAuthorization
*DESCRIPTION:	Function to set device authorization
*PARAMETER:     deviceHandle, userAuthorizationStatus
*RETURNVALUE:	NA
*SWFL:          NA
*HISTORY:
*0.1     26.06.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vSetProjectionDeviceAuthorization(uint32 deviceHandle,
      ::midw_smartphoneint_fi_types::T_e8_UserAuthorizationStatus userAuthorizationStatus)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetProjectionDeviceAuthorization"));
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetProjectionDeviceAuthorization deviceHandle = %d", deviceHandle));
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetProjectionDeviceAuthorization userAuthorizationStatus = %d", userAuthorizationStatus));

   _smartphoneFIProxy->sendProjectionDeviceAuthorizationSet(*this, deviceHandle, userAuthorizationStatus);
}


/*********************************************************************************************
                                Method: SetAccessoryDisplayContext
*********************************************************************************************/
/************************************************************************
*FUNCTION:		onSetAccessoryDisplayContextResult
*DESCRIPTION:	Function to get result of SetAccessoryDisplayContext
*PARAMETER:     proxy, result
*RETURNVALUE:	NA
*SWFL:          2583
*HISTORY:
*0.1     05.10.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onSetAccessoryDisplayContextResult(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::SetAccessoryDisplayContextResult>& result)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onSetAccessoryDisplayContextResult"));
   ETG_TRACE_USR4(("ConnectionClientHandler::onSetAccessoryDisplayContextResult ResponseCode = %d", result->getResponseCode()));
}


/************************************************************************
*FUNCTION:		onSetAccessoryDisplayContextError
*DESCRIPTION:	Function to get result of SetAccessoryDisplayContext
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          2583
*HISTORY:
*0.1     05.10.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onSetAccessoryDisplayContextError(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::AccessoryDisplayContextError>& error)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onSetAccessoryDisplayContextError"));
}


/************************************************************************
*FUNCTION:		vSetAccessoryDisplayContext
*DESCRIPTION:	Function to send details
*PARAMETER:     setAccessoryDisplayContextDetails
*RETURNVALUE:	NA
*SWFL:          2583
*HISTORY:
*0.1     05.10.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vSetAccessoryDisplayContext(const SetAccessoryDisplayContextDetails& setAccessoryDisplayContextDetails)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetAccessoryDisplayContext"));

   _smartphoneFIProxy->sendSetAccessoryDisplayContextStart(*this,
         setAccessoryDisplayContextDetails.tDeviceHandle,
         setAccessoryDisplayContextDetails.enDisplayContextInfo,
         setAccessoryDisplayContextDetails.enDisplayConstraintInfo,
         setAccessoryDisplayContextDetails.enContextResponsibility);
}


/**************************************************************************************
*FUNCTION:		onCourierMessage
*DESCRIPTION:	Function to switch to homescreen
*PARAMETER:     oMsg
*RETURNVALUE:	NA
*SWFL:          2583, 2573
*HISTORY:
*0.1     12.10.18      Athira Sreekumar       Initial Version
*0.2     02.04.19      Sundeep Parvathikar    Modified Version w.r.t context switch
**************************************************************************************/
bool ConnectionClientHandler::onCourierMessage(const SwitchToHomeScreenReq& oMsg)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::SwitchToHomeScreenReq()"));
   if (bGetActiveDeviceHandleValidity())
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::SwitchToHomeScreenReq deviceHandle = %d", tGetActiveDeviceHandle()));
      /*
            SwitchApplicationReq* switchApplicationReq = COURIER_MESSAGE_NEW(SwitchApplicationReq)(APPID_APPHMI_MASTER, SURFACEID_MAIN_SURFACE_MASTER);
            if (switchApplicationReq != 0)
            {
               switchApplicationReq->Post();
            }
      */
      midw_smartphoneint_fi_types::T_AccessoryDisplayContext l_AccessoryDisplayContext(
         /*DisplayContext*/ midw_smartphoneint_fi_types::T_e8_DisplayContext__DISPLAY_CONTEXT_NATIVE,
         /*DisplayFlag*/ true //Stop SPI Display Projection
      );

      midw_smartphoneint_fi_types::T_DisplayConstraint l_DisplayConstraint(
         /*Transfer Type*/ midw_smartphoneint_fi_types::T_e8_ResourceTransferType__TRANSFER_TYPE_TAKE,
         /*Transfer Priority*/ midw_smartphoneint_fi_types::T_e8_ResourceTransferPriority__TRANSFER_PRIORITY_USERINITIATED,
         /*Take Constraint*/ midw_smartphoneint_fi_types::T_e8_ResourceTransferConstraint__TRANSFER_CONSTRAINT_USERINITIATED,
         /*Borrow Constraint*/ midw_smartphoneint_fi_types::T_e8_ResourceTransferConstraint__TRANSFER_CONSTRAINT_NA, "");

      SetAccessoryDisplayContextDetails l_SetAccessoryDisplayContextDetails;

      l_SetAccessoryDisplayContextDetails.tDeviceHandle = tGetActiveDeviceHandle();
      l_SetAccessoryDisplayContextDetails.enDisplayContextInfo = l_AccessoryDisplayContext;
      l_SetAccessoryDisplayContextDetails.enDisplayConstraintInfo = l_DisplayConstraint;
      l_SetAccessoryDisplayContextDetails.enContextResponsibility = midw_smartphoneint_fi_types::T_e8_DisplayInfo__DISPLAY_CONSTRAINT;

      vSetAccessoryDisplayContext(l_SetAccessoryDisplayContextDetails);
   }
   ContextSwitchClientHandler::poGetInstance()->requestContextSwitchApplication(
      (uint8)2
      , (uint8)APPID_APPHMI_SPI
      , (APPLICATIONSWITCH_TYPES::enActivityIDs)ContextSwitchClientHandler::poGetInstance()->getActivityIDforCurrentView()
      , (uint8)APPID_APPHMI_MASTER
      , (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_HOME);

   //vDisableSPILayers(); redundent and has been removed
   return true;
}


/*********************************************************************************************
                                Method: SendKeyEvent
*********************************************************************************************/
/************************************************************************
*FUNCTION:		onSendKeyEventResult
*DESCRIPTION:	Function to get result of SendKeyEvent
*PARAMETER:     proxy, result
*RETURNVALUE:	NA
*SWFL:          2582
*HISTORY:
*0.1     31.10.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onSendKeyEventResult(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::SendKeyEventResult>& result)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onSendKeyEventResult"));
   ETG_TRACE_USR4(("ConnectionClientHandler::onSendKeyEventResult ResponseCode = %d", result->getResponseCode()));
   ETG_TRACE_USR4(("ConnectionClientHandler::onSendKeyEventResult ErrorCode = %d", result->getErrorCode()));
}


/************************************************************************
*FUNCTION:		onSendKeyEventError
*DESCRIPTION:	Function to get result of SendKeyEvent
*PARAMETER:     proxy, error
*RETURNVALUE:	NA
*SWFL:          2582
*HISTORY:
*0.1     31.10.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::onSendKeyEventError(const ::boost::shared_ptr<midw_smartphoneint_fi::Midw_smartphoneint_fiProxy>& proxy,
      const ::boost::shared_ptr<midw_smartphoneint_fi::SendKeyEventError>& error)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onSendKeyEventError"));
}


/************************************************************************
*FUNCTION:		vSendKeyEvent
*DESCRIPTION:	Function to send details regarding switch to SPI Home
*PARAMETER:     keyMode, keyCode
*RETURNVALUE:	NA
*SWFL:          2582
*HISTORY:
*0.1     05.10.18      Athira Sreekumar       Initial Version
************************************************************************/
void ConnectionClientHandler::vSendKeyEvent(::midw_smartphoneint_fi_types::T_e8_KeyMode keyMode,
      ::midw_smartphoneint_fi_types::T_e32_KeyCode keyCode)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vSendKeyEvent"));

   _smartphoneFIProxy->sendSendKeyEventStart(*this,
         tGetActiveDeviceHandle(),
         keyMode,
         keyCode);
}


/************************************************************************
*FUNCTION:		onCourierMessage
*DESCRIPTION:	Function to switch to SPI Home
*PARAMETER:     oMsg
*RETURNVALUE:	NA
*SWFL:          2582
*HISTORY:
*0.1     12.10.18      Athira Sreekumar       Initial Version
************************************************************************/
bool ConnectionClientHandler::onCourierMessage(const SwitchToSPIHomeReq& oMsg)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::SwitchToSPIHomeReq()"));

   vSendKeyEvent(midw_smartphoneint_fi_types::T_e8_KeyMode__KEY_PRESS,
                 midw_smartphoneint_fi_types::T_e32_KeyCode__DEV_HOME);

   vSendKeyEvent(midw_smartphoneint_fi_types::T_e8_KeyMode__KEY_RELEASE,
                 midw_smartphoneint_fi_types::T_e32_KeyCode__DEV_HOME);
   return true;
}


/*********************************************************************************************
                                Property: BluetoothDeviceStatus
*********************************************************************************************/
/*********************************************************************************************
*FUNCTION:		onBluetoothDeviceStatusStatus
*DESCRIPTION:	Update if the device should be paired or not
*PARAMETER:     proxy, status
*RETURNVALUE:	NA
*SWFL:          2608,2609
*HISTORY:
*0.1     04.04.19      Bruce Mario Savio Netto       Initial Version
*0.2     17.05.19      Sundeep Parvathikar - Extracting A2DP and HFP, calling vOnSessionActiveOperations
*********************************************************************************************/
void ConnectionClientHandler::onBluetoothDeviceStatusStatus(const ::boost::shared_ptr< ::midw_smartphoneint_fi::Midw_smartphoneint_fiProxy >& proxy,
      const ::boost::shared_ptr< ::midw_smartphoneint_fi::BluetoothDeviceStatusStatus >& status)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onBluetoothDeviceStatusStatus()"));
   //m_bPairingReqd == true means the BT device requires/needs pairing,
   //m_bPairingReqd == false means the BT device is paired.
   ETG_TRACE_USR4(("ConnectionClientHandler::onBluetoothDeviceStatusStatus status->getConnectedProfiles() = %d ", status->getConnectedProfiles()));

   PairingStatus = status->getConnectedProfiles();

   const bool m_isBT_A2DP_Active = ConnectionClientHandler::getInstance()->isBTConnectedProfilesActive_A2DP(PairingStatus);
   const bool m_isBT_HFP_Active = ConnectionClientHandler::getInstance()->isBTConnectedProfilesActive_HFP(PairingStatus);

   ETG_TRACE_USR4(("ConnectionClientHandler::onBluetoothDeviceStatusStatus isBTConnectedProfilesActive_A2DP() = %d ", m_isBT_A2DP_Active));

   ETG_TRACE_USR4(("ConnectionClientHandler::onBluetoothDeviceStatusStatus isBTConnectedProfilesActive_HFP() = %d ", m_isBT_HFP_Active));

   ETG_TRACE_USR4(("ConnectionClientHandler::onBluetoothDeviceStatusStatus Pairing Required = %d", PairingStatus));

   vOnSessionActiveOperations(m_isBT_HFP_Active , m_isBT_A2DP_Active);//BT Popup handling
}


/************************************************************************
*FUNCTION:		onBluetoothDeviceStatusError
*DESCRIPTION:	Update if the device should be paired or not
*PARAMETER:     proxy, status
*RETURNVALUE:	NA
*SWFL:          2608,2609
*HISTORY:
*0.1     04.04.19      Bruce Mario Savio Netto       Initial Version
************************************************************************/
void ConnectionClientHandler::onBluetoothDeviceStatusError(const ::boost::shared_ptr< ::midw_smartphoneint_fi::Midw_smartphoneint_fiProxy >& proxy,
      const ::boost::shared_ptr< ::midw_smartphoneint_fi::BluetoothDeviceStatusError >& error)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onBluetoothDeviceStatusError()"));
}


/************************************************************************
*FUNCTION:		vOnDeviceRemovedPopupHandle
*DESCRIPTION:	Update pop ups on device removed
*PARAMETER:     None
*RETURNVALUE:	NA
*SWFL:          2608,2609
*HISTORY:
*0.1     07.11.18      Bruce Mario Savio Netto        Initial Version
************************************************************************/
void ConnectionClientHandler::vOnDeviceRemovedPopupHandle()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vOnDeviceRemovedPopupHandle()"));
   m_SPIBTPopupShow = false;
   POST_MSG((COURIER_MESSAGE_NEW(SPI_BT_Pairing_Popup_Close)()));
   ETG_TRACE_USR4(("ConnectionClientHandler::vOnDeviceRemovedPopupHandle CourierMsg Sent: SPI_BT_Pairing_Popup_Close"));
}


/**********************************************************************************************
*FUNCTION:		vOnSessionActiveOperations
*DESCRIPTION:	Update functionalities when the SPI session is ACTIVE
*PARAMETER:     const bool &  , const bool &
*RETURNVALUE:	NA
*SWFL:          2607,2608
*HISTORY:
*0.1     18.12.18      Bruce Mario Savio Netto        Initial Version
*0.2     17.05.19      Sundeep Parvathikar	1. Parameters to the fun added
											2. checkinh for m_SessionStatus
											3. else if block added to call vOnDeviceRemovedPopupHandle
**********************************************************************************************/
void ConnectionClientHandler::vOnSessionActiveOperations(const bool& isBT_HFP_Active, const bool& isBT_A2DP_Active)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vOnSessionActiveOperations() isBT_HFP_Active = %d ", isBT_HFP_Active));
   ETG_TRACE_USR4(("ConnectionClientHandler::vOnSessionActiveOperations() isBT_A2DP_Active = %d ", isBT_A2DP_Active));

   if ((!isBT_HFP_Active) &&
         (m_ActiveDeviceCategory == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN) &&
         (m_SessionStatus == ::midw_smartphoneint_fi_types::T_e8_SessionStatus__SESSION_ACTIVE) &&
         (m_SPIBTPopupShow == false))
   {
      m_SPIBTPopupShow = true;
      POST_MSG((COURIER_MESSAGE_NEW(SPI_BT_Pairing_Popup)()));
      ETG_TRACE_USR4(("ConnectionClientHandler::vOnSessionActiveOperations CourierMsg Sent: SPI_BT_Pairing_Popup"));
   }
   else if (m_SPIBTPopupShow)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::vOnSessionActiveOperations else calling vOnDeviceRemovedPopupHandle"));
      vOnDeviceRemovedPopupHandle();
   }
   ETG_TRACE_USR4(("ConnectionClientHandler::vOnSessionActiveOperations()"));
}


/************************************************************************
*FUNCTION:		onSelectDeviceResult
*DESCRIPTION:	Update the device result of the currently selected device
*PARAMETER:     None
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     22.02.19      Bruce Mario Savio Netto        Initial Version
************************************************************************/
void ConnectionClientHandler::onSelectDeviceResult(const ::boost::shared_ptr< ::midw_smartphoneint_fi::Midw_smartphoneint_fiProxy >& proxy,
      const ::boost::shared_ptr< ::midw_smartphoneint_fi::SelectDeviceResult >& result)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onSelectDeviceResult: Received.. "));
}


/************************************************************************
*FUNCTION:		onSelectDeviceResult
*DESCRIPTION:	Update the device result of the currently selected device
*PARAMETER:     None
*RETURNVALUE:	NA
*SWFL:
*HISTORY:
*0.1     22.02.19      Bruce Mario Savio Netto        Initial Version
************************************************************************/
void ConnectionClientHandler::onSelectDeviceError(const ::boost::shared_ptr< ::midw_smartphoneint_fi::Midw_smartphoneint_fiProxy >& proxy,
      const ::boost::shared_ptr< ::midw_smartphoneint_fi::SelectDeviceError >& error)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::onSelectDeviceError: Received.. "));
}


/************************************************************************
*FUNCTION:		onCourierMessage
*DESCRIPTION:	Function to Connect/Disconnect a Device
*PARAMETER:    DeviceConnectionMsg
*RETURNVALUE:	bool
*SWFL:         2445, 2506, 2505, 2503, 2502, 2501
*HISTORY:
*0.1     22.02.19      Bruce Mario Savio Netto     Initial Version
************************************************************************/
bool ConnectionClientHandler::onCourierMessage(const DeviceConnectionMsg& oMsg)
{
   ETG_TRACE_USR4(("ConnectionClientHandler::DeviceConnectionMsg() STARTS"));

   ::midw_smartphoneint_fi_types::T_e8_DeviceCategory category = (::midw_smartphoneint_fi_types::T_e8_DeviceCategory) tGetActiveDeviceCategory();

   //Bruce
   if (m_SessionStatus ==::midw_smartphoneint_fi_types::T_e8_SessionStatus__SESSION_ACTIVE)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::DeviceConnectionMsg()::Session Active Message"));
      _smartphoneFIProxy->sendSelectDeviceStart(*this ,
            tGetActiveDeviceHandle(),   /* deviceHandle */
            ::midw_smartphoneint_fi_types::T_e8_DeviceConnectionType__USB_CONNECTED,  /* DeviceConnectionType */
            ::midw_smartphoneint_fi_types::T_e8_DeviceConnectionReq__DEV_DISCONNECT,     /* DeviceConnectionReq */
            ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_DISABLED,           /* DAPUsage */
            ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_DISABLED,                       /* CDBUsage */
            //::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN,
            category,
            ::midw_smartphoneint_fi_types::T_e8_SelectionType__SELECTION_TYPE_USER_TRIGGERED);
      ETG_TRACE_USR4(("ConnectionClientHandler::DeviceConnectionMsg()::Session Active Message Complete"));
   }
   else if (m_SessionStatus ==::midw_smartphoneint_fi_types::T_e8_SessionStatus__SESSION_INACTIVE)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::DeviceConnectionMsg()::Session Inactive Message"));
      _smartphoneFIProxy->sendSelectDeviceStart(*this ,
            tGetActiveDeviceHandle(),   /* deviceHandle */
            ::midw_smartphoneint_fi_types::T_e8_DeviceConnectionType__USB_CONNECTED,  /* DeviceConnectionType */
            ::midw_smartphoneint_fi_types::T_e8_DeviceConnectionReq__DEV_CONNECT,     /* DeviceConnectionReq */
            ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_DISABLED,           /* DAPUsage */
            ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_DISABLED,                       /* CDBUsage */
            //::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN,
            category,
            ::midw_smartphoneint_fi_types::T_e8_SelectionType__SELECTION_TYPE_USER_TRIGGERED);
      ETG_TRACE_USR4(("ConnectionClientHandler::DeviceConnectionMsg()::Session Inactive Message Complete"));
      if (mParkBrakeInfo == midw_smartphoneint_fi_types:: T_e8_ParkBrake__PARK_BRAKE_NOT_ENGAGED)
      {
         POST_MSG((COURIER_MESSAGE_NEW(SPI_ParkedCondition_Popup)()));/*Connect/Disconnect button*/
      }
   }
   ETG_TRACE_USR4(("ConnectionClientHandler::DeviceConnectionMsg() ENDS"));
   return true;
   //return false;
}


/****************************************************************************************************
*FUNCTION:		isBTConnectedProfilesActive_A2DP
*DESCRIPTION:	(return false- A2DP profile is not active, return true - A2DP profile is active)
*PARAMETER:     const uint8 &
*RETURNVALUE:	bool
*SWFL:          NA
*HISTORY:		NA
*0.1     30.04.19      Sundeep Parvathikar       Initial Version
****************************************************************************************************/
bool ConnectionClientHandler::isBTConnectedProfilesActive_A2DP(const uint8& PairingStatus)
{
   /*
   	HFP A2DP	PairingStatus
   	0	0		0
   	0	1		1
   	1	0		2
   	1	1		3
   */
   if (PairingStatus == UNKNOWN_INFO)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::isBTConnectedProfilesActive_A2DP() if"));
      return false;
   }
   else
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::isBTConnectedProfilesActive_A2DP() else"));
      return isKthBitSet(PairingStatus, A2DP_BIT_POSITION);
   }
}


/****************************************************************************************************
*FUNCTION:		isBTConnectedProfilesActive_A2DP
*DESCRIPTION:	(return false- HFP profile is not active, return true - HFP profile is active)
*PARAMETER:     const uint8 &
*RETURNVALUE:	bool
*SWFL:          NA
*HISTORY:		NA
*0.1     30.04.19      Sundeep Parvathikar       Initial Version
****************************************************************************************************/
bool ConnectionClientHandler::isBTConnectedProfilesActive_HFP(const uint8& PairingStatus)
{
   /*
   	HFP A2DP	PairingStatus
   	0	0		0
   	0	1		1
   	1	0		2
   	1	1		3
   */
   if (PairingStatus == UNKNOWN_INFO)
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::isBTConnectedProfilesActive_HFP() if"));
      return false;
   }
   else
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::isBTConnectedProfilesActive_HFP() else"));
      return isKthBitSet(PairingStatus, HFP_BIT_POSITION);
   }
}


/************************************************************************
*FUNCTION:		readKdsForTechnology
*DESCRIPTION:	Function to get the KDS value based on Technology
*PARAMETER:     NA
*RETURNVALUE:	NA
************************************************************************/
void ConnectionClientHandler::readKdsForTechnology()
{
   dp_tclKdsSystemConfiguration1 _KdsSystemConfigurationInst;
   _KdsSystemConfigurationInst.u8GetMySPIN(_technologyMySpin);
   _KdsSystemConfigurationInst.u8GetCarPlay(_technologyCP);
   _KdsSystemConfigurationInst.u8GetAndroidAuto(_technologyAA);

   ETG_TRACE_USR4(("ConnectionClientHandler::readKdsForTechnology() MySpin = %d", _technologyMySpin));
   ETG_TRACE_USR4(("ConnectionClientHandler::readKdsForTechnology() CarPlay = %d", _technologyCP));
   ETG_TRACE_USR4(("ConnectionClientHandler::readKdsForTechnology() AndroidAuto = %d", _technologyAA));
}


int ConnectionClientHandler::getKDSVehicleType()
{
   uint8 u8VehicleType[2] = {0};
   uint16 u16VehicleType = 0;
   ETG_TRACE_USR4(("ConnectionClientHandler::getKDSVehicleType()"));
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "VehicleType", &u8VehicleType[0], 2))
   {
      u16VehicleType = static_cast<uint16>((u8VehicleType[0] << 8) | u8VehicleType[1]);
   }
   if (u16VehicleType == DP_INF4CV_VEHICLETYPE_SMART)
   {
      sm_variant_smart = 1;
   }
   if ((u16VehicleType == DP_INF4CV_VEHICLETYPE_TRUCK) || (u16VehicleType == DP_INF4CV_VEHICLETYPE_COACH))
   {
      sm_variant_smart = 0;
   }
   return u16VehicleType;
}


/************************************************************************
*FUNCTION:		ismySPINEnabled
*DESCRIPTION:	Function to get the KDS value based on Technology
*PARAMETER:     NA
*RETURNVALUE:	NA
************************************************************************/
bool ConnectionClientHandler::ismySPINEnabled()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::ismySPINEnabled() MySpin = %d", _technologyMySpin));
   return (bool)_technologyMySpin;
}


/************************************************************************
*FUNCTION:		isCarPlayEnabled
*DESCRIPTION:	Function to get the KDS value based on Technology
*PARAMETER:     NA
*RETURNVALUE:	NA
************************************************************************/
bool ConnectionClientHandler::isCarPlayEnabled()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::isCarPlayEnabled() CarPlay = %d", _technologyCP));
   return (bool)_technologyCP;
}


/************************************************************************
*FUNCTION:		isAndroidAutoEnabled
*DESCRIPTION:	Function to get the KDS value based on Technology
*PARAMETER:     NA
*RETURNVALUE:	NA
************************************************************************/
bool ConnectionClientHandler::isAndroidAutoEnabled()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::isAndroidAutoEnabled() AndroidAuto = %d", _technologyAA));
   return (bool)_technologyAA;
}


/************************************************************************
*FUNCTION:		vSendSelectDeviceStart
*DESCRIPTION:	Function to Disconnect a Device connected via USB
*PARAMETER:     NA
*RETURNVALUE:	NA
************************************************************************/
void ConnectionClientHandler::vSendSelectDeviceStart()
{
   ::midw_smartphoneint_fi_types::T_e8_DeviceCategory category = (::midw_smartphoneint_fi_types::T_e8_DeviceCategory) tGetActiveDeviceCategory();
   ETG_TRACE_USR4(("ConnectionClientHandler::vSendSelectDeviceStart() Starts"));
   _smartphoneFIProxy->sendSelectDeviceStart(*this ,
         tGetActiveDeviceHandle(),   /* deviceHandle */
         ::midw_smartphoneint_fi_types::T_e8_DeviceConnectionType__USB_CONNECTED,  /* DeviceConnectionType */
         ::midw_smartphoneint_fi_types::T_e8_DeviceConnectionReq__DEV_DISCONNECT,     /* DeviceConnectionReq */
         ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_DISABLED,           /* DAPUsage */
         ::midw_smartphoneint_fi_types::T_e8_EnabledInfo__USAGE_DISABLED,                       /* CDBUsage */
         category,
         ::midw_smartphoneint_fi_types::T_e8_SelectionType__SELECTION_TYPE_USER_TRIGGERED);
}


/************************************************************************
*FUNCTION:		AAPMySPinStandby
*DESCRIPTION:	Function to launch Android Auto and Myspin after coming from standby mode
*PARAMETER:     NA
*RETURNVALUE:	Void
************************************************************************/
void ConnectionClientHandler::AAPMySPinStandby()
{
   if (tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN ||
         tGetActiveDeviceCategory() == ::midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_ANDROIDAUTO)
   {
      if (bGetActiveDeviceHandleValidity())
      {
         vLaunchApp(tGetActiveDeviceHandle(),
                    (::midw_smartphoneint_fi_types::T_e8_DeviceCategory)tGetActiveDeviceCategory(),
                    ::midw_smartphoneint_fi_types::T_e8_DiPOAppType__DIPO_NOT_USED);
         ETG_TRACE_USR4(("SystemPowerModeClientHandler::onHmiStateStatus() Calling Launch App"));
      }
   }
}


/************************************************************************
*FUNCTION:		vSetMySPINDisplayAttributesOnStartUp
*DESCRIPTION:	Set Display Attributes for MYSPIN technology at startup
*PARAMETER:     NA
*RETURNVALUE:	NA
*SWFL:          1993, 2581, 2604
*Bug			644654
*HISTORY:
*0.1	 12.02.2020	   Shivani Priya    Initial Version
************************************************************************/
void ConnectionClientHandler::vSetMySPINDisplayAttributesOnStartUp()
{
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetMySPINDisplayAttributesOnStartUp()"));
   ETG_TRACE_USR4(("ConnectionClientHandler::vSetMySPINDisplayAttributesOnStartUp DeviceCategory = %d", m_ActiveDeviceCategory));

   if (_smartphoneFIProxy->isAvailable() && ismySPINEnabled())
   {
      std::vector<midw_smartphoneint_fi_types::T_DisplayLayerAttributes> l_DisplayLayers;
      //FOR MYSPIN DEVICE
      midw_smartphoneint_fi_types::T_DisplayLayerAttributes l_DisplayLayerAttr_MYSPIN(
         /*DeviceCategory*/      midw_smartphoneint_fi_types::T_e8_DeviceCategory__DEV_TYPE_MYSPIN,
         /*TouchLayerID*/        3501,
         /*TouchSurfaceID*/      80,
         /*VideoLayerID*/        3500,
         /*VideoSurfaceID*/      59,
         /*LayerWidth*/          706,
         /*LayerHeight*/         480,
         /*LayerWidthinMM*/      137.2059,
         /*LayerHeightinMM*/     80.712,
         /*PixelAspectRatio*/    "1.69",
         /*DPI*/                 160,
         /*DisplayTouchType*/    midw_smartphoneint_fi_types::T_e8_DisplayTouchType__e8MULTI_TOUCH,
         /*PriorityIndex*/       0);

      l_DisplayLayers.push_back(l_DisplayLayerAttr_MYSPIN);

      midw_smartphoneint_fi_types::T_DisplayAttributes l_DisplayAttributes(
         /*ScreenHeight*/            480,
         /*ScreenWidth*/             700,
         /*ScreenHeightMm*/          186,
         /*ScreenWidthMm*/           127,
         /*DisplayType*/             midw_smartphoneint_fi_types::T_e8_DisplayType__e8CAPACITIVE,
         /*DisplayLayerAttributes*/  l_DisplayLayers);

      //Sending Required Screen Configuration to be displayed
      ETG_TRACE_USR4(("ConnectionClientHandler::vSetMySPINDisplayAttributesOnStartUp Sending Initlaized Display Attributes"));
      _smartphoneFIProxy->sendSetDisplayAttributesStart(*this, l_DisplayAttributes);
   }
   else
   {
      ETG_TRACE_USR4(("ConnectionClientHandler::vSetMySPINDisplayAttributesOnStartUp SmartPhoneFiProxy is not Available"));
   }
}


} // namespace Core
} // namespace App
