/* ***************************************************************************************
* FILE:          PhoneCallModel.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PhoneCallModel.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

#include "util/StrUtf8.h"
#include "AppHmi_PhoneStateMachine.h"
#include "AppHmi_PhoneConstants.h"
#include "PhoneCallModel.h"
#include "hmi_trace_if.h"
#ifdef WINDOWS_SIMULATION
#include <iostream>
#include "AppHmi_PhoneStateMachineSEMLibB.h"
using namespace std;
#endif

#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_PHONE_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_FILE_PREFIX            App::Core::PhonePhoneCallModel::
#include "trcGenProj/Header/PhoneCallModel.cpp.trc.h"
#endif

namespace App {
namespace Core {

PhoneCallModel* PhoneCallModel::poPhoneCallModel = NULL;
/************************************************************************
*FUNCTION: 	   PhoneCallModel
*DESCRIPTION:  Constructor
*PARAMETER: NA
*
*RETURNVALUE: 	NA
*
*HISTORY:
15/09/2017 Shwetha K
Initial Revision

************************************************************************/
PhoneCallModel::PhoneCallModel()

{
}


/************************************************************************
*FUNCTION: 	   ~PhoneCallModel
*DESCRIPTION:  Destructor
*PARAMETER: NA
*
*RETURNVALUE: 	NA
*
*HISTORY:
15/09/2017 Shwetha K
Initial Revision

************************************************************************/
PhoneCallModel::~PhoneCallModel()
{
}


/**
* @brief   : Incomming message for HMI data model a message component
* @param   :
//*/
//bool PhoneCallModel::onCourierMessage(const Courier::StartupMsg& /*oMsg*/)
//{
//
//	return true;
//}
/************************************************************************
*FUNCTION: 	   OnUpdCallStatus
*DESCRIPTION:  obtains call status from service and updates the same to CallProgress,Message progress data binding update
*PARAMETER: tU16 CallInst-> call instance(1st call or 2nd call)
 std::string CallDuration-> call duration in hr:mm:ss or mm:ss
 std::string CallerID-> contact name if present, else unknown
 btphn_tenPhoneCallStatus CallStatus-> call status of current call
 std::string PhoneType-> phone type (business,other,mobile,private)
*
*RETURNVALUE: 	NA
*
*HISTORY:
15/09/2017 Shwetha K
Initial Revision

************************************************************************/
void PhoneCallModel::OnUpdCallStatus(tU16 CallInst, std::string CallDuration, std::string CallerID,  btphn_tenPhoneCallStatus CallStatus, std::string PhoneType)
{
   ETG_TRACE_USR4(("PhoneCallModel : OnUpdCallStatus()"));
   bool bRetVal = false;
   /* Retrieving text from textDB for callerID of unknown text */
   const Candera::String CALLERID_UNKNOWN_STR          = LANGUAGE_STRING(TextId_0x0ECE, "Unknown");
   /* Getting Call Status */
   UInt8 u8CallStatus = CallStatus;
   std::string strCallerID = CallerID.c_str();
   bRetVal = bDataUpdate(u8CallStatus, PhoneType);

   ETG_TRACE_USR4(("cobCallStatusResp update"));
   //SUZUKI-18257
   //if caller id is unknown : send %1 to gui. %1 is mapped to Unknown(0x0edc)
   if (BTPHN_EN_IDLE != u8CallStatus)
   {
      if (BTPHN_C_VALUE_ZERO == CallerID.compare(BTPHN_C_CONTACT_UNKNOWN))
      {
         (*oCallProgress).mCallerID = strCallerID.c_str();
         ETG_TRACE_USR4(("Unknown: %s", BTPHN_C_CALLER_ID_UNKNOWN));

         /* Update gadget callerID value with unknown text */
         (*oCallProgress).mGadgetCallerID = CALLERID_UNKNOWN_STR;
      }
      else if (CallerID.empty())
      {
         (*oCallProgress).mCallerID = BTPHN_C_CALLER_ID_UNKNOWN;
         ETG_TRACE_USR4(("(*oCallProgress).mCallerID : %s", ((*oCallProgress).mCallerID.GetCString())));

         /* Update gadget callerID value with unknown text */
         (*oCallProgress).mGadgetCallerID = CALLERID_UNKNOWN_STR;
      }
      else
      {
         (*oCallProgress).mCallerID = strCallerID.c_str();
         ETG_TRACE_USR4(("Id: %s", strCallerID.c_str()));

         /* Update gadget callerID value with valid input text */
         (*oCallProgress).mGadgetCallerID = strCallerID.c_str();
      }
   }
   else
   {
      (*oCallProgress).mCallerID = "";
      (*oCallProgress).mGadgetCallerID = "";
   }

   ETG_TRACE_USR4(("Caller Name : %s", strCallerID.c_str()));
   (*oCallProgress).mContactType = PhoneType.c_str();

   if (u8CallStatus != BTPHN_EN_DISCONNECTING)
   {
      // Apart from IDLE  call status, update empty call duration for the internally updated call status of DIAL_START as well as to avoid updating of invalid values
      if ((u8CallStatus == BTPHN_EN_IDLE) || (u8CallStatus == BTPHN_EN_DIAL_START))
      {
         (*oCallProgress).mCallDuration = "";
      }
      else
      {
         (*oCallProgress).mCallDuration = CallDuration.c_str();
      }
      ETG_TRACE_USR4(("OnUpdCallStatus call  duration update"));
   }

   // send update being sent multiple time, hence moving it to the end
   oCallProgress.MarkAllItemsModified();
   bRetVal = bRetVal && oCallProgress.SendUpdate(true);

   if (false == bRetVal)
   {
      ETG_TRACE_ERR(("PhoneCallModel : OnUpdCallStatus(), Error in Call Status  Updated"));
   }
}


/************************************************************************
*FUNCTION: PhoneCallModel::vDeletecalldetails
 Bug -730719
************************************************************************/
void PhoneCallModel:: vDeletecalldetails()
{
   ETG_TRACE_USR4(("PhoneCallModel::vDeletecalldetails called"));
   (*oCallProgress).mGadgetCallerID = "";
   (*oCallProgress).mCallDuration = "";
   oCallProgress.MarkAllItemsModified();
   oCallProgress.SendUpdate();
}


bool PhoneCallModel::bDataUpdate(UInt8 u8CallStatus, std::string PhoneType)
{
   bool bRetVal = false;
   ETG_TRACE_USR4(("PhoneCallModel::vDataUpdate"));
   switch (u8CallStatus)
   {
      case BTPHN_C_HEXA_TWO:
      {
         ETG_TRACE_USR4(("OnUpdCallStatus update ACTIVE"));
         /* Updating Microphone Mute Status */
         (*oMicEnable).mEnabled = BTPHN_C_BUTTON_ENABLED;
         /* Updating DialPad Button Databinding*/
         (*oDialPadButton).mEnabled = BTPHN_C_BUTTON_ENABLED;
         /* Updating Call Status  Databinding*/
         (*oCallProgress).mCallStatus = BTPHN_C_CALL_STATUS_ACTIVE;
         break;
      }
      case BTPHN_C_HEXA_THREE:
      {
         ETG_TRACE_USR4(("OnUpdCallStatus update HOLD"));
         /* Updating Microphone Mute Status */
         (*oMicEnable).mEnabled = BTPHN_C_BUTTON_DISABLED;
         /* Updating DialPad Button Databinding*/
         (*oDialPadButton).mEnabled = BTPHN_C_BUTTON_DISABLED;
         /* Updating Call Status Databinding*/
         (*oCallProgress).mCallStatus = BTPHN_C_DEVICE_ONHOLD;
         break;
      }

      case BTPHN_C_HEXA_EIGHT:
      {
         ETG_TRACE_USR4(("OnUpdCallStatus update Ringing"));
         (*oIncomingCall).mContactType = PhoneType.c_str();
         (*oIncomingCall).mCount = (Candera::String)BTPHN_C_FIRST_INCOMING;
         oIncomingCall.MarkItemModified(ItemKey::IncomingCall::ContactTypeItem);
         break;
      }
      case BTPHN_EN_DIAL_ERROR:
      {
         /* Error Handling done in BL*/
         ETG_TRACE_USR4(("OnUpdCallStatus update dial  error"));
         IsDialed = BTPHN_C_VALUE_ZERO;
         break;
      }
      default:
      {
         ETG_TRACE_USR4(("OnUpdCallStatus update IDLE"));
         break;
      }
   }
   oDialPadButton.MarkItemModified(ItemKey::DialPadButton::EnabledItem);
   bRetVal = oIncomingCall.SendUpdate(true);
   bRetVal =  bRetVal && oDialPadButton.SendUpdate(true);
   oMicEnable.MarkItemModified(ItemKey::MicEnable::EnabledItem);
   bRetVal = bRetVal && oMicEnable.SendUpdate(true);
   return bRetVal;
}


/************************************************************************
*FUNCTION: 	   OnUpdIncomingCallStatus
*DESCRIPTION:  obtains incoming call status from service and updates the same to incoming call databinding update
*PARAMETER: std::string ContactName-> caller name of the incoming call
 std::string TelePhoneNo->phone number of the caller
 btphn_tenPhoneIncomingCallStatus CallStatus-> call status(it has to be ringing if there is an incoming call)
*
*RETURNVALUE: 	NA
*
*HISTORY:
15/09/2017 Shwetha K
Initial Revision
*13.08.2019 Rev 0.2, Palani K, SYSFL-5565

************************************************************************/
void PhoneCallModel::OnUpdIncomingCallStatus(std::string ContactName, std::string TelePhoneNo, btphn_tenPhoneIncomingCallStatus CallStatus, std::string DeviceName)
{
#ifdef WINDOWS_SIMULATION
   (*oIncomingCall).mCount = (Candera::String)BTPHN_C_FIRST_INCOMING;
#endif // WINDOWS_SIMULATION
   ETG_TRACE_USR4(("ContactName:%s", ContactName.c_str()));

   if (BTPHN_EN_INRINGING == CallStatus)
   {
      if (BTPHN_C_VALUE_ZERO == ContactName.compare(BTPHN_C_CONTACT_UNKNOWN))
      {
         (*oIncomingCall).mCallerID = BTPHN_C_CALLER_ID_UNKNOWN;
         ETG_TRACE_USR4(("(*oIncomingCall).mCallerID : %s", ((*oIncomingCall).mCallerID.GetCString())));
      }
      else if (ContactName.empty() && TelePhoneNo.empty())
      {
         (*oIncomingCall).mCallerID = BTPHN_C_CALLER_ID_UNKNOWN;
         ETG_TRACE_USR4(("(*oIncomingCall).mCallerID : %s", ((*oIncomingCall).mCallerID.GetCString())));
      }
      else
      {
         (*oIncomingCall).mCallerID = ContactName.c_str();
         ETG_TRACE_USR4(("(*oIncomingCall).mCallerID : %s", ((*oIncomingCall).mCallerID.GetCString())));
      }
      (*oIncomingCall).mContactType = BTPHN_C_CONTACT_TYPE_PHONE_GENRIC;
      (*oIncomingCall).mDeviceName = DeviceName.c_str();
      ETG_TRACE_USR4(("(*oIncomingCall).mDeviceName : %s", ((*oIncomingCall).mDeviceName.GetCString())));
   }

   oIncomingCall.MarkAllItemsModified();
   oIncomingCall.SendUpdate(true);
   ETG_TRACE_USR4(("Call Count: %s", ((*oIncomingCall).mCount.GetCString())));
}


/************************************************************************
*FUNCTION: 	   OnUpdMicMuteStatus
*DESCRIPTION:  Updates Microphone mute status to GUI
*PARAMETER :	MuteStatus -> 0->Phone mute , 1-> phone unmute
*RETURNVALUE: 	NA
*
*HISTORY:
15/09/2017 Shwetha K
Initial Revision

************************************************************************/
void PhoneCallModel::OnUpdMicMuteStatus(bool MuteStatus)
{
   ETG_TRACE_USR4(("PhoneCallModel : OnUpdMicMuteStatus() MuteStatus =%d ", MuteStatus));
   (*oMicEnable).mIsSelected = MuteStatus;
   oMicEnable.MarkItemModified(ItemKey::MicEnable::IsSelectedItem);
   bool bRetVal = oMicEnable.SendUpdate(true);

   if (true == bRetVal)
   {
      ETG_TRACE_USR4(("PhoneCallModel : OnUpdMicMuteStatus() "
                      "Microphone status Updated"));
   }
   else
   {
      ETG_TRACE_ERR(("PhoneCallModel : OnUpdMicMuteStatus(), Error in "
                     "Microphone status update"));
   }
}


/************************************************************************
*FUNCTION: PhoneCallModel::OnUpdMultipleCallStatusResp
*DESCRIPTION: Invoked when there is an active multiple call  update by service
*PARAMETER: std::string FirstCallerID-> caller name of 1st call
 std::string FirstCallDuration->call duration of 1st call
 tU8 u8FirstCallStatus-> call status of 1st call
	std::string FirstContactType-> contact phone type of 1st call
	std::string SecondCallerID->caller name of 2nd call
	std::string SecondCallDuration->call duration of 2nd call
	tU8 u8SecondCallStatus->call status of 2nd call
	std::string SecondContactType->contact phone type of 2nd call
*RETURNVALUE: 	void
*HISTORY:
15/09/2017 Shwetha K
Initial Revision
************************************************************************/
void PhoneCallModel::OnUpdMultipleCallStatusResp(std::string FirstCallerID, std::string FirstCallDuration, tU8 u8FirstCallStatus,
      std::string FirstContactType, std::string SecondCallerID, std::string SecondCallDuration, tU8 u8SecondCallStatus,
      std::string SecondContactType)
{
   ETG_TRACE_USR4(("PhoneCallModel::OnUpdMultipleCallStatusResp()"));
   tBool bRetVal = false;
   (*oMicEnable).mEnabled = BTPHN_C_BUTTON_ENABLED;

   if ((BTPHN_EN_ACTIVE == u8FirstCallStatus) &&
         (BTPHN_EN_ON_HOLD == u8SecondCallStatus))
   {
      bRetVal = bMultipleCallStatusUpdate(
                   FirstCallerID.c_str(),
                   SecondCallerID.c_str(),
                   FirstCallDuration.c_str(),
                   SecondCallDuration.c_str(),
                   (Candera::String) BTPHN_C_CALL_STATUS_ACTIVE,
                   (Candera::String) BTPHN_C_DEVICE_ONHOLD,
                   FirstContactType.c_str(),
                   SecondContactType.c_str(),
                   (Candera::String) BTPHN_C_CALL_MULTIPLE,
                   BTPHN_C_VALUE_ONE
                );
      OnUpdMultipleConfCallStatus(FirstCallerID.c_str(), FirstCallDuration.c_str());
      ETG_TRACE_USR4(("First Call Active"));
   }
   else if ((BTPHN_EN_ON_HOLD == u8FirstCallStatus) &&
            (BTPHN_EN_ACTIVE == u8SecondCallStatus))
   {
      bRetVal = bMultipleCallStatusUpdate(
                   SecondCallerID.c_str(),
                   FirstCallerID.c_str(),
                   SecondCallDuration.c_str(),
                   FirstCallDuration.c_str(),
                   (Candera::String) BTPHN_C_CALL_STATUS_ACTIVE,
                   (Candera::String) BTPHN_C_DEVICE_ONHOLD,
                   SecondContactType.c_str(),
                   FirstContactType.c_str(),
                   (Candera::String) BTPHN_C_CALL_MULTIPLE,
                   BTPHN_C_VALUE_ONE
                );
      OnUpdMultipleConfCallStatus(SecondCallerID.c_str(), SecondCallDuration.c_str());
      ETG_TRACE_USR4(("Second Call Active"));
   }
   else if (((BTPHN_EN_CONFERENCE == u8FirstCallStatus) &&
             (BTPHN_EN_CONFERENCE == u8SecondCallStatus)) ||
            ((BTPHN_EN_ACTIVE == u8FirstCallStatus) &&
             (BTPHN_EN_ACTIVE == u8SecondCallStatus)))
   {
      bRetVal = bMultipleCallStatusUpdate(
                   FirstCallerID.c_str(),
                   SecondCallerID.c_str(),
                   FirstCallDuration.c_str(),
                   SecondCallDuration.c_str(),
                   (Candera::String) BTPHN_C_CALL_STATUS_ACTIVE,
                   (Candera::String) BTPHN_C_CALL_STATUS_ACTIVE,
                   FirstContactType.c_str(),
                   SecondContactType.c_str(),
                   (Candera::String) BTPHN_C_CALL_CONFERENCE,
                   BTPHN_C_VALUE_ZERO
                );
      OnUpdMultipleConfCallStatus(BTPHN_C_CALL_CONFERENCE, FirstCallDuration.c_str());
      ETG_TRACE_USR4(("Conference Active"));
   }

   else if ((BTPHN_EN_ON_HOLD == u8FirstCallStatus) &&
            (BTPHN_EN_ON_HOLD == u8SecondCallStatus))
   {
      bRetVal = bMultipleCallStatusUpdate(
                   FirstCallerID.c_str(),
                   SecondCallerID.c_str(),
                   FirstCallDuration.c_str(),
                   SecondCallDuration.c_str(),
                   (Candera::String) BTPHN_C_DEVICE_ONHOLD,
                   (Candera::String) BTPHN_C_DEVICE_ONHOLD,
                   FirstContactType.c_str(),
                   SecondContactType.c_str(),
                   (Candera::String) BTPHN_C_CALL_CONFERENCE,
                   BTPHN_C_VALUE_ZERO
                );
      (*oMicEnable).mEnabled = BTPHN_C_BUTTON_DISABLED;
      OnUpdMultipleConfCallStatus(BTPHN_C_CALL_CONFERENCE, FirstCallDuration.c_str());
      ETG_TRACE_USR4(("Conference on Hold"));
   }
   else
   {
      vMultipleCallStatusInitatingUpdate
      (
         FirstCallerID.c_str(),
         SecondCallerID.c_str(),
         u8FirstCallStatus,
         u8SecondCallStatus,
         FirstContactType.c_str(),
         SecondContactType.c_str(),
         FirstCallDuration.c_str(),
         SecondCallDuration.c_str());

      (*oMicEnable).mEnabled = BTPHN_C_BUTTON_DISABLED;
   }
   oMicEnable.MarkAllItemsModified();
   oMicEnable.SendUpdate(true);
}


/************************************************************************
*FUNCTION: PhoneCallModel::bMultipleCallStatusUpdate
*DESCRIPTION: Invoked by OnUpdMultipleCallStatusResp when there is change in call status of already placed multiple call
*PARAMETER:	Candera::String strFirstCallerID,
Candera::String strSecondCallerID,
Candera::String strFirstCallDuration,
Candera::String strSecondCallDuration,
UInt8 u8FirstCallStatus,
UInt8 u8SecondCallStatus,
Candera::String strFirstCallContactType,
Candera::String strSecondCallContactType,
Candera::String strCallStatusType,
Bool bIsEnabled
*RETURNVALUE: 	void
*HISTORY:
*18.01.2018 Rev. 0.1 shwetha K
* 		  Initial Revision.
************************************************************************/
bool PhoneCallModel::bMultipleCallStatusUpdate(
   Candera::String strFirstCallerID,
   Candera::String strSecondCallerID,
   Candera::String strFirstCallDuration,
   Candera::String strSecondCallDuration,
   Candera::String strFirstCallStatus,
   Candera::String strSecondCallStatus,
   Candera::String strFirstCallContactType,
   Candera::String strSecondCallContactType,
   Candera::String strCallStatusType,
   tBool bIsEnabled
)
{
   ETG_TRACE_USR4(("PhoneCallModel::bMultipleCallStatusUpdate()"));
   //SUZUKI-18257
   if (strFirstCallerID == BTPHN_C_CONTACT_UNKNOWN)
   {
      (*oMultipleCallProgress).mActiveCallerID = BTPHN_C_CALLER_ID_UNKNOWN;
   }
   else
   {
      (*oMultipleCallProgress).mActiveCallerID = strFirstCallerID;
   }
   if (strSecondCallerID == BTPHN_C_CONTACT_UNKNOWN)
   {
      (*oMultipleCallProgress).mOnHoldCallerID = BTPHN_C_CALLER_ID_UNKNOWN;
   }
   else
   {
      (*oMultipleCallProgress).mOnHoldCallerID = strSecondCallerID;
   }

   (*oMultipleCallProgress).mActiveCallDuration = strFirstCallDuration;
   (*oMultipleCallProgress).mOnHoldCallDuration = strSecondCallDuration;
   (*oMultipleCallProgress).mActiveCallStatus = strFirstCallStatus;
   (*oMultipleCallProgress).mOnHoldCallerStatus = strSecondCallStatus;
   (*oMultipleCallProgress).mActiveContactType = strFirstCallContactType;
   (*oMultipleCallProgress).mOnHoldContactType = strSecondCallContactType;
   (*oMultipleCallProgress).mCallType = strCallStatusType;
   (*oMultipleCallProgress).mIsEnableDialPad = bIsEnabled;
   (*oMultipleCallProgress).mIsEnableSwap = bIsEnabled;
   oMultipleCallProgress.MarkAllItemsModified();
   bool retval = oMultipleCallProgress.SendUpdate(true);
   return retval;
}


/************************************************************************
*FUNCTION: PhoneCallModel::bMultipleCallStatusUpdate
*DESCRIPTION: Invoked by OnUpdMultipleCallStatusResp when multiple call is initiated
*PARAMETER:	Candera::String strFirstCallerID,
Candera::String strSecondCallerID,
UInt8 u8FirstCallStatus,
UInt8 u8SecondCallStatus,
Candera::String strFirstCallContactType,
Candera::String strSecondCallContactType,
Candera::String strFirstCallDuration,
Candera::String strSecondCallDuration

*RETURNVALUE: 	void
*HISTORY:
*18.01.2018 Rev. 0.1 shwetha K
* 		  Initial Revision.
************************************************************************/
void PhoneCallModel::vMultipleCallStatusInitatingUpdate
(
   Candera::String strFirstCallerID,
   Candera::String strSecondCallerID,
   UInt8 u8FirstCallStatus,
   UInt8 u8SecondCallStatus,
   Candera::String strFirstCallContactType,
   Candera::String strSecondCallContactType,
   Candera::String strFirstCallDuration,
   Candera::String strSecondCallDuration)
{
   ETG_TRACE_USR4(("PhoneCallModel :vMultipleCallStatusInitatingUpdate"));
   tBool bRetVal = false;
   if (BTPHN_EN_DIALING == u8SecondCallStatus)
   {
      ETG_TRACE_USR4(("PhoneCallModel :"
                      "Multiple Call Status : Second Call Dialling /n"));
      (*oCallProgress).mCallerID = strSecondCallerID;
      (*oCallProgress).mContactType = strSecondCallContactType;
      (*oCallProgress).mCallDuration = BTPHN_C_ZERO_CALL_DURATION;
      /* Update gadget callerID value with valid input text */
      (*oCallProgress).mGadgetCallerID = strSecondCallerID;
      oCallProgress.MarkAllItemsModified();
      bRetVal = oCallProgress.SendUpdate(true);
   }

   else if (BTPHN_EN_DIALING == u8FirstCallStatus)
   {
      ETG_TRACE_USR4(("PhoneCallModel :"
                      "Multiple Call Status : First Call Dialling /n"));
      (*oCallProgress).mCallerID = strFirstCallerID;
      (*oCallProgress).mContactType = strFirstCallContactType;
      (*oCallProgress).mCallDuration = BTPHN_C_ZERO_CALL_DURATION;
      /* Update gadget callerID value with valid input text */
      (*oCallProgress).mGadgetCallerID = strFirstCallerID;
      oCallProgress.MarkAllItemsModified();
      bRetVal = oCallProgress.SendUpdate(true);
   }
   else if (BTPHN_EN_RINGTONE == u8FirstCallStatus)
   {
      (*oIncomingCall).mCallerID = strFirstCallerID;
      ETG_TRACE_USR4(("(*oIncomingCall).mCallerID : %s", ((*oIncomingCall).mCallerID.GetCString())));
      (*oIncomingCall).mContactType = strFirstCallContactType;
      (*oIncomingCall).mCount =
         (Candera::String)BTPHN_C_SECOND_INCOMING;
      oIncomingCall.MarkAllItemsModified();
      bRetVal = oIncomingCall.SendUpdate(true);
      (*oCallProgress).mCallerID = strSecondCallerID;
      (*oCallProgress).mContactType = strSecondCallContactType;
      (*oCallProgress).mCallDuration = strSecondCallDuration;
      /* Update gadget callerID value with valid input text */
      (*oCallProgress).mGadgetCallerID = strSecondCallerID;
      oCallProgress.MarkAllItemsModified();
      bRetVal = bRetVal && oCallProgress.SendUpdate(true);
   }
   else if (BTPHN_EN_RINGTONE == u8SecondCallStatus)
   {
      (*oIncomingCall).mCallerID = strSecondCallerID;
      ETG_TRACE_USR4(("(*oIncomingCall).mCallerID : %s", ((*oIncomingCall).mCallerID.GetCString())));
      (*oIncomingCall).mContactType = strSecondCallContactType;
      (*oIncomingCall).mCount =
         (Candera::String)BTPHN_C_SECOND_INCOMING;
      oIncomingCall.MarkAllItemsModified();
      bRetVal = oIncomingCall.SendUpdate(true);
      (*oCallProgress).mCallerID = strFirstCallerID;
      (*oCallProgress).mContactType = strFirstCallContactType;
      (*oCallProgress).mCallDuration = strFirstCallDuration;
      /* Update gadget callerID value with valid input text */
      (*oCallProgress).mGadgetCallerID = strFirstCallerID;
      oCallProgress.MarkAllItemsModified();
      bRetVal = bRetVal && oCallProgress.SendUpdate(true);
   }
   else
   {
      ETG_TRACE_USR4(("PhoneCallModel :"
                      "Multiple Call Status : No Active Call :Error/n"));
   }
   if (true == bRetVal)
   {
      ETG_TRACE_USR4(("PhoneCallModel : vMultipleCallStatusInitatingUpdate() "
                      "Multiparty Call Databinding Updated "));
   }
   else
   {
      ETG_TRACE_ERR(("btphn_tclBTPhoneModel : Error in "
                     "Multiparty Call Databinding Update"));
   }
}


/************************************************************************
*FUNCTION: PhoneCallModel::OnUpdMultipleConfCallStatus
*DESCRIPTION: Invoked by model to update the conference and multiple call info in the phone tile of Homescreen scene
*PARAMETER:	CallerID, CallDuration
Bug:280287
*RETURNVALUE: 	void
*HISTORY:
*18.01.2018 Rev. 0.1 Vidya
* 		  Initial Revision.
************************************************************************/

void PhoneCallModel::OnUpdMultipleConfCallStatus(Candera::String CallerID, Candera::String CallDuration)
{
   ETG_TRACE_USR4(("PhoneCallModel : OnUpdMultipleConfCallStatus"));
   tBool bRetVal = false;
   /* Retrieving text from textDB for callerID of conference call text */
   const Candera::String CALLERID_CONFERENCE_STR          = LANGUAGE_STRING(TextId_0x050C, "Conference call");
   (*oCallProgress).mCallerID = CallerID;
   (*oCallProgress).mCallDuration = CallDuration;
   if (CallerID == BTPHN_C_CALL_CONFERENCE)
   {
      /* Update gadget callerID value with coference call text */
      (*oCallProgress).mGadgetCallerID = CALLERID_CONFERENCE_STR;
   }
   else
   {
      /* Update gadget callerID value with valid input text */
      (*oCallProgress).mGadgetCallerID = CallerID;
   }
   oCallProgress.MarkItemModified(ItemKey::CallProgress::CallerIDItem);
   oCallProgress.MarkItemModified(ItemKey::CallProgress::CallDurationItem);
   oCallProgress.MarkItemModified(ItemKey::CallProgress::GadgetCallerIDItem);
   bRetVal = oCallProgress.SendUpdate(true);
   if (true == bRetVal)
   {
      ETG_TRACE_USR4(("PhoneCallModel : OnUpdMultipleConfCallStatus()  Databinding Updated "));
   }
   else
   {
      ETG_TRACE_USR4(("PhoneCallModel : OnUpdMultipleConfCallStatus()  Error in Databinding Updated "));
   }
}


/////////////
/************************************************************************
*FUNCTION: PhoneCallModel::OnUpdVehicleAudioSource
*DESCRIPTION: Invoked by client when there is change in vehicle audio status
*PARAMETER:	AudioStatus -> call in handset or in vehicle

*RETURNVALUE: 	void
*HISTORY:
*18.01.2018 Rev. 0.1 shwetha K
* 		  Initial Revision.
************************************************************************/
void PhoneCallModel::OnUpdVehicleAudioSource(bool AudioStatus)
{
   ETG_TRACE_USR4(("PhoneCallModel : OnUpdVehicleAudioSource() AudioStatus= %d ", AudioStatus));
   (*oVehicleAudioStatus).mIsSelected = AudioStatus;
   oVehicleAudioStatus.MarkItemModified
   (ItemKey::VehicleAudioStatus::IsSelectedItem);
   tBool bRetVal = oVehicleAudioStatus.SendUpdate(true);
   if (true == bRetVal)
   {
      ETG_TRACE_USR4(("PhoneCallModel : OnUpdVehicleAudioSource() "
                      "Vehicle audio status Updated"));
   }
   else
   {
      ETG_TRACE_ERR(("PhoneCallModel : OnUpdVehicleAudioSource(), Error in "
                     "Vehicle audio status update"));
   }
}


/************************************************************************
*FUNCTION: PhoneCallModel::OnUpdNetworkOperatorName
*DESCRIPTION: Invoked by client when there is a change in Network operator name
*PARAMETER:	strNetworkOperator-> operator name

*RETURNVALUE: 	void
*HISTORY:
*18.01.2018 Rev. 0.1 shwetha K
* 		  Initial Revision.
*19.07.2019 Rev 0.2 Sai Prabha Bug Id : 454189,509066

************************************************************************/
void PhoneCallModel::OnUpdNetworkOperatorName(std::string acNetworkOperator)
{
   ETG_TRACE_USR4(("PhoneCallModel : OnUpdNetworkOperatorName() name =%s", acNetworkOperator.c_str()));
   /* Define vector values for normal and extended "devicename/no phone connected/bluetooth off" text node as to dynamically change the maximum size of text node */
   Candera::Vector2 textNodesizeNormal(DEVNAME_TEXTNODE_NORMAL_MAXSIZE_X, DEVNAME_TEXTNODE_NORMAL_MAXSIZE_Y);
   Candera::Vector2 textNodesizeExtended(DEVNAME_TEXTNODE_EXTENDED_MAXSIZE_X, DEVNAME_TEXTNODE_EXTENDED_MAXSIZE_Y);
   /* By default, set the normal text node x-y maximum values*/
   (*oOperatorName).mTextNodeMaxSize = textNodesizeNormal;
#ifdef VARIANT_S_FTR_ENABLE_BLUETOOTH_ON_OFF
   if (ServiceBT::poGetInstance()->GetBluetoothOnOffStatus() == true)
   {
#endif
      /*update network operator name only if the device is connected*/
      if (ServiceBT::poGetInstance() && (0 != ServiceBT::poGetInstance()->GetNoOfHFPDevicesConnected()))
      {
         if (BTPHN_C_EQUAL != acNetworkOperator.compare(""))
         {
            std::string strOperatorname = acNetworkOperator.c_str();
            if (('"' == strOperatorname.at(0)) && ('"' == strOperatorname.at(strOperatorname.length())))
            {
               acNetworkOperator = strOperatorname.substr(1, strOperatorname.length() - 1);
            }
            ETG_TRACE_USR2(("PhoneCallModel::OnUpdNetworkOperatorName: %s", acNetworkOperator.c_str()));
            (*oOperatorName).mOperatorName = acNetworkOperator.c_str();
         }
         else
         {
            ETG_TRACE_USR2(("PhoneCallModel::Invalid NetworkOperator name"));
            (*oOperatorName).mOperatorName = "";// bug Id : 454189,509066 //ebg1kor
         }
      }
      else
      {
         ETG_TRACE_USR2(("ServiceTel:: No BT device name is connected"));
         (*oOperatorName).mOperatorName = " ";
         /* Check for overall HFP and AVP status to update device status */
         OnUpdDeviceConnectionStatus();
      }
#ifdef VARIANT_S_FTR_ENABLE_BLUETOOTH_ON_OFF
   }

   else if (ServiceBT::poGetInstance()->GetBluetoothOnOffStatus() == false)
   {
      ETG_TRACE_USR2(("PhoneCallModel::Bluetooth OFF"));
      (*oOperatorName).mOperatorName = Candera::String(TextId_0x170F);//"Bluetooth off";
      /* On bluetooth off, set the extended text node x-y maximum values*/
      (*oOperatorName).mTextNodeMaxSize = textNodesizeExtended;
   }
#endif

   oOperatorName.MarkAllItemsModified();
   tBool bRetVal = oOperatorName.SendUpdate(true);
   if (true == bRetVal)
   {
      ETG_TRACE_USR4(("PhoneCallModel : OnUpdNetworkOperatorName() "
                      "network operator status Updated"));
   }
   else
   {
      ETG_TRACE_ERR(("PhoneCallModel : OnUpdNetworkOperatorName(), Error in "
                     "network operator status update"));
   }
}


/************************************************************************
*FUNCTION: PhoneCallModel::OnUpdDeviceConnectionStatus
*DESCRIPTION: To process if there is a overall change in HFP/AVP status
*PARAMETER:	None
*RETURNVALUE: 	void
*HISTORY:28-01-2020; Palani K;
************************************************************************/
void PhoneCallModel::OnUpdDeviceConnectionStatus()
{
   ETG_TRACE_USR4(("PhoneCallModel : OnUpdDeviceConnectionStatus"));
   /* Retrieving text from textDB for no phone connected text */
   const Candera::String NO_PHONE_CONNECTED_STR          = LANGUAGE_STRING(TextId_0x0EED, "No phone connected.");
   /* Define vector values for extended "no phone connected" text node as to dynamically change the maximum size of text node */
   Candera::Vector2 textNodesizeExtended(DEVNAME_TEXTNODE_EXTENDED_MAXSIZE_X, DEVNAME_TEXTNODE_EXTENDED_MAXSIZE_Y);
#ifdef VARIANT_S_FTR_ENABLE_BLUETOOTH_ON_OFF
   if (ServiceBT::poGetInstance()->GetBluetoothOnOffStatus() == true)
   {
#endif
      if (ServiceBT::poGetInstance())
      {
         /* Update "No phone connected" when there is no device connected with either HFP or with AVP */
         if ((BTPHN_C_DEFAULT_VALUE_ZERO == ServiceBT::poGetInstance()->GetNoOfHFPDevicesConnected()) &&
               (BTPHN_C_DEFAULT_VALUE_ZERO == ServiceBT::poGetInstance()->u8GetAVPDevHandle()))
         {
            ETG_TRACE_USR4(("PhoneCallModel:: No phone connected"));
            (*oOperatorName).mOperatorName = NO_PHONE_CONNECTED_STR;
            /* On no phone connected, set the extended text node x-y maximum values*/
            (*oOperatorName).mTextNodeMaxSize = textNodesizeExtended;
         }
         /* Show empty device name when AVP alone enabled, as it doesn't get device name */
         else if ((BTPHN_C_DEFAULT_VALUE_ZERO == ServiceBT::poGetInstance()->GetNoOfHFPDevicesConnected()) &&
                  (BTPHN_C_DEFAULT_VALUE_ZERO != ServiceBT::poGetInstance()->u8GetAVPDevHandle()))
         {
            ETG_TRACE_USR4(("PhoneCallModel:: Only AVP connected"));
            (*oOperatorName).mOperatorName = " ";
         }
         else
         {
         }
      }
      oOperatorName.MarkAllItemsModified();
      tBool bRetVal = oOperatorName.SendUpdate(true);
      if (true == bRetVal)
      {
         ETG_TRACE_USR4(("PhoneCallModel : OnUpdDeviceConnectionStatus() "
                         "Device connection status Updated"));
      }
      else
      {
         ETG_TRACE_ERR(("PhoneCallModel : OnUpdDeviceConnectionStatus(), Error in "
                        "Device connection status update"));
      }
#ifdef VARIANT_S_FTR_ENABLE_BLUETOOTH_ON_OFF
   }
#endif
}


/************************************************************************
*FUNCTION: PhoneCallModel::OnUpdCallDuration
*DESCRIPTION: Invoked by client when there is a change in call duration to be displayed on home screen
*RETURNVALUE: 	void
*HISTORY:
*18.01.2018 Rev. 0.1 shwetha K
* 		  Initial Revision.
************************************************************************/
void PhoneCallModel::OnUpdCallDuration(std::string sCallDuration)
{
   /*update null when no device is connected*/
   //if (ServiceBT::poGetInstance() && (FALSE == ServiceBT::poGetInstance()->bGetDeviceConnectedInfo()))
   {
      (*oCallProgress).mCallDuration = sCallDuration.c_str();
      oCallProgress.MarkItemModified
      (ItemKey::CallProgress::CallDurationItem);
      tBool bRetVal = oCallProgress.SendUpdate(true);
      if (true == bRetVal)
      {
         ETG_TRACE_USR4(("PhoneCallModel : OnUpdCallDuration Updated"));
      }
      else
      {
         ETG_TRACE_ERR(("PhoneCallModel : OnUpdNetworkOperatorName(), Error in updation"));
      }
   }
}


/************************************************************************


*FUNCTION: 	   bUpdateSRIconStatusMsg
*DESCRIPTION:  Updates Icon in SIRI
*PARAMETER:
*              corfoMsg:	(->I)
*                 Courier message from the Message Router
*RETURNVALUE: 	BOOL
*
*HISTORY:
*19.06.14 Rev. 0.1 shwetha K
* 		  Initial Revision.
************************************************************************/
tVoid PhoneCallModel::OnUpdSRIconStatus(tBool bVRIconToGUI)
{
   ETG_TRACE_USR4(("BTSiriModel : OnUpdSRIconStatusMsg\n"));

   if (bVRIconToGUI)
   {
      (*oBTVRIconDataBindingSourceIndex).mIconSRStatusIndex = 1;
   }
   else
   {
      (*oBTVRIconDataBindingSourceIndex).mIconSRStatusIndex = 0;
   }

   /* Mark all items for updation */
   oBTVRIconDataBindingSourceIndex.MarkAllItemsModified();
   tBool bDataUpdated = oBTVRIconDataBindingSourceIndex.SendUpdate(true);

   if (false == bDataUpdated)
   {
      ETG_TRACE_USR4(("BTSiriModel : bUpdateSRIconStatusMsg, Data updation failed"));
   }
   else
   {
      ETG_TRACE_USR4(("BTSiriModel : bUpdateSRIconStatusMsg, Data updation Successfully"));
   }
}


}
}
