/* ***************************************************************************************
* FILE:          CallHistoryModel.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  CallHistoryModel.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2017-2020 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#include "gui_std_if.h"

#include "util/StrUtf8.h"
#include "AppHmi_PhoneStateMachine.h"
#include "AppHmi_PhoneConstants.h"
#endif

#include "CallHistoryModel.h"
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#include "../PhoneBook/ServicePhonebook.h"
#include "hmi_trace_if.h"
#include <iostream>
#include <sstream>
#ifdef WINDOWS_SIMULATION
#include <iostream>
#include "AppHmi_PhoneStateMachineSEMLibB.h"
#endif

#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_PHONE_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_FILE_PREFIX            App::Core::PhoneCallHistoryModel::
#include "trcGenProj/Header/CallHistoryModel.cpp.trc.h"
#endif

#endif
using namespace std;
namespace App {
namespace Core {

CallHistoryModel* CallHistoryModel::m_poCallHistoryModel = NULL;

/************************************************************************
*FUNCTION: 	   CallHistoryModel
*DESCRIPTION:  Constructor
*PARAMETER: NA
*
*RETURNVALUE: 	NA
*
*HISTORY:
15/09/2017 Shwetha K
Initial Revision

************************************************************************/
CallHistoryModel::CallHistoryModel()
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   : u8BTLinkLost(0),
     u8BatteryIndicator(0),
     u8RoamingStatus(0),
     u8SignalStrength(0),
     bRetVal(false),
     bRoamingStatus(false)
#endif
{
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR4(("CallHistoryModel : CallHistoryModel()"));
   ListRegistry::s_getInstance().addListImplementation(LISTID_CALLLIST, this);
#endif
}


/************************************************************************
*FUNCTION: 	   ~CallHistoryModel
*DESCRIPTION:  Destructor
*PARAMETER: NA
*
*RETURNVALUE: 	NA
*
*HISTORY:
15/09/2017 Shwetha K
Initial Revision

************************************************************************/
CallHistoryModel::~CallHistoryModel()
{
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR4(("BTCallHistoryModel : ~CallHistoryModel()"));
   ListRegistry::s_getInstance().removeListImplementation(LISTID_CALLLIST);
#endif
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
bool CallHistoryModel::Init()
{
   return true;
}


/**
* @brief   : Incomming message for HMI data model a message component
* @param   :
//*/
bool CallHistoryModel::onCourierMessage(const Courier::StartupMsg& /*oMsg*/)
{
   (*m_ListDataSize).mCallListSize = BTPHN_C_VALUE_ZERO;
   (*m_ListID).mCallListId = LISTID_CALLLIST;
   m_ListID.MarkAllItemsModified();
   m_ListID.SendUpdate(true);
   m_ListDataSize.MarkItemModified(ItemKey::ListDataSize::CallListSizeItem);
   m_ListDataSize.SendUpdate(true);

   return true;
}


/**
* This function creates ListDataProviderBuilder and fills Call history list items to it.
* Each row in call history list includes Call Type, Contact name, call count, call date in the same order.
* Once all the data is filled to list builder object, it posts ListDateProviderResMsg courier message by passing a shared pointer to lister builder object.
* This courier message is handled by framework to update the list
* @param[in] - arCallListSlice - call history list item
*
* @param[out]- returns void
* System Requirements : (SWFL-305, SWFL-306)
initial rev:Shwetha K
*/
void CallHistoryModel::OnUpdSetCallListElement(std::vector<btphn_trCallHistoryInfo>& arCallListSlice,
      tU16 u16CallListSize, tU16 u16StartIndex_CallList, tU16 u16lastItemIndex_CallList)
{
   ETG_TRACE_USR4(("CallHistoryModel :: OnUpdSetCallListElement"));
   char callcount_string[5];
   const Candera::String CALL_LIST_UNKNOWN_STR          = LANGUAGE_STRING(TextId_0x0ECB, "Unknown");

   if (u16CallListSize > BTPHN_C_CALL_LIST_MAX_NUMBER_SIZE)
   {
      u16CallListSize = BTPHN_C_CALL_LIST_MAX_NUMBER_SIZE;
   }
   OnUpdCallListSizeToModel(u16CallListSize);
   ListDataProviderBuilder listBuilder(LISTID_CALLLIST);
   CallList_ListItemData ListData[u16lastItemIndex_CallList + 1];

   tU8 dataIndex = 0;
   std::string sName;
   most_PhonBk_fi_types::T_e8_PhonBkPreferredPhoneBookSortOrder sortorder = T_e8_PhonBkPreferredPhoneBookSortOrder__e8PREFERRED_SORT_ORDER_FIRSTNAME;

   if (ServicePhoneBook::poGetInstance() != NULL)
   {
      sortorder = ServicePhoneBook::poGetInstance()->enGetSortOrder(); //temporary untill P_OPT scene implementation
   }
   tU8 u8CallListSize = arCallListSlice.size();
   if (u8CallListSize == 0)
   {
      ETG_TRACE_FATAL(("CallHistoryModel :: arCallListSlice.size() = 0 "));
      return;
   }

   /*logic to find index of last item to be updated into call history list*/
   ETG_TRACE_USR4(("CallHistoryModel :: OnUpdSetCallListElemente= %d ", u8CallListSize));

   /*uncomment for testing*/
   //std::cout << std::endl << m_StartIndex_CallList << std::endl << m_WindowSize_CallList << std::endl << lastItemIndex_CallList << std::endl << arCallListSlice.size();

   /*loop from starting index till last index to fill CallList data elements- call type icon*/

   for (tU8 u8Index = 0, dataIndex = u16StartIndex_CallList;
         ((u8Index < u8CallListSize) && (dataIndex <= u16lastItemIndex_CallList) && (u8Index < u16CallListSize));
         u8Index++, dataIndex++)
   {
      /*invoking GetCallInfoIcon function to obtain call type*/
      ListData[dataIndex].mCallInfoIcon = GetCallInfoIcon(arCallListSlice[u8Index].enCallHistoryType);

      if ((arCallListSlice[u8Index].copcLastName.empty()) && (arCallListSlice[u8Index].copcFirstName.empty()))
      {
         sName = arCallListSlice[u8Index].copcPhoneNumber;
      }
      /*invoking GetContactname fuction to obtain contact name  */
      else
      {
         sName = AppHmiPhone_Helper::GetContactname(arCallListSlice[u8Index].copcFirstName,
                 arCallListSlice[u8Index].copcLastName, sortorder);
      }

      /* If the received caller name is empty show Unknown string otherwise show the received valid string */
      if (sName.empty())
      {
         ListData[dataIndex].mCallerName = CALL_LIST_UNKNOWN_STR;
      }
      else
      {
         ListData[dataIndex].mCallerName = sName.c_str();
      }
      //conversion from integer to string type
      stringstream ss_CallCount_str;
      ss_CallCount_str << arCallListSlice[u8Index].u8CallCount;
      string CallCount_str = ss_CallCount_str.str();
      memset(callcount_string, 0, sizeof(callcount_string));
      snprintf(callcount_string, sizeof(callcount_string), "%1.0d", arCallListSlice[u8Index].u8CallCount);
      ETG_TRACE_FATAL(("CallHistoryModel :: callcount_string:%s", callcount_string));
      ListData[dataIndex].mCallNumberOfTimes = callcount_string;
      ListData[dataIndex].mCallDate = arCallListSlice[u8Index].CallHistorydate.c_str();

      /*add each row items to the list*/
      listBuilder.AddItem(dataIndex, 0, "Layer_CallListEntry_1").AddDataBindingUpdater<CallList_ListItemDataBindingSource>(ListData[dataIndex]);
   }
   /*Courier message to base class to update the list*/
   tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider(u16StartIndex_CallList, u16CallListSize);
   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
}


#endif
/**
* This function assigns correct index values to each of the call list type.
* @param[in] - enCallHistoryType - call type value : 0- CCH, 1- MCH, 2- ICH, 3-OCH
*
* @param[out]- returns CallType- integer value of the index
initial rev:Shwetha K
*/
tU8 CallHistoryModel::GetCallInfoIcon(btphn_tenCallHistoryType enCallHistoryType)
{
   tU8 CallType = INVALID_CALLHISTORYICONTYPE;
//	ETG_TRACE_USR4((" Call Type = %d", ETG_CENUM(btphn_tenCallHistoryType, (btphn_tenCallHistoryType)enCallHistoryType)));
   switch (enCallHistoryType)
   {
      case 0:
         CallType = 0;//index value 0= CCH
         break;
      case 1:
         CallType = 3;//index value 3= MCH
         break;
      case 2:
         CallType = 1;//index value 1= ICH
         break;
      case 3:
         CallType = 2;//index value 2= OCH
         break;
   }
   return CallType;
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
/**
* This function updates BTTeltales databinding properties with correct index values
* It updates Icon_PhoneStatus based on link lost status,battery indicator status,roaming status and signal strength
* @param[in] - void
*
* @param[out]- returns bool; true if databinding update is successfully, otherwise false
* initial rev:Shwetha K
* Rev 0.1 Sai Prabha ,452238: Battery status and Network Strength is not observed on Native HMI and outgoing call page
*/
bool CallHistoryModel::bUpdateBTTelltaleUpdate()
{
   ETG_TRACE_USR4(("CallHistoryModel : bUpdateBTTelltaleUpdate()"));

   bool bRetVal = false;
   /*logic used to update Icon_PhoneStatus*/
   UInt8 u8StausIndicator = (u8BTLinkLost > u8BatteryIndicator) ? u8BTLinkLost : u8BatteryIndicator;
   ETG_TRACE_USR4(("CallHistoryModel Status u8BatteryIndicator : %d", u8StausIndicator));

   //u8StausIndicator = (u8StausIndicator > u8RoamingStatus) ? u8StausIndicator : u8RoamingStatus;
   //ETG_TRACE_USR4(("CallHistoryModel Status u8RoamingStatus : %d", u8StausIndicator));
   /* bug 977293 : In this function roaming status need not to be verified as it will be taken care in function OnUpdCellNetworkStatus().
                   Also here if raoming status check is enabled then when the roaming is true the status indicator value will be 6(No image) where the databinding
   			   will be updated wrongly.i.e., actual signal strength value will not be updated */

   u8StausIndicator = (u8StausIndicator > u8SignalStrength) ? u8StausIndicator : u8SignalStrength;
   ETG_TRACE_USR4(("CallHistoryModel Status u8StausIndicator : %d", u8StausIndicator));

   switch (u8StausIndicator)
   {
      case BTPHN_C_HEXA_0X01F:
      {
         (*oBTTelltale).mImagePath = 3;//index value 3-icon_phonestatus_bt_lost
         (*oRoamingenable).mEnabled = false;
         ETG_TRACE_USR4(("index value 3-icon_phonestatus_bt_lost"));
         break;
      }
      case BTPHN_C_HEXA_0X00F:
      {
         (*oBTTelltale).mImagePath = 5;//index value 5-icon_phonestatus_battery_0
         (*oRoamingenable).mEnabled = false;
         ETG_TRACE_USR4(("index value 5-icon_phonestatus_battery_0"));
         break;
      }
      case BTPHN_C_HEXA_0X00E:
      {
         (*oBTTelltale).mImagePath = 4;//index value 4-icon_phonestatus_battery_1
         (*oRoamingenable).mEnabled = false;
         ETG_TRACE_USR4(("index value 4-icon_phonestatus_battery_1"));
         break;
      }
      case BTPHN_C_HEXA_SIX:
      {
         (*oBTTelltale).mImagePath = 6;//index value 6-icon_phonestatus_reception_roaming_1
         (*oRoamingenable).mEnabled = true;
         ETG_TRACE_USR4(("index value 6-icon_phonestatus_reception_roaming_1"));
         break;
      }
      case BTPHN_C_HEXA_FIVE:
      {
         (*oBTTelltale).mImagePath = 2;//index value 2-icon_phonestatus_signalstrength_5
         (*oRoamingenable).mEnabled = true;
         ETG_TRACE_USR4(("index value 2-icon_phonestatus_signalstrength_5"));
         break;
      }
      case BTPHN_C_HEXA_FOUR:
      {
         (*oBTTelltale).mImagePath = 1;//index value 1-icon_phonestatus_signalstrength_4
         (*oRoamingenable).mEnabled = true;
         ETG_TRACE_USR4(("index value 1-icon_phonestatus_signalstrength_4"));
         break;
      }
      case BTPHN_C_HEXA_THREE:
      {
         (*oBTTelltale).mImagePath = 0;//index value 0-icon_phonestatus_signalstrength_3
         (*oRoamingenable).mEnabled = true;
         ETG_TRACE_USR4(("index value 0-icon_phonestatus_signalstrength_3"));
         break;
      }
      case BTPHN_C_HEXA_TWO:
      {
         (*oBTTelltale).mImagePath = 10;//index value 10-icon_phonestatus_signalstrength_2
         (*oRoamingenable).mEnabled = true;
         ETG_TRACE_USR4(("index value 10-icon_phonestatus_signalstrength_2"));
         break;
      }
      case BTPHN_C_HEXA_ONE:
      {
         (*oBTTelltale).mImagePath = 9;//index value 9-icon_phonestatus_signalstrength_1
         (*oRoamingenable).mEnabled = true;
         ETG_TRACE_USR4(("index value 9-icon_phonestatus_signalstrength_1"));
         break;
      }
      case BTPHN_C_HEXA_ZERO:
      {
         (*oBTTelltale).mImagePath = 8;//index value 8-icon_phonestatus_signalstrength_0
         (*oRoamingenable).mEnabled = true;
         ETG_TRACE_USR4(("index value 8-icon_phonestatus_signalstrength_0"));
         break;
      }
      default:
      {
         break;
      }
   }
   // ETG_TRACE_USR4(("btphn_tclBTPhoneModel Status Imagepath : %s", (*oBTTelltale).mImagePath.GetCString()));
   oRoamingenable.MarkAllItemsModified();
   oRoamingenable.SendUpdate(true);
   (*oBTTelltale).mLoadAssetBitmap = true;
   oBTTelltale.MarkAllItemsModified();
   bRetVal = oBTTelltale.SendUpdate(true);

   return bRetVal;
}


/**
* This function resets the flag u8BatteryIndicator so that UI updates when active/passive device toggle happens.
* it is called when the ActiveDeviceResult is obtained from the Service
*
* @param[out]- returns void
initial rev: Sai Prabha, Bug Fix :280286: Battery seen on connecting phone
*/
void CallHistoryModel ::SetBatteryIndicator()
{
   u8BatteryIndicator = 0;
}


/**
* This function updates  PhoneStatusDataBindingSource properties with appropriate index value
*Iit is updated based on the battery status of the phone which is received from service
* @param[in] - stateofcharge- battery status obtained from client handler
*
* @param[out]- returns void
initial rev: Sai Prabha, Bug Fix :280286: Battery seen on connecting phone
*/
void CallHistoryModel::OnUpdCellStateOfCharge(tU8 stateofcharge)
{
//Requirement specifies that the battery be shown only when it has one bar, or very less
   switch (stateofcharge)
   {
      case BTPHN_C_HEXA_ONE:
      {
         u8BatteryIndicator = BTPHN_C_HEXA_0X00E;
         (*oPhoneStatus).mBatteryStatus = 1;//index value 4-icon_phonestatus_battery_1
         (*oPhoneStatus).mBatteryStatusinGadget = 1;
         ETG_TRACE_USR4(("index value 4-icon_phonestatus_battery_1"));
         break;
      }
      case BTPHN_C_HEXA_TWO:
      {
         (*oPhoneStatus).mBatteryStatusinGadget = 2;//index value 3-icon_phonestatus_battery_2
         ETG_TRACE_USR4(("index value 3-icon_phonestatus_battery_2"));
         break;
      }
      case BTPHN_C_HEXA_THREE:
      {
         (*oPhoneStatus).mBatteryStatusinGadget = 3;//index value 2-icon_phonestatus_battery_3
         ETG_TRACE_USR4(("index value 2-icon_phonestatus_battery_3"));
         break;
      }
      case BTPHN_C_HEXA_FOUR:
      {
         (*oPhoneStatus).mBatteryStatusinGadget = 4;//index value 1-icon_phonestatus_battery_4
         ETG_TRACE_USR4(("index value 1-icon_phonestatus_battery_4"));
         break;
      }
      case BTPHN_C_HEXA_FIVE:
      {
         (*oPhoneStatus).mBatteryStatusinGadget = 4;//index value 0-icon_phonestatus_battery_5
         ETG_TRACE_USR4(("index value 0-icon_phonestatus_battery_5"));
         break;
      }
      case BTPHN_C_HEXA_ZERO:
      {
         u8BatteryIndicator = BTPHN_C_HEXA_0X00F;
         (*oPhoneStatus).mBatteryStatus = 0;//index value 5-icon_phonestatus_battery_0
         (*oPhoneStatus).mBatteryStatusinGadget = 0 ;
         ETG_TRACE_USR4(("index value 5-icon_phonestatus_battery_0"));
         break;
      }
      default:
      {
         ETG_TRACE_USR4(("Battery Level Unknown"));
         (*oPhoneStatus).mBatteryStatus = 6;//default icon
         (*oPhoneStatus).mBatteryStatusinGadget = 6;
         break;
      }
   }
   bRetVal = bUpdateBTTelltaleUpdate();
   oPhoneStatus.MarkItemModified(ItemKey::PhoneStatus::BatteryStatusItem);
   oPhoneStatus.MarkItemModified(ItemKey::PhoneStatus::BatteryStatusinGadgetItem);

   bRetVal = bRetVal && oPhoneStatus.SendUpdate(true);
   if (false == bRetVal)
   {
      /*ETG_TRACE_ERR(("btphn_tclBTPhoneModel : cobBatteryLevelResp(), Error in "
      	"Battery Level Updated"));*/
   }
}


/**
* This function updates  signal status to GUI
*I
* @param[in] - CellSignalStrength- signal status received from client
*
* @param[out]- returns void
* initial rev:Sai Prabha Bug Fix : 272500: Signal Strength not correct in Home screen - Phone tile
*/
void CallHistoryModel::OnUpdCellSignalStrength(tU8 CellSignalStrength)
{
   u8SignalStrength = CellSignalStrength;
   bRetVal = bUpdateBTTelltaleUpdate();
   ETG_TRACE_USR4(("Network Signal Level: %d", u8SignalStrength));
   switch (u8SignalStrength)
   {
      case BTPHN_C_HEXA_ZERO:
      {
         (*oPhoneStatus).mSignalStatus = 0;//icon_signalstrength_1
         ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength - 0"));
         break;
      }
      case BTPHN_C_HEXA_ONE:
      {
         (*oPhoneStatus).mSignalStatus = 1;//icon_signalstrength_1
         ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength - 1"));
         break;
      }
      case BTPHN_C_HEXA_TWO:
      {
         (*oPhoneStatus).mSignalStatus = 2;//icon_signalstrength_2
         ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength-2"));
         break;
      }
      case BTPHN_C_HEXA_THREE:
      {
         (*oPhoneStatus).mSignalStatus = 3;//icon_signalstrength_3
         ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength-3"));
         break;
      }
      case BTPHN_C_HEXA_FOUR:
      {
         (*oPhoneStatus).mSignalStatus = 4;//icon_signalstrength_4
         ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength-4"));
         break;
      }
      case BTPHN_C_HEXA_FIVE:
      {
         (*oPhoneStatus).mSignalStatus = 5;//icon_signalstrength_5
         ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength-5"));
         break;
      }
      default:
      {
         (*oPhoneStatus).mSignalStatus = 12;//icon_signalstrength_0
         ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength-6"));
         break;
      }
   }
   if (bRoamingStatus)
   {
      switch (u8SignalStrength)
      {
         case BTPHN_C_HEXA_ZERO:
         {
            (*oPhoneStatus).mSignalStatusinGadget = 0;//icon_signalstrength_1
            ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength - 0"));
            break;
         }
         case BTPHN_C_HEXA_ONE:
         {
            (*oPhoneStatus).mSignalStatusinGadget = 11;//icon_signalstrength_1
            ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength - 6"));
            break;
         }
         case BTPHN_C_HEXA_TWO:
         {
            (*oPhoneStatus).mSignalStatusinGadget = 7;//icon_signalstrength_2
            ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength-7"));
            break;
         }
         case BTPHN_C_HEXA_THREE:
         {
            (*oPhoneStatus).mSignalStatusinGadget = 8;//icon_signalstrength_3
            ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength-8"));
            break;
         }
         case BTPHN_C_HEXA_FOUR:
         {
            (*oPhoneStatus).mSignalStatusinGadget = 9;//icon_signalstrength_4
            ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength-9"));
            break;
         }
         case BTPHN_C_HEXA_FIVE:
         {
            (*oPhoneStatus).mSignalStatusinGadget = 10;//icon_signalstrength_5
            ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength-10"));
            break;
         }
         default:
         {
            (*oPhoneStatus).mSignalStatusinGadget = 12;//icon_signalstrength_0
            ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength-0"));
            break;
         }
      }
   }
   else
   {
      switch (u8SignalStrength)
      {
         case BTPHN_C_HEXA_ZERO:
         {
            (*oPhoneStatus).mSignalStatusinGadget = 0;//icon_signalstrength_1
            ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength - 0"));
            break;
         }
         case BTPHN_C_HEXA_ONE:
         {
            (*oPhoneStatus).mSignalStatusinGadget = 1;//icon_signalstrength_1
            ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength - 1"));
            break;
         }
         case BTPHN_C_HEXA_TWO:
         {
            (*oPhoneStatus).mSignalStatusinGadget = 2;//icon_signalstrength_2
            ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength-2"));
            break;
         }
         case BTPHN_C_HEXA_THREE:
         {
            (*oPhoneStatus).mSignalStatusinGadget = 3;//icon_signalstrength_3
            ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength-3"));
            break;
         }
         case BTPHN_C_HEXA_FOUR:
         {
            (*oPhoneStatus).mSignalStatusinGadget = 4;//icon_signalstrength_4
            ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength-4"));
            break;
         }
         case BTPHN_C_HEXA_FIVE:
         {
            (*oPhoneStatus).mSignalStatusinGadget = 5;//icon_signalstrength_5
            ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength-5"));
            break;
         }
         default:
         {
            (*oPhoneStatus).mSignalStatusinGadget = 12;//icon_signalstrength_0
            ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength-6"));
            break;
         }
      }
   }
   oPhoneStatus.MarkItemModified(ItemKey::PhoneStatus::SignalStatusItem);

   bRetVal = bRetVal && oPhoneStatus.SendUpdate(true);
   if (false == bRetVal)
   {
      ETG_TRACE_USR4(("CallHistoryModel -OnUpdCellSignalStrength icon update error"));
   }
}


/**
* This function updates  roaming and network status to GUI
*
* @param[in] - NetworkServicestate- network state received from client
RoamingStatus- roaming state received from client
*
* @param[out]- returns void
initial rev: Sai Prabha, Bug Fix :272500: Signal Strength not correct in Home screen - Phone tile
*/
void CallHistoryModel::OnUpdCellNetworkStatus(btphn_tenNetworkServiceState NetworkServicestate, bool RoamingStatus)
{
   bRoamingStatus = RoamingStatus;

   if (TRUE == bRoamingStatus)
   {
      u8RoamingStatus = BTPHN_C_HEXA_SIX; //change for list MBSP
      ETG_TRACE_ERR(("btphn_tclBTPhoneModel : OnUpdCellNetworkStatus()- %d", u8RoamingStatus));
   }
   else
   {
      u8RoamingStatus = BTPHN_C_HEXA_ZERO;
   }
   (*oPhoneStatus).mNetworkStatus = bRoamingStatus;
   oPhoneStatus.MarkItemModified(ItemKey::PhoneStatus::NetworkStatusItem);
   bRetVal = oPhoneStatus.SendUpdate(true);

   bRetVal = bRetVal && bUpdateBTTelltaleUpdate();
   if (false == bRetVal)
   {
      ETG_TRACE_ERR(("btphn_tclBTPhoneModel : cobNetworkStatusResp(), Error in "
                     "Network Roaming Status Updated"));
   }
   else
   {
      ETG_TRACE_ERR(("btphn_tclBTPhoneModel : cobNetworkStatusResp(), NetworkStatus "
                     "NULL pointer"));
   }
}


/**
* This function updates  CAll history list size and is list empty data binding values based on the list size received from client
*I
* @param[in] - corfu16TotalListSize- call history list size
*
* @param[out]- returns void
* System Requirements : (SWFL-305, SWFL-306)
initial rev:Shwetha K
*/
void CallHistoryModel::OnUpdCallListSizeToModel(tU16 corfu16TotalListSize)
{
   ETG_TRACE_USR4(("CallHistoryModel : OnUpdCallListSizeToModel corfu16TotalListSize: %d ", corfu16TotalListSize));
   /*update call hisory list size*/
   (*m_ListDataSize).mCallListSize = corfu16TotalListSize;
   m_ListDataSize.MarkItemModified(ItemKey::ListDataSize::CallListSizeItem);
   m_ListDataSize.SendUpdate(true);
   /*update list empty status */
   if (BTPHN_C_VALUE_ZERO == corfu16TotalListSize)
   {
      //vClearCallListinGUI();
      ETG_TRACE_USR4(("CallHistoryModel : OnUpdCallListSizeToModel: update visible "));
      (*oIsListNotempty).mCallList = BTPHN_C_VALUE_ONE;
   }
   else
   {
      ETG_TRACE_USR4(("CallHistoryModel : OnUpdCallListSizeToModel: update invisible "));
      (*oIsListNotempty).mCallList = BTPHN_C_VALUE_ZERO;
   }
   oIsListNotempty.MarkAllItemsModified();
   oIsListNotempty.SendUpdate(true);
}


/**
* This function sends empty call list to be displayed
*I
* @param[in] - void
*
* @param[out]- returns void
initial rev:Shwetha K
*/
void CallHistoryModel::vClearCallListinGUI()
{
   ETG_TRACE_USR4(("CallHistoryModel : vClearCallListinGUI "));
   ListDataProviderBuilder listBuilder(LISTID_CALLLIST);
   tSharedPtrDataProvider dataProvider = listBuilder.CreateDataProvider();
   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
}


/**
* This function responsible to diaplay miss call icon when there is a misscall for the connected device phone
*
* @param[in] - u32NumOfMissedCall -> number of misscall for the connected device
*
* @param[out]- returns void
* initial rev:Sai Prabha
* Bug Fix: 547937: Battery and Network symbol appears/disappear on toggling phone icon button present on phone tile in Home screen
*/
void CallHistoryModel::OnUpdMissedCallIndicator(tU32 u32NumOfMissedCall)
{
   ETG_TRACE_USR4(("CallHistoryModel : OnUpdMissedCallIndicator u32NumOfMissedCall: %d ", u32NumOfMissedCall));
   /*check if device is connected*/
   if ((ServiceBT::poGetInstance()->GetNoOfHFPDevicesConnected() > 0) && (Sm_ActiveSource == 0))
   {
      if (u32NumOfMissedCall > BTPHN_C_VALUE_ZERO)
      {
         (*oPhoneStatus).mMissedCallStatus = 1;
      }
      else
      {
         (*oPhoneStatus).mMissedCallStatus = 0;
      }
   }

   /*default updates*/
   else
   {
      (*oPhoneStatus).mMissedCallStatus = 0;//display blank
      (*oPhoneStatus).mBatteryStatus = 6;// display blank
      (*oPhoneStatus).mSignalStatusinGadget = 6;//display blank
      (*oPhoneStatus).mBatteryStatusinGadget = 6; //display blank
   }
   oPhoneStatus.MarkAllItemsModified();
   oPhoneStatus.SendUpdate(true);
}


#endif
}


}
