/**************************************************************************************
* @file         : <VehicleSrvClient.cpp>
* @author       : <ECG> <INF4CV>
* @addtogroup   : <AppHmi_Phone>
* @brief        :
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "VehicleSrvClient.h"
#include "AppHmi_PhoneStateMachine.h"
#include "ServiceTel.h"
#include "Core/SpeedLockStatusHandler/SpeedLockStatusHandler.h"
#include "Core/Microphone/MicClientHandler.h"
#include "Core/PhoneHall.h"
#define UTFUTIL_S_IMPORT_INTERFACE_GENERIC
#include "utf_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_PHONE_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_PHONE
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_Phone_"
#define ETG_I_FILE_PREFIX                 App::Core::VehicleSrvClient::
#include "trcGenProj/Header/VehicleSrvClient.cpp.trc.h"
#endif


using namespace ::vehicle_main_fi_types;
using namespace ::VEHICLE_MAIN_FI;

static const tU8 e8RCDS_ERROR = 255;

namespace App {
namespace Core {
/**
 * @Constructor
 */

VehicleSrvClient* VehicleSrvClient::_VehicleSrvClient = NULL;
VehicleSrvClient::VehicleSrvClient():
   _vehicleProxy(::VEHICLE_MAIN_FIProxy::createProxy("vehicleMainFiPort", *this)),
   _hmiDataProxy(::HmiDataProxy::createProxy("hmiDataServicePort", *this)), // sys settings TBC
   m_U8BatteryStatus(0),
   m_U8CallDuraErrCode(0),
   m_U8SignalStatus(0),
   m_U8CallStatus(0),
   m_U8DownloadStatus(0),
   m_bRoamStatus(false),
   m_U8errorCodeSNA(PHONEMW_SNA),
   m_u16CallDurationInSec(0),
   m_u8CallHistorySyncState(0),
   m_bPhoneAvailable(false),
   m_bWifiAvailable(false),
   m_bSmartPhoneVRAvailable(false),
   m_u8BTLinkState(0),
   m_u16SpeedValue(0)
{
   ETG_TRACE_USR4(("VehicleSrvClient Constructor is called"));
   if (_vehicleProxy)
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _vehicleProxy->getPortName());
      ETG_TRACE_USR4(("VehicleSrvClient Constructor after creating proxy"));
   }
   // sys settings TBC
   if (_hmiDataProxy)
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _hmiDataProxy->getPortName());
   }
   // sys settings TBC
   LanguageMappingTable [0].LangIndex = 12;
   LanguageMappingTable [0].LangString =  "en_GB" ;
   LanguageMappingTable [1].LangIndex =  23;
   LanguageMappingTable [1].LangString = "de_DE" ;

   LanguageMappingTable [2].LangIndex = 10;
   LanguageMappingTable [2].LangString = "nl_NL" ;
   LanguageMappingTable [3].LangIndex = 21;
   LanguageMappingTable [3].LangString  = "fr_FR" ;
   LanguageMappingTable [4].LangIndex = 29;
   LanguageMappingTable [4].LangString =  "it_IT";
   LanguageMappingTable [5].LangIndex = 46 ;
   LanguageMappingTable [5].LangString = "es_ES" ;
   LanguageMappingTable [6].LangIndex = 39;
   LanguageMappingTable [6].LangString = "pt_PT" ;
   LanguageMappingTable [7].LangIndex = 27;
   LanguageMappingTable [7]. LangString = "hu_HU";
   LanguageMappingTable [8].LangIndex = 42;
   LanguageMappingTable [8].LangString = "ru_RU" ;
   LanguageMappingTable [9].LangIndex = 8;
   LanguageMappingTable [9].LangString = "cs_CZ" ;
   LanguageMappingTable [10].LangIndex = 9;
   LanguageMappingTable [10].LangString  =  "da_DK" ;
   LanguageMappingTable [11].LangIndex = 35;
   LanguageMappingTable [11].LangString =  "nb_NO"  ;
   LanguageMappingTable [12].LangIndex = 38;
   LanguageMappingTable [12].LangString =   "pl_PL" ;
   LanguageMappingTable [13].LangIndex = 49;
   LanguageMappingTable [13].LangString  = "sv_SE";
   LanguageMappingTable [14].LangIndex = 13;
   LanguageMappingTable [14].LangString  =  "en_US" ;
   LanguageMappingTable [15].LangIndex = 24 ;
   LanguageMappingTable [15].LangString = "el_GR" ;
   LanguageMappingTable [16].LangIndex = 52;
   LanguageMappingTable [16].LangString   =   "tr_TR" ;
   LanguageMappingTable [17].LangIndex = 47;
   LanguageMappingTable [17].LangString = "es_MX" ;
   LanguageMappingTable [18].LangIndex = 22;
   LanguageMappingTable [18].LangString  = "fr_CA" ;
}


/**
* @Destructor
*/

VehicleSrvClient::~VehicleSrvClient()
{
   ETG_TRACE_USR4(("VehicleSrvClient destructor is called"));
}


/**
 * @onAvailable
 */
void VehicleSrvClient::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   StartupSync::getInstance().onAvailable(proxy, stateChange);
   ETG_TRACE_USR4(("VehicleSrvClient onAvailable is called"));
   if (_vehicleProxy && (proxy == _vehicleProxy))
   {
      _vehicleProxy->sendLanguageUpReg(*this);
      _vehicleProxy->sendSpeedUpReg(*this);
      _vehicleProxy->sendPhoneCallCommandUpReg(*this);
      _vehicleProxy->sendPhonePwrCmdUpReg(*this);
      _vehicleProxy->sendSMARTVRUpReg(*this);
      ETG_TRACE_USR4(("VehicleSrvClient onAvailable KDS Check for Generic CAN"));
      //KDS supported features check
      vCheckKDSFeaturesAvailable();
   }
   else
   {
      ETG_TRACE_USR4(("VehicleSrvClient onAvailable Vehicle Proxy is not available"));
   }

   if (_hmiDataProxy && (proxy == _hmiDataProxy))
   {
      _hmiDataProxy->sendSpeedLockStatusRegister(*this);
   }
}


/**
 * @onUnavailable
 */
void VehicleSrvClient::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   StartupSync::getInstance().onUnavailable(proxy, stateChange);
   ETG_TRACE_USR4(("VehicleSrvClient onAvailable is called"));
   if (_vehicleProxy && (proxy == _vehicleProxy))
   {
      _vehicleProxy->sendLanguageRelUpRegAll();
      _vehicleProxy->sendSpeedRelUpRegAll();
      _vehicleProxy->sendPhoneCallCommandRelUpRegAll();
      _vehicleProxy->sendPhonePwrCmdRelUpRegAll();
      _vehicleProxy->sendSMARTVRRelUpRegAll();
   }

   if (_hmiDataProxy && (proxy == _hmiDataProxy))
   {
      _hmiDataProxy->sendSpeedLockStatusDeregisterAll();
   }
}


void VehicleSrvClient::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("VehicleSrvClient register property is called"));
   if (_vehicleProxy && (proxy == _vehicleProxy))
   {
      ETG_TRACE_USR4(("VehicleSrvClient register property is called"));
      _vehicleProxy->sendLanguageUpReg(*this);
      _vehicleProxy->sendPhoneCallCommandUpReg(*this);
      _vehicleProxy->sendPhonePwrCmdUpReg(*this);
      _vehicleProxy->sendSMARTVRUpReg(*this);
   }

   if (_hmiDataProxy && (proxy == _hmiDataProxy))
   {
      _hmiDataProxy->sendSpeedLockStatusRegister(*this);
   }
}


void VehicleSrvClient::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("VehicleSrvClient deregister property is called"));
   if (_vehicleProxy && (proxy == _vehicleProxy))
   {
      _vehicleProxy->sendLanguageRelUpRegAll();
      _vehicleProxy->sendPhoneCallCommandRelUpRegAll();
      _vehicleProxy->sendPhonePwrCmdRelUpRegAll();
      _vehicleProxy->sendSMARTVRRelUpRegAll();
   }

   if (_hmiDataProxy && (proxy == _hmiDataProxy))
   {
      _hmiDataProxy->sendSpeedLockStatusDeregisterAll();
   }
}


/*
 * onSetLanguageResult - ASF framework callback which is after SetLanguage
 * @param[in] proxy
 * @param[in] result
 * @param[out] None
 * @return void
 */
void VehicleSrvClient::onSetLanguageResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/,  const ::boost::shared_ptr< SetLanguageResult >& result)
{
   ETG_TRACE_USR4(("onSetLanguageResult received :  %d", result->getResult()));
}


/*
 * onSetLanguageError - ASF framework callback which is after SetLanguage
 * @param[in] proxy
 * @param[in] error
 * @param[out] None
 * @return void
 */
void VehicleSrvClient::onSetLanguageError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< SetLanguageError >& /*error*/)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onSetLanguageError"));
}


/*
 * onLanguageStatus - ASF framework callback which is triggered on every setLanguage()
 * @param[in] proxy
 * @param[in] status
 * @param[out] None
 * @return voidl
 * todo :check As of now, after setLanguage, onLanguageStatus() is received twice from the VD_Vehicle data.
 * First update represents, previously configured Language ID and the second one represents currently configured ID.*/

void VehicleSrvClient::onLanguageStatus(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/,  const ::boost::shared_ptr< LanguageStatus >& status)
{
   std::vector<T_Language_SourceTable> langTable = status->getLangTable();
   uint8 langIndex = status->getLanguage();

   ETG_TRACE_USR4(("onLanguageStatus langIndex=%d", langIndex));
   std::string langauge = getLanguageId(langIndex);

   //COURIER_MESSAGE_NEW(Courier::SetCultureReqMsg)(Courier::ItemId(langauge.c_str()))->Post();
}


std::string VehicleSrvClient::getLanguageId(int index)
{
   std::string langId = "";
   for (int i = 0; i < (sizeof(LanguageMappingTable) / sizeof(LanguageMappingTable[0])); i++)
   {
      if (LanguageMappingTable[i].LangIndex == index)
      {
         ETG_TRACE_USR4(("onLanguageStatus Language string=%s", LanguageMappingTable[i].LangString.c_str()));
         langId = LanguageMappingTable[i].LangString.c_str();
      }
   }
   return langId;
}


/*
 * onLanguageError - ASF framework callback which is triggered if there is any error in setLanguage()
 * @param[in] proxy
 * @param[in] error
 * @param[out] None
 * @return void
 */
void VehicleSrvClient::onLanguageError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< LanguageError >& /*error*/)
{
   ETG_TRACE_USR4(("onLanguageError"));
}


#if 0  // Enable on FI Implementation
void VehicleSrvClient::onCallStatusError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::CallStatusError >& error)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onCallStatusError"));
}


void VehicleSrvClient::onCallStatusResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::CallStatusResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onCallStatusResult"));
}


void VehicleSrvClient::onCallDurationError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::CallDurationError >& error)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onCallDurationError"));
}


void VehicleSrvClient::onCallDurationResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::CallDurationResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onCallDurationResult"));
}


void VehicleSrvClient::onPhone_AvailError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::Phone_AvailError >& error)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onPhone_AvailError"));
}


void VehicleSrvClient::onPhone_AvailResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::Phone_AvailResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onPhone_AvailResult"));
}


void VehicleSrvClient::onGSMFieldstrengthError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::GSMFieldstrengthError >& error)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onGSMFieldstrengthError"));
}


void VehicleSrvClient::onGSMFieldstrengthResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::GSMFieldstrengthResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onGSMFieldstrengthResult"));
}


void VehicleSrvClient::onPhoneBatteryLevelError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhoneBatteryLevelError >& error)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onPhoneBatteryLevelError"));
}


void VehicleSrvClient::onPhoneBatteryLevelResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhoneBatteryLevelResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onPhoneBatteryLevelResult"));
}


#endif

/************************************************
Function 	: onPhoneCallStatusError
Parameters 	:
Description : This function provides an error in case
*             call statusis not sent to CAN
SYSFL/SWFL :
Author     : Bruce Netto 08/02/2021
************************************************/
void VehicleSrvClient::onPhoneCallStatusError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhoneCallStatusError >& error)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onPhoneCallStatusError"));
}


/************************************************
Function 	: onPhoneCallStatusResult
Parameters 	:
Description : This function is an acknowledgement in case
*             call status is sent to CAN
SYSFL/SWFL :
Author     : Bruce Netto 08/02/2021
************************************************/
void VehicleSrvClient::onPhoneCallStatusResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhoneCallStatusResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onPhoneCallStatusResult"));
}


/************************************************
Function 	: onCallDurationError
Parameters 	:
Description : This function provides an error in case
*             call duration is not sent to CAN
SYSFL/SWFL :
Author     : Bruce Netto 02/02/2021
************************************************/
void VehicleSrvClient::onCallDurationError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::CallDurationError >& error)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onCallDurationError"));
}


/************************************************
Function 	: onCallDurationResult
Parameters 	:
Description : This function is an acknowledgement in case
*             call duration is sent to CAN
SYSFL/SWFL :
Author     : Bruce Netto 02/02/2021
************************************************/
void VehicleSrvClient::onCallDurationResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::CallDurationResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onCallDurationResult"));
}


/************************************************
Function 	: onGSMSignalStrengthError
Parameters 	:
Description : This function is an error in case
*             once GSM status is not sent to CAN
SYSFL/SWFL :
Author     : Bruce Netto 02/02/2021
************************************************/
void VehicleSrvClient::onGSMSignalStrengthError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::GSMSignalStrengthError >& error)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onGSMSignalStrengthError"));
}


/************************************************
Function 	: onGSMSignalStrengthResult
Parameters 	:
Description : This function is an acknowledgement in case
*             GSM status is received in CAN
SYSFL/SWFL :
Author     : Bruce Netto 02/02/2021
************************************************/
void VehicleSrvClient::onGSMSignalStrengthResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::GSMSignalStrengthResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onGSMSignalStrengthResult"));
}


/************************************************
Function 	: onPhoneStatusRoamingError
Parameters 	:
Description : This function is an error in case
*             phone roaming status is not sent to CAN
SYSFL/SWFL :
Author     : Bruce Netto 05/02/2021
************************************************/
void VehicleSrvClient::onPhoneStatusRoamingError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhoneStatusRoamingError >& error)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onPhoneStatusRoamingError"));
}


/************************************************
Function 	: onPhoneStatusRoamingResult
Parameters 	:
Description : This function is an acknowledgement in case
*             phone roaming status is sent to CAN
SYSFL/SWFL :
Author     : Bruce Netto 05/02/2021
************************************************/
void VehicleSrvClient::onPhoneStatusRoamingResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhoneStatusRoamingResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onPhoneStatusRoamingResult"));
}


/************************************************
Function 	: onPhoneBatteryLevelStatusError
Parameters 	:
Description : This function is an error in case
*             once battery status is not sent to CAN
SYSFL/SWFL :
Author     : Bruce Netto 05/02/2021
************************************************/
void VehicleSrvClient::onPhoneBatteryLevelStatusError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhoneBatteryLevelStatusError >& error)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onPhoneBatteryLevelStatusError"));
}


/************************************************
Function 	: onPhoneBatteryLevelStatusResult
Parameters 	:
Description : This function is an acknowledgement in case
*             battery status is sent to CAN
SYSFL/SWFL :
Author     : Bruce Netto 02/02/2021
************************************************/
void VehicleSrvClient::onPhoneBatteryLevelStatusResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhoneBatteryLevelStatusResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onPhoneBatteryLevelStatusResult"));
}


/************************************************
Function 	: onPhoneBookSyncstatusError
Parameters 	:
Description : This function is an error in case
*             phone book sync is not sent to CAN
SYSFL/SWFL :
Author     : Bruce Netto 10/02/2021
************************************************/
void VehicleSrvClient::onPhoneBookSyncstatusError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhoneBookSyncstatusError >& error)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onPhoneBookSyncstatusError"));
}


/************************************************
Function 	: onPhoneBookSyncstatusResult
Parameters 	:
Description : This function is an acknowledgement in case
*             phone book sync is sent to CAN
SYSFL/SWFL :
Author     : Bruce Netto 10/02/2021
************************************************/
void VehicleSrvClient::onPhoneBookSyncstatusResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhoneBookSyncstatusResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onPhoneBookSyncstatusResult"));
}


/************************************************
Function 	: onCallHistorySyncStatusError
Parameters 	:
Description : This function is an error in case
*             call history sync is not sent to CAN
SYSFL/SWFL :
Author     : Bruce Netto 10/02/2021
************************************************/
void VehicleSrvClient::onCallHistorySyncStatusError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::CallHistorySyncStatusError >& error)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onCallHistorySyncStatusError"));
}


/************************************************
Function 	: onCallHistorySyncStatusResult
Parameters 	:
Description : This function is an acknowledgement in case
*             call history sync is sent to CAN
SYSFL/SWFL :
Author     : Bruce Netto 10/02/2021
************************************************/
void VehicleSrvClient::onCallHistorySyncStatusResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::CallHistorySyncStatusResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onCallHistorySyncStatusResult"));
}


/************************************************
Function 	: onPhoneBTLinkStatusError
Parameters 	:
Description : This function is an error in case
*             BT Link status is not sent to CAN
SYSFL/SWFL :
Author     : Teena 11/02/2021
************************************************/
void VehicleSrvClient::onPhoneBTLinkStatusError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhoneBTLinkStatusError >& error)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onPhoneBTLinkStatusError"));
}


/************************************************
Function 	: onPhoneBTLinkStatusResult
Parameters 	:
Description : This function is an acknowledgement in case
*             BT Link status is sent to CAN
SYSFL/SWFL :
Author     : Teena 11/02/2021
************************************************/
void VehicleSrvClient::onPhoneBTLinkStatusResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhoneBTLinkStatusResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onPhoneBTLinkStatusResult"));
}


/************************************************
Function 	: onPhoneVRAvailableError
Parameters 	:
Description : This function is an error in case
*             Phone VR Available is not received
SYSFL/SWFL :
Author     : Bruce 09/06/2021
************************************************/
void VehicleSrvClient::onPhoneVRAvailableError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhoneVRAvailableError >& error)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onPhoneVRAvailableError"));
}


/************************************************
Function 	: onPhoneVRAvailableResult
Parameters 	:
Description : This function is an acknowledgement in case
*             Phone VR Available information is sent to CAN
SYSFL/SWFL :
Author     : Bruce 09/06/2021
************************************************/
void VehicleSrvClient::onPhoneVRAvailableResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhoneVRAvailableResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onPhoneVRAvailableResult"));
}


/************************************************
Function 	: onPhoneAvailableStatusError
Parameters 	:
Description : This function is an error in case
*             Phone Available is not received
SYSFL/SWFL :
Author     : Bruce 24/02/2021
************************************************/
void VehicleSrvClient::onPhoneAvailableStatusError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhoneAvailableStatusError >& error)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onPhoneAvailableStatusError"));
}


/************************************************
Function 	: onPhoneAvailableStatusResult
Parameters 	:
Description : This function is an acknowledgement in case
*             Phone Available information is sent to CAN
SYSFL/SWFL :
Author     : Bruce 24/02/2021
************************************************/
void VehicleSrvClient::onPhoneAvailableStatusResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhoneAvailableStatusResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onPhoneAvailableStatusResult"));
}


/************************************************
Function 	: onDeviceStatusWifiAvailableError
Parameters 	:
Description : This function is an error in case
*             WIFI available is not received
SYSFL/SWFL :
Author     : Bruce 24/02/2021
************************************************/
void VehicleSrvClient::onDeviceStatusWifiAvailableError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::DeviceStatusWifiAvailableError >& error)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onDeviceStatusWifiAvailableError"));
}


/************************************************
Function 	: onDeviceStatusWifiAvailableResult
Parameters 	:
Description : This function is an acknowledgement in case
*             WIFI Available information is sent to CAN
SYSFL/SWFL :
Author     : Bruce 24/02/2021
************************************************/
void VehicleSrvClient::onDeviceStatusWifiAvailableResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::DeviceStatusWifiAvailableResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onDeviceStatusWifiAvailableResult"));
}


/************************************************
Function 	: onDeviceStatusBTStatusError
Parameters 	:
Description : This function is an error in case
*             BT STATUS available is not received by CAN
SYSFL/SWFL :
Author     : Bruce 24/02/2021
************************************************/
void VehicleSrvClient::onDeviceStatusBTStatusError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::DeviceStatusBTStatusError >& error)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onDeviceStatusBTStatusError"));
}


/************************************************
Function 	: onDeviceStatusBTStatusResult
Parameters 	:
Description : This function is an acknowledgement in case
*             BT STATUS information is sent to CAN
SYSFL/SWFL :
Author     : Bruce 24/02/2021
************************************************/
void VehicleSrvClient::onDeviceStatusBTStatusResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::DeviceStatusBTStatusResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onDeviceStatusBTStatusResult"));
}


/************************************************
Function 	: updatePhonebookDownloadSync
Parameters 	:
Description : This function sends phone book
*             sync level to CAN
SYSFL/SWFL :
Author     : Teena
*            v0.1 Bruce Netto 10/02/2021 sending Update to CAN
************************************************/
void VehicleSrvClient::updatePhonebookDownloadSync(tU8 u8DownloadStatus)
{
   if (_vehicleProxy != NULL)
   {
      if (u8DownloadStatus == e8RCDS_ERROR)
      {
         m_U8DownloadStatus = CAN_ERROR;
      }
      else if (u8DownloadStatus == PBDS_INPROGRESS || u8DownloadStatus == PBDS_LOADING)
      {
         m_U8DownloadStatus = CAN_PHONEBOOK_LOADING;
      }
      else if (u8DownloadStatus == PBDS_COMPLETED)
      {
         m_U8DownloadStatus = CAN_PHONEBOOK_READY;
      }
      else if (u8DownloadStatus == PBDS_NOT_STARTED)
      {
         m_U8DownloadStatus = CAN_PHONEBOOK_INIT;
      }
      if (PhoneDataPoolConfig::getInstance()->isBluetoothSupported() == KDS_PHONE_AVAILABLE)
      {
         _vehicleProxy->sendPhoneBookSyncstatusStart(*this, m_U8DownloadStatus);
         ETG_TRACE_USR4(("VehicleSrvClient::updatePhonebookDownloadSync:Phonebook Download sync is : %d", m_U8DownloadStatus));
      }
      else
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updatePhonebookDownloadSync:Phone_Available is 0 in KDS"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("VehicleSrvClient::updatePhonebookDownloadSync:_vehicleProxy is NULL"));
   }
}


/************************************************
Function 	: updatePhoneBatteryLevel
Parameters 	:
Description : This function sends Phone battery
*             level to CAN
SYSFL/SWFL :
Author     : Teena
*            v0.1 Bruce Netto 02/02/2021 sending Update to CAN
************************************************/
void VehicleSrvClient::updatePhoneBatteryLevel(tU8 u8BatteryChargeStatus)
{
   if (_vehicleProxy != NULL)
   {
      m_U8BatteryStatus = u8BatteryChargeStatus;
      if (PhoneDataPoolConfig::getInstance()->isBluetoothSupported() == KDS_PHONE_AVAILABLE)
      {
         _vehicleProxy->sendPhoneBatteryLevelStatusStart(*this, m_U8BatteryStatus);
         ETG_TRACE_USR4(("VehicleSrvClient::updatePhoneBatteryLevel:Battery level is : %d", m_U8BatteryStatus));
      }
      else
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updatePhoneBatteryLevel:Phone_Available is 0 in KDS"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("VehicleSrvClient::updatePhoneBatteryLevel:_vehicleProxy is NULL"));
   }
}


/************************************************
Function 	: updateGSMSignalStrength
Parameters 	:
Description : This function is to send GSM signal
*             strength to CAN
SYSFL/SWFL :
Author     : Teena
*            v0.1 Bruce Netto 02/02/2021 sending Update to CAN
************************************************/
void VehicleSrvClient::updateGSMSignalStrength(tU8 u8SignalstrengthStatus)
{
   if (_vehicleProxy != NULL)
   {
      m_U8SignalStatus = u8SignalstrengthStatus;
      if (PhoneDataPoolConfig::getInstance()->isBluetoothSupported() == KDS_PHONE_AVAILABLE)
      {
         _vehicleProxy->sendGSMSignalStrengthStart(*this, m_U8SignalStatus);
         ETG_TRACE_USR4(("VehicleSrvClient::updateGSMSignalStrength:Signal strength status is : %d", m_U8SignalStatus));
      }
      else
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updateGSMSignalStrength:Phone_Available is 0 in KDS"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("VehicleSrvClient::updateGSMSignalStrength:_vehicleProxy is NULL"));
   }
}


/************************************************
Function 	: updateCallDuration
Parameters 	:
Description : This function is to send CallDuration
*             to CAN
SYSFL/SWFL :
Author     : Teena
*            v0.1 Bruce Netto 02/02/2021 sending Update to CAN
************************************************/
void VehicleSrvClient::updateCallDuration(const btphn_trCallStatus& corfrCallStatus)
{
   ETG_TRACE_USR4(("VehicleSrvClient:: updateCallDuration"));
   if (_vehicleProxy != NULL)
   {
      m_u16CallDurationInSec = ((corfrCallStatus.u8CallDurationHr * 3600) + (corfrCallStatus.u8CallDurationMin * 60) + corfrCallStatus.u8CallDurationSec);
      if (PhoneDataPoolConfig::getInstance()->isBluetoothSupported() == KDS_PHONE_AVAILABLE)
      {
         _vehicleProxy->sendCallDurationStart(*this, m_u16CallDurationInSec);
         ETG_TRACE_USR4(("VehicleSrvClient::updateCallDuration:: Hour is %d", corfrCallStatus.u8CallDurationHr));
         ETG_TRACE_USR4(("VehicleSrvClient::updateCallDuration:: Minute is %d", corfrCallStatus.u8CallDurationMin));
         ETG_TRACE_USR4(("VehicleSrvClient::updateCallDuration:: Second is %d", corfrCallStatus.u8CallDurationSec));
         ETG_TRACE_USR4(("VehicleSrvClient::updateCallDuration:: duration in seconds is %d", m_u16CallDurationInSec));
      }
      else
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updateCallDuration:Phone_Available is 0 in KDS"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("VehicleSrvClient::updateCallDuration:_vehicleProxy is NULL"));
   }
}


/************************************************
Function 	: updatePhoneRoamingStatus
Parameters 	:
Description : This function sends Phone Roaming
*             level to CAN
SYSFL/SWFL :
Author     : v0.1 Bruce Netto 05/02/2021 sending Update to CAN
************************************************/
void VehicleSrvClient::updatePhoneRoamingStatus(bool bRoamingStatus)
{
   if (_vehicleProxy != NULL)
   {
      m_bRoamStatus = bRoamingStatus;
      if (PhoneDataPoolConfig::getInstance()->isBluetoothSupported() == KDS_PHONE_AVAILABLE)
      {
         _vehicleProxy->sendPhoneStatusRoamingStart(*this, m_bRoamStatus);
         ETG_TRACE_USR4(("VehicleSrvClient::updatePhoneRoamingStatus:Roaming Status is : %d", m_bRoamStatus));
      }
      else
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updatePhoneRoamingStatus:Phone_Available is 0 in KDS"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("VehicleSrvClient::updatePhoneRoamingStatus:_vehicleProxy is NULL"));
   }
}


/************************************************
Function 	: updatePhoneBatteryLevelError
Parameters 	:
Description : This function is to send Phone battery
*             level error codes to CAN
SYSFL/SWFL :
Author     : Teena
*            v0.1 Bruce Netto 02/02/2021 sending Update to CAN
************************************************/
void VehicleSrvClient::updatePhoneBatteryLevelError(tU8 u8ChargeStatusErrorCode)
{
   ETG_TRACE_USR4(("VehicleSrvClient::updatePhoneBatteryLevelError"));
   if (_vehicleProxy != NULL)
   {
      if (PhoneDataPoolConfig::getInstance()->isBluetoothSupported() == KDS_PHONE_AVAILABLE)
      {
         if (u8ChargeStatusErrorCode == 0)
         {
            ETG_TRACE_USR4(("VehicleSrvClient:: updatePhoneBatteryLevelError ZERO"));
            return;
         }
         else if (u8ChargeStatusErrorCode == m_U8errorCodeSNA)
         {
            m_U8BatteryStatus = CAN_SNA;
            _vehicleProxy->sendPhoneBatteryLevelStatusStart(*this, m_U8BatteryStatus);
            ETG_TRACE_USR4(("VehicleSrvClient:: updatePhoneBatteryLevelError CAN_SNA"));
         }
         else
         {
            m_U8BatteryStatus = CAN_ERROR;
            _vehicleProxy->sendPhoneBatteryLevelStatusStart(*this, m_U8BatteryStatus);
            ETG_TRACE_USR4(("VehicleSrvClient:: updatePhoneBatteryLevelError CAN_ERROR"));
         }
      }
      else
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updatePhoneBatteryLevelError:Phone_Available is 0 in KDS"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("VehicleSrvClient::updatePhoneBatteryLevelError:_vehicleProxy is NULL"));
   }
}


/************************************************
Function 	: updatePhonebookDownloadError
Parameters 	:
Description : This function sends Phone battery
*             level to CAN
SYSFL/SWFL :
Author     : Teena
*            v0.1 Bruce Netto 10/02/2021 sending Update to CAN
************************************************/
void VehicleSrvClient::updatePhonebookDownloadError(tU8 u8DownloadErrorCode)
{
   ETG_TRACE_USR4(("VehicleSrvClient::updatePhonebookDownloadError"));
   if (_vehicleProxy != NULL)
   {
      if (PhoneDataPoolConfig::getInstance()->isBluetoothSupported() == KDS_PHONE_AVAILABLE)
      {
         if (u8DownloadErrorCode == 0)
         {
            ETG_TRACE_USR4(("VehicleSrvClient:: updatePhonebookDownloadError ZERO"));
            return;
         }
         else if (u8DownloadErrorCode == m_U8errorCodeSNA)
         {
            m_U8DownloadStatus = CAN_SNA;
            _vehicleProxy->sendPhoneBookSyncstatusStart(*this, m_U8DownloadStatus);
            ETG_TRACE_USR4(("VehicleSrvClient:: updatePhonebookDownloadError CAN_SNA"));
         }
         else
         {
            m_U8DownloadStatus = CAN_ERROR;
            _vehicleProxy->sendPhoneBookSyncstatusStart(*this, m_U8DownloadStatus);
            ETG_TRACE_USR4(("VehicleSrvClient:: updatePhonebookDownloadError CAN_ERROR"));
         }
      }
      else
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updatePhonebookDownloadError:Phone_Available is 0 in KDS"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("VehicleSrvClient::updatePhonebookDownloadError:_vehicleProxy is NULL"));
   }
}


/************************************************
Function 	: updateGSMSignalStrengthError
Parameters 	:
Description : function to send the GSM Error to CAN
SYSFL/SWFL :
Author     : Teena
*            v0.1 Bruce Netto 02/02/2021 sending Update to CAN
************************************************/
void VehicleSrvClient::updateGSMSignalStrengthError(tU8 u8SignalStatusErrorCode)
{
   ETG_TRACE_USR4(("VehicleSrvClient:: updateGSMSignalStrengthError"));
   if (_vehicleProxy != NULL)
   {
      if (PhoneDataPoolConfig::getInstance()->isBluetoothSupported() == KDS_PHONE_AVAILABLE)
      {
         if (u8SignalStatusErrorCode == 0)
         {
            ETG_TRACE_USR4(("VehicleSrvClient:: updateGSMSignalStrengthError ZERO"));
            return;
         }
         else if (u8SignalStatusErrorCode == m_U8errorCodeSNA)
         {
            m_U8SignalStatus = CAN_SNA;
            _vehicleProxy->sendGSMSignalStrengthStart(*this, m_U8SignalStatus);
            ETG_TRACE_USR4(("VehicleSrvClient:: updateGSMSignalStrengthError CAN_SNA"));
         }
         else
         {
            m_U8SignalStatus = CAN_ERROR;
            _vehicleProxy->sendGSMSignalStrengthStart(*this, m_U8SignalStatus);
            ETG_TRACE_USR4(("VehicleSrvClient:: updateGSMSignalStrengthError CAN_ERROR"));
         }
      }
      else
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updateGSMSignalStrengthError:Phone_Available is 0 in KDS"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("VehicleSrvClient::updateGSMSignalStrengthError:_vehicleProxy is NULL"));
   }
}


/************************************************
Function 	: updateCallDurationError
Parameters 	:
Description : function to send the CallDuration Error to CAN
SYSFL/SWFL :
Author     : Teena
*            v0.1 Bruce Netto 02/02/2021 sending Update to CAN
************************************************/
void VehicleSrvClient::updateCallDurationError(tU8 u8CallDurationErrorCode)
{
   ETG_TRACE_USR4(("VehicleSrvClient:: updateCallDurationError"));
   if (_vehicleProxy != NULL)
   {
      if (PhoneDataPoolConfig::getInstance()->isBluetoothSupported() == KDS_PHONE_AVAILABLE)
      {
         if (u8CallDurationErrorCode == 0)
         {
            ETG_TRACE_USR4(("VehicleSrvClient:: updateCallDurationError ZERO"));
            return;
         }
         else if (u8CallDurationErrorCode == m_U8errorCodeSNA)
         {
            //m_U8CallDuraErrCode = CAN_SNA;
            _vehicleProxy->sendCallDurationStart(*this, 0xFFFF);
            ETG_TRACE_USR4(("VehicleSrvClient:: updateCallDurationError CAN_SNA"));
         }
         else
         {
            //m_U8CallDuraErrCode = CAN_ERROR;
            _vehicleProxy->sendCallDurationStart(*this, 0xFFFE);
            ETG_TRACE_USR4(("VehicleSrvClient:: updateCallDurationError CAN_ERROR"));
         }
      }
      else
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updateCallDurationError:Phone_Available is 0 in KDS"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("VehicleSrvClient::updateCallDurationError:_vehicleProxy is NULL"));
   }
}


/************************************************
Function 	: updatePhoneRoamingError
Parameters 	:
Description : This function is to send Phone roaming
*             error codes to CAN
SYSFL/SWFL :
Author     : v0.1 Bruce Netto 05/02/2021 sending Update to CAN
************************************************/
void VehicleSrvClient::updatePhoneRoamingError(tU8 u8RoamingErrorCode)
{
   ETG_TRACE_USR4(("VehicleSrvClient::updatePhoneRoamingError"));
   if (_vehicleProxy != NULL)
   {
      if (PhoneDataPoolConfig::getInstance()->isBluetoothSupported() == KDS_PHONE_AVAILABLE)
      {
         if (u8RoamingErrorCode == 0)
         {
            ETG_TRACE_USR4(("VehicleSrvClient:: updatePhoneRoamingError ZERO"));
            return;
         }
         else if (u8RoamingErrorCode == m_U8errorCodeSNA)
         {
            m_bRoamStatus = CAN_SECONDARY_SNA;
            _vehicleProxy->sendPhoneStatusRoamingStart(*this, m_bRoamStatus);
            ETG_TRACE_USR4(("VehicleSrvClient:: updatePhoneRoamingError CAN_ROAMING_SNA"));
         }
         else
         {
            m_bRoamStatus = CAN_SECONDARY_ERROR;
            _vehicleProxy->sendPhoneStatusRoamingStart(*this, m_bRoamStatus);
            ETG_TRACE_USR4(("VehicleSrvClient:: updatePhoneRoamingError CAN_ROAMING_ERROR"));
         }
      }
      else
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updatePhoneRoamingError:Phone_Available is 0 in KDS"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("VehicleSrvClient::updatePhoneRoamingError:_vehicleProxy is NULL"));
   }
}


/************************************************
Function 	: updateCallStatusError
Parameters 	:
Description : This function is to send call status
*             error codes to CAN
SYSFL/SWFL :
Author     : v0.1 Bruce Netto 08/02/2021 sending Update to CAN
************************************************/
void VehicleSrvClient::updateCallStatusError(tU8 u8CallStatusErrorCode)
{
   ETG_TRACE_USR4(("VehicleSrvClient::updateCallStatusError"));
   if (_vehicleProxy != NULL)
   {
      if (PhoneDataPoolConfig::getInstance()->isBluetoothSupported() == KDS_PHONE_AVAILABLE)
      {
         if (u8CallStatusErrorCode == 0)
         {
            ETG_TRACE_USR4(("VehicleSrvClient:: updateCallStatusError ZERO"));
            return;
         }
         else if (u8CallStatusErrorCode == m_U8errorCodeSNA)
         {
            m_U8CallStatus = CAN_SNA;
            _vehicleProxy->sendPhoneCallStatusStart(*this, m_U8CallStatus);
            ETG_TRACE_USR4(("VehicleSrvClient:: updateCallStatusError CAN_SNA"));
         }
         else
         {
            m_U8CallStatus = CAN_ERROR;
            _vehicleProxy->sendPhoneCallStatusStart(*this, m_U8CallStatus);
            ETG_TRACE_USR4(("VehicleSrvClient:: updateCallStatusError CAN_ERROR"));
         }
      }
      else
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updateCallStatusError:Phone_Available is 0 in KDS"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("VehicleSrvClient::updateCallStatusError:_vehicleProxy is NULL"));
   }
}


void VehicleSrvClient::updateCallStatus()
{
   ETG_TRACE_USR4(("VehicleSrvClient::updateCallStatus"));
   //Constants found in the types.Xhcdl file
   if (ServiceTel::poGetInstance()->getIncomingCallStatus() == true)
   {
      m_U8CallStatus	= BTPHN_CAN_INCOMMING;
   }
   else if (ServiceTel::poGetInstance()->getCallDialingStatus() == true)
   {
      m_U8CallStatus = BTPHN_CAN_OUTGOING;
   }
   else if (ServiceTel::poGetInstance()->getOngoingCallStatus() == true)
   {
      m_U8CallStatus = BTPHN_CAN_ACTIVE;
   }
   else if (ServiceTel::poGetInstance()->getIdleCallStatus() == true)
   {
      m_U8CallStatus = BTPHN_CAN_IDLE;
   }
   if (_vehicleProxy != NULL)
   {
      if (PhoneDataPoolConfig::getInstance()->isBluetoothSupported() == KDS_PHONE_AVAILABLE)
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updateCallStatus:status is :%d", m_U8CallStatus));
         _vehicleProxy->sendPhoneCallStatusStart(*this, m_U8CallStatus);
      }
      else
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updateCallStatus:Phone_Available is 0 in KDS"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("VehicleSrvClient::updateCallStatus:_vehicleProxy is NULL"));
   }
}


/************************************************
Function 	: updatePrivateMode
Parameters 	:
Description : This function is to send call status
*             for private mode and dial error
SYSFL/SWFL :
Author     : v0.1 Bruce Netto 09/02/2021 sending Update to CAN
************************************************/
void VehicleSrvClient::updatePrivateDialFailMode(int statusType)
{
   ETG_TRACE_USR4(("VehicleSrvClient::updatePrivateDialFailMode"));
   //Constants for Private Mode and Dial failed as remaining values are part of types.Xhcdl

   switch (statusType)
   {
      case BTPHN_EN_PRIVATE_MODE :
      {
         //Private Mode:set status to 4
         m_U8CallStatus = BTPHN_CAN_PRIVATE_MODE;
      }
      break;
      case BTPHN_EN_DIALING_FAILED :
      {
         //Dialing Failed:set status to 5
         m_U8CallStatus = BTPHN_CAN_DIALING_FAILED;
      }
      break;
      default:
         break;
   }
   if (_vehicleProxy != NULL)
   {
      if (PhoneDataPoolConfig::getInstance()->isBluetoothSupported() == KDS_PHONE_AVAILABLE)
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updatePrivateDialFailMode:status is :%d", m_U8CallStatus));
         _vehicleProxy->sendPhoneCallStatusStart(*this, m_U8CallStatus);
      }
      else
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updatePrivateDialFailMode:Phone_Available is 0 in KDS"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("VehicleSrvClient::updatePrivateDialFailMode:_vehicleProxy is NULL"));
   }
}


/***********************************************************************
*DESCRIPTION	: This function is called when PhoneCal command is received from Vehicle service interface.
   	              Currently Call ACCEPT and Call REJECT shall be supported
   	              SYSFL-8200
                   Phone call command
                   0: reserved
                   1: Accept Call
                   2: Reject/End Call
                   3: Transfer Call - Not impl - To be implemented
                   4: Mute - Not impl - To be implemented
                   5: Demute - Not impl - To be implemented
                   14: Error
                   15: Don’t care

*RETURNVALUE	: NA
************************************************************************/
void VehicleSrvClient::onPhoneCallCommandStatus(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhoneCallCommandStatus >& status)
{
   // Const based on Interface def
   const tU8 CAN_ACCEPT = 1;
   const tU8 CAN_REJECT = 2;
   const tU8 TRANSFER_CALL = 3;
   const tU8 MUTE = 4;
   const tU8 DEMUTE = 5;
   const tU8 ERROR = 14;
   const tU8 DONTCARE = 15;

   ETG_TRACE_USR4(("VehicleSrvClient::onPhoneCallCommandStatus called."));
   if (proxy == _vehicleProxy)
   {
      tU8 U8PhoneCallCommandValue = status->getPhoneCallCommandValue();
      ETG_TRACE_USR4(("VehicleSrvClient::onPhoneCallCommandStatus PhoneCallCommand  = %d", U8PhoneCallCommandValue));

      // Reusing the MCP short press key handling to address all the Accept and Reject scenario
      if (CAN_ACCEPT == U8PhoneCallCommandValue)
      {
         POST_MSG((COURIER_MESSAGE_NEW(onHardKeyPress)(HARDKEYCODE_HK_PHONE_ACCEPT, hmibase::HardKeyStateEnum(Enum_hmibase_HARDKEYSTATE_UP))));
      }
      else if (CAN_REJECT == U8PhoneCallCommandValue)
      {
         POST_MSG((COURIER_MESSAGE_NEW(onHardKeyPress)(HARDKEYCODE_HK_PHONE_REJECT, hmibase::HardKeyStateEnum(Enum_hmibase_HARDKEYSTATE_UP))));
      }
      else if (TRANSFER_CALL == U8PhoneCallCommandValue)
      {
         //Transfer call : TransferVehicleAudioReq
         ETG_TRACE_USR4(("VehicleSrvClient::onPhoneCallCommandStatus : %d", U8PhoneCallCommandValue));
         POST_MSG((COURIER_MESSAGE_NEW(TransferVehicleAudioReqFromCAN)()));
      }
      else if (MUTE == U8PhoneCallCommandValue)
      {
         ETG_TRACE_USR4(("VehicleSrvClient::onPhoneCallCommandStatus : %d", U8PhoneCallCommandValue));
         //mute
         if (ServiceTel::poGetInstance()->getMicrophoneMuteStatus() == FALSE)
         {
            ETG_TRACE_USR4(("VehicleSrvClient::onPhoneCallCommandStatus :MicMuteStateReqFromCAN is false"));
            POST_MSG((COURIER_MESSAGE_NEW(MicMuteStateReqFromCAN)()));
         }
         else
         {
            ETG_TRACE_USR4(("VehicleSrvClient::onPhoneCallCommandStatus :MicMuteStateReqFromCAN is true"));
         }
      }
      else if (DEMUTE == U8PhoneCallCommandValue)
      {
         //demute
         if (ServiceTel::poGetInstance()->getMicrophoneMuteStatus() == TRUE)
         {
            ETG_TRACE_USR4(("VehicleSrvClient::onPhoneCallCommandStatus :MicMuteStateReqFromCAN is true"));
            POST_MSG((COURIER_MESSAGE_NEW(MicMuteStateReqFromCAN)()));
         }
         else
         {
            ETG_TRACE_USR4(("VehicleSrvClient::onPhoneCallCommandStatus :MicMuteStateReqFromCAN is false"));
         }
      }
      else if (ERROR == U8PhoneCallCommandValue || DONTCARE == U8PhoneCallCommandValue)
      {
         ETG_TRACE_USR4(("VehicleSrvClient::onPhoneCallCommandStatus : %d", U8PhoneCallCommandValue));
         ETG_TRACE_USR4(("Error condition or dont care condition entered"));
      }
      else
      {
         ETG_TRACE_ERR(("VehicleSrvClient::onPhoneCallCommandStatus - Invalid Phonecall command"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("VehicleSrvClient::onPhoneCallCommandStatus  service is not available"));
   }
}


void VehicleSrvClient::onPhoneCallCommandError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& /* proxy */, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhoneCallCommandError >& /* error*/)
{
   ETG_TRACE_COMP(("VehicleSrvClient::onPhoneCallCommandError Called."));
}


/************************************************
Function 	: onPhonePwrCmdError
Parameters 	:
Description : This function is trigerred by the Vehicle
*             component when an error of phone power switch occurs
SYSFL/SWFL :
Author     : Bruce Netto 29/01/2021
************************************************/
void VehicleSrvClient::onPhonePwrCmdError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhonePwrCmdError >& error)
{
   ETG_TRACE_ERR(("VehicleSrvClient::onPhonePwrCmdError service is not available"));
}


/************************************************
Function 	: onPhonePwrCmdStatus
Parameters 	:
Description : This function is trigerred by the Vehicle component when the Phone
*             power switch in System Settings is turned ON/oFF
SYSFL/SWFL :
Author     : Bruce Netto 29/01/2021
************************************************/
void VehicleSrvClient::onPhonePwrCmdStatus(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::PhonePwrCmdStatus >& status)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onPhonePwrCmdStatus"));
   if (_vehicleProxy && (proxy == _vehicleProxy))
   {
      ETG_TRACE_USR4(("VehicleSrvClient::onPhonePwrCmdStatus Phone Command Value is: %d", status->getPhonePwrCmdValue()));
#ifdef VARIANT_S_FTR_ENABLE_BLUETOOTH_ON_OFF
      if (status->getPhonePwrCmdValue() == PHONE_POWERCMD_ON)
      {
         ETG_TRACE_USR4(("VehicleSrvClient::onPhonePwrCmdStatus Phone Power is SWITCHED ON"));
         ServiceBT::poGetInstance()->BTONOFFStart(true);
      }
      else if (status->getPhonePwrCmdValue() == PHONE_POWERCMD_OFF)
      {
         ETG_TRACE_USR4(("VehicleSrvClient::onPhonePwrCmdStatus Phone Power is SWITCHED OFF"));
         ServiceBT::poGetInstance()->BTONOFFStart(false);
      }
      else if (status->getPhonePwrCmdValue() == PHONE_POWERCMD_DONTCARE)
      {
         ETG_TRACE_USR4(("VehicleSrvClient::onPhonePwrCmdStatus Phone Power is DONT CARE"));
      }
#endif
   }
   else
   {
      ETG_TRACE_USR4(("VehicleSrvClient::onPhonePwrCmdStatus Vehicle Proxy is not available"));
   }
}


/************************************************
Function 	: onSMARTVRError
Parameters 	:
Description : This function is trigerred by the Vehicle
*             component when an error of VR command occurs
SYSFL/SWFL :
Author     : Bruce Netto 09/06/2021
************************************************/
void VehicleSrvClient::onSMARTVRError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
                                      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SMARTVRError >& error)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onSMARTVRError"));
}


/************************************************
Function 	: onSMARTVRStatus
Parameters 	:
Description : This function is trigerred by the Vehicle component when the VR
*             command is initiated for phone by the user
SYSFL/SWFL  : CRQ 1053847 Currently Disabled
Author      : Bruce Netto 09/06/2021
************************************************/
void VehicleSrvClient::onSMARTVRStatus(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
                                       const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SMARTVRStatus >& status)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onSMARTVRStatus"));
   if (KDS_PHONE_AVAILABLE == PhoneDataPoolConfig::getInstance()->isBluetoothSupported())
   {
      ETG_TRACE_USR4(("VehicleSrvClient::onSMARTVRStatus: Phone_Available is 1 in KDS"));
      if (KDS_SMARTVR_AVAILABLE == PhoneDataPoolConfig::getInstance()->isVRSupported())
      {
         ETG_TRACE_USR4(("VehicleSrvClient::onSMARTVRStatus: SmartVR_Available is 1 in KDS"));
         //if true means the device checked is the current active device
         //and the Siri is not disabled on the device
         if (ServiceTel::poGetInstance()->bGetActiveSmartVRDevice() == true)
         {
            ETG_TRACE_USR4(("VehicleSrvClient::onSMARTVRStatus: SIRI is not Disabled"));
            if (PHONE_SMARTVR_START == status->getSMARTVRValue())
            {
               ETG_TRACE_USR4(("VehicleSrvClient::onSMARTVRStatus Start handleBTVRViaGenCAN"));
               //Start the BT VR once trigerred via GEneric CAN Command
               POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(GenericCANBTVRReq)()));
            }
            else if (PHONE_SMARTVR_STOP == status->getSMARTVRValue())
            {
               ETG_TRACE_USR4(("VehicleSrvClient::onSMARTVRStatus Stop BTVRSessionExitAppReqMsg"));
               //Stop the VR via Generic CAN
               POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(BTVRSessionExitAppReqMsg)()));
            }
            else if (PHONE_SMARTVR_DONTCARE == status->getSMARTVRValue())
            {
               ETG_TRACE_USR4(("VehicleSrvClient::onSMARTVRStatus is DONT CARE"));
            }
         }
         else
         {
            ETG_TRACE_USR4(("VehicleSrvClient::onSMARTVRStatus: SIRI is Disabled"));
         }
      }
      else
      {
         ETG_TRACE_USR4(("VehicleSrvClient::onSMARTVRStatus SmartVR_Available is 0 in KDS"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("VehicleSrvClient::onSMARTVRStatus: Phone_Available is 0 in KDS"));
   }
}


/***********************************************************************
*DESCRIPTION	: This function is called whenever there is a change in the speed value from the vehicle data.
   	              To update the speed lock status value based on the system setting enable/Disable.
   	              [SWFL-3787][SYSFL-8952]
*RETURNVALUE	: NA
************************************************************************/
void VehicleSrvClient::onSpeedStatus(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SpeedStatus >& status)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onSpeedStatus Called."));
   if (proxy == _vehicleProxy)
   {
      ETG_TRACE_USR4(("VehicleSrvClient::onSpeedStatus speedvalue  = %d", status->getSpeedValue()));
      SpeedLockStatusHandler::getInstance().updateSpeedValue(status->getSpeedValue());
      setSpeedValue(status->getSpeedValue());
   }
   else
   {
      ETG_TRACE_USR1(("VehicleSrvClient::onSpeedStatus  service is not available"));
   }
}


void VehicleSrvClient::onSpeedError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SpeedError >& /*error*/)
{
   ETG_TRACE_COMP(("VehicleSrvClient::onSpeedError"));
}


void VehicleSrvClient::setSpeedValue(uint16 l_u16SpeedValue)
{
   ETG_TRACE_USR4(("VehicleSrvClient::setSpeedValue speedvalue  = %d", l_u16SpeedValue));
   m_u16SpeedValue = l_u16SpeedValue;
}


int VehicleSrvClient::getSpeedValue()
{
   ETG_TRACE_USR4(("VehicleSrvClient::getSpeedValue speedvalue  = %d", m_u16SpeedValue));
   return m_u16SpeedValue;
}


/***********************************************************************
*DESCRIPTION	: This function is called whenever there is a change in the system setting speedlock status in the current ignition cycle.
   	              To update the speed lock status value based on the system setting enable/Disable.
   	              [SWFL-3797][SYSFL-8955]
*RETURNVALUE	: NA
************************************************************************/

void VehicleSrvClient::onSpeedLockStatusUpdate(const ::boost::shared_ptr< HmiDataProxy >& proxy,
      const ::boost::shared_ptr< SpeedLockStatusUpdate >& update)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onSpeedLockStatusUpdate()"));

   if (proxy == _hmiDataProxy)
   {
      ETG_TRACE_USR4(("VehicleSrvClient::onSpeedLockStatusUpdate() : %d", update->getSpeedLockStatus()));
      SpeedLockStatusHandler::getInstance().updateSysSettingSpeedLockStatus(update->getSpeedLockStatus());
   }
   else
   {
      ETG_TRACE_USR4(("VehicleSrvClient::onSpeedLockStatusUpdate() - Invalid hmiDataProxy Obj"));
   }
}


void VehicleSrvClient::onSpeedLockStatusError(const ::boost::shared_ptr< HmiDataProxy >& /* proxy */,
      const ::boost::shared_ptr< SpeedLockStatusError >& /* error */)
{
   ETG_TRACE_USR4(("VehicleDataClientHandler::onSpeedLockStatusError()"));
}


/***********************************************************************
*DESCRIPTION	: This function is called whenever there is an error in the download status for an active device
   	              [SWFL-][SYSFL-]
*RETURNVALUE	: NA
************************************************************************/
void VehicleSrvClient::updateCallHistorySyncStatus(tU8 CallListDownloadState)
{
   ETG_TRACE_USR4(("VehicleSrvClient::updateCallHistorySyncStatus()"));
   if (_vehicleProxy != NULL)
   {
      if (CallListDownloadState == e8RCDS_ERROR)
      {
         m_u8CallHistorySyncState = CAN_ERROR;
      }
      else
      {
         m_u8CallHistorySyncState = CallListDownloadState;
      }

      if (PhoneDataPoolConfig::getInstance()->isBluetoothSupported() == KDS_PHONE_AVAILABLE)
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updateCallHistorySyncStatus():m_u8CallHistorySyncState value is %d", m_u8CallHistorySyncState));
         _vehicleProxy->sendCallHistorySyncStatusStart(*this, m_u8CallHistorySyncState);
      }
      else
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updateCallHistorySyncStatus:Phone_Available is 0 in KDS"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("updateCallHistorySyncStatus:_vehicleProxy is NULL"));
   }
}


void VehicleSrvClient::updateCallHistorySyncError(tU8 CallListDownloadErrorCode)
{
   ETG_TRACE_USR4(("VehicleSrvClient::updateCallHistorySyncError Entered()"));
   if (_vehicleProxy != NULL)
   {
      if (PhoneDataPoolConfig::getInstance()->isBluetoothSupported() == KDS_PHONE_AVAILABLE)
      {
         if (CallListDownloadErrorCode == 0)
         {
            ETG_TRACE_USR4(("VehicleSrvClient:: updateCallHistorySyncError ZERO"));
            return;
         }
         else if (CallListDownloadErrorCode == m_U8errorCodeSNA)
         {
            m_u8CallHistorySyncState = CAN_SNA;
            _vehicleProxy->sendCallHistorySyncStatusStart(*this, m_u8CallHistorySyncState);
            ETG_TRACE_USR4(("VehicleSrvClient:: updateCallHistorySyncError CAN_SNA"));
         }
         else
         {
            m_u8CallHistorySyncState = CAN_ERROR;
            _vehicleProxy->sendCallHistorySyncStatusStart(*this, m_u8CallHistorySyncState);
            ETG_TRACE_USR4(("VehicleSrvClient:: updateCallHistorySyncError CAN_ERROR"));
         }
      }
      else
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updateCallHistorySyncError:Phone_Available is 0 in KDS"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("VehicleSrvClient::updateCallHistorySyncError:_vehicleProxy is NULL"));
   }
}


/***********************************************************************
*DESCRIPTION	: This function is called whenever there is an update for the BT link status
   	              [SWFL-][SYSFL-]
*RETURNVALUE	: NA
************************************************************************/
void VehicleSrvClient::updateBTLinkStatus(tU8 BTLinkState)
{
   ETG_TRACE_USR4(("VehicleSrvClient::updateBTLinkStatus()"));
   if (_vehicleProxy != NULL)
   {
      m_u8BTLinkState = BTLinkState;
      ETG_TRACE_USR4(("VehicleSrvClient:: updateBTLinkStatus %d", m_u8BTLinkState));
      if (PhoneDataPoolConfig::getInstance()->isBluetoothSupported() == KDS_PHONE_AVAILABLE)
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updateBTLinkStatus:sendPhoneBTLinkStatusStart"));
         _vehicleProxy->sendPhoneBTLinkStatusStart(*this, m_u8BTLinkState);
      }
      else
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updateBTLinkStatus:Phone_Available is 0 in KDS"));
      }
   }
}


/***********************************************************************
*DESCRIPTION	: This function is called whenever there is BT Link status ErrorCode
   	              [SWFL-][SYSFL-]
*RETURNVALUE	: NA
************************************************************************/
void VehicleSrvClient::updateBTLinkStatusError(tU8 BTLinkStateErrorCode)
{
   ETG_TRACE_USR4(("VehicleSrvClient::updateBTLinkStatusError Entered()"));
   if (_vehicleProxy != NULL)
   {
      ETG_TRACE_USR4(("VehicleSrvClient::updateBTLinkStatusError Error Code: %d", BTLinkStateErrorCode));
      if (PhoneDataPoolConfig::getInstance()->isBluetoothSupported() == KDS_PHONE_AVAILABLE)
      {
         if (BTLinkStateErrorCode == 0)
         {
            ETG_TRACE_USR4(("VehicleSrvClient:: updateBTLinkStatusError ZERO"));
            return;
         }
         else if (BTLinkStateErrorCode == m_U8errorCodeSNA)
         {
            m_u8BTLinkState = CAN_SNA;
            _vehicleProxy->sendPhoneBTLinkStatusStart(*this, m_u8BTLinkState);
            //_vehicleProxy->sendDeviceStatusBTStatusStart(*this, CAN_SECONDARY_SNA);
            ETG_TRACE_USR4(("VehicleSrvClient:: uupdateBTLinkStatusError CAN_SNA"));
         }
         else
         {
            m_u8BTLinkState = CAN_ERROR;
            _vehicleProxy->sendPhoneBTLinkStatusStart(*this, m_u8BTLinkState);
            //_vehicleProxy->sendDeviceStatusBTStatusStart(*this, CAN_SECONDARY_ERROR);
            ETG_TRACE_USR4(("VehicleSrvClient:: uupdateBTLinkStatusError CAN_ERROR"));
         }
      }
      else
      {
         ETG_TRACE_USR4(("VehicleSrvClient::updateBTLinkStatusError:Phone_Available is 0 in KDS"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("updateBTLinkStatusError:_vehicleProxy is NULL"));
   }
}


/************************************************
Function 	: vCheckKDSFeaturesAvailable
Parameters 	:
Description : This function checks if Phone supported
*             KDS features are enabled or disabled
SYSFL/SWFL  :
Author      : Bruce Netto 24/01/2021
*             v1.0 added Smart VR available 09/06/2021
************************************************/
void VehicleSrvClient::vCheckKDSFeaturesAvailable()
{
   ETG_TRACE_USR4(("VehicleSrvClient::vCheckKDSFeaturesAvailable"));

   m_bPhoneAvailable = PhoneDataPoolConfig::getInstance()->isBluetoothSupported();
   m_bWifiAvailable = PhoneDataPoolConfig::getInstance()->isWIFISupported();
   m_bSmartPhoneVRAvailable = PhoneDataPoolConfig::getInstance()->isVRSupported();

   ETG_TRACE_USR4(("VehicleSrvClient::vCheckKDSFeaturesAvailable BT PhoneAvailable: %d // WiFIAvailable: %d // SmartVRAvailable: %d", m_bPhoneAvailable, m_bWifiAvailable, m_bSmartPhoneVRAvailable));
   ETG_TRACE_USR4(("VehicleSrvClient::vCheckKDSFeaturesAvailable BT PhoneAvailable: %d // WiFIAvailable: %d", m_bPhoneAvailable, m_bWifiAvailable));
   if (_vehicleProxy != NULL)
   {
      _vehicleProxy->sendPhoneAvailableStatusStart(*this, m_bPhoneAvailable);
      _vehicleProxy->sendDeviceStatusWifiAvailableStart(*this, m_bWifiAvailable);
      _vehicleProxy->sendPhoneVRAvailableStart(*this, m_bSmartPhoneVRAvailable);
      ETG_TRACE_USR4(("VehicleSrvClient::vCheckKDSFeaturesAvailable Sending BT PhoneAvailable: %d // WiFIAvailable: %d // SmartVRAvailable: %d to CAN", m_bPhoneAvailable, m_bWifiAvailable, m_bSmartPhoneVRAvailable));
      ETG_TRACE_USR4(("VehicleSrvClient::vCheckKDSFeaturesAvailable Sending BT PhoneAvailable: %d // WiFIAvailable: %d to CAN", m_bPhoneAvailable, m_bWifiAvailable));
   }
}


/************************************************
Function 	: vInitGenericCanInterfaces
Parameters 	:
Description :  This function initializes the Generic CAN
*              Interfaces to SNA, Done During onAvailable()
*              and Disconnsction of Phone
SYSFL/SWFL :
Author     : Bruce Netto 05/06/2021
************************************************/
void VehicleSrvClient::vInitGenericCanInterfaces()
{
   ETG_TRACE_USR4(("VehicleSrvClient::vInitGenericCanInterfaces"));
   //On Available of VehicleClient Initialize all values for GEN CAN
   _vehicleProxy->sendPhoneBTLinkStatusStart(*this, CAN_SNA);
   _vehicleProxy->sendCallHistorySyncStatusStart(*this, CAN_SNA);
   _vehicleProxy->sendPhoneCallStatusStart(*this, CAN_SNA);
   _vehicleProxy->sendPhoneStatusRoamingStart(*this, CAN_SECONDARY_SNA);
   _vehicleProxy->sendCallDurationStart(*this, 0xFFFF);
   _vehicleProxy->sendGSMSignalStrengthStart(*this, CAN_SNA);
   _vehicleProxy->sendPhoneBookSyncstatusStart(*this, CAN_SNA);
   _vehicleProxy->sendPhoneBatteryLevelStatusStart(*this, CAN_SNA);
   ETG_TRACE_USR4(("VehicleSrvClient::vInitGenericCanInterfaces GEN CAN Interfaces have been Initialized with SNA"));
}


/************************************************
Function 	: vUpdateBTOnOffStatus
Parameters 	:
Description : This function updates the BT phones ON or OFF status to CAN
SYSFL/SWFL :
Author     : Bruce Netto 24/01/2021
************************************************/
void VehicleSrvClient::vUpdateBTOnOffStatus(tU8 BTOnOff)
{
   ETG_TRACE_USR4(("VehicleSrvClient::vUpdateBTOnOffStatus"));
   if (_vehicleProxy != NULL)
   {
      if (PhoneDataPoolConfig::getInstance()->isBluetoothSupported() == KDS_PHONE_AVAILABLE)
      {
         ETG_TRACE_USR4(("VehicleSrvClient::vUpdateBTOnOffStatus BT Status for CAN is: %d ", BTOnOff));
         _vehicleProxy->sendDeviceStatusBTStatusStart(*this, BTOnOff);
      }
      else
      {
         ETG_TRACE_USR4(("VehicleSrvClient::vUpdateBTOnOffStatus: Phone_Available is 0 in KDS"));
         _vehicleProxy->sendDeviceStatusBTStatusStart(*this, BT_STATUS_SNA);
      }
   }
}


/************************************************
Function 	: vUpdateCallInfo
Parameters 	:
Description : This function updates the BT phones ON or OFF status to CAN
SYSFL/SWFL :
Author     : Bruce Netto 16/03/2022
************************************************/
void VehicleSrvClient::vUpdateCallInfo(int callLongDatatype, std::string strlongDataCallNum)
{
   ETG_TRACE_USR4(("VehicleSrvClient::vUpdateCallInfo"));
   vPrepareLongData(callLongDatatype, strlongDataCallNum);
}


/************************************************
Function 	: vUpdatePhoneInfo
Parameters 	:
Description : This function updates the BT phones ON or OFF status to CAN
SYSFL/SWFL :
Author     : Bruce Netto 16/03/2022
************************************************/
void VehicleSrvClient::vUpdatePhoneInfo(int phoneLongDataType, std::string strlongDataPhoneName)
{
   ETG_TRACE_USR4(("VehicleSrvClient::vUpdatePhoneInfo"));
   vPrepareLongData(phoneLongDataType, strlongDataPhoneName);
}


/************************************************
Function 	: vPrepareLongData
Parameters 	:
Description : This function updates the BT phones ON or OFF status to CAN
SYSFL/SWFL :
Author     : Bruce Netto 16/03/2022
************************************************/
void VehicleSrvClient::vPrepareLongData(int longDataType, std::string longdatainfo)
{
   ETG_TRACE_USR4(("VehicleSrvClient::vPrepareLongData"));
   //Concatenate Element type and terminating character
   if (!longdatainfo.empty())
   {
      ETG_TRACE_USR4(("VehicleSrvClient::vPrepareLongData for MediaType: %d and Information: %s", longDataType, longdatainfo.c_str()));
      //Vector creation
      std::vector<uint8> phoneinfo;
      ::vehicle_main_fi_types::T_CAN_Mesg  phoneCANMsg;
      //Push Mediatype into first position in the vector
      phoneinfo.push_back(longDataType);
      //Setting TcanMsg initial parameter also with Mediatype
      phoneCANMsg.setU8MediaType(longDataType);
      //Push the Media Long Data into our local vector from position index 1 as 0 is filled with media type
      for (uint i = 0; i < longdatainfo.size(); ++i)
      {
         phoneinfo.push_back(longdatainfo[i]);
         //ETG_TRACE_USR4(("VehicleSrvClient::vPrepareLongData longData at %d: %s", i, longdatainfo[i]));
         ETG_TRACE_USR4(("VehicleSrvClient::vPrepareLongData longData at %d: %d", i, longdatainfo[i]));
      }
      //Updating the trailing character
      phoneinfo.push_back('\0');
      //Adding the local vector to TcanMsg second parameter
      phoneCANMsg.setListData(phoneinfo);
      ETG_TRACE_USR4(("VehicleSrvClient::vPrepareLongData longData is packed ready to send"));
      if (longDataType == LONGDATATYPE_CALLNAME || longDataType == LONGDATATYPE_CALLNUM)
      {
         ETG_TRACE_USR4(("VehicleSrvClient::vPrepareLongData longData sendcallinfo interface call"));
         _vehicleProxy->sendSendCallInfoStart(*this, phoneCANMsg);
      }
      if (longDataType == LONGDATATYPE_PHONENAME || longDataType == LONGDATATYPE_PHONENETWORK)
      {
         ETG_TRACE_USR4(("VehicleSrvClient::vPrepareLongData longData sendphoneinfo interface call"));
         _vehicleProxy->sendSendPhoneInfoStart(*this, phoneCANMsg);
      }
   }
   else
   {
      ETG_TRACE_USR4(("VehicleSrvClient::vPrepareLongData the vector is empty"));
   }
}


/************************************************
Function 	: onSendPhoneInfoError
Parameters 	:
Description : This function updates the BT phones ON or OFF status to CAN
SYSFL/SWFL :
Author     : Bruce Netto 16/03/2022
************************************************/
void VehicleSrvClient::onSendPhoneInfoError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SendPhoneInfoError >& error)
{
   ETG_TRACE_ERR(("VehicleSrvClient::onSendPhoneInfoError"));
}


/************************************************
Function 	: onSendPhoneInfoResult
Parameters 	:
Description : This function updates the BT phones ON or OFF status to CAN
SYSFL/SWFL :
Author     : Bruce Netto 16/03/2022
************************************************/
void VehicleSrvClient::onSendPhoneInfoResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SendPhoneInfoResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onSendPhoneInfoResult"));
}


/************************************************
Function 	: onSendCallInfoError
Parameters 	:
Description : This function updates the BT phones ON or OFF status to CAN
SYSFL/SWFL :
Author     : Bruce Netto 16/03/2022
************************************************/
void VehicleSrvClient::onSendCallInfoError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SendCallInfoError >& error)
{
   ETG_TRACE_ERR(("VehicleSrvClient::onSendCallInfoError"));
}


/************************************************
Function 	: onSendCallInfoResult
Parameters 	:
Description : This function updates the BT phones ON or OFF status to CAN
SYSFL/SWFL :
Author     : Bruce Netto 16/03/2022
************************************************/
void VehicleSrvClient::onSendCallInfoResult(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy,
      const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SendCallInfoResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onSendCallInfoResult"));
}


// sys settings TBC
}


}
