/**************************************************************************************
* @file         : ClockSrvClient.cpp
* @author       : Palani K
* @addtogroup   : Phone
* @brief        : To support time and date related features for Phone module
* @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include <hmibase/hall_std_if.h>
#include "ClockSrvClient.h"

#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_PHONE_HALL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ClockSrvClient.cpp.trc.h"
#endif

using namespace clock_main_fi;
using namespace clock_main_fi_types;

namespace App {
namespace Core {

ClockSrvClient* ClockSrvClient::ClockSrvClientObj = NULL;

/**
 * Constructor of class ClockSrvClient
 */
ClockSrvClient::ClockSrvClient():
   _ClockProxy(Clock_main_fiProxy::createProxy("clockFiPort", *this))
{
   if (_ClockProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _ClockProxy->getPortName());
   }
   m_u8DateFormat = clock_main_fi_types::VDCLK_TEN_NewDateFormat__VDCLK_EN_DF_dd_mm_yyyy_Dash;
}


/**
 * Destructor of class ClockSrvClient
 */
ClockSrvClient::~ClockSrvClient()
{
}


void ClockSrvClient::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("ClockSrvClient::onAvailable"));
   StartupSync::getInstance().onAvailable(proxy, stateChange);
   if (proxy == _ClockProxy)
   {
      ETG_TRACE_USR4(("ClockSrvClient: onAvailable called With State:%d, Action: UpReg", stateChange.getCurrentState()));
      _ClockProxy->sendNewDateFormatUpReg(*this);
   }
}


void ClockSrvClient::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("ClockSrvClient::onUnavailable"));
   StartupSync::getInstance().onUnavailable(proxy, stateChange);
   _ClockProxy->sendNewDateFormatRelUpRegAll();
}


void ClockSrvClient::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   if (proxy == _ClockProxy)
   {
      ETG_TRACE_USR4(("ClockSrvClient: registerProperties called With State:%d, Action: UpReg", stateChange.getCurrentState()));
      _ClockProxy->sendNewDateFormatUpReg(*this);
   }
}


void ClockSrvClient::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   if (proxy == _ClockProxy)
   {
      ETG_TRACE_USR4(("ClockSrvClient: deregisterProperties called With State:%d", stateChange.getCurrentState()));
      _ClockProxy->sendNewDateFormatRelUpRegAll();
   }
}


/**
 * onNewDateFormatError - Error handling for DateFormatProperty
 * @param[in] proxy
 * @param[in] error
 * @param[out] none
 * @return void
 * HISTORY:
 * 26.08.2019 pka5cob
 * Rev 0.1 Initial Revision.
 */
void ClockSrvClient::onNewDateFormatError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& /*proxy*/, const ::boost::shared_ptr< clock_main_fi::NewDateFormatError >& /*error*/)
{
   ETG_TRACE_ERR(("ClockSrvClient:onNewDateFormatError is called"));
}


/**
 * onNewDateFormatStatus - Status handling for DateFormatProperty
 * @param[in] proxy
 * @param[in] Status
 * @param[out] none
 * @return void
 * HISTORY:
 * 26.08.2019 pka5cob
 * Rev 0.1 Initial Revision.
 */
void ClockSrvClient::onNewDateFormatStatus(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy, const ::boost::shared_ptr< clock_main_fi::NewDateFormatStatus >& status)
{
   if (proxy && (proxy == _ClockProxy))
   {
      if (status->hasEnDateFormat())
      {
         ETG_TRACE_USR4(("onNewDateFormatStatus received with TimeFormat: %d", ETG_CENUM(enDateFormat, (enDateFormat) status->getEnDateFormat())));
         m_u8DateFormat = status->getEnDateFormat();
      }
   }
}


/**
 * evaluateDateFormat - Date string formation based on the current set date format.
 * @param[in] day
 * @param[in] month
 * @param[in] year
 * @param[out] none
 * @return std::string
 * HISTORY:
 * 26.08.2019 pka5cob
 * Rev 0.1 Initial Revision.
 */
std::string ClockSrvClient::evaluateDateFormat(std::string day, std::string month, std::string year)
{
   std::string formatedDate = "";
   ETG_TRACE_USR4(("ClockSrvClient::evaluateDateFormat entry"));
   switch (m_u8DateFormat)
   {
      case ::clock_main_fi_types::VDCLK_TEN_NewDateFormat__VDCLK_EN_DF_dd_mm_yyyy_Dash:
         formatedDate.append(day);
         formatedDate.append("-");
         formatedDate.append(month);
         if (year.size() != 0)
         {
            formatedDate.append("-");
            formatedDate.append(year);
         }
         ETG_TRACE_USR4(("ClockSrvClient::format of new date %s", formatedDate.c_str()));
         break;
      case ::clock_main_fi_types::VDCLK_TEN_NewDateFormat__VDCLK_EN_DF_mm_dd_yyyy_Dash:
         formatedDate.append(month);
         formatedDate.append("-");
         formatedDate.append(day);
         if (year.size() != 0)
         {
            formatedDate.append("-");
            formatedDate.append(year);
         }
         ETG_TRACE_USR4(("ClockSrvClient::format of new date %s", formatedDate.c_str()));
         break;
      case ::clock_main_fi_types::VDCLK_TEN_NewDateFormat__VDCLK_EN_DF_yyyy_mm_dd_Dash:
         if (year.size() != 0)
         {
            formatedDate.append(year);
            formatedDate.append("-");
         }
         formatedDate.append(month);
         formatedDate.append("-");
         formatedDate.append(day);
         ETG_TRACE_USR4(("ClockSrvClient::format of new date %s", formatedDate.c_str()));
         break;
      case ::clock_main_fi_types::VDCLK_TEN_NewDateFormat__VDCLK_EN_DF_dd_mm_yyyy_Slash:
         formatedDate.append(day);
         formatedDate.append("/");
         formatedDate.append(month);
         if (year.size() != 0)
         {
            formatedDate.append("/");
            formatedDate.append(year);
         }
         ETG_TRACE_USR4(("ClockSrvClient::format of new date %s", formatedDate.c_str()));
         break;
      case ::clock_main_fi_types::VDCLK_TEN_NewDateFormat__VDCLK_EN_DF_mm_dd_yyyy_Slash:
         formatedDate.append(month);
         formatedDate.append("/");
         formatedDate.append(day);
         if (year.size() != 0)
         {
            formatedDate.append("/");
            formatedDate.append(year);
         }
         ETG_TRACE_USR4(("ClockSrvClient::format of new date %s", formatedDate.c_str()));
         break;
      case ::clock_main_fi_types::VDCLK_TEN_NewDateFormat__VDCLK_EN_DF_dd_mm_yyyy_Dot:
         formatedDate.append(day);
         formatedDate.append(".");
         formatedDate.append(month);
         formatedDate.append(".");
         if (year.size() != 0)
         {
            formatedDate.append(year);
         }
         ETG_TRACE_USR4(("ClockSrvClient::format of new date %s", formatedDate.c_str()));
         break;
      default:
         ETG_TRACE_USR4(("DateFormat unset!"));
         break;
   }
   return formatedDate;
}


} //Core
} //App
