/****************************************************************************
* Copyright (c) 2019-2020 Robert Bosch Car Multimedia GmbH
* duplication and disclosure to third parties is prohibited.
*
* FILE                : ApplicationSwitchClientHandler.cpp
* COMPONENT Name      : AppHmi_Phone
* DESCRIPTION         : DBus client for Application switch intrerface.
* AUTHOR              : Ramesh Kesavan
* Revision History    : 0.1
* Date 23.08.2017     : Initial version
****************************************************************************/

/*****************************************************************
| includes
|----------------------------------------------------------------*/
#include "ApplicationSwitchClientHandler.h"
#include "hmi_trace_if.h"
#include <string>
#include "Core/Microphone/MicClientHandler.h"
#include "../VehicleSrvClient/VehicleSrvClient.h"
#include "App/Core/Telephone/ServiceTel.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_PHONE_HALL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TRACE_CHANNEL TR_TTFIS_APPHMI_PHONE
#define ETG_I_TTFIS_CMD_PREFIX "APPHMI_PHONE_"
#define ETG_I_FILE_PREFIX App::Core::ApplicationSwitchClientHandler::
#include "trcGenProj/Header/ApplicationSwitchClientHandler.cpp.trc.h"
#endif

using namespace ::bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch;

namespace App {
namespace Core {


ApplicationSwitchClientHandler::ApplicationSwitchClientHandler(const std::string& portName):
   m_enPhoneContextStatus(EN_PHONE_CONTEXT_UNKNOWN),
   m_bIsPhoneContextBgRequested(false),
   m_enCurrentPhoneStatus(EN_PHONE_CONTEXT_BG),
   m_bPhoneContextRVCStatusflag(false),
   m_isPermanentActivityID(true),
   m_bPrevAppflyinState(false),
   m_bAppflyinInitialState(true),
   m_entargetActivityID(enActivityIDs__eActivityID_PHONE_CALL_LIST),
   m_u8TargetAppID(APPID_APPHMI_MASTER), //Home screen set as Default application
   m_enActivityID(enActivityIDs__eActivityID_NONE),
   m_enCurrPermanentViewActivityID(enActivityIDs__eActivityID_PHONE_DEVICE_SELECTION),
   m_enCurrIntermediateViewActivityID(enActivityIDs__eActivityID_NONE),
   m_poApplicationSwitchClient(ApplicationSwitchProxy::createProxy(portName, *this))
{
   if (m_poApplicationSwitchClient.get())
   {
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler m_poApplicationSwitchClient.get is TRUE"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, std::string(portName));
   }
   _applicationSwitchClientComponent = ApplicationSwitchClientComponent::poGetInstance();
}


ApplicationSwitchClientHandler::~ApplicationSwitchClientHandler()
{
   /*if (NULL != m_poSelfAppSwitchClient)
   {
   delete m_poSelfAppSwitchClient;
   m_poSelfAppSwitchClient = NULL;
   }*/
}


void ApplicationSwitchClientHandler::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onAvailable"));
   StartupSync::getInstance().onAvailable(proxy, stateChange);

   if (m_poApplicationSwitchClient == proxy)
   {
      m_poApplicationSwitchClient->sendMapout_Activate_DeactivateRegister(*this);
      m_poApplicationSwitchClient->sendSig_ActivateContextRegister(*this);
      m_poApplicationSwitchClient->sendSig_ControlPanelButtonPressRegister(*this);
   }
   else
   {
      ETG_TRACE_USR4(("m_poApplicationSwitchClient is not available"));
   }
}


void ApplicationSwitchClientHandler::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onUnavailable"));
   StartupSync::getInstance().onUnavailable(proxy, stateChange);
   m_poApplicationSwitchClient->sendMapout_Activate_DeactivateDeregisterAll();
   m_poApplicationSwitchClient->sendSig_ActivateContextDeregisterAll();
   m_poApplicationSwitchClient->sendSig_ControlPanelButtonPressDeregisterAll();
}


void ApplicationSwitchClientHandler::registerProperties(const ::boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::registerProperties()"));

   if (m_poApplicationSwitchClient == proxy)
   {
      m_poApplicationSwitchClient->sendMapout_Activate_DeactivateRegister(*this);
      m_poApplicationSwitchClient->sendSig_ActivateContextRegister(*this);
      m_poApplicationSwitchClient->sendSig_ControlPanelButtonPressRegister(*this);
   }
   else
   {
      ETG_TRACE_USR4(("m_poApplicationSwitchClient is not available"));
   }
}


void ApplicationSwitchClientHandler::deregisterProperties(const ::boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::deregisterProperties()"));
   m_poApplicationSwitchClient->sendMapout_Activate_DeactivateDeregisterAll();
}


void ApplicationSwitchClientHandler::onActivateDeactivateRVCSceneError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< ActivateDeactivateRVCSceneError >& error)
{
}


void ApplicationSwitchClientHandler::onActivateDeactivateRVCSceneResponse(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< ActivateDeactivateRVCSceneResponse >& response)
{
}


void ApplicationSwitchClientHandler::onRequestApplcationSwitchResponse(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< RequestApplcationSwitchResponse >& response)
{
}


void ApplicationSwitchClientHandler::onRequestApplcationSwitchError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< RequestApplcationSwitchError >& error)
{
}


void ApplicationSwitchClientHandler::onRequestShowHideWaitSceneError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< RequestShowHideWaitSceneError >& error)
{
}


void ApplicationSwitchClientHandler::onRequestShowHideWaitSceneResponse(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< RequestShowHideWaitSceneResponse >& response)
{
}


void ApplicationSwitchClientHandler::onMapout_Activate_DeactivateError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Mapout_Activate_DeactivateError >& error)
{
}


void ApplicationSwitchClientHandler::onMapout_Activate_DeactivateSignal(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Mapout_Activate_DeactivateSignal >& signal)
{
}


void ApplicationSwitchClientHandler::onSig_ActivateContextError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Sig_ActivateContextError >& error)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextError"));
}


void ApplicationSwitchClientHandler::onSig_ActivateContextSignal(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Sig_ActivateContextSignal >& signal)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal %d", signal->getTargetactivityId()));
   m_entargetActivityID = signal->getTargetactivityId();
   m_u8TargetAppID = signal->getTargetappId();

   /* On change of application ID to other than phone, reset the background context flag to false */
   /*    if (m_u8TargetAppID != APPID_APPHMI_PHONE)
      {
         m_bIsPhoneContextBgRequested = false;
      } */

   if ((enActivityIDs__eActivityID_RVC == m_entargetActivityID) ||
         (enActivityIDs__eActivityID_RVC_SETTINGS == m_entargetActivityID))
   {
      g_SM_IsRVCViewEnabled = true;
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal %d -  RVC Enabled. ", signal->getTargetactivityId()));
   }
   else
   {
      g_SM_IsRVCViewEnabled = false;
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal %d - RVC Disabled. ", signal->getTargetactivityId()));
   }

   /*
   IF(PHONEIN FOREGROUND AND rvc ENABLED) - Switch to RVC view while Phone in FG
   {
   	RESULT : Phone will be in BG becasue of RVC, dont request and forward context switch.
   			 because, phone in BG because of RVC spl case

   			 RVC_SPL_FLAG -set
   }
   elss if(Phone in Background and RVC is disabled) -  - switch to any application from RVC while phone in BG
   {
   	// because RVC disable will come before Phone comes to foreground
   	if (RVC_SPL_FLAG ==true)
   	{
   		if set - phone was in FG when RVC was enaged
   		RVC_SPL_FLAG - reset to flase.
   	}
   	else
   	{
   		if not set - other application was in FG when RVC was enaged
   	}
   }
   else
   {
   	Phone in Foreground and RvC is disabled - switch to any application except RVC while phone in FG
   	-- FM-> accept incoming call -> Active call -> Home Screen -> E.RVC -> Terminate (back req) -> D.RVC ->  home
   	-- FM-> accept incoming call -> Active call -> Home Screen -> E.RVC -> D.RVC -> Home -> Terminate (back req)

   	Phone in Backgorund and RVC is enabled - Switch to RVC view while Anyother application in Foreground
   	-- FM -> E.RVC -> accept incoming call(PTT) -> Active call -> D.RVC -> Active -> Terminate (back req) -> FM
   	-- FM -> E.RVC -> accept incoming call(PTT) -> Active call -> Terminate (back req) -> D.RVC -> FM
   	-- FM -> E.RVC -> accept incoming call(PTT) -> Active call -> D.RVC -> Active -> Home Screen -> Terminate (back req) -> : HomeScreen
   	-- FM -> E.RVC -> accept incoming call(PTT) -> Active call -> Home Screen(Not possible) -> Terminate (back req) -> D.RVC ->  : FM
   }
   */
   if ((m_enCurrentPhoneStatus == EN_PHONE_CONTEXT_FG) && (g_SM_IsRVCViewEnabled == true))
   {
      m_bPhoneContextRVCStatusflag = true;
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal: RVCStatusflag TRUE"));
   }
   else if ((m_enCurrentPhoneStatus == EN_PHONE_CONTEXT_BG) && (g_SM_IsRVCViewEnabled == false))
   {
      if (m_bPhoneContextRVCStatusflag == true)
      {
         m_bPhoneContextRVCStatusflag = false;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal: RVCStatusflag FALSE"));
      }
      else
      {
         //if flag is not set - other application was in FG when RVC was enaged
      }
   }
   else
   {
      //Phone in Foreground and RvC is disabled - switch to any application except RVC while phone in FG
      //Phone in Backgorund and RVC is enabled - Switch to RVC view while Anyother application in Foreground
   }

   /* Evaluate activity ID's based on the internal state of the phone component, irrespective of request being placed by master component */
   if ((g_SM_SpiSessionStatus != SPI_SESSION_ANDROID_AUTO_ACTIVE) && (g_SM_SpiSessionStatus != SPI_SESSION_CARPLAY_ACTIVE))
   {
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal: Evaluate the phone activity ID "));

      /* Evaluate activity ID's based on the internal call state of the phone component */
      if (((ServiceTel::poGetInstance()->getCallDialingStatus() == true) || (ServiceTel::poGetInstance()->getOngoingCallStatus() == true)) &&
            ((m_entargetActivityID == enActivityIDs__eActivityID_PHONE_CALL_CONNECTING) || (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_ON_CALL) ||
             (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_ON_MULTIPARTY_CALL) || (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_CALL_DIALPAD) ||
             (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_MULTIPLE_CALL_DIALPAD) || (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_CALL_LIST) ||
             (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_BT_VR)))
      {
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::OnUpdonSig_ActivateContextSignal evaluated internal call state and BTVR "));
         OnUpdonSig_ActivateContextSignal_Phone();
      }
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      /* Evaluate activity ID's based on the internal intercom call state of the phone component */
      else if (((IntercomClienthandler::poGetInstance()->getIntercomCallDialingStatus() == true) || (IntercomClienthandler::poGetInstance()->getIntercomOngoingCallStatus() == true) || (IntercomClienthandler::poGetInstance()->getIntercomConfCallStatus() == true)) &&
               ((m_entargetActivityID == enActivityIDs__eActivityID_PHONE_ON_INTERCOMCALL) || (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_ON_I_CONTACTS) ||
                (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_ON_I_CALL_CONNECTING) || (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_ON_I_CONFERENCECALL)))
      {
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal Intercom"));
         OnUpdonSig_ActivateContextSignal_Intercom();
      }
#endif
      else if (((ServiceTel::poGetInstance()->getCallDialingStatus() == true) || (ServiceTel::poGetInstance()->getOngoingCallStatus() == true)) &&
               ((m_entargetActivityID == enActivityIDs__eActivityID_PHONE_CALL_CONNECTING) || (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_ON_CALL) ||
                (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_ON_MULTIPARTY_CALL) || (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_CALL_DIALPAD) ||
                (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_MULTIPLE_CALL_DIALPAD) || (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_ON_INTERCOMCALL) ||
                (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_ON_I_CONTACTS) || (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_ON_I_CALL_CONNECTING) ||
                (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_ON_I_CONFERENCECALL)))
      {
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::OnUpdonSig_ActivateContextSignal evaluated internal call state and BTVR "));
         OnUpdonSig_ActivateContextSignal_Phone();
      }
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      else if (((IntercomClienthandler::poGetInstance()->getIntercomCallDialingStatus() == true) || (IntercomClienthandler::poGetInstance()->getIntercomOngoingCallStatus() == true) || (IntercomClienthandler::poGetInstance()->getIntercomConfCallStatus() == true)) &&
               ((m_entargetActivityID == enActivityIDs__eActivityID_PHONE_ON_INTERCOMCALL) || (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_ON_I_CALL_CONNECTING) || (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_ON_I_CONFERENCECALL) ||
                (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_CALL_LIST) || (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_BT_VR) ||
                (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_CONTACT_LIST) || (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_DIALPAD) ||
                (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_CALL_LIST_LOADING) || (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_CONTACTS_LOADING) ||
                (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_CALL_CONNECTING) || (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_ON_CALL) ||
                (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_ON_MULTIPARTY_CALL) || (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_CALL_DIALPAD) ||
                (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_MULTIPLE_CALL_DIALPAD)))
      {
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal Intercom"));
         OnUpdonSig_ActivateContextSignal_Intercom();
      }
#endif

      else
      {
         // There is not active call, when coming to foreground
         switch (signal->getTargetactivityId())
         {
            case enActivityIDs__eActivityID_PHONE_CALL_CONNECTING:
            case enActivityIDs__eActivityID_PHONE_CALL_DISCONNECTING:
            case enActivityIDs__eActivityID_PHONE_ON_CALL:
            case enActivityIDs__eActivityID_PHONE_ON_MULTIPARTY_CALL:
            case enActivityIDs__eActivityID_PHONE_CALL_DIALPAD:
            case enActivityIDs__eActivityID_PHONE_MULTIPLE_CALL_DIALPAD:
               // TO be revisited in RVC  SYSFL
               //Whenever there is a request for transition ID's, assign the previous permanent activity ID's.
               if (ServiceBT::poGetInstance()->GetNoOfHFPDevicesConnected() != 0)
               {
                  m_entargetActivityID = m_enCurrPermanentViewActivityID;
                  ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal Permanent ID "));
               }
               else
               {
                  //On phone context re-entry, if previous HFP devices got disconnected, go to device selection screen.
                  m_entargetActivityID = enActivityIDs__eActivityID_PHONE_DEVICE_SELECTION;
                  ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal Phone device selection "));
               }
               break;

#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
            case enActivityIDs__eActivityID_PHONE_ON_INTERCOMCALL:
            case enActivityIDs__eActivityID_PHONE_ON_I_CALL_CONNECTING:
            case enActivityIDs__eActivityID_PHONE_ON_I_CONFERENCECALL:
            {
               if (true == IntercomClienthandler::poGetInstance()->getIntercomServiceAvailability())
               {
                  m_entargetActivityID = m_enCurrPermanentViewActivityID;
                  ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal Intercom call not active"));
               }
               else
               {
                  m_entargetActivityID = enActivityIDs__eActivityID_PHONE_DEVICE_SELECTION;
                  ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal Phone device selection "));
               }
            }
            break;
#endif

            case enActivityIDs__eActivityID_PHONE_BT_VR:
            {
               ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal BTVR case "));
               if (ServiceBT::poGetInstance()->GetNoOfHFPDevicesConnected() != 0)
               {
                  /* Assign BTVR screen only when the middleware BTVR status is already set as true */
                  if ((TRUE == bBTVRViewShown) && (PhoneDataPoolConfig::getInstance()->isVRSupported()))
                  {
                     m_entargetActivityID = enActivityIDs__eActivityID_PHONE_BT_VR;
                     ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal BTVR active "));
                  }
                  else
                  {
                     ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal BTVR not active "));
                     m_entargetActivityID = m_enCurrPermanentViewActivityID;
                  }
               }
               else
               {
                  //On phone context re-entry, if previous HFP devices got disconnected, go to device selection screen.
                  m_entargetActivityID = enActivityIDs__eActivityID_PHONE_DEVICE_SELECTION;
                  ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal Phone device selection "));
               }
            }
            break;
            default:
               //By default, assign the whatever values comes from Master
               m_entargetActivityID = signal->getTargetactivityId();
               ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal default "));
               break;
         }
      }
   }
   else
   {
      //If SPI session is active, assign Phone selection ID.
      m_entargetActivityID = enActivityIDs__eActivityID_PHONE_DEVICE_SELECTION;
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal: SPI session is active "));
   }

   // only Process the CASES which doesnot involve in contest switch
   ProcessExternalSceneTransitionTriggers();
}


void ApplicationSwitchClientHandler::OnUpdonSig_ActivateContextSignal_Phone()
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::OnUpdonSig_ActivateContextSignal_Phone evaluated internal call state and BTVR "));
   /* Assign the activity ID of call_connecting when call dialling is in progress */
   if (ServiceTel::poGetInstance()->getCallDialingStatus() == true)
   {
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::OnUpdonSig_ActivateContextSignal_Phone Call Connecting "));
      m_entargetActivityID = enActivityIDs__eActivityID_PHONE_CALL_CONNECTING;
   }
   /* Assign the activity ID of respective CALL View when active call is in progress */
   else
   {
      if (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_ON_MULTIPARTY_CALL)
      {
         m_entargetActivityID = enActivityIDs__eActivityID_PHONE_ON_MULTIPARTY_CALL;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::OnUpdonSig_ActivateContextSignal_Phone Multiparty Call"));
      }
      else if (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_CALL_DIALPAD)
      {
         m_entargetActivityID = enActivityIDs__eActivityID_PHONE_CALL_DIALPAD;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::OnUpdonSig_ActivateContextSignal_Phone Call Dialpad "));
      }
      else if (m_entargetActivityID == enActivityIDs__eActivityID_PHONE_MULTIPLE_CALL_DIALPAD)
      {
         m_entargetActivityID = enActivityIDs__eActivityID_PHONE_MULTIPLE_CALL_DIALPAD;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::OnUpdonSig_ActivateContextSignal_Phone Multiple Call Dialpad  "));
      }
      else
      {
         if (ServiceTel::poGetInstance()->bIsMultipleCallActive() == false)
         {
            // only one call instance
            m_entargetActivityID = enActivityIDs__eActivityID_PHONE_ON_CALL;
            ETG_TRACE_USR4(("ApplicationSwitchClientHandler::OnUpdonSig_ActivateContextSignal_Phone On Call"));
         }
         else if (ServiceTel::poGetInstance()->bIsMultipleCallActive() == true)
         {
            //More than one call instance, Multiparty or Conference Call
            m_entargetActivityID = enActivityIDs__eActivityID_PHONE_ON_MULTIPARTY_CALL;
            ETG_TRACE_USR4(("ApplicationSwitchClientHandler::OnUpdonSig_ActivateContextSignal_Phone Multiparty Call"));
         }
      }
   }
}


#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
void ApplicationSwitchClientHandler::OnUpdonSig_ActivateContextSignal_Intercom()
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal: On Intercom Calls check"));
   if (true == IntercomClienthandler::poGetInstance()->getIntercomServiceAvailability())
   {
      if (IntercomClienthandler::poGetInstance()->getIntercomCallDialingStatus() == true)
      {
         m_entargetActivityID = enActivityIDs__eActivityID_PHONE_ON_I_CALL_CONNECTING;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal: On Intercom Dialing show Intercom Contacts Scene"));
      }
      else if (IntercomClienthandler::poGetInstance()->getIntercomOngoingCallStatus() == true)
      {
         m_entargetActivityID = enActivityIDs__eActivityID_PHONE_ON_INTERCOMCALL;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal: Intercom Call Scene"));
      }
      else if (IntercomClienthandler::poGetInstance()->getIntercomConfCallStatus() == true)
      {
         m_entargetActivityID = enActivityIDs__eActivityID_PHONE_ON_I_CONFERENCECALL;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal: Intercom Call Scene"));
      }
      else
      {
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal: Intercom Call Scene %d", m_entargetActivityID));
      }
   }
   else
   {
      m_entargetActivityID = enActivityIDs__eActivityID_PHONE_DEVICE_SELECTION;
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal: Intercom Unavailable"));
   }
}


#endif
void ApplicationSwitchClientHandler::onActiveRegionIdError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< ActiveRegionIdError >& error)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onActiveRegionIdError"));
}


void ApplicationSwitchClientHandler::onActiveRegionIdUpdate(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< ActiveRegionIdUpdate >& update)
{
}


void ApplicationSwitchClientHandler::displayPhoneContextScene()
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene %d m_enActivityID %d", m_entargetActivityID, m_enActivityID));
   if (m_enActivityID != m_entargetActivityID)
   {
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene current activityId and target activityId are diffterent"));
      switch (m_entargetActivityID)
      {
         case enActivityIDs__eActivityID_PHONE_DEVICE_SELECTION:
         {
            ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene::Posting ActivateAddDeviceMsg"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateAddDeviceMsg)()));

            break;
         }
         // Implementing following changes for SMART customer bugfix
         case enActivityIDs__eActivityID_PHONE_CALL_LIST:
         {
            ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene::Posting ActivateCallListMsg"));
#ifndef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateCallListMsg)()));
#else
            POST_MSG((COURIER_MESSAGE_NEW(PhoneAppflyinReq)()));
#endif
            break;
         }
         case enActivityIDs__eActivityID_PHONE_CALL_LIST_LOADING:
         {
            ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene::Posting ActivateCallListLoadingMsg"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateCallListLoadingMsg)()));
            break;
         }
         case enActivityIDs__eActivityID_PHONE_CONTACTS_LOADING:
         {
            ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene::Posting ActivateContactListLoadingMsg"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateContactListLoadingMsg)()));
            break;
         }
         case enActivityIDs__eActivityID_PHONE_CONTACT_LIST:
         {
            ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene::Posting ActivateContactListMsg"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateContactListMsg)()));
            break;
         }
         case enActivityIDs__eActivityID_PHONE_CONTACTS_CALL:
         {
            ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene::Posting ActivateOnContactCallMsg"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnContactCallMsg)()));
            break;
         }
         case enActivityIDs__eActivityID_PHONE_DIALPAD:
         {
            ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene::Posting ActivateDialPadMsg"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateDialPadMsg)()));
            break;
         }
         case enActivityIDs__eActivityID_PHONE_CALL_CONNECTING:
         {
            ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene::Posting ActivateCallConnectingSceneMsg"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateCallConnectingSceneMsg)()));
            break;
         }
         case enActivityIDs__eActivityID_PHONE_ON_CALL:
         {
            ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene::Posting ActivateOnCallSceneMsg"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnCallSceneMsg)()));
            break;
         }
         case enActivityIDs__eActivityID_PHONE_ON_MULTIPARTY_CALL:
         {
            ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene::switch to Multi party call"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnMultipleCallSceneMsg)()));
            break;
         }
         case enActivityIDs__eActivityID_PHONE_SETTINGS_VIEW:
         {
            ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene::switch to Phone Setting"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnSettingSceneMsg)()));
            break;
         }
         case enActivityIDs__eActivityID_PHONE_BT_VR:
         {
            /* Post the Courier message only when Android auto session and Carplay is not active */
            if ((g_SM_SpiSessionStatus != SPI_SESSION_ANDROID_AUTO_ACTIVE) && (g_SM_SpiSessionStatus != SPI_SESSION_CARPLAY_ACTIVE))
            {
               ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene::switch to BT_VR scene"));

               // BTVR Trigger is already handled in the following three case
               // case 1 : triggered via notch display - serviced by ProcessExternalSceneTransitionTriggers method on onSig_ActivateContextSignal interface call
               // case 2 : Triggered via MCP/SWC when Phone in BG - on context switch serivce -  onSig_ActivateContextSignal->ProcessExternalSceneTransitionTriggers method
               // case 3 : Triggered via MCP/SWC when phone in FG - serviceed by sendRequestForContextSwitching_internal method in Internal PhoneHall.
               //
               // This case is just needed to trigger the default screen (add device in case of context switch request)

               // Phone App screen surface change is needed to avoid context switch request expire
               // on activating add device, ContextSwitchHandler: onAplicationSwitchComplete is obtained.
               //POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateAddDeviceMsg)()))
               /* Creating BTVRStatusResp courier message */
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
               if (MicClientHandler::poGetInstance()->getMicStatus() == false) //MIC Check for BT VR
               {
                  ETG_TRACE_USR4(("ApplicationSwitchClientHandler : MIC is not Active Allowing BTVR Request"));
#endif
                  if (TRUE == bBTVRViewShown && PhoneDataPoolConfig::getInstance()->isVRSupported())
                  {
                     ETG_TRACE_USR4(("ApplicationSwitchClientHandler::switch to BT_VR scene entered"));
                     POST_MSG((COURIER_MESSAGE_NEW(BTVRStatusResp)(true)));
                  }
                  else
                  {
                     //On receiving BTVR request when already BTVR is active, the active BTVR session shall need to be exited.
                     ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene::Toggle From BT_VR scene"));
                     POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(BTVRSessionExitAppReqMsg)()));
                  }
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
               }
               else
               {
                  ETG_TRACE_USR4(("ApplicationSwitchClientHandler::switch to BT_VR scene is not started"));
               }
#endif
               // tobe removed when BTVR is changed from Context switch to Interface. As of now, on showing BTVR,
               // SendContext is not done in SM, as a result, Master Reset to Context switch to Home Screen
               // This lead to phone in fg without Appflying at end of call. and CANT NAVIGATE TO HOME again.
               // To avoid this Request Expire send context to master on BTVR display
               //requestSendContext((uint8)2 , (uint8)APPID_APPHMI_PHONE, enActivityIDs__eActivityID_PHONE_BT_VR);
            }
            break;
         }
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
         case enActivityIDs__eActivityID_PHONE_ON_INTERCOMCALL:
         {
            ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene::Posting ActivateOnIntercomCallSceneMsg"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnIntercomCallSceneMsg)()));
            break;
         }
         case enActivityIDs__eActivityID_PHONE_ON_I_CONTACTS:
         {
            ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene::Posting ActivateOnIContactsSceneMsg"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnIContactsSceneMsg)()));
            break;
         }
         case enActivityIDs__eActivityID_PHONE_ON_I_CONFERENCECALL:
         {
            ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene::Posting ActivateOnIntercomConfCallSceneMsg"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnIntercomConfCallSceneMsg)()));
            break;
         }
         case enActivityIDs__eActivityID_PHONE_ON_I_CALL_CONNECTING:
         {
            ETG_TRACE_USR4(("ApplicationSwitchClientHandler::displayPhoneContextScene::Posting ActivateOnICallConnectingSceneMsg"));
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnICallConnectingSceneMsg)()));
            break;
         }
#endif
         default:
         {
            break;
         }
      }
   }
}


void ApplicationSwitchClientHandler::ProcessExternalSceneTransitionTriggers()
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::ProcessExternalSceneTransitionTriggers %d", m_entargetActivityID));

   switch (m_entargetActivityID)
   {
      /*  case enActivityIDs__eActivityID_PHONE_CALL_CONNECTING:
        {
           ETG_TRACE_USR4(("ApplicationSwitchClientHandler::ProcessExternalSceneTransitionTriggers::Posting ActivateCallConnectingSceneMsg"));
           POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateCallConnectingSceneMsg)()));
           break;
        }
        case enActivityIDs__eActivityID_PHONE_ON_CALL:
        {
           ETG_TRACE_USR4(("ApplicationSwitchClientHandler::ProcessExternalSceneTransitionTriggers::Posting ActivateOnCallSceneMsg"));
           POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnCallSceneMsg)()));
           break;
        }
        case enActivityIDs__eActivityID_PHONE_ON_MULTIPARTY_CALL:
        {
           ETG_TRACE_USR4(("ApplicationSwitchClientHandler::ProcessExternalSceneTransitionTriggers::switch to Multi party call - To be Handled"));
           break;
        }*/
      case enActivityIDs__eActivityID_PHONE_SETTINGS_VIEW:
      {
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::ProcessExternalSceneTransitionTriggers::switch to Phone Setting"));
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnSettingSceneMsg)()));
         break;
      }
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      /* case enActivityIDs__eActivityID_PHONE_ON_INTERCOMCALL:
      {
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::ProcessExternalSceneTransitionTriggers::Posting ActivateOnIntercomCallSceneMsg"));
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnIntercomCallSceneMsg)()));
         break;
      } */
      case enActivityIDs__eActivityID_PHONE_ON_I_CONTACTS:
      {
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::ProcessExternalSceneTransitionTriggers::Posting ActivateOnIContactsSceneMsg"));
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnIContactsSceneMsg)()));
         break;
      }
         /*  case enActivityIDs__eActivityID_PHONE_ON_I_CONFERENCECALL:
          {
             ETG_TRACE_USR4(("ApplicationSwitchClientHandler::ProcessExternalSceneTransitionTriggers::Posting ActivateOnIntercomConfCallSceneMsg"));
             POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ActivateOnIntercomConfCallSceneMsg)()));
             break;
          } */
#endif
      default:
      {
         break;
      }
   }
}


/************************************************************************
*FUNCTION:		getActivityIDforCurrentView
*DESCRIPTION:	Updates the master with current activity ID
*PARAMETER:		void
*RETURNVALUE:	bool
*SWFL:			General
*HISTORY:
*0.1     19.07.19      Sai Prabha   Initial Version
************************************************************************/
APPLICATIONSWITCH_TYPES::enActivityIDs ApplicationSwitchClientHandler::getActivityIDforCurrentView(void)
{
   // reset IntermediateViewActivityID on every new update, this was if there is no update in intermediate iD, it will be cleared
   // DO NOT RESET = Permanent Scene IDs.
   m_enCurrIntermediateViewActivityID = enActivityIDs__eActivityID_NONE;
   switch (CurrentView)
   {
      case AppHmi_PhoneModule_PhoneScenes_P_CallList:
      {
         m_enActivityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_PHONE_CALL_LIST;
         m_enCurrPermanentViewActivityID = m_enActivityID;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIDforCurrentView() m_enActivityID = %d" , m_enActivityID));

         break;
      }
      case AppHmi_PhoneModule_PhoneScenes_P_Contacts:
      {
         m_enActivityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_PHONE_CONTACT_LIST;
         m_enCurrPermanentViewActivityID = m_enActivityID;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIDforCurrentView() m_enActivityID = %d" , m_enActivityID));

         break;
      }
      case AppHmi_PhoneModule_PhoneScenes_P_Call:
      {
         m_enActivityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_PHONE_ON_CALL;
         m_enCurrIntermediateViewActivityID = m_enActivityID;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIDforCurrentView() m_enActivityID = %d", m_enActivityID));

         break;
      }
      case AppHmi_PhoneModule_PhoneScenes_P_Call_Connecting:
      {
         m_enActivityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_PHONE_CALL_CONNECTING;
         m_enCurrIntermediateViewActivityID = m_enActivityID;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIDforCurrentView() m_enActivityID = %d ", m_enActivityID));

         break;
      }
      case AppHmi_PhoneModule_PhoneScenes_P_MultipleCall:
      {
         m_enActivityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_PHONE_ON_MULTIPARTY_CALL;
         m_enCurrIntermediateViewActivityID = m_enActivityID;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIDforCurrentView() m_enActivityID = %d ", m_enActivityID));

         break;
      }
      case AppHmi_PhoneModule_PhoneScenes_P_PhoneSelection:
      {
         m_enActivityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_PHONE_DEVICE_SELECTION;
         m_enCurrPermanentViewActivityID = m_enActivityID;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIDforCurrentView() m_enActivityID = %d ", m_enActivityID));

         break;
      }
      case AppHmi_PhoneModule_PhoneScenes_P_OPT:
      {
         m_enActivityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_PHONE_SETTINGS_VIEW;
         m_enCurrIntermediateViewActivityID = m_enActivityID;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIDforCurrentView() m_enActivityID = %d ", m_enActivityID));

         break;
      }
      case AppHmi_PhoneModule_PhoneScenes_P_DialPad:
      {
         m_enActivityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_PHONE_DIALPAD;
         m_enCurrPermanentViewActivityID = m_enActivityID;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIDforCurrentView() m_enActivityID = %d ", m_enActivityID));

         break;
      }
      case AppHmi_PhoneModule_PhoneScenes_P_CallList_Loading:
      {
         m_enActivityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_PHONE_CALL_LIST_LOADING;
         m_enCurrPermanentViewActivityID = m_enActivityID;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIDforCurrentView() m_enActivityID = %d ", m_enActivityID));

         break;
      }
      case AppHmi_PhoneModule_PhoneScenes_P_Contacts_Loading:
      {
         m_enActivityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_PHONE_CONTACTS_LOADING;
         m_enCurrPermanentViewActivityID = m_enActivityID;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIDforCurrentView() m_enActivityID = %d ", m_enActivityID));

         break;
      }
      case AppHmi_PhoneModule_PhoneScenes_P_Contacts_Call:
      {
         m_enActivityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_PHONE_CONTACTS_CALL;
         m_enCurrPermanentViewActivityID = m_enActivityID;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIDforCurrentView() m_enActivityID = %d ", m_enActivityID));

         break;
      }
      case AppHmi_PhoneModule_PhoneScenes_P_Call_Disconnecting:
      {
         m_enActivityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_PHONE_CALL_DISCONNECTING;
         m_enCurrIntermediateViewActivityID = m_enActivityID;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIDforCurrentView() m_enActivityID = %d ", m_enActivityID));

         break;
      }
      case AppHmi_PhoneModule_PhoneScenes_P_Call_DialPad:
      {
         m_enActivityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_PHONE_CALL_DIALPAD;
         m_enCurrIntermediateViewActivityID = m_enActivityID;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIDforCurrentView() m_enActivityID = %d", m_enActivityID));

         break;
      }
      case AppHmi_PhoneModule_PhoneScenes_P_MultipleCall_DialPad:
      {
         m_enActivityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_PHONE_MULTIPLE_CALL_DIALPAD;
         m_enCurrIntermediateViewActivityID = m_enActivityID;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIDforCurrentView() m_enActivityID = %d ", m_enActivityID));

         break;
      }
#ifdef VARIANT_S_FTR_ENABLE_INTERCOM_SUPPORT
      case AppHmi_PhoneModule_PhoneScenes_P_IntercomCall:
      {
         m_enActivityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_PHONE_ON_INTERCOMCALL;
         m_enCurrIntermediateViewActivityID = m_enActivityID;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIDforCurrentView() m_enActivityID = %d", m_enActivityID));

         break;
      }
      case AppHmi_PhoneModule_PhoneScenes_P_Intercom_ConferenceCall:
      {
         m_enActivityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_PHONE_ON_I_CONFERENCECALL;
         m_enCurrIntermediateViewActivityID = m_enActivityID;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIDforCurrentView() m_enActivityID = %d", m_enActivityID));

         break;
      }
      case AppHmi_PhoneModule_PhoneScenes_I_Contacts:
      {
         m_enActivityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_PHONE_ON_I_CONTACTS;
         m_enCurrPermanentViewActivityID = m_enActivityID;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIDforCurrentView() m_enActivityID = %d", m_enActivityID));

         break;
      }
#endif
      case AppHmi_PhoneModule_PhoneScenes_P_SR_SIRI_Main:
      {
         m_enActivityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_PHONE_BT_VR;
         m_enCurrIntermediateViewActivityID = m_enActivityID;
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIDforCurrentView() m_enActivityID = %d ", m_enActivityID));

         break;
      }
      default :
      {
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIDforCurrentView() default case entered"));
         break;
      }
   }
   return m_enActivityID;
}


/************************************************************************
*FUNCTION:			onCourierMessage
*DESCRIPTION:
*PARAMETER:
*RETURNVALUE:
*SWFL:
*HISTORY:
*0.1     19.07.19      Sai Prabha  Initial Version
************************************************************************/
bool ApplicationSwitchClientHandler::onCourierMessage(const ViewPhoneActivityId& oMsg)
{
   ETG_TRACE_USR3(("PhoneHall::onCourierMessage(ViewPhoneActivityId)"));

   APPLICATIONSWITCH_TYPES::enActivityIDs  activityID = getActivityIDforCurrentView();
   requestSendContext((uint8)2 , (uint8)APPID_APPHMI_PHONE, activityID);

   return true;
}


/************************************************************************
*FUNCTION:			onCourierMessage
*DESCRIPTION:       Bt on/off status for Phone appflyin enable/disable
*PARAMETER:         Bt on/off status
*RETURNVALUE:
*SWFL:
*HISTORY:
*0.1     28.04.20      Palani K  Initial Version
************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_BLUETOOTH_ON_OFF
bool ApplicationSwitchClientHandler::onCourierMessage(const BtOnOffStatusAppFlyInReq& oMsg)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onCourierMessage(BtOnOffStatusAppFlyInReq)"));
   std::vector<AppFlyInStateInfo> temp_AppFlyInStateInfo;
   AppFlyInStateInfo temp_obj;

   temp_obj.setRegionId((uint8)2);
   temp_obj.setAppFlyInIndex((uint8)ITEM_PHONE);

   /* When BT is enabled, enable Phone AppFlyIn else disable the same */
   if (oMsg.GetBtOnOffStatus() == true)
   {
      temp_obj.setStatus((uint8)BUTTON_STATE_ENABLED);
   }
   else
   {
      temp_obj.setStatus((uint8)BUTTON_STATE_DISABLED);
   }

   temp_AppFlyInStateInfo.clear();
   temp_AppFlyInStateInfo.push_back(temp_obj) ;

   if (m_poApplicationSwitchClient)
   {
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::requestSendConfigureAppFlyInStatus %d %d %d", temp_obj.getRegionId(), temp_obj.getAppFlyInIndex(), temp_obj.getStatus()));
      //For first time to be able to pass we need to force the appflyin request in constructor ourselves
      //this way the actual update coming will always be the second
      if (m_bPrevAppflyinState != oMsg.GetBtOnOffStatus() || (true == m_bAppflyinInitialState))
      {
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::requestSendConfigureAppFlyInStatus request sent"));
         m_bAppflyinInitialState = false;
         m_poApplicationSwitchClient->sendConfigureAppFlyInStatusRequest(temp_AppFlyInStateInfo);
         m_bPrevAppflyinState = oMsg.GetBtOnOffStatus();
      }
      else
      {
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::requestSendConfigureAppFlyInStatus request not sent Appflyin values variable is same as previous"));
      }
   }

   return true;
}


#endif

/************************************************************************
*FUNCTION:			requestSendContext
*DESCRIPTION:
*PARAMETER:
*RETURNVALUE:
*SWFL:
*HISTORY:
*0.1     19.07.19      Sai Prabha  Initial Version
************************************************************************/
void ApplicationSwitchClientHandler::requestSendContext(uint8  regionId, uint8  sourceAppId, APPLICATIONSWITCH_TYPES::enActivityIDs  sourceActivityId)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::requestSendContext"));
   if (m_poApplicationSwitchClient)
   {
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::requestSendContext %d %d %d", regionId, sourceAppId, sourceActivityId));
      m_poApplicationSwitchClient->sendSendContextRequest(*this, regionId, sourceAppId, sourceActivityId);
   }
}


/************************************************************************
*FUNCTION:			onSendContextError
*DESCRIPTION:
*PARAMETER:
*RETURNVALUE:
*SWFL:
*HISTORY:
*0.1     19.07.19      Sai Prabha  Initial Version
************************************************************************/
void ApplicationSwitchClientHandler::onSendContextError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< SendContextError >& error)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSendContextError()"));
}


/************************************************************************
*FUNCTION:			onSendContextResponse
*DESCRIPTION:
*PARAMETER:
*RETURNVALUE:
*SWFL:
*HISTORY:
*0.1     19.07.19      Sai Prabha  Initial Version
************************************************************************/

void ApplicationSwitchClientHandler::onSendContextResponse(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< SendContextResponse >& response)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSendContextResponse()"));
}


/************************************************************************
*FUNCTION:		SetPhoneContextStatus
*DESCRIPTION:	Sets the status of Phone when context switch was requested
*PARAMETER:		en_phoneContextStatus
*RETURNVALUE:	void
*SWFL:			General
*HISTORY:
*0.1     02.02.20      Ramesh Kesavan  Initial Version
************************************************************************/
void ApplicationSwitchClientHandler::SetPhoneContextStatus(en_phoneContextStatus argPhoneContextStatus)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::SetPhoneContextStatus(%d)", argPhoneContextStatus));
   m_enPhoneContextStatus = argPhoneContextStatus;
}


/************************************************************************
*FUNCTION:		GetPhoneContextStatus
*DESCRIPTION:	returns the status of Phone when context switch was requested
*PARAMETER:		void
*RETURNVALUE:	en_phoneContextStatus
*SWFL:			General
*HISTORY:
*0.1     02.02.20    Ramesh Kesavan    Initial Version
************************************************************************/
en_phoneContextStatus ApplicationSwitchClientHandler::GetPhoneContextStatus()
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::GetPhoneContextStatus() return %d", m_enPhoneContextStatus));
   return m_enPhoneContextStatus;
}


/************************************************************************
*FUNCTION:		SetPhoneContextBgReqFlag
*DESCRIPTION:	Sets the status of Phone background context request
*PARAMETER:		bool
*RETURNVALUE:	void
*SWFL:			General
*HISTORY:
*0.1     10.12.20      Palani K  Initial Version
************************************************************************/
void ApplicationSwitchClientHandler::SetPhoneContextBgReqFlag(bool BgReqFlagStatus)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::SetPhoneContextBgReqFlag(%d)", BgReqFlagStatus));
   m_bIsPhoneContextBgRequested = BgReqFlagStatus;
}


/************************************************************************
*FUNCTION:		GetPhoneContextBgReqFlag
*DESCRIPTION:	returns the status of Phone background context request flag status
*PARAMETER:		void
*RETURNVALUE:	bool
*SWFL:			General
*HISTORY:
*0.1     10.12.20    Palani K    Initial Version
************************************************************************/
bool ApplicationSwitchClientHandler::GetPhoneContextBgReqFlag()
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::GetPhoneContextBgReqFlag() return %d", m_bIsPhoneContextBgRequested));
   return m_bIsPhoneContextBgRequested;
}


/************************************************************************
*FUNCTION:		GetPhoneContextRVCStatus
*DESCRIPTION:	returns the status of Phone context RVC status flag
*PARAMETER:		void
*RETURNVALUE:	bool
*SWFL:			General
*HISTORY:
*0.1     13.11.20    Palani K    Initial Version
************************************************************************/
bool ApplicationSwitchClientHandler::GetPhoneContextRVCStatus()
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::GetPhoneContextRVCStatus() return %d", m_bPhoneContextRVCStatusflag));
   return m_bPhoneContextRVCStatusflag;
}


/************************************************************************
*FUNCTION:		SetCurrentPhoneStatus
*DESCRIPTION:	Sets the Currect context information from Phone hall
*PARAMETER:		argCurrentPhoneStatus
*RETURNVALUE:	void
*SWFL:			General
*HISTORY:
*0.1     02.02.20      Ramesh Kesavan  Initial Version
************************************************************************/
void ApplicationSwitchClientHandler::SetCurrentPhoneStatus(en_phoneContextStatus argCurrentPhoneStatus)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::SetCurrentPhoneStatus(%d)", argCurrentPhoneStatus));
   if (argCurrentPhoneStatus == EN_PHONE_CONTEXT_BG)
   {
      m_enActivityID = (APPLICATIONSWITCH_TYPES::enActivityIDs)enActivityIDs__eActivityID_NONE;
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::SetCurrentPhoneStatus m_enActivityID: %d", m_enActivityID));
   }
   m_enCurrentPhoneStatus = argCurrentPhoneStatus;
}


/************************************************************************
*FUNCTION:		getCurrentActivityID
*DESCRIPTION:	returns the status of current activityID of the whole HMI system
*PARAMETER:		void
*RETURNVALUE:	enActivityIDs
*SWFL:			General
*HISTORY:
*0.1     03.09.20    Palani K    Initial Version
************************************************************************/
APPLICATIONSWITCH_TYPES::enActivityIDs ApplicationSwitchClientHandler::getCurrentActivityID()
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getCurrentActivityID() return %d", m_entargetActivityID));
   return m_entargetActivityID;
}


/************************************************************************
*FUNCTION:		getTargetAppID
*DESCRIPTION:	returns the current targetAppID of the whole HMI system
*PARAMETER:		void
*RETURNVALUE:	uint8
*SWFL:			General
*HISTORY:
*0.1     26.11.20    Palani K    Initial Version
************************************************************************/
uint8 ApplicationSwitchClientHandler::getTargetAppID()
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getTargetAppID() return %d", m_u8TargetAppID));
   return m_u8TargetAppID;
}


/************************************************************************
*FUNCTION: restorePhoneApplicationScene
*DESCRIPTION: Used to update the display context whenever, we need back go to previous scene
*PARAMETER: EnablePermanentView - should force last permanent view or should display last intermediate view
*RETURNVALUE:void
*SWFL:General
*HISTORY:
   * 0.1     02.02.20      Ramesh Kesavan  Initial Version
************************************************************************/
void ApplicationSwitchClientHandler::restorePhoneApplicationScene(bool EnablePermanentView)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::restorePhoneApplicationScene(%d)", EnablePermanentView));

   if (true == EnablePermanentView)
   {
      m_entargetActivityID = m_enCurrPermanentViewActivityID;
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::restorePhoneApplicationScene - set to Display Permanent scene : %d ", m_entargetActivityID));
   }
   else if (m_enCurrIntermediateViewActivityID != enActivityIDs__eActivityID_NONE)
   {
      m_entargetActivityID = m_enCurrIntermediateViewActivityID;
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::restorePhoneApplicationScene - set to Display intermediate scene  : %d ", m_entargetActivityID));
   }
   else
   {
      m_entargetActivityID = m_enCurrPermanentViewActivityID;
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::restorePhoneApplicationScene - set to Display Permanent scene : %d ", m_entargetActivityID));
   }

   requestSendContext((uint8)2 , (uint8)APPID_APPHMI_PHONE, m_entargetActivityID);
}


/************************************************************************
*FUNCTION:		onSig_ControlPanelButtonPressSignal
*DESCRIPTION:	Launch BTVR through notch panel
************************************************************************/
void ApplicationSwitchClientHandler::onSig_ControlPanelButtonPressError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Sig_ControlPanelButtonPressError >& error)
{
   ETG_TRACE_ERR(("ApplicationSwitchClientComponent:: onSig_ControlPanelButtonPressError"));
}


void ApplicationSwitchClientHandler::onSig_ControlPanelButtonPressSignal(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Sig_ControlPanelButtonPressSignal >& signal)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler:: onSig_ControlPanelButtonPressSignal"));
   if (m_poApplicationSwitchClient == proxy)
   {
      ControlPanelButtonPressInfo* oMsg =  COURIER_MESSAGE_NEW(ControlPanelButtonPressInfo)(signal->getButtonId(), signal->getButtonState());
      if (NULL != oMsg)
      {
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler:: onSig_ControlPanelButtonPressSignal ControlPanelButtonPressInfo posted "));
         oMsg->Post();
      }
      else
      {
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler:: Null Pointer"));
      }
   }
}


}
}
