/**************************************************************************************
* @file         : NaviDataPoolConfig.cpp
* @author       : ECG5-Atchuta Sashank Kappagantu
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "NaviDataPoolConfig.h"
#include <vector>
#include <algorithm>
#include <sstream>
#include <limits>


// Instantiate the static class object
NaviDataPoolConfig* NaviDataPoolConfig::_dpNavi = NULL;

#ifdef DP_DATAPOOL_ID
#include "vehicle_main_fi_typesConst.h"
using namespace ::vehicle_main_fi_types;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_HALL
#define ETG_I_TRACE_CHANNEL TR_TTFIS_APPHMI_NAVIGATION
#define ETG_I_TTFIS_CMD_PREFIX "APPHMI_NAVIGATION_"
#define ETG_I_FILE_PREFIX App::Core::NaviDataPoolConfig::
#include "trcGenProj/Header/NaviDataPoolConfig.cpp.trc.h"
#endif

const unsigned int maxISOCodeLen = 6;
const uint8 INVALID_CONFIG_VALUE = std::numeric_limits<uint8>::max();
const tU16 SENSORCONFIGGNSS_KDS_KEY = 0x0530;
const tU8 SATELLITESOURCE_MASK = 0xF8;   /* 0xF8 = 1111 1000 */


NaviDataPoolConfig::~NaviDataPoolConfig()
{
}


NaviDataPoolConfig::NaviDataPoolConfig()
{
   DP_vCreateDatapool();
}


NaviDataPoolConfig* NaviDataPoolConfig::getInstance()
{
   if (_dpNavi == NULL)
   {
      createInstance();
   }

   return _dpNavi;
}


void NaviDataPoolConfig::createInstance()
{
   _dpNavi = new NaviDataPoolConfig();
}


void NaviDataPoolConfig::deleteInstance()
{
   if (_dpNavi != NULL)
   {
      delete _dpNavi;
      _dpNavi = NULL;
   }
}


void NaviDataPoolConfig::setDpVDLanguageIndexSystem(uint8 languageIndex)
{
   ETG_TRACE_USR4(("NaviDataPoolConfig::setDpVDLanguageIndexSystem store LanguageIndex = %d", languageIndex));
   _dpVDLanguageIndexSystem.vSetData((uint8)languageIndex);
}


void NaviDataPoolConfig::setDpVDLanguageIndexSds(uint8 languageIndex)
{
   _dpVDLanguageIndexSds.vSetData((uint8)languageIndex);
}


void NaviDataPoolConfig::setDpVDLanguageIndexCluster(uint8 languageIndex)
{
   _dpVDLanguageIndexCluster.vSetData((uint8)languageIndex);
}


void NaviDataPoolConfig::setDpVDLanguageIndexTrafficText(uint8 languageIndex)
{
   _dpVDLanguageIndexTrafficText.vSetData((uint8)languageIndex);
}


void NaviDataPoolConfig::setDpLanguageIndexSystem(uint8 languageIndex)
{
   _dpLanguageIndexSystem.vSetData((uint8)languageIndex);
}


void NaviDataPoolConfig::setDpLanguageISOCodeSystem(std::string languageISOCodeSystem)
{
   tChar strISOCode [maxISOCodeLen] = "";

   snprintf(strISOCode, sizeof(strISOCode), "%s", languageISOCodeSystem.c_str());
   _dpLanguageISOCodeSystem.s32SetData(strISOCode);
   ETG_TRACE_USR4(("NaviDataPoolConfig::setDpLanguageISOCodeSystem store LanguageISOCodeSystem = %s", strISOCode));
}


void NaviDataPoolConfig::setDpLanguageIndexSds(uint8 languageIndexSds)
{
   _dpLanguageIndexSds.vSetData(languageIndexSds);
}


void NaviDataPoolConfig::setDpLanguageISOCodeSds(std::string languageISOCodeSds)
{
   tChar strISOCode [maxISOCodeLen] = "";

   snprintf(strISOCode, sizeof(strISOCode), "%s", languageISOCodeSds.c_str());
   _dpLanguageISOCodeSds.s32SetData(strISOCode);
   ETG_TRACE_USR4(("NaviDataPoolConfig::setDpLanguageISOCodeSds store LanguageISOCodeSds = %s", strISOCode));
}


void NaviDataPoolConfig::setDpLanguageIndexCluster(uint8 languageIndexCluster)
{
   _dpLanguageIndexCluster.vSetData(languageIndexCluster);
}


void NaviDataPoolConfig::setDpLanguageISOCodeCluster(std::string languageISOCodeCluster)
{
   tChar strISOCode [maxISOCodeLen] = "";

   snprintf(strISOCode, sizeof(strISOCode), "%s", languageISOCodeCluster.c_str());
   _dpLanguageISOCodeCluster.s32SetData(strISOCode);
   ETG_TRACE_USR4(("NaviDataPoolConfig::setDpLanguageISOCodeCluster store LanguageISOCodeCluster = %s", strISOCode));
}


void NaviDataPoolConfig::setDpLanguageIndexTrafficText(uint8 languageIndexTrafficText)
{
   _dpLanguageIndexTrafficText.vSetData(languageIndexTrafficText);
}


void NaviDataPoolConfig::setDpLanguageISOCodeTrafficText(std::string languageISOCodeTrafficText)
{
   tChar strISOCode [maxISOCodeLen] = "";

   snprintf(strISOCode, sizeof(strISOCode), "%s", languageISOCodeTrafficText.c_str());
   _dpLanguageISOCodeTrafficText.s32SetData(strISOCode);
   ETG_TRACE_USR4(("NaviDataPoolConfig::setDpLanguageISOCodeTrafficText store LanguageISOCodeTrafficText = %s", strISOCode));
}


void NaviDataPoolConfig::setDpLightMode(enLightMode lightMode)
{
   _dpLightMode.s32SetData((uint8)lightMode);
}


void NaviDataPoolConfig::setDpDateFormat(uint8 dateFormat)
{
   _dpDateFormat.s32SetData(dateFormat);
}


void NaviDataPoolConfig::setDpTimeFormat(enTimeFormat timeFormat)
{
   _dpTimeFormat.s32SetData((uint8)timeFormat);
}


void NaviDataPoolConfig::setDpEstimatedTimeMode(unsigned char estimatedTimeMode)
{
   _dpEstimatedTimeMode.s32SetData(estimatedTimeMode);
}


void NaviDataPoolConfig::setDpEstimatedTimeType(unsigned char estimatedTimeType)
{
   _dpEstimatedTimeType.s32SetData(estimatedTimeType);
}


void NaviDataPoolConfig::setDpLocalTimeOffset(int localTimeOffset)
{
   _dpLocalTimeOffset.s32SetData((uint32)localTimeOffset);
}


void NaviDataPoolConfig::setDpClockAutoModeActivated(bool clockAutoModeActivated)
{
   _dpClockAutoModeActivated.s32SetData((tBool)clockAutoModeActivated);
}


void NaviDataPoolConfig::setDpIsGuidanceActive(bool isGuidanceActive)
{
   ETG_TRACE_COMP(("Set Guidance Active in Datapool"));
   _dpIsGuidanceActive.s32SetData(isGuidanceActive);
}


void NaviDataPoolConfig::setDpDistanceUnitSystem(unsigned char distanceUnitSystem)
{
   _dpDistanceUnitSystem.s32SetData(distanceUnitSystem);
}


void NaviDataPoolConfig::setDpLatitudeDirection(enLatitudeCoordinatesDirection latitudeDirection)
{
   _dpLatitudeCoordinatesDirection.s32SetData(static_cast<tU8>(latitudeDirection));
}


void NaviDataPoolConfig::setDpLongitudeDirection(enLongitudeCoordinatesDirection longitudeDirection)
{
   _dpLongitudeCoordinatesDirection.s32SetData(static_cast<tU8>(longitudeDirection));
}


void NaviDataPoolConfig::setDpCoordinatiateDisplayFormat(enCoordinatesDisplayType coordinateDisplayFormat)
{
   _dpCoordinateDisplayFormat.s32SetData(static_cast<tU8>(coordinateDisplayFormat));
}


void NaviDataPoolConfig::setMyPOIsAvailability(bool isAvailable)
{
   _dpMyPOIsAvailability.s32SetData(isAvailable);
}


enRegionType NaviDataPoolConfig::getDpRegionType()
{
   uint8 u8Region = 255;

   if (DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "DestinationRegion1", &u8Region, 1))
   {
      ETG_TRACE_COMP(("NaviDataPoolConfig::getDpRegionType(%d)", u8Region));
      return (enRegionType)u8Region;
   }

   return (enRegionType)u8Region;
}


uint8 NaviDataPoolConfig::getDpVDLanguageIndexSystem()
{
   uint8 languageIndexSystem = (uint8)_dpVDLanguageIndexSystem.tGetData();
   // In case if data pool language is invalid, reading language from KDS for default language
   if ((languageIndexSystem == (uint8)T_e8_Language_Code__Unknown) ||
         (languageIndexSystem == (uint8)T_e8_Language_Code__Unused) ||
         (languageIndexSystem == (uint8)T_e8_Language_Code__UnSupported))
   {
      dp_tclKdsSystemConfiguration1 language;
      uint8 kdsDataStatus = language.u8GetDefaultLanguage(languageIndexSystem);
      ETG_TRACE_USR4(("NaviDataPoolConfig:: From KDS getDpVDLanguageIndexSystem kdsDataStatus = (%d)", kdsDataStatus));
   }
   ETG_TRACE_USR4(("NaviDataPoolConfig::getDpVDLanguageIndexSystem(%d)", languageIndexSystem));
   return languageIndexSystem;
}


uint8 NaviDataPoolConfig::getDpVDLanguageIndexSds()
{
   uint8 languageIndexSds = (uint8)_dpVDLanguageIndexSds.tGetData();
   ETG_TRACE_USR4(("NaviDataPoolConfig::getDpVDLanguageIndexSds(%d)", languageIndexSds));
   return languageIndexSds;
}


uint8 NaviDataPoolConfig::getDpVDLanguageIndexCluster()
{
   uint8 languageIndexCluster = (uint8)_dpVDLanguageIndexCluster.tGetData();
   ETG_TRACE_USR4(("NaviDataPoolConfig::getDpVDLanguageIndexCluster(%d)", languageIndexCluster));
   return languageIndexCluster;
}


uint8 NaviDataPoolConfig::getDpVDLanguageIndexTrafficText()
{
   return (uint8)_dpVDLanguageIndexTrafficText.tGetData();
}


uint8 NaviDataPoolConfig::getDpLanguageIndexSystem()
{
   return (uint8)_dpLanguageIndexSystem.tGetData();
}


std::string NaviDataPoolConfig::getDpLanguageISOCodeSystem()
{
   tChar strISOCode [maxISOCodeLen] = "";

   _dpLanguageISOCodeSystem.s32GetData(static_cast<tString>(strISOCode), sizeof(strISOCode));
   std::string languageISOCode = strISOCode;
   return languageISOCode;
}


uint8 NaviDataPoolConfig::getDpLanguageIndexSds()
{
   return (uint8)_dpLanguageIndexSds.tGetData();
}


std::string NaviDataPoolConfig::getDpLanguageISOCodeSds()
{
   tChar strISOCode [maxISOCodeLen] = "";

   _dpLanguageISOCodeSds.s32GetData(static_cast<tString>(strISOCode), sizeof(strISOCode));
   std::string languageISOCode = strISOCode;
   return languageISOCode;
}


uint8 NaviDataPoolConfig::getDpLanguageIndexCluster()
{
   return (uint8)_dpLanguageIndexCluster.tGetData();
}


std::string NaviDataPoolConfig::getDpLanguageISOCodeCluster()
{
   tChar strISOCode [maxISOCodeLen] = "";

   _dpLanguageISOCodeCluster.s32GetData(static_cast<tString>(strISOCode), sizeof(strISOCode));
   std::string languageISOCode = strISOCode;
   return languageISOCode;
}


uint8 NaviDataPoolConfig::getDpLanguageIndexTrafficText()
{
   return (uint8)_dpLanguageIndexTrafficText.tGetData();
}


std::string NaviDataPoolConfig::getDpLanguageISOCodeTrafficText()
{
   tChar strISOCode [maxISOCodeLen] = "";

   _dpLanguageISOCodeTrafficText.s32GetData(static_cast<tString>(strISOCode), sizeof(strISOCode));
   std::string languageISOCode = strISOCode;
   return languageISOCode;
}


/*
enRegionType NaviDataPoolConfig::getDpRegionType()
{
   uint8 u8Region = 255;

   if (DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "DestinationRegion1", &u8Region, 1))
   {
      ETG_TRACE_COMP(("NaviDataPoolConfig::getDpRegionType(%d)", u8Region));
      return (enRegionType)u8Region;
   }

   return (enRegionType)u8Region;
}*/


float NaviDataPoolConfig::getDpDefaultPositionLongitude() const
{
   uint8 longitudeBuf[4];
   float longitude = 0.f;

   if (DP_S32_NO_ERR == DP_s32GetConfigItem("NavDefaultPos", "Longitude", (uint8*) &longitudeBuf, 4))
   {
      //Motorola-HighLow-order needs to be reversed
      int32 longitudeInt = static_cast<int32>((longitudeBuf[0] << 24) | (longitudeBuf[1] << 16) | (longitudeBuf[2] << 8) | longitudeBuf[3]);
      //convert from integer representation to float
      longitude = (360.f / 4294967295.f) * longitudeInt;
      ETG_TRACE_COMP(("NaviDataPoolConfig::getDpDefaultPositionLongitude(): longitude %f", longitude));
   }
   return longitude;
}


float NaviDataPoolConfig::getDpDefaultPositionLatitude() const
{
   uint8 latitudeBuf[4];
   float latitude = 0.f;

   if (DP_S32_NO_ERR == DP_s32GetConfigItem("NavDefaultPos", "Latitude", (uint8*) &latitudeBuf, 4))
   {
      //Motorola-HighLow-order needs to be reversed
      int32 latitudeInt = static_cast<int32>((latitudeBuf[0] << 24) | (latitudeBuf[1] << 16) | (latitudeBuf[2] << 8) | latitudeBuf[3]);
      //convert from integer representation to float
      latitude = (360.f / 4294967295.f) * latitudeInt;
      ETG_TRACE_COMP(("NaviDataPoolConfig::getDpDefaultPositionLatitude(): latitude %f", latitude));
   }
   return latitude;
}


float NaviDataPoolConfig::getDpDefaultPositionHeading() const
{
   uint8 defaultHeading = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("NavDefaultPos", "Heading", &defaultHeading, 1))
   {
      ETG_TRACE_COMP(("NaviDataPoolConfig::getDpDefaultPositionHeading(): %d", defaultHeading));
   }
   //convert from counterclockwise to clockwise
   defaultHeading = 0xFF - defaultHeading;
   //convert to float representation
   return (360.f / 256.f) * defaultHeading;
}


/*std::string NaviDataPoolConfig::getDpNavigationUnitId() const
{
   char navigationUnitIdBuffer[8 + 1] = "";
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("NavUnitID", "", (uint8*)navigationUnitIdBuffer, 8))
   {
      navigationUnitIdBuffer[8] = '\0';
      ETG_TRACE_COMP(("NaviDataPoolConfig::getDpNavigationUnitId(): %s", navigationUnitIdBuffer));
   }
   return ::std::string(navigationUnitIdBuffer);
} */


bool NaviDataPoolConfig::getDpIsGuidanceActive()
{
   ETG_TRACE_COMP(("Get Guidance Active in Datapool "));
   ETG_TRACE_COMP((" The Guidance state is %d", _dpIsGuidanceActive.tGetData()));
   return (bool)_dpIsGuidanceActive.tGetData();
}


enLightMode NaviDataPoolConfig::getDpLightMode()
{
   ETG_TRACE_COMP(("NaviDataPoolConfig::getDpLightMode(%d)", _dpLightMode.tGetData()));
   return (enLightMode)_dpLightMode.tGetData();
}


uint8 NaviDataPoolConfig::getDpDateFormat()
{
   ETG_TRACE_COMP(("NaviDataPoolConfig::getDpDateFormat(%d)", _dpDateFormat.tGetData()));
   return _dpDateFormat.tGetData();
}


enTimeFormat NaviDataPoolConfig::getDpTimeFormat()
{
   ETG_TRACE_COMP(("NaviDataPoolConfig::getDpTimeFormat(%d)", _dpTimeFormat.tGetData()));
   return (enTimeFormat)_dpTimeFormat.tGetData();
}


unsigned char NaviDataPoolConfig::getDpEstimatedTimeMode()
{
   return _dpEstimatedTimeMode.tGetData();
}


unsigned char NaviDataPoolConfig::getDpEstimatedTimeType()
{
   return _dpEstimatedTimeType.tGetData();
}


int NaviDataPoolConfig::getDpLocalTimeOffset()
{
   ETG_TRACE_COMP(("NaviDataPoolConfig::getDpLocalTimeOffset(%d)", _dpLocalTimeOffset.tGetData()));
   return (int)_dpLocalTimeOffset.tGetData();
}


bool NaviDataPoolConfig::getDpClockAutoModeActivated()
{
   ETG_TRACE_COMP(("NaviDataPoolConfig::getDpClockAutoModeActivated(%d)", _dpClockAutoModeActivated.tGetData()));
   return (bool)_dpClockAutoModeActivated.tGetData();
}


/*
unsigned int NaviDataPoolConfig::getTcuConfig()
{
   tU8 tCUType = 0;
   dp_tclKdsVehicleInformation kdsVehicleInformationInst;
   kdsVehicleInformationInst.u8GetTCU(tCUType);
   return (unsigned int)tCUType;
}*/


enVehicleType NaviDataPoolConfig::getKDSVehicleType()
{
   uint8 u8VehicleType[2] = {0};
   uint16 u16VehicleType = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "VehicleType", &u8VehicleType[0], 2))
   {
      ETG_TRACE_USR4(("NaviDataPoolConfig::getKDSVehicleType, First byte : %x, Second byte : %x", u8VehicleType[0], u8VehicleType[1]));

      //Motorola-HighLow-order needs to be reversed
      u16VehicleType = static_cast<uint16>((u8VehicleType[0] << 8) | u8VehicleType[1]);
      ETG_TRACE_COMP(("NaviDataPoolConfig::getKDSVehicleType, Vehicle type : %d", u16VehicleType));
   }
   return (enVehicleType)u16VehicleType;
}


uint16 NaviDataPoolConfig::getKDSWeightOfVehicle()
{
   uint8 u8VehicleWeight[2] = {0};
   uint16 u16VehicleWeight = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "WeightOfVehicle", &u8VehicleWeight[0], 2))
   {
      ETG_TRACE_USR4(("NaviDataPoolConfig::getKDSWeightOfVehicle, First byte : %x, Second byte : %x", u8VehicleWeight[0], u8VehicleWeight[1]));

      //Motorola-HighLow-order needs to be reversed
      u16VehicleWeight = static_cast<uint16>((u8VehicleWeight[0] << 8) | u8VehicleWeight[1]);
      ETG_TRACE_COMP(("NaviDataPoolConfig::getKDSWeightOfVehicle, Weight : %d", u16VehicleWeight));
   }
   return u16VehicleWeight;
}


uint16 NaviDataPoolConfig::getKDSWidthOfVehicle()
{
   uint8 u8VehicleWidth[2] = {0};
   uint16 u16VehicleWidth = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "WidthOfVehicle", &u8VehicleWidth[0], 2))
   {
      ETG_TRACE_USR4(("NaviDataPoolConfig::getKDSWidthOfVehicle, First byte : %x, Second byte : %x", u8VehicleWidth[0], u8VehicleWidth[1]));

      //Motorola-HighLow-order needs to be reversed
      u16VehicleWidth = static_cast<uint16>((u8VehicleWidth[0] << 8) | u8VehicleWidth[1]);
      ETG_TRACE_COMP(("NaviDataPoolConfig::getKDSWidthOfVehicle, Width: %d", u16VehicleWidth));
   }

   return u16VehicleWidth;
}


uint16 NaviDataPoolConfig::getKDSHeightOfVehicle()
{
   uint8 u8VehicleHeight[2] = {0};
   uint16 u16VehicleHeight = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "HeightOfVehicle", &u8VehicleHeight[0], 2))
   {
      ETG_TRACE_USR4(("NaviDataPoolConfig::getKDSHeightOfVehicle, First byte : %x, Second byte : %x", u8VehicleHeight[0], u8VehicleHeight[1]));

      //Motorola-HighLow-order needs to be reversed
      u16VehicleHeight = static_cast<uint16>((u8VehicleHeight[0] << 8) | u8VehicleHeight[1]);
      ETG_TRACE_COMP(("NaviDataPoolConfig::getKDSHeightOfVehicle, Height : %d", u16VehicleHeight));
   }
   return u16VehicleHeight;
}


uint16 NaviDataPoolConfig::getKDSLengthOfVehicle()
{
   uint8 u8VehicleLength[2] = {0};
   uint16 u16VehicleLength = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "LengthOfVehicle", &u8VehicleLength[0], 2))
   {
      ETG_TRACE_USR4(("NaviDataPoolConfig::getKDSLengthOfVehicle, First byte : %x, Second byte : %x", u8VehicleLength[0], u8VehicleLength[1]));

      //Motorola-HighLow-order needs to be reversed
      u16VehicleLength = static_cast<uint16>((u8VehicleLength[0] << 8) | u8VehicleLength[1]);
      ETG_TRACE_COMP(("NaviDataPoolConfig::getKDSLengthOfVehicle, Length : %d", u16VehicleLength));
   }
   return u16VehicleLength;
}


uint8 NaviDataPoolConfig::getKDSTrailerType()
{
   uint8 u8TrailerType = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "TrailerType", &u8TrailerType, 1))
   {
      ETG_TRACE_USR4(("NaviDataPoolConfig::getKDSTrailerType, Byte value : %x", u8TrailerType));

      // Trailer type is stored in bit positions 0 - 3
      u8TrailerType = u8TrailerType & 0x0F;
      ETG_TRACE_COMP(("NaviDataPoolConfig::getKDSTrailerType, Trailer type : %d", u8TrailerType));
   }
   return u8TrailerType;
}


uint8 NaviDataPoolConfig::getKDSNumberOfAxes()
{
   uint8 u8NumberOfAxes = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "NumberOfAxes", &u8NumberOfAxes, 1))
   {
      ETG_TRACE_COMP(("NaviDataPoolConfig::getKDSNumberOfAxes, Number of axes : %d", u8NumberOfAxes));
   }
   return u8NumberOfAxes;
}


uint16 NaviDataPoolConfig::getKDSMaximumLoadPerAxle()
{
   uint8 u8MaximumLoadPerAxle[2] = {0};
   uint16 u16MaximumLoadPerAxle = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "MaximumLoadPerAxle", &u8MaximumLoadPerAxle[0], 2))
   {
      ETG_TRACE_USR4(("NaviDataPoolConfig::getKDSMaximumLoadPerAxle, First byte : %x, Second byte : %x", u8MaximumLoadPerAxle[0], u8MaximumLoadPerAxle[1]));

      //Motorola-HighLow-order needs to be reversed
      u16MaximumLoadPerAxle = static_cast<uint16>((u8MaximumLoadPerAxle[0] << 8) | u8MaximumLoadPerAxle[1]);
      ETG_TRACE_COMP(("NaviDataPoolConfig::getKDSMaximumLoadPerAxle, Maximum Load per Axle : %d", u16MaximumLoadPerAxle));
   }
   return u16MaximumLoadPerAxle;
}


void NaviDataPoolConfig::readKDSEntry(tU16 u16KdsKey, tU16 u16Len, tU8* u8Buf) const
{
   OSAL_tIODescriptor tIoKdsHandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_KDS, OSAL_EN_READWRITE);
   ETG_TRACE_USR4(("NaviDataPoolConfig::readKDSEntry(%d)", tIoKdsHandle));
   if (OSAL_ERROR != tIoKdsHandle)
   {
      tsKDSEntry sKDSEntryData;
      sKDSEntryData.u16Entry = u16KdsKey;
      sKDSEntryData.u16EntryLength = u16Len;
      sKDSEntryData.u16EntryFlags = M_KDS_ENTRY_FLAG_NONE;
      sKDSEntryData.au8EntryData[0] = 0;

      tS32 s32NbElements = OSAL_s32IORead(tIoKdsHandle, (tPS8)&sKDSEntryData, (sizeof(sKDSEntryData)));
      if (OSAL_ERROR != s32NbElements)
      {
         (tVoid)memcpy((tVoid*)u8Buf, sKDSEntryData.au8EntryData, u16Len);
         ETG_TRACE_ERRMEM(("NaviDataPoolConfig::readKDSEntry s32NbElements(%d)", s32NbElements));
      }
      else
      {
         ETG_TRACE_ERRMEM(("NaviDataPoolConfig::readKDSEntry OSAL error while reading(%s)", OSAL_coszErrorText(OSAL_u32ErrorCode())));
      }
      (tVoid)OSAL_s32IOClose(tIoKdsHandle);
   }
}


void NaviDataPoolConfig::setSatelliteSource(uint8_t satelliteSource)
{
   ETG_TRACE_USR4(("NaviDataPoolConfig::setSatelliteSource(%d)", satelliteSource));
   tU8 u8WriteData[DP_U8_KDSLEN_SENSORCONFIGGNSS_COMPLETE] = {0};
   tU8* u8gnssSensorInfo =  u8WriteData;
   readKDSEntry(SENSORCONFIGGNSS_KDS_KEY, DP_U8_KDSLEN_SENSORCONFIGGNSS_COMPLETE, u8gnssSensorInfo);
   u8gnssSensorInfo[11] &= SATELLITESOURCE_MASK; // Mask the last 3 bits of the 11 th Byte to set the new satellite configuration
   u8WriteData[11] = u8gnssSensorInfo[11] | satelliteSource;
   OSAL_tIODescriptor tIoKdsHandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_KDS, OSAL_EN_READWRITE);

   if (OSAL_ERROR != tIoKdsHandle)
   {
      if (OSAL_ERROR == OSAL_s32IOControl(tIoKdsHandle, OSAL_C_S32_IOCTRL_KDS_WRITE_ENABLE, 1))
      {
         ETG_TRACE_ERRMEM(("NaviDataPoolConfig::setSatelliteSource OSAL error during write enable(%s)", OSAL_coszErrorText(OSAL_u32ErrorCode())));
      }
      else
      {
         tsKDSEntry sKDSEntryData;
         sKDSEntryData.u16Entry = SENSORCONFIGGNSS_KDS_KEY;
         sKDSEntryData.u16EntryLength = sizeof(u8WriteData);
         sKDSEntryData.u16EntryFlags = M_KDS_ENTRY_FLAG_NONE;
         memcpy(sKDSEntryData.au8EntryData, u8WriteData, sizeof(u8WriteData));
         if (OSAL_ERROR == OSAL_s32IOWrite(tIoKdsHandle, (tPS8)&sKDSEntryData, (sizeof(sKDSEntryData))))
         {
            ETG_TRACE_ERRMEM(("NaviDataPoolConfig::setSatelliteSource OSAL error while writing(%s)", OSAL_coszErrorText(OSAL_u32ErrorCode())));
         }
         if (OSAL_ERROR == OSAL_s32IOControl(tIoKdsHandle, OSAL_C_S32_IOCTRL_KDS_WRITE_BACK, 0))
         {
            ETG_TRACE_ERRMEM(("NaviDataPoolConfig::setSatelliteSource OSAL error after writing(%s)", OSAL_coszErrorText(OSAL_u32ErrorCode())));
         }
         if (OSAL_ERROR == OSAL_s32IOControl(tIoKdsHandle, OSAL_C_S32_IOCTRL_KDS_WRITE_ENABLE, 0))
         {
            ETG_TRACE_ERRMEM(("NaviDataPoolConfig::setSatelliteSource OSAL error during write disable(%s)", OSAL_coszErrorText(OSAL_u32ErrorCode())));
         }
      }
      (tVoid)OSAL_s32IOClose(tIoKdsHandle);
   }
}


uint8 NaviDataPoolConfig::getKDSSatelliteSource()
{
   uint8 u8Gps, u8Glonass, u8Galileo, u8SatelliteSource;
   u8Gps = u8Glonass = u8Galileo = u8SatelliteSource = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("SensorConfigGNSS", "GPS", &u8Gps, 1))
   {
      ETG_TRACE_COMP(("NaviDataPoolConfig::getKDSSatelliteSource, GPS : %d", u8Gps));
   }
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("SensorConfigGNSS", "GLONASS", &u8Glonass, 1))
   {
      ETG_TRACE_COMP(("NaviDataPoolConfig::getKDSSatelliteSource, GLONASS : %d", u8Glonass));
   }
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("SensorConfigGNSS", "GALILEO", &u8Galileo, 1))
   {
      ETG_TRACE_COMP(("NaviDataPoolConfig::getKDSSatelliteSource, GALILEO : %d", u8Galileo));
   }
   u8SatelliteSource = u8Gps | (u8Glonass <<= 1) | (u8Galileo <<= 2);
   return u8SatelliteSource;
}


enVariantSelection NaviDataPoolConfig::getVariantSelectionType()
{
   uint8 u8VariantSelectionType = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("AIVIVariantCoding", "VariantSelection", (uint8*) &u8VariantSelectionType, 1))
   {
      ETG_TRACE_COMP(("NaviDataPoolConfig::getVariantSelectionType(%d)", u8VariantSelectionType));
   }
   else
   {
      ETG_TRACE_COMP(("Error!!! NaviDataPoolConfig::getVariantSelectionType(%d)", u8VariantSelectionType));
   }
   return (enVariantSelection)u8VariantSelectionType;
}


unsigned char NaviDataPoolConfig::getDpDistanceUnitSystem()
{
   return (_dpDistanceUnitSystem.tGetData());
}


enLatitudeCoordinatesDirection NaviDataPoolConfig::getDpLatitudeDirection()
{
   return ((enLatitudeCoordinatesDirection)_dpLatitudeCoordinatesDirection.tGetData());
}


enLongitudeCoordinatesDirection NaviDataPoolConfig::getDpLongitudeDirection()
{
   return ((enLongitudeCoordinatesDirection)_dpLongitudeCoordinatesDirection.tGetData());
}


enCoordinatesDisplayType NaviDataPoolConfig::getDpCoordinatiateDisplayFormat()
{
   return ((enCoordinatesDisplayType)_dpCoordinateDisplayFormat.tGetData());
}


bool NaviDataPoolConfig::getMyPOIsAvailability()
{
   ETG_TRACE_COMP(("NaviDataPoolConfig::getMyPOIsAvailability(%d)", _dpMyPOIsAvailability.tGetData()));
   return (bool)_dpMyPOIsAvailability.tGetData();
}


/*
bool NaviDataPoolConfig::getUInt8ConfigurationValueAsString(
   const std::string& category,
   const std::string& configurationItem,
   std::string& value)
{
   uint8 current = INVALID_CONFIG_VALUE;

   const int resultCode = DP_s32GetConfigItem(category.c_str(), configurationItem.c_str(), &current, 1);
   const bool success = (DP_S32_NO_ERR == resultCode);

   if (success)
   {
      // In absence of C++11 and to_string()
      std::ostringstream stream;
      stream << int(current);
      value = stream.str();
   }
   else
   {
      ETG_TRACE_ERR(("NaviDataPoolConfig::getConfigurationValueAsString() - category [%25s], configuration item [%25s], error code %d",
                     category.c_str(), configurationItem.c_str(), resultCode));
   }

   return success;
}


bool NaviDataPoolConfig::getGyroMountingAngleConfiguration(
   std::string& angleRX,
   std::string& angleRY,
   std::string& angleRZ,
   std::string& angleSX,
   std::string& angleSY,
   std::string& angleSZ,
   std::string& angleTX,
   std::string& angleTY,
   std::string& angleTZ)
{
   bool ok = getUInt8ConfigurationValueAsString(NaviDataPoolConsts::CONFIG_GROUP_SENSOR_CONFIG_GYRO, NaviDataPoolConsts::CONFIG_KEY_GYRO_ANGLE_RX, angleRX);
   ok &= getUInt8ConfigurationValueAsString(NaviDataPoolConsts::CONFIG_GROUP_SENSOR_CONFIG_GYRO, NaviDataPoolConsts::CONFIG_KEY_GYRO_ANGLE_RY, angleRY);
   ok &= getUInt8ConfigurationValueAsString(NaviDataPoolConsts::CONFIG_GROUP_SENSOR_CONFIG_GYRO, NaviDataPoolConsts::CONFIG_KEY_GYRO_ANGLE_RZ, angleRZ);
   ok &= getUInt8ConfigurationValueAsString(NaviDataPoolConsts::CONFIG_GROUP_SENSOR_CONFIG_GYRO, NaviDataPoolConsts::CONFIG_KEY_GYRO_ANGLE_SX, angleSX);
   ok &= getUInt8ConfigurationValueAsString(NaviDataPoolConsts::CONFIG_GROUP_SENSOR_CONFIG_GYRO, NaviDataPoolConsts::CONFIG_KEY_GYRO_ANGLE_SY, angleSY);
   ok &= getUInt8ConfigurationValueAsString(NaviDataPoolConsts::CONFIG_GROUP_SENSOR_CONFIG_GYRO, NaviDataPoolConsts::CONFIG_KEY_GYRO_ANGLE_SZ, angleSZ);
   ok &= getUInt8ConfigurationValueAsString(NaviDataPoolConsts::CONFIG_GROUP_SENSOR_CONFIG_GYRO, NaviDataPoolConsts::CONFIG_KEY_GYRO_ANGLE_TX, angleTX);
   ok &= getUInt8ConfigurationValueAsString(NaviDataPoolConsts::CONFIG_GROUP_SENSOR_CONFIG_GYRO, NaviDataPoolConsts::CONFIG_KEY_GYRO_ANGLE_TY, angleTY);
   ok &= getUInt8ConfigurationValueAsString(NaviDataPoolConsts::CONFIG_GROUP_SENSOR_CONFIG_GYRO, NaviDataPoolConsts::CONFIG_KEY_GYRO_ANGLE_TZ, angleTZ);
   return ok;
} */


#endif
