/**************************************************************************************
* @file         : EngineeringMenuHandler.cpp
* @author       : Atchuta Sashank Kappagantu
* @addtogroup   : AppHmi_Testmode
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hmi_trace_if.h"
#include <vector>
#include "EngineeringMenuHandler.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/EngineeringMenuHandler.cpp.trc.h"
#endif

using namespace navmiddleware;
static const char* const ENGG_MENU_LIST_MENUITEM = "EnggMenuListItem";

EngineeringMenuHandler::EngineeringMenuHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _listId(0)
   , _startIndex(0)
   , _windowElementSize(0)
   , isDemoModeItemPressed(false)
   , isMenuTypeItemPressed(false)
{
   ETG_TRACE_USR4(("EngineeringMenuHandler::Constructor Called"));
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_TESTMODE_NAVIGATION, this);
}


EngineeringMenuHandler::~EngineeringMenuHandler()
{
}


void EngineeringMenuHandler::initialize()
{
   _navMiddleware.registerEngineeringMenuPropertyUpdateCallback(*this);
}


void EngineeringMenuHandler::deinitialize()
{
   _navMiddleware.unregisterEngineeringMenuPropertyUpdateCallback(*this);
}


bool EngineeringMenuHandler::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   const navmiddleware::EngineeringInfos& engineeringInfos = _navMiddleware.getEngineeringInfos();

   if (oMsg.GetListId() == LIST_ID_TESTMODE_NAVIGATION)
   {
      // we are handling this request
      _listId = oMsg.GetListId();
      _startIndex = oMsg.GetStartIndex();
      _windowElementSize = oMsg.GetWindowElementSize();

      ETG_TRACE_USR4(("EngineeringMenuHandler::onCourierMessage(ListDateProviderReqMsg(listID %d, startIndex %d, elementSize %d))", _listId, _startIndex, _windowElementSize));
      // FIXME: ENGINEERING_MENU_CHANGED does not come after resetEngineeringMenu()
      updateEngineeringMenuBinding();

      tSharedPtrDataProvider dataProvider = getEngineeringMenuListDataProvider(_startIndex, _windowElementSize);
      dataProvider->setCacheOnOff(false);
      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
   }

   return false;
}


bool EngineeringMenuHandler::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   ListDataProviderBuilder modifiedListItems(LIST_ID_TESTMODE_NAVIGATION);

   unsigned int listID = oMsg.GetListId();
   unsigned int rowIdx = (unsigned int)oMsg.GetHdl();
   unsigned int hdlCol = (unsigned int)oMsg.GetSubHdl();
   const enReaction  reaction = oMsg.GetReaction();

   if (LIST_ID_TESTMODE_NAVIGATION == listID)
   {
      if (oMsg.GetReaction() == enRelease)
      {
         if (rowIdx == MENUITEM0) // Demo mode item
         {
            isDemoModeItemPressed = true;
         }
         else if (rowIdx == MENUITEM2 || rowIdx == MENUITEM6 || rowIdx == MENUITEM8 || rowIdx == MENUITEM9 || rowIdx == MENUITEM10) // Menu type items, which will lode the same screen with changed list conent
         {
            isMenuTypeItemPressed = true;
         }
         else
         {
            // do nothing
         }

         const navmiddleware::EngineeringInfos& engineeringInfos = _navMiddleware.getEngineeringInfos();
         const ::std::vector<std::string>& engineeringInfoVector = engineeringInfos.getMenuItems();

         if (engineeringInfoVector.size() > rowIdx)
         {
            _navMiddleware.selectEngineeringMenuEntry(rowIdx);
         }
         updateMenuEntries();
      }
   }

   return false;
}


bool EngineeringMenuHandler::onCourierMessage(const ResetEngineeringMenuReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);

   ETG_TRACE_USR4(("EngineeringMenuHandler::onCourierMessage(ResetEngineeringMenuReqMsg())"));

   _navMiddleware.resetEngineeringMenu();

   return true;
}


bool EngineeringMenuHandler::onCourierMessage(const DeselectEngineeringMenuReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);

   ETG_TRACE_USR4(("EngineeringMenuHandler::onCourierMessage(DeselectEngineeringMenuReqMsg())"));

   _navMiddleware.deselectEngineeringMenuEntry();

   return true;
}


bool EngineeringMenuHandler::onCourierMessage(const EnterEngineeringMenuReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);

   ETG_TRACE_USR4(("EngineeringMenuHandler::onCourierMessage(EnterEngineeringMenuReqMsg())"));

   _navMiddleware.enterEngineeringMenu();

   return true;
}


bool EngineeringMenuHandler::onCourierMessage(const ExitEngineeringMenuReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);

   ETG_TRACE_USR4(("EngineeringMenuHandler::onCourierMessage(ExitEngineeringMenuReqMsg())"));

   _navMiddleware.exitEngineeringMenu();

   return true;
}


void EngineeringMenuHandler::updateMenuEntries()
{
   const navmiddleware::EngineeringInfos& engineeringInfos = _navMiddleware.getEngineeringInfos();

   updateEngineeringMenuBinding();

   if (!engineeringInfos.isMenu())
   {
      ETG_TRACE_USR4(("SubMenu text is %s", engineeringInfos.getValue().c_str()));
      (*_engineeringMenuDetailedText).mEngineeringMenuTextDetails = engineeringInfos.getValue().c_str();
      _engineeringMenuDetailedText.MarkItemModified(ItemKey::EngineeringMenuDetails::EngineeringMenuTextDetailsItem);
      POST_MSG((COURIER_MESSAGE_NEW(EngineeringMenuResponseMsg)()));
   }
   else if (isDemoModeItemPressed)
   {
      updateListElemets(); // Send only ListUpdate Msg to demo mode item
      isDemoModeItemPressed = false;
      (*_engineeringMenuDetailedText).mIsDemoModeEnabled = _navMiddleware.isDemoModeActive();
      _engineeringMenuDetailedText.MarkItemModified(ItemKey::EngineeringMenuDetails::IsDemoModeEnabledItem);
   }
   else if (isMenuTypeItemPressed)
   {
      POST_MSG((COURIER_MESSAGE_NEW(EngineeringMenuResponseMsg)())); // Need complete reload of list(recreation of screen) to stop flickering and incomplete load of list data
      isMenuTypeItemPressed = false;
   }
   _engineeringMenuDetailedText.SendUpdate(true);
}


void EngineeringMenuHandler::onPropertyUpdateEngineeringMenuInfoChanged()
{
   const navmiddleware::EngineeringInfos& engineeringInfos = _navMiddleware.getEngineeringInfos();

   ETG_TRACE_USR4(("EngineeringMenuHandler::onPropertyUpdateEngineeringMenuInfoChanged(isMenu %d)", engineeringInfos.isMenu()));

   updateMenuEntries();
   tSharedPtrDataProvider dataProvider = getEngineeringMenuListDataProvider(_startIndex, _windowElementSize);
   dataProvider->setCacheOnOff(false);
   POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider))); //Update the list with complete data change
}


void EngineeringMenuHandler::updateListElemets()
{
   ListDataProviderBuilder listBuilder(LIST_ID_TESTMODE_NAVIGATION);

   const navmiddleware::EngineeringInfos& engineeringInfos = _navMiddleware.getEngineeringInfos();
   const ::std::vector<std::string>& infos = engineeringInfos.getMenuItems();
   ::std::vector<std::string>::const_iterator info = infos.begin();
   listBuilder.AddItem(
      0UL, // - identifies the row
      0UL, // - unused
      ENGG_MENU_LIST_MENUITEM) // 0 - Name(Id) of the button
   .AddData((*info).c_str()) // 1 - Text
   .AddData(true); // highlight the text to mark an active element
   POST_MSG((COURIER_MESSAGE_NEW(ListDataProviderUpdMsg)(listBuilder.CreateDataProviderUpdater())));
}


void EngineeringMenuHandler::updateEngineeringMenuBinding()
{
   const navmiddleware::EngineeringInfos& engineeringInfos = _navMiddleware.getEngineeringInfos();

   (*_engineeringMenuTitle).mTitleDetail = engineeringInfos.getTitle().c_str();
   _engineeringMenuTitle.MarkItemModified(ItemKey::EngineeringMenuTitle::TitleDetailItem);

   if (_engineeringMenuTitle.SendUpdate() == false)
   {
      ETG_TRACE_ERR(("_engineeringMenuTitle sending failed!"));
   }

   (*_engineeringMenuListCheck).mEngineeringMenuTopListStatus = (engineeringInfos.getSelectedMenuLevel() > 0 ? true : false);
   _engineeringMenuListCheck.MarkItemModified(ItemKey::EngineeringMenuListCheck::EngineeringMenuTopListStatusItem);
   (*_engineeringMenuListCheck).mEngineeringMenuListStatus = engineeringInfos.isMenu();
   _engineeringMenuListCheck.MarkItemModified(ItemKey::EngineeringMenuListCheck::EngineeringMenuListStatusItem);

   if (_engineeringMenuListCheck.HasModifiedItems())
   {
      if (!(_engineeringMenuListCheck.SendUpdate()))
      {
         ETG_TRACE_ERR(("_engineeringMenuListCheck sending failed!"));
      }
   }
}


tSharedPtrDataProvider EngineeringMenuHandler::getEngineeringMenuListDataProvider(unsigned int startIndex, unsigned int windowElementSize) const
{
   ListDataProviderBuilder listBuilder(LIST_ID_TESTMODE_NAVIGATION);

   const navmiddleware::EngineeringInfos& engineeringInfos = _navMiddleware.getEngineeringInfos();
   const ::std::vector<std::string>& infos = engineeringInfos.getMenuItems();

   unsigned int idx = startIndex;
   unsigned int listSize = 0;

   for (::std::vector<std::string>::const_iterator info = infos.begin(); (info != infos.end()); ++info, ++idx)
   {
      listBuilder.AddItem(
         idx, // - identifies the row
         0UL, // - unused
         ENGG_MENU_LIST_MENUITEM) // 0 - Name(Id) of the button
      .AddData((*info).c_str())
      .AddData(true); // 1 - Text
      ++listSize;
   }

   return listBuilder.CreateDataProvider(startIndex, listSize);
}
