/**************************************************************************************
* @file         : StreamedMapSettingsListHandler.cpp
* @author       : ECG-Naveen Thangamalayan
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include <vector>
#include <string> // std::string
#include <sstream> // std::to_string
#include "StreamedMapSettingsListHandler.h"
#include "NaviSettingsUtil.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/StreamedMapSettingsListHandler.cpp.trc.h"
#endif

#if defined (HALL_TO_MDW_COM)

static const Candera::UInt32 TIMEOUT_30_SEC_DURATION = 30000;
static const Candera::UInt32 TIMEOUT_40_SEC_DURATION = 40000;
static const Candera::UInt32 TIMEOUT_50_SEC_DURATION = 50000;
static const Candera::UInt32 TIMEOUT_60_SEC_DURATION = 60000;

StreamedMapSettingsListHandler::StreamedMapSettingsListHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _timerDuration(TIMEOUT_30_SEC_DURATION)
   , _infoStore(infoStore)
   , _navMiddleware(navMiddleware)
{
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_STREAM_MAP_DISPLAY_SETTING, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_STREAM_MAP_DROPDOWN_DURATION, this);
}


StreamedMapSettingsListHandler::~StreamedMapSettingsListHandler()
{
}


void StreamedMapSettingsListHandler::initialize()
{
   const Candera::UInt8 MAX_LIST_SIZE = 5;
   const Candera::UInt8 LIST_ID_NONE = 0;

   _streamMapSettingsListItem.reserve(MAX_LIST_SIZE);
   _streamMapSettingsListItem.push_back(StreamMapSettingsListItem(LANGUAGE_STRING(TextId_0x1423, "Map duration"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_DROPDOWN, STREAMED_MAP_SETTING_DUARTION, StreamMapTimerDuration[STREAMED_MAP_30_SEC], LIST_ID_STREAM_MAP_DROPDOWN_DURATION));
   _streamMapSettingsListItem.push_back(StreamMapSettingsListItem(LANGUAGE_STRING(TextId_0x1424, "Map icons"), IS_VISIBLE_FALSE, LIST_BUTTON_TYPE_BUTTON, STREAMED_MAP_SETTING_ICONS));
   _streamMapSettingsListItem.push_back(StreamMapSettingsListItem(LANGUAGE_STRING(TextId_0x1425, "Map POI"), IS_VISIBLE_FALSE, LIST_BUTTON_TYPE_BUTTON, STREAMED_MAP_SETTING_POI));
   _streamMapSettingsListItem.push_back(StreamMapSettingsListItem(LANGUAGE_STRING(TextId_0x1426, "Map type"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_BUTTON, STREAMED_MAP_SETTING_MAP_TYPE));
   _streamMapSettingsListItem.push_back(StreamMapSettingsListItem(LANGUAGE_STRING(TextId_0x1427, "Map content"), IS_VISIBLE_FALSE, LIST_BUTTON_TYPE_BUTTON, STREAMED_MAP_SETTING_MAP_CONTENT));
}


void StreamedMapSettingsListHandler::deinitialize()
{
}


tSharedPtrDataProvider StreamedMapSettingsListHandler::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_USR4(("StreamedMapSettingsListHandler::getListDataProvider(List ID %d)", oMsg.GetListId()));

   int listId = oMsg.GetListId();
   tSharedPtrDataProvider dataProvider;

   switch (listId)
   {
      case LIST_ID_STREAM_MAP_DISPLAY_SETTING:
      {
         dataProvider = getMainListDataProvider(oMsg);
         break;
      }
      case LIST_ID_STREAM_MAP_DROPDOWN_DURATION:
      {
         dataProvider = configureDropdownList(listId);
         break;
      }
      default:
      {
         break;
      }
   }

   return dataProvider;
}


tSharedPtrDataProvider StreamedMapSettingsListHandler::getMainListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_USR4(("StreamedMapSettingsListHandler::getMainListDataProvider(List ID %d)", oMsg.GetListId()));

   DataItemContext listItemName = "ListItemText";
   ListDataProviderBuilder listBuilder(LIST_ID_STREAM_MAP_DISPLAY_SETTING);

   for (std::vector<StreamMapSettingsListItem>::iterator itemNumber = _streamMapSettingsListItem.begin(); itemNumber < _streamMapSettingsListItem.end(); ++itemNumber)
   {
      //The below condition will be removed when streamed map middleware interfaces is availble for other options
      if ((itemNumber->_order == STREAMED_MAP_SETTING_MAP_TYPE) || (itemNumber->_order == STREAMED_MAP_SETTING_DUARTION))
      {
         listItemName = "Combobox_ListItem4";
         if (itemNumber->_listButtonType == LIST_BUTTON_TYPE_DROPDOWN)
         {
            listBuilder.AddExpandableItem(
               itemNumber->_order,
               itemNumber->_dropDownListId,
               listItemName)
            .AddData(itemNumber->_text)
            .AddData(itemNumber->_dropDownListId)
            .AddData(itemNumber->_dropDownText);
         }
         else if (itemNumber->_listButtonType == LIST_BUTTON_TYPE_BUTTON)
         {
            listItemName = "ListItemText";
            listBuilder.AddItem(
               itemNumber->_order,
               0UL,  //    - unused
               listItemName)  //0 - Name(Id) of the button
            .AddData(itemNumber->_text)
            .AddData(itemNumber->_isVisible);
         }
      }
   }
   return listBuilder.CreateDataProvider();
}


tSharedPtrDataProvider StreamedMapSettingsListHandler::configureDropdownList(::Courier::UInt32 listID)
{
   ETG_TRACE_USR4(("StreamedMapSettingsListHandler::configureDropdownList(List ID %d)", listID));

   //initializes the provider with the list id and the default data context
   const char* DATA_CONTEXT_DROPDOWN_LIST_ITEM = "DropdownListButton";
   ListDataProviderBuilder listBuilder(listID, DATA_CONTEXT_DROPDOWN_LIST_ITEM);
   const Candera::UInt8 MAX_MAP_VIEW_DROPDOWN_SIZE = 4;

   if (LIST_ID_STREAM_MAP_DROPDOWN_DURATION == listID)
   {
      for (Candera::UInt8 idx = 0; idx < MAX_MAP_VIEW_DROPDOWN_SIZE; idx++)
      {
         listBuilder.AddItem(idx).AddData(StreamMapTimerDuration[idx]);
      }
   }

   return listBuilder.CreateDataProvider();
}


bool StreamedMapSettingsListHandler::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   ListProviderEventInfo info;
   bool isMsgConsumed = false;
   bool isListUpdated = true;
   Candera::UInt32 listId = oMsg.GetListId();
   const enReaction  reaction = oMsg.GetReaction();

   // ToDo: Commented out since they are unused. To be analyzed if the below code can be removed
   //navmiddleware::settings::MapView currentMapView =  _navMiddleware.getChangeMapViewSettings().getMapView();
   //bool multiViewActive = ((currentMapView == navmiddleware::settings::MAPVIEW_2D_SPLIT) || (currentMapView == navmiddleware::settings::MAPVIEW_3D_SPLIT));

   if (enRelease == reaction)
   {
      StreamMapSettingsListItemValue rowIdx = static_cast<StreamMapSettingsListItemValue>(oMsg.GetHdl());
      if ((rowIdx != STREAMED_MAP_SETTING_DUARTION) && (LIST_ID_STREAM_MAP_DISPLAY_SETTING == listId))
      {
         ETG_TRACE_USR4(("StreamedMapSettingsListHandler::onCourierMessage(ButtonListItemUpdMsg), List Id : %d", listId));
         POST_MSG((COURIER_MESSAGE_NEW(StreamedMapSubOptionsListUpdateMsg)(rowIdx)));
      }
      else if (LIST_ID_STREAM_MAP_DROPDOWN_DURATION == listId)
      {
         StreamMapSettingsTimeListItemValue rowIdx = static_cast<StreamMapSettingsTimeListItemValue>(oMsg.GetHdl());
         StreamMapSettingsListItem& streamMapDurationListElement = _streamMapSettingsListItem.at(STREAMED_MAP_SETTING_DUARTION);
         ETG_TRACE_USR4(("StreamedMapSettingsListHandler::onCourierMessage(ButtonListItemUpdMsg), List Id : %d, Row index : %d", listId, rowIdx));

         switch (rowIdx)
         {
            case STREAMED_MAP_30_SEC:
            {
               streamMapDurationListElement._dropDownText = StreamMapTimerDuration[STREAMED_MAP_30_SEC];
               _timerDuration = TIMEOUT_30_SEC_DURATION;
               break;
            }
            case STREAMED_MAP_40_SEC:
            {
               streamMapDurationListElement._dropDownText = StreamMapTimerDuration[STREAMED_MAP_40_SEC];
               _timerDuration = TIMEOUT_40_SEC_DURATION;
               break;
            }
            case STREAMED_MAP_50_SEC:
            {
               streamMapDurationListElement._dropDownText = StreamMapTimerDuration[STREAMED_MAP_50_SEC];
               _timerDuration = TIMEOUT_50_SEC_DURATION;
               break;
            }
            case STREAMED_MAP_60_SEC:
            {
               streamMapDurationListElement._dropDownText = StreamMapTimerDuration[STREAMED_MAP_60_SEC];
               _timerDuration = TIMEOUT_60_SEC_DURATION;
               break;
            }
            default:
            {
               isListUpdated = false;
            }
         }
         if (true == isListUpdated)
         {
            //ToDo: To be reworked
            /*if (BUTTON_TOGGLE_ENABLED_TRUE == _enableStreamMap)
            {
               POST_MSG((COURIER_MESSAGE_NEW(StartPopupTimerReqMsg)(AppHmi_NavigationModule_NavigationScenes_N_StreamedMap_SubOptions, _timerDuration)));
            }*/
            ListRegistry::s_getInstance().updateList(LIST_ID_STREAM_MAP_DISPLAY_SETTING);
         }
         isMsgConsumed = true;
      }
   }
   return isMsgConsumed;
}


bool StreamedMapSettingsListHandler::onCourierMessage(const ExpiredTimerResMsg& oMsg)
{
   ETG_TRACE_USR4(("StreamedMapSettingsListHandler::ExpiredPopupTimerResMsg()"));

   POST_MSG((COURIER_MESSAGE_NEW(StartPopupTimerReqMsg)(AppHmi_NavigationModule_NavigationScenes_N_StreamedMap_SubOptions, _timerDuration)));

   return true;
}


// mup6cob [2020-CW22]: The below functionality has been moved to HMIModelMap since Streamed map settings option is currently disabled
//bool StreamedMapSettingsListHandler::onCourierMessage(const StartStopMapStreamReqMsg& oMsg)
//{
//   bool startMapStreaming = oMsg.GetStartMapStream();
//   ETG_TRACE_USR4(("StreamedMapSettingsListHandler::StartStopMapStreamReqMsg(Status - %d)", startMapStreaming));
//
//   if (startMapStreaming == false)
//   {
//      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(SetMapCameraModeReqMsg)(STOP_MAP_STREAM)));
//   }
//   else
//   {
//      //mup6cob [2019-CW25]: Since timer expiry message is not received, as a temporary workaround - ExpiredTimerResMsg is directly posted.
//      //mup6cob [2019-CW29]: Since ExpiredTimerResMsg is not received consecutively on time out, posting of ExpiredTimerResMsg is commented out
//      //POST_MSG((COURIER_MESSAGE_NEW(EnableStreamedMapTimerReqMsg)(startMapStreaming, 10)));
//      //POST_MSG((COURIER_MESSAGE_NEW(ExpiredTimerResMsg)()));
//
//      //mup6cob [CW29]: Directly message is posted to start streaming in 2D
//      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(SetMapCameraModeReqMsg)(STREAMED_MAP_2D_MAP)));
//   }
//
//   return true;
//}


#endif // HALL_TO_MDW_COM
