/**************************************************************************************
* @file         : SpeedLimitSettingsListHandler.cpp
* @author       : ECG-Naveen Thangamalayan
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2018 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include  <vector>
#include "SpeedLimitSettingsListHandler.h"
#include "Map/MapScreenDataUtils.h"
#include "NaviSettingsUtil.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SpeedLimitSettingsListHandler.cpp.trc.h"
#endif

#ifdef  HALL_TO_MDW_COM

static const char* const DATA_CONTEXT_SENTIVITY_INC_CONTROL = "Layer_TouchArea_increment";
static const char* const DATA_CONTEXT_SENTIVITY_DEC_CONTROL = "Layer_TouchArea_decrement";
static const int SPEEDLIMIT_SENSITIVITY_MIN = 0;
static const int SPEEDLIMIT_SENSITIVITY_MAX = 30;

#ifdef ENABLE_FEATURE_SPEED_LIMIT
IsButtonToggleEnabled getSpeedLimitIndicationOnMap(navmiddleware::NavMiddleware& navMiddleware)
{
   IsButtonToggleEnabled retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   navmiddleware::settings::SafetyGuidanceSettings safetyGuidanceSettings = navMiddleware.getSafetyGuidanceSettings();
   if (safetyGuidanceSettings.isSpeedLimitActive())
   {
      retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   }
   else
   {
      retValue = BUTTON_TOGGLE_ENABLED_FALSE;
   }
   ETG_TRACE_USR4(("getSpeedLimitIndicationOnMap(%d)", retValue));

   return retValue;
}


IsButtonToggleEnabled getIsSpeedLimitWarningEnabled(navmiddleware::NavMiddleware& navMiddleware)
{
   IsButtonToggleEnabled retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   navmiddleware::settings::SafetyGuidanceSettings safetyGuidanceSettings = navMiddleware.getSafetyGuidanceSettings();
   if (safetyGuidanceSettings.isOverSpeedWarningActive())
   {
      retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   }
   else
   {
      retValue = BUTTON_TOGGLE_ENABLED_FALSE;
   }
   ETG_TRACE_USR4(("getIsSpeedLimitWarningEnabled(%d)", retValue));

   return retValue;
}
#endif

IsButtonToggleEnabled getIsVehicleRestrictionsWarningEnabled(navmiddleware::NavMiddleware& navMiddleware)
{
   IsButtonToggleEnabled retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   navmiddleware::settings::GuidanceSettings guidanceSettings = navMiddleware.getGuidanceSettings();
   if (guidanceSettings.getVehicleRestrictionsInformation())
   {
      retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   }
   else
   {
      retValue = BUTTON_TOGGLE_ENABLED_FALSE;
   }
   ETG_TRACE_USR4(("getIsVehicleRestrictionsWarningEnabled(%d)", retValue));

   return retValue;
}


SpeedLimitSettingsListHandler::SpeedLimitSettingsListHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : NavMiddlewarePropertyUpdateConsumer(navMiddleware)
   , _listId(0)
   , _startIndex(0)
   , _windowElementSize(0)
{
   //Register the ListID
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_SPEED_LIMIT_OPTION, this);
}


SpeedLimitSettingsListHandler::~SpeedLimitSettingsListHandler()
{
}


void SpeedLimitSettingsListHandler::initialize()
{
   const Candera::UInt8 MAX_LISTSIZE = 4;

   _speedLimitOptionsList.reserve(MAX_LISTSIZE);
#ifdef ENABLE_FEATURE_SPEED_LIMIT
   _speedLimitOptionsList.push_back(SpeedLimitListItem(LANGUAGE_STRING(TextId_0x1434, "Speed limit sensitivity in seconds"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_INC_DEC_VALUE, SETTING_SPEED_LIMIT_SENSITIVITY));
   _speedLimitOptionsList.push_back(SpeedLimitListItem(LANGUAGE_STRING(TextId_0x1431, "Speed limit indication on map"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, SETTING_SPEED_LIMIT_INDICATION_ON_MAP, getSpeedLimitIndicationOnMap));
   _speedLimitOptionsList.push_back(SpeedLimitListItem(LANGUAGE_STRING(TextId_0x1432, "Speed limit warning"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, SETTING_SPEED_LIMIT_WARNING, getIsSpeedLimitWarningEnabled));
#endif
   _speedLimitOptionsList.push_back(SpeedLimitListItem(LANGUAGE_STRING(TextId_0x1955, "Vehicle restriction warning"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, SETTING_VEHICLE_RESTRICTIONS_WARNING, getIsVehicleRestrictionsWarningEnabled));
}


void SpeedLimitSettingsListHandler::deinitialize()
{
   _navMiddleware.unregisterMapPropertyUpdateCallback(*this);
}


tSharedPtrDataProvider SpeedLimitSettingsListHandler::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   _listId = oMsg.GetListId();
   _startIndex = oMsg.GetStartIndex();
   _windowElementSize = oMsg.GetWindowElementSize();
   tSharedPtrDataProvider dataProvider;

   switch (_listId)
   {
      case LIST_ID_SPEED_LIMIT_OPTION:
      {
         dataProvider = getListDataProviderMainList();
         break;
      }
      default:
      {
         break;
      }
   }

   return dataProvider;
}


bool SpeedLimitSettingsListHandler::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   ListProviderEventInfo info;
   bool isMsgConsumed = false;
   Candera::UInt32 listId = oMsg.GetListId();
   unsigned int rowIdx = (unsigned int)oMsg.GetHdl();
   const enReaction reaction = oMsg.GetReaction();

   if ((enRelease == reaction) && (LIST_ID_SPEED_LIMIT_OPTION == listId))
   {
      ETG_TRACE_USR4(("SpeedLimitSettingsListHandler::onCourierMessage(ButtonListItemUpdMsg), List Id : %d", oMsg.GetListId()));
      navmiddleware::settings::SafetyGuidanceSettings safetyGuidanceSettings = _navMiddleware.getSafetyGuidanceSettings();
      bool listUpdateNeeded = true;
      switch (rowIdx)
      {
      #ifdef ENABLE_FEATURE_SPEED_LIMIT
         case SETTING_SPEED_LIMIT_INDICATION_ON_MAP:
         {
            safetyGuidanceSettings.isSpeedLimitActive() ? safetyGuidanceSettings.setSpeedLimit(false) : safetyGuidanceSettings.setSpeedLimit(true);
            break;
         }
         case SETTING_SPEED_LIMIT_WARNING:
         {
            safetyGuidanceSettings.isOverSpeedWarningActive() ? safetyGuidanceSettings.setOverSpeedWarning(false) : safetyGuidanceSettings.setOverSpeedWarning(true);
            break;
         }
         case SETTING_SPEED_LIMIT_SENSITIVITY:
         {
            unsigned int hdlCol = (unsigned int)oMsg.GetSubHdl();

            if (1 == hdlCol)
            {
               decreaseSpeedLimitSensitivity(safetyGuidanceSettings);
            }
            else if (2 == hdlCol)
            {
               increaseSpeedLimitSensitivity(safetyGuidanceSettings);
            }
            else
            {
               listUpdateNeeded = false;
            }
            break;
         }
      #endif
         case SETTING_VEHICLE_RESTRICTIONS_WARNING:
         {
            navmiddleware::settings::GuidanceSettings guidanceSettings = _navMiddleware.getGuidanceSettings();
            guidanceSettings.getVehicleRestrictionsInformation() ? guidanceSettings.setVehicleRestrictionsInformation(false) : guidanceSettings.setVehicleRestrictionsInformation(true);
            _navMiddleware.updateGuidanceSettings(guidanceSettings);
            break;
         }
         default:
         {
            listUpdateNeeded = false;
            break;
         }
      }
      if (true == listUpdateNeeded)
      {
         _navMiddleware.updateSafetyGuidanceSettings(safetyGuidanceSettings);
         ListRegistry::s_getInstance().updateList(LIST_ID_SPEED_LIMIT_OPTION);
      }
      isMsgConsumed = true;
   }

   return isMsgConsumed;
}


bool SpeedLimitSettingsListHandler::onCourierMessage(const ApplyMapViewSettingsReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   _navMiddleware.applySettings();

   return true;
}


tSharedPtrDataProvider SpeedLimitSettingsListHandler::getListDataProviderMainList()
{
   DataItemContext listItemName = "Layer_CheckboxOptions";
   ListDataProviderBuilder listBuilder(LIST_ID_SPEED_LIMIT_OPTION);
   Candera::String currentOption;

   for (std::vector<SpeedLimitListItem>::iterator itemNumber = _speedLimitOptionsList.begin(); itemNumber != _speedLimitOptionsList.end(); ++itemNumber)
   {
      if (itemNumber->_listButtonType == LIST_BUTTON_TYPE_TOGGLE)
      {
         listItemName = "Layer_CheckboxOptions";
         listBuilder.AddItem(
            itemNumber->_order,
            0UL,  //    - unused
            listItemName)  //0 - Name(Id) of the button
         .AddData(itemNumber->_text)
         .AddData((itemNumber->_isVisible == IS_VISIBLE_FALSE) ? true : itemNumber->getButtonToggleState(_navMiddleware));
      }
   #ifdef ENABLE_FEATURE_SPEED_LIMIT
      else if (itemNumber->_listButtonType == LIST_BUTTON_TYPE_INC_DEC_VALUE)
      {
         listItemName = "Layer_Spinner";
         SettingSliderData settingSliderData;
         settingSliderData.mCurrentValue = _navMiddleware.getSafetyGuidanceSettings().getSpeedLimitSensitivity();
         settingSliderData.mDecreasable = (SPEEDLIMIT_SENSITIVITY_MIN == settingSliderData.mCurrentValue) ? false : true;
         settingSliderData.mIncreasable = (SPEEDLIMIT_SENSITIVITY_MAX == settingSliderData.mCurrentValue) ? false : true;

         if (_navMiddleware.getSafetyGuidanceSettings().isOverSpeedWarningActive())
         {
            settingSliderData.mButtonIsActive = IS_VISIBLE_TRUE;
         }
         else
         {
            settingSliderData.mButtonIsActive = IS_VISIBLE_FALSE;
            settingSliderData.mDecreasable = false;
            settingSliderData.mIncreasable = false;
         }

         listBuilder.AddItem(
            itemNumber->_order,
            0UL, // - unused
            listItemName) // 0 - Name(Id) of the button
         .AddData(itemNumber->_text)//1
         .AddData(itemNumber->_isVisible)//2
         .AddId( // 3 - Name(Id) of the Decerement button
            itemNumber->_order,
            1UL)
         .AddId( // 4 - Name(Id) of the Increment button
            itemNumber->_order,
            2UL)
         .AddData(settingSliderData.mCurrentValue)// 5 - button visibility
         .AddDataBindingUpdater<SettingSliderDataBindingSource>(settingSliderData);
      }
   #endif
      else
      {
         //do nothing
      }
   }

   return listBuilder.CreateDataProvider();
}

#ifdef ENABLE_FEATURE_SPEED_LIMIT
void SpeedLimitSettingsListHandler::increaseSpeedLimitSensitivity(navmiddleware::settings::SafetyGuidanceSettings& safetyGuidanceSettings)
{
   if (safetyGuidanceSettings.getSpeedLimitSensitivity() < SPEEDLIMIT_SENSITIVITY_MAX)
   {
      safetyGuidanceSettings.setSpeedLimitSensitivity((safetyGuidanceSettings.getSpeedLimitSensitivity() + 1));
   }
}


void SpeedLimitSettingsListHandler::decreaseSpeedLimitSensitivity(navmiddleware::settings::SafetyGuidanceSettings& safetyGuidanceSettings)
{
   if (safetyGuidanceSettings.getSpeedLimitSensitivity() > SPEEDLIMIT_SENSITIVITY_MIN)
   {
      safetyGuidanceSettings.setSpeedLimitSensitivity((safetyGuidanceSettings.getSpeedLimitSensitivity() - 1));
   }
}
#endif


#endif // HALL_TO_MDW_COM
