/**************************************************************************************
* @file         : PremiumTrafficListHandler.cpp
* @author       : ECG-Naveen Thangamalayan
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2022 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include  <vector>
#include "PremiumTrafficListHandler.h"
#include "Map/MapScreenDataUtils.h"
#include "NaviSettingsUtil.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/PremiumTrafficListHandler.cpp.trc.h"
#endif

#ifdef  HALL_TO_MDW_COM

static const unsigned int PAN_CONNECTION_TIMER_DURATION = 300000;   // (5 minutes in milliseconds)
static const unsigned int PAN_POLLING_TIMER_DURATION    = 10000;    // (10 seconds in milliseconds)
static const int MAX_NUM_UPDATE_FREQUENCY = 6;
static const Candera::UInt32 TRAFFIC_FREQUENCY_VALUE[MAX_NUM_UPDATE_FREQUENCY] = { 0, 120, 300, 6000, 9000, 12000 };


PremiumTrafficListHandler::PremiumTrafficListHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : NavMiddlewarePropertyUpdateConsumer(navMiddleware)
   , InfoStorePropertyUpdateConsumer(infoStore)
{
   ETG_TRACE_USR4(("PremiumTrafficListHandler::PremiumTrafficListHandler()"));
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_PREMIUM_TRAFFIC, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_PREMIUM_TRAFFIC_FREQUENCY, this);
}


PremiumTrafficListHandler::~PremiumTrafficListHandler()
{
   ETG_TRACE_USR4(("PremiumTrafficListHandler::~PremiumTrafficListHandler()"));
}


void PremiumTrafficListHandler::initialize()
{
   ETG_TRACE_USR4(("PremiumTrafficListHandler::initialize()"));
   _infoStore.registerDataPropertyUpdateCallback(*this);
   _panConnectionTimer.setTimeout(0, PAN_CONNECTION_TIMER_DURATION, COURIER_MESSAGE_NEW(PANConnectionTimerExpiredMsg)());
   _panPollingTimer.setTimeout(0, PAN_POLLING_TIMER_DURATION, COURIER_MESSAGE_NEW(PANPollingTimerExpiredMsg)());
   setBTTetheringState(BT_TETHERING_STATE__NotApplicable);

   _trafficSettings = _navMiddleware.getTrafficSettings();
   bool premiumTrafficState = _trafficSettings.isUsePremiumTrafficEnabled();
   setPrivacyMode(!premiumTrafficState);
   if (true == premiumTrafficState)
   {
      startPANConnection();
   }

   const Candera::UInt8 MAX_LISTSIZE = 3;
   _premiumTrafficList.reserve(MAX_LISTSIZE);
   _premiumTrafficList.push_back(PremiumTrafficListItem(LANGUAGE_STRING(TextId_0x1498, "Enable premium traffic*"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, SETTING_ENABLE_PREMIUM_TRAFFIC));
   _premiumTrafficList.push_back(PremiumTrafficListItem(LANGUAGE_STRING(TextId_0x1499, "Update for each route"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, SETTING_UPDATE_FOR_EACH_ROUTE));
   _premiumTrafficList.push_back(PremiumTrafficListItem(LANGUAGE_STRING(TextId_0x149A, "Update Frequency"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_DROPDOWN, SETTING_UPDATE_FREQUENCY, PremiumTrafficDropDownListLabel[getUpdateFrequencyIndex()], LIST_ID_PREMIUM_TRAFFIC_FREQUENCY));
}


void PremiumTrafficListHandler::deinitialize()
{
   ETG_TRACE_USR4(("PremiumTrafficListHandler::deinitialize()"));
   setBTTetheringState(BT_TETHERING_STATE__NotApplicable);
   _infoStore.unregisterDataPropertyUpdateCallback(*this);
}


tSharedPtrDataProvider PremiumTrafficListHandler::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   tSharedPtrDataProvider dataProvider;
   switch (oMsg.GetListId())
   {
      case LIST_ID_PREMIUM_TRAFFIC:
      {
         dataProvider = getListDataProviderMainList();
         break;
      }
      case LIST_ID_PREMIUM_TRAFFIC_FREQUENCY:
      {
         dataProvider = configureDropdownList(oMsg.GetListId());
         break;
      }
      default:
      {
         break;
      }
   }
   return dataProvider;
}


bool PremiumTrafficListHandler::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   bool bIsMsgConsumed = false;

   const Courier::ViewId TRAFFIC_DISCLAIMER_SCENENAME = Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#N_OPT_PremiumTraffic_Disclaimer");
   const Courier::Identifier IdOkButton               = Courier::Identifier("Layer_OK/ButtonWidget");

   const Courier::ViewId sceneName      = oMsg.GetView();
   const Courier::Identifier senderInfo = oMsg.GetSender();
   const enReaction reaction            = oMsg.GetEnReaction();

   if ((TRAFFIC_DISCLAIMER_SCENENAME == sceneName) && (IdOkButton == senderInfo) && (enPress == reaction))
   {
      ETG_TRACE_USR4(("PremiumTrafficListHandler::onCourierMessage(ButtonReactionMsg) OK button press"));
      _trafficSettings.setUsePremiumTraffic(true);
      applySettings();
      ListRegistry::s_getInstance().updateList(LIST_ID_PREMIUM_TRAFFIC);
      bIsMsgConsumed = true;
   }
   return bIsMsgConsumed;
}


bool PremiumTrafficListHandler::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   bool isMsgConsumed = false;
   Candera::UInt32 listId = oMsg.GetListId();
   const enReaction reaction = oMsg.GetReaction();

   if ((enRelease == reaction) && ((LIST_ID_PREMIUM_TRAFFIC_FREQUENCY == listId) || (LIST_ID_PREMIUM_TRAFFIC == listId)))
   {
      isMsgConsumed = true;
      bool isListUpdated = true;
      unsigned int rowIdx = (unsigned int)oMsg.GetHdl();

      if (LIST_ID_PREMIUM_TRAFFIC_FREQUENCY == listId)
      {
         changePremiumTrafficUpdateFrequency(rowIdx);
      }
      else   // listId is LIST_ID_PREMIUM_TRAFFIC
      {
         switch (rowIdx)
         {
            case SETTING_ENABLE_PREMIUM_TRAFFIC:
            {
               ETG_TRACE_USR4(("PremiumTrafficListHandler::onCourierMessage(ButtonListItemUpdMsg) for LIST_ID_PREMIUM_TRAFFIC::SETTING_ENABLE_PREMIUM_TRAFFIC"));

               if (true == EXT_bIsPremiumTrafficDisclaimerShown)
               {
                  _trafficSettings.isUsePremiumTrafficEnabled() ? _trafficSettings.setUsePremiumTraffic(false) : _trafficSettings.setUsePremiumTraffic(true);
               }
               else
               {
                  if (_trafficSettings.isUsePremiumTrafficEnabled())
                  {
                     _trafficSettings.setUsePremiumTraffic(false);
                  }
                  else
                  {
                     POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ev_N_PremiumTraffic_Disclaimer_Msg)()));
                  }
               }
               applySettings();
               break;
            }
            case SETTING_UPDATE_FOR_EACH_ROUTE:
            {
               ETG_TRACE_USR4(("PremiumTrafficListHandler::onCourierMessage(ButtonListItemUpdMsg) for LIST_ID_PREMIUM_TRAFFIC::SETTING_UPDATE_FOR_EACH_ROUTE"));

               _trafficSettings.isDownloadWheneverDestinationIsSetEnabled() ? _trafficSettings.setDownloadWheneverDestinationIsSet(false) : _trafficSettings.setDownloadWheneverDestinationIsSet(true);
               break;
            }
            default:
            {
               ETG_TRACE_USR4(("PremiumTrafficListHandler::onCourierMessage(ButtonListItemUpdMsg) for LIST_ID_PREMIUM_TRAFFIC::DEFAULT"));
               isListUpdated = false;
               break;
            }
         }
      }

      if (isListUpdated == true)
      {
         ListRegistry::s_getInstance().updateList(LIST_ID_PREMIUM_TRAFFIC);
      }
   }
   return isMsgConsumed;
}


bool PremiumTrafficListHandler::onCourierMessage(const FetchPremiumTrafficSettingsReqMsg& oMsg)
{
   ETG_TRACE_USR4(("PremiumTrafficListHandler::onCourierMessage(FetchPremiumTrafficSettingsReqMsg)"));
   COURIER_UNUSED(oMsg);

   _trafficSettings = _navMiddleware.getTrafficSettings();
   return true;
}


bool PremiumTrafficListHandler::onCourierMessage(const ApplyPremiumTrafficSettingsReqMsg& oMsg)
{
   ETG_TRACE_USR4(("PremiumTrafficListHandler::onCourierMessage(ApplyPremiumTrafficSettingsReqMsg)"));
   COURIER_UNUSED(oMsg);

   applySettings();
   return true;
}


bool PremiumTrafficListHandler::onCourierMessage(const ApplySettingsReqMsg& oMsg)
{
   ETG_TRACE_USR4(("PremiumTrafficListHandler::onCourierMessage(ApplySettingsReqMsg)"));
   COURIER_UNUSED(oMsg);

   applySettings();
   return false;   // Generic settings message forwarded to other settings handler also
}


void PremiumTrafficListHandler::applySettings()
{
   ETG_TRACE_USR4(("PremiumTrafficListHandler::applySettings()"));

   navmiddleware::settings::TrafficSettings trafficSettings = _navMiddleware.getTrafficSettings();
   if (_trafficSettings != trafficSettings)
   {
      _navMiddleware.updateTrafficSettings(_trafficSettings);
      _navMiddleware.applySettings();

      if (_trafficSettings.isUsePremiumTrafficEnabled() != trafficSettings.isUsePremiumTrafficEnabled())
      {
         if (true == _trafficSettings.isUsePremiumTrafficEnabled())
         {
            setPrivacyMode(false);
            startPANConnection();
         }
         else
         {
            setPrivacyMode(true);
            stopPANConnection();
         }
      }
   }
}


bool PremiumTrafficListHandler::onCourierMessage(const PANConnectionStatusMsg& oMsg)
{
   bool isPANConnected = oMsg.GetIsConnected();
   ETG_TRACE_USR4(("PremiumTrafficListHandler::onCourierMessage(PANConnectionStatusMsg), isPANConnected : %d", isPANConnected));

   if (true == _navMiddleware.getTrafficSettings().isUsePremiumTrafficEnabled())
   {
      if (true == isPANConnected)
      {
         stopPANTimers();
         setBTTetheringState(BT_TETHERING_STATE__Success);
      }
      else
      {
         setBTTetheringState(BT_TETHERING_STATE__NoPhoneConnected);
      }
   }
   return true;
}


void PremiumTrafficListHandler::setBTTetheringState(BTTetheringState tetheringState)
{
   if ((*_btTetheringStateData).mTetheringIndex != tetheringState)
   {
      ETG_TRACE_USR4(("PremiumTrafficListHandler::setBTTetheringState(), BT tethering state : %d", tetheringState));
      (*_btTetheringStateData).mTetheringIndex = static_cast<uint8_t>(tetheringState);
      _btTetheringStateData.MarkAllItemsModified();
      _btTetheringStateData.SendUpdate(true);
   }
}


void PremiumTrafficListHandler::setPremiumTrafficState(bool premiumTrafficState)
{
   ETG_TRACE_USR4(("PremiumTrafficListHandler::setPremiumTrafficState(), Premium traffic state : %d", premiumTrafficState));

   navmiddleware::settings::TrafficSettings trafficSettings = _navMiddleware.getTrafficSettings();
   trafficSettings.setUsePremiumTraffic(premiumTrafficState);
   _navMiddleware.updateTrafficSettings(trafficSettings);
   _navMiddleware.applySettings();

   _trafficSettings = trafficSettings;
}


void PremiumTrafficListHandler::setPrivacyMode(bool privacyMode)
{
   ETG_TRACE_USR4(("PremiumTrafficListHandler::setPrivacyMode(), privacyMode : %d", privacyMode));

   ::navmiddleware::ConfigurationInfos::PrivacyMode navmidPrivacyMode =
      (true == privacyMode) ? navmiddleware::ConfigurationInfos::PRIVACY_MODE__ON : navmiddleware::ConfigurationInfos::PRIVACY_MODE__OFF;
   _navMiddleware.setPrivacyMode(navmidPrivacyMode);
   _navMiddleware.applySettings();
}


void PremiumTrafficListHandler::changePremiumTrafficUpdateFrequency(Candera::UInt32 selectedRowId)
{
   ETG_TRACE_USR4(("PremiumTrafficListHandler::changePremiumTrafficUpdateFrequency(), Row ID: %d, Frequency : %d", selectedRowId, TRAFFIC_FREQUENCY_VALUE[selectedRowId]));

   _premiumTrafficList.at(SETTING_UPDATE_FREQUENCY)._dropDownText = PremiumTrafficDropDownListLabel[selectedRowId];
   navmiddleware::ValidValue<int32_t> updateFrequency(TRAFFIC_FREQUENCY_VALUE[selectedRowId]);
   _trafficSettings.setTrafficRequestFrequencyInSecond(updateFrequency);
}


int PremiumTrafficListHandler::getUpdateFrequencyIndex()
{
   int updateFrequency = getUpdateFrequency();
   int updateFrequencyIndex = 0;
   for (int index = 0; index < MAX_NUM_UPDATE_FREQUENCY; index++)
   {
      if (updateFrequency == TRAFFIC_FREQUENCY_VALUE[index])
      {
         updateFrequencyIndex = index;
         break;
      }
   }
   ETG_TRACE_USR4(("PremiumTrafficListHandler::getUpdateFrequencyIndex(), Update frequency index : %d", updateFrequencyIndex));
   return updateFrequencyIndex;
}


int PremiumTrafficListHandler::getUpdateFrequency()
{
   int updateFrequency = (_trafficSettings.getTrafficRequestFrequencyInSecond().isValid())
                         ? _trafficSettings.getTrafficRequestFrequencyInSecond().getValue() : 0;
   return updateFrequency;
}


tSharedPtrDataProvider PremiumTrafficListHandler::configureDropdownList(::Courier::UInt32 listID)
{
   ETG_TRACE_USR4(("PremiumTrafficListHandler::configureDropdownList()"));
   const char* DATA_CONTEXT_DROPDOWN_LIST_ITEM = "DropdownListButton";
   const Candera::UInt8 MAX_MAP_VIEW_DROPDOWN_SIZE = 6;

   ListDataProviderBuilder listBuilder(listID, DATA_CONTEXT_DROPDOWN_LIST_ITEM);
   if (LIST_ID_PREMIUM_TRAFFIC_FREQUENCY == listID)
   {
      for (Candera::UInt8 idx = 0; idx < MAX_MAP_VIEW_DROPDOWN_SIZE; idx++)
      {
         listBuilder.AddItem(idx).AddData(PremiumTrafficDropDownListLabel[idx]);
      }
   }
   else
   {
      return tSharedPtrDataProvider();
   }

   return listBuilder.CreateDataProvider();
}


tSharedPtrDataProvider PremiumTrafficListHandler::getListDataProviderMainList()
{
   ETG_TRACE_USR4(("PremiumTrafficListHandler::getListDataProviderMainList()"));
   DataItemContext listItemName = "Layer_CheckboxOptions";
   ListDataProviderBuilder listBuilder(LIST_ID_PREMIUM_TRAFFIC);
   Candera::String currentOption;

   for (std::vector<PremiumTrafficListItem>::iterator itemNumber = _premiumTrafficList.begin(); itemNumber != _premiumTrafficList.end(); ++itemNumber)
   {
      if ((itemNumber->_listButtonType == LIST_BUTTON_TYPE_DROPDOWN) && (itemNumber->_order == SETTING_UPDATE_FREQUENCY))
      {
         itemNumber->_dropDownText = PremiumTrafficDropDownListLabel[getUpdateFrequencyIndex()];
         listItemName = "ComboBox_ListItem3";
         listBuilder.AddExpandableItem(
            itemNumber->_order,
            itemNumber->_dropDownListId,
            listItemName)
         .AddData(itemNumber->_text)
         .AddData(itemNumber->_dropDownListId)
         .AddData(itemNumber->_dropDownText);
      }
      else if ((itemNumber->_listButtonType == LIST_BUTTON_TYPE_TOGGLE) && (itemNumber->_order == SETTING_ENABLE_PREMIUM_TRAFFIC))
      {
         listItemName = "Layer_CheckboxOptions";
         listBuilder.AddItem(
            itemNumber->_order,
            0UL,           //  - unused
            listItemName)  //0 - Name(Id) of the button
         .AddData(itemNumber->_text)
         .AddData(!_trafficSettings.isUsePremiumTrafficEnabled());   // Set to 0 if check-box is to be enabled and vice-versa for disabled
      }
      else if ((itemNumber->_listButtonType == LIST_BUTTON_TYPE_TOGGLE) && (itemNumber->_order == SETTING_UPDATE_FOR_EACH_ROUTE))
      {
         listItemName = "Layer_CheckboxOptions";
         listBuilder.AddItem(
            itemNumber->_order,
            0UL,           //  - unused
            listItemName)  //0 - Name(Id) of the button
         .AddData(itemNumber->_text)
         .AddData(!_trafficSettings.isDownloadWheneverDestinationIsSetEnabled());   // Set to 0 if check-box is to be enabled and vice-versa for disabled
      }
      else
      {
         ETG_TRACE_USR4(("PremiumTrafficListHandler::getListDataProviderMainList(), Neither check-box nor drop-down"));
      }
   }
   return listBuilder.CreateDataProvider();
}


bool PremiumTrafficListHandler::onPropertyUpdateBTConnectionChanged()
{
   if (true == _panConnectionTimer.running())
   {
      ETG_TRACE_USR4(("PremiumTrafficListHandler::onPropertyUpdateBTConnectionChanged(), isDeviceConnected : %d", _infoStore.getBTConnectedDeviceAvailable()));
      if (true == _infoStore.getBTConnectedDeviceAvailable())
      {
         setBTTetheringState(BT_TETHERING_STATE__InProgress);
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(StartStopBtTetheringReqMsg)(true)));
         _panPollingTimer.start();
      }
      else
      {
         setBTTetheringState(BT_TETHERING_STATE__NoPhoneConnected);
         _panPollingTimer.stop();
      }
   }
   return true;
}


bool PremiumTrafficListHandler::onCourierMessage(const PANConnectionTimerExpiredMsg& oMsg)
{
   ETG_TRACE_USR4(("PremiumTrafficListHandler::onCourierMessage(PANConnectionTimerExpiredMsg)"));
   if (!_infoStore.isNonNaviVariant())
   {
      POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_PremiumTrafficDisabled"))));
   }
   stopPANConnection();
   setPremiumTrafficState(false);
   setPrivacyMode(true);
   ListRegistry::s_getInstance().updateList(LIST_ID_PREMIUM_TRAFFIC);
   return true;
}


bool PremiumTrafficListHandler::onCourierMessage(const PANPollingTimerExpiredMsg& oMsg)
{
   ETG_TRACE_USR4(("PremiumTrafficListHandler::onCourierMessage(PANPollingTimerExpiredMsg)"));

   if (true == _panConnectionTimer.running() && (true == _infoStore.getBTConnectedDeviceAvailable()))
   {
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(StartStopBtTetheringReqMsg)(true)));
      _panPollingTimer.start();
   }
   return true;
}


void PremiumTrafficListHandler::startPANConnection()
{
   ETG_TRACE_USR4(("PremiumTrafficListHandler::startPANConnection()"));

   POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(StartStopBtTetheringReqMsg)(true)));

   _panConnectionTimer.start();   // Starts 5 minutes timer to wait for successful PAN connection
   if (true == _infoStore.getBTConnectedDeviceAvailable())
   {
      setBTTetheringState(BT_TETHERING_STATE__InProgress);
      _panPollingTimer.start();   // Starts 10 seconds timer to wait for response to PAN connection request
   }
   else
   {
      setBTTetheringState(BT_TETHERING_STATE__NoPhoneConnected);
   }
}


void PremiumTrafficListHandler::stopPANConnection()
{
   ETG_TRACE_USR4(("PremiumTrafficListHandler::stopPANConnection()"));

   POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(StartStopBtTetheringReqMsg)(false)));
   setBTTetheringState(BT_TETHERING_STATE__NotApplicable);
   stopPANTimers();
}


void PremiumTrafficListHandler::stopPANTimers()
{
   ETG_TRACE_USR4(("PremiumTrafficListHandler::stopPANTimers()"));
   if (true == _panPollingTimer.running())
   {
      _panPollingTimer.stop();
   }
   if (true == _panConnectionTimer.running())
   {
      _panConnectionTimer.stop();
   }
}


#endif // HALL_TO_MDW_COM
