/**************************************************************************************
* @file         : PositionInfoDataUtils.cpp
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "PositionInfoDataUtils.h"
#include <iostream>
#include <string>
//#include <regex>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/PositionInfoDataUtils.cpp.trc.h"
#endif

#if defined (HALL_TO_MDW_COM)

static const float HEADING_STEP_SIZE = 11.25f;
static const int DAY = 6;
static const int MONTH = 4;
static const int YEAR = 2;
static const std::string DOUBLE_DASH = "--";
static const std::string EMPTY = " ";
static const std::string DASH = "-";
static const std::string DOT = ".";
static const std::string SLASH = "/";
static const std::string DEFAULT_ZERO_STRING = "0";
static const std::string DEFAULT_DAY_MONTH_STRING = "00";
static const std::string DEFAULT_YEAR_STRING = "0000";

using namespace navmiddleware;


DataBindingItem<GPSInfoDataBindingSource>& getGPSInfo()
{
   static DataBindingItem<GPSInfoDataBindingSource> sGPSInfo;
   return sGPSInfo;
}


DataBindingItem<WhereAmIScreenDataDataBindingSource>& getWhereAmIScreenData()
{
   static DataBindingItem<WhereAmIScreenDataDataBindingSource> sWhereAmIScreenData;
   return sWhereAmIScreenData;
}


void setGpsSensorInfos(const GnssDataInfo& gnssDataInfo, navmiddleware::DateFormat& dateFormat)
{
   std::stringstream stringStreamGnsstype;
   std::stringstream stringStreamReceivedSatelliteNum;
   std::stringstream stringStreamUsedSatelliteNum;

   /**
   * Updates the value of the time received from satellite
   */
   if ((*getGPSInfo()).mTimeInfo != gnssDataInfo.getTime().c_str())
   {
      (*getGPSInfo()).mTimeInfo = gnssDataInfo.getTime().c_str();
      getGPSInfo().MarkItemModified(ItemKey::GPSInfo::TimeInfoItem);
   }

   ::std::string dateInfo = gnssDataInfo.getGnssDataInfo();
   ::std::string year, month, day;
   char* strptr;
   char* delimiter = ":,";
   strptr = strtok(const_cast<char*>(dateInfo.c_str()), delimiter);

   for (int idx = 0; strptr != NULL; idx++)
   {
      ::std::string temp(strptr);

      //condition check is made to differetiate the day month and year from the GNSS data info using the delimiters
      //the substring starts at index 1 as index 0 is " "
      if (idx == DAY)
      {
         temp.substr(1) != DEFAULT_ZERO_STRING ? (day = temp.substr(1)) : (day = DEFAULT_DAY_MONTH_STRING);
      }
      else if (idx == MONTH)
      {
         temp.substr(1) != DEFAULT_ZERO_STRING ? (month = temp.substr(1)) : (month = DEFAULT_DAY_MONTH_STRING);
      }
      else if (idx == YEAR)
      {
         temp.substr(1) != DEFAULT_ZERO_STRING ? (year = temp.substr(1)) : (year = DEFAULT_YEAR_STRING);
      }
      else
      {
         // do nothing
      }

      strptr = strtok(NULL, delimiter);
   }
   updateDateBasedonFormat(day, month, year, dateFormat);

   if ((*getGPSInfo()).mHeadingInfo != gnssDataInfo.getHeading().c_str())
   {
      (*getGPSInfo()).mHeadingInfo = gnssDataInfo.getHeading().c_str();
      getGPSInfo().MarkItemModified(ItemKey::GPSInfo::HeadingInfoItem);
   }

   /**
   * Updates the number of Used Satellites for Information
   */
   stringStreamUsedSatelliteNum << gnssDataInfo.getNumberOfUsedSatellites();
   if ((*getGPSInfo()).mCapturedSatelliteNumber != stringStreamUsedSatelliteNum.str().c_str())
   {
      (*getGPSInfo()).mCapturedSatelliteNumber = stringStreamUsedSatelliteNum.str().c_str();
      getGPSInfo().MarkItemModified(ItemKey::GPSInfo::CapturedSatelliteNumberItem);
   }

   /**
   * Updates the number of Received Satellites for Information
   */
   stringStreamReceivedSatelliteNum << gnssDataInfo.getNumberOfReceivedSatellites();
   if ((*getGPSInfo()).mSearchingSatelliteNumber != stringStreamReceivedSatelliteNum.str().c_str())
   {
      (*getGPSInfo()).mSearchingSatelliteNumber = stringStreamReceivedSatelliteNum.str().c_str();
      getGPSInfo().MarkItemModified(ItemKey::GPSInfo::SearchingSatelliteNumberItem);
   }

   /**
   * Updates the satellite type used for Information
   */
   stringStreamGnsstype << gnssDataInfo.getSatelliteTypeUsed();
   if ((*getGPSInfo()).mGNSSTypeInfo != stringStreamGnsstype.str().c_str())
   {
      (*getGPSInfo()).mGNSSTypeInfo = stringStreamGnsstype.str().c_str();
      getGPSInfo().MarkItemModified(ItemKey::GPSInfo::GNSSTypeInfoItem);
   }
}


void updateDateBasedonFormat(const std::string& day, const std::string& month, const std::string& year, navmiddleware::DateFormat& dateFormat)
{
   ::std::string date;

   switch (dateFormat)
   {
      case DATEFORMAT_DD_MM_YYYY_SLASH:
         date = day + SLASH + month + SLASH + year;
         break;
      case DATEFORMAT_MM_DD_YYYY_SLASH:
         date = month + SLASH + day + SLASH + year;
         break;
      case DATEFORMAT_YYYY_MM_DD_SLASH:
         date = year + SLASH + month + SLASH + day;
         break;
      case DATEFORMAT_DD_MM_YYYY_DASH:
         date = day + DASH + month + DASH + year;
         break;
      case DATEFORMAT_MM_DD_YYYY_DASH:
         date = month + DASH + day + DASH + year;
         break;
      case DATEFORMAT_YYYY_MM_DD_DASH:
         date = year + DASH + month + DASH + day;
         break;
      case DATEFORMAT_DD_MM_YYYY_DOT:
         date = day + DOT + month + DOT + year;
         break;
      default:
         break;
   }

   if ((*getGPSInfo()).mDateValue != date.c_str())
   {
      (*getGPSInfo()).mDateValue = date.c_str();
      getGPSInfo().MarkItemModified(ItemKey::GPSInfo::DateValueItem);
   }
}


void setGpsSensorlatlongInfos(const navmiddleware::GnssDataInfo& gnssDataInfo, bool isDayModeActive, bool isDemoModeActive)
{
   std::stringstream altitudeValue, longitudeValue, latitudeValue;
   //Altitude ,Latitude,Longitude information is dependent on gnss mode
   navmiddleware::GnssMode gnssMode = gnssDataInfo.getGnssMode();

   setSatelliteIconVisibility(gnssDataInfo, isDayModeActive, isDemoModeActive);

   Candera::String stringGnssMode;
   switch (gnssMode)
   {
      case navmiddleware::GNSS_MODE__3D_FIX:
         stringGnssMode = Candera::String(LANGUAGE_STRING(TextId_0x14A0, "3D_FIX"));
         break;
      case navmiddleware::GNSS_MODE__2D_FIX:
         stringGnssMode = Candera::String(LANGUAGE_STRING(TextId_0x14A1, "2D_FIX"));
         break;
      case navmiddleware::GNSS_MODE__NO_FIX:
         stringGnssMode = Candera::String(LANGUAGE_STRING(TextId_0x14A2, "NO_FIX"));
         break;
      default:
         stringGnssMode = "UNKNOWN";
         break;
   }
   (*getGPSInfo()).mGnssModeType = stringGnssMode;
   getGPSInfo().MarkItemModified(ItemKey::GPSInfo::GnssModeTypeItem);

   if (gnssDataInfo.getHemisphereLatitude() == HEMISPHERE_LATITUDE__NORTHERN)
   {
      latitudeValue << gnssDataInfo.getLatitude().c_str() << " " << "N";
   }
   else if (gnssDataInfo.getHemisphereLatitude() == HEMISPHERE_LATITUDE__SOUTHERN)
   {
      latitudeValue << gnssDataInfo.getLatitude().c_str() << " " << "S";
   }
   else
   {
      latitudeValue << gnssDataInfo.getLatitude().c_str();
   }

   if (gnssDataInfo.getHemisphereLongitude() == HEMISPHERE_LONGITUDE__WESTERN)
   {
      longitudeValue << gnssDataInfo.getLongitude().c_str() << " " << "W";
   }
   else if (gnssDataInfo.getHemisphereLongitude() == HEMISPHERE_LONGITUDE__EASTERN)
   {
      longitudeValue << gnssDataInfo.getLongitude().c_str() << " " << "E";
   }
   else
   {
      longitudeValue << gnssDataInfo.getLongitude().c_str();
   }

   //3D_FIX : Show coordinates and altitude
   if (gnssMode == GNSS_MODE__3D_FIX)
   {
      altitudeValue << gnssDataInfo.getAltitudeAsString().c_str();
   }
   //2D_FIX : Only show coordinates, but not altitude
   else if (gnssMode == GNSS_MODE__2D_FIX)
   {
      altitudeValue << "" << DOUBLE_DASH;;
   }
   //NO_FIX  and UNKNOWN :show "--" do not show coordinate(lat. / long.) and altitude, because values could be completely wrong
   else
   {
      altitudeValue.str(EMPTY);
      altitudeValue << "" << DOUBLE_DASH;
      latitudeValue.str(EMPTY);
      latitudeValue << "" << DOUBLE_DASH;
      longitudeValue.str(EMPTY);
      longitudeValue << "" << DOUBLE_DASH;
   }

   /**
   * Updates the Altitude Value of current location
   */
   if ((*getGPSInfo()).mAltitudeValue != altitudeValue.str().c_str())
   {
      (*getGPSInfo()).mAltitudeValue = altitudeValue.str().c_str();
      getGPSInfo().MarkItemModified(ItemKey::GPSInfo::AltitudeValueItem);
   }

   /**
   * Updates the Latitude Value of current location
   */
   if ((*getGPSInfo()).mLatViaSensor != latitudeValue.str().c_str())
   {
      (*getGPSInfo()).mLatViaSensor = latitudeValue.str().c_str();
      getGPSInfo().MarkItemModified(ItemKey::GPSInfo::LatViaSensorItem);
   }

   /**
   * Updates the Longitude Value of current location
   */
   if ((*getGPSInfo()).mLongViaSensor != longitudeValue.str().c_str())
   {
      (*getGPSInfo()).mLongViaSensor = longitudeValue.str().c_str();
      getGPSInfo().MarkItemModified(ItemKey::GPSInfo::LongViaSensorItem);
   }
}


void setSatelliteIconVisibility(const navmiddleware::GnssDataInfo& gnssDataInfo, bool isDayModeActive, bool isDemoModeActive)
{
   ETG_TRACE_USR4(("PositionInfoDataUtils::setSatelliteIconVisibility(), isDemoModeActive : %d", isDemoModeActive));
   DataBindingItem<GPSInfoDataBindingSource>& gpsInfo = getGPSInfo();
   if(true == isDemoModeActive)
   {
      if((*gpsInfo).mSatelliteIconVisibility != false)
      {
         ETG_TRACE_USR4(("PositionInfoDataUtils::setSatelliteIconVisibility(), Icon hidden since demo mode is active"));
         (*gpsInfo).mSatelliteIconVisibility = false;
         gpsInfo.MarkItemModified(ItemKey::GPSInfo::SatelliteIconVisibilityItem);
      }
   }
   else
   {
      navmiddleware::GnssMode gnssMode = gnssDataInfo.getGnssMode();
      bool isGnssNoFix = ((GNSS_MODE__2D_FIX == gnssMode) || (GNSS_MODE__3D_FIX == gnssMode)) ? false : true;
      if ((*gpsInfo).mSatelliteIconVisibility != isGnssNoFix)
      {
         (*gpsInfo).mSatelliteIconVisibility = isGnssNoFix;
         ETG_TRACE_USR4(("PositionInfoDataUtils::setSatelliteIconVisibility(), isVisible : %d", isGnssNoFix));

         gpsInfo.MarkItemModified(ItemKey::GPSInfo::SatelliteIconVisibilityItem);
         setSatelliteIcon(isDayModeActive);
      }
   }
   sendGpsInfoData();
}


void setSatelliteIcon(bool isDayModeActive)
{
   if(true == (*getGPSInfo()).mSatelliteIconVisibility)
   {
      ETG_TRACE_USR4(("PositionInfoDataUtils::setSatelliteIcon(), isDayModeActive : %d", isDayModeActive));
      if (isDayModeActive)
      {
         (*getGPSInfo()).mSatelliteIcon = ImageLoader::getAssetBitmapImage("AppHmi_NavigationModule#Images#Common#Icon_Satellite_unavailable");
      }
      else
      {
         (*getGPSInfo()).mSatelliteIcon = ImageLoader::getAssetBitmapImage("AppHmi_NavigationModule#Images#Common#Icon_Satellite_unavailable_Nightmode");
      }
      getGPSInfo().MarkItemModified(ItemKey::GPSInfo::SatelliteIconItem);
      sendGpsInfoData();
   }
}


bool sendGpsInfoData()
{
   bool retVal = true;
   if (getGPSInfo().HasModifiedItems()
         && (getGPSInfo().SendUpdate() == false))
   {
      retVal = false;
   }

   return retVal;
}


void setWhereAmIScreenDataWhereAmIStatusInfos(const navmiddleware::WhereAmIInfo& whereAmIInfo)
{
   // current street info : street number and current street name
   bool isValidAddress = false;
   std::stringstream stringStreamCurrentStreetInfo;
   if (!(whereAmIInfo.getCurrentStreetInfo().m_streetNumber.empty()))
   {
      stringStreamCurrentStreetInfo << whereAmIInfo.getCurrentStreetInfo().m_streetNumber << "/";
   }

   stringStreamCurrentStreetInfo << ((!(whereAmIInfo.getCurrentStreetInfo().m_streetName.empty())) ? whereAmIInfo.getCurrentStreetInfo().m_streetName : "UNKNOWN STREET NAME"/*LANGUAGE_STRING(TextId_IT_00971, "UNKNOWN STREET NAME").GetCString()*/);

   if ((*getWhereAmIScreenData()).mCurrentStreetInfo != stringStreamCurrentStreetInfo.str().c_str())
   {
      (*getWhereAmIScreenData()).mCurrentStreetInfo = stringStreamCurrentStreetInfo.str().c_str();
      getWhereAmIScreenData().MarkItemModified(ItemKey::WhereAmIScreenData::CurrentStreetInfoItem);
   }

   // current city
   if ((*getWhereAmIScreenData()).mCurrentCityInfo != whereAmIInfo.getCurrentStreetInfo().m_cityName.c_str())
   {
      (*getWhereAmIScreenData()).mCurrentCityInfo = whereAmIInfo.getCurrentStreetInfo().m_cityName.c_str();
      getWhereAmIScreenData().MarkItemModified(ItemKey::WhereAmIScreenData::CurrentCityInfoItem);
   }
}


bool sendWhereAmIScreenData()
{
   bool retVal = true;
   if (getWhereAmIScreenData().HasModifiedItems()
         && (getWhereAmIScreenData().SendUpdate() == false))
   {
      retVal = false;
   }

   return retVal;
}


float convertHeadingIndexToHeading(int headingIndex)
{
   return static_cast<float>(headingIndex * HEADING_STEP_SIZE);
}


int convertHeadingToHeadingIndex(float heading)
{
   // calculate the current heading of car
   int headingIndex = static_cast<int>(heading / HEADING_STEP_SIZE + 0.5);

   return headingIndex;
}


#endif // HALL_TO_MDW_COM
