/**************************************************************************************
* @file         : NavigationSettingsListHandler.cpp
* @author       : ECG-Ramesh Kesavan
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "NavigationSettingsListHandler.h"
#include "NaviSettingsUtil.h"
#include "CgiExtensions/ImageLoader.h"
#include <vector>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/NavigationSettingsListHandler.cpp.trc.h"
#endif

static const char* const DATA_CONTEXT_TEXT_LIST_ITEM = "Layer_TextListItem";
static const char* const DATA_CONTEXT_COMBOBOX_LIST_ITEMS_2 = "Combobox_ListItem2";

static const char* routeOptionsBorderIcons[MAX_SIZE_ROUTE_TYPE][2] =
{
   {"AppHmi_NavigationModule#Images#N_Map#Icon_Border_Shortest_Night", "AppHmi_NavigationModule#Images#N_Map#Icon_Border_Shortest_Day"},
   {"AppHmi_NavigationModule#Images#N_Map#Icon_Border_Fastest_Night",  "AppHmi_NavigationModule#Images#N_Map#Icon_Border_Fastest_Day" },
   {"AppHmi_NavigationModule#Images#N_Map#Icon_Border_Economy_Night",  "AppHmi_NavigationModule#Images#N_Map#Icon_Border_Economy_Day" }
};


#ifdef  HALL_TO_MDW_COM

using namespace navmiddleware::settings;


NavigationSettingsListHandler::NavigationSettingsListHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _listId(LIST_ID_NAV_SETTINGS)
   , _currentSatelliteSource(SATELLITE_SOURCE_GPS_GLONASS)
   , _previousSatelliteSource(SATELLITE_SOURCE_GPS_GLONASS)
{
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_NAV_SETTINGS, this);
}


NavigationSettingsListHandler::~NavigationSettingsListHandler()
{
}


void NavigationSettingsListHandler::initialize()
{
   ETG_TRACE_USR4(("NavigationSettingsListHandler::initialize()"));

   configureCarsor();

   _currentSatelliteSource = SATELLITE_SOURCE_GPS_GLONASS;
   if (KDS_SATELLITE_SRC__GPS_AND_GALILEO == _infoStore.getSatelliteSource())
   {
      _currentSatelliteSource = SATELLITE_SOURCE_GPS_GALILEO;
   }

   _previousSatelliteSource = _currentSatelliteSource;
   _navigationSettingsList.reserve(MAX_SIZE_NAVIGATION_SETTINGS_LIST);
   _navigationSettingsList.push_back(NavigationSettingsListItem(NavigationSettings_ListItemText[0], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_BUTTON, ROUTE_PREFERENCES));
   _navigationSettingsList.push_back(NavigationSettingsListItem(NavigationSettings_ListItemText[1], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_BUTTON, MAP_VIEW));
   _navigationSettingsList.push_back(NavigationSettingsListItem(NavigationSettings_ListItemText[2], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_BUTTON, GUIDANCE_SETTINGS));
   _navigationSettingsList.push_back(NavigationSettingsListItem(NavigationSettings_ListItemText[3], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_BUTTON, TRAFFIC_ICON_SETTING));
   _navigationSettingsList.push_back(NavigationSettingsListItem(NavigationSettings_ListItemText[4], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_BUTTON, Position_And_GPSInfo));
   _navigationSettingsList.push_back(NavigationSettingsListItem(NavigationSettings_ListItemText[5], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_BUTTON, MY_POIS));
   _navigationSettingsList.push_back(NavigationSettingsListItem(NavigationSettings_ListItemText[6], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_BUTTON, VEHICLE_PARAMETERS));
   _navigationSettingsList.push_back(NavigationSettingsListItem(NavigationSettings_ListItemText[7], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_DROPDOWN, SATELLITE_SOURCE, SatelliteSourceDropDownListLabel[_currentSatelliteSource], LIST_ID_DROPDOWN_SATELLITE_SOURCE));
   _navigationSettingsList.push_back(NavigationSettingsListItem(NavigationSettings_ListItemText[8], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_BUTTON, RESET_ALL_NAVIGATION));
}


void NavigationSettingsListHandler::deinitialize()
{
}


tSharedPtrDataProvider NavigationSettingsListHandler::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   _listId = oMsg.GetListId();
   ListDataProviderBuilder listBuilder(_listId);

   if (LIST_ID_NAV_SETTINGS == _listId)
   {
      ETG_TRACE_USR4(("NavigationSettingsListHandler::getListDataProvider(), List Id : %d", _listId));
      int listIndex = 0;
      for (std::vector<NavigationSettingsListItem>::iterator listItem = _navigationSettingsList.begin();
            ((listItem != _navigationSettingsList.end()) && (true == listItem->_isVisible));
            ++listItem, ++listIndex)
      {
         if (listItem->_listButtonType == LIST_BUTTON_TYPE_BUTTON)
         {
            listBuilder.AddItem(
               listIndex, // - identifies the row
               0UL, // - unused
               DATA_CONTEXT_TEXT_LIST_ITEM)
            .AddData(listItem->_text);
         }
         else if (listItem->_listButtonType == LIST_BUTTON_TYPE_DROPDOWN)
         {
            bool _satelliteSourceHighlight[MAX_LIST_SATELLITE_SOURCE] = {0};
            _satelliteSourceHighlight[_currentSatelliteSource] = true;
            listBuilder.AddItem(listIndex, 0, DATA_CONTEXT_COMBOBOX_LIST_ITEMS_2)
            .AddData(NavigationSettings_ListItemText[listIndex])
            .AddId(listIndex, 1).AddData(SatelliteSourceDropDownListLabel[0]).AddData(_satelliteSourceHighlight[0])
            .AddId(listIndex, 2).AddData(SatelliteSourceDropDownListLabel[1]).AddData(_satelliteSourceHighlight[1])
            .AddData(SatelliteSourceDropDownListLabel[_currentSatelliteSource]);
         }
         else
         {
         }
      }
   }
   else
   {
      //invalid listId
      return tSharedPtrDataProvider();
   }

   return listBuilder.CreateDataProvider();
}


bool NavigationSettingsListHandler::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   bool msgProcessed = false;
   unsigned int listID = oMsg.GetListId();
   unsigned int rowIdx = (unsigned int)oMsg.GetHdl();
   const enReaction  reaction = oMsg.GetReaction();

   if ((reaction == enRelease) && (LIST_ID_NAV_SETTINGS == listID))
   {
      ETG_TRACE_USR4(("NavigationSettingsListHandler::onCourierMessage(ButtonListItemUpdMsg), List Id : %d, Row Id : %d", listID, rowIdx));
      msgProcessed = true;
      if (rowIdx == RESET_ALL_NAVIGATION)
      {
         POST_MSG((COURIER_MESSAGE_NEW(UpdateDeletePopupTextReqMsg)(POPUP_RESET_ALL_SETTINGS)));
         ListRegistry::s_getInstance().updateList(listID);
      }
      else if (rowIdx == SATELLITE_SOURCE)
      {
         _currentSatelliteSource = (enSatelliteSourceOption)(oMsg.GetSubHdl() - 1);
         ::hmibase::widget::combobox::ComboBoxReqMsg* cbToggleMsg = COURIER_MESSAGE_NEW(::hmibase::widget::combobox::ComboBoxReqMsg)(0, ::hmibase::widget::combobox::enComboBoxAction::Close);
         if (cbToggleMsg != NULL)
         {
            cbToggleMsg->Post();
         }
         ListRegistry::s_getInstance().updateList(listID);
      }
      else
      {
         msgProcessed = false;
      }
   }
   return msgProcessed;
}


bool NavigationSettingsListHandler::onCourierMessage(const CheckIsSatelliteSourceChangedReqMsg& oMsg)
{
   ETG_TRACE_USR4(("NavigationSettingsListHandler::onCourierMessage(CheckIsSatelliteSourceChangedReqMsg"));

   if (_previousSatelliteSource != _currentSatelliteSource)
   {
      _previousSatelliteSource = _currentSatelliteSource;
      POST_MSG((COURIER_MESSAGE_NEW(UpdateDeletePopupTextReqMsg)(POPUP_SATELLITE_SOURCE_CHANGE)));

      uint8_t kdsSatelliteSource = KDS_SATELLITE_SRC__GPS_AND_GLONASS;
      if (SATELLITE_SOURCE_GPS_GALILEO == _currentSatelliteSource)
      {
         kdsSatelliteSource = KDS_SATELLITE_SRC__GPS_AND_GALILEO;
      }
      _infoStore.setSatelliteSource(kdsSatelliteSource);
   }
   return true;
}


bool NavigationSettingsListHandler::onCourierMessage(const ResetAllNaviSettingsReqMsg& /* oMsg */)
{
   ETG_TRACE_USR4(("NavigationSettingsListHandler::onCourierMessage(ResetAllNaviSettingsReqMsg"));

   bool msgProcessed = false;  // Set to false so that other navigation settings list handler can process the message
   const Candera::String DATA_CONTEXT__TEXT_RESETPOPUP_LINE1 = LANGUAGE_STRING(TextId_0x145F, "All navigation settings have been");
   const Candera::String DATA_CONTEXT__TEXT_RESETPOPUP_LINE2 = LANGUAGE_STRING(TextId_0x1460, "returned to their defaults.");

   (*_PopupText).mTextLine1 = DATA_CONTEXT__TEXT_RESETPOPUP_LINE1;
   (*_PopupText).mTextLine2 = DATA_CONTEXT__TEXT_RESETPOPUP_LINE2;
   _PopupText.MarkAllItemsModified();
   _PopupText.SendUpdate(true);
   _navMiddleware.restoreDefaultSettings(navmiddleware::settings::SETTINGS_GROUP__ALL);
   _navMiddleware.applySettings();
   POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_MultiLine_Text"))));
   resetNaviHmiSettings();
   // update the list after Navi reset.
   ListRegistry::s_getInstance().updateList(LIST_ID_NAV_SETTINGS);

   configureCarsor();

   return msgProcessed;
}


bool NavigationSettingsListHandler::onCourierMessage(const ApplySettingsReqMsg& oMsg)
{
   ETG_TRACE_USR4(("NavigationSettingsListHandler::onCourierMessage(ApplySettingsReqMsg"));

   COURIER_UNUSED(oMsg);
   _navMiddleware.applySettings();

   return true;
}


void NavigationSettingsListHandler::resetNaviHmiSettings()
{
   ETG_TRACE_USR4(("NavigationSettingsListHandler::resetNaviHmiSettings()"));

   // Reset HMI settings which are handled from Infostore
   _infoStore.setEstimatedTimeMode(InfoStore::ARRIVAL_TIME);
   _infoStore.setEstimatedTimeType(InfoStore::DESTINATION);

   // TODO: Enable following reset on MapScreen Data and AudioDisplay in Map Feature
   // Sets the correct map headder configuration on reset
   // updateMapHeader(_navMiddleware.getMapSettings().isAudioDisplayOnMapEnabled());
   // POST_MSG((COURIER_MESSAGE_NEW(AudioDisplayOnMapStatusReqMsg)(_navMiddleware.getMapSettings().isAudioDisplayOnMapEnabled())));

   //Posting Default volume to audio manager
   const unsigned int guidanceVolume = 12;
   POST_MSG((COURIER_MESSAGE_NEW(SetGuidanceVolumeReqMsg)((Candera::UInt8)(guidanceVolume))));

   _currentSatelliteSource = (_infoStore.isNarRegion()) ? SATELLITE_SOURCE_GPS_GALILEO : SATELLITE_SOURCE_GPS_GLONASS;
   _infoStore.setSatelliteSource((_infoStore.isNarRegion()) ? KDS_SATELLITE_SRC__GPS_AND_GALILEO : KDS_SATELLITE_SRC__GPS_AND_GLONASS);
}


void NavigationSettingsListHandler::configureCarsor()
{
   ETG_TRACE_USR4(("NavigationSettingsListHandler::configureCarsor()"));

   MapCarsorSettings carsorSettings = _navMiddleware.getMapCarsorSettings();
   MapCarsorSettings::MapCarsorType carsorType = carsorSettings.getMapCarsorType();
   enVehicleType vehicleType = _infoStore.getVehicleType();
   ETG_TRACE_USR4(("NavigationSettingsListHandler::configureCarsor(), Vehicle Type : %d, Carsor type : %d", vehicleType, carsorType));

   bool isCarsorUpdated = false;
   switch (vehicleType)
   {
      case TRUCK:
      {
         if (MapCarsorSettings::MAP_CARSOR_TYPE__CONCEPT_CAR_2 != carsorType)
         {
            carsorSettings.setMapCarsorType(MapCarsorSettings::MAP_CARSOR_TYPE__CONCEPT_CAR_2);
            isCarsorUpdated = true;
         }
         break;
      }
      case COACH:
      {
         if (MapCarsorSettings::MAP_CARSOR_TYPE__CONCEPT_CAR_4 != carsorType)
         {
            carsorSettings.setMapCarsorType(MapCarsorSettings::MAP_CARSOR_TYPE__CONCEPT_CAR_4);
            isCarsorUpdated = true;
         }
         break;
      }
      default:
      {
         // Default carsor image corresponds to blue arrow
         if (MapCarsorSettings::MAP_CARSOR_TYPE__CONCEPT_CAR_1 != carsorType)
         {
            carsorSettings.setMapCarsorType(MapCarsorSettings::MAP_CARSOR_TYPE__CONCEPT_CAR_1);
            isCarsorUpdated = true;
         }
      }
   } // End of switch(vehicleType)

   if (true == isCarsorUpdated)
   {
      _navMiddleware.updateMapCarsorSettings(carsorSettings);
      _navMiddleware.applySettings();
   }
}


void NavigationSettingsListHandler::configureRouteOptionsBorder(bool isDayModeActive)
{
   ETG_TRACE_USR4(("NavigationSettingsListHandler::configureRouteOptionsBorder(%d)", isDayModeActive));

   int iconIndex = static_cast<int>(isDayModeActive);
   (*_routeOptionsBorder).mShortestIcon = ImageLoader::getAssetBitmapImage(routeOptionsBorderIcons[ROUTE_TYPE_SHORTEST][iconIndex]);
   (*_routeOptionsBorder).mFastestIcon = ImageLoader::getAssetBitmapImage(routeOptionsBorderIcons[ROUTE_TYPE_FASTEST][iconIndex]);
   (*_routeOptionsBorder).mOptimisedIcon = ImageLoader::getAssetBitmapImage(routeOptionsBorderIcons[ROUTE_TYPE_ECONOMIC][iconIndex]);
   _routeOptionsBorder.MarkAllItemsModified();
   _routeOptionsBorder.SendUpdate(true);
}


#endif // HALL_TO_MDW_COM
