/**************************************************************************************
* @file         : MapSettingsShowPOIListHandler.cpp
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include <vector>
#include "MapSettingsShowPOIListHandler.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/MapSettingsShowPOIListHandler.cpp.trc.h"
#endif

#if defined (HALL_TO_MDW_COM)


MapSettingsShowPOIListHandler::MapSettingsShowPOIListHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _listId(0)
   , _startIndex(0)
   , _windowElementSize(0)
{
}


MapSettingsShowPOIListHandler::~MapSettingsShowPOIListHandler()
{
}


void MapSettingsShowPOIListHandler::initialize()
{
   _mapDynamicIconSettings = _navMiddleware.getMapDynamicIconSettings();
}


void MapSettingsShowPOIListHandler::deinitialize()
{
}


bool MapSettingsShowPOIListHandler::onCourierMessage(const ShowPoisOnMapViewReqMsg& oMsg)
{
   bool isListShown = oMsg.GetIsListShown();
   ETG_TRACE_USR4(("MapSettingsShowPOIListHandler::onCourierMessage(ShowPoisOnMapViewReqMsg), Is list shown : %d", isListShown));

   if (true == isListShown)
   {
      // On entering into the scene, POI icon's visibility status is fetched from navmiddleware
      _mapDynamicIconSettings = _navMiddleware.getMapDynamicIconSettings();
   }
   else
   {
      // On exiting from the scene, POI icon's visibility status is updated to navmiddleware. ApplySettings is invoked in exit action of N_OPT state (Top state for navigation settings)
      _navMiddleware.updateMapDynamicIconSettings(_mapDynamicIconSettings);
   }

   return true;
}


bool MapSettingsShowPOIListHandler::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   bool msgProcessed = false;
   if (oMsg.GetListId() == LIST_ID_MAP_SHOW_POI_SETTINGS)
   {
      _listId = oMsg.GetListId();
      _startIndex = oMsg.GetStartIndex();
      _windowElementSize = oMsg.GetWindowElementSize();

      ETG_TRACE_USR4(("MapSettingsShowPOIListHandler::onCourierMessage(ListDateProviderReqMsg), List ID: %d, startIndex %d, elementSize %d", _listId, _startIndex, _windowElementSize));
      updateMapDynamicIconSettingsList();
      msgProcessed = true;
   }

   return msgProcessed;
}


bool MapSettingsShowPOIListHandler::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   bool msgProcessed = false;
   if ((oMsg.GetListId() == LIST_ID_MAP_SHOW_POI_SETTINGS) && (oMsg.GetReaction() == enRelease))
   {
      unsigned int index = oMsg.GetHdl();
      ETG_TRACE_USR4(("MapSettingsShowPOIListHandler::onCourierMessage(ButtonListItemUpdMsg), List Id : %d, Row Id : %d", oMsg.GetListId(), index));

      index = index - _startIndex;
      toggleStatusUpdateToMDW(index);
      msgProcessed = true;
   }

   return msgProcessed;
}


void MapSettingsShowPOIListHandler::toggleStatusUpdateToMDW(const unsigned int& selectedRowId)
{
   std::vector<navmiddleware::settings::MapDynamicIconSettings::PoiCategoryInfo>& mapDynamicIconSettingsList = _mapDynamicIconSettings.getPoiCategoryInfoList();

   if (selectedRowId < mapDynamicIconSettingsList.size())
   {
      mapDynamicIconSettingsList.at(selectedRowId).m_poiCategoryVisibilityOnMap = !(mapDynamicIconSettingsList.at(selectedRowId).m_poiCategoryVisibilityOnMap);
      ETG_TRACE_USR4(("MapSettingsShowPOIListHandler::toggleStatusUpdateToMDW(), POI category name : %s", mapDynamicIconSettingsList.at(selectedRowId).m_poiCategoryName.c_str()));
      ETG_TRACE_USR4(("MapSettingsShowPOIListHandler::toggleStatusUpdateToMDW(), Row Id : %d, Modified visibility : %d", selectedRowId, mapDynamicIconSettingsList.at(selectedRowId).m_poiCategoryVisibilityOnMap));

      updateMapDynamicIconSettingsList();
   }
}


tSharedPtrDataProvider MapSettingsShowPOIListHandler::getMapDynamicIconSelectionList() const
{
   ETG_TRACE_USR4(("MapSettingsShowPOIListHandler::getMapDynamicIconSelectionList()"));

   ListDataProviderBuilder listBuilder(_listId);
   DataItemContext listItemName = "ShowPOIListItem";
   const  std::vector<navmiddleware::settings::MapDynamicIconSettings::PoiCategoryInfo>& mapDynamicIconSettingsList = _mapDynamicIconSettings.getPoiCategoryInfoList();
   std::vector<navmiddleware::settings::MapDynamicIconSettings::PoiCategoryInfo>::const_iterator it = mapDynamicIconSettingsList.begin();

   unsigned int idx = _startIndex;
   if (_listId == LIST_ID_MAP_SHOW_POI_SETTINGS)
   {
      if (!(mapDynamicIconSettingsList.empty()))
      {
         for (; (idx < _windowElementSize) && (it != mapDynamicIconSettingsList.end()); ++it, ++idx)
         {
            ETG_TRACE_USR4(("MapSettingsShowPOIListHandler::getMapDynamicIconSelectionList(), POI category name : %s", (*it).m_poiCategoryName.c_str()));
            ETG_TRACE_USR4(("MapSettingsShowPOIListHandler::getMapDynamicIconSelectionList(), Row Id : %d, Visibility : %d", idx, (*it).m_poiCategoryVisibilityOnMap));

            ShowPOIsOnMapItemData imageItem;
            if ((true == it->m_poiCategoryIcon.isValid()) && (0 < it->m_poiCategoryIcon.getValue().getBlobSize()))
            {
               imageItem.mIconNormalBitmap = loadImage((const char*)it->m_poiCategoryIcon.getValue().getBlob(), it->m_poiCategoryIcon.getValue().getBlobSize(), true);
            }
            else
            {
               ETG_TRACE_ERR(("MapSettingsShowPOIListHandler::getMapDynamicIconSelectionList(), POI category icon unavailable"));
               imageItem.mIconNormalBitmap = ImageLoader::getAssetBitmapImage("AppHmi_NavigationModule#Images#N_CategoryRoot#Icon_Category_3_normal");
            }

            listBuilder.AddItem(
               idx, // - identifies the row
               0UL, // - unused
               listItemName)                                 // 0 - Name(Id) of the button
            .AddData((*it).m_poiCategoryName.c_str())        // 1 - Text
            .AddData(!(*it).m_poiCategoryVisibilityOnMap)    // 2 - Check-box state.
            // Note: Check-box behavior is such that when FALSE is assigned, check-box is enabled and vice versa. Hence negation added to the above line.
            .AddDataBindingUpdater<ShowPOIsOnMapItemDataBindingSource>(imageItem);   // POI category icon
         }
      }
      else
      {
         ETG_TRACE_USR4(("MapSettingsShowPOIListHandler::getMapDynamicIconSelectionList(), List is empty"));
      }
   }

   return listBuilder.CreateDataProvider();
}


void MapSettingsShowPOIListHandler::updateMapDynamicIconSettingsList()
{
   tSharedPtrDataProvider dataProvider = getMapDynamicIconSelectionList();
   if (dataProvider != NULL)
   {
      ETG_TRACE_USR4(("MapSettingsShowPOIListHandler::updateMapDynamicIconSettingsList(), startIndex = %d, listSize = %d", dataProvider->startIndex(), dataProvider->listSize()));
      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
   }
}


#endif // HALL_TO_MDW_COM
