/**************************************************************************************
* @file         : MapSettingsListHandler.cpp
* @author       : ECG-Naveen Thangamalayan
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include  <vector>
#include "MapSettingsListHandler.h"
#include "Map/MapScreenDataUtils.h"


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/MapSettingsListHandler.cpp.trc.h"
#endif

static const char* const DATA_CONTEXT_TEXT_LIST_ITEM = "Layer_Others";
static const char* const DATA_CONTEXT_COMBOBOX_LIST_ITEMS_4 = "ComboBox_ListItem4";
static const char* const DATA_CONTEXT_COMBOBOX_LIST_ITEMS_2 = "ComboBox_ListItem2";


#ifdef  HALL_TO_MDW_COM


MapSettingsListHandler::MapSettingsListHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : NavMiddlewarePropertyUpdateConsumer(navMiddleware)
   , _listId(0)
   , _startIndex(0)
   , _windowElementSize(0)
   , _infoStore(infoStore)
   , _mapViewSettingOption(MAP_SETTINGS_2D_MAP)
   , _guidanceViewSettingOption(LEFTSIDE_INFO_INTERSECTION)
{
   //Register the ListID
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_MAP_SETTINGS, this);
   resetSelectedOption();
}


MapSettingsListHandler::~MapSettingsListHandler()
{
}


void MapSettingsListHandler::initialize()
{
   _navMiddleware.registerMapPropertyUpdateCallback(*this);
   navmiddleware::settings::MapView currentMapView =  _navMiddleware.getChangeMapViewSettings().getMapView();

   _guidanceViewSettingOption = (navmiddleware::settings::MAP_HMI_SPLITMODE_FULL_MAP == _navMiddleware.getMapViewSettings().getMapHMISplitMode()) ? LEFTSIDE_INFO_FULLMAP : LEFTSIDE_INFO_INTERSECTION;
   bool multiViewActive = ((currentMapView == navmiddleware::settings::MAPVIEW_2D_SPLIT) || (currentMapView == navmiddleware::settings::MAPVIEW_3D_SPLIT));
   //(*getMapScreenData()).mEnableSplitMapWidgets = multiViewActive;
   //getMapScreenData().MarkItemModified(ItemKey::MapScreenData::EnableSplitMapWidgetsItem);
   //sendMapScreenData();

   _mapSettingsList.reserve(MAX_SIZE_MAPSETTINGS_LIST);
   _mapSettingsList.push_back(MapSettingsListItem(MapSettingsListItemText[0], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_DROPDOWN, MAP_SETTINGS_MAP_VIEW, DualMapViewDropDownListLabel[currentMapView], LIST_ID_DROPDOWN_MAP_VIEW));
   _mapSettingsList.push_back(MapSettingsListItem(MapSettingsListItemText[1], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_DROPDOWN, MAP_SETTINGS_GUIDANCE_VIEW, LeftSideInfoDropDownListLabel[_guidanceViewSettingOption], LIST_ID_DROPDOWN_GUIDANCE_VIEW));
   _mapSettingsList.push_back(MapSettingsListItem(MapSettingsListItemText[2], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_BUTTON, MAP_SETTINGS_SHOW_POI_ON_MAP));
   _mapSettingsList.push_back(MapSettingsListItem(MapSettingsListItemText[3], IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_BUTTON, MAP_SETTINGS_MORE_OPTION));
}


void MapSettingsListHandler::resetSelectedOption()
{
   memset(_mapView, 0, MAX_SIZE_MAPSETTINGS_LIST * sizeof(_mapView[0]));
   memset(_guidanceView, 0, MAX_SIZE_GUIDANCE_VIEW_OPTION * sizeof(_guidanceView[0]));
}


void MapSettingsListHandler::deinitialize()
{
   _navMiddleware.unregisterMapPropertyUpdateCallback(*this);
}


tSharedPtrDataProvider MapSettingsListHandler::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   _listId = oMsg.GetListId();
   tSharedPtrDataProvider dataProvider;
   if (_listId == LIST_ID_MAP_SETTINGS)
   {
      _mapViewSettingOption = static_cast<enMapViewSettingOptions>(_navMiddleware.getChangeMapViewSettings().getMapView());
      _guidanceViewSettingOption = (navmiddleware::settings::MAP_HMI_SPLITMODE_FULL_MAP == _navMiddleware.getMapViewSettings().getMapHMISplitMode()) ? LEFTSIDE_INFO_FULLMAP : LEFTSIDE_INFO_INTERSECTION;
      _currentMapView = DualMapViewDropDownListLabel[_mapViewSettingOption];
      _currentGuidanceView = LeftSideInfoDropDownListLabel[_guidanceViewSettingOption];

      resetSelectedOption();
      if (_infoStore.isSecondaryMapStreamingActive())
      {
         (*_streamedMapSettingData).mStreamedMapTypeSettingStatus = true;
         _mapView[MAP_SETTINGS_2D_MAP] = true;
         _guidanceView[LEFTSIDE_INFO_FULLMAP] = true;
      }
      else
      {
         (*_streamedMapSettingData).mStreamedMapTypeSettingStatus = false;
         _mapView[_mapViewSettingOption] = true;
         _guidanceView[_guidanceViewSettingOption] = true;
      }
      _streamedMapSettingData.MarkAllItemsModified();
      _streamedMapSettingData.SendUpdate(true);

      ETG_TRACE_USR4(("MapSettingsListHandler::getListDataProvider(%d)", oMsg.GetListId()));
      ListDataProviderBuilder listBuilder(LIST_ID_MAP_SETTINGS);

      listBuilder.AddItem(0, 0, DATA_CONTEXT_COMBOBOX_LIST_ITEMS_4)
      .AddData(MapSettingsListItemText[0])
      .AddId(0, 1).AddData(DualMapViewDropDownListLabel[0]).AddData(_mapView[0])
      .AddId(0, 2).AddData(DualMapViewDropDownListLabel[1]).AddData(_mapView[1])
      .AddId(0, 3).AddData(DualMapViewDropDownListLabel[2]).AddData(_mapView[2])
      .AddId(0, 4).AddData(DualMapViewDropDownListLabel[3]).AddData(_mapView[3])
      .AddData(_currentMapView);
      listBuilder.AddItem(1, 0, DATA_CONTEXT_COMBOBOX_LIST_ITEMS_2)
      .AddData(MapSettingsListItemText[1])
      .AddId(1, 1).AddData(LeftSideInfoDropDownListLabel[0]).AddData(_guidanceView[0])
      .AddId(1, 2).AddData(LeftSideInfoDropDownListLabel[1]).AddData(_guidanceView[1])
      .AddData(_currentGuidanceView);
      listBuilder.AddItem(2, 0, DATA_CONTEXT_TEXT_LIST_ITEM)
      .AddData(MapSettingsListItemText[2]);
      listBuilder.AddItem(3, 0, DATA_CONTEXT_TEXT_LIST_ITEM)
      .AddData(MapSettingsListItemText[3]);
      dataProvider = listBuilder.CreateDataProvider();
   }

   return dataProvider;
}


bool MapSettingsListHandler::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   bool isMsgConsumed = false;
   Candera::UInt32 listId = oMsg.GetListId();

   if (oMsg.GetListId() == LIST_ID_MAP_SETTINGS && oMsg.GetReaction() == enRelease)
   {
      ETG_TRACE_USR4(("MapSettingsListHandler::ButtonListItemUpdMsg: ListId: %d, Hdl: %d", oMsg.GetListId(), oMsg.GetHdl()));
      processComboBox(oMsg.GetHdl(), oMsg.GetSubHdl());
      isMsgConsumed = true;
   }
   return isMsgConsumed;
}


void MapSettingsListHandler::processComboBox(Courier::UInt32 hdl, Courier::UInt32 subHdl)
{
   ETG_TRACE_USR4(("MapSettingsListHandler::processComboBox: Hdl: %d, SubHdl: %d", hdl, subHdl));
   switch (hdl)
   {
      case MAP_SETTINGS_MAP_VIEW:
      {
         subHdl = subHdl - 1;
         ETG_TRACE_USR4(("MapSettingsListHandler::processComboBox:  SubHdl: %d", subHdl));
         if ((_infoStore.isSecondaryMapStreamingActive() == false)
               || ((_infoStore.isSecondaryMapStreamingActive() == true) && (subHdl == 0 /*MAP_SETTINGS_2D_MAP*/)))
         {
            if (((_infoStore.isNo3D_TypeSelected() == true) && (subHdl < 2))
                  || (_infoStore.isNo3D_TypeSelected() == false))
            {
               changeDualMapViewSettings(subHdl);
            }
         }
         else
         {
            ETG_TRACE_USR4(("MapSettingsListHandler::processComboBox:, Secondary map streaming is active, split map view is disabled"));
         }
         break;
      }
      case MAP_SETTINGS_GUIDANCE_VIEW:
      {
         subHdl = subHdl - 1;
         if ((_infoStore.isSecondaryMapStreamingActive() == false)
               || ((_infoStore.isSecondaryMapStreamingActive() == true) && (subHdl == 1 /*LEFTSIDE_INFO_FULLMAP*/)))
         {
            changeLeftSideInfoSettings(subHdl);
         }
         else
         {
            ETG_TRACE_USR4(("MapSettingsListHandler::processComboBox:, Secondary map streaming is active, split map view is disabled"));
         }
         break;
      }
      default:
         break;
   }
   ::hmibase::widget::combobox::ComboBoxReqMsg* cbToggleMsg = COURIER_MESSAGE_NEW(::hmibase::widget::combobox::ComboBoxReqMsg)(0, ::hmibase::widget::combobox::enComboBoxAction::Close); //ComboBoxAction enComboBoxAction::Enum::ComboBoxClose:
   if (cbToggleMsg != NULL)
   {
      cbToggleMsg->Post();
   }

   ListRegistry::s_getInstance().updateList(_listId);
}


bool MapSettingsListHandler::onCourierMessage(const ApplyMapViewSettingsReqMsg& oMsg)
{
   ETG_TRACE_USR4(("MapSettingsListHandler::onCourierMessage(ApplyMapViewSettingsReqMsg)"));

   COURIER_UNUSED(oMsg);
   _navMiddleware.applySettings();

   return true;
}


void MapSettingsListHandler::changeDualMapViewSettings(Candera::UInt32 selectedRowId)
{
   ETG_TRACE_USR4(("MapSettingsListHandler::changeDualMapViewSettings(), Row Id : %d", selectedRowId));

   navmiddleware::settings::MapView selectedMapView = navmiddleware::settings::MAPVIEW_2D;
   _mapViewSettingOption = static_cast<enMapViewSettingOptions>(selectedRowId);

   switch (selectedRowId)
   {
      case MAP_SETTINGS_2D_MAP:
      {
         selectedMapView = navmiddleware::settings::MAPVIEW_2D;
         break;
      }
      case MAP_SETTINGS_2D_MAP_SPLIT:
      {
         selectedMapView = navmiddleware::settings::MAPVIEW_2D_SPLIT;
         break;
      }
      case MAP_SETTINGS_3D_MAP:
      {
         selectedMapView = navmiddleware::settings::MAPVIEW_3D;
         break;
      }
      case MAP_SETTINGS_3D_MAP_SPLIT:
      {
         selectedMapView = navmiddleware::settings::MAPVIEW_3D_SPLIT;
         break;
      }
      default:
      {
         //do nothing
      }
   }
   //(*getMapScreenData()).mEnableSplitMapWidgets = isSplitMapActive;
   //getMapScreenData().MarkItemModified(ItemKey::MapScreenData::EnableSplitMapWidgetsItem);
   //sendMapScreenData();
   navmiddleware::settings::ChangeMapViewSettings mapViewSettings = _navMiddleware.getChangeMapViewSettings();
   mapViewSettings.setMapView(selectedMapView);
   _navMiddleware.updateChangeMapViewSettings(mapViewSettings);
   ListRegistry::s_getInstance().updateList(LIST_ID_MAP_SETTINGS);
}


void MapSettingsListHandler::changeLeftSideInfoSettings(Candera::UInt32 selectedRowId)
{
   ETG_TRACE_USR4(("MapSettingsListHandler::changeLeftSideInfoSettings(), Row Id : %d", selectedRowId));

   navmiddleware::settings::MapHMISplitMode selectedMapHMISplitModeView = navmiddleware::settings::MAP_HMI_SPLITMODE_INTERSECTION;
   _guidanceViewSettingOption = static_cast<enGuidanceViewInfoSettingOptions>(selectedRowId);
   switch (selectedRowId)
   {
      case LEFTSIDE_INFO_INTERSECTION:
      {
         selectedMapHMISplitModeView = navmiddleware::settings::MAP_HMI_SPLITMODE_INTERSECTION;
         break;
      }
      case LEFTSIDE_INFO_FULLMAP:
      {
         selectedMapHMISplitModeView = navmiddleware::settings::MAP_HMI_SPLITMODE_FULL_MAP;
         POST_MSG((COURIER_MESSAGE_NEW(SetMapCameraModeReqMsg)(MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN)));
         break;
      }
      default:
      {
         break;
      }
   }

   navmiddleware::settings::MapViewSettings mapViewSettings = _navMiddleware.getMapViewSettings();
   mapViewSettings.setMapHMISplitMode(selectedMapHMISplitModeView);
   _navMiddleware.updateMapViewSettings(mapViewSettings);
   ListRegistry::s_getInstance().updateList(LIST_ID_MAP_SETTINGS);
}


bool MapSettingsListHandler::onCourierMessage(const UpdateMapSettingsListReqMsg& oMsg)
{
   ETG_TRACE_USR4(("MapSettingsListHandler::onCourierMessage(UpdateMapSettingsListReqMsg)"));

   ListRegistry::s_getInstance().updateList(LIST_ID_MAP_SETTINGS);
   return true;
}


void MapSettingsListHandler::onPropertyMapPresentationModeChanged()
{
   ETG_TRACE_USR4(("MapSettingsListHandler::onPropertyMapPresentationModeChanged()"));

   navmiddleware::settings::MapView currentMapView = _navMiddleware.getChangeMapViewSettings().getMapView();
   MapSettingsListItem& dualMapViewListElement = _mapSettingsList.at(MAP_SETTINGS_DUAL_MAP_VIEW);
   dualMapViewListElement._dropDownText =  DualMapViewDropDownListLabel[currentMapView];
   ListRegistry::s_getInstance().updateList(LIST_ID_MAP_SETTINGS);
}


#endif // HALL_TO_MDW_COM
