/**************************************************************************************
* @file         : GuidanceSettingsHandler.cpp
* @author       : ECG-Ramesh Kesavan
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2018 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "GuidanceSettingsHandler.h"
#include "Map/MapScreenDataUtils.h"
#include <vector>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/GuidanceSettingsHandler.cpp.trc.h"
#endif

#ifdef  HALL_TO_MDW_COM

const int GUIDANCE_VOLUME_STEP = 1;
const int GUIDANCE_VOLUME_MIN = 5;
const int GUIDANCE_VOLUME_MAX = 40;
const int CHECKBOX_ENABLED = 0;
const int CHECKBOX_DISABLED = 1;

IsButtonToggleEnabled getEnableGuidanceVoice(navmiddleware::NavMiddleware& navMiddleware)
{
   IsButtonToggleEnabled retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   if (navMiddleware.getGuidanceSettings().getVoiceGuidanceMode())
   {
      retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   }
   else
   {
      retValue = BUTTON_TOGGLE_ENABLED_FALSE;
   }
   ETG_TRACE_USR4(("GuidanceSettingsHandler::getEnableGuidanceVoice(%d)", retValue));

   return retValue;
}


IsButtonToggleEnabled getTrafficAnnounmentStatus(navmiddleware::NavMiddleware& navMiddleware)
{
   IsButtonToggleEnabled retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   if (navMiddleware.getGuidanceSettings().getTrafficAnnouncements())
   {
      retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   }
   else
   {
      retValue = BUTTON_TOGGLE_ENABLED_FALSE;
   }
   ETG_TRACE_USR4(("GuidanceSettingsHandler::getTrafficAnnounmentStatus(%d)", retValue));

   return retValue;
}


IsButtonToggleEnabled getTollBoothGuideStatus(navmiddleware::NavMiddleware& navMiddleware)
{
   IsButtonToggleEnabled retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   if (navMiddleware.getGuidanceSettings().getTollBoothInformation())
   {
      retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   }
   else
   {
      retValue = BUTTON_TOGGLE_ENABLED_FALSE;
   }
   ETG_TRACE_USR4(("GuidanceSettingsHandler::getTollBoothGuideStatus(%d)", retValue));

   return retValue;
}


IsButtonToggleEnabled getTrafficSignStatus(navmiddleware::NavMiddleware& navMiddleware)
{
   IsButtonToggleEnabled retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   if (navMiddleware.getGuidanceSettings().getTrafficSignInformation())
   {
      retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   }
   else
   {
      retValue = BUTTON_TOGGLE_ENABLED_FALSE;
   }
   ETG_TRACE_USR4(("GuidanceSettingsHandler::getTrafficSignStatus(%d)", retValue));

   return retValue;
}


IsButtonToggleEnabled getBorderCrossingStatus(navmiddleware::NavMiddleware& navMiddleware)
{
   IsButtonToggleEnabled retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   if (navMiddleware.getGuidanceSettings().getBorderCrossingInformation())
   {
      retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   }
   else
   {
      retValue = BUTTON_TOGGLE_ENABLED_FALSE;
   }
   ETG_TRACE_USR4(("GuidanceSettingsHandler::getBorderCrossingStatus(%d)", retValue));

   return retValue;
}


GuidanceSettingsHandler::GuidanceSettingsHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : NavMiddlewarePropertyUpdateConsumer(navMiddleware)
   , InfoStorePropertyUpdateConsumer(infoStore)
   , _currentGuidanceVolume(0)
{
}


GuidanceSettingsHandler::~GuidanceSettingsHandler()
{
}


void GuidanceSettingsHandler::initialize()
{
   _infoStore.registerDataPropertyUpdateCallback(*this);
   navmiddleware::settings::GuidanceSettings guidanceSettings = _navMiddleware.getGuidanceSettings();
   _currentGuidanceVolume = _infoStore.getGuidanceVolume();
   setMuteButtonStatus(_navMiddleware.getGuidanceSettings().getVoiceGuidanceMode());

   const Candera::UInt8 MAX_LISTSIZE = 7;
   _guidanceSettingList.reserve(MAX_LISTSIZE);
   _guidanceSettingList.push_back(GuidanceSettingListItem(LANGUAGE_STRING(TextId_0x1435, "Safety settings"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_BUTTON, SETTING_TRAFFIC_SAFETY_SETTING));
   _guidanceSettingList.push_back(GuidanceSettingListItem(LANGUAGE_STRING(TextId_0x1449, "Guidance volume"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_INC_DEC_VALUE, SETTING_GUIDANCE_VOLUME));
   _guidanceSettingList.push_back(GuidanceSettingListItem(LANGUAGE_STRING(TextId_0x144A, "Guidance voice"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, SETTING_GUIDANCE_VOICE, getEnableGuidanceVoice));
   _guidanceSettingList.push_back(GuidanceSettingListItem(LANGUAGE_STRING(TextId_0x1403, "Traffic announcement"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, SETTING_TRAFFIC_ANNOUNCEMENT, getTrafficAnnounmentStatus));
   _guidanceSettingList.push_back(GuidanceSettingListItem(LANGUAGE_STRING(TextId_0x141B, "Toll booth guide"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, SETTING_TOLL_BOTH_GUIDE, getTollBoothGuideStatus));
   _guidanceSettingList.push_back(GuidanceSettingListItem(LANGUAGE_STRING(TextId_0x148F, "Border crossing"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, SETTING_BORDER_CROSSING, getBorderCrossingStatus));
   _guidanceSettingList.push_back(GuidanceSettingListItem(LANGUAGE_STRING(TextId_0x1490, "Traffic sign"), IS_VISIBLE_TRUE, LIST_BUTTON_TYPE_TOGGLE, SETTING_TRAFFIC_ICON, getTrafficSignStatus));

   ListRegistry::s_getInstance().addListImplementation(LIST_ID_NAV_GUIDANCE_OPTION, this);
   _infoStore.registerDataPropertyUpdateCallback(*this);
   _navMiddleware.registerGuidancePropertyUpdateCallback(*this);
}


void GuidanceSettingsHandler::deinitialize()
{
   _infoStore.unregisterDataPropertyUpdateCallback(*this);
   _navMiddleware.unregisterGuidancePropertyUpdateCallback(*this);
}


tSharedPtrDataProvider GuidanceSettingsHandler::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   Candera::UInt32 listId = oMsg.GetListId();
   Candera::UInt32 startIndex = oMsg.GetStartIndex();
   Candera::UInt32 windowElementSize = oMsg.GetWindowElementSize();
   tSharedPtrDataProvider dataProvider;

   DataItemContext listItemName = "Layer_Checkbox";
   ListDataProviderBuilder listBuilder(LIST_ID_NAV_GUIDANCE_OPTION);
   bool isHardRestrictionPresent = _infoStore.getIsHardRestrictionPresent();

   if (listId == LIST_ID_NAV_GUIDANCE_OPTION)
   {
      ETG_TRACE_USR4(("GuidanceSettingsHandler::ListDateProviderReqMsg(ListID %d, startIndex %d, elementSize %d)", listId, startIndex, windowElementSize));
      for (std::vector<GuidanceSettingListItem>::iterator itemNumber = _guidanceSettingList.begin(); itemNumber != _guidanceSettingList.end(); ++itemNumber)
      {
         if (itemNumber->_listButtonType == LIST_BUTTON_TYPE_INC_DEC_VALUE)
         {
            listItemName = "Layer_GuidanceVolume";
            GuidanceSettingsData guidanceVolumeData;
            guidanceVolumeData.mCurrentGuidanceVolume = _currentGuidanceVolume;
            listBuilder.AddItem(
               itemNumber->_order,
               0UL, // - unused
               listItemName) // 0 - Name(Id) of the button
            .AddData(itemNumber->_text)//1
            .AddData(itemNumber->_isVisible)//2
            .AddId( // 3 - Name(Id) of the Decerement button
               itemNumber->_order,
               1UL)
            .AddId( // 4 - Name(Id) of the Increment button
               itemNumber->_order,
               2UL)
            .AddData(true)// 5 - button visibility
            .AddDataBindingUpdater<GuidanceSettingsDataBindingSource>(guidanceVolumeData);
         }
         else if (itemNumber->_listButtonType == LIST_BUTTON_TYPE_TOGGLE)
         {
            bool enableCheckbox = (isHardRestrictionPresent && (itemNumber->_order == SETTING_GUIDANCE_VOICE)) ? false : true;
            listItemName = "Layer_Checkbox";
            listBuilder.AddItem(
               itemNumber->_order,
               0UL,  //    - unused
               listItemName)  //0 - Name(Id) of the button
            .AddData(itemNumber->_text)
            .AddData((itemNumber->_isVisible == IS_VISIBLE_FALSE) ? true : itemNumber->getButtonToggleState(_navMiddleware))
            .AddData(enableCheckbox);
         }
         else if (itemNumber->_listButtonType == LIST_BUTTON_TYPE_BUTTON)
         {
            listItemName = "Layer_ListButton";
            listBuilder.AddItem(
               itemNumber->_order,
               0UL,  //    - unused
               listItemName)  //0 - Name(Id) of the button
            .AddData(itemNumber->_text)
            .AddData(itemNumber->_isVisible);
         }
      }
      dataProvider = listBuilder.CreateDataProvider();
   }
   return dataProvider;
}


bool GuidanceSettingsHandler::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   bool isMsgConsumed = false;
   Candera::UInt32 listId = oMsg.GetListId();
   unsigned int rowIdx = (unsigned int)oMsg.GetHdl();
   const enReaction  reaction = oMsg.GetReaction();
   bool isListUpdated = true;
   if (enRelease == reaction)
   {
      if (LIST_ID_NAV_GUIDANCE_OPTION == listId)
      {
         ETG_TRACE_USR4(("GuidanceSettingsHandler::ButtonListItemUpdMsg(ListID %d, rowId %d)", listId, rowIdx));
         isMsgConsumed = true;
         navmiddleware::settings::GuidanceSettings guidanceSettings = _navMiddleware.getGuidanceSettings();
         switch (rowIdx)
         {
            case SETTING_GUIDANCE_VOLUME:
            {
               const Candera::UInt8 decrementBtnID = 1;
               const Candera::UInt8 incrementBtnID = 2;
               unsigned int hdlCol = (unsigned int)oMsg.GetSubHdl();
               if (incrementBtnID == hdlCol)
               {
                  // TODO: Enable update Volume when notify opition of infostore is enabled
                  updateGuidanceVolume(true);
               }
               else if (decrementBtnID == hdlCol)
               {
                  // TODO: Enable update Volume when notify opition of infostore is enabled
                  updateGuidanceVolume(false);
               }
               break;
            }
            case SETTING_GUIDANCE_VOICE:
            {
               guidanceSettings.getVoiceGuidanceMode() ? guidanceSettings.setVoiceGuidanceMode(false) : guidanceSettings.setVoiceGuidanceMode(true);
               break;
            }
            case SETTING_TRAFFIC_SAFETY_SETTING:
            {
               break;
            }
            case SETTING_TRAFFIC_ANNOUNCEMENT:
            {
               guidanceSettings.getTrafficAnnouncements() ? guidanceSettings.setTrafficAnnouncements(false) : guidanceSettings.setTrafficAnnouncements(true);
               break;
            }
            case SETTING_TOLL_BOTH_GUIDE:
            {
               guidanceSettings.getTollBoothInformation() ? guidanceSettings.setTollBoothInformation(false) : guidanceSettings.setTollBoothInformation(true);
               break;
            }
            case SETTING_BORDER_CROSSING:
            {
               guidanceSettings.getBorderCrossingInformation() ? guidanceSettings.setBorderCrossingInformation(false) : guidanceSettings.setBorderCrossingInformation(true);
               break;
            }
            case SETTING_TRAFFIC_ICON:
            {
               guidanceSettings.getTrafficSignInformation() ? guidanceSettings.setTrafficSignInformation(false) : guidanceSettings.setTrafficSignInformation(true);
               break;
            }
            default:
            {
               isListUpdated = false;
               break;
            }
         }

         if (isListUpdated == true)
         {
            _navMiddleware.updateGuidanceSettings(guidanceSettings);
            ListRegistry::s_getInstance().updateList(LIST_ID_NAV_GUIDANCE_OPTION);
         }
      }
   }
   return isMsgConsumed;
}


bool GuidanceSettingsHandler::onPropertyUpdateGuidanceVolumeChanged()
{
   ETG_TRACE_USR4(("GuidanceSettingsHandler::onPropertyUpdateGuidanceVolumeChanged(%d)", _infoStore.getGuidanceVolume()));

   _currentGuidanceVolume = _infoStore.getGuidanceVolume();
   ListRegistry::s_getInstance().updateList(LIST_ID_NAV_GUIDANCE_OPTION);

   return true;
}


bool GuidanceSettingsHandler::onCourierMessage(const ApplyGuidanceSettingsReqMsg& oMsg)
{
   ETG_TRACE_USR4(("GuidanceSettingsHandler::onCourierMessage(ApplyGuidanceSettingsReqMsg)"));

   COURIER_UNUSED(oMsg);
   _navMiddleware.applySettings();

   return true;
}


void GuidanceSettingsHandler::updateGuidanceVolume(const bool incrementVolume)
{
   int newVolumeLevel = _currentGuidanceVolume;
   if (incrementVolume)
   {
      newVolumeLevel += GUIDANCE_VOLUME_STEP;
      if (newVolumeLevel > GUIDANCE_VOLUME_MAX)
      {
         newVolumeLevel = GUIDANCE_VOLUME_MAX;
      }
   }
   else
   {
      newVolumeLevel -= GUIDANCE_VOLUME_STEP;
      if (newVolumeLevel < GUIDANCE_VOLUME_MIN)
      {
         newVolumeLevel = GUIDANCE_VOLUME_MIN;
      }
   }
   ETG_TRACE_USR4(("GuidanceSettingsHandler::updateGuidanceVolume(%d , %d)", incrementVolume, newVolumeLevel));
   if (newVolumeLevel != _currentGuidanceVolume)
   {
      //Posting volume change to audio manager
      POST_MSG((COURIER_MESSAGE_NEW(SetGuidanceVolumeReqMsg)((Candera::UInt8)(newVolumeLevel))));

      // Inform middleware to playback the volume change.
      _navMiddleware.speakSamplePhrase(navmiddleware::SAMPLE_PHRASE__VOICE_OUTPUT);
   }
}


bool GuidanceSettingsHandler::onPropertyUpdateGuidanceSettingsChanged()
{
   ETG_TRACE_USR4(("GuidanceSettingsHandler::onPropertyUpdateGuidanceSettingsChanged()"));

   navmiddleware::settings::GuidanceSettings guidanceSettings = _navMiddleware.getGuidanceSettings();
   setMuteButtonStatus(_navMiddleware.getGuidanceSettings().getVoiceGuidanceMode());

   return true;
}


void GuidanceSettingsHandler::setMuteButtonStatus(bool isMute)
{
   ETG_TRACE_USR4(("GuidanceSettingsHandler::setMuteButtonStatus(), isMute : %d", isMute));
   DataBindingItem<MapScreenDataDataBindingSource>& mapScreenData = getMapScreenData();
   (*mapScreenData).mIsGuidanceVoiceEnabled = isMute;
   if (true == isMute)
   {
      (*mapScreenData).mMuteUnmuteText = LANGUAGE_STRING(TextId_0x0217, "Mute guidance");
      (*mapScreenData).mMuteUnmuteNormalIcon = ImageLoader::getAssetBitmapImage("AppHmi_NavigationModule#Images#N_Guidance#Icon_Mute_normal");
      (*mapScreenData).mMuteUnmutePressedIcon = ImageLoader::getAssetBitmapImage("AppHmi_NavigationModule#Images#N_Guidance#Icon_Mute_touched");
      (*mapScreenData).mMuteUnmuteDisabledIcon = ImageLoader::getAssetBitmapImage("AppHmi_NavigationModule#Images#N_Guidance#Icon_Mute_disabledBmp");
   }
   else
   {
      (*mapScreenData).mMuteUnmuteText = LANGUAGE_STRING(TextId_0x021E, "Unmute guidance");
      (*mapScreenData).mMuteUnmuteNormalIcon = ImageLoader::getAssetBitmapImage("AppHmi_NavigationModule#Images#N_Guidance#Icon_unMute_normalBmp");
      (*mapScreenData).mMuteUnmutePressedIcon = ImageLoader::getAssetBitmapImage("AppHmi_NavigationModule#Images#N_Guidance#Icon_unMute_touchedBmp");
      (*mapScreenData).mMuteUnmuteDisabledIcon = ImageLoader::getAssetBitmapImage("AppHmi_NavigationModule#Images#N_Guidance#Icon_unMute_disabledBmp");
   }
   mapScreenData.MarkItemModified(ItemKey::MapScreenData::IsGuidanceVoiceEnabledItem);
   mapScreenData.MarkItemModified(ItemKey::MapScreenData::MuteUnmuteTextItem);
   mapScreenData.MarkItemModified(ItemKey::MapScreenData::MuteUnmuteNormalIconItem);
   mapScreenData.MarkItemModified(ItemKey::MapScreenData::MuteUnmutePressedIconItem);
   mapScreenData.MarkItemModified(ItemKey::MapScreenData::MuteUnmuteDisabledIconItem);
   sendMapScreenData();
}


bool GuidanceSettingsHandler::onPropertyUpdateHardRestrictionStatusChanged()
{
   bool isHardRestrictionPresent = _infoStore.getIsHardRestrictionPresent();
   bool isVoiceGuidanceEnabled = _navMiddleware.getGuidanceSettings().getVoiceGuidanceMode();
   ETG_TRACE_USR4(("GuidanceSettingsHandler::onPropertyUpdateHardRestrictionStatusChanged(), isHardRestrictionPresent : %d, isVoiceGuidanceEnabled : %d", isHardRestrictionPresent, isVoiceGuidanceEnabled));

   if (isVoiceGuidanceEnabled)
   {
      _navMiddleware.muteGuidance(!isHardRestrictionPresent);
   }
   ListRegistry::s_getInstance().updateList(LIST_ID_NAV_GUIDANCE_OPTION);
   return true;
}


#endif // HALL_TO_MDW_COM
