/**************************************************************************************
* @file         : DetailedRoutePreferenceListHandler.cpp
* @author       : ECG-Ramesh Kesavan
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "DetailedRoutePreferenceListHandler.h"
#include "NaviSettingsUtil.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/DetailedRoutePreferenceListHandler.cpp.trc.h"
#endif

static const char* const DATA_CONTEXT_TEXT_LIST_ITEM = "Layer_TextListItem";
static const char* const DATA_CONTEXT_COMBOBOX_LIST_ITEMS_3 = "Combobox_ListItem3";
static const char* const DATA_CONTEXT_ROUTE_SETTINGS_TOGGLE_OPTION = "Layer_CheckboxOptions";

#ifdef  HALL_TO_MDW_COM

const int MAX_ITEMS_3 = 3;
const int MAX_CHK_BOX_ITEM = 9;

const char* EMPTY_STR = "";


Candera::String DetailedRoutePreferenceListHandler::getUseRealTimeTrafficInfo(navmiddleware::NavMiddleware& navMiddleware)
{
   Candera::String retValue = EMPTY_STR;

   switch (navMiddleware.getRouteOptionSettings().getUseRealTimeTrafficInfo())
   {
      case navmiddleware::settings::USE_REALTIME_TRAFFIC_AUTOMATIC:
      {
         retValue = UseTrafficInfoStrings[0];
         break;
      }
      case navmiddleware::settings::USE_REALTIME_TRAFFIC_PROMPT:
      {
         retValue = UseTrafficInfoStrings[1];
         break;
      }
      case navmiddleware::settings::USE_REALTIME_TRAFFIC_OFF:
      {
         retValue = UseTrafficInfoStrings[2];
         break;
      }
      default:
      {
         retValue = UseTrafficInfoStrings[0];
         break;
      }
   }
   _currentTrafficInformationOption = static_cast<enRoadInformationOptions>(navMiddleware.getRouteOptionSettings().getUseRealTimeTrafficInfo() + 1); //Adding 1 because HMI index starts from 1 and MIDW is from 0
   ETG_TRACE_USR4(("DetailedRoutePreferenceListHandler::getUseRealTimeTrafficInfo(%s)", retValue.GetCString()));

   return retValue;
}


Candera::String DetailedRoutePreferenceListHandler::getUseTollRoadInfo(navmiddleware::NavMiddleware& navMiddleware)
{
   Candera::String retValue = EMPTY_STR;

   switch (navMiddleware.getRouteOptionSettings().getUseTollRoads())
   {
      case navmiddleware::settings::USE_TOLL_ROADS_VALUE__ALWAYS:
      {
         retValue = UseTrafficInfoStrings[0];
         break;
      }
      case navmiddleware::settings::USE_TOLL_ROADS_VALUE__ASK:
      {
         retValue = UseTrafficInfoStrings[1];
         break;
      }
      case navmiddleware::settings::USE_TOLL_ROADS_VALUE__NEVER:
      {
         retValue = UseTrafficInfoStrings[2];
         break;
      }
      default:
      {
         retValue = UseTrafficInfoStrings[0];
         break;
      }
   }
   _currentUseTollRoadOption = static_cast<enRoadInformationOptions>(navMiddleware.getRouteOptionSettings().getUseTollRoads() + 1); //Adding 1 because HMI index starts from 1 and MIDW is from 0
   ETG_TRACE_USR4(("DetailedRoutePreferenceListHandler::getUseRealTimeTrafficInfo(%s)", retValue.GetCString()));

   return retValue;
}


DetailedRoutePreferenceListHandler::DetailedRoutePreferenceListHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : NavMiddlewarePropertyUpdateConsumer(navMiddleware)
   , _isDropdownEnabled(false)
   , _listId(LIST_ID_NAV_ROUTE_PREFERENCE_OTHERS)
   , _currentUseTrafficInfoStr(EMPTY_STR)
   , _currentTrafficInformationOption(OPTION_ALWAYS)
   , _currentUseTollRoadOption(OPTION_ALWAYS)
   , _currentUseTollRoadOptionStr(EMPTY_STR)
{
   // configuring static list item
   for (int idx = 0; idx < MAX_CHK_BOX_ITEM; idx++)
   {
      _isCheckboxEnabled[idx] = false;
   }

   ListRegistry::s_getInstance().addListImplementation(LIST_ID_NAV_ROUTE_PREFERENCE_OTHERS, this);
   resetSelectedOption();
}


DetailedRoutePreferenceListHandler::~DetailedRoutePreferenceListHandler()
{
}


/********************************************
/* NAME    : DetailedRoutePreferenceListHandler
/* SYSFL    : null,
****************************************************/
void DetailedRoutePreferenceListHandler::initialize()
{
}


/********************************************
/* NAME    : DetailedRoutePreferenceListHandler
/* SYSFL    : SYSFL-3191,
****************************************************/
void DetailedRoutePreferenceListHandler::deinitialize()
{
}


void DetailedRoutePreferenceListHandler::resetSelectedOption()
{
   memset(_useTrafficInfo, 0, MAX_SIZE_USE_TRAFFIC_INFO * sizeof(_useTrafficInfo[0]));
   memset(_useTollRoad, 0, MAX_SIZE_USE_TRAFFIC_INFO * sizeof(_useTollRoad[0]));
}


/********************************************
/* NAME    : DetailedRoutePreferenceListHandler
/* SYSFL    : null,
****************************************************/
tSharedPtrDataProvider DetailedRoutePreferenceListHandler::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   tSharedPtrDataProvider dataProvider;
   if (oMsg.GetListId() == LIST_ID_NAV_ROUTE_PREFERENCE_OTHERS)
   {
      ETG_TRACE_USR4(("DetailedRoutePreferenceListHandler::getListDataProvider(%d)", oMsg.GetListId()));
      ListDataProviderBuilder listBuilder(LIST_ID_NAV_ROUTE_PREFERENCE_OTHERS);
      getCurrentValues();
      resetSelectedOption();
      _useTrafficInfo[_currentTrafficInformationOption - 1] = true;
      _useTollRoad[_currentUseTollRoadOption - 1] = true;

      listBuilder.AddItem(0, 0, DATA_CONTEXT_COMBOBOX_LIST_ITEMS_3)
      .AddData(RouteSettingOthers_ListItemText[0])
      .AddId(0, 1).AddData(UseTrafficInfoStrings[0]).AddData(_useTrafficInfo[0])
      .AddId(0, 2).AddData(UseTrafficInfoStrings[1]).AddData(_useTrafficInfo[1])
      .AddId(0, 3).AddData(UseTrafficInfoStrings[2]).AddData(_useTrafficInfo[2])
      .AddData(_currentUseTrafficInfoStr);
      listBuilder.AddItem(1, 0, DATA_CONTEXT_COMBOBOX_LIST_ITEMS_3)
      .AddData(RouteSettingOthers_ListItemText[1])
      .AddId(1, 1).AddData(UseTrafficInfoStrings[0]).AddData(_useTollRoad[0])
      .AddId(1, 2).AddData(UseTrafficInfoStrings[1]).AddData(_useTollRoad[1])
      .AddId(1, 3).AddData(UseTrafficInfoStrings[2]).AddData(_useTollRoad[2])
      .AddData(_currentUseTollRoadOptionStr);

      const int INDEX_CORRECTION_FACTOR = 2;
      for (unsigned long index = 0; index < MAX_CHK_BOX_ITEM; ++index)
      {
         CheckBoxListItemData item;
         item.mListItemName = RouteSettingOthers_ListItemText[INDEX_CORRECTION_FACTOR + index];
         item.mToggleIconSwitchIndex = _isCheckboxEnabled[index];

         listBuilder.AddItem(
            (INDEX_CORRECTION_FACTOR + index),
            0UL,  //    - unused
            DATA_CONTEXT_ROUTE_SETTINGS_TOGGLE_OPTION)  //0 - Name(Id) of the button
         .AddData(RouteSettingOthers_ListItemText[INDEX_CORRECTION_FACTOR + index])
         .AddData(_isCheckboxEnabled[index])
         .AddData(true);
      }

      dataProvider = listBuilder.CreateDataProvider();
   }
   return dataProvider;
}


/********************************************
/* NAME    : DetailedRoutePreferenceListHandler
/* SYSFL    : SYSFL-3392, SYSFL-3436, SYSFL-3437,
****************************************************/
bool DetailedRoutePreferenceListHandler::onCourierMessage(const ChangeUseTrafficInfoReqMsg& oMsg)
{
   ETG_TRACE_USR4(("DetailedRoutePreferenceListHandler::onCourierMessage(ChangeUseTrafficInfoReqMsg"));
   unsigned int selectedOption = oMsg.GetUseTrafficInfoOption();

   if (selectedOption <= MAX_ITEMS_3)
   {
      // Subtract 1 to match the selection and enumeration index,
      // handle for first option in dropdown is 1. However, the enumeration value for same is 0
      _currentTrafficInformationOption = static_cast<enRoadInformationOptions>(selectedOption);
      _currentUseTrafficInfoStr = UseTrafficInfoStrings[selectedOption - 1U];
      setTrafficInformationOption();
   }

   ListRegistry::s_getInstance().updateList(LIST_ID_NAV_ROUTE_PREFERENCE_OTHERS);

   return true;
}


/********************************************
/* NAME    : DetailedRoutePreferenceListHandler
/* SYSFL    : SYSFL-3392, SYSFL-3436, SYSFL-3437,
****************************************************/

bool DetailedRoutePreferenceListHandler::onCourierMessage(const ChangeUseTollRoadReqMsg& oMsg)
{
   ETG_TRACE_USR4(("DetailedRoutePreferenceListHandler::onCourierMessage(ChangeUseTollRoadReqMsg"));
   unsigned int selectedOption = oMsg.GetUseTollRoadOption();

   if (selectedOption <= MAX_ITEMS_3)
   {
      // Subtract 1 to match the selection and enumeration index,
      // handle for first option in dropdown is 1. However, the enumeration value for same is 0
      _currentUseTollRoadOption = static_cast<enRoadInformationOptions>(selectedOption);
      _currentUseTollRoadOptionStr = UseTrafficInfoStrings[selectedOption - 1U];
      setUseTollRoadInformation();
   }

   ListRegistry::s_getInstance().updateList(LIST_ID_NAV_ROUTE_PREFERENCE_OTHERS);

   return true;
}


/********************************************
/* NAME    : DetailedRoutePreferenceListHandler
/* SYSFL    : null,
****************************************************/
bool DetailedRoutePreferenceListHandler::onCourierMessage(const ApplyDetailedRoutePreferenceOptionsReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR4(("DetailedRoutePreferenceListHandler::onCourierMessage(ApplyDetailedRoutePreferenceOptionsReqMsg)"));
   _navMiddleware.applySettings();

   return true;
}


/********************************************
/* NAME    : DetailedRoutePreferenceListHandler
/* SYSFL    : null,
****************************************************/
bool DetailedRoutePreferenceListHandler::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   bool msgProcessed = false;
   unsigned int listID = oMsg.GetListId();
   unsigned int rowIdx = (unsigned int)oMsg.GetHdl();

   if (oMsg.GetReaction() == enRelease)
   {
      if (LIST_ID_NAV_ROUTE_PREFERENCE_OTHERS == listID)
      {
         ETG_TRACE_USR4(("DetailedRoutePreferenceListHandler::onCourierMessage(ButtonListItemUpdMsg), List Id : %d, Row Id: %d ", listID, rowIdx));
         bool listUpdateNeeded = false;
         navmiddleware::settings::RouteOptionsSettings currentRouteOptionSettings = _navMiddleware.getRouteOptionSettings();

         switch (rowIdx)
         {
            case USE_REAL_TIME_TRAFFIC_INFORMATION:
            case USE_TOLL_ROADS:
            {
               ::hmibase::widget::combobox::ComboBoxReqMsg* cbToggleMsg = COURIER_MESSAGE_NEW(::hmibase::widget::combobox::ComboBoxReqMsg)(0, ::hmibase::widget::combobox::enComboBoxAction::Close); //ComboBoxAction enComboBoxAction::Enum::ComboBoxClose:
               if (cbToggleMsg != NULL)
               {
                  cbToggleMsg->Post();
               }
               break;
            }
            case MINIMIZE_FERRIES:
            {
               (navmiddleware::settings::USE_FERRIES_ON == currentRouteOptionSettings.getUseFerries()) ? currentRouteOptionSettings.setUseFerries(navmiddleware::settings::USE_FERRIES_OFF)
               : currentRouteOptionSettings.setUseFerries(navmiddleware::settings::USE_FERRIES_ON);
               listUpdateNeeded = true;
               break;
            }
            case MINIMIZE_TRAIN_TRANSPORT:
            {
               (navmiddleware::settings::USE_CAR_SHUTTLE_TRAIN_ON == currentRouteOptionSettings.getUseCarShuttleTrain()) ? currentRouteOptionSettings.setUseCarShuttleTrain(navmiddleware::settings::USE_CAR_SHUTTLE_TRAIN_OFF)
               : currentRouteOptionSettings.setUseCarShuttleTrain(navmiddleware::settings::USE_CAR_SHUTTLE_TRAIN_ON);
               listUpdateNeeded = true;
               break;
            }
            case MINIMIZE_SUBSCRIPTION_ROADS:
            {
               (navmiddleware::settings::USE_SUBSCRIPTION_ROADS_ON == currentRouteOptionSettings.getUseSubscriptionRoads()) ? currentRouteOptionSettings.setUseSubscriptionRoads(navmiddleware::settings::USE_SUBSCRIPTION_ROADS_OFF)
               : currentRouteOptionSettings.setUseSubscriptionRoads(navmiddleware::settings::USE_SUBSCRIPTION_ROADS_ON);
               listUpdateNeeded = true;
               break;
            }
            case MINIMIZE_UNVERIFIED_ROADS:
            {
               (navmiddleware::settings::USE_IPD_ON == currentRouteOptionSettings.getUseIPD()) ? currentRouteOptionSettings.setUseIPD(navmiddleware::settings::USE_IPD_OFF)
               : currentRouteOptionSettings.setUseIPD(navmiddleware::settings::USE_IPD_ON);
               listUpdateNeeded = true;
               break;
            }
            case MINIMIZE_UNPAVED_ROADS:
            {
               (navmiddleware::settings::USE_UNPAVED_ROADS_ON == currentRouteOptionSettings.getUseUnpavedRoads()) ? currentRouteOptionSettings.setUseUnpavedRoads(navmiddleware::settings::USE_UNPAVED_ROADS_OFF)
               : currentRouteOptionSettings.setUseUnpavedRoads(navmiddleware::settings::USE_UNPAVED_ROADS_ON);
               listUpdateNeeded = true;
               break;
            }
            case MINIMIZE_HOV_ROADS:
            {
               (navmiddleware::settings::USE_HOV_LANES_ON == currentRouteOptionSettings.isHOVLaneGuidanceActive()) ? currentRouteOptionSettings.setHOVLaneGuidanceActive(navmiddleware::settings::USE_HOV_LANES_OFF)
               : currentRouteOptionSettings.setHOVLaneGuidanceActive(navmiddleware::settings::USE_HOV_LANES_ON);
               listUpdateNeeded = true;
               break;
            }
            case AVOID_BORDER_CROSSING:
            {
               (navmiddleware::settings::USE_BORDER_CROSSING_ON == currentRouteOptionSettings.getUseBorderCrossings()) ? currentRouteOptionSettings.setUseBorderCrossings(navmiddleware::settings::USE_BORDER_CROSSING_OFF)
               : currentRouteOptionSettings.setUseBorderCrossings(navmiddleware::settings::USE_BORDER_CROSSING_ON);
               listUpdateNeeded = true;
               break;
            }
            case AVOID_U_TURNS:
            {
               (navmiddleware::settings::USE_UTURNS_ON == currentRouteOptionSettings.getUseUTurns()) ? currentRouteOptionSettings.setUseUTurns(navmiddleware::settings::USE_UTURNS_OFF)
               : currentRouteOptionSettings.setUseUTurns(navmiddleware::settings::USE_UTURNS_ON);
               listUpdateNeeded = true;
               break;
            }
            case AVOID_TUNNELS:
            {
               (navmiddleware::settings::USE_TUNNELS_ON == currentRouteOptionSettings.getUseTunnels()) ? currentRouteOptionSettings.setUseTunnels(navmiddleware::settings::USE_TUNNELS_OFF)
               : currentRouteOptionSettings.setUseTunnels(navmiddleware::settings::USE_TUNNELS_ON);
               listUpdateNeeded = true;
               break;
            }
            default:
            {
               break;
            }
         }

         if (listUpdateNeeded)
         {
            _navMiddleware.updateRouteOptionSettings(currentRouteOptionSettings);
            ListRegistry::s_getInstance().updateList(LIST_ID_NAV_ROUTE_PREFERENCE_OTHERS);
         }
         msgProcessed = true;
      }// end of if (lisdID)
   } // end of if  (Button Reaction)

   return msgProcessed;
}


/********************************************
/* NAME    : DetailedRoutePreferenceListHandler
/* SYSFL    : null,
****************************************************/
void DetailedRoutePreferenceListHandler::getCurrentValues()
{
   _currentUseTrafficInfoStr = getUseRealTimeTrafficInfo(_navMiddleware);
   _currentUseTollRoadOptionStr = getUseTollRoadInfo(_navMiddleware);
   _isCheckboxEnabled[0] = getMinimizeFerriesStatus();
   _isCheckboxEnabled[1] = getMinimizeTrainTransportStatus();
   _isCheckboxEnabled[2] = getMinimizeSubscriptionRoadStatus();
   _isCheckboxEnabled[3] = getMinimizeUnverifiedRoadStatus();
   _isCheckboxEnabled[4] = getMinimizeUnpavedRoadsStatus();
   _isCheckboxEnabled[5] = getMinimizeHOVLanesStatus();
   _isCheckboxEnabled[6] = getAvoidBorderCrossingStatus();
   _isCheckboxEnabled[7] = getAvoidUTurnStatus();
   _isCheckboxEnabled[8] = getAvoidTunnelStatus();
}


/********************************************
/* NAME    : DetailedRoutePreferenceListHandler
/* SYSFL    : SYSFL-3392, SYSFL-3436, SYSFL-3437,
****************************************************/
void DetailedRoutePreferenceListHandler::setTrafficInformationOption()
{
   navmiddleware::settings::RouteOptionsSettings routeOptionsSettings = _navMiddleware.getRouteOptionSettings();
   ETG_TRACE_USR4(("DetailedRoutePreferenceListHandler::setTrafficInformationOption()"));

   switch (_currentTrafficInformationOption)
   {
      case OPTION_ALWAYS:
      {
         routeOptionsSettings.setUseRealTimeTrafficInfo(navmiddleware::settings::USE_REALTIME_TRAFFIC_AUTOMATIC);
         break;
      }
      case OPTION_PROMPT:
      {
         // to check whether USE_REALTIME_TRAFFIC_PROMPT is same as Accept
         routeOptionsSettings.setUseRealTimeTrafficInfo(navmiddleware::settings::USE_REALTIME_TRAFFIC_PROMPT);
         break;
      }
      case OPTION_NEVER:
      {
         routeOptionsSettings.setUseRealTimeTrafficInfo(navmiddleware::settings::USE_REALTIME_TRAFFIC_OFF);
         break;
      }
      default:
      {
         ETG_TRACE_ERR(("DetailedRoutePreferenceListHandler::setTrafficInformationOption()  wrong index"));
         break;
      }
   }

   _navMiddleware.updateRouteOptionSettings(routeOptionsSettings);
}


/********************************************
/* NAME    : DetailedRoutePreferenceListHandler
/* SYSFL    : null,
****************************************************/
void DetailedRoutePreferenceListHandler::setUseTollRoadInformation()
{
   navmiddleware::settings::RouteOptionsSettings routeOptionsSettings = _navMiddleware.getRouteOptionSettings();
   ETG_TRACE_USR4(("DetailedRoutePreferenceListHandler::setUseTollRoadInformation(), Use toll : %d", _currentUseTollRoadOption));

   switch (_currentUseTollRoadOption)
   {
      case OPTION_ALWAYS:
      {
         routeOptionsSettings.setUseTollRoads(navmiddleware::settings::USE_TOLL_ROADS_VALUE__ALWAYS);
         break;
      }
      case OPTION_PROMPT:
      {
         // to check whether USE_REALTIME_TRAFFIC_PROMPT is same as Accept
         routeOptionsSettings.setUseTollRoads(navmiddleware::settings::USE_TOLL_ROADS_VALUE__ASK);
         break;
      }
      case OPTION_NEVER:
      {
         routeOptionsSettings.setUseTollRoads(navmiddleware::settings::USE_TOLL_ROADS_VALUE__NEVER);
         break;
      }
      default:
      {
         ETG_TRACE_ERR(("DetailedRoutePreferenceListHandler::setTrafficInformationOption()  wrong index"));
         break;
      }
   }

   _navMiddleware.updateRouteOptionSettings(routeOptionsSettings);
}


/********************************************
/* NAME    : DetailedRoutePreferenceListHandler
/* SYSFL    : null,
****************************************************/
IsButtonToggleEnabled DetailedRoutePreferenceListHandler::getMinimizeUnpavedRoadsStatus()
{
   IsButtonToggleEnabled retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   if (navmiddleware::settings::USE_UNPAVED_ROADS_ON == _navMiddleware.getRouteOptionSettings().getUseUnpavedRoads())
   {
      retValue = BUTTON_TOGGLE_ENABLED_FALSE;
   }
   else
   {
      retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   }
   ETG_TRACE_USR4(("DetailedRoutePreferenceListHandler::getMinimizeUnpavedRoadsStatus(%d)", retValue));

   return retValue;
}


/********************************************
/* NAME    : DetailedRoutePreferenceListHandler
/* SYSFL    : null,
****************************************************/
IsButtonToggleEnabled DetailedRoutePreferenceListHandler::getMinimizeHOVLanesStatus()
{
   IsButtonToggleEnabled retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   if (navmiddleware::settings::USE_HOV_LANES_ON == _navMiddleware.getRouteOptionSettings().isHOVLaneGuidanceActive())
   {
      retValue = BUTTON_TOGGLE_ENABLED_FALSE;
   }
   else
   {
      retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   }
   ETG_TRACE_USR4(("DetailedRoutePreferenceListHandler::getMinimizeHOVLanesStatus(%d)", retValue));

   return retValue;
}


/********************************************
/* NAME    : DetailedRoutePreferenceListHandler
/* SYSFL    : null,
****************************************************/
IsButtonToggleEnabled DetailedRoutePreferenceListHandler::getMinimizeFerriesStatus()
{
   IsButtonToggleEnabled retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   if (navmiddleware::settings::USE_FERRIES_ON == _navMiddleware.getRouteOptionSettings().getUseFerries())
   {
      retValue = BUTTON_TOGGLE_ENABLED_FALSE;
   }
   else
   {
      retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   }
   ETG_TRACE_USR4(("DetailedRoutePreferenceListHandler::getMinimizeFerriesStatus(%d)", retValue));

   return retValue;
}


/********************************************
/* NAME    : DetailedRoutePreferenceListHandler
/* SYSFL    : null,
****************************************************/
IsButtonToggleEnabled DetailedRoutePreferenceListHandler::getMinimizeTrainTransportStatus()
{
   IsButtonToggleEnabled retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   if (navmiddleware::settings::USE_CAR_SHUTTLE_TRAIN_ON == _navMiddleware.getRouteOptionSettings().getUseCarShuttleTrain())
   {
      retValue = BUTTON_TOGGLE_ENABLED_FALSE;
   }
   else
   {
      retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   }
   ETG_TRACE_USR4(("DetailedRoutePreferenceListHandler::getMinimizeTrainTransportStatus(%d)", retValue));

   return retValue;
}


/********************************************
/* NAME    : DetailedRoutePreferenceListHandler
/* SYSFL    : null,
****************************************************/
IsButtonToggleEnabled DetailedRoutePreferenceListHandler::getMinimizeSubscriptionRoadStatus()
{
   IsButtonToggleEnabled retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   if (navmiddleware::settings::USE_SUBSCRIPTION_ROADS_ON == _navMiddleware.getRouteOptionSettings().getUseSubscriptionRoads())
   {
      retValue = BUTTON_TOGGLE_ENABLED_FALSE;
   }
   else
   {
      retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   }
   ETG_TRACE_USR4(("DetailedRoutePreferenceListHandler::getMinimizeSubscriptionRoadStatus(%d)", retValue));

   return retValue;
}


/********************************************
/* NAME    : DetailedRoutePreferenceListHandler
/* SYSFL    : null,
****************************************************/
IsButtonToggleEnabled DetailedRoutePreferenceListHandler::getMinimizeUnverifiedRoadStatus()
{
   IsButtonToggleEnabled retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   if (navmiddleware::settings::USE_IPD_ON == _navMiddleware.getRouteOptionSettings().getUseIPD())
   {
      retValue = BUTTON_TOGGLE_ENABLED_FALSE;
   }
   else
   {
      retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   }
   ETG_TRACE_USR4(("DetailedRoutePreferenceListHandler::getMinimizeUnverifiedRoadStatus(%d)", retValue));

   return retValue;
}


/********************************************
/* NAME    : DetailedRoutePreferenceListHandler
/* SYSFL    : null,
****************************************************/
IsButtonToggleEnabled DetailedRoutePreferenceListHandler::getAvoidBorderCrossingStatus()
{
   IsButtonToggleEnabled retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   if (navmiddleware::settings::USE_BORDER_CROSSING_ON == _navMiddleware.getRouteOptionSettings().getUseBorderCrossings())
   {
      retValue = BUTTON_TOGGLE_ENABLED_FALSE;
   }
   else
   {
      retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   }
   ETG_TRACE_USR4(("DetailedRoutePreferenceListHandler::getAvoidBorderCrossingStatus(%d)", retValue));

   return retValue;
}


/********************************************
/* NAME    : DetailedRoutePreferenceListHandler
/* SYSFL    : null,
****************************************************/
IsButtonToggleEnabled DetailedRoutePreferenceListHandler::getAvoidUTurnStatus()
{
   IsButtonToggleEnabled retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   if (navmiddleware::settings::USE_UTURNS_ON == _navMiddleware.getRouteOptionSettings().getUseUTurns())
   {
      retValue = BUTTON_TOGGLE_ENABLED_FALSE;
   }
   else
   {
      retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   }
   ETG_TRACE_USR4(("DetailedRoutePreferenceListHandler::getAvoidUTurnsStatus(), UseUTurns Value : %d", _navMiddleware.getRouteOptionSettings().getUseUTurns()));

   return retValue;
}


/********************************************
/* NAME    : DetailedRoutePreferenceListHandler
/* SYSFL    : null,
****************************************************/
IsButtonToggleEnabled DetailedRoutePreferenceListHandler::getAvoidTunnelStatus()
{
   IsButtonToggleEnabled retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   if (navmiddleware::settings::USE_TUNNELS_ON == _navMiddleware.getRouteOptionSettings().getUseTunnels())
   {
      retValue = BUTTON_TOGGLE_ENABLED_FALSE;
   }
   else
   {
      retValue = BUTTON_TOGGLE_ENABLED_TRUE;
   }
   ETG_TRACE_USR4(("DetailedRoutePreferenceListHandler::getAvoidTunnelStatus(), UseTunnels Value : %d", _navMiddleware.getRouteOptionSettings().getUseTunnels()));

   return retValue;
}


#endif // HALL_TO_MDW_COM
