/**************************************************************************************
* @file         : WaypointListHandler.h
* @author       : KanagaDurga Balakrishnan
* @addtogroup   :
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "util/StrUtf8.h"
#include "AppHmi_NavigationStateMachine.h"
#include "WaypointListHandler.h"
#include "hmi_trace_if.h"
#include "CgiExtensions/ImageLoader.h"
#include "Common/Util/StringUtils.h"
#include "RouteDataUtils.h"
#include "vector"
#include "algorithm"
#include "iostream"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/WaypointListHandler.cpp.trc.h"
#endif

static const char* const WAYPOINT_LIST_ITEM = "WaypointListItem";
static const int WAYPOINT_DELETEALL_POPUP = 8;
static const int WAYPOINT_DELETE_POPUP = 7;


using namespace navmiddleware;

WaypointListHandler::WaypointListHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _isWaypointListShown(false)
   , _listId(0)
   , _startIndex(0)
   , _listSize(0)
{
}


WaypointListHandler::~WaypointListHandler()
{
}


void WaypointListHandler::initialize()
{
   _navMiddleware.registerRoutePropertyUpdateCallback(*this);
   _navMiddleware.registerGuidancePropertyUpdateCallback(*this);
}


void WaypointListHandler::deinitialize()
{
   _navMiddleware.unregisterRoutePropertyUpdateCallback(*this);
   _navMiddleware.unregisterGuidancePropertyUpdateCallback(*this);
}


bool WaypointListHandler::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   bool isMsgConsumed = false;

   if (oMsg.GetListId() == LIST_ID_WAYPOINTS)
   {
      ETG_TRACE_USR4(("WaypointListHandler::onCourierMessage(ListDateProviderReqMsg)"));

      _listId = oMsg.GetListId();
      _startIndex = oMsg.GetStartIndex();
      _navMiddleware.requestWaypointList(navmiddleware::WAYPOINT_LIST_MODE__VIEW);
      isMsgConsumed = true;
   }

   return isMsgConsumed;
}


bool WaypointListHandler::onCourierMessage(const ShowWaypointListViewReqMsg& oMsg)
{
   _isWaypointListShown = oMsg.GetIsListShown();
   ETG_TRACE_USR4(("WaypointListHandler::onCourierMessage(ShowWaypointListViewReqMsg), Is waypoint list shown : %d", _isWaypointListShown));

   return true;
}


bool WaypointListHandler::onCourierMessage(const WaypointInsertReqMsg& oMsg)
{
   ETG_TRACE_USR4(("WaypointListHandler::onCourierMessage(WaypointInsertReqMsg), List size : %d", _listSize));

   if (_listSize > 0)
   {
      unsigned int newWayPointIndex = _listSize - 1;   // insert before destination
      _navMiddleware.insertWaypoint(newWayPointIndex, navmiddleware::WAYPOINT_EDIT_MODE__WITH_RECALCULATION);
   }

   return true;
}


void WaypointListHandler::onPropertyUpdateRouteWaypointInfoChanged()
{
   unsigned int newListSize = _navMiddleware.getWaypointInfos().getWaypointInfos().size();
   ETG_TRACE_USR4(("WaypointListHandler::onPropertyUpdateRouteWaypointInfoChanged(), Waypoint list size (including destination) : %d", newListSize));

   // This condition is met when waypoint is inserted or deleted
   if ((0 != _listSize) && (1 == abs(newListSize - _listSize)))
   {
      _infoStore.setIsWaypointListModified(true);
      ETG_TRACE_USR4(("WaypointListHandler::onPropertyUpdateRouteWaypointInfoChanged(), Waypoint inserted/deleted"));
   }
   if (newListSize > 0)
   {
      _infoStore.setWaypointListSize(newListSize - 1);   // Waypoint list size maintained in infostore does not include destination
   }
   _listSize = newListSize;

   if (true == _isWaypointListShown)
   {
      POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(getListDataProvider())));
   }

   //Check for skip waypoint popup when waypoints list update is received.
   checkForSkipWaypoints(_navMiddleware.getWaypointInfos().isSkipCurrentWaypoint(), _navMiddleware.getWaypointInfos().getCurrentDestinationIndex());
}


void WaypointListHandler::onPropertyUpdateArrivalInfosChanged()
{
   // Fix for RTC bug 863711.
   const ArrivalInfos& arrivalInfos = _navMiddleware.getArrivalInfos();
   bool isCalcComplete = arrivalInfos.isRouteCalcCompletedForAllDestinations();
   bool isWaypointListModified = _infoStore.getIsWaypointListModified();
   ETG_TRACE_USR4(("WaypointListHandler::onPropertyUpdateArrivalInfosChanged(), isRouteCalcCompletedForAllDestinations: %d,  isWaypointListModified : %d", isCalcComplete, isWaypointListModified));

   if ((true == isCalcComplete) && (true == isWaypointListModified))
   {
      _infoStore.setIsWaypointListModified(false);
      _navMiddleware.startGuidance();
   }
}


void WaypointListHandler::onPropertyUpdateGuidanceStateChanged()
{
   bool guidanceState = _navMiddleware.isGuidanceActive();
   ETG_TRACE_USR4(("WaypointListHandler::onPropertyUpdateGuidanceStateChanged(), Guidance Active: %d", guidanceState));

   if (true == guidanceState)
   {
      if (0 == _listSize)
      {
         // To update the waypoint list with destination entry details when guidance is started
         _navMiddleware.requestWaypointList(navmiddleware::WAYPOINT_LIST_MODE__VIEW);
      }
   }
   else
   {
      _infoStore.setIsWaypointListModified(false);
      _infoStore.setWaypointIndex(0);
      _infoStore.setWaypointListIdx(0);
      _infoStore.setWaypointListSize(0);
      _listSize = 0;
   }
}


void WaypointListHandler::checkForSkipWaypoints(bool isSkipWaypoints, unsigned int index)
{
   ETG_TRACE_USR4(("WaypointListHandler::checkForSkipWaypoints(), isSkipWaypoints : %d, index : %d", isSkipWaypoints, index));

   if (isSkipWaypoints)
   {
      POST_MSG((COURIER_MESSAGE_NEW(UpdateDeletePopupTextReqMsg)(POPUP_SKIP_WAYPOINT)));
   }
   else
   {
      //do nothing
   }
}


bool WaypointListHandler::onCourierMessage(const SkipWaypointReqMsg& oMsg)
{
   ETG_TRACE_USR4(("WaypointListHandler::onCourierMessage(SkipWaypointReqMsg)"));

   _navMiddleware.skipWaypoint(oMsg.GetSkipWaypoint(), _navMiddleware.getWaypointInfos().getCurrentDestinationIndex());

   return true;
}


tSharedPtrDataProvider WaypointListHandler::getListDataProvider()
{
   ETG_TRACE_USR4(("WaypointListHandler::getListDataProvider()"));
   ListDataProviderBuilder listBuilder(_listId);
   WaypointListInfoData item;

   const navmiddleware::WaypointInfos& wayPointInfos = _navMiddleware.getWaypointInfos();
   const std::vector< navmiddleware::WaypointInfos::WaypointInfo >& waypointInfoList = wayPointInfos.getWaypointInfos();
   unsigned int actDestListSize = waypointInfoList.size();
   _infoStore.setWaypointIndex(waypointInfoList.size() - 1);

   unsigned int idx = wayPointInfos.getCurrentDestinationIndex();
   static const char* const waypointNormalIconName[11] =
   {
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoint1_1_normalBmp",
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoint2_1_normalBmp",
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoint3_1_normalBmp",
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoint4_1_normalBmp",
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoint5_1_normalBmp" ,
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoints_6_normal",
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoints_7_normal",
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoints_8_normal",
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoints_9_normal",
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoints_10_normal",
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoints_11_normal",
   };

   static const char* const waypointTouchedIconName[11] =
   {
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoint1_1_touchedBmp",
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoint2_1_touchedBmp",
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoint3_1_touchedBmp",
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoint4_1_touchedBmp",
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoint5_1_touchedBmp",
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoints_6_touched",
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoints_7_touched",
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoints_8_touched",
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoints_9_touched",
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoints_10_touched",
      "AppHmi_NavigationModule#Images#N_Waypoint#Icon_Waypoints_11_touched",
   };

   if (LIST_ID_WAYPOINTS == _listId)
   {
      for (::std::vector<navmiddleware::WaypointInfos::WaypointInfo>::const_iterator editRouteListItem = waypointInfoList.begin() + wayPointInfos.getCurrentDestinationIndex();
            editRouteListItem != waypointInfoList.end(); editRouteListItem++)
      {
         item.mWaypointListItem = editRouteListItem->m_locationString.c_str();
         ETG_TRACE_USR4(("WaypointListHandler::m_locationString:%s", editRouteListItem->m_locationString.c_str()));
         item.mWaypointNormalIcon = ImageLoader::getAssetBitmapImage(waypointNormalIconName[idx]);
         item.mWaypointPressedIcon = ImageLoader::getAssetBitmapImage(waypointTouchedIconName[idx]);
         if (idx == actDestListSize - 1)
         {
            item.mWaypointNormalIcon = ImageLoader::getAssetBitmapImage("AppHmi_NavigationModule#Images#N_Waypoint#Icon_Destination_3_normalBmp");
            item.mWaypointPressedIcon = ImageLoader::getAssetBitmapImage("AppHmi_NavigationModule#Images#N_Waypoint#Icon_Destination_3_touched");
         }

         listBuilder.AddItem(idx,
                             0UL,
                             WAYPOINT_LIST_ITEM).AddDataBindingUpdater<WaypointListInfoDataBindingSource>(item);
         idx++;
      }
   }

   return listBuilder.CreateDataProvider(_startIndex, actDestListSize - wayPointInfos.getCurrentDestinationIndex());
}


bool WaypointListHandler::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   bool isMessageProcessed = false;
   ListProviderEventInfo info;

   if (oMsg.GetEnReaction() == enRelease)
   {
      if (ListProviderEventInfo::GetItemIdentifierInfo(oMsg.GetSender(), info, LIST_ID_WAYPOINTS))
      {
         unsigned int listIdx = (unsigned int)info.getHdlRow();
         listIdx = listIdx - _startIndex;
         _infoStore.setWaypointListIdx(listIdx);
         _infoStore.setIsDetailInfoRequested(true, InfoStoreBase::DETAIL_INFO_WAYPOINT);
         // we are handling this request
         ETG_TRACE_USR4(("WaypointListHandler::onCourierMessage(ButtonReactionMsg(listID %d, row %d, column %d))", info.getListId(), info.getHdlRow(), info.getHdlCol()));
         const navmiddleware::WaypointInfos& wayPointInfos = _navMiddleware.getWaypointInfos();
         const std::vector< navmiddleware::WaypointInfos::WaypointInfo >& waypointInfoList = wayPointInfos.getWaypointInfos();
         navmiddleware::WaypointInfos::WaypointInfo editRouteListItem = waypointInfoList.at(listIdx);
         navmiddleware::WaypointInfos::WaypointInfo destinationItem = waypointInfoList.back();

         if (listIdx < waypointInfoList.size())
         {
            std::vector<PosWGS84<double> > positionVector;
            positionVector.push_back(PosWGS84<double>(editRouteListItem.m_coordinate.getLongitude(), editRouteListItem.m_coordinate.getLatitude()));

            _infoStore.setLatitude(editRouteListItem.m_coordinate.getLatitude());
            _infoStore.setLongitude(editRouteListItem.m_coordinate.getLongitude());

            InfoStoreBase::CoordinatesToBeShownInMap& coordinatesToBeShownInMap = _infoStore.getCoordinatesToBeShownInMap();
            coordinatesToBeShownInMap._coordinates = positionVector;

            // Get the address details from the middleware by providing the latitude and longitude co ordinate
            navmiddleware::GeoCoordinateDegree geocoordinates(editRouteListItem.m_coordinate.getLatitude(), editRouteListItem.m_coordinate.getLongitude());

            _navMiddleware.setLocationWithWaypointIndex(listIdx);
            _infoStore.setIsDetailInfoRequested(true, InfoStoreBase::DETAIL_INFO_WAYPOINT);
            _navMiddleware.requestLocationAttributes();
         }

         //To check Whether waypoint destination has been selected
         if (destinationItem == editRouteListItem)
         {
            EXT_bIsWaypointDestinationIndex = 1;
         }
         else
         {
            EXT_bIsWaypointDestinationIndex = 0;
         }
      }
   }

   return isMessageProcessed;
}


bool WaypointListHandler::onCourierMessage(const WaypointDeleteReqMsg& oMsg)
{
   ETG_TRACE_USR4(("WaypointListHandler::onCourierMessage(WaypointDeleteReqMsg)"));

   bool isMessageProcessed = true;
   int activeWaypointIndex = _infoStore.getWaypointListIdx();

   if (oMsg.GetPopupType() == WAYPOINT_DELETEALL_POPUP)  // delete all waypoint popup
   {
      std::vector<unsigned int> waypointsToBeDeleted;
      waypointsToBeDeleted.push_back(_navMiddleware.getWaypointInfos().getWaypointInfos().size() - 1); // destination index for deleting all waypoints
      _navMiddleware.deleteWaypoints(waypointsToBeDeleted, navmiddleware::WAYPOINT_EDIT_MODE__WITH_RECALCULATION);
      _infoStore.setWaypointIndex(0);
      _infoStore.setWaypointListSize(0);
   }
   else if ((oMsg.GetPopupType() == WAYPOINT_DELETE_POPUP) && (EXT_bIsWaypointDestinationIndex == 0)) //delete waypoint popup
   {
      std::vector<unsigned int> waypointsToBeDeleted;
      waypointsToBeDeleted.push_back(activeWaypointIndex);
      _navMiddleware.deleteWaypoints(waypointsToBeDeleted, navmiddleware::WAYPOINT_EDIT_MODE__WITH_RECALCULATION);
      _infoStore.setWaypointIndex(_infoStore.getWaypointIndex() - 1);
   }
   else if ((oMsg.GetPopupType() == WAYPOINT_DELETE_POPUP) && (EXT_bIsWaypointDestinationIndex == 1))
   {
      std::vector<unsigned int> waypointsToBeDeleted;
      waypointsToBeDeleted.push_back(activeWaypointIndex);
      _navMiddleware.deleteWaypoints(waypointsToBeDeleted, navmiddleware::WAYPOINT_EDIT_MODE__WITH_RECALCULATION);
      _infoStore.setWaypointIndex(0);
      _infoStore.setWaypointListSize(0);
   }
   else
   {
      //Do nothing
   }

   return isMessageProcessed;
}


bool WaypointListHandler::onCourierMessage(const ShowPopupTextMsg& oMsg)
{
   ETG_TRACE_USR4(("WaypointListHandler::onCourierMessage(ShowPopupTextMsg)"));

   const Candera::String HOME_EDIT_POPUP_TXT_LINE1 = LANGUAGE_STRING(TextId_0x09FE, "Please enter an address or pick one from");
   const Candera::String HOME_EDIT_POPUP_TXT_LINE2 = LANGUAGE_STRING(TextId_0x09FF, "the map, select details, then store as");
   const Candera::String HOME_EDIT_POPUP_TXT_LINE3 = LANGUAGE_STRING(TextId_0x0E37, "home address.");

   const Candera::String WAYPOINT_POPUP_TXT_LINE1 = LANGUAGE_STRING(TextId_0x0F16, "Maximum number of waypoint entries");
   const Candera::String WAYPOINT_POPUP_TXT_LINE2 = LANGUAGE_STRING(TextId_0x0F17, "has been reached. Please delete one");
   const Candera::String WAYPOINT_POPUP_TXT_LINE3 = LANGUAGE_STRING(TextId_0x0F18, "of your started waypoints.");

   const Candera::String LOCATION_OFF_MAP_POPUP_TXT_LINE1 = LANGUAGE_STRING(TextId_0x0FC8, "Selected point lies outside the area covered");
   const Candera::String LOCATION_OFF_MAP_POPUP_TXT_LINE2 = LANGUAGE_STRING(TextId_0x0FC9, "by the map data. Please select a position that");
   const Candera::String LOCATION_OFF_MAP_POPUP_TXT_LINE3 = LANGUAGE_STRING(TextId_0x0FCA, "lies within the area covered by the map data.");

   if (oMsg.GetPopupType() == POPUP_FAVORITES_HOME_EDIT)
   {
      (*_multiLinePopupText).mThreeLinePopupText1 = HOME_EDIT_POPUP_TXT_LINE1;
      (*_multiLinePopupText).mThreeLinePopupText2 = HOME_EDIT_POPUP_TXT_LINE2;
      (*_multiLinePopupText).mThreeLinePopupText3 = HOME_EDIT_POPUP_TXT_LINE3;
   }
   else if (oMsg.GetPopupType() == POPUP_MAX_WAYPOINTS_REACHED)
   {
      (*_multiLinePopupText).mThreeLinePopupText1 = WAYPOINT_POPUP_TXT_LINE1;
      (*_multiLinePopupText).mThreeLinePopupText2 = WAYPOINT_POPUP_TXT_LINE2;
      (*_multiLinePopupText).mThreeLinePopupText3 = WAYPOINT_POPUP_TXT_LINE3;
   }
   else if (oMsg.GetPopupType() == POPUP_LOCATION_OFF_MAP)
   {
      (*_multiLinePopupText).mThreeLinePopupText1 = LOCATION_OFF_MAP_POPUP_TXT_LINE1;
      (*_multiLinePopupText).mThreeLinePopupText2 = LOCATION_OFF_MAP_POPUP_TXT_LINE2;
      (*_multiLinePopupText).mThreeLinePopupText3 = LOCATION_OFF_MAP_POPUP_TXT_LINE3;
   }
   else
   {
      //Do nothing
   }
   _multiLinePopupText.MarkAllItemsModified();
   _multiLinePopupText.SendUpdate();

   POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_HomeEdit_Info_ThreeLine"))));

   return true;
}
