/**************************************************************************************
* @file         : CalculateRouteHandler.cpp
* @author       : ECG5-Naveen Thangamalayan
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include <algorithm>
#include <functional>
#include <utility>

#include "CalculateRouteHandler.h"
#include "Map/MapScreenDataUtils.h"
#include "CgiExtensions/ImageLoader.h"
#include "Settings/VehicleSettingsHelper.h"
#include "App/Core/VehicleDataClient/ClusterDataClientHandler.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/CalculateRouteHandler.cpp.trc.h"
#endif

static const std::string OPEN_BRACKET_STRING = "(";
static const std::string CLOSE_BRACKET_STRING = ")";
static const std::string FULL_STOP_STRING = ".";
static const std::string COMMA_STRING = ",";
static const std::string SPACE_STRING = " ";
static const std::string EMPTY_STRING = "";
static const unsigned int MAX_VEHICLE_RESTRICTIONS_SHOWN = 3;

#ifdef HALL_TO_MDW_COM

static const int INVALID_ROUTE_INDEX = 0xFF;

typedef std::pair< ::navmiddleware::settings::RouteCriterion, enRouteTypeOptions > RouteCriterionPair_t; // (Navmid criterion, hmi criterion)
static const RouteCriterionPair_t criterionPairs[MAX_SIZE_ROUTE_TYPE] =
{
   RouteCriterionPair_t(::navmiddleware::settings::ROUTE_CRITERION_FAST,     ROUTE_TYPE_FASTEST),
   RouteCriterionPair_t(::navmiddleware::settings::ROUTE_CRITERION_ECONOMIC, ROUTE_TYPE_ECONOMIC),
   RouteCriterionPair_t(::navmiddleware::settings::ROUTE_CRITERION_SHORT,    ROUTE_TYPE_SHORTEST)
};


CalculateRouteHandler::CalculateRouteHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _isCalculatingRoutePopUpVisible(false)
   , _activeAlternativeIndex(INVALID_ROUTE_INDEX)
   , _isGuidanceActive(false)
   , _isGuidanceRetriggered(false)
   , _isGuidanceTriggerFromGadget(false)
   , _checkForTollRoadOnRoute(false)
   , _isSoftRestrictionPopupTriggered(false)
   , _navmidActiveRouteCriterion(ROUTE_CRITERION_ECONOMIC)
{
   setMultipleRouteOptionsState(false);
   setRouteTabState(false);
}


CalculateRouteHandler::~CalculateRouteHandler()
{
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::initialize()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::initialize()"));

   //Should be called when RouteGuidance Simulation should be tested
   _navMiddleware.registerGuidancePropertyUpdateCallback(*this);
   _navMiddleware.registerRoutePropertyUpdateCallback(*this);
   _infoStore.registerDataPropertyUpdateCallback(*this);
   initializeMoreRoute();
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::deinitialize()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::deinitialize()"));

   _navMiddleware.unregisterGuidancePropertyUpdateCallback(*this);
   _navMiddleware.unregisterRoutePropertyUpdateCallback(*this);
   _infoStore.unregisterDataPropertyUpdateCallback(*this);
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
bool CalculateRouteHandler::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   const Courier::ViewId GADGET_SCENE_NAME = Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#N_Navigation_Gadget");
   const Courier::Identifier  IdNaviIconBtn = Courier::Identifier("TouchArea_NaviIcon/ButtonWidget");
   const Courier::Identifier senderInfo = oMsg.GetSender();
   const Courier::ViewId sceneName = oMsg.GetView();
   const enReaction  btnReaction = oMsg.GetEnReaction();
   bool isMsgConsumed = false;

   if (enRelease == btnReaction)
   {
      if ((sceneName == GADGET_SCENE_NAME) && (senderInfo == IdNaviIconBtn) && (_infoStore.getIsNavMiddlewareStarted() == true))
      {
         ETG_TRACE_USR4(("CalculateRouteHandler::onCourierMessage(ButtonReactionMsg), N_Navigation_Gadget scene"));
         if (EXT_bIsGuidanceActive == true)
         {
            cancelRouteCalculationAndGuidance();
         }
         else
         {
            _isGuidanceTriggerFromGadget = true;
            if (true == _infoStore.getIsDisclaimerAccepted())
            {
               loadLastDestinationOnStartup();
            }
            else
            {
               POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchReqMsg)(CONTEXT_SWITCH_FROM_HOMESCREEN, CONTEXT_SWITCH_TO_MAP)));
            }
         }
         isMsgConsumed = true;
      }
   }

   return isMsgConsumed;
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
bool CalculateRouteHandler::onCourierMessage(const RetriggerLastGuidedRouteReqMsg& oMsg)
{
   ETG_TRACE_USR4(("CalculateRouteHandler::onCourierMessage(RetriggerLastGuidedRouteReqMsg)"));

   _isGuidanceRetriggered = true;
   _navMiddleware.stopGuidance();

   return true;
}


/********************************************
/* NAME        : CalculateRouteHandler
/* SYSFL       : null,
/* DESCRIPTION : This courier message is sent when guidance is initiated from speech component
****************************************************/
bool CalculateRouteHandler::onCourierMessage(const StartRouteGuidanceFromSpeechReqMsg& oMsg)
{
   ETG_TRACE_USR4(("CalculateRouteHandler::onCourierMessage(StartRouteGuidanceFromSpeechReqMsg)"));

   if (true == _infoStore.getIsDisclaimerAccepted())
   {
      // If multiple route options are shown in map scene and if RG is started from SDS,
      // then multiple route calculation should be stopped before starting RG to destination set in SDS.
      // In this case, RG is started after receiving ROUTE_CALCULATION_STOPPED property update.
      if (true == (*_switchRouteIndex).mIsAlternativeRouteBtnEnabled)
      {
         cancelMultipleRouteCalculation();
      }
      else
      {
         startRouteGuidance(navmiddleware::settings::MORE_ROUTES_TYPE_ALTERNATIVE);
      }
   }
   return true;
}


/********************************************
/* NAME        : CalculateRouteHandler
/* SYSFL       : null,
/* DESCRIPTION : This courier message is sent on press of Guidance button in Address detailed info scene
****************************************************/
bool CalculateRouteHandler::onCourierMessage(const StartRouteGuidanceReqMsg& oMsg)
{
   ETG_TRACE_USR4(("CalculateRouteHandler::onCourierMessage(StartRouteGuidanceReqMsg)"));

   startRouteGuidance(_navMiddleware.getRouteOptionSettings().getMoreRoutesType());
   return true;
}


/********************************************
/* NAME        : CalculateRouteHandler
/* SYSFL       : null,
/* DESCRIPTION : Starts route guidance to single/multiple route
****************************************************/
void CalculateRouteHandler::startRouteGuidance(navmiddleware::settings::MoreRoutesTypeValue routeType)
{
   ETG_TRACE_USR4(("CalculateRouteHandler::startRouteGuidance(), Route type : %d", routeType));

   _isGuidanceActive = false;
   _infoStore.setRouteCalculationStatus(false);
   _isCalculatingRoutePopUpVisible = false;

   if (routeType == navmiddleware::settings::MORE_ROUTES_TYPE_ALTERNATIVE) //Route type is Single
   {
      EXT_bIsGuidanceActive = true;
      _infoStore.setGuidanceStatefromSDS(false);
      _navMiddleware.startRouteCalculationAndGuidance();
      POST_MSG((COURIER_MESSAGE_NEW(ShowMapActiveRGReqMsg)()));
      _infoStore.setIsGuidanceActive(true, true);
      _infoStore.setIsTourEnabled(false);
      setMultipleRouteOptionsState(false);
      setRouteTabState(true);
      _infoStore.setMultipleRouteCalculationStatus(false);
   }
   else //Route type is Multiple
   {
      resetMultipleRouteButtonConfiguration();
      setMultipleRouteOptionsState(true);
      _navMiddleware.startRouteCalculation();
   }
   displayCalculateRoutePopup(true);
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
bool CalculateRouteHandler::onCourierMessage(const CancelRouteGuidanceReqMsg& oMsg)
{
   ETG_TRACE_USR4(("CalculateRouteHandler::onCourierMessage(CancelRouteGuidanceReqMsg)"));

   cancelRouteCalculationAndGuidance();
   return false;
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::cancelRouteCalculationAndGuidance()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::cancelRouteCalculationAndGuidance()"));

   _navMiddleware.stopRouteCalculation();
   _navMiddleware.stopGuidance();
   resetGuidanceData();
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::resetGuidanceData()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::resetGuidanceData()"));

   EXT_bIsGuidanceActive = false;

   const Candera::String BLOCK_BUTTON_TEXT = LANGUAGE_STRING(TextId_0x063C, "Block route list");
   static const char* ROUTELIST_BLOCKED_ICON_NORMAL = "AppHmi_NavigationModule#Images#N_BlockRouteList#Icon_Blocked_1_normal";
   static const char* ROUTELIST_UNBLOCKED_ICON_TOUCHED = "AppHmi_NavigationModule#Images#N_BlockRouteList#Icon_Unblocked_2_touched";
   (*_guidanceBlockUnblockButton).mBlockUnblockNormalIcon = ImageLoader::getAssetBitmapImage(ROUTELIST_BLOCKED_ICON_NORMAL);
   (*_guidanceBlockUnblockButton).mBlockUnblockPressedIcon = ImageLoader::getAssetBitmapImage(ROUTELIST_UNBLOCKED_ICON_TOUCHED);
   (*_guidanceBlockUnblockButton).mBlockUnblockText = BLOCK_BUTTON_TEXT;
   _guidanceBlockUnblockButton.MarkAllItemsModified();
   _guidanceBlockUnblockButton.SendUpdate(true);

   _infoStore.setRouteCalculationStatus(false);
   _infoStore.setIsGuidanceActive(false, true);
   _infoStore.setMultipleRouteCalculationStatus(false);

   _activeAlternativeIndex = INVALID_ROUTE_INDEX;
   _isGuidanceRetriggered = false;
   _checkForTollRoadOnRoute = false;
   _isSoftRestrictionPopupTriggered = false;

   setSoftRestrictionStatus(false);
   setHardRestrictionStatus(false);
   setMultipleRouteOptionsState(false);
   setRouteTabState(false);

   POST_MSG((COURIER_MESSAGE_NEW(SetMapCameraModeReqMsg)(MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN)));
   POST_MSG((COURIER_MESSAGE_NEW(ShowMapReqMsg)()));  // To initiate state transition from N_Map_ActiveRG to N_Map
   POST_MSG((COURIER_MESSAGE_NEW(StopGuidanceMsg)()));
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::onPropertyUpdateRouteCalculationStopped()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateRouteCalculationStopped()"));

   if (true == _isCalculatingRoutePopUpVisible)
   {
      displayCalculateRoutePopup(false);
   }

   // If multiple route options was shown in map while initiating RG from navi gadget or SDS,
   // then RG is initiated here only after multiple route calculation is stopped.
   if (true == _isGuidanceTriggerFromGadget)
   {
      loadLastDestinationOnStartup();
   }
   if (true == _infoStore.getGuidanceStatefromSDS())
   {
      startRouteGuidance(navmiddleware::settings::MORE_ROUTES_TYPE_ALTERNATIVE);
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
bool CalculateRouteHandler::onPropertyUpdateEstimatedTimeModeChanged()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateEstimatedTimeModeChanged()"));

   if (true == _infoStore.getMultipleRouteCalculationStatus())
   {
      for (int index = 0; index < _buttonConfiguration.size(); index++)
      {
         std::string timeOnMap = (InfoStore::ARRIVAL_TIME == _infoStore.getEstimatedTimeMode())
                                 ? _buttonConfiguration.at(index)._etaToFinalDestination
                                 : _buttonConfiguration.at(index)._timeToFinalDestination;

         _buttonConfiguration.at(index).setTimeToDestination(timeOnMap.c_str());
         sendAlternativeRouteItemsUpdate();
      }
   }
   return true;
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
bool CalculateRouteHandler::onPropertyUpdateDisclaimerStatusChanged()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateDisclaimerStatusChanged(), Is disclaimer accepted: %d", _infoStore.getIsDisclaimerAccepted()));

   if (true == _infoStore.getIsDisclaimerAccepted())
   {
      if (true == _isGuidanceTriggerFromGadget)
      {
         loadLastDestinationOnStartup();
      }
      else if (true == _infoStore.getGuidanceStatefromSDS())
      {
         startRouteGuidance(navmiddleware::settings::MORE_ROUTES_TYPE_ALTERNATIVE);
      }
   }
   return true;
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
bool CalculateRouteHandler::onPropertyUpdateHmiAppStateChanged()
{
   if (false == _infoStore.getIsDisclaimerAccepted())
   {
      hmibase::hmiappstates hmiAppState = _infoStore.getHmiAppState();
      ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateHmiAppStateChanged(%d)", hmiAppState));
      if (hmibase::TO_BACKGROUND == hmiAppState)
      {
         if (true == _isGuidanceTriggerFromGadget)
         {
            _isGuidanceTriggerFromGadget = false;
         }
         else if (true == _infoStore.getGuidanceStatefromSDS())
         {
            _infoStore.setGuidanceStatefromSDS(false);
         }
      }
   }
   return true;
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
bool CalculateRouteHandler::onCourierMessage(const ToggleRouteOverViewBtnMsg& oMsg)
{
   ETG_TRACE_USR4(("CalculateRouteHandler::onCourierMessage(ToggleRouteOverViewBtnMsg)"));

   //Hide Compass/ODR when route overview is activated
   (*_routeOverViewBtnState).mIsRouteOverviewBtnEnabled = !oMsg.GetIsRouteOverViewActive();
   _routeOverViewBtnState.MarkAllItemsModified();
   _routeOverViewBtnState.SendUpdate();

   return true;
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
bool CalculateRouteHandler::onCourierMessage(const RouteStartGuidanceReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR4(("CalculateRouteHandler::onCourierMessage(RouteStartGuidanceReqMsg), Index : %d", _activeAlternativeIndex));
   if (INVALID_ROUTE_INDEX != _activeAlternativeIndex)
   {
      const RouteCalculationInfo& routeCalcInfo = _navMiddleware.getRouteCalculationInfo();
      const ::std::vector<RouteCalculationInfo::RouteCalculationInfoAlternative>& routeCalcInfoAlternative = routeCalcInfo.getRouteCalculationInfoAlternatives();
      for (int index = 0; index < routeCalcInfoAlternative.size(); index++)
      {
         if (_navmidActiveRouteCriterion == routeCalcInfoAlternative[index].m_routeCriterion)
         {
            bool isHardRestrictionViolated = ((true == routeCalcInfoAlternative[index].m_routeRestrictionStatus.m_isHardPhysicalRestrictionViolated)
                                              || (true == routeCalcInfoAlternative[index].m_routeRestrictionStatus.m_isHardGeneralRestrictionViolated))
                                             ? true : false;
            bool isSoftRestrictionViolated = routeCalcInfoAlternative[index].m_routeRestrictionStatus.m_isSoftPhysicalRestrictionViolated;
            if (true == isHardRestrictionViolated)
            {
               // "Hard restriction" confirmation popup is displayed.
               POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_LCVHardRestrictionConfirm_Multiple"))));
            }
            else if (true == isSoftRestrictionViolated)
            {
               displaySoftRestrictionsPopup();
            }
            else
            {
               startGuidanceWithRouteCriterion();
            }
            break;
         }
      }
   }
   return true;
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::startGuidanceWithRouteCriterion()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::startGuidanceWithRouteCriterion()"));

   EXT_bIsGuidanceActive = true;
   resetMultipleRouteButtonConfiguration();

   _infoStore.setMultipleRouteCalculationStatus(false);
   _infoStore.setIsGuidanceActive(true, true);
   _navMiddleware.selectHighlightedRoute(navmiddleware::MAP_VIEW_ID__PRIMARY);
   _navMiddleware.startGuidance();
   POST_MSG((COURIER_MESSAGE_NEW(ShowMapActiveRGReqMsg)()));

   setMultipleRouteOptionsState(false);
   setRouteTabState(true);
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
bool CalculateRouteHandler::onCourierMessage(const HighlightActiveRouteReqMsg& oMsg)
{
   ETG_TRACE_USR4(("CalculateRouteHandler::onCourierMessage(HighlightActiveRouteReqMsg)"));

   enRouteTypeOptions activeRouteCriterion = oMsg.GetRouteCriterion();
   _activeAlternativeIndex = _buttonConfiguration.at(activeRouteCriterion)._indexAlternative;
   _navmidActiveRouteCriterion = convertRouteCriterionFromHmiToNavmid(activeRouteCriterion);
   activeMoreRouteOption(activeRouteCriterion);
   sendAlternativeRouteItemsUpdate();
   _navmidActiveRouteCriterion = convertRouteCriterionFromHmiToNavmid(activeRouteCriterion);
   ETG_TRACE_USR4(("CalculateRouteHandler::onCourierMessage(HighlightActiveRouteReqMsg), Index : %d", _activeAlternativeIndex));

   if (INVALID_ROUTE_INDEX != _activeAlternativeIndex)
   {
      _navMiddleware.highlightRoute(MAP_VIEW_ID__PRIMARY, _activeAlternativeIndex);
      setMultipleRouteGuidanceBtnState(true);
   }
   return true;
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::activeMoreRouteOption(enRouteTypeOptions activeRouteCriterion)
{
   ETG_TRACE_USR1(("CalculateRouteHandler::activeMoreRouteOption at %d", activeRouteCriterion));

   for (unsigned int buttonNumber = 0; buttonNumber < _buttonConfiguration.size(); ++buttonNumber)
   {
      if (activeRouteCriterion == buttonNumber)
      {
         _buttonConfiguration.at(buttonNumber).setActiveRouteOption(true);
      }
      else
      {
         _buttonConfiguration.at(buttonNumber).setActiveRouteOption(false);
      }
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::onPropertyUpdateGuidanceStateChanged()
{
   _isGuidanceActive = _navMiddleware.isGuidanceActive();
   ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateGuidanceStateChanged(), Is guidance active : %d", _isGuidanceActive));

   if (_navMiddleware.isFinalDestinationReached())
   {
      resetGuidanceData();
   }
   if (false == _isGuidanceActive)
   {
      // If calculating route pop-up is displayed when guidance is stopped, then pop-up has to be closed
      _checkForTollRoadOnRoute = false;
      displayCalculateRoutePopup(false);

      setSoftRestrictionStatus(false);
      setHardRestrictionStatus(false);

      if (_isGuidanceRetriggered)
      {
         _navMiddleware.restartGuidanceToLastDestination();
         _isGuidanceRetriggered = false;
      }
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::onPropertyUpdateGuidanceHorizonEventChanged()
{
   navmiddleware::HorizonEventInfo horizonEventInfo = _navMiddleware.getHorizonEventInfo();
   uint16_t horizonEventMask = horizonEventInfo.getHorizonEventTypeMask();

   if (HORIZON_EVENT_TYPE__VEHICLE_RESTRICTIONS_APPROACH == (horizonEventMask & HORIZON_EVENT_TYPE__VEHICLE_RESTRICTIONS_APPROACH))
   {
      ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateGuidanceHorizonEventChanged HORIZON_EVENT_TYPE__VEHICLE_RESTRICTIONS_APPROACH"));
      const Candera::String HORIZON_EVENT_VEHICLE_RESTRICTIONS = LANGUAGE_STRING(TextId_0x1951, "Caution: The road ahead has the following vehicle restrictions -");
      std::string vehicleRestrictionsText = HORIZON_EVENT_VEHICLE_RESTRICTIONS.GetCString();

      int totalActiveVehicleRestrictions = 0;
      std::vector<VehicleRestrictionsPriorityType> vehicleRestrictionTypeInfos;
      navmiddleware::VehicleRestrictionsInfo vehicleRestrictionsInfo = horizonEventInfo.getVehicleRestrictionsInfo();

      int endIndex = (true == _infoStore.isTruckType()) ? VEHICLE_RESTRICTIONS_GOODS_TYPE : VEHICLE_RESTRICTIONS_LENGTH;
      for (int index = VEHICLE_RESTRICTIONS_WEIGHT; index <= endIndex; ++index)
      {
         std::string currentVehicleRestrictionsInfoText = getVehicleRestrictionsInfoText(vehicleRestrictionsInfo, index);
         if (!currentVehicleRestrictionsInfoText.empty())
         {
            vehicleRestrictionTypeInfos.push_back((VehicleRestrictionsPriorityType)index);
            if (totalActiveVehicleRestrictions < MAX_VEHICLE_RESTRICTIONS_SHOWN)
            {
               ++totalActiveVehicleRestrictions;
               vehicleRestrictionsText += SPACE_STRING + currentVehicleRestrictionsInfoText + COMMA_STRING;
            }
         }
      }
      ::App::Core::ClusterDataClientHandler::getInstance()->updateVehicleRestrictions(vehicleRestrictionTypeInfos);

      if (totalActiveVehicleRestrictions > 0)
      {
         if (_navMiddleware.getGuidanceSettings().getVehicleRestrictionsInformation())
         {
            POST_MSG((COURIER_MESSAGE_NEW(PlayBeepReqMsg)(hmibase::BEEPTYPE_SPEED_WARN)));
         }

         vehicleRestrictionsText.replace(vehicleRestrictionsText.size() - 1, FULL_STOP_STRING.length(), FULL_STOP_STRING);
         (*_vehicleRestrictionsInfoPopupText).mMultiLineText = vehicleRestrictionsText.c_str();
         _vehicleRestrictionsInfoPopupText.MarkItemModified(ItemKey::FourLinePopupText::MultiLineTextItem);
         _vehicleRestrictionsInfoPopupText.SendUpdate(true);
         if (!_infoStore.isNonNaviVariant())
         {
            POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_FourLine_SingleButton"))));
         }
      }
   }
   if ((HORIZON_EVENT_TYPE__BORDER_CROSSING == (horizonEventMask & HORIZON_EVENT_TYPE__BORDER_CROSSING))
         && _navMiddleware.getGuidanceSettings().getBorderCrossingInformation())
   {
      ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateGuidanceHorizonEventChanged HORIZON_EVENT_TYPE__BORDER_CROSSING"));
      (*_routeInfoPopupText).mTextLine1 = LANGUAGE_STRING(TextId_0x171E, "Caution: This road leads to a border crossing!");
      (*_routeInfoPopupText).mTextLine2 = LANGUAGE_STRING(TextId_0x171F, "");
      _routeInfoPopupText.MarkItemModified(ItemKey::RouteInfoPopupText::TextLine1Item);
      _routeInfoPopupText.MarkItemModified(ItemKey::RouteInfoPopupText::TextLine2Item);
      _routeInfoPopupText.SendUpdate(true);
      if (!_infoStore.isNonNaviVariant())
      {
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_RouteInfo"))));
      }
   }
   if ((HORIZON_EVENT_TYPE__TOLL_BOOTH_APPROACH == (horizonEventMask & HORIZON_EVENT_TYPE__TOLL_BOOTH_APPROACH))
         && _navMiddleware.getGuidanceSettings().getTollBoothInformation())
   {
      ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateGuidanceHorizonEventChanged HORIZON_EVENT_TYPE__TOLL_BOOTH_APPROACH"));
      (*_routeInfoPopupText).mTextLine1 = LANGUAGE_STRING(TextId_0x1914, "Caution: This road contains toll collection!");
      (*_routeInfoPopupText).mTextLine2 = LANGUAGE_STRING(TextId_0x1915, "");
      _routeInfoPopupText.MarkItemModified(ItemKey::RouteInfoPopupText::TextLine1Item);
      _routeInfoPopupText.MarkItemModified(ItemKey::RouteInfoPopupText::TextLine2Item);
      _routeInfoPopupText.SendUpdate(true);
      if (!_infoStore.isNonNaviVariant())
      {
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_RouteInfo"))));
      }
   }
   if ((HORIZON_EVENT_TYPE__USB_POI_APPROACH == (horizonEventMask & HORIZON_EVENT_TYPE__USB_POI_APPROACH))
         && _navMiddleware.getGuidanceSettings().isUsbPoiWarningsEnabled())
   {
      ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateGuidanceHorizonEventChanged HORIZON_EVENT_TYPE__USB_POI_APPROACH"));
      POST_MSG((COURIER_MESSAGE_NEW(PlayBeepReqMsg)(hmibase::BEEPTYPE_POI)));
   }
}


std::string CalculateRouteHandler::getVehicleRestrictionsInfoText(navmiddleware::VehicleRestrictionsInfo& vehicleRestrictionsInfo, int vehicleRestrictionsType)
{
   std::string currentVehicleRestrictionsInfoText = EMPTY_STRING;
   navmiddleware::DistanceUnitSystem unitSystemForRestriction = vehicleRestrictionsInfo.getUnitSystem();
   navmiddleware::DistanceUnitSystem systemMeasurementUnit = static_cast<navmiddleware::DistanceUnitSystem>(_infoStore.getDistanceUnitSystem());
   bool isUnitSystemMetricForRestriction = (navmiddleware::DISTANCEUNITSYSTEM_METRIC == unitSystemForRestriction) ? true : false;
   switch (vehicleRestrictionsType)
   {
      case VEHICLE_RESTRICTIONS_WEIGHT:
      {
         unsigned int weightValue = vehicleRestrictionsInfo.getMaxWeightOfVehicle();
         if ((weightValue != 0) &&
               (navmiddleware::VehicleRestrictionsInfo::RESTRICTION_TYPE__HARD_PHYSICAL == vehicleRestrictionsInfo.getRestrictionTypeForWeight()))
         {
            navmiddleware::DistanceUnitSystem unitSystemForWeightRestriction = vehicleRestrictionsInfo.getUnitSystemForWeightRestriction();
            bool isUnitSystemMetricForWeightRestriction = (navmiddleware::DISTANCEUNITSYSTEM_METRIC == unitSystemForWeightRestriction) ? true : false;
            ETG_TRACE_USR4(("CalculateRouteHandler::getVehicleRestrictionsInfoText(), Weight value : %d", weightValue));
            currentVehicleRestrictionsInfoText = VehicleSettingsHelper::convertVehicleRestrictionsDimensionToString(VEHICLE_RESTRICTIONS_WEIGHT, weightValue, systemMeasurementUnit, isUnitSystemMetricForWeightRestriction);
            currentVehicleRestrictionsInfoText = WeightText.GetCString() + OPEN_BRACKET_STRING + currentVehicleRestrictionsInfoText + CLOSE_BRACKET_STRING;
         }
         break;
      }
      case VEHICLE_RESTRICTIONS_WEIGHT_PER_AXLE:
      {
         unsigned int weightPerAxleValue = vehicleRestrictionsInfo.getMaxWeightPerAxleOfVehicle();
         if ((weightPerAxleValue != 0) &&
               (navmiddleware::VehicleRestrictionsInfo::RESTRICTION_TYPE__HARD_PHYSICAL == vehicleRestrictionsInfo.getRestrictionTypeForWeight()))
         {
            navmiddleware::DistanceUnitSystem unitSystemForWeightRestriction = vehicleRestrictionsInfo.getUnitSystemForWeightRestriction();
            bool isUnitSystemMetricForWeightRestriction = (navmiddleware::DISTANCEUNITSYSTEM_METRIC == unitSystemForWeightRestriction) ? true : false;
            ETG_TRACE_USR4(("CalculateRouteHandler::getVehicleRestrictionsInfoText(), Weight per axle value : %d", weightPerAxleValue));
            currentVehicleRestrictionsInfoText = VehicleSettingsHelper::convertVehicleRestrictionsDimensionToString(VEHICLE_RESTRICTIONS_WEIGHT_PER_AXLE, weightPerAxleValue, systemMeasurementUnit, isUnitSystemMetricForWeightRestriction);
            currentVehicleRestrictionsInfoText = WeightPerAxleText.GetCString() + OPEN_BRACKET_STRING + currentVehicleRestrictionsInfoText + CLOSE_BRACKET_STRING;
         }
         break;
      }
      case VEHICLE_RESTRICTIONS_WIDTH:
      {
         unsigned int widthValue = vehicleRestrictionsInfo.getWidthOfVehicle();
         if ((widthValue != 0) &&
               (navmiddleware::VehicleRestrictionsInfo::RESTRICTION_TYPE__HARD_PHYSICAL == vehicleRestrictionsInfo.getRestrictionTypeForDimension()))
         {
            ETG_TRACE_USR4(("CalculateRouteHandler::getVehicleRestrictionsInfoText(), Width value : %d", widthValue));
            currentVehicleRestrictionsInfoText = VehicleSettingsHelper::convertVehicleRestrictionsDimensionToString(VEHICLE_RESTRICTIONS_WIDTH, widthValue, systemMeasurementUnit, isUnitSystemMetricForRestriction);
            currentVehicleRestrictionsInfoText = WidthText.GetCString() + OPEN_BRACKET_STRING + currentVehicleRestrictionsInfoText + CLOSE_BRACKET_STRING;
         }
         break;
      }
      case VEHICLE_RESTRICTIONS_HEIGHT:
      {
         unsigned int heightValue = vehicleRestrictionsInfo.getHeightOfVehicle();
         if ((heightValue != 0) &&
               (navmiddleware::VehicleRestrictionsInfo::RESTRICTION_TYPE__HARD_PHYSICAL == vehicleRestrictionsInfo.getRestrictionTypeForDimension()))
         {
            ETG_TRACE_USR4(("CalculateRouteHandler::getVehicleRestrictionsInfoText(), Height value : %d", heightValue));
            currentVehicleRestrictionsInfoText = VehicleSettingsHelper::convertVehicleRestrictionsDimensionToString(VEHICLE_RESTRICTIONS_HEIGHT, heightValue, systemMeasurementUnit, isUnitSystemMetricForRestriction);
            currentVehicleRestrictionsInfoText = HeightText.GetCString() + OPEN_BRACKET_STRING + currentVehicleRestrictionsInfoText + CLOSE_BRACKET_STRING;
         }
         break;
      }
      case VEHICLE_RESTRICTIONS_LENGTH:
      {
         unsigned int lengthValue = vehicleRestrictionsInfo.getLengthOfVehicle();
         if ((lengthValue != 0) &&
               (navmiddleware::VehicleRestrictionsInfo::RESTRICTION_TYPE__HARD_PHYSICAL == vehicleRestrictionsInfo.getRestrictionTypeForDimension()))
         {
            ETG_TRACE_USR4(("CalculateRouteHandler::getVehicleRestrictionsInfoText(), Length value : %d", lengthValue));
            currentVehicleRestrictionsInfoText = VehicleSettingsHelper::convertVehicleRestrictionsDimensionToString(VEHICLE_RESTRICTIONS_LENGTH, lengthValue, systemMeasurementUnit, isUnitSystemMetricForRestriction);
            currentVehicleRestrictionsInfoText = LengthText.GetCString() + OPEN_BRACKET_STRING + currentVehicleRestrictionsInfoText + CLOSE_BRACKET_STRING;
         }
         break;
      }
      case VEHICLE_RESTRICTIONS_GOODS_TYPE:
      {
         const ValidValue<navmiddleware::VehicleRestrictionsInfo::CargoType> cargoTypeRestriction = vehicleRestrictionsInfo.getCargoTypeRestriction();
         if ((cargoTypeRestriction.isValid() &&
               (navmiddleware::VehicleRestrictionsInfo::RESTRICTION_TYPE__HARD_PHYSICAL == vehicleRestrictionsInfo.getRestrictionTypeForCargo())))
         {
            if (cargoTypeRestriction.getValue().m_hasWaterPolluting)
            {
               currentVehicleRestrictionsInfoText = HazardousDropDownListLabel[VEHICLE_SETTINGS_WATER_POLLUTING].GetCString();
            }
            else if (cargoTypeRestriction.getValue().m_hasExplosive)
            {
               currentVehicleRestrictionsInfoText = HazardousDropDownListLabel[VEHICLE_SETTINGS_EXPLOSIVES].GetCString();
            }
            else if (cargoTypeRestriction.getValue().m_hasGasses)
            {
               currentVehicleRestrictionsInfoText = HazardousDropDownListLabel[VEHICLE_SETTINGS_GAS].GetCString();
            }
            currentVehicleRestrictionsInfoText = GoodsTypeText.GetCString() + OPEN_BRACKET_STRING + currentVehicleRestrictionsInfoText + CLOSE_BRACKET_STRING;
         }
         break;
      }
      default:
      {
         break;
      }
   }
   return currentVehicleRestrictionsInfoText;
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::onPropertyUpdateRouteCalculationStarted()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateRouteCalculationStarted()"));

   if ((false == EXT_bIsGuidanceActive) && (MORE_ROUTES_TYPE_MULTIPLE == _navMiddleware.getRouteOptionSettings().getMoreRoutesType()))
   {
      const enMapCameraAndViewModeMode& mapCameraViewMode = _infoStore.getMapCameraViewMode();
      ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateRouteCalculationStarted(), mapCameraViewMode : %d", mapCameraViewMode));
      if (MAP_CAMERA_AND_VIEW_MODE_ROUTE_OVERVIEW_MULTIPLE != mapCameraViewMode)
      {
         ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateRouteCalculationStarted(), Camera mode set to Route overview multiple"));
         _infoStore.setMultipleRouteCalculationStatus(true);

         // Camera mode is set to ROUTE_OVERVIEW_MULTIPLE so that route calculation property updates are received for multiple routes
         POST_MSG((COURIER_MESSAGE_NEW(SetMapCameraModeReqMsg)(MAP_CAMERA_AND_VIEW_MODE_ROUTE_OVERVIEW_MULTIPLE)));
      }
   }
   if (false == _isCalculatingRoutePopUpVisible)
   {
      displayCalculateRoutePopup(true);
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::loadLastDestinationOnStartup()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::loadLastDestinationOnStartup()"));

   // If multiple route options are shown in map scene and if RG is started from Navigation tile in home-screen,
   // then multiple route calculation should be stopped before starting RG to last destination.
   // In this case, RG is started to last destination after receiving ROUTE_CALCULATION_STOPPED property update.
   if (true == (*_switchRouteIndex).mIsAlternativeRouteBtnEnabled)
   {
      cancelMultipleRouteCalculation();
   }
   else
   {
      _isGuidanceTriggerFromGadget = false;
      _navMiddleware.restartGuidanceToLastDestination(); //Start the guidance to the last previously guided destination/route from Destination Memory.
      setRouteTabState(true);
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::onPropertyUpdateGuidanceLastGuidedItemDetailsAvailable()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateGuidanceLastGuidedItemDetailsAvailable()"));

   const navmiddleware::DestinationMemoryItem& lastDestination = _navMiddleware.getLastGuidedItem();
   std::stringstream traceOut("No Message");
   if (lastDestination.hasEntry()) //If previously guidance was active for a address
   {
      EXT_bIsGuidanceActive = true;
      ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateGuidanceLastGuidedItemDetailsAvailable(), Start guidance to provided entry"));
      traceOut.str("");
      traceOut << lastDestination.getEntry().getDestination().getName() << " with longitude " << lastDestination.getEntry().getDestination().getLongitude()
               << " and latitude " << lastDestination.getEntry().getDestination().getLatitude();
   }
   else if (lastDestination.hasRoute())  //If previously guidance was active for a route
   {
      if (_navMiddleware.getLastGuidedRouteNumberOfWaypointsReached() == lastDestination.getRoute().getWaypointList().size())
      {
         ETG_TRACE_ERR(("CalculateRouteHandler::onPropertyUpdateGuidanceLastGuidedItemDetailsAvailable() Destination was reached !!!"));
         _infoStore.setIsGuidanceActive(false);
      }
      else
      {
         EXT_bIsGuidanceActive = true;
         traceOut.str("");
         traceOut << lastDestination.getRoute().getName() << " with no of waypoints = " << lastDestination.getRoute().getWaypointList().size();
      }
   }
   else
   {
      ETG_TRACE_ERR(("CalculateRouteHandler::onPropertyUpdateGuidanceLastGuidedItemDetailsAvailable(), UnHandled state %d %d",
                     lastDestination.hasEntry(), lastDestination.hasRoute()));
   }
   ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateGuidanceLastGuidedItemDetailsAvailable(), Start guidance to %s", traceOut.str().c_str()));
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::initializeMoreRoute()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::initializeMoreRoute()"));

   _buttonConfiguration.reserve(MAX_SIZE_ROUTE_TYPE);
   _buttonConfiguration.push_back(
      ButtonMoreRouteType(
         std::pair<Candera::String*, Courier::DataItemKey>(&((*_alternativeRouteItems).mDistanceToDestinationFirst), ItemKey::AlternativeRouteItems::DistanceToDestinationFirstItem),
         std::pair<Candera::String*, Courier::DataItemKey>(&((*_alternativeRouteItems).mTimeToDestinationFirst), ItemKey::AlternativeRouteItems::TimeToDestinationFirstItem),
         std::pair<bool*, Courier::DataItemKey>(&((*_alternativeRouteItems).mMoreRouteFirstBtnActive), ItemKey::AlternativeRouteItems::MoreRouteFirstBtnActiveItem),
         std::pair<bool*, Courier::DataItemKey>(&((*_alternativeRouteItems).mMoreRouteFirstBtnEnabled), ItemKey::AlternativeRouteItems::MoreRouteFirstBtnEnabledItem),
         std::pair<bool*, Courier::DataItemKey>(&((*_alternativeRouteItems).mMoreRouteFirstBtnLoading), ItemKey::AlternativeRouteItems::MoreRouteFirstBtnLoadingItem),
         std::pair<bool*, Courier::DataItemKey>(&((*_alternativeRouteItems).mMoreRouteFirstBtnTouchable), ItemKey::AlternativeRouteItems::MoreRouteFirstBtnTouchableItem),
         &_alternativeRouteItems));
   _buttonConfiguration.push_back(
      ButtonMoreRouteType(
         std::pair<Candera::String*, Courier::DataItemKey>(&((*_alternativeRouteItems).mDistanceToDestinationSecond), ItemKey::AlternativeRouteItems::DistanceToDestinationSecondItem),
         std::pair<Candera::String*, Courier::DataItemKey>(&((*_alternativeRouteItems).mTimeToDestinationSecond), ItemKey::AlternativeRouteItems::TimeToDestinationSecondItem),
         std::pair<bool*, Courier::DataItemKey>(&((*_alternativeRouteItems).mMoreRouteSecondBtnActive), ItemKey::AlternativeRouteItems::MoreRouteSecondBtnActiveItem),
         std::pair<bool*, Courier::DataItemKey>(&((*_alternativeRouteItems).mMoreRouteSecondBtnEnabled), ItemKey::AlternativeRouteItems::MoreRouteSecondBtnEnabledItem),
         std::pair<bool*, Courier::DataItemKey>(&((*_alternativeRouteItems).mMoreRouteSecondBtnLoading), ItemKey::AlternativeRouteItems::MoreRouteSecondBtnLoadingItem),
         std::pair<bool*, Courier::DataItemKey>(&((*_alternativeRouteItems).mMoreRouteSecondBtnTouchable), ItemKey::AlternativeRouteItems::MoreRouteSecondBtnTouchableItem),
         &_alternativeRouteItems));
   _buttonConfiguration.push_back(
      ButtonMoreRouteType(
         std::pair<Candera::String*, Courier::DataItemKey>(&((*_alternativeRouteItems).mDistanceToDestinationThird), ItemKey::AlternativeRouteItems::DistanceToDestinationThirdItem),
         std::pair<Candera::String*, Courier::DataItemKey>(&((*_alternativeRouteItems).mTimeToDestinationThird), ItemKey::AlternativeRouteItems::TimeToDestinationThirdItem),
         std::pair<bool*, Courier::DataItemKey>(&((*_alternativeRouteItems).mMoreRouteThirdBtnActive), ItemKey::AlternativeRouteItems::MoreRouteThirdBtnActiveItem),
         std::pair<bool*, Courier::DataItemKey>(&((*_alternativeRouteItems).mMoreRouteThirdBtnEnabled), ItemKey::AlternativeRouteItems::MoreRouteThirdBtnEnabledItem),
         std::pair<bool*, Courier::DataItemKey>(&((*_alternativeRouteItems).mMoreRouteThirdBtnLoading), ItemKey::AlternativeRouteItems::MoreRouteThirdBtnLoadingItem),
         std::pair<bool*, Courier::DataItemKey>(&((*_alternativeRouteItems).mMoreRouteThirdBtnTouchable), ItemKey::AlternativeRouteItems::MoreRouteThirdBtnTouchableItem),
         &_alternativeRouteItems));
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::onPropertyUpdateArrivalInfosAlternativesChanged()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateArrivalInfosAlternativesChanged()"));

   const navmiddleware::ArrivalInfosAlternatives& arrivalAlternatives = _navMiddleware.getArrivalInfosAlternatives();
   _infoStore.setUserSelectedRouteAlternative(_navMiddleware.getArrivalInfosAlternatives().getIndexActiveAlternative());
   ::navmiddleware::RouteCalculationStatus routeCalcStatus = arrivalAlternatives.getAlternativeCalculationStatusFinalDest();
   ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateArrivalInfosAlternativesChanged(), Route calculation status : %d", routeCalcStatus));

   if (navmiddleware::settings::MORE_ROUTES_TYPE_MULTIPLE == _navMiddleware.getRouteOptionSettings().getMoreRoutesType())
   {
      if (::navmiddleware::ROUTE_CALCULATION_STATUS__UNDEFINED != routeCalcStatus)
      {
         bindToMultipleRouteCriteria();

         if ((::navmiddleware::ROUTE_CALCULATION_STATUS__COMPLETED == routeCalcStatus)
               && (INVALID_ROUTE_INDEX != _activeAlternativeIndex))
         {
            // On re-entry into multiple route overview scene (after route calculation),
            // _activeAlternativeIndex will have the index corresponding to the route option last selected by the user in overview scene.
            // Route corresponding to _activeAlternativeIndex is highlighted.
            _navMiddleware.highlightRoute(navmiddleware::MAP_VIEW_ID__PRIMARY, _activeAlternativeIndex);
         }
      }
      else
      {
         resetMultipleRouteButtonConfiguration();
      }
   }
   else // single route
   {
      setMultipleRouteOptionsState(false);
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::bindToMultipleRouteCriteria()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::bindToMultipleRouteCriteria()"));

   const ArrivalInfosAlternatives& arrivalInfosAlternatives = _navMiddleware.getArrivalInfosAlternatives();
   const ::std::vector<ArrivalInfosAlternatives::ArrivalInfoAlternative>& arrivalInfosAlternative = arrivalInfosAlternatives.getArrivalInfosAlternatives();
   unsigned int numAlternatives = arrivalInfosAlternative.size();
   ETG_TRACE_USR4(("CalculateRouteHandler::bindToMultipleRouteCriteria(), numAlternatives is %d", numAlternatives));

   for (int idx = 0; idx < numAlternatives; idx++)
   {
      if (true == _isCalculatingRoutePopUpVisible)
      {
         displayCalculateRoutePopup(false);
      }
      ::navmiddleware::settings::RouteCriterion navmidRouteCriterion = arrivalInfosAlternative[idx].m_routeCriterion;
      enRouteTypeOptions hmiRouteCriterion = convertRouteCriterionFromNavmidToHmi(navmidRouteCriterion);

      if ((0 != strcmp(arrivalInfosAlternative[idx].m_distanceToFinalDestination.c_str(),
                       _buttonConfiguration.at(hmiRouteCriterion)._distanceToDestination.first->GetCString()))
            || (0 != strcmp(arrivalInfosAlternative[idx].m_etaToFinalDestination.c_str(),
                            _buttonConfiguration.at(hmiRouteCriterion)._etaToFinalDestination.c_str())))
      {
         _buttonConfiguration.at(hmiRouteCriterion)._etaToFinalDestination = arrivalInfosAlternative[idx].m_etaToFinalDestination;
         _buttonConfiguration.at(hmiRouteCriterion)._timeToFinalDestination = arrivalInfosAlternative[idx].m_timeToFinalDestination;
         std::string timeOnMap = (InfoStore::ARRIVAL_TIME == _infoStore.getEstimatedTimeMode())
                                 ? _buttonConfiguration.at(hmiRouteCriterion)._etaToFinalDestination
                                 : _buttonConfiguration.at(hmiRouteCriterion)._timeToFinalDestination;

         _buttonConfiguration.at(hmiRouteCriterion).setRouteOptionLoadingState(false);
         _buttonConfiguration.at(hmiRouteCriterion).setDistanceToDestination(arrivalInfosAlternative[idx].m_distanceToFinalDestination.c_str());
         _buttonConfiguration.at(hmiRouteCriterion).setTimeToDestination(timeOnMap.c_str());
         _buttonConfiguration.at(hmiRouteCriterion)._indexAlternative = arrivalInfosAlternative[idx].m_indexOfReferencedAlternative;
         sendAlternativeRouteItemsUpdate();

         ETG_TRACE_USR4(("CalculateRouteHandler::bindToMultipleRouteCriteria(), Hmi route criterion : %d, Route index : %d", hmiRouteCriterion, _buttonConfiguration.at(hmiRouteCriterion)._indexAlternative));
      }
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::setMultipleRouteGuidanceBtnState(bool isEnabled)
{
   ETG_TRACE_USR4(("CalculateRouteHandler::setMultipleRouteGuidanceBtnState(%d)", isEnabled));

   if ((*_multipleRouteGuidanceBtn).mButtonIsActive != isEnabled)
   {
      (*_multipleRouteGuidanceBtn).mButtonIsActive = isEnabled;
      _multipleRouteGuidanceBtn.MarkAllItemsModified();
      _multipleRouteGuidanceBtn.SendUpdate();
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::setMultipleRouteOptionsState(bool isEnabled)
{
   ETG_TRACE_USR4(("CalculateRouteHandler::setMultipleRouteOptionsState(%d)", isEnabled));

   if ((*_switchRouteIndex).mIsAlternativeRouteBtnEnabled != isEnabled)
   {
      (*_switchRouteIndex).mIsAlternativeRouteBtnEnabled = isEnabled;
      _switchRouteIndex.MarkAllItemsModified();
      _switchRouteIndex.SendUpdate();
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::setRouteTabState(bool isEnabled)
{
   ETG_TRACE_USR4(("CalculateRouteHandler::setRouteTabState(%d)", isEnabled));

   if ((*_switchIndexData).mIndexValue != isEnabled)
   {
      (*_switchIndexData).mIndexValue = isEnabled;
      _switchIndexData.MarkAllItemsModified();
      _switchIndexData.SendUpdate();
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::sendAlternativeRouteItemsUpdate()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::sendAlternativeRouteItemsUpdate()"));

   if (_alternativeRouteItems.HasModifiedItems())
   {
      if (!(_alternativeRouteItems.SendUpdate()))
      {
         ETG_TRACE_ERR(("_alternativeRouteItems update failed!"));
      }
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::showTollRoadPopup()
{
   navmiddleware::settings::UseTollRoadsValue useTollRoadsNaviSettings = _navMiddleware.getRouteOptionSettings().getUseTollRoads();
   navmiddleware::settings::UseTollRoadsValue useTollRoadsRouteSettings = _navMiddleware.getRouteOptionSettingsFromRoute().getUseTollRoads();
   ETG_TRACE_USR4(("CalculateRouteHandler::showTollRoadPopup(), useTollRoadsNaviSettings : %d, useTollRoadsRouteSettings : %d", useTollRoadsNaviSettings, useTollRoadsRouteSettings));

   if ((navmiddleware::settings::USE_TOLL_ROADS_VALUE__ASK == useTollRoadsNaviSettings)
         && (navmiddleware::settings::USE_TOLL_ROADS_VALUE__ALWAYS == useTollRoadsRouteSettings) && (!_infoStore.isNonNaviVariant()))
   {
      POST_MSG((COURIER_MESSAGE_NEW(PopupTypeUpdateMsg)(POPUP_AVOID_TOLL_ROAD)));
      POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_AvoidTollRoad"))));
   }
   else if (((navmiddleware::settings::USE_TOLL_ROADS_VALUE__ASK == useTollRoadsNaviSettings)
             && (navmiddleware::settings::USE_TOLL_ROADS_VALUE__NEVER == useTollRoadsRouteSettings))
            || (navmiddleware::settings::USE_TOLL_ROADS_VALUE__NEVER == useTollRoadsNaviSettings))
   {
      const Candera::String DATA_CONTEXT__TEXT_FIRSTLINE = LANGUAGE_STRING(TextId_0x1916, "This route contains toll road despite avoiding.");
      const Candera::String DATA_CONTEXT__TEXT_SECONDLINE = LANGUAGE_STRING(TextId_0x1917, "");
      (*_routeInfoPopupText).mTextLine1 = DATA_CONTEXT__TEXT_FIRSTLINE;
      (*_routeInfoPopupText).mTextLine2 = DATA_CONTEXT__TEXT_SECONDLINE;
      _routeInfoPopupText.MarkItemModified(ItemKey::RouteInfoPopupText::TextLine1Item);
      _routeInfoPopupText.MarkItemModified(ItemKey::RouteInfoPopupText::TextLine2Item);
      _routeInfoPopupText.SendUpdate(true);
      if (!_infoStore.isNonNaviVariant())
      {
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_RouteInfo"))));
      }
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
bool CalculateRouteHandler::onCourierMessage(const AvoidTollRoadReqMsg& oMsg)
{
   ETG_TRACE_USR4(("CalculateRouteHandler::onCourierMessage(AvoidTollRoadReqMsg)"));
   COURIER_UNUSED(oMsg);

   navmiddleware::settings::RouteOptionsSettings routeOptionsSettings = _navMiddleware.getRouteOptionSettingsFromRoute();
   routeOptionsSettings.setUseTollRoads(navmiddleware::settings::USE_TOLL_ROADS_VALUE__NEVER);
   _navMiddleware.applySettingsToRoute(routeOptionsSettings);
   return true;
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
bool CalculateRouteHandler::onCourierMessage(const CancelMultipleRouteReqMsg& oMsg)
{
   ETG_TRACE_USR4(("CalculateRouteHandler::onCourierMessage(CancelMultipleRouteReqMsg)"));

   cancelMultipleRouteCalculation();
   return true;
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::cancelMultipleRouteCalculation()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::cancelMultipleRouteCalculation()"));

   //Cancel planned alternative route only when the route is calculated
   if ((EXT_bIsGuidanceActive == false) && (true == (*_switchRouteIndex).mIsAlternativeRouteBtnEnabled))
   {
      _infoStore.setMultipleRouteCalculationStatus(false);
      resetMultipleRouteButtonConfiguration();
      setMultipleRouteOptionsState(false);
      setRouteTabState(false);

      _navMiddleware.stopRouteCalculation();
   }
   POST_MSG((COURIER_MESSAGE_NEW(SetMapCameraModeReqMsg)(MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN)));
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::onPropertyUpdateRouteDestinationsChanged()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateRouteDestinationsChanged()"));

   handleTollRoadOnRoute();
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::handleTollRoadOnRoute()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::handleTollRoadOnRoute()"));

   if ((true == _checkForTollRoadOnRoute) && (true == _isGuidanceActive))
   {
      if (true == _navMiddleware.getArrivalInfos().isRouteCalcCompletedForAllDestinations())
      {
         const DestinationInfos& destinationInfos = _navMiddleware.getDestinationInfos();
         const ::std::vector<DestinationInfos::DestinationInfo>& destinationInfo = destinationInfos.getDestinationInfos();
         if (0 < destinationInfo.size())
         {
            ValidValue<RouteInfoProperties> routeInfoProperties = destinationInfo.at(0).m_destinationRouteInfoProperties;
            if (routeInfoProperties.isValid())
            {
               _checkForTollRoadOnRoute = false;
               RouteInfoProperties::Status tollRoadStatus = routeInfoProperties.getValue().m_tollRoad;
               ETG_TRACE_USR4(("CalculateRouteHandler::handleTollRoadOnRoute(), tollRoadStatus : %d", tollRoadStatus));
               if (RouteInfoProperties::STATUS__PRESENT_DESPITE_AVOIDING == tollRoadStatus)
               {
                  showTollRoadPopup();
               }
            }
         }
      }
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::onPropertyUpdateRouteCalcInfoChanged()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateRouteCalcInfoChanged()"));

   const RouteCalculationInfo& routeCalcInfo = _navMiddleware.getRouteCalculationInfo();
   RouteCalculationInfo::Status routeCalcStatus = routeCalcInfo.getStatus();
   ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateRouteCalcInfoChanged(), Route calculation status : %d", routeCalcStatus));

   // If route calculation is completed/failed, "Calculating route" popup is closed.
   if (RouteCalculationInfo::STATUS__ROUTE_AVAILABLE == routeCalcStatus)
   {
      if (true == EXT_bIsGuidanceActive) // For single route, route calculation and guidance is started. Hence check is added for active guidance state
      {
         _checkForTollRoadOnRoute = false;
         if (true == _isCalculatingRoutePopUpVisible)
         {
            displayCalculateRoutePopup(false);
         }

         const ::std::vector<RouteCalculationInfo::RouteCalculationInfoAlternative>& routeCalcInfoAlternative = routeCalcInfo.getRouteCalculationInfoAlternatives();
         if (0 < routeCalcInfoAlternative.size())
         {
            // Check for hard physical or hard general restriction
            if ((true == routeCalcInfoAlternative[0].m_routeRestrictionStatus.m_isHardPhysicalRestrictionViolated)
                  || (true == routeCalcInfoAlternative[0].m_routeRestrictionStatus.m_isHardGeneralRestrictionViolated))
            {
               ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateRouteCalcInfoChanged(), Hard restriction present in route"));
               setHardRestrictionStatus(true);
               //'Route updated' restriction information pop-up is displayed
               if (!_infoStore.isNonNaviVariant())
               {
                  POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_LCVHardRestrictionInfo_ActiveRG"))));
               }
               _checkForTollRoadOnRoute = true;
            }
            // check for soft physical restriction
            else if (true == routeCalcInfoAlternative[0].m_routeRestrictionStatus.m_isSoftPhysicalRestrictionViolated)
            {
               ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateRouteCalcInfoChanged(), Soft restriction present in route"));
               if (false == _isSoftRestrictionPopupTriggered)
               {
                  displaySoftRestrictionsPopup();
               }
               else
               {
                  _checkForTollRoadOnRoute = true;
               }
            }
            else   // No vehicle restrictions
            {
               ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateRouteCalcInfoChanged(), No restrictions in route"));
               _checkForTollRoadOnRoute = true;
            }
         }
      }
      else  // For multiple route, only route calculation is started and completion status is handled in onPropertyUpdateArrivalInfosAlternativesChanged
      {
         // Check for LCV restrictions
         int numberofRoutesRestricted = 0;
         const ::std::vector<RouteCalculationInfo::RouteCalculationInfoAlternative>& routeCalcInfoAlternative = routeCalcInfo.getRouteCalculationInfoAlternatives();
         for (int routeIndex = 0; routeIndex < routeCalcInfoAlternative.size(); routeIndex++)
         {
            bool isRestrictionViolated = ((true == routeCalcInfoAlternative[routeIndex].m_routeRestrictionStatus.m_isHardPhysicalRestrictionViolated)
                                          || (true == routeCalcInfoAlternative[routeIndex].m_routeRestrictionStatus.m_isHardGeneralRestrictionViolated))
                                         ? true : false;

            navmiddleware::settings::RouteCriterion navmidRouteCriterion = routeCalcInfoAlternative[routeIndex].m_routeCriterion;
            enRouteTypeOptions hmiRouteCriterion = convertRouteCriterionFromNavmidToHmi(navmidRouteCriterion);
            _buttonConfiguration.at(hmiRouteCriterion).setRouteOptionButtonState(!isRestrictionViolated);
            if (isRestrictionViolated)
            {
               numberofRoutesRestricted += 1;
               _buttonConfiguration.at(hmiRouteCriterion).setActiveRouteOption(false);
            }
            // Route index corresponding to route criterion selected is set as _activeAlternativeIndex.
            if ((true == _buttonConfiguration.at(hmiRouteCriterion).getActiveRouteOption()) &&
                  (true == _buttonConfiguration.at(hmiRouteCriterion).getRouteOptionButtonState()))
            {
               _navmidActiveRouteCriterion = navmidRouteCriterion;
               _activeAlternativeIndex = _buttonConfiguration.at(hmiRouteCriterion)._indexAlternative;
               if ((false == getMultipleRouteGuidanceBtnState()) && (INVALID_ROUTE_INDEX != _activeAlternativeIndex))
               {
                  ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateRouteCalcInfoChanged(), Active index : %d", _activeAlternativeIndex));
                  _navMiddleware.highlightRoute(navmiddleware::MAP_VIEW_ID__PRIMARY, _activeAlternativeIndex);
                  setMultipleRouteGuidanceBtnState(true);
               }
            }
            sendAlternativeRouteItemsUpdate();
         }
         static ::navmiddleware::RouteCalculationStatus prevRouteCalcStatus = ROUTE_CALCULATION_STATUS__UNDEFINED;
         const navmiddleware::ArrivalInfosAlternatives& arrivalAlternatives = _navMiddleware.getArrivalInfosAlternatives();
         ::navmiddleware::RouteCalculationStatus routeCalcStatus = arrivalAlternatives.getAlternativeCalculationStatusFinalDest();
         ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateRouteCalcInfoChanged(), Number of routes with hard restrictions : %d", numberofRoutesRestricted));
         if ((ROUTE_CALCULATION_STATUS__COMPLETED == routeCalcStatus) && (0 != numberofRoutesRestricted))
         {
            if (numberofRoutesRestricted < MAX_SIZE_ROUTE_TYPE)
            {
               ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateRouteCalcInfoChanged(), Route calculation status (prev, new) : (%d, %d)", prevRouteCalcStatus, routeCalcStatus));
               if (prevRouteCalcStatus != routeCalcStatus)  // To avoid display of restriction popup multiple times
               {
                  displayHardRestrictionsMultipleInfoPopup(true);
               }
               // If the route corresponding to default route criterion has a restriction,
               // then find the next possible route without restriction and set it to active.
               navmiddleware::settings::RouteCriterion navmidRouteCriterion = _navMiddleware.getRouteCriteriaSettings().getRouteCriterion();
               enRouteTypeOptions hmiRouteCriterion = convertRouteCriterionFromNavmidToHmi(navmidRouteCriterion);
               if (false == _buttonConfiguration.at(hmiRouteCriterion).getRouteOptionButtonState())
               {
                  for (int routeIndex = 0; routeIndex < MAX_SIZE_ROUTE_TYPE; routeIndex++)
                  {
                     if (true == _buttonConfiguration.at(hmiRouteCriterion).getRouteOptionButtonState())
                     {
                        _activeAlternativeIndex = _buttonConfiguration.at(routeIndex)._indexAlternative;
                        _navmidActiveRouteCriterion = convertRouteCriterionFromHmiToNavmid(static_cast<enRouteTypeOptions>(routeIndex));
                        _buttonConfiguration.at(routeIndex).setActiveRouteOption(true);
                        sendAlternativeRouteItemsUpdate();
                        if ((false == getMultipleRouteGuidanceBtnState()) && (INVALID_ROUTE_INDEX != _activeAlternativeIndex))
                        {
                           _navMiddleware.highlightRoute(navmiddleware::MAP_VIEW_ID__PRIMARY, _activeAlternativeIndex);
                           setMultipleRouteGuidanceBtnState(true);
                        }
                        break;
                     }
                  }
               }
            }
            else
            {
               displayHardRestrictionsMultipleInfoPopup();   // 'Hard restriction' information popup is displayed

               // Enable all the three buttons and highlight the button corresponding to route criterion selected by user in settings screen
               for (unsigned int buttonNumber = 0; buttonNumber < _buttonConfiguration.size(); ++buttonNumber)
               {
                  _buttonConfiguration.at(buttonNumber).setRouteOptionButtonState(true);
               }
               navmiddleware::settings::RouteCriterion navmidRouteCriterion = _navMiddleware.getRouteCriteriaSettings().getRouteCriterion();
               enRouteTypeOptions hmiRouteCriterion = convertRouteCriterionFromNavmidToHmi(navmidRouteCriterion);
               _buttonConfiguration.at(hmiRouteCriterion).setActiveRouteOption(true);
               _activeAlternativeIndex = _buttonConfiguration.at(hmiRouteCriterion)._indexAlternative;
               sendAlternativeRouteItemsUpdate();

               if (INVALID_ROUTE_INDEX != _activeAlternativeIndex)
               {
                  _navMiddleware.highlightRoute(navmiddleware::MAP_VIEW_ID__PRIMARY, _activeAlternativeIndex);
                  setMultipleRouteGuidanceBtnState(true);
               }
            }
         }
         prevRouteCalcStatus = routeCalcStatus;
      }
   }

   if (RouteCalculationInfo::STATUS__ERROR == routeCalcStatus)
   {
      displayCalculateRoutePopup(false);
      const ValidValue<RouteCalculationInfo::Error>& errorInfo = routeCalcInfo.getError();
      if (true == errorInfo.isValid())
      {
         ETG_TRACE_USR1(("CalculateRouteHandler::onPropertyUpdateRouteCalcInfoChanged(), Error status : %d", errorInfo.getValue()));

         (*_routeInfoPopupText).mTextLine1 = LANGUAGE_STRING(TextId_0x0F9B, "Route calculation has failed. Please try again.");
         (*_routeInfoPopupText).mTextLine2 = LANGUAGE_STRING(TextId_0x0F9C, "");
         _routeInfoPopupText.MarkAllItemsModified();
         _routeInfoPopupText.SendUpdate(true);
         if (!_infoStore.isNonNaviVariant())
         {
            POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_RouteInfo"))));
         }
      }
      stopPlannedOrGuidedRoute();
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::stopPlannedOrGuidedRoute()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::stopPlannedOrGuidedRoute()"));

   // If waypoint is not added, the planned or guided route is stopped on route calculation error or restriction violations.
   if (false == _infoStore.getIsWaypointListModified())
   {
      cancelRouteCalculationAndGuidance();
   }
   else // If waypoint is added, only planned route is stopped. No changes to guided route.
   {
      _infoStore.setIsWaypointListModified(false);
      _navMiddleware.stopRouteCalculation();
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::onPropertyUpdateSoftPhysicalRestrictionChanged()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateSoftPhysicalRestrictionChanged()"));

   if (true == _navMiddleware.isGuidanceActive())
   {
      const RouteCalculationInfo& routeCalcInfo = _navMiddleware.getRouteCalculationInfo();
      const ::std::vector<RouteCalculationInfo::RouteCalculationInfoAlternative>& routeCalcInfoAlternative = routeCalcInfo.getRouteCalculationInfoAlternatives();
      if (0 < routeCalcInfoAlternative.size())
      {
         bool isSoftRestrictionPresent = routeCalcInfoAlternative[0].m_routeRestrictionStatus.m_isSoftPhysicalRestrictionViolated;
         if (true == _isSoftRestrictionPopupTriggered)
         {
            setSoftRestrictionStatus(isSoftRestrictionPresent);
         }
         else if (true == isSoftRestrictionPresent)
         {
            displaySoftRestrictionsPopup();
         }
      }
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::onPropertyUpdateHardRestrictionChanged()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateHardRestrictionChanged()"));

   if (true == _navMiddleware.isGuidanceActive())
   {
      const RouteCalculationInfo& routeCalcInfo = _navMiddleware.getRouteCalculationInfo();
      const ::std::vector<RouteCalculationInfo::RouteCalculationInfoAlternative>& routeCalcInfoAlternative = routeCalcInfo.getRouteCalculationInfoAlternatives();
      if (0 < routeCalcInfoAlternative.size())
      {
         bool isHardRestrictionPresent = routeCalcInfoAlternative[0].m_routeRestrictionStatus.m_isHardPhysicalRestrictionViolated
                                         || routeCalcInfoAlternative[0].m_routeRestrictionStatus.m_isHardGeneralRestrictionViolated;

         // Hard restriction popup should be shown to user when switching from full guidance to reduced guidance mode.
         if ((true == isHardRestrictionPresent) && (false == _infoStore.getIsHardRestrictionPresent()) && (!_infoStore.isNonNaviVariant()))
         {
            //'Route updated' restriction information pop-up is displayed
            ETG_TRACE_USR4(("CalculateRouteHandler::onPropertyUpdateHardRestrictionChanged() Route updated popup shown"));
            POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_LCVHardRestrictionInfo_ActiveRG"))));
         }
         setHardRestrictionStatus(isHardRestrictionPresent);
      }
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
bool CalculateRouteHandler::onCourierMessage(const ShowRouteCalcPopupReqMsg& oMsg)
{
   ETG_TRACE_USR4(("CalculateRouteHandler::onCourierMessage(ShowRouteCalcPopupReqMsg)"));

   displayCalculateRoutePopup(oMsg.GetShowPopup());

   return true;
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::displayCalculateRoutePopup(bool displayPopup)
{
   ETG_TRACE_USR4(("CalculateRouteHandler::displayCalculateRoutePopup(), Display popup : %d, Is popup visible : %d", displayPopup, _isCalculatingRoutePopUpVisible));

   if ((true == displayPopup) && (false == _isCalculatingRoutePopUpVisible))
   {
      (*_PopupText).mTextLine1 = LANGUAGE_STRING(TextId_0x0148, "Calculating route");
      _PopupText.MarkItemModified(ItemKey::MultiLinePopuptext::TextLine1Item);
      _PopupText.SendUpdate(true);
      POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_Dest_CalcRoute"))));
      _isCalculatingRoutePopUpVisible = true;

      _infoStore.setRouteCalculationStatus(true);
   }
   else if ((false == displayPopup) && (true == _isCalculatingRoutePopUpVisible))
   {
      POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_Dest_CalcRoute"))));
      POST_MSG((COURIER_MESSAGE_NEW(RouteGuidancePopUpCloseReqMsg)()));
      _isCalculatingRoutePopUpVisible = false;

      _infoStore.setRouteCalculationStatus(false);
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::displayHardRestrictionsMultipleInfoPopup(bool isMultipleRoutesRestricted)
{
   ETG_TRACE_USR4(("CalculateRouteHandler::displayHardRestrictionsMultiplePopup(%d)", isMultipleRoutesRestricted));

   if (false == isMultipleRoutesRestricted)
   {
      POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_LCVHardRestrictionInfo_Multiple"))));
   }
   else
   {
      (*_routeRestrictionsPopupText).mTextLine1 = LANGUAGE_STRING(TextId_0x1A0F, "One or more routes could not be offered owing to");
      (*_routeRestrictionsPopupText).mTextLine2 = LANGUAGE_STRING(TextId_0x1A10, "traffic regulations.");
      (*_routeRestrictionsPopupText).mTextLine3 = LANGUAGE_STRING(TextId_0x1A11, "");
      _routeRestrictionsPopupText.MarkAllItemsModified();
      _routeRestrictionsPopupText.SendUpdate(true);
      if (!_infoStore.isNonNaviVariant())
      {
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_LCVRestrictionInfo"))));
      }
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::displaySoftRestrictionsPopup()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::displaySoftRestrictionsPopup()"));

   if (true == EXT_bIsGuidanceActive)
   {
      _isSoftRestrictionPopupTriggered = true;
   }
   if (!_infoStore.isNonNaviVariant())
   {
      POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_SoftPhysicalRestrictionInfo"))));
      POST_MSG((COURIER_MESSAGE_NEW(PopupTypeUpdateMsg)(POPUP_SOFT_PHYSICAL_RESTRICTIONS)));   //Send pop-up type information to controller.
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::setSoftRestrictionStatus(bool isPresent)
{
   ETG_TRACE_USR4(("CalculateRouteHandler::setSoftRestrictionStatus(), isSoftRestrictionPresent : %d", isPresent));

   if (isPresent != (*_softRestrictionIcon).mIsSoftPhysicalRestriction)
   {
      (*_softRestrictionIcon).mIsSoftPhysicalRestriction = isPresent;
      _softRestrictionIcon.MarkAllItemsModified();
      _softRestrictionIcon.SendUpdate(true);
   }
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::setHardRestrictionStatus(bool isPresent)
{
   ETG_TRACE_USR4(("CalculateRouteHandler::setHardRestrictionStatus(), isHardRestrictionPresent : %d", isPresent));

   if ((*_hardRestrictionData).mIsFullRGEnabled != !(isPresent))
   {
      (*_hardRestrictionData).mIsFullRGEnabled = !(isPresent);
      _hardRestrictionData.MarkAllItemsModified();
      _hardRestrictionData.SendUpdate(true);
   }
   _infoStore.setIsHardRestrictionPresent(isPresent);
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
/* DESCRIPTION : This courier message is sent on click of yes button in Hard restriction confirmation popup
****************************************************/
bool CalculateRouteHandler::onCourierMessage(const HardRestrictionConfirmMultipleReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_USR4(("CalculateRouteHandler::onCourierMessage(HardRestrictionConfirmMultipleReqMsg)"));

   setHardRestrictionStatus(true);
   startGuidanceWithRouteCriterion();
   _checkForTollRoadOnRoute = true;
   handleTollRoadOnRoute();
   return true;
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
bool CalculateRouteHandler::onCourierMessage(const SoftRestrictionMsg& oMsg)
{
   bool isRestrictionAccepted = oMsg.GetIsRestrictionAccepted();
   ETG_TRACE_USR4(("CalculateRouteHandler::onCourierMessage(SoftRestrictionMsg), isRestrictionAccepted : %d", isRestrictionAccepted));

   if (false == EXT_bIsGuidanceActive)  // Display of soft restriction pop-up in multiple route overview scene
   {
      if (true == isRestrictionAccepted)
      {
         startGuidanceWithRouteCriterion();
         _isSoftRestrictionPopupTriggered = true;
      }
   }
   else
   {
      if (true == isRestrictionAccepted)
      {
         setSoftRestrictionStatus(true);
         _checkForTollRoadOnRoute = true;
         handleTollRoadOnRoute();
      }
      else
      {
         stopPlannedOrGuidedRoute();
      }
   }
   return true;
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
enRouteTypeOptions CalculateRouteHandler::convertRouteCriterionFromNavmidToHmi(RouteCriterion navmidRouteCriterion) const
{
   enRouteTypeOptions hmiRouteCriterion = ROUTE_TYPE_ECONOMIC;
   for (int index = 0; index < MAX_SIZE_ROUTE_TYPE; index++)
   {
      if (static_cast<int>(navmidRouteCriterion) == static_cast<int>(criterionPairs[index].first))
      {
         hmiRouteCriterion = criterionPairs[index].second;
         break;
      }
   }
   return hmiRouteCriterion;
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
::navmiddleware::settings::RouteCriterion CalculateRouteHandler::convertRouteCriterionFromHmiToNavmid(enRouteTypeOptions hmiRouteCriterion) const
{
   RouteCriterion navmidRouteCriterion = ROUTE_CRITERION_ECONOMIC;
   for (int index = 0; index < MAX_SIZE_ROUTE_TYPE; index++)
   {
      if (static_cast<int>(hmiRouteCriterion) == static_cast<int>(criterionPairs[index].second))
      {
         navmidRouteCriterion = criterionPairs[index].first;
         break;
      }
   }
   return navmidRouteCriterion;
}


/********************************************
/* NAME    : CalculateRouteHandler
/* SYSFL    : null,
****************************************************/
void CalculateRouteHandler::resetMultipleRouteButtonConfiguration()
{
   ETG_TRACE_USR4(("CalculateRouteHandler::resetMultipleRouteButtonConfiguration()"));

   _activeAlternativeIndex = INVALID_ROUTE_INDEX;
   setMultipleRouteGuidanceBtnState(false);
   // Resetting distance and ETA for Fast/Economic/Short routes
   for (int buttonNumber = 0; buttonNumber < _buttonConfiguration.size(); buttonNumber++)
   {
      _buttonConfiguration.at(buttonNumber).setDistanceToDestination("");
      _buttonConfiguration.at(buttonNumber).setTimeToDestination("");
      _buttonConfiguration.at(buttonNumber).setRouteOptionButtonState(true);
      _buttonConfiguration.at(buttonNumber).setActiveRouteOption(false);
      _buttonConfiguration.at(buttonNumber).setRouteOptionLoadingState(true);
      _buttonConfiguration.at(buttonNumber)._indexAlternative = INVALID_ROUTE_INDEX;
   }
   // Route criterion set by the user in navigation settings is highlighted by default
   navmiddleware::settings::RouteCriterion navmidRouteCriterion = _navMiddleware.getRouteCriteriaSettings().getRouteCriterion();
   enRouteTypeOptions hmiRouteCriterion = convertRouteCriterionFromNavmidToHmi(navmidRouteCriterion);
   _buttonConfiguration.at(hmiRouteCriterion).setActiveRouteOption(true);
   sendAlternativeRouteItemsUpdate();
}


#endif // HALL_TO_MDW_COM
