/**************************************************************************************
* @file         : NavDataUpdateUtils.cpp
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2018 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "NavDataUpdateUtils.h"
#include <iomanip>


#define ENUM_STRING_START(var) switch(var){
#define ENUM_STRING(ENUM) case ENUM: return ::std::string(#ENUM);
#define ENUM_STRING_END(defaultString) default: return ::std::string(#defaultString);}


using namespace navmiddleware;


static const unsigned int DEVICE_STATUS__UNKNOWN = 255;
static const unsigned int ENUM_UNKNOWN = 255;


/** Convert function for UpdateStatus */
ENUM_CONVERT_START(getMappedUpdateStatus, NavDataUpdateStatus::DataUpdateStatus, enNavDataUpdate__UpdateStatus)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__IDLE, NAVDATAUPDATE__UPDATESTATUS__IDLE)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_STARTED, NAVDATAUPDATE__UPDATESTATUS__AVAILABLE_UPDATES_STARTED)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_IN_PROGRESS, NAVDATAUPDATE__UPDATESTATUS__AVAILABLE_UPDATES_IN_PROGRESS)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_FINISHED, NAVDATAUPDATE__UPDATESTATUS__AVAILABLE_UPDATES_FINISHED)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_ABORTED_BY_ERROR, NAVDATAUPDATE__UPDATESTATUS__AVAILABLE_UPDATES_ABORTED_BY_ERROR)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_ABORTED_BY_USER, NAVDATAUPDATE__UPDATESTATUS__AVAILABLE_UPDATES_ABORTED_BY_USER)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_STARTED, NAVDATAUPDATE__UPDATESTATUS__DOWNLOAD_STARTED)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_IN_PROGRESS, NAVDATAUPDATE__UPDATESTATUS__DOWNLOAD_IN_PROGRESS)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_STOPPED, NAVDATAUPDATE__UPDATESTATUS__DOWNLOAD_STOPPED)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_FINISHED, NAVDATAUPDATE__UPDATESTATUS__DOWNLOAD_FINISHED)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_ABORTED_BY_ERROR, NAVDATAUPDATE__UPDATESTATUS__DOWNLOAD_ABORTED_BY_ERROR)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_ABORTED_BY_USER, NAVDATAUPDATE__UPDATESTATUS__DOWNLOAD_ABORTED_BY_USER)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_STARTED, NAVDATAUPDATE__UPDATESTATUS__INSTALLATION_STARTED)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_IN_PROGRESS, NAVDATAUPDATE__UPDATESTATUS__INSTALLATION_IN_PROGRESS)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_STOPPED, NAVDATAUPDATE__UPDATESTATUS__INSTALLATION_STOPPED)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_FINISHED, NAVDATAUPDATE__UPDATESTATUS__INSTALLATION_FINISHED)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_ABORTED_BY_USER, NAVDATAUPDATE__UPDATESTATUS__INSTALLATION_ABORTED_BY_USER)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__UPDATE_ABORTED_BY_USER, NAVDATAUPDATE__UPDATESTATUS__UPDATE_ABORTED_BY_USER)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_UPDATE_SUCCESSFULLY_FINISHED, NAVDATAUPDATE__UPDATESTATUS__DOWNLOAD_UPDATE_SUCCESSFULLY_FINISHED)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__USB_REMOVED, NAVDATAUPDATE__UPDATESTATUS__USB_REMOVED)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__UPDATE_SOURCE_ERROR, NAVDATAUPDATE__UPDATESTATUS__UPDATE_SOURCE_ERROR)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__FULL_UPDATE_REQUIRED, NAVDATAUPDATE__UPDATESTATUS__FULL_UPDATE_REQUIRED)
ENUM_CONVERT_ENTRY(NavDataUpdateStatus::DATAUPDATESTATUS__UNKNOWN, NAVDATAUPDATE__UPDATESTATUS__UNKNOWN)
ENUM_CONVERT_END(NAVDATAUPDATE__UPDATESTATUS__NOT_HANDLED)


/** Convert function for DeviceStatus */
ENUM_CONVERT_START(getMappedDeviceStatus, DeviceStatus, enNavDataUpdate__DeviceStatus)
ENUM_CONVERT_ENTRY(DEVICESTATUS__AVAILABLE, NAVDATAUPDATE__DEVICESTATUS__AVAILABLE)
ENUM_CONVERT_ENTRY(DEVICESTATUS__INC_UPDATE, NAVDATAUPDATE__DEVICESTATUS__INC_UPDATE)
ENUM_CONVERT_ENTRY(DEVICESTATUS__FULL_UPDATE, NAVDATAUPDATE__DEVICESTATUS__FULL_UPDATE)
ENUM_CONVERT_END(NAVDATAUPDATE__DEVICESTATUS__NOT_AVAILABLE)


/** Convert function for ExportStatus */
/*(ENUM_CONVERT_START(getMappedExportStatus, NavDataUpdateUSBDeviceStatus, enNavDataUpdate__ExportStatus)
ENUM_CONVERT_ENTRY(NAVDATAUPDATE_USB_DEVICE_STATUS__SUCCESS, NAVDATAUPDATE__EXPORTSTATUS__SUCCESS)
ENUM_CONVERT_ENTRY(NAVDATAUPDATE_USB_DEVICE_STATUS__FAILURE, NAVDATAUPDATE__EXPORTSTATUS__FAILURE)
ENUM_CONVERT_ENTRY(NAVDATAUPDATE_USB_DEVICE_STATUS__OUT_OF_SPACE_ERROR, NAVDATAUPDATE__EXPORTSTATUS__OUT_OF_SPACE_ERROR)
ENUM_CONVERT_ENTRY(NAVDATAUPDATE_USB_DEVICE_STATUS__REMOVED_ERROR, NAVDATAUPDATE__EXPORTSTATUS__REMOVED_ERROR)
ENUM_CONVERT_END(NAVDATAUPDATE__EXPORTSTATUS__NOT_HANDLED)*/


unsigned int getDeviceStatus(const NavDataDeviceInfos& deviceInfo, DeviceType deviceType, unsigned int& deviceId)
{
   std::vector<NavDataDeviceInfo>::const_iterator navDataDeviceInfo = deviceInfo.getNavDataDeviceInfos().begin();

   unsigned int deviceStatus = DEVICE_STATUS__UNKNOWN;

   for (; ((navDataDeviceInfo != deviceInfo.getNavDataDeviceInfos().end()) && (DEVICE_STATUS__UNKNOWN == deviceStatus)); ++navDataDeviceInfo)
   {
      if (deviceType == navDataDeviceInfo->getDeviceType())
      {
         deviceStatus = navDataDeviceInfo->getDeviceStatus();
         deviceId = navDataDeviceInfo->getDeviceId();
      }
   }

   return deviceStatus;
}


bool isDeviceActive(const NavDataDeviceInfos& deviceInfo, DeviceType deviceType, unsigned int& deviceId)
{
   return (DEVICE_STATUS__UNKNOWN != getDeviceStatus(deviceInfo, deviceType, deviceId));
}


bool isDeviceAvailable(const NavDataDeviceInfos& deviceInfo, DeviceType deviceType, unsigned int& deviceId)
{
   return (DEVICESTATUS__AVAILABLE == getDeviceStatus(deviceInfo, deviceType, deviceId));
}


bool isDeviceContainsIncUpdate(const NavDataDeviceInfos& deviceInfo, DeviceType deviceType, unsigned int& deviceId)
{
   return (DEVICESTATUS__INC_UPDATE == getDeviceStatus(deviceInfo, deviceType, deviceId));
}


bool isDeviceContainsFullUpdate(const NavDataDeviceInfos& deviceInfo, DeviceType deviceType, unsigned int& deviceId)
{
   return (DEVICESTATUS__FULL_UPDATE == getDeviceStatus(deviceInfo, deviceType, deviceId));
}


::std::string getDataUpdateStatusAsString(NavDataUpdateStatus::DataUpdateStatus navDataUpdateStatus)
{
   ENUM_STRING_START(navDataUpdateStatus)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__IDLE)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_STARTED)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_IN_PROGRESS)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_FINISHED)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_ABORTED_BY_ERROR)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__AVAILABLE_UPDATES_ABORTED_BY_USER)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_STARTED)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_IN_PROGRESS)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_STOPPED)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_FINISHED)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_ABORTED_BY_ERROR)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_ABORTED_BY_USER)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_STARTED)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_IN_PROGRESS)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_STOPPED)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_FINISHED)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__INSTALLATION_ABORTED_BY_USER)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__UPDATE_ABORTED_BY_USER)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__DOWNLOAD_UPDATE_SUCCESSFULLY_FINISHED)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__USB_REMOVED)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__UPDATE_SOURCE_ERROR)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__FULL_UPDATE_REQUIRED)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__OUT_OF_SPACE)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__INVALID_UPDATE_PACKAGE)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__SUBSCRIPTION_EXPIRED)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__SUBSCRIPTION_VALIDATION_IMPOSSIBLE)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__ABNORMAL_TERMINATION)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__INCOMPATIBLE_UPDATE_PACKAGE)
   ENUM_STRING(NavDataUpdateStatus::DATAUPDATESTATUS__UNKNOWN)
   ENUM_STRING_END(NavDataUpdateStatus::DATAUPDATESTATUS__UNKNOWN)
}


::std::string getAvailableUpdatesRequestTypeAsString(AvailableUpdatesRequestType requestType)
{
   ENUM_STRING_START(requestType)
   ENUM_STRING(AVAILABLE_UPDATES_REQUEST_TYPE__INITIAL_REQUEST)
   ENUM_STRING(AVAILABLE_UPDATES_REQUEST_TYPE__MODIFY_SELECTION_REQUEST)
   ENUM_STRING(AVAILABLE_UPDATES_REQUEST_TYPE__CANCEL_SELECTION_REQUEST)
   ENUM_STRING_END(ENUM_UNKNOWN)
}


Candera::String constructUpdateTime(unsigned int timeInSeconds)
{
   const unsigned int hours = timeInSeconds / 3600;
   const unsigned int minutes = (timeInSeconds % 3600) / 60;
   const unsigned int seconds = timeInSeconds % 60;

   App::DataModel::util::MultiLangTextBuilder textBuilder;

   if (hours > 0)
   {
      textBuilder.AddData(hours);
      textBuilder.AddData(" ");
      textBuilder.AddData(LANGUAGE_STRING(TEXTID_UNAVAILABLE, "hours"));
   }

   if ((minutes > 0) || ((minutes == 0) && (hours > 0) && (seconds > 0)))
   {
      if (!textBuilder.GetData().IsEmpty())
      {
         textBuilder.AddData(" ");
      }
      textBuilder.AddData(minutes);
      textBuilder.AddData(" ");
      textBuilder.AddData(LANGUAGE_STRING(TEXTID_UNAVAILABLE, "minutes"));
   }

   if ((seconds > 0) || ((seconds == 0) && (hours == 0) && (minutes == 0)))
   {
      if (!textBuilder.GetData().IsEmpty())
      {
         textBuilder.AddData(" ");
      }
      textBuilder.AddData(seconds);
      textBuilder.AddData(" ");
      textBuilder.AddData(LANGUAGE_STRING(TEXTID_UNAVAILABLE, "seconds"));
   }

   return textBuilder.GetData();
}


Candera::String constructDataSize(uint64_t dataSizeInBytes)
{
   std::stringstream dataSizeStringStream;
   const std::streamsize NUM_OF_DIGITS = 2;

   if (dataSizeInBytes >= (1024 * 1024 * 1024))
   {
      const float gigaBytes = static_cast<float>(dataSizeInBytes) / (1024.0f * 1024.0f * 1024.0f);
      dataSizeStringStream << std::setprecision(NUM_OF_DIGITS) << std::fixed << gigaBytes << " GB";
   }
   else if (dataSizeInBytes >= (1024 * 1024))
   {
      const float megaBytes = static_cast<float>(dataSizeInBytes) / (1024.0f * 1024.0f);
      dataSizeStringStream << std::setprecision(0) << std::fixed << megaBytes << " MB";
   }
   else if (dataSizeInBytes >= 1024)
   {
      const float kiloBytes = static_cast<float>(dataSizeInBytes) / 1024.0f;
      dataSizeStringStream << std::setprecision(0) << std::fixed << kiloBytes << " KB";
   }
   else
   {
      dataSizeStringStream << dataSizeInBytes << " Bytes";
   }

   return Candera::String(dataSizeStringStream.str().c_str());
}


Candera::String constructOverviewInfo(const NavDataUpdateOverviewInfo& navDataUpdateOverviewInfo)
{
   App::DataModel::util::MultiLangTextBuilder textBuilder;
   if (navDataUpdateOverviewInfo.getTotalNumberOfUpdates() > 0)
   {
      //textBuilder.AddData(LANGUAGE_STRING(TextId_New_Mapupdate_0043, "The navigation function can not use during map updating."));
      //textBuilder.AddData(" ");
      //textBuilder.AddData(LANGUAGE_STRING(TextId_New_Mapupdate_0044, "Please do not turn off your vehicle before complete map update."));
      //textBuilder.AddData(" ");
      //textBuilder.AddData(LANGUAGE_STRING(TextId_New_Mapupdate_0010, "Would you like to update to new map version?"));
      //textBuilder.AddData("\n");
      //textBuilder.AddData(LANGUAGE_STRING(TextId_New_Mapupdate_0011, "Current Map Version:"));
      //textBuilder.AddData(" ");
      //textBuilder.AddData(navDataUpdateOverviewInfo.getCurrentVersion().c_str());
      //textBuilder.AddData("\n");
      //textBuilder.AddData(LANGUAGE_STRING(TextId_New_Mapupdate_0012, "New Map Version"));
      //textBuilder.AddData(" ");
      //textBuilder.AddData(navDataUpdateOverviewInfo.getNewVersion().c_str());
      //textBuilder.AddData("\n");
      //// @ TODO : Currently what information has to be shown to the user in update area is not confirmed, So currently not showing any information regarding this in the screen.
      ////textBuilder.AddData((LANGUAGE_STRING(TextId_New_Mapupdate_0013, "Updating Area")));
      ////textBuilder.AddData(": NA");
      ////textBuilder.AddData("\n");
      //textBuilder.AddData(LANGUAGE_STRING(TextId_New_Mapupdate_0014, "Estimate Update Time"));
      //textBuilder.AddData(" ");
      //textBuilder.AddData(constructUpdateTime(navDataUpdateOverviewInfo.getEstimatedTime()));
      //textBuilder.AddData("\n");
      //textBuilder.AddData(LANGUAGE_STRING(TextId_New_Mapupdate_0015, "Data Size"));
      //textBuilder.AddData(" ");
      //textBuilder.AddData(constructDataSize(navDataUpdateOverviewInfo.getDataSize()));
   }

   return textBuilder.GetData();
}
