/**************************************************************************************
* @file         : MapCameraAndViewHandler.cpp
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "MapUtils.h"
#include "MapScreenDataUtils.h"
#include "MapCameraAndViewHandler.h"
#include "MapUtils.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/MapCameraAndViewHandler.cpp.trc.h"
#endif
static const int SDS_GADGET_CAMERA_POSITION_X_VALUE = 322;
static const int SDS_GADGET_CAMERA_POSITION_Y_VALUE = 100;
static const int DEFAULT_ZOOM_SCALE_FOR_ADJUST_CURRENT_LOCATION_SCREEN = 800;

#ifdef HALL_TO_MDW_COM

using namespace navmiddleware;

MapCameraAndViewHandler::MapCameraAndViewHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _requestedMapCameraAndViewMode(MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN)
   , _lastRequestedMapCameraAndViewMode(MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN)
   , _lastManeuverViewType(MANEUVERVIEWTYPE_NONE)
{
}


MapCameraAndViewHandler::~MapCameraAndViewHandler()
{
}


void MapCameraAndViewHandler::initialize()
{
   _navMiddleware.registerMapPropertyUpdateCallback(*this);
   _infoStore.registerDataPropertyUpdateCallback(*this);
}


void MapCameraAndViewHandler::deinitialize()
{
   _navMiddleware.unregisterMapPropertyUpdateCallback(*this);
   _infoStore.unregisterDataPropertyUpdateCallback(*this);
}


void MapCameraAndViewHandler::handleMapCameraAndViewMode(enMapCameraAndViewModeMode mapCameraAndViewMode)
{
   ETG_TRACE_USR1(("MapCameraAndViewHandler::handleMapCameraAndViewMode(%d)", mapCameraAndViewMode));

   bool junctionViewActive =  isJunctionViewActive(_navMiddleware);
   bool multiViewActive = isMultiViewActive(_navMiddleware);
   bool isGoStraightIndicationActive = false;//_infoStore.getGoStraightIndicationStatus();

   _requestedMapCameraAndViewMode = mapCameraAndViewMode;
   _requestedViewConfigurationVector = getViewModeConfigurationForCameraMode(_requestedMapCameraAndViewMode, junctionViewActive, multiViewActive, isGoStraightIndicationActive);

   // check if surface sync is necessary
   bool needSurfaceSync = false;
   if (_lastRequestedViewConfigurationVector.size() && _requestedViewConfigurationVector.size())
   {
      if (_lastRequestedViewConfigurationVector.size() != _requestedViewConfigurationVector.size() ||
            (_lastRequestedViewConfigurationVector[0].m_viewDimensions != _requestedViewConfigurationVector[0].m_viewDimensions))
      {
         needSurfaceSync = true;
      }
   }

   if (needSurfaceSync)
   {
      // do surface synchronization
      //Athira: commented out during CGI Migration as suggested [Sept 2018]
      //startSurfaceSyncronization();
      startSurfaceSyncronization();
   }
   else
   {
      // just process the new camera mode without surface synchronization
      handleMapCameraAndViewModeInternal();
   }
}


bool MapCameraAndViewHandler::onPrepareSurfaces()
{
   handleMapCameraAndViewModeInternal();

   return true;
}


bool MapCameraAndViewHandler::onPropertyUpdateMapCameraAndViewModesDoneChanged()
{
   ETG_TRACE_USR1(("MapCameraAndViewHandler::onPropertyUpdateMapCameraAndViewModesDoneChanged()"));
   if (_navMiddleware.getMapCameraModeInfo().getMapCameraModeInfo() == MAP_MODE_WIDGET)
   {
      //ETG_TRACE_USR1(("Camera mode(%d)", _navMiddleware.getMapCameraModeInfo()));
      //POST_MSG((COURIER_MESSAGE_NEW(ActivateGadgetReqMsg)()));
   }
   //Athira: commented out during CGI Migration as suggested [Sept 2018]
   //surfacesPrepared();
   surfacesPrepared();
   return true;
}


bool MapCameraAndViewHandler::onPropertyUpdateMapSettingsChanged()
{
   ETG_TRACE_USR1(("MapCameraAndViewHandler::onPropertyUpdateMapSettingsChanged"));

   ////we do manual surface sync for change in Dual Map View or  setting only if intersection view is not shown
   //if (!((isIntersectionMapActive(_navMiddleware) || isJunctionViewActive(_navMiddleware)) && (_requestedMapCameraAndViewMode == MAP_CAMERA_AND_VIEW_MODE_MAP_JUNCTION_AND_INTERSECTION_SCREEN)))
   //{
   //   startSurfaceSyncronization(true);
   //}
   //POST_MSG((COURIER_MESSAGE_NEW(ActivateMapReqMsg)()));

   return true;
}


bool MapCameraAndViewHandler::onPropertyUpdateTMIDChanged()
{
   ETG_TRACE_USR4(("MapCameraAndViewHandler::onPropertyUpdateTMIDChanged()"));

   startSurfaceSyncronization(true);

   return true;
}


void MapCameraAndViewHandler::handleMapCameraAndViewModeInternal()
{
   ETG_TRACE_USR1(("MapCameraAndViewHandler::handleMapCameraAndViewModeInternal(), View mode : %d", _requestedMapCameraAndViewMode));

   navmiddleware::ManeuverViewType maneuverViewType = _navMiddleware.getManeuverViewInfo().getManeuverViewType();
   navmiddleware::ValidValue<int32_t> scale;

   switch (_requestedMapCameraAndViewMode)
   {
      case MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN:
      {
         configureCrossHair(false);
         _navMiddleware.setMapCameraAndViewModes(navmiddleware::MAP_VIEW_GROUP__HEADUNIT, _requestedViewConfigurationVector);
         _navMiddleware.switchToLinkedMode(MAP_VIEW_ID__PRIMARY);
         if (_requestedViewConfigurationVector.size() > 1)
         {
            _navMiddleware.switchToLinkedMode(MAP_VIEW_ID__SECONDARY);
         }
         (*_mapMainScreenStateData).mMapMainScreenStateValue = MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN;
         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_LOCATION_DETAILS:
      {
         if (_infoStore.getCoordinatesToBeShownInMap()._coordinates.size())
         {
            _navMiddleware.setMapCameraAndViewModes(navmiddleware::MAP_VIEW_GROUP__HEADUNIT, _requestedViewConfigurationVector);
            configureCrossHair(true);

            _navMiddleware.centerMapToGeoPosition(MAP_VIEW_ID__PRIMARY, navmiddleware::GeoCoordinateDegree(_infoStore.getCoordinatesToBeShownInMap()._coordinates.front()._latitude,
                                                  _infoStore.getCoordinatesToBeShownInMap()._coordinates.front()._longitude),
                                                  scale,
                                                  CameraAnimation__IMMEDIATE);
         }
         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_LEFT_SPLITMAP_SCALE:
      {
         configureCrossHair(false);
         _navMiddleware.setMapCameraAndViewModes(navmiddleware::MAP_VIEW_GROUP__HEADUNIT, _requestedViewConfigurationVector);
         _navMiddleware.switchToLinkedMode(MAP_VIEW_ID__PRIMARY, navmiddleware::CameraAnimation__IMMEDIATE);
         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_MAP_SCROLL:
      {
         _navMiddleware.setMapCameraAndViewModes(navmiddleware::MAP_VIEW_GROUP__HEADUNIT, _requestedViewConfigurationVector);

         if (EXT_bCenterMapToCurrentPosition == 1)   // In this use-case, map has to be centered on CCP. EXT_bCenterMapToCurrentPosition is set to 1 in SM.
         {
            // This condition check is met on entry into N_Map_Move scene from N-Map scene or "By Map" option.
            _navMiddleware.centerMapToGeoPosition(MAP_VIEW_ID__PRIMARY, navmiddleware::GeoCoordinateDegree(_navMiddleware.getPositionStatusInfo().getLatitude(), _navMiddleware.getPositionStatusInfo().getLongitude()), scale);

            // EXT_bCenterMapToCurrentPosition is reset so that in other use-cases, map is centered on the coordinates updated in infostore.
            EXT_bCenterMapToCurrentPosition = 0;
         }
         else
         {
            if (_infoStore.getCoordinatesToBeShownInMap()._coordinates.size())
            {
               _navMiddleware.centerMapToGeoPosition(MAP_VIEW_ID__PRIMARY, navmiddleware::GeoCoordinateDegree(_infoStore.getCoordinatesToBeShownInMap()._coordinates.front()._latitude, _infoStore.getCoordinatesToBeShownInMap()._coordinates.front()._longitude), scale);
            }
         }
         configureCrossHair(true);
         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_MAP_JUNCTION_AND_INTERSECTION_SCREEN:
      {
         if ((_lastRequestedMapCameraAndViewMode != _requestedMapCameraAndViewMode) || (_lastManeuverViewType != maneuverViewType))
         {
            configureCrossHair(false);
            _navMiddleware.setMapCameraAndViewModes(navmiddleware::MAP_VIEW_GROUP__HEADUNIT, _requestedViewConfigurationVector);
            _navMiddleware.switchToLinkedMode(MAP_VIEW_ID__PRIMARY);
         }
         if (_requestedViewConfigurationVector.size() > 1)
         {
            _navMiddleware.showIntersectionMap(MAP_VIEW_ID__SECONDARY, CameraAnimation__IMMEDIATE, INTERSECTION_MAP_MODE_MANUAL);
         }
         (*_mapMainScreenStateData).mMapMainScreenStateValue = MAP_CAMERA_AND_VIEW_MODE_MAP_JUNCTION_AND_INTERSECTION_SCREEN;
         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_ROUTE_OVERVIEW_MULTIPLE:
      case MAP_CAMERA_AND_VIEW_MODE_ROUTE_OVERVIEW:
      {
         //if (_lastRequestedMapCameraAndViewMode != _requestedMapCameraAndViewMode)
         //{
         configureCrossHair(false);
         _navMiddleware.setMapCameraAndViewModes(navmiddleware::MAP_VIEW_GROUP__HEADUNIT, _requestedViewConfigurationVector);
         _navMiddleware.switchToFreeMode(MAP_VIEW_ID__PRIMARY);
         settings::MoreRoutesTypeValue currentMoreRouteType = _navMiddleware.getRouteOptionSettings().getMoreRoutesType();
         //_navMiddleware.showRouteOverview(MAP_VIEW_ID__PRIMARY, navmiddleware::SHOW_ALTERNATIVES_NONE,
         //   currentMoreRouteType, navmiddleware::DESTINATION_PREFERENCE_LAST);
         navmiddleware::ShowAlternatives alternativesPreference = (MAP_CAMERA_AND_VIEW_MODE_ROUTE_OVERVIEW == _requestedMapCameraAndViewMode) ? navmiddleware::SHOW_ALTERNATIVES_NONE : navmiddleware::SHOW_ALTERNATIVES_ALL;
         _navMiddleware.showRouteOverview(MAP_VIEW_ID__PRIMARY, alternativesPreference,
                                          currentMoreRouteType, navmiddleware::DESTINATION_PREFERENCE_LAST);

         /* if (currentMoreRouteType == settings::MORE_ROUTES_TYPE_ALTERNATIVE)
         {
            _navMiddleware.showRouteOverview(MAP_VIEW_ID__PRIMARY, navmiddleware::SHOW_ALTERNATIVES_NONE,
               currentMoreRouteType, navmiddleware::DESTINATION_PREFERENCE_LAST);
         }
         else
         {
            _navMiddleware.showRouteOverview(MAP_VIEW_ID__PRIMARY, navmiddleware::SHOW_ALTERNATIVES_ALL,
               currentMoreRouteType, navmiddleware::DESTINATION_PREFERENCE_LAST);
         }*/
         //}
         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_PICK_POI_50PERCENT:
      {
         configureCrossHair(false);
         _navMiddleware.setMapCameraAndViewModes(navmiddleware::MAP_VIEW_GROUP__HEADUNIT, _requestedViewConfigurationVector);
         const InfoStoreBase::CoordinatesToBeShownInMap& coordinates = _infoStore.getCoordinatesToBeShownInMap();
         navmiddleware::ValidValue<int> scale(coordinates._scale);

         ::std::vector< navmiddleware::GeoCoordinateDegree > optionalCoordinates;
         _navMiddleware.showLocationsInMap(MAP_VIEW_ID__PRIMARY, convert(coordinates._coordinates), optionalCoordinates, coordinates._mapIconInfo, scale, navmiddleware::CameraAnimation__ADAPTIVE);

         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_MAP_COLOR_VIEW_RIGHT:
      {
         configureCrossHair(false);
         _navMiddleware.setMapCameraAndViewModes(navmiddleware::MAP_VIEW_GROUP__HEADUNIT, _requestedViewConfigurationVector);
         _navMiddleware.switchToLinkedMode(MAP_VIEW_ID__PRIMARY);
         break;
      }
      //case STREAMED_MAP_OVERVIEW_MAP: // Not yet implemented from middleware side
      //{
//#ifndef WIN32
      //configureCrossHair(false);
      //_navMiddleware.setMapCameraAndViewModes(navmiddleware::MAP_VIEW_GROUP__CLUSTER, _requestedViewConfigurationVector);
      //_navMiddleware.showRouteOverview(MAP_VIEW_ID__THREE, navmiddleware::SHOW_ALTERNATIVES_NONE,
      //   _navMiddleware.getRouteOptionSettings().getMoreRoutesType(), navmiddleware::DESTINATION_PREFERENCE_LAST);
//#endif
      //break;
      //}
      case STREAMED_MAP_2D_MAP:
      case STREAMED_MAP_3D_MAP:
      {
#ifndef WIN32
         ETG_TRACE_USR1(("MapCameraAndViewHandler::handleMapCameraAndViewModeInternal(), Streamed map pitch : %d", _requestedMapCameraAndViewMode));

         //Update orientation of Stream Map to 2D or 3D
         MapRepresentation mapRepresentation = _navMiddleware.getMapRepresentationInfo().getMapRepresentation();
         mapRepresentation.m_mapPitch = (STREAMED_MAP_2D_MAP == _requestedMapCameraAndViewMode) ? navmiddleware::MAP_PITCH_2D : navmiddleware::MAP_PITCH_3D;
         mapRepresentation.m_mapOrientation = navmiddleware::settings::MAP_ORIENTATION_HEAD_UP;

         //Update view
         _navMiddleware.setMapCameraAndViewModes(navmiddleware::MAP_VIEW_GROUP__CLUSTER, _requestedViewConfigurationVector);
         _navMiddleware.switchToLinkedMode(MAP_VIEW_ID__THREE);
         _navMiddleware.setMapRepresentation(MAP_VIEW_ID__THREE, mapRepresentation, CameraAnimation__FAST);

         // unFreezeMap() is invoked here because specifically for Renault variant in PresCtrl, the streamed map will be created in frozen state (in MapStateManager.cpp)
         // This change was introduced in nav_int_2018_33.4 SDK.
         // To start map streaming, unfreeze has to be invoked.
         _navMiddleware.unFreezeMap(MAP_VIEW_ID__THREE);
#endif
         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_CARSOR_LEFT_OVERVIEW:
      {
         configureCrossHair(false);
         _navMiddleware.setMapCameraAndViewModes(MAP_VIEW_GROUP__HEADUNIT, _requestedViewConfigurationVector);
         _navMiddleware.switchToLinkedMode(MAP_VIEW_ID__PRIMARY);
         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_TRAFFIC_DETOUR_50PERCENT:
      {
         configureCrossHair(false);
         _navMiddleware.setMapCameraAndViewModes(MAP_VIEW_GROUP__HEADUNIT, _requestedViewConfigurationVector);
         _navMiddleware.showTrafficDetourInMap(MAP_VIEW_ID__PRIMARY, TRAFFIC_ROUTE_OVERVIEW_MODE__DETOUR, CameraAnimation__IMMEDIATE);
         break;
      }
      default:
      {
         ETG_TRACE_ERR(("Unknown map camera mode requested!"));
         break;
      }
   }
   _mapMainScreenStateData.MarkAllItemsModified();
   _mapMainScreenStateData.SendUpdate();
   _lastRequestedMapCameraAndViewMode = _requestedMapCameraAndViewMode;
   _lastRequestedViewConfigurationVector = _requestedViewConfigurationVector;
   _lastManeuverViewType = maneuverViewType;
}


std::vector<navmiddleware::ViewModeConfiguration> MapCameraAndViewHandler::getViewModeConfigurationForCameraMode(enMapCameraAndViewModeMode mapCameraMode, bool junctionViewActive, bool multiViewActive, bool isGoStraightIndicationActive)
{
   ETG_TRACE_USR4(("MapCameraAndViewHandler::getViewConfigurationForCameraMode(), View mode : %d", mapCameraMode));

   // The below coordinates are used for 2 camera modes - LOCATION_DETAILS and CARSOR_LEFT_OVERVIEW
   const short DETAIL_INFO_MAP_WINDOW_LEFTX = 30;
   const short DETAIL_INFO_MAP_WINDOW_LEFTY = 93;
   const short DETAIL_INFO_MAP_WINDOW_WIDTH = 196;
   const short DETAIL_INFO_MAP_WINDOW_HEIGHT = 254;

   std::vector<navmiddleware::ViewModeConfiguration> viewConfigurationVector;
   short screenWidth = (short)_infoStore.getScreenWidth();
   short screenHeight = (short)_infoStore.getScreenHeight();

   short screenWidth50Percentage = (short)(screenWidth * 0.5);
   const short MAP_FOOTER_HEIGHT = 58;
   bool splitScreenStatus = false;

   switch (mapCameraMode)
   {
      case MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN:
      {
         if (multiViewActive)
         {
            navmiddleware::ViewModeConfiguration viewConfigurationPrimaryView = initializeViewConfiguration(screenWidth - screenWidth50Percentage, 0, screenWidth50Percentage, screenHeight, 0, 0, screenWidth50Percentage, screenHeight);
            viewConfigurationPrimaryView.m_mapViewId = MAP_VIEW_ID__PRIMARY;
            viewConfigurationPrimaryView.m_mapCameraMode = MAP_MODE_CARSOR;
            viewConfigurationVector.push_back(viewConfigurationPrimaryView);
            navmiddleware::ViewModeConfiguration viewConfigurationSecondaryView = initializeViewConfiguration(0, 0, screenWidth50Percentage, screenHeight, 0, 0, screenWidth50Percentage, screenHeight);
            viewConfigurationSecondaryView.m_mapViewId = MAP_VIEW_ID__SECONDARY;
            viewConfigurationSecondaryView.m_mapCameraMode = MAP_MODE_CARSOR;
            viewConfigurationVector.push_back(viewConfigurationSecondaryView);
            splitScreenStatus = (false == _infoStore.getMultipleRouteCalculationStatus()) ? true : false;
         }
         else
         {
            navmiddleware::ViewModeConfiguration viewConfigurationPrimaryView = initializeViewConfiguration(0, 0, screenWidth, screenHeight, 0, 0, screenWidth, screenHeight);
            viewConfigurationPrimaryView.m_mapViewId = MAP_VIEW_ID__PRIMARY;
            viewConfigurationPrimaryView.m_mapCameraMode = MAP_MODE_CARSOR;
            viewConfigurationVector.push_back(viewConfigurationPrimaryView);
         }
         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_LEFT_SPLITMAP_SCALE:
      {
         navmiddleware::ViewModeConfiguration viewConfigurationPrimaryView = initializeViewConfiguration(0, 0, screenWidth, screenHeight, 0, 0, screenWidth, screenHeight);
         viewConfigurationPrimaryView.m_mapViewId = MAP_VIEW_ID__PRIMARY;
         viewConfigurationPrimaryView.m_mapCameraMode = MAP_MODE_SETTINGS_SECONDARY_MAP_SCALE;
         viewConfigurationVector.push_back(viewConfigurationPrimaryView);
         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_LOCATION_DETAILS:
      {
         navmiddleware::ViewModeConfiguration viewConfigurationPrimaryView = initializeViewConfiguration(0, 0, screenWidth, screenHeight,
               DETAIL_INFO_MAP_WINDOW_LEFTX, DETAIL_INFO_MAP_WINDOW_LEFTY,
               DETAIL_INFO_MAP_WINDOW_WIDTH, DETAIL_INFO_MAP_WINDOW_HEIGHT);
         viewConfigurationPrimaryView.m_mapViewId = MAP_VIEW_ID__PRIMARY;
         viewConfigurationPrimaryView.m_mapCameraMode = MAP_MODE_FREE;
         viewConfigurationVector.push_back(viewConfigurationPrimaryView);
         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_ROUTE_OVERVIEW_MULTIPLE:
      case MAP_CAMERA_AND_VIEW_MODE_ROUTE_OVERVIEW:
      {
         const Candera::UInt16 LEFT_X = (mapCameraMode == MAP_CAMERA_AND_VIEW_MODE_ROUTE_OVERVIEW) ? 0 : 300; //The multiple route overview starts at (300, 100) to avoid display of ccp or destnation flag under the route options.
         const Candera::UInt16 LEFT_Y = 100;
         const Candera::UInt16 WINDOW_WIDTH = (mapCameraMode == MAP_CAMERA_AND_VIEW_MODE_ROUTE_OVERVIEW) ? screenWidth : 480; //The width of multiple route overview is set to 480 to avoid clipping.
         const Candera::UInt16 WINDOW_HEIGHT = (mapCameraMode == MAP_CAMERA_AND_VIEW_MODE_ROUTE_OVERVIEW) ? screenHeight - (LEFT_Y + MAP_FOOTER_HEIGHT) : 350;

         navmiddleware::ViewModeConfiguration viewConfigurationPrimaryView = initializeViewConfiguration(0, 0, screenWidth, screenHeight, LEFT_X, LEFT_Y, WINDOW_WIDTH, WINDOW_HEIGHT);

         viewConfigurationPrimaryView.m_mapViewId = MAP_VIEW_ID__PRIMARY;
         viewConfigurationPrimaryView.m_mapCameraMode = MAP_MODE_ROUTE_OVERVIEW;
         viewConfigurationVector.push_back(viewConfigurationPrimaryView);
         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_MAP_SCROLL:
      {
         navmiddleware::ViewModeConfiguration viewConfigurationPrimaryView = initializeViewConfiguration(0, 0, screenWidth, screenHeight, 0, 0, screenWidth, screenHeight);
         viewConfigurationPrimaryView.m_mapViewId = MAP_VIEW_ID__PRIMARY;
         viewConfigurationPrimaryView.m_mapCameraMode = MAP_MODE_FREE;
         viewConfigurationVector.push_back(viewConfigurationPrimaryView);
         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_MAP_JUNCTION_AND_INTERSECTION_SCREEN:
      {
         if (junctionViewActive || isGoStraightIndicationActive)
         {
            navmiddleware::ViewModeConfiguration viewConfigurationPrimaryView = initializeViewConfiguration(screenWidth - screenWidth50Percentage, 0, screenWidth50Percentage, screenHeight, 0, 0, screenWidth50Percentage, screenHeight);
            viewConfigurationPrimaryView.m_mapViewId = MAP_VIEW_ID__PRIMARY;
            viewConfigurationPrimaryView.m_mapCameraMode = MAP_MODE_CARSOR;
            viewConfigurationVector.push_back(viewConfigurationPrimaryView);
         }
         else
         {
            const short MAP_WINDOW_LEFTX = 25;
            const short MAP_WINDOW_LEFTY = 77;
            const short MAP_WINDOW_WIDTH = 288;
            const short MAP_WINDOW_HEIGHT = 351;

            navmiddleware::ViewModeConfiguration viewConfigurationPrimaryView = initializeViewConfiguration(screenWidth - screenWidth50Percentage, 0, screenWidth50Percentage, screenHeight, 0, 0, screenWidth50Percentage, screenHeight);
            viewConfigurationPrimaryView.m_mapViewId = MAP_VIEW_ID__PRIMARY;
            viewConfigurationPrimaryView.m_mapCameraMode = MAP_MODE_CARSOR;
            viewConfigurationVector.push_back(viewConfigurationPrimaryView);
            navmiddleware::ViewModeConfiguration viewConfigurationSecondaryView = initializeViewConfiguration(0, 0, screenWidth50Percentage, screenHeight, 0, MAP_WINDOW_LEFTY, screenWidth50Percentage, screenHeight - MAP_WINDOW_LEFTY);
            viewConfigurationSecondaryView.m_mapViewId = MAP_VIEW_ID__SECONDARY;
            viewConfigurationSecondaryView.m_mapCameraMode = MAP_MODE_INTERSECTION_MAP;
            viewConfigurationVector.push_back(viewConfigurationSecondaryView);
         }
         splitScreenStatus = true;
         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_PICK_POI_50PERCENT:
      {
         const short SPLIT_POI_MAP_WINDOW_LEFTX = 449;
         const short SPLIT_POI_MAP_WINDOW_LEFTY = 101;
         const short SPLIT_POI_MAP_WINDOW_WIDTH = 250;
         const short SPLIT_POI_WINDOW_HEIGHT = 250;
         navmiddleware::ViewModeConfiguration viewConfigurationPrimaryView = initializeViewConfiguration(0, 0, screenWidth, screenHeight, SPLIT_POI_MAP_WINDOW_LEFTX, SPLIT_POI_MAP_WINDOW_LEFTY, SPLIT_POI_MAP_WINDOW_WIDTH, SPLIT_POI_WINDOW_HEIGHT);

         viewConfigurationPrimaryView.m_mapViewId = MAP_VIEW_ID__PRIMARY;
         viewConfigurationPrimaryView.m_mapCameraMode = MAP_MODE_LOCATION_OVERVIEW;
         viewConfigurationVector.push_back(viewConfigurationPrimaryView);
         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_MAP_COLOR_VIEW_RIGHT:
      {
         const short MAP_WINDOW_LEFTX = 417;
         const short MAP_WINDOW_LEFTY = 73;
         const short MAP_WINDOW_WIDTH = 367;
         const short MAP_WINDOW_HEIGHT = 377;

         navmiddleware::ViewModeConfiguration viewConfigurationPrimaryView = initializeViewConfiguration(0, 0, screenWidth, screenHeight, MAP_WINDOW_LEFTX, MAP_WINDOW_LEFTY, MAP_WINDOW_WIDTH, MAP_WINDOW_HEIGHT);
         viewConfigurationPrimaryView.m_mapViewId = MAP_VIEW_ID__PRIMARY;
         viewConfigurationPrimaryView.m_mapCameraMode = MAP_MODE_CARSOR;
         viewConfigurationVector.push_back(viewConfigurationPrimaryView);
         break;
      }
      case STREAMED_MAP_OVERVIEW_MAP:
      case STREAMED_MAP_2D_MAP:
      case STREAMED_MAP_3D_MAP:
      {
         const int SECONDARY_MAP_HEIGHT = 576;
         const int SECONDARY_MAP_WIDTH = 1024;
         const ::std::size_t RENAULT_VIDEO_TYPE_1 = 1;

         navmiddleware::ViewModeConfiguration viewConfigurationStreamView = initializeViewConfiguration(0, 0, SECONDARY_MAP_WIDTH, SECONDARY_MAP_HEIGHT, 0, 0, SECONDARY_MAP_WIDTH, SECONDARY_MAP_HEIGHT);
         viewConfigurationStreamView.m_mapViewId = MAP_VIEW_ID__THREE;
         viewConfigurationStreamView.m_screenId = SCREEN_ID__CLUSTER;
         viewConfigurationStreamView.m_mapCameraMode = MAP_MODE_CARSOR;

         //TO DO: IP Address and Port number should be read from KDS instead of hard-coding it.
         //mup6cob [2020-CW15]: Region Id for which the request is to be processed should be determined by invoking InfoStoreBase::getMapOutRequest()
         //Based on the region id, the corresponding IP address and port number should be set here.
         std::string ipAddress = "239.0.0.1";
         uint16_t portNumber = 5004;

         //MapVideo Stream Options
         navmiddleware::ViewModeOptions::MapVideoStreamInfo mapVideoStreamInfo;
         mapVideoStreamInfo.setIpAddress(ipAddress);
         mapVideoStreamInfo.setPortNumber(portNumber);
         mapVideoStreamInfo.setVideoType(RENAULT_VIDEO_TYPE_1);

         navmiddleware::ViewModeOptions viewModeOptions = viewConfigurationStreamView.m_viewModeOptions.getValue();
         viewModeOptions.setMapVideoStreamInfo(ValidValue< ::navmiddleware::ViewModeOptions::MapVideoStreamInfo >(mapVideoStreamInfo));
         viewConfigurationStreamView.m_viewModeOptions.setValue(viewModeOptions);

         viewConfigurationVector.push_back(viewConfigurationStreamView);
         ETG_TRACE_USR1(("MapCameraAndViewHandler::getViewModeConfigurationForCameraMode start secondary map streaming"));

         break;
      }
      case STOP_MAP_STREAM:
      {
         std::vector<navmiddleware::ViewModeConfiguration> viewConfigurationVector;
         _navMiddleware.setMapCameraAndViewModes(MAP_VIEW_GROUP__CLUSTER, viewConfigurationVector);
         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_CARSOR_LEFT_OVERVIEW:
      {
         navmiddleware::ViewModeConfiguration viewConfigurationPrimaryView = initializeViewConfiguration(0, 0, screenWidth, screenHeight,
               DETAIL_INFO_MAP_WINDOW_LEFTX, DETAIL_INFO_MAP_WINDOW_LEFTY,
               DETAIL_INFO_MAP_WINDOW_WIDTH, DETAIL_INFO_MAP_WINDOW_HEIGHT);
         viewConfigurationPrimaryView.m_mapViewId = MAP_VIEW_ID__PRIMARY;
         viewConfigurationPrimaryView.m_mapCameraMode = MAP_MODE_CARSOR;
         viewConfigurationVector.push_back(viewConfigurationPrimaryView);
         break;
      }
      case MAP_CAMERA_AND_VIEW_MODE_TRAFFIC_DETOUR_50PERCENT:
      {
         const short SPLIT_TRAFFIC_MAP_WINDOW_LEFTX = 449;
         const short SPLIT_TRAFFIC_MAP_WINDOW_LEFTY = 101;
         const short SPLIT_TRAFFIC_MAP_WINDOW_WIDTH = 250;
         const short SPLIT_TRAFFIC_WINDOW_HEIGHT = 250;
         navmiddleware::ViewModeConfiguration viewConfigurationPrimaryView = initializeViewConfiguration(0, 0, screenWidth, screenHeight, SPLIT_TRAFFIC_MAP_WINDOW_LEFTX, SPLIT_TRAFFIC_MAP_WINDOW_LEFTY, SPLIT_TRAFFIC_MAP_WINDOW_WIDTH, SPLIT_TRAFFIC_WINDOW_HEIGHT);

         viewConfigurationPrimaryView.m_mapViewId = MAP_VIEW_ID__PRIMARY;
         viewConfigurationPrimaryView.m_mapCameraMode = MAP_MODE_TRAFFIC_DETOUR;
         viewConfigurationVector.push_back(viewConfigurationPrimaryView);
         break;
      }
      default:
      {
         ETG_TRACE_ERR(("Unknown map camera mode requested!"));
         break;
      }
   }
   _infoStore.setSplitScreenEnabled(splitScreenStatus);
   return viewConfigurationVector;
}


navmiddleware::ViewModeConfiguration MapCameraAndViewHandler::initializeViewConfiguration(short viewX, short viewY, short viewWidth, short viewHeight, short viewAreaX, short viewAreaY, short viewAreaWidth, short viewAreaHeight)
{
   navmiddleware::ViewModeConfiguration viewConfiguration;
   viewConfiguration.m_mapViewId = MAP_VIEW_ID__PRIMARY;
   navmiddleware::ViewDimensions viewDimeansions;
   viewDimeansions.m_top = viewY;
   viewDimeansions.m_left = viewX;
   viewDimeansions.m_width = viewWidth;
   viewDimeansions.m_height = viewHeight;
   viewConfiguration.m_viewDimensions = viewDimeansions;
   navmiddleware::ViewDimensions activeAreaDimensions;
   activeAreaDimensions.m_top = viewAreaY;
   activeAreaDimensions.m_left = viewAreaX;
   activeAreaDimensions.m_width = viewAreaWidth;
   activeAreaDimensions.m_height = viewAreaHeight;
   viewConfiguration.m_activeAreaDimensions = activeAreaDimensions;

   navmiddleware::ViewModeOptions mapViewModeOptions;
   ValidValue<bool> isGuidanceArrowVisible(!(_infoStore.getIsHardRestrictionPresent()));
   mapViewModeOptions.setGuidanceArrowVisibilityState(isGuidanceArrowVisible);

   viewConfiguration.m_viewModeOptions.setValue(mapViewModeOptions);
   return viewConfiguration;
}


void MapCameraAndViewHandler::configureCrossHair(bool showCrossHair)
{
   if (showCrossHair == true)
   {
      _navMiddleware.configureCrossHair(MAP_VIEW_ID__PRIMARY, CROSSHAIR_VISIBILITY_ON);
   }
   else
   {
      _navMiddleware.configureCrossHair(MAP_VIEW_ID__PRIMARY, CROSSHAIR_VISIBILITY_OFF);
   }
}


int32_t MapCameraAndViewHandler::getDefaultScaleValue(const int32_t scale) const
{
   const std::vector< navmiddleware::NominalScaleInfos::NominalScaleInfo >& nominalScaleInfoVector = _navMiddleware.getNominalScaleInfos().getNominalScaleInfos();
   int32_t defaultScale = _infoStore.getDefaultNominalMapScale();
   if (!nominalScaleInfoVector.empty())
   {
      defaultScale = findDefaultMapScaleFromNominalScales(nominalScaleInfoVector, scale);
   }
   ETG_TRACE_USR1(("MapCameraAndViewHandler::getDefaultScaleValue current Scale %d Nominal scale %d", scale, defaultScale));
   return defaultScale;
}


#endif // HALL_TO_MDW_COM
