/**************************************************************************************
* @file         : HMIModelComponent.cpp
* @author       : ECG5-Naveen Thangamalayan
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include <algorithm>
#include <functional>

#include "Common/Util/HMIModelIncludes.h"
#include "Common/Util/TraceCommandHandler.h"
#include "Map/MapScreenDataUtils.h"
#include "HMIModelComponent.h"
#include "Destination/DestinationAddressInputHandlerNarRegion.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/HMIModelComponent.cpp.trc.h"
#endif
#include "Common/Util/NavMiddlewareStartup.h"

#ifdef HALL_TO_MDW_COM

using namespace navmiddleware;

HMIModelComponent::HMIModelComponent()
   : _destinationAddressInputHandler(NULL),
     _destinationMatchListHandler(_navMiddleware, _infoStore),
     _destinationLatLongSpellerHandler(_navMiddleware, _infoStore),
     _hmiModelMap(_navMiddleware, _infoStore),
     _mapPickingHandler(_navMiddleware, _infoStore),
     _userpoiListHandler(_navMiddleware, _infoStore),
     _poiListHandler(_navMiddleware, _infoStore),
     _favoritesListHandler(_navMiddleware, _infoStore),
     _trafficListHandler(_navMiddleware, _infoStore),
     _waypointListHandler(_navMiddleware, _infoStore),
     _routeListHandler(_navMiddleware, _infoStore),
     _previousDestinationListHandler(_navMiddleware, _infoStore),
     _addressDetailedInfo(_navMiddleware, _infoStore),
     _hmiModelSettings(_navMiddleware, _infoStore),
     _hmiModelPosition(_navMiddleware, _infoStore),
     _calculateRouteHandler(_navMiddleware, _infoStore),
     _stackedPOIOnMapListHandler(_navMiddleware, _infoStore),
     _hmiModelGuidance(_navMiddleware, _infoStore),
     _mapPickingOptionHandler(_navMiddleware, _infoStore),
     _hmiModelNavDataUpdate(_navMiddleware, _infoStore),
     _hmiModelNavigationDataUpdateService(_navMiddleware, _infoStore),
     _engineeringmenuhandler(_navMiddleware, _infoStore),
     _isNavAppInForeground(false),
     _hmiModelNavigationService(_navMiddleware, _infoStore),
     _hmiModelNavigationSDSService(_navMiddleware, _infoStore),
     _isRestartRGNeeded(false),
     _isReinitialized(false)
{
   // add sub models to hmi sub models list
   _hmiSubModels.push_back(&_favoritesListHandler);
   _hmiSubModels.push_back(&_destinationMatchListHandler);
   _hmiSubModels.push_back(&_destinationLatLongSpellerHandler);
   _hmiSubModels.push_back(&_hmiModelGuidance);
   _hmiSubModels.push_back(&_hmiModelMap);
   _hmiSubModels.push_back(&_userpoiListHandler);
   _hmiSubModels.push_back(&_poiListHandler);
   _hmiSubModels.push_back(&_mapPickingHandler);
   _hmiSubModels.push_back(&_previousDestinationListHandler);
   _hmiSubModels.push_back(&_hmiModelSettings);
   _hmiSubModels.push_back(&_trafficListHandler);
   _hmiSubModels.push_back(&_waypointListHandler);
   _hmiSubModels.push_back(&_routeListHandler);
   _hmiSubModels.push_back(&_addressDetailedInfo);
   _spellerHandler = SpellerHandler::getInstance();
   _hmiSubModels.push_back(&_hmiModelPosition);
   _hmiSubModels.push_back(&_calculateRouteHandler);
   _hmiSubModels.push_back(&_stackedPOIOnMapListHandler);
   _hmiSubModels.push_back(&_mapPickingOptionHandler);
   _hmiSubModels.push_back(&_hmiModelNavDataUpdate);
   _hmiSubModels.push_back(&_hmiModelNavigationService);
   _hmiSubModels.push_back(&_hmiModelNavigationSDSService);
   _hmiSubModels.push_back(&_hmiModelNavigationDataUpdateService);
   _hmiSubModels.push_back(&_engineeringmenuhandler);
}


HMIModelComponent::~HMIModelComponent()
{
   if(NULL != _destinationAddressInputHandler)
   {
      delete _destinationAddressInputHandler;
      _destinationAddressInputHandler = NULL;
   }
}


bool HMIModelComponent::initialize()
{
   ETG_TRACE_COMP(("HMIModelComponent::initialize()"));

   if (checkNavMiddlewareVersion(_navMiddleware))
   {
      ETG_TRACE_COMP(("navMiddleware version ok - starting up!"));

      _infoStore.initialize();
      _isRestartRGNeeded = _infoStore.getIsGuidanceActive();

      if(NULL == _destinationAddressInputHandler)
      {
         _destinationAddressInputHandler = (true == _infoStore.isNarRegion())
                                           ? new DestinationAddressInputHandlerNarRegion(_navMiddleware, _infoStore)
                                           : new DestinationAddressInputHandler(_navMiddleware, _infoStore);
         _hmiSubModels.push_back(_destinationAddressInputHandler);
      }

      // initialize middleware and wait for started trigger
      _navMiddleware.registerLifeCyclePropertyUpdateCallback(*this);
      initializeMiddleware(_navMiddleware, _infoStore);

      setNavigationGadgetStatus(NAVI_GADGET_STATUS__INITIALIZING, false);
      // FocusChangedUpdMsg::Subscribe(Courier::ComponentType::Model, Courier::ComponentType::View);

      initNaviSettingsTextIDs();
      TraceCommandHandler::initialize(_navMiddleware, _infoStore);
   }
   else
   {
      ETG_TRACE_ERR(("navMiddleware version nok - not starting up!"));
   }
   return true;
}


void HMIModelComponent::savePersistency()
{
   //ETG_TRACE_COMP(("HMIModelComponent::savePersistency"));

   // CAREFUL: this is called after deinitialize()
   // it may also be called multiple times
}


bool HMIModelComponent::deinitialize()
{
   ETG_TRACE_COMP(("HMIModelComponent::deinitialize()"));

   std::for_each(_hmiSubModels.begin(), _hmiSubModels.end(), std::mem_fun(&HMIModelBase::deinitialize));
   std::for_each(_hmiSubModels.begin(), _hmiSubModels.end(), std::mem_fun(&HMIModelBase::setDeinitialized));

   // store current settings
   _navMiddleware.applySettings();
   _navMiddleware.stop();
   _navMiddleware.unregisterLifeCyclePropertyUpdateCallback(*this);

   _infoStore.deinitialize();

   return true;
}


bool HMIModelComponent::reinitialize()
{
   ETG_TRACE_COMP(("HMIModelComponent::reinitialize()"));

   if (checkNavMiddlewareVersion(_navMiddleware))
   {
      ETG_TRACE_COMP(("navMiddleware version ok - restarting middleware!"));

      _infoStore.initialize();
      _isRestartRGNeeded = _infoStore.getIsGuidanceActive();
      _isReinitialized = true;

      setNavigationGadgetStatus(NAVI_GADGET_STATUS__INITIALIZING, false);

      //Re-initialize middleware and wait for started trigger
      _navMiddleware.registerLifeCyclePropertyUpdateCallback(*this);
      reinitializeMiddleware(_navMiddleware, _infoStore);
   }
   return true;
}


bool HMIModelComponent::onPropertyUpdateNavMiddlewareStarted()
{
   ETG_TRACE_COMP(("HMIModelComponent::onPropertyUpdateNavMiddlewareStarted()"));

   _infoStore.setIsNavMiddlewareStarted(true);
   // initialize sub models so that they can directly access middleware features
   std::for_each(_hmiSubModels.begin(), _hmiSubModels.end(), std::mem_fun(&HMIModelBase::initialize));
   std::for_each(_hmiSubModels.begin(), _hmiSubModels.end(), std::mem_fun(&HMIModelBase::setInitialized));

   // only send notifications for persistent values once HMIModel initialized
   startNavigationLimitedModeTimer();
   _infoStore.notifyDataUpdatesFromDataPool();
   setNavigationGadgetStatus(NAVI_GADGET_STATUS__NO_GUIDANCE, true);

   POST_MSG((COURIER_MESSAGE_NEW(NavigationLifeCycleStatusUpdMsg)(LIFECYCLE_STATUS__STARTED)));
   if ((true == _isRestartRGNeeded) && (false == _isReinitialized))
   {
      _isRestartRGNeeded = false;
      _calculateRouteHandler.loadLastDestinationOnStartup();
   }
   return true;
}


bool HMIModelComponent::onPropertyUpdateNavMiddlewareStopped()
{
   ETG_TRACE_COMP(("HMIModelComponent::onPropertyUpdateNavMiddlewareStopped()"));

   POST_MSG((COURIER_MESSAGE_NEW(NavigationLifeCycleStatusUpdMsg)(LIFECYCLE_STATUS__STOPPED)));

   _infoStore.setIsNavMiddlewareStarted(false);
   setNavigationGadgetStatus(NAVI_GADGET_STATUS__NO_DATA, false);

   return true;
}


bool HMIModelComponent::onPropertyUpdateNavDataUpdateStarted()
{
   ETG_TRACE_COMP(("HMIModelComponent::onPropertyUpdateNavDataUpdateStarted()"));

   _infoStore.setNavDataUpdateRunning(true);
   setNavigationGadgetStatus(NAVI_GADGET_STATUS__INITIALIZING, true);   // Enable navigation tile in home-screen (if not already enabled because of inconsistent map data)

   std::for_each(_hmiSubModels.begin(), _hmiSubModels.end(), std::mem_fun(&HMIModelBase::navDataUpdateStarted));
   std::for_each(_hmiSubModels.begin(), _hmiSubModels.end(), std::mem_fun(&HMIModelBase::setNavDataUpdateStarted));
//   POST_MSG((COURIER_MESSAGE_NEW(NavigationLifeCycleStatusUpdMsg)(LIFECYCLE_STATUS__NAVDATAUPDATE_STARTED)));

   return true;
}


bool HMIModelComponent::onPropertyUpdateNavDataUpdateFinished()
{
   ETG_TRACE_COMP(("HMIModelComponent::onPropertyUpdateNavDataUpdateFinished()"));
   _infoStore.setNavDataUpdateRunning(false);
   _infoStore.setNavDataUpdateRecovery(false);
   std::for_each(_hmiSubModels.begin(), _hmiSubModels.end(), std::mem_fun(&HMIModelBase::navDataUpdateFinished));
   std::for_each(_hmiSubModels.begin(), _hmiSubModels.end(), std::mem_fun(&HMIModelBase::setNavDataUpdateFinished));
//   POST_MSG((COURIER_MESSAGE_NEW(NavigationLifeCycleStatusUpdMsg)(LIFECYCLE_STATUS__NAVDATAUPDATE_FINISHED)));

   return true;
}


void HMIModelComponent::onPropertyUpdateNavDataRecovery()
{
   ETG_TRACE_COMP(("HMIModelComponent::onPropertyUpdateNavDataRecovery()"));

   std::for_each(_hmiSubModels.begin(), _hmiSubModels.end(), std::mem_fun(&HMIModelBase::recoverNavData));
   std::for_each(_hmiSubModels.begin(), _hmiSubModels.end(), std::mem_fun(&HMIModelBase::setNavDataUpdateRecovery));
   _infoStore.notifyDataUpdatesRecovery();
   _infoStore.setNavDataUpdateRecovery(true);
//   POST_MSG((COURIER_MESSAGE_NEW(NavigationLifeCycleStatusUpdMsg)(LIFECYCLE_STATUS__NAVDATA_RECOVERY)));
}


bool HMIModelComponent::onCourierMessage(const Courier::StartupMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_COMP(("HMIModelComponent::onCourierMessage(StartupMsg)"));

   initialize();

   return true;
}


bool HMIModelComponent::onCourierMessage(const Courier::ShutdownMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_COMP(("HMIModelComponent::onCourierMessage(ShutdownMsg)"));

   deinitialize();

   return true;
}


bool HMIModelComponent::onCourierMessage(const ApplicationStateUpdMsg& oMsg)
{
   ETG_TRACE_USR1(("HMIModelComponent::onCourierMessage(ApplicationStateUpdMsg(%d))", oMsg.GetState()));
   _infoStore.setHmiAppState(oMsg.GetState());
   if (oMsg.GetState() == hmibase::IN_FOREGROUND)
   {
      EXT_bIsNaviAppRunning = _isNavAppInForeground = true;
   }
   else
   {
      EXT_bIsNaviAppRunning = _isNavAppInForeground = false;
      // When Application is moved to Background, the _currentActiveNaviContext should be reset to NONE. Hence the below msg is posted to NaviApplicationSwitchClientHandler
      POST_MSG((COURIER_MESSAGE_NEW(SendContextReqMsg)(CONTEXT_SWITCH_FROM_NONE)));
   }
   return false;
}


bool HMIModelComponent::onCourierMessage(const HKStateUpdMsg& oMsg)
{
   ETG_TRACE_USR1(("HMIModelComponent::onCourierMessage(HKStateUpdMsg)"));

   if (false == _isNavAppInForeground)
   {
      POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchReqMsg)(CONTEXT_SWITCH_FROM_NONE, CONTEXT_SWITCH_TO_MAP)));
   }
   else if (false == _infoStore.isNavDataUpdateRunning())
   {
      POST_MSG((COURIER_MESSAGE_NEW(HKStatusChangedMsg)()));
   }
   else
   {
      ETG_TRACE_USR4(("HMIModelComponent::onCourierMessage(HKStateUpdMsg), Ignored since map update is in progress"));
   }

   return true;
}


bool HMIModelComponent::onCourierMessage(const DeactivateLimitedModeReqMsg& oMsg)
{
   COURIER_UNUSED(oMsg);
   ETG_TRACE_COMP(("HMIModelComponent::onCourierMessage(DeactivateLimitedModeReqMsg)"));

   deactivateNavigationLimitedMode(_navMiddleware);
   if ((true == _isReinitialized) && (true == _isRestartRGNeeded))
   {
      _isReinitialized = _isRestartRGNeeded = false;
      _calculateRouteHandler.loadLastDestinationOnStartup();
   }
   return true;
}


bool HMIModelComponent::onCourierMessage(const SetBackButtonLevelReqMsg& oMsg)
{
   Candera::UInt8 backButtonLevel = oMsg.GetBackButtonLevel();

   switch (backButtonLevel)
   {
      case BACK_BUTTON_LEVEL_ONE:
      {
         (*_navBackButtonLevel).mBackButtonLevel = BACK_BUTTON_LEVEL_ONE;
         break;
      }
      case BACK_BUTTON_LEVEL_TWO:
      {
         (*_navBackButtonLevel).mBackButtonLevel = BACK_BUTTON_LEVEL_TWO;
         break;
      }
      case BACK_BUTTON_LEVEL_THREE:
      {
         (*_navBackButtonLevel).mBackButtonLevel = BACK_BUTTON_LEVEL_THREE;
         break;
      }
      default :
      {
         (*_navBackButtonLevel).mBackButtonLevel = BACK_BUTTON_LEVEL_ONE;
      }
   }
   _navBackButtonLevel.MarkItemModified(ItemKey::NaviBackButtonLevel::BackButtonLevelItem);
   _navBackButtonLevel.SendUpdate();
   return true;
}


bool HMIModelComponent::onCourierMessage(const ValidateNaviEntryConditionReqMsg& oMsg)
{
   ETG_TRACE_USR1(("HMIModelComponent::onCourierMessage(ValidateNaviEntryConditionReqMsg)"));

   _infoStore.setIsDisclaimerAccepted(oMsg.GetIsDisclaimerConfirmed());
   return false;
}


#endif // HALL_TO_MDW_COM
