/**************************************************************************************
* @file         : ManeuverViewHandler.cpp
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "Common/Util/ImageUtils.h"
#include "Map/MapScreenDataUtils.h"
#include "ManeuverViewHandler.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ManeuverViewHandler.cpp.trc.h"
#endif

#if defined (HALL_TO_MDW_COM)


ManeuverViewHandler::ManeuverViewHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
{
}


ManeuverViewHandler::~ManeuverViewHandler()
{
}


void ManeuverViewHandler::initialize()
{
   _navMiddleware.registerGuidancePropertyUpdateCallback(*this);
   _navMiddleware.registerMapPropertyUpdateCallback(*this);
   _infoStore.registerDataPropertyUpdateCallback(*this);
}


void ManeuverViewHandler::deinitialize()
{
   _navMiddleware.unregisterGuidancePropertyUpdateCallback(*this);
   _navMiddleware.unregisterMapPropertyUpdateCallback(*this);
   _infoStore.unregisterDataPropertyUpdateCallback(*this);
}


void ManeuverViewHandler::onPropertyUpdateManeuverViewChanged()
{
   ETG_TRACE_USR4(("ManeuverViewHandler::onPropertyUpdateManeuverViewChanged()"));
   if (_navMiddleware.isGuidanceActive())
   {
      processJunctionView();
      setIntersectionMapMode(isIntersectionMapActive(_navMiddleware));
   }
}


void ManeuverViewHandler::onPropertyUpdateMapVideoStreamStatusChanged()
{
   if (_navMiddleware.isGuidanceActive() && isIntersectionMapActive(_navMiddleware))
   {
      navmiddleware::MapVideoStreamStatusInfo streamInfo = _navMiddleware.getMapVideoStreamStatusInfo();
      navmiddleware::MapVideoStreamStatusInfo::MapVideoStreamStatus status = streamInfo.getMapVideoStreamStatus();
      ETG_TRACE_USR4(("ManeuverViewHandler::onPropertyUpdateMapVideoStreamStatusChanged, Stream status : %d", status));

      bool enableIntersectionMapMode = (navmiddleware::MapVideoStreamStatusInfo::MAP_VIDEO_STREAM_STATUS__AVAILABLE == status) ? false : true;
      setIntersectionMapMode(enableIntersectionMapMode);
   }
}


bool ManeuverViewHandler::onPropertyUpdateMapCameraViewModeChanged()
{
   bool isUpdateRequired = false;
   bool isVisible = false;
   bool isJunctionView = isJunctionViewActive(_navMiddleware);
   const enMapCameraAndViewModeMode& mapCameraViewMode = _infoStore.getMapCameraViewMode();
   ETG_TRACE_USR4(("ManeuverViewHandler::onPropertyUpdateMapCameraViewModeChanged(), isJunctionViewActive : %d, MapCameraViewMode : %d", isJunctionView, mapCameraViewMode));

   if ((MAP_CAMERA_AND_VIEW_MODE_MAP_JUNCTION_AND_INTERSECTION_SCREEN == mapCameraViewMode) && (isJunctionView))
   {
      // On moving from any other map view such as route overview/map scroll to junction view
      if (!isJunctionViewVisible())
      {
         isUpdateRequired = true;
         isVisible = true;
      }
   }
   else
   {
      // On moving from Junction view to any other map view such as route overview/map scroll
      if (isJunctionViewVisible())
      {
         isUpdateRequired = true;
         isVisible = false;
      }
   }
   if (true == isUpdateRequired)
   {
      (*_junctionViewImageData).mHlgImageVisible = isVisible;
      (*_junctionViewImageData).mToggleLaneGuidanceVisibility = !isVisible;
      _junctionViewImageData.MarkItemModified(ItemKey::HlgImageData::HlgImageVisibleItem);
      _junctionViewImageData.MarkItemModified(ItemKey::HlgImageData::ToggleLaneGuidanceVisibilityItem);
      sendJunctionViewImageData();
   }
   return true;
}


void ManeuverViewHandler::processJunctionView()
{
   ETG_TRACE_USR4(("ManeuverViewHandler::processJunctionView()"));
   if (isJunctionViewActive(_navMiddleware))
   {
      if (updateJunctionViewImage())
      {
         setJunctionMapMode(true);
      }
      else
      {
         setJunctionMapMode(false);
      }
   }
   else
   {
      setJunctionMapMode(false);
   }
}


void ManeuverViewHandler::setJunctionMapMode(bool isJunctionView)
{
   const enMapCameraAndViewModeMode& mapCameraViewMode = _infoStore.getMapCameraViewMode();
   ETG_TRACE_USR4(("ManeuverViewHandler::setJunctionMapMode(), isJunctionView : %d, MapCameraViewMode : %d", isJunctionView, mapCameraViewMode));

   bool isUpdateRequired = false;
   if ((*_junctionViewImageData).mHlgImageVisible != isJunctionView)
   {
      if (isJunctionView)
      {
         if (MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN == mapCameraViewMode)
         {
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(SetMapCameraModeReqMsg)(MAP_CAMERA_AND_VIEW_MODE_MAP_JUNCTION_AND_INTERSECTION_SCREEN)));
         }
      }
      else
      {
         if (MAP_CAMERA_AND_VIEW_MODE_MAP_JUNCTION_AND_INTERSECTION_SCREEN == mapCameraViewMode)
         {
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(SetMapCameraModeReqMsg)(MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN)));
         }
      }
   }
}


bool ManeuverViewHandler::isJunctionViewVisible()
{
   return (*_junctionViewImageData).mHlgImageVisible;
}


bool ManeuverViewHandler::updateJunctionViewImage()
{
   ETG_TRACE_USR4(("ManeuverViewHandler::updateJunctionViewImage"));

   if (_navMiddleware.getManeuverViewInfo().getBlobImageData() != NULL)
   {
      (*_junctionViewImageData).mHlgImage = loadImage((const char*)_navMiddleware.getManeuverViewInfo().getBlobImageData(), _navMiddleware.getManeuverViewInfo().getBlobImageDataSize(), false);
      _junctionViewImageData.MarkItemModified(ItemKey::HlgImageData::HlgImageItem);
      return sendJunctionViewImageData();
   }
   else
   {
      ETG_TRACE_USR4(("ManeuverViewHandler::updateJunctionViewImage Not Available"));
      return false;
   }
}


bool ManeuverViewHandler::sendJunctionViewImageData()
{
   bool retVal = true;
   if (_junctionViewImageData.HasModifiedItems())
   {
      ETG_TRACE_USR4(("ManeuverViewHandler::sendJunctionViewImageData()"));
      if (_junctionViewImageData.SendUpdate() == false)
      {
         ETG_TRACE_ERR(("_junctionViewImageData update failed!"));
         retVal = false;
      }
   }

   return retVal;
}


void ManeuverViewHandler::setIntersectionMapMode(bool isIntersectionView)
{
   const enMapCameraAndViewModeMode& mapCameraViewMode = _infoStore.getMapCameraViewMode();
   ETG_TRACE_USR4(("ManeuverViewHandler::setIntersectionMapMode(), isIntersectionView : %d, MapCameraViewMode : %d", isIntersectionView, mapCameraViewMode));
   if (isIntersectionView)
   {
      if ((navmiddleware::settings::MAP_HMI_SPLITMODE_INTERSECTION == _navMiddleware.getMapViewSettings().getMapHMISplitMode()))
      {
         // Condition 1: For switching from full map to intersection view
         // Condition 2: For switching from one intersection view to another when there are consecutive intersections in the route
         if ((MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN == mapCameraViewMode)
               || (MAP_CAMERA_AND_VIEW_MODE_MAP_JUNCTION_AND_INTERSECTION_SCREEN == mapCameraViewMode))
         {
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(SetMapCameraModeReqMsg)(MAP_CAMERA_AND_VIEW_MODE_MAP_JUNCTION_AND_INTERSECTION_SCREEN)));
         }
      }
   }
   else
   {
      if (MAP_CAMERA_AND_VIEW_MODE_MAP_JUNCTION_AND_INTERSECTION_SCREEN == mapCameraViewMode)
      {
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(SetMapCameraModeReqMsg)(MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN)));
      }
   }
}


#endif // HALL_TO_MDW_COM
