/**************************************************************************************
* @file         : FavoritesListHandler.h
* @author       : KanagaDurga Balakrishnan
* @addtogroup   :
* @brief        :
* @copyright    : (c) -2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "util/StrUtf8.h"
#include "AppHmi_NavigationStateMachine.h"
#include "FavoritesListHandler.h"
#include "hmi_trace_if.h"
#include "CgiExtensions/ImageLoader.h"
#include "Common/Util/StringUtils.h"
#include "Route/RouteDataUtils.h"
#include "AddressDetailedInfo/AddressDetailedInfo.h"

using namespace navmiddleware;

//Identifier for FavAddfav Scene
static const Courier::Identifier  IdClickSaveWithDisplayedName = Courier::Identifier("Layer_SaveWithDisplayedName/ButtonWidget");
static const Courier::Identifier  IdClickSaveAsHome = Courier::Identifier("Layer_SaveAsHome/ButtonWidget");

//Identifier for FavSetHome Scene
static const Courier::Identifier  IdClickCurrentpositionbtn = Courier::Identifier("Layer_CurrentPosition/ButtonWidget");

//Identifier for AdressIputHome Scene
static const Courier::Identifier  IdClickAddressHomebtn = Courier::Identifier("Layer_Guidance/ButtonWidget");

//Identifier for VehicleProfileInfo Scene
static const Courier::Identifier  IdClickVehicleInfobtn = Courier::Identifier("Layer_StartGuidance/ButtonWidget");

//List Identifier
static const char* const DATA_CONTEXT_LIST_FAVOTITE_ITEM = "FavoriteEntryListItem";
static const char* const DATA_CONTEXT_LIST_REPLACE_FAVORITE_ITEM = "Replace_FavoriteEntryListItem";

static const char* IMG_ADDRESSINPUT_HOME_ICON_ENABLED = "AppHmi_NavigationModule#Images#N_NewDest_AddressInput#icon_Home_normalBmp";
static const char* IMG_ADDRESSINPUT_HOME_ICON_TOUCHED = "AppHmi_NavigationModule#Images#N_NewDest_AddressInput#icon_Home_activeBmp";
static const char* IMG_ADDRESSINPUT_HOME_ICON_DISABLED = "AppHmi_NavigationModule#Images#N_NewDest_AddressInput#icon_Home_disabledBmp";
static const char* IMG_FAVORITE_ICON_NORMAL = "AppHmi_NavigationModule#Images#N_Favorites#icon_Favorites_2_normal";
static const char* IMG_FAVORITE_ICON_TOUCHED = "AppHmi_NavigationModule#Images#N_Favorites#icon_Favorites_2_touched";
static const char* IMG_HOME_ICON_NOMRAL = "AppHmi_NavigationModule#Images#N_Favorites#icon_Home_normal";
static const char* IMG_HOME_ICON_TOUCHED = "AppHmi_NavigationModule#Images#N_Favorites#icon_Home_touched";
static const char* IMG_GUIDANCE_ICON_ENABLED = "AppHmi_NavigationModule#Images#Common#Icon_Guidance_touchedBmp";
static const char* IMG_GUIDANCE_ICON_DISABLED = "AppHmi_NavigationModule#Images#Common#Icon_Guidance_normalBmp";

static const Candera::String REPLACE_FAV_TXT_LINE1 = LANGUAGE_STRING(TextId_0x067F, "Maximum number of entries reached in the");
static const Candera::String REPLACE_FAV_TXT_LINE2 = LANGUAGE_STRING(TextId_0x0680, "favourites list. Would you like to replace an");
static const Candera::String REPLACE_FAV_TXT_LINE3 = LANGUAGE_STRING(TextId_0x0E36, "existing one?");

static const int EDIT_ICON = 1;
static const int FAVORITE_ICON = 0;
static const int HIDE_DELETE_BUTTON = 0;
static const int SHOW_DELETE_BUTTON = 1;
static const int DELETE_SELECTED_FAVORITE = 2;
static const int DELETE_ALL_FAVORITES = 3;
static const int DELETE_SELECTED_HOME_FAVORITE = 4;
static const int MAXIMUM_FAVORITE = 30;

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/FavoritesListHandler.cpp.trc.h"
#endif

using namespace navmiddleware;

FavoritesListHandler::FavoritesListHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , _listID(0)
   , _startIndex(0)
   , _windowElementSize(0)
   , _isAddressBookModified(false)
   , _editNameOKPress("")
   , _isAddressHomeClicked(false)
   , _previousScene(0)
   , _isEntryReplace(false)
{
}


FavoritesListHandler::~FavoritesListHandler()
{
}


void FavoritesListHandler::initialize()
{
   ETG_TRACE_USR4(("FavoritesListHandler::initialize()"));

   _navMiddleware.registerDestinationMemoryPropertyUpdateCallback(*this);
   _infoStore.registerDataPropertyUpdateCallback(*this);
   _infoStore.setAddressBookSortByCategory(navmiddleware::DestinationMemoryEntry::SORTINGALGORITHM_ALPHABETICAL_BY_ALIAS_ASC);
}


void FavoritesListHandler::deinitialize()
{
   _navMiddleware.unregisterDestinationMemoryPropertyUpdateCallback(*this);
   _infoStore.unregisterDataPropertyUpdateCallback(*this);
}


bool FavoritesListHandler::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   ::Courier::UInt32 listID = oMsg.GetListId();

   if ((listID == LIST_ID_NAV_FAVORITES) || (listID == LIST_ID_NAV_REPLACE_FAVORITES))
   {
      ETG_TRACE_USR4(("FavoritesListHandler::onCourierMessage(ListDateProviderReqMsg), ListID : %d", listID));
      _startIndex = oMsg.GetStartIndex();
      _windowElementSize = oMsg.GetWindowElementSize();
      _listID = listID;

      ETG_TRACE_USR4(("FavoritesListHandler::onCourierMessage(ListDateProviderReqMsg(listID %d, startIndex %d, windowSize %d))", _listID, _startIndex, _windowElementSize));

      _infoStore.setListFocusIndex(0);
      _infoStore.setListStartIndex(_startIndex);

      navmiddleware::ValidValue<navmiddleware::DestinationMemoryAttributesGroup> groupFilter;
      if (0 == _startIndex)
      {
         // Since first entry in HMI Favorites list is HOME entry, one less address book entry is requested from navmiddleware
         _navMiddleware.requestEntryList(DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK, groupFilter, _startIndex, _windowElementSize - 1, _infoStore.getAddressBookSortByCategory());
      }
      else
      {
         _navMiddleware.requestEntryList(DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK, groupFilter, _startIndex - 1, _windowElementSize, _infoStore.getAddressBookSortByCategory());
      }
      return true;
   }

   return false;
}


bool FavoritesListHandler::onCourierMessage(const SetPreviosulyActiveSceneName& oMsg)
{
   _previousScene = oMsg.GetSceneName();

   return true;
}


void FavoritesListHandler::onPropertyUpdateDestinationMemoryStoreEntryStatusChanged()
{
   DestinationMemoryStoreEntryStatus destinationMemoryStoreEntryStatus = _navMiddleware.getStoreEntryStatus();
   ETG_TRACE_USR4(("FavoritesListHandler::onPropertyUpdateDestinationMemoryStoreEntryStatusChanged(), StoreEntryStatus : %d", destinationMemoryStoreEntryStatus.getStatus()));

   if (destinationMemoryStoreEntryStatus.getStatus() == DestinationMemoryStoreEntryStatus::STOREENTRYSTATUS__OK)
   {
      // Request details of the most recent destination added to favorite list
      _navMiddleware.requestEntryDetails(destinationMemoryStoreEntryStatus.getStoredEntryId());
   }
}


void FavoritesListHandler::onPropertyUpdateDestinationMemoryEntryDetailsAvailable()
{
   navmiddleware::DestinationMemoryEntryCategory entryCategory = _infoStore.getDestinationMemoryStatus()._category;
   if ((navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__HOME == entryCategory)
         || (navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK == entryCategory))
   {
      ETG_TRACE_USR4(("FavoritesListHandler::onPropertyUpdateDestinationMemoryEntryDetailsAvailable()"));

      const DestinationMemoryEntry& destinationMemoryEntry = _navMiddleware.getEntryDetails();
      if (navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__HOME == entryCategory)
      {
         ETG_TRACE_USR4(("FavoritesListHandler::onPropertyUpdateDestinationMemoryEntryDetailsAvailable(), Home Id : %d", destinationMemoryEntry.getId()));

         // Whenever there is a change in HOME entry, requestEntryList API is invoked for HOME
         // so that the property update HOME_LIST_CHANGED is handled in both FavoritesListHandler and HMIModelNavigationService
         navmiddleware::ValidValue<navmiddleware::DestinationMemoryAttributesGroup> groupFilter;
         _navMiddleware.requestEntryList(DESTINATION_MEMORY_ENTRY_CATEGORY__HOME,
                                         groupFilter,
                                         (uint32_t)_startIndex,
                                         -1,    // numElements set to -1 so that all elements available will be returned
                                         _infoStore.getAddressBookSortByCategory());

         POST_MSG((COURIER_MESSAGE_NEW(ShowDetailedAddressMsg)()));
         POST_MSG((COURIER_MESSAGE_NEW(SaveAsHomeBtnSelectionMsg)()));
      }
      else if (navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK == entryCategory)
      {
         _infoStore.setDestinationAttributesInfo(destinationMemoryEntry.getAttributes().getValue());

         if (true == _isAddressBookModified)
         {
            _infoStore.setDestinationAttributesInfoName(_editNameOKPress.GetCString());

            ETG_TRACE_USR4(("FavoritesListHandler::onPropertyUpdateDestinationMemoryEntryDetailsAvailable(), Modified favorite name for Address Id : %d", destinationMemoryEntry.getId()));
            _navMiddleware.modifyEntryAttributes(destinationMemoryEntry.getId(), _infoStore.getDestinationAttributesInfo());
            _isAddressBookModified = false;
         }
         else
         {
            _infoStore.setDestinationAttributesInfoName(destinationMemoryEntry.getDestination().getName());
         }

         POST_MSG((COURIER_MESSAGE_NEW(ShowDetailedAddressMsg)()));
      }
   }
}


void FavoritesListHandler::onPropertyUpdateDestinationMemoryAddressBookEntryRemoved()
{
   ETG_TRACE_USR4(("FavoritesListHandler::onPropertyUpdateDestinationMemoryAddressBookEntryRemoved()"));

   // After address book entry to be replaced is removed, new address book entry is stored
   const navmiddleware::DestinationMemoryStatus& destinationMemoryStatus = _navMiddleware.getDestinationMemoryStatus();
   if (true == _isEntryReplace)
   {
      _isEntryReplace = false;
      _infoStore.setDestinationMemoryStatus(InfoStoreBase::DestinationMemoryStatus(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK, UINT64_MAX, ""));
      _navMiddleware.storeEntry(DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK);
   }
}


void FavoritesListHandler::onPropertyUpdateDestinationMemoryHomeEntryRemoved()
{
   ETG_TRACE_USR4(("FavoritesListHandler::onPropertyUpdateDestinationMemoryHomeEntryRemoved()"));

   _infoStore.setDestinationMemoryHomeStatus(false);

   // After home entry to be replaced is removed, new home entry is stored
   if (true == _isEntryReplace)
   {
      _isEntryReplace = false;
      _infoStore.setDestinationMemoryStatus(InfoStoreBase::DestinationMemoryStatus(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__HOME, UINT64_MAX, ""));
      _navMiddleware.storeEntry(DESTINATION_MEMORY_ENTRY_CATEGORY__HOME);
   }
}


void FavoritesListHandler::onPropertyUpdateDestinationMemoryAddressBookStatusChanged()
{
   static bool isFirstEntry = true;
   const navmiddleware::DestinationMemoryStatus& destinationMemoryStatus = _navMiddleware.getDestinationMemoryStatus();

   _infoStore.setNumberOfAddressBookEntries(destinationMemoryStatus.getNumberOfAddressBookEntries());
   _infoStore.setNumberOfStoredRouteEntries(destinationMemoryStatus.getNumberOfRoutes());

   ETG_TRACE_USR4(("FavoritesListHandler::onPropertyUpdateDestinationMemoryAddressBookStatusChanged(), Number of favorites : %d", destinationMemoryStatus.getNumberOfAddressBookEntries()));
   if (destinationMemoryStatus.getNumberOfAddressBookEntries() < MAXIMUM_FAVORITE)
   {
      EXT_bIsMaxFavReached = FALSE;
   }
   else
   {
      EXT_bIsMaxFavReached = TRUE;
   }

   // After navmiddleware is first initialized, this property update is received after which request is sent to fetch home entry details
   if (true == isFirstEntry)
   {
      isFirstEntry = false;
      navmiddleware::ValidValue<navmiddleware::DestinationMemoryAttributesGroup> groupFilter;
      _navMiddleware.requestEntryList(DESTINATION_MEMORY_ENTRY_CATEGORY__HOME,
                                      groupFilter,
                                      (uint32_t)_startIndex,
                                      -1,    // numElements set to -1 so that all elements available will be returned
                                      _infoStore.getAddressBookSortByCategory());
   }
}


void FavoritesListHandler::onPropertyUpdateDestinationMemoryHomeListChanged()
{
   ETG_TRACE_USR4(("FavoritesListHandler::onPropertyUpdateDestinationMemoryHomeListChanged()"));

   const navmiddleware::DestinationMemoryEntryList& destinationMemoryEntryList = _navMiddleware.getEntryList();
   const ::std::vector<navmiddleware::DestinationMemoryEntry>& infos = destinationMemoryEntryList.getEntryList();

   if (0 < infos.size())
   {
      navmiddleware::DestinationMemoryEntry homeEntry = infos.at(0);
      std::vector<PosWGS84<double> > positionVector;
      positionVector.push_back(PosWGS84<double>(homeEntry.getDestination().getLongitude(), homeEntry.getDestination().getLatitude()));

      ETG_TRACE_USR4(("FavoritesListHandler::onPropertyUpdateDestinationMemoryHomeListChanged(), Home Id : %d", homeEntry.getId()));
      _infoStore.setDestinationMemoryHomeStatus(true);
      _infoStore.setDestinationMemoryHomeId(homeEntry.getId());
      _infoStore.setDestinationHomeAttribute(homeEntry.getAttributes().getValue());
      _infoStore.setDestinationHomeCoordinates(positionVector);
   }
}


void FavoritesListHandler::onPropertyUpdateDestinationMemoryAddressBookListChanged()
{
   ETG_TRACE_USR4(("FavoritesListHandler::onPropertyUpdateDestinationMemoryAddressBookListChanged(), listId = %d", _listID));
   if (_listID)
   {
      tSharedPtrDataProvider dataProvider;
      if (_listID == LIST_ID_NAV_FAVORITES)
      {
         dataProvider = getAddressBookDestinationListDataProvider(_listID, DATA_CONTEXT_LIST_FAVOTITE_ITEM, _startIndex, _windowElementSize);
      }
      else if (_listID == LIST_ID_NAV_REPLACE_FAVORITES)
      {
         _isEntryReplace = false;
         dataProvider = getAddressBookDestinationListDataProvider(_listID, DATA_CONTEXT_LIST_REPLACE_FAVORITE_ITEM, _startIndex, _windowElementSize);
      }

      // Null pointer check for data Provider
      if (!dataProvider.PointsToNull())
      {
         POST_MSG((COURIER_MESSAGE_NEW(ListDateProviderResMsg)(dataProvider)));
      }
      else
      {
         ETG_TRACE_ERR(("FavoritesListHandler::onPropertyUpdateDestinationMemoryAddressBookListChanged(), dataprovider is null "));
      }
      _listID = 0;
   }
}


tSharedPtrDataProvider FavoritesListHandler::getAddressBookDestinationListDataProvider(unsigned int listID, const char* itemID, unsigned int startIndex, unsigned int windowElementSize) const
{
   ETG_TRACE_USR4(("FavoritesListHandler::getAddressBookDestinationListDataProvider(), listId = %d", listID));

   ListDataProviderBuilder listBuilder(listID, itemID);
   const navmiddleware::DestinationMemoryEntryList& destinationMemoryEntryList = _navMiddleware.getEntryList();
   const ::std::vector<navmiddleware::DestinationMemoryEntry>& infos = destinationMemoryEntryList.getEntryList();
   const navmiddleware::DestinationMemoryStatus& destinationMemoryStatus = _navMiddleware.getDestinationMemoryStatus();

   ::std::vector<navmiddleware::DestinationMemoryEntry>::const_iterator info = infos.begin();

   unsigned int idx = startIndex;
   unsigned int actDestListSize = destinationMemoryStatus.getNumberOfAddressBookEntries() + 1;   // One home entry is added to the address book entries to get the HMI Favorites list size

   if ((itemID == DATA_CONTEXT_LIST_FAVOTITE_ITEM) || (itemID == DATA_CONTEXT_LIST_REPLACE_FAVORITE_ITEM))
   {
      StoredLocationIconData item;
      // By checking the list item index 0 and updating static text Home
      if (0 == idx)
      {
         ETG_TRACE_USR4(("FavoritesListHandler::getAddressBookDestinationListDataProvider(), List index : 0, Location name : Home"));
         item.mLocationNameInfo = LANGUAGE_STRING(TextId_0x0932, "Home");
         item.mLocationIconNormalBitmap = ImageLoader::getAssetBitmapImage(IMG_HOME_ICON_NOMRAL);
         item.mLocationIconPressedBitmap = ImageLoader::getAssetBitmapImage(IMG_HOME_ICON_TOUCHED);
         listBuilder.AddItem(
            idx,
            0UL,
            itemID)
         .AddDataBindingUpdater<StoredLocationIconDataBindingSource>(item);
         idx++;
      }

      for (; (idx < startIndex + windowElementSize) && (info != infos.end()); ++info, ++idx)
      {
         ETG_TRACE_USR4(("FavoritesListHandler::getAddressBookDestinationListDataProvider(), List index : %d", idx));

         const navmiddleware::ValidValue< navmiddleware::DestinationMemoryEntryAttributes >& destinationMemoryEntryAttributes = info->getAttributes();
         const navmiddleware::DestinationMemoryEntryAttributes& destinationAttributeDetails = destinationMemoryEntryAttributes.getValue();

         item.mLocationNameInfo = destinationAttributeDetails.getName().c_str();
         item.mLocationIconNormalBitmap = ImageLoader::getAssetBitmapImage(IMG_FAVORITE_ICON_NORMAL);
         item.mLocationIconPressedBitmap = ImageLoader::getAssetBitmapImage(IMG_FAVORITE_ICON_TOUCHED);
         listBuilder.AddItem(
            idx,
            0UL,
            itemID)
         .AddDataBindingUpdater<StoredLocationIconDataBindingSource>(item);

         ETG_TRACE_USR4(("FavoritesListHandler::getAddressBookDestinationListDataProvider(), Location name : %s", item.mLocationNameInfo.GetCString()));
      }
   }
   ETG_TRACE_USR4(("FavoritesListHandler::getAddressBookDestinationListDataProvider(), listSize = %d", actDestListSize));
   return listBuilder.CreateDataProvider(startIndex, actDestListSize);
}


bool FavoritesListHandler::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   bool isMessageProcessed = false;

   if (oMsg.GetEnReaction() == enRelease)
   {
      ListProviderEventInfo info;
      ListProviderEventInfo::GetItemIdentifierInfo(oMsg.GetSender(), info);

      unsigned int listIdx = (unsigned int)info.getHdlRow();
      listIdx = listIdx - _startIndex;
      const Courier::Identifier senderInfo = oMsg.GetSender();
      const Courier::ViewId sceneName = oMsg.GetView();

      if (info.getListId() == LIST_ID_NAV_FAVORITES)
      {
         ButtonClickOnFavoriteList(listIdx);
      }
      else if (info.getListId() == LIST_ID_NAV_REPLACE_FAVORITES)
      {
         ButtonClickOnReplaceFavoriteList(listIdx);
      }
      else
      {
         ButtonClickOnFavoritesScenes(senderInfo, sceneName);
      }
   }

   return isMessageProcessed;
}


void FavoritesListHandler::ButtonClickOnFavoriteList(unsigned int listIdx)
{
   ETG_TRACE_USR4(("FavoritesListHandler::ButtonClickOnFavoriteList(), List index : %d", listIdx));

   const navmiddleware::DestinationMemoryStatus& destinationMemoryStatus = _navMiddleware.getDestinationMemoryStatus();
   const navmiddleware::DestinationMemoryEntryList& destinationMemoryEntryList = _navMiddleware.getEntryList();
   const ::std::vector<navmiddleware::DestinationMemoryEntry>& destinationInfoVector = destinationMemoryEntryList.getEntryList();

   // on button click in favorite, update infostore category to home/address
   if (listIdx == 0)
   {
      //if home entries are not available then show FavSetHome View
      if (destinationMemoryStatus.getNumberOfHomeEntries() == 0)
      {
         POST_MSG((COURIER_MESSAGE_NEW(FavoritesHometouchMsg)()));
      }
      else
      {
         const InfoStoreBase::DestinationMemoryHomeWorkInfo& homeInfo = _infoStore.getDestinationMemoryHomeInfo();
         InfoStoreBase::CoordinatesToBeShownInMap& coordinatesToBeShownInMap = _infoStore.getCoordinatesToBeShownInMap();
         coordinatesToBeShownInMap._coordinates = homeInfo._coordinates;
         _infoStore.setDestinationMemoryStatus(InfoStoreBase::DestinationMemoryStatus(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__HOME,
                                               homeInfo._id,
                                               homeInfo._destinationMemoryAttributesInfo.getName()));
         _navMiddleware.setLocationWithDestinationMemoryId(homeInfo._id);
         _infoStore.setIsDetailInfoRequested(true, InfoStoreBase::DETAIL_INFO_FAVORITES);
         _navMiddleware.requestLocationAttributes();

         ETG_TRACE_USR4(("FavoritesListHandler::ButtonClickOnFavoriteList(), Home ID : %d", homeInfo._id));
         POST_MSG((COURIER_MESSAGE_NEW(FavHomeBtnSelectionMsg)()));
      }
   }
   else
   {
      if (0 == _startIndex)
      {
         listIdx = listIdx - 1;   //decrementing the list index by 1 because list item starts from index 1, index 0 is Home
      }
      if (listIdx < destinationInfoVector.size())
      {
         navmiddleware::DestinationMemoryEntry destinationMemoryEntry = destinationInfoVector[listIdx];
         const navmiddleware::ValidValue< navmiddleware::DestinationMemoryEntryAttributes >& destinationAttributes = destinationMemoryEntry.getAttributes();
         const navmiddleware::DestinationMemoryEntryAttributes& destinationAttributeDetails = destinationAttributes.getValue();

         // use a stream string helper because of missing target support for printing out two separate strings within the ETG trace
         DestinationMemoryEntry::Id destinationId = destinationMemoryEntry.getId();
         std::stringstream traceOut;
         traceOut << "Address ID " << destinationId
                  << ", location name " << destinationAttributeDetails.getName()
                  << ", longitude " << destinationMemoryEntry.getDestination().getLongitude()
                  << " and latitude " << destinationMemoryEntry.getDestination().getLatitude();
         ETG_TRACE_USR4(("FavoritesListHandler::ButtonClickOnFavoriteList(), Details : %s", traceOut.str().c_str()));

         std::vector<PosWGS84<double> > positionVector;
         positionVector.push_back(PosWGS84<double>(destinationMemoryEntry.getDestination().getLongitude(), destinationMemoryEntry.getDestination().getLatitude()));
         InfoStoreBase::CoordinatesToBeShownInMap& coordinatesToBeShownInMap = _infoStore.getCoordinatesToBeShownInMap();
         coordinatesToBeShownInMap._coordinates = positionVector;

         _infoStore.setDestinationMemoryStatus(InfoStoreBase::DestinationMemoryStatus(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK,
                                               destinationMemoryEntry.getId(), destinationAttributeDetails.getName()));
         _navMiddleware.setLocationWithDestinationMemoryId(destinationId);
         _infoStore.setIsDetailInfoRequested(true, InfoStoreBase::DETAIL_INFO_FAVORITES);
         _navMiddleware.requestLocationAttributes();
      }
   }
}


void FavoritesListHandler::ButtonClickOnReplaceFavoriteList(unsigned int listIdx)
{
   ETG_TRACE_USR4(("FavoritesListHandler::ButtonClickOnReplaceFavoriteList(), List index : %d", listIdx));

   const navmiddleware::DestinationMemoryEntryList& destinationMemoryEntryList = _navMiddleware.getEntryList();
   const ::std::vector<navmiddleware::DestinationMemoryEntry>& destinationInfoVector = destinationMemoryEntryList.getEntryList();

   if (listIdx == 0)
   {
      if (true == _infoStore.getDestinationMemoryHomeInfo()._status)
      {
         // If home entry is to replaced, last stored home entry is removed and after removal is successful, new location is stored as home.
         // Refer onPropertyUpdateDestinationMemoryHomeEntryRemoved()
         _isEntryReplace = true;
         _navMiddleware.removeAllEntries(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__HOME);
      }
      else // If home entry is not set, the new location is saved as HOME
      {
         _infoStore.setDestinationMemoryStatus(InfoStoreBase::DestinationMemoryStatus(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__HOME, UINT64_MAX, ""));
         _navMiddleware.storeEntry(DESTINATION_MEMORY_ENTRY_CATEGORY__HOME);
      }

   }
   else
   {
      if (0 == _startIndex)
      {
         listIdx = listIdx - 1;   //decrementing the list index by 1 because list item starts from index 1, index 0 is Home
      }
      if (listIdx < destinationInfoVector.size())
      {
         _isEntryReplace = true;
         navmiddleware::DestinationMemoryEntry destinationMemoryEntry = destinationInfoVector[listIdx];
         removeFavoritesEntry(destinationMemoryEntry.getId());
         ETG_TRACE_USR4(("FavoritesListHandler::ButtonClickOnReplaceFavoriteList(), Replacing favorite entry with Id : %d", destinationMemoryEntry.getId()));
      }
   }
   //Detailed info should be shown with current selected address
   POST_MSG((COURIER_MESSAGE_NEW(SaveWithDisplayedNameBtnSelectionMsg)()));
}


void FavoritesListHandler::ButtonClickOnFavoritesScenes(const Courier::Identifier senderInfo, const Courier::ViewId sceneName)
{
   const Courier::ViewId ADDRESS_DETAILEDINFO_SCENENAME = Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#N_Address_DetailedInfo");
   const Courier::ViewId DESTINATION_INPUT_SCENENAME    = Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#N_NewDest_AddressInput");
   const Courier::ViewId VEHICLE_PROFILEINFO_SCENENAME  = Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_Dest_VehicleInfo");

   //When Save with displayed name button in N_Fav_AddFavorites is clicked
   if (senderInfo == IdClickSaveWithDisplayedName)
   {
      ETG_TRACE_USR4(("FavoritesListHandler::ButtonClickOnFavoritesScenes(), Save with displayed name"));

      _infoStore.setDestinationMemoryStatus(InfoStoreBase::DestinationMemoryStatus(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK, UINT64_MAX, ""));
      _navMiddleware.storeEntry(DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK);

      //Detailed info should be shown by saving the address entry in Favorites scene
      POST_MSG((COURIER_MESSAGE_NEW(SaveWithDisplayedNameBtnSelectionMsg)()));
      updateDetailedInfo();
   }

   else if (senderInfo == IdClickSaveAsHome)
   {
      ETG_TRACE_USR4(("FavoritesListHandler::ButtonClickOnFavoritesScenes(), Save as home"));

      const navmiddleware::DestinationMemoryStatus& destinationMemoryStatus = _navMiddleware.getDestinationMemoryStatus();
      if (destinationMemoryStatus.getNumberOfHomeEntries() == 0)
      {
         _infoStore.setDestinationMemoryStatus(InfoStoreBase::DestinationMemoryStatus(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__HOME, UINT64_MAX, ""));
         _navMiddleware.storeEntry(DESTINATION_MEMORY_ENTRY_CATEGORY__HOME);
         updateDetailedInfo();
      }
      else
      {
         //popup should be shown to replace the home with current address as home
         POST_MSG((COURIER_MESSAGE_NEW(ShowReplaceFavPopup)()));
      }
   }

   else if (senderInfo == IdClickCurrentpositionbtn)
   {
      ETG_TRACE_USR4(("FavoritesListHandler::ButtonClickOnFavoritesScenes(), Save current position as home"));

      // Get the current vehicle position and save the location
      const navmiddleware::PositionStatusInfo& position = _navMiddleware.getPositionStatusInfo();
      _navMiddleware.setLocationWithCoordinates(position.getLatitude(), position.getLongitude());
      _infoStore.setDestinationMemoryStatus(InfoStoreBase::DestinationMemoryStatus(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__HOME, UINT64_MAX, ""));
      _navMiddleware.storeEntry(DESTINATION_MEMORY_ENTRY_CATEGORY__HOME);

      POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(EDIT_ICON, SHOW_DELETE_BUTTON, 1, 0)));
      POST_MSG((COURIER_MESSAGE_NEW(FavHomeBackCurPosMsg)()));
   }

   else if (senderInfo == IdClickAddressHomebtn || senderInfo == IdClickVehicleInfobtn)
   {
      if (sceneName == DESTINATION_INPUT_SCENENAME)
      {
         //Set the current entered address as destination for RouteGuidance when Add Home is not active
         _navMiddleware.setLocationWithCurrentAddressInput();

         //Store latitude longtitude information of selected address input
         const navmiddleware::AddressInputInfo& addresInputInfo = _navMiddleware.getFormBasedAddressInputInfo();
         _infoStore.setLatitude(addresInputInfo.getCurrentCoordinates().getValue().getLatitude());
         _infoStore.setLongitude(addresInputInfo.getCurrentCoordinates().getValue().getLongitude());

         if (_isAddressHomeClicked == true)
         {
            _infoStore.setDestinationMemoryStatus(InfoStoreBase::DestinationMemoryStatus(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__HOME, UINT64_MAX, ""));
            _navMiddleware.storeEntry(DESTINATION_MEMORY_ENTRY_CATEGORY__HOME);
            POST_MSG((COURIER_MESSAGE_NEW(AddressInputHometouchMsg)()));
         }
         else
         {
            recalculateRouteGuidance(_navMiddleware, _infoStore, true);
         }
      }
      else if (sceneName == ADDRESS_DETAILEDINFO_SCENENAME || sceneName == VEHICLE_PROFILEINFO_SCENENAME)
      {
         recalculateRouteGuidance(_navMiddleware, _infoStore, true);
      }
   }
   else
   {
      // do nothing
   }
}


void FavoritesListHandler::updateDetailedInfo()
{
   ETG_TRACE_USR4(("FavoritesListHandler::updateDetailedInfo()"));

   switch (_previousScene)
   {
      case NAVI_HMI_SM_C_FAVORITE_DETAILEDINFO:
      case NAVI_HMI_SM_C_HOME_DETAILEDINFO:
      {
         POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(1, true, true, 0)));
         break;
      }
      case NAVI_HMI_SM_C_WAYPOINT_DETAILEDINFO:
      {
         POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(FAVORITE_ICON, true, false, 0)));
         break;
      }
      case NAVI_HMI_SM_C_LASTDESTINATION_DETAILEDINFO:
      {
         POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(FAVORITE_ICON, true, true, 0)));
         break;
      }
      case NAVI_HMI_SM_C_ROUTE_DESTINATIONINFO:
      {
         POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(FAVORITE_ICON, HIDE_DELETE_BUTTON, false, 0)));
         break;
      }
      default:
      {
         POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(FAVORITE_ICON, HIDE_DELETE_BUTTON, 1, 0)));
         break;
      }
   }
}


bool FavoritesListHandler::onCourierMessage(const ReplaceHomeFavReqMsg&)
{
   ETG_TRACE_USR4(("FavoritesListHandler::onCourierMessage(ReplaceHomeFavReqMsg())"));

   // If home entry is to be replaced, last store home entry is removed and after removal is successful, new location is stored as home.
   // Refer onPropertyUpdateDestinationMemoryHomeEntryRemoved()
   _isEntryReplace = true;
   _navMiddleware.removeAllEntries(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__HOME);

   return true;
}


bool FavoritesListHandler::onCourierMessage(const FavDeleteReqMsg& oMsg)
{
   ETG_TRACE_USR4(("FavoritesListHandler::onCourierMessage(FavDeleteReqMsg())"));

   unsigned int deleteType = oMsg.GetPopupType();
   bool isHomeEntry = oMsg.GetIsHomeEntry();

   if (deleteType == POPUP_FAVORITES_DELETE_ALL) // if delete all favorite button is pressed from popup
   {
      ETG_TRACE_USR4(("FavoritesListHandler::onCourierMessage(FavDeleteReqMsg()), Delete all favorites"));

      _navMiddleware.removeAllEntries(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK);
      _navMiddleware.removeAllEntries(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__HOME);
   }
   else if (deleteType == POPUP_FAVORITES_DELETE)
   {
      if (!isHomeEntry)
      {
         navmiddleware::DestinationMemoryEntry::Id deleteId = _infoStore.getDestinationMemoryStatus()._id;
         ETG_TRACE_USR4(("FavoritesListHandler::onCourierMessage(FavDeleteReqMsg()), Delete selected favorite, Address Id : %d", deleteId));
         removeFavoritesEntry(deleteId);
      }
      else
      {
         ETG_TRACE_USR4(("FavoritesListHandler::onCourierMessage(FavDeleteReqMsg()), Delete home"));
         _navMiddleware.removeAllEntries(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__HOME);
      }
   }
   else
   {
      //Do nothing
   }

   return true;
}


bool FavoritesListHandler::onCourierMessage(const MaxFavPopupVisibilityReqsMsg& oMsg)
{
   ETG_TRACE_USR4(("FavoritesListHandler::onCourierMessage(MaxFavPopupVisibilityReqsMsg()))"));
   // Trigger Detail info Pop up
   if (true == oMsg.GetIsPopupEnabled())
   {
      (*_multiLinePopupText).mThreeLinePopupText1 = REPLACE_FAV_TXT_LINE1;
      (*_multiLinePopupText).mThreeLinePopupText2 = REPLACE_FAV_TXT_LINE2;
      (*_multiLinePopupText).mThreeLinePopupText3 = REPLACE_FAV_TXT_LINE3;
      _multiLinePopupText.MarkAllItemsModified();
      _multiLinePopupText.SendUpdate();

      POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_ThreeLine_TwoButton"))));
   }
   else
   {
      POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_ThreeLine_TwoButton"))));
   }
   return true;
}


bool FavoritesListHandler::onCourierMessage(const SpellerOKBtnPressedMsg& oMsg)
{
   bool isMessageProcessed = false;
   const Courier::UInt32 sceneName = oMsg.GetSceneType();

   SpellerHandler* spellerHandler = SpellerHandler::getInstance();
   _editNameOKPress = SpellerHandler::getInstance()->GetEditFieldText();
   if (0 != strcmp(_editNameOKPress.GetCString(), ""))
   {
      _isAddressBookModified = true;
      if (AppHmi_NavigationModule_NavigationScenes_N_Favorites == sceneName)
      {
         ETG_TRACE_USR4(("FavoritesListHandler::onCourierMessage(SpellerOKBtnPressedMsg), Favorites scene"));
         _navMiddleware.requestEntryDetails(_infoStore.getDestinationMemoryStatus()._id);
      }
      else if (AppHmi_NavigationModule_NavigationScenes_N_Fav_AddFavorites == sceneName)
      {
         ETG_TRACE_USR4(("FavoritesListHandler::onCourierMessage(SpellerOKBtnPressedMsg), Add Favorites scene"));
         _infoStore.setDestinationMemoryStatus(InfoStoreBase::DestinationMemoryStatus(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK, UINT64_MAX, ""));
         _navMiddleware.storeEntry(DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK);
      }
   }

   return isMessageProcessed;
}


bool FavoritesListHandler::onCourierMessage(const EnableGuidanceButton_NewAddressReqMsg& oMsg)
{
   ETG_TRACE_USR4(("FavoritesListHandler::onCourierMessage(EnableGuidanceButton_NewAddressReqMsg()))"));

   if (oMsg.GetGuidanceEnable() == 1)
   {
      (*_addressInputButtonVisibility).mDetailInfoButtonVisibility = true;
      (*_addressInputButtonVisibility).mGuidance_HomeNormalIconEnabled = ImageLoader::getAssetBitmapImage(IMG_GUIDANCE_ICON_ENABLED);
      (*_addressInputButtonVisibility).mGuidance_HomePressedIconEnabled = ImageLoader::getAssetBitmapImage(IMG_GUIDANCE_ICON_ENABLED);
      (*_addressInputButtonVisibility).mGuidance_HomeIconDisabled = ImageLoader::getAssetBitmapImage(IMG_GUIDANCE_ICON_DISABLED);
      _isAddressHomeClicked = false;
   }
   else
   {
      //if home address input then show home icon and detailed info button should be disabled
      (*_addressInputButtonVisibility).mDetailInfoButtonVisibility = false;
      (*_addressInputButtonVisibility).mGuidance_HomeNormalIconEnabled = ImageLoader::getAssetBitmapImage(IMG_ADDRESSINPUT_HOME_ICON_ENABLED);
      (*_addressInputButtonVisibility).mGuidance_HomePressedIconEnabled = ImageLoader::getAssetBitmapImage(IMG_ADDRESSINPUT_HOME_ICON_TOUCHED);
      (*_addressInputButtonVisibility).mGuidance_HomeIconDisabled = ImageLoader::getAssetBitmapImage(IMG_ADDRESSINPUT_HOME_ICON_DISABLED);
      _isAddressHomeClicked = true;
   }

   _addressInputButtonVisibility.MarkAllItemsModified();
   _addressInputButtonVisibility.SendUpdate(true);

   return true;
}


void FavoritesListHandler::removeFavoritesEntry(const navmiddleware::DestinationMemoryEntry::Id& entryId)
{
   ETG_TRACE_USR4(("FavoritesListHandler::removeFavoritesEntry(), Address Id : %d", entryId));

   if (UINT64_MAX != entryId)
   {
      ::std::vector<navmiddleware::DestinationMemoryEntry::Id> destinationInfoId;
      destinationInfoId.push_back(entryId);
      _navMiddleware.removeEntry(destinationInfoId);
   }
}
