/**************************************************************************************
* @file         : DestinationLatLongSpellerHandler.h
* @author       : ECG-Ramesh Kesavan
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#ifndef DESTINATION_LATLONG_SPELLER_HANDLER_H
#define DESTINATION_LATLONG_SPELLER_HANDLER_H

#include "Common/Util/HMIModelIncludes.h"
#include "navmid/InfoTypes.h"

#ifdef HALL_TO_MDW_COM

class DestinationLatLongSpellerHandler : public HMIModelBase
{
   public:
      DestinationLatLongSpellerHandler(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore);
      virtual ~DestinationLatLongSpellerHandler();

      virtual void initialize();
      virtual void deinitialize();

      // Incoming property updates from middleware
      PROPERTY_UPDATE_MIDDLEWARE_BEGIN()
      ON_PROPERTY_MIDDLEWARE_UNUSED()
      PROPERTY_UPDATE_MIDDLEWARE_END()

      // Incoming property updates from info store
      PROPERTY_UPDATE_INFOSTORE_BEGIN()
      ON_PROPERTY_INFOSTORE_UNUSED()
      PROPERTY_UPDATE_INFOSTORE_END()

      // Incoming courier messages from HMI
      COURIER_MSG_MAP_BEGIN(TR_CLASS_APPHMI_NAVIGATION_COURIER_PAYLOAD_MODEL_COMP)
      ON_COURIER_MESSAGE(Courier::UpdateModelMsg)
      ON_COURIER_MESSAGE(ToggleSouthNorthReqMsg)
      ON_COURIER_MESSAGE(ToggleEastWestReqMsg)
      ON_COURIER_MESSAGE(InitializeLatLongSpellerEntryReqMsg)
      ON_COURIER_MESSAGE(ChangeActiveEditFieldReqMsg)
      ON_COURIER_MESSAGE(ChangeCoordinateFormatReqMsg)
      ON_COURIER_MESSAGE(ShowCoordinatesInMapReqMsg)
      ON_COURIER_MESSAGE(ResetLatLongPopUpActiveStatusReqMsg)
      COURIER_MSG_MAP_DELEGATE_DEFAULT_BEGIN()
      COURIER_MSG_MAP_DELEGATE_DEFAULT_ENDS()
      COURIER_MSG_MAP_ENDS()

      // Incoming events
      COURIER_BINDING_MAP_BEGIN()
      COURIER_BINDING_ITEM_CHANGE(LatLongSpellerPressedKeyInfoItem)
      COURIER_BINDING_ITEM_CHANGE(LatLongSpellerTextReceivedItem)
      COURIER_BINDING_MAP_END()

      /** Called when the pressed key item changed */
      bool onCourierBindingItemChange_LatLongSpellerPressedKeyInfoItem(const Courier::Request& request);

      /** Called when the text item changed */
      bool onCourierBindingItemChange_LatLongSpellerTextReceivedItem(const Courier::Request& request);

      /** Called on press of South/North button */
      bool onCourierMessage(const ToggleSouthNorthReqMsg& oMsg);

      /** Called on press of East/West button */
      bool onCourierMessage(const ToggleEastWestReqMsg& oMsg);

      /** Called to initialize the speller on Entry */
      bool onCourierMessage(const InitializeLatLongSpellerEntryReqMsg& oMsg);

      /** Called on press of Map button */
      bool onCourierMessage(const ShowCoordinatesInMapReqMsg& oMsg);

      /** Called on press of EditField in latlong view */
      bool onCourierMessage(const ChangeActiveEditFieldReqMsg& oMsg);

      /** Called on press of Change button */
      bool onCourierMessage(const ChangeCoordinateFormatReqMsg& oMsg);

      /** Called on closing popup */
      bool onCourierMessage(const ResetLatLongPopUpActiveStatusReqMsg& oMsg);

   private:
      /**
      * Updates the active edit field text to the edit field widget
      */
      void changeActiveEditFieldText();

      /**
      * Change the active edit field BG image
      *
      * @input: optional parameter to be set on changing co-ordinat formats
      */
      void changeActiveEditFieldBGImage(bool coordinateFormatChanged = false);

      /**
      * Set the default display value to both edit field
      */
      void clearAllEditFields();

      /**
      * Check all fields for input and only if all the fields are not empty, setLatLongButton will be enabled
      */
      void checkForInputsInEditFields();

      /**
      * Change the coordinate display format from degree to decimal and vice versa
      */
      void changeDisplayFormat();

      /**
      * Get the latitude string from edit fields and convert to double
      *
      * @return:
      *        - value of latitude in double
      */
      double getLatitudeValue();

      /**
      * Get the longitude string from edit fields and convert to double
      *
      * @return:
      *        - value of longitude in double
      */
      double getLongitudeValue();

      /**
      * Check if entered coordinate value is valid or not
      * input :
      *         - Co-oridinate values
      * @return:
      *        - true if valid and false if invalid
      */
      bool isEnteredCoordinateValid(const PosWGS84<double>& coordinate);

      /**
      * Set the entered coordinate to infoStore to be shown on map
      */
      void setCoordinateToBeShownInMapToInfoStore();

      /**
      * input :
      *         - Co-oridinate values
      * Set for negative & positive value of coordinate base on direction (South/North & West/East)
      */
      void adaptEnteredCoordinateBasedOnSouthWestDirections(PosWGS84<double>& coordinate);

      /**
      * Called to process the entered coordinate
      */
      void processEnteredCoordinate();

      /**
      * Called to send databinding update
      */
      void sendLatLongCoordinatesDirectionUpdate();

      /**
      * Convert coordinate from degree format to decimal format
      * input :
      *         inputString - Co-ordinate values as string,
      *         offset      - Repsent whether the string is latitude or longitude values,
      * @return:
      *        - result coordinate string in degree format
      */
      std::string getCoordinateDecimalFormat(const std::string& inputString, int offset);

      /**
      * Initialize lat long setting base on region
      */
      void initializeLatLongSetting();

      /**
      * Resets the Data binding of the Lat long input data
      */
      void resetLatLongDataValues();

      /**
      * Sets the Presskey Data binding to the input string
      * input :
      *         inputStr - Text data to be set
      */
      void setLatLongSpellerPressedKeyInfoData(const char* inputStr);

      /**
      * Handles string updates in Edifield For DD Format
      */
      void updateSpellerPressedKeyInfoForDDFormat();

      /**
      * Handles string updates in Edifield For DM Format
      */
      void updateSpellerPressedKeyInfoForDMFormat();

      /**
      * Handles string updates in Edifield For DMS Format
      */
      void updateSpellerPressedKeyInfoForDMSFormat();

      /**
      * Handles string updates in Edifield For DMS Latitude
      */
      bool HandleLatitudeValuesForDMSFormat();

      /**
      * Handles string updates in Edifield For DMS Latitude
      */
      bool HandleLongitudeValuesForDMSFormat();

      /**
      * Update ActiveEditField Enumeration for DM co-ordinate format
      */
      void UpdateActiveEditFieldForDMFormat(const enLatLongInputDataType currentDataType);

      /**
      * Update ActiveEditField Enumeration for DMS co-ordinate format
      */
      void UpdateActiveEditFieldForDMSFormat(const enLatLongInputDataType currentDataType);

      /**
      * update the Background Images (focused/unfocused images) of Latitude edit field ONLY
      */
      void UpdateLatitudeEditFieldBackgroundImages(const enActivateEditField currentEditField, const char* currentImage, int CurrentIteration);

      /**
      * update the Background Images (focused/unfocused images) of Longitude edit field ONLY
      */
      void UpdateLongitudeEditFieldBackgroundImages(const enActivateEditField currentEditField, const char* currentImage, int CurrentIteration);

      /**
      * Enables the display of invalid co-ordinates popup
      */
      void showInvalidCoordPopup();

      // Binding source instances
      DataBindingItem<LatLongSpellerPressedKeyInfoDataBindingSource>    _latLongSpellerPressedKeyInfoData;
      DataBindingItem<LatLongSpellerTextReceivedDataBindingSource>      _latLongSpellerTextReceivedInfoData;
      DataBindingItem<LatLongCoordinatesDirectionDataBindingSource>     _latLongCoordinatesDirection;
      DataBindingItem<LatLongDataValuesDataBindingSource>               _latlongDataValues;
      DataBindingItem<LatLongEditFieldBGImageItemDataBindingSource>     _latlongEditFieldBGImage;
      DataBindingItem<LatLongCoordinatesFormatDataBindingSource>        _latLongCoordinatesFormatIndex;
      DataBindingItem<EnableSetLatLongButtonDataBindingSource>          _enableSetLatLongButton;
      DataBindingItem<AddressInputValDetailsDataBindingSource>          _addressInputValDetailsData;
      DataBindingItem<MultiLinePopuptextDataBindingSource>              _multiLineSingleBtnPopupText;

      enActivateEditField _activeEditField;	//! field which is focused
      enActivateEditField _previousActiveEditField;
      enLongitudeCoordinatesDirection _longitudeCoordinatesDirection;
      enLatitudeCoordinatesDirection _latitudeCoordinatesDirection;
      enCoordinatesDisplayType _coordinatesDisplayType;

      PosWGS84<double> _coordinate;

      bool _ignoreTextReceivedMsg;		// set to true when the text update from the widget should be ignored
      bool _ignoreEmptyString;			// set to true when the empty text update from the widget should be ignored
      bool _isSetLatLongButtonEnabled;	// set to true when there are entries in all the edit fields
      bool _isGeocoordiantesInvalid;	// set to true when the entries are invalid and data should not be reset

      std::string _currentInputStr;

      std::string _latitudeDegreesStr;
      std::string _longitudeDegreesStr;

      std::string _latitudeMinutesStr;
      std::string _longitudeMinutesStr;

      std::string _latitudeSecondsStr;
      std::string _longitudeSecondsStr;

      std::string _latitudeDecimalsStr;
      std::string _longitudeDecimalsStr;

      FEATSTD_MAKE_CLASS_UNCOPYABLE(DestinationLatLongSpellerHandler);
};


#endif // HALL_TO_MDW_COM
#endif // DESTINATION_LATLONG_SPELLER_HANDLER_H
