/**************************************************************************************
* @file         : SurfaceSynchronisationHandler.h
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2018 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#ifndef SURFACE_SYNCHRONIZATION_HANDLER_H
#define SURFACE_SYNCHRONIZATION_HANDLER_H

/**
* Helper class to handle surface synchronization. Just derive from this class, implement the pure virtual methods and use it as described below.
*/
class SurfaceSynchronizationHandler
{
   public:
      SurfaceSynchronizationHandler();
      virtual ~SurfaceSynchronizationHandler();

      // Incoming courier messages from HMI
      COURIER_MSG_MAP_BEGIN(TR_CLASS_APPHMI_NAVIGATION_COURIER_PAYLOAD_MODEL_COMP)
      ON_COURIER_MESSAGE(Courier::LayerResMsg)
      COURIER_MSG_MAP_END()

      /** Broadcast response for a layer request call */
      bool onCourierMessage(const Courier::LayerResMsg& oMsg);

      static void traceCmd_startSurfaceSynchronization();
      static void traceCmd_surfacesPrepared();

      static void setMapLayerVisibility(const bool& isVisible);

   protected:
      /** Called from client code to start a surface synchronization */
      bool startSurfaceSyncronization(bool manualSurfaceSync = false);

      /** Called from client code to start a surface synchronization based on the given surfaces */
      bool startSurfaceSyncronization(const std::vector<unsigned int>& surfaces, const std::vector<unsigned int>& layers);

      /** Trigger has to be implemented by client code to prepare all surfaces while the screen is frozen */
      virtual bool onPrepareSurfaces() = 0;

      /** Confirmation by client code that all surfaces are refreshed and sync can be removed again */
      void surfacesPrepared();

   private:
      /** Call to ilm to sync surfaces with the given surface ids */
      bool setSynchronizedSurfaces(std::vector<unsigned int>& surfaces);

      /** Call to ilm to remove the sync again */
      bool removeSynchronizedSurfaces(std::vector<unsigned int>& surfaces);

      /** Send layer request messages and subscribe to result when requested */
      void sendLayerRequestMessages(bool activate, bool subscribe);

      void clear();

      enum surfaceSynchronizationState
      {
         SURFACE_SYNCHRONIZATION_DEACTIVATED,
         SURFACE_SYNCHRONIZATION_WITH_LAYER,
         SURFACE_SYNCHRONIZATION_WITHOUT_LAYER
      };
      surfaceSynchronizationState _surfaceSynchronizationState;

      std::vector<unsigned int> _surfaces;
      std::vector<unsigned int> _layers;
      unsigned int _numLayerResponsesReceived;
      bool _manualSurfaceSync;

      FEATSTD_MAKE_CLASS_UNCOPYABLE(SurfaceSynchronizationHandler);
};


#endif // SURFACE_SYNCHRONIZATION_HANDLER_H
