/**************************************************************************************
* @file         : InfoStoreBase.cpp
* @author       : ECG5-Atchuta Sashank Kappagantu
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) -2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "App/datapool/NaviDataPoolConfig.h"
#include "EnvironmentUtils.h"
#include "InfoStoreBase.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/InfoStoreBase.cpp.trc.h"
#endif

InfoStoreBase::InfoStoreBase()
   : _writeLazyToInfoStore(false)
   , _waypointListRequestNeeded(true)
   , _waypointIndex(0)
   , _waypointListIdx(0)
   , _waypointListSize(0)
   , _isWaypointListModified(false)
   , _tcuState(NAVI_TCU_STATE_UNKNOWN)
   , _longitude(0.0)
   , _latitude(0.0)
   , _languageIndexSystem(0)
   , _languageISOCodeSystem("")
   , _languageIndexSds(0)
   , _languageISOCodeSds("")
   , _languageIndexCluster(0)
   , _languageISOCodeCluster("")
   , _languageIndexTrafficText(0)
   , _languageISOCodeTrafficText("")
   , _isTourEnabled(false)
   , _isAddDestinationSelected(false)

   , _distanceUnitSystem(DISTANCEUNITSYSTEM_IMPERIAL)
     //, _fuelUnit(FUELUNIT_UNKNOWN)
     //, _temperatureUnit(TEMPERATUREUNIT_UNKNOWN)
     //, _handBrakeState(HANDBRAKESTATE_UNKNOWN)
     //, _reverseGearState(REVERSEGEARSTATE_UNKNOWN)
   , _speedLockState(SPEEDLOCKSTATE_OFF)
   , _hmiDataServiceSpeedLockStatus(false)
   , _estimatedTimeMode(ARRIVAL_TIME)
   , _estimatedTimeType(DESTINATION)

   , _regionType(REGION_TYPE__UNKNOWN)
   , _defaultPosition()
   , _navigationUnitId("")
   , _addressInfoDetailsName("")
   , _addressInfoDetailsAddress("")
   , _addressInfoDetailsPhoneNumber("")
   , _lightMode(LIGHT_MODE_UNDEFINED)
   , _dateFormat(navmiddleware::DATEFORMAT_UNSET)
   , _timeFormat(TF_UNSET)
   , _countryCode(0)
   , _isAbleToCall(false)
   , _isBTDeviceConnected(false)
   , _guidanceVolume(5)
   , _meterConnected(false)
   , _isE2EEnabled(false)
   , _screenWidth(0)
   , _screenHeight(0)
   , _speedLimit(0)
   , _speedLimitDisplay(false)
   , _speedLimitUnit(SPEED_UNIT_UNKNOWN)
   , _freeSpeed(false)
   , _overSpeed(false)
   , _trafficReceptionState(navmiddleware::TRAFFICRECEPTIONSTATE__UNKNOWN)
   , _sxmSubscriptionStatus(false)
   , _moveLocationFlag(false)
   , _storedLocationPosition(0)
   , _isGuidanceActive(false)
   , _isHardRestrictionPresent(false)
   , _isNavMiddlewareStarted(false)
   , _isNaviServiceRequestedData(false)
   , _isDisclaimerAccepted(false)
   , _numberOfStoredRouteEntries(0)
   , _routeListElementIndexToBeDeleted(0)
   , _selectedHmiTurnIndex(-1)
   , _selectedMWTurnIndex(-1)
   , _numberOfAddressBookEntries(0)
   , _avoidAreaDeleteOptionEnable(false)
   , _addressBookSortByCategory(navmiddleware::DestinationMemoryEntry::SORTINGALGORITHM_ALPHABETICAL_BY_ALIAS_ASC)
   , _avoidAreaPreviewStyle(navmiddleware::AVOID_AREA_PREVIEW_STYLE_ALLOW_FREEWAYS)
   , _addressBookSortByGroupType(navmiddleware::DESTINATION_MEMORY_ATTRIBUTES_GROUP__NONE)
   , _addressBookSortByGroupIndex(0)
   , _routeCalculationStatus(false)
   , _speechGuidanceStatus(false)
   , _isMulipleRouteCalculated(false)
     // , _mediaDeviceConnected(MEDIA_DEVICE_CONNECTED_NONE)
   , _vehicleType(TRUCK)
   , _variantSelectionType(NAVI)
   , _isNavDataUpdateRunning(false)
   , _isNavDataUpdateRecovery(false)
   , _vehicleTypeEV(0xFF)
   , _tcuConfig(0)
   , _displayAspectRatio(DISPLAY_ASPECT_RATIO_15_BY_9)
   , _localTimeOffset(0)
   , _isClockAutoMode(false)
   , _isStateEntry(false)
   , _refineSpellerStatus(false)
   , _avoidAreaAddNewFlow(false)
   , _isGoStraightIndicationActive(false)
   , _userSelectedRouteAlternative(0)
   , _mapScale(0)
   , _listStartIndex(0)
   , _listFocusIndex(0)
   , _latitudeCoordinatesDirection(SOUTH_DIRECTION)
   , _longitudeCoordinatesDirection(EAST_DIRECTION)
   , _coordinatesDisplayType(DEGREES)
   , _defaultNominalMapScale(0)
   , _weekDay(0)
   , _clockHour(0)
   , _clockMinutes(0)
   , _addressCoordinateErrorFlag(false)
   , _sortOrder(navmiddleware::LOCATIONSORTORDER__BY_AIR_DISTANCE_NEAREST_FIRST)
   , _isSWUpdating(false)
   , _isDetailInfoRequested(false)
   , _addressDetailInfoType(DETAIL_INFO_DEFAULT)
   , _spmSystemState(SPM_SYSTEM_STATE__STATE_MAX)
   , _hmiAppState(hmibase::IN_BACKGROUND)
   , _isSplitScreenEnabled(false)
   , _satelliteSource(0)
   , _isBlackListValid(0)
   , _mapCameraViewMode(MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN)
{
   for (int regionId = 0; regionId < MAX_REG; regionId++)
   {
      _mapOutStatus[regionId].regionId = static_cast<REGION>(regionId);
   }
}


InfoStoreBase::~InfoStoreBase()
{
}


void InfoStoreBase::setTcuState(TcuState tcuState, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setTcuState(%d)", tcuState));

   if (_tcuState != tcuState
         || forceUpdate)
   {
      _tcuState = tcuState;
      notify(NAVI_TCU_CONNECTED_STATE_CHANGED);
   }
}


void InfoStoreBase::setLanguageIndexAndISOCodeRelevantForNavigation(
   unsigned char languageIndexSystem, const std::string& languageISOCodeSystem,
   unsigned char languageIndexSds, const std::string& languageISOCodeSds,
   unsigned char languageIndexCluster, const std::string& languageISOCodeCluster,
   bool forceUpdate)
{
   std::stringstream traceOut;
   traceOut << "System: " << (int)languageIndexSystem << "/" << languageISOCodeSystem << ", ";
   traceOut << "SDS: " << (int)languageIndexSds << "/" << languageISOCodeSds << ", ";
   traceOut << "Cluster: " << (int)languageIndexCluster << "/" << languageISOCodeCluster;

   ETG_TRACE_COMP(("InfoStore::setLanguageIndexAndISOCodeRelevantForNavigation(%s)", traceOut.str().c_str()));

   if (_languageIndexSystem != languageIndexSystem || _languageISOCodeSystem != languageISOCodeSystem
         || _languageIndexSds != languageIndexSds || _languageISOCodeSds != languageISOCodeSds
         || _languageIndexCluster != languageIndexCluster || _languageISOCodeCluster != languageISOCodeCluster
         || forceUpdate
      )
   {
      ETG_TRACE_COMP(("At least one language changed. Update Datapool values and notify clients!"));

      if (forceUpdate || _languageIndexSystem != languageIndexSystem || _languageISOCodeSystem != languageISOCodeSystem)
      {
         // update System language
         _languageIndexSystem = languageIndexSystem;
         _languageISOCodeSystem = languageISOCodeSystem;
         if (!_writeLazyToInfoStore)
         {
            NaviDataPoolConfig::getInstance()->setDpLanguageIndexSystem(_languageIndexSystem);
            NaviDataPoolConfig::getInstance()->setDpLanguageISOCodeSystem(languageISOCodeSystem);
         }
      }

      if (forceUpdate || _languageIndexSds != languageIndexSds || _languageISOCodeSds != languageISOCodeSds)
      {
         // update SDS language
         _languageIndexSds = languageIndexSds;
         _languageISOCodeSds = languageISOCodeSds;
         if (!_writeLazyToInfoStore)
         {
            NaviDataPoolConfig::getInstance()->setDpLanguageIndexSds(_languageIndexSds);
            NaviDataPoolConfig::getInstance()->setDpLanguageISOCodeSds(languageISOCodeSds);
         }
      }

      if (forceUpdate || _languageIndexCluster != languageIndexCluster || _languageISOCodeCluster != languageISOCodeCluster)
      {
         // update Cluster language
         _languageIndexCluster = languageIndexCluster;
         _languageISOCodeCluster = languageISOCodeCluster;
         if (!_writeLazyToInfoStore)
         {
            NaviDataPoolConfig::getInstance()->setDpLanguageIndexCluster(_languageIndexCluster);
            NaviDataPoolConfig::getInstance()->setDpLanguageISOCodeCluster(languageISOCodeCluster);
         }
      }

      notify(NAVI_LANGUAGE_CHANGED);
   }
}


void InfoStoreBase::setCountryCode(unsigned int countryCode, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setCountryCode(%d)", countryCode));

   _countryCode = countryCode;
   notify(NAVI_COUNTRYCODE_CHANGED);
}


void InfoStoreBase::setTrafficTextLanguageIndexAndISOCode(
   unsigned char languageIndexTrafficText, const std::string& languageISOCodeTrafficText, bool forceUpdate)
{
   std::stringstream traceOut;
   traceOut << "TrafficText: " << (int)languageIndexTrafficText << "/" << languageISOCodeTrafficText;

   ETG_TRACE_COMP(("InfoStore::setTrafficTextLanguageIndexAndISOCode(%s)", traceOut.str().c_str()));

   if (_languageIndexTrafficText != languageIndexTrafficText
         || _languageISOCodeTrafficText != languageISOCodeTrafficText
         || forceUpdate)
   {
      ETG_TRACE_COMP(("TrafficText language has changed. Update Datapool values and notify clients!"));

      // update TrafficText language
      _languageIndexTrafficText = languageIndexTrafficText;
      _languageISOCodeTrafficText = languageISOCodeTrafficText;
      if (!_writeLazyToInfoStore)
      {
         NaviDataPoolConfig::getInstance()->setDpLanguageIndexTrafficText(_languageIndexTrafficText);
         NaviDataPoolConfig::getInstance()->setDpLanguageISOCodeTrafficText(_languageISOCodeTrafficText);
      }

      notify(NAVI_LANGUAGE_CHANGED);
   }
}


void InfoStoreBase::setDistanceUnitSystem(DistanceUnitSystem distanceUnitSystem, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setDistanceUnitSystem(%d)", distanceUnitSystem));

   if (_distanceUnitSystem != distanceUnitSystem
         || forceUpdate)
   {
      _distanceUnitSystem = distanceUnitSystem;
      notify(NAVI_DISTANCE_UNIT_SYSTEM_CHANGED);

      if (!_writeLazyToInfoStore)
      {
         NaviDataPoolConfig::getInstance()->setDpDistanceUnitSystem(static_cast<unsigned char>(_distanceUnitSystem));
      }
   }
}


void InfoStoreBase::setLongitudeCoordinatesDirection(enLongitudeCoordinatesDirection longitudeCoordinatesDirection, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setLongitudeCoordinatesDirection(%d)", longitudeCoordinatesDirection));
   if (_longitudeCoordinatesDirection != longitudeCoordinatesDirection
         || forceUpdate)
   {
      _longitudeCoordinatesDirection = longitudeCoordinatesDirection;
      notify(NAVI_LONGITUDE_COORDINATE_DIRECTION);
      NaviDataPoolConfig::getInstance()->setDpLongitudeDirection(_longitudeCoordinatesDirection);
   }
}


void InfoStoreBase::setLatitudeCoordinatesDirection(enLatitudeCoordinatesDirection latitudeCoordinatesDirection, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setLongitudeCoordinatesDirection(%d)", latitudeCoordinatesDirection));
   if (_latitudeCoordinatesDirection != latitudeCoordinatesDirection
         || forceUpdate)
   {
      _latitudeCoordinatesDirection = latitudeCoordinatesDirection;
      notify(NAVI_LATITUDE_COORDINATE_DIRECTION);
      NaviDataPoolConfig::getInstance()->setDpLatitudeDirection(latitudeCoordinatesDirection);
   }
}


void InfoStoreBase::setCoordinatesDisplayType(enCoordinatesDisplayType coordinatesDisplayType, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setCoordinatesDisplayType(%d)", coordinatesDisplayType));
   if (_coordinatesDisplayType != coordinatesDisplayType
         || forceUpdate)
   {
      _coordinatesDisplayType = coordinatesDisplayType;
      notify(NAVI_DISPLAY_COORDINATE_FORMAT);
      NaviDataPoolConfig::getInstance()->setDpCoordinatiateDisplayFormat(_coordinatesDisplayType);
   }
}


//void InfoStoreBase::setFuelUnit(FuelUnit fuelUnit, bool forceUpdate)
//{
//   ETG_TRACE_COMP(("InfoStoreBase::setFuelUnit(%d)", fuelUnit));
//
//   if (_fuelUnit != fuelUnit
//         || forceUpdate)
//   {
//      _fuelUnit = fuelUnit;
//      //notify(NAVI_FUEL_UNIT_CHANGED);
//   }
//}


//void InfoStoreBase::setTemperatureUnit(TemperatureUnit temperatureUnit, bool forceUpdate)
//{
//   ETG_TRACE_COMP(("InfoStoreBase::setTemperatureUnit(%d)", temperatureUnit));
//
//   if (_temperatureUnit != temperatureUnit
//         || forceUpdate)
//   {
//      _temperatureUnit = temperatureUnit;
//      //notify(NAVI_TEMPERATURE_UNIT_CHANGED);
//   }
//}
//
//
//void InfoStoreBase::setHandBrakeState(HandBrakeState handBrakeState, bool forceUpdate)
//{
//   ETG_TRACE_COMP(("InfoStoreBase::setHandBrakeState(%d)", handBrakeState));
//
//   if (_handBrakeState != handBrakeState
//         || forceUpdate)
//   {
//      _handBrakeState = handBrakeState;
//      //notify(NAVI_HANDBRAKE_STATE_CHANGED);
//   }
//}
//
//
//void InfoStoreBase::setReverseGearState(ReverseGearState reverseGearState, bool forceUpdate)
//{
//   ETG_TRACE_COMP(("InfoStoreBase::setReverseGearState(%d)", reverseGearState));
//
//   if (_reverseGearState != reverseGearState
//         || forceUpdate)
//   {
//      _reverseGearState = reverseGearState;
//      //notify(NAVI_REVERSEGEAR_STATE_CHANGED);
//   }
//}


void InfoStoreBase::setSpeedLockState(SpeedLockState speedLockState, bool hmiDataServiceSpeedLockStatus, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setSpeedLockState(%d, %d)", speedLockState, hmiDataServiceSpeedLockStatus));

   if (((_speedLockState != speedLockState) || (_hmiDataServiceSpeedLockStatus != hmiDataServiceSpeedLockStatus)) && !getEnvVarAsBool(ENV_HMI_CFG_DISABLE_SPEEDLOCK, false)
         || forceUpdate)
   {
      _speedLockState = speedLockState;
      _hmiDataServiceSpeedLockStatus = hmiDataServiceSpeedLockStatus;
      if (true == _hmiDataServiceSpeedLockStatus)
      {
         ETG_TRACE_USR4(("InfoStoreBase::Speed Lock is enabled in system settings"));
         if (true == _speedLockState)
         {
            POST_MSG((COURIER_MESSAGE_NEW(DisplaySpeedLockPopuponSpellerReqMsg)()));
         }
         else
         {
            POST_MSG((COURIER_MESSAGE_NEW(HideSpeedLockPopuponSpellerReqMsg)()));
         }
      }
      else
      {
         ETG_TRACE_USR4(("InfoStoreBase::Speed Lock is disabled in system settings"));
         POST_MSG((COURIER_MESSAGE_NEW(HideSpeedLockPopuponSpellerReqMsg)()));
      }
      notify(NAVI_SPEEDLOCK_STATE_CHANGED);
   }
}


void InfoStoreBase::setEstimatedTimeMode(EstimatedTimeMode estimatedTimeMode, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setEstimatedTimeMode(%d)", estimatedTimeMode));

   if (_estimatedTimeMode != estimatedTimeMode
         || forceUpdate)
   {
      _estimatedTimeMode = estimatedTimeMode;
      if (!_writeLazyToInfoStore)
      {
         NaviDataPoolConfig::getInstance()->setDpEstimatedTimeMode(static_cast<unsigned char>(_estimatedTimeMode));
      }
      notify(NAVI_ESTIMATED_TIME_MODE_CHANGED);
   }
}


void InfoStoreBase::setEstimatedTimeType(EstimatedTimeType estimatedTimeType, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setEstimatedTimeType(%d)", estimatedTimeType));

   if (_estimatedTimeType != estimatedTimeType
         || forceUpdate)
   {
      _estimatedTimeType = estimatedTimeType;
      if (!_writeLazyToInfoStore)
      {
         NaviDataPoolConfig::getInstance()->setDpEstimatedTimeType(static_cast<unsigned char>(_estimatedTimeType));
      }
      notify(NAVI_ESTIMATED_TIME_TYPE_MODE_CHANGED);
   }
}


void InfoStoreBase::setRegionType(enRegionType regionType, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setRegionType regionType(%d)", regionType));
   if (_regionType != regionType
         || forceUpdate)
   {
      _regionType = regionType;
      ETG_TRACE_COMP(("InfoStoreBase::setRegionType _regionType(%d)", _regionType));
      //ETG_TRACE_COMP(("InfoStoreBase::Address setRegionType this:(%u) _regionType(%u)", *this, &_regionType));

      notify(NAVI_REGION_TYPE_CHANGED);
   }
}


void InfoStoreBase::setDefaultPosition(float latitude, float longitude, float heading)
{
   _defaultPosition._pos._latitude = latitude;
   _defaultPosition._pos._longitude = longitude;
   _defaultPosition._heading = heading;
}


void InfoStoreBase::setNavigationUnitId(const ::std::string& navigationUnitId)
{
   _navigationUnitId = navigationUnitId;
}


void InfoStoreBase::setAddressInfoDetailsName(const ::std::string& addressInfoDetailsName)
{
   _addressInfoDetailsName = addressInfoDetailsName;
}


void InfoStoreBase::setAddressInfoDetailsAddress(const ::std::string& addressInfoDetailsAddress)
{
   _addressInfoDetailsAddress = addressInfoDetailsAddress;
}


void InfoStoreBase::setAddressInfoDetailsPhoneNumber(const ::std::string& addressInfoDetailsPhoneNumber)
{
   _addressInfoDetailsPhoneNumber = addressInfoDetailsPhoneNumber;
}


void InfoStoreBase::setIsDetailInfoRequested(bool isDetailInfoRequested, enAddressDetailInfoType addressDetailInfoType)
{
   ETG_TRACE_COMP(("InfoStoreBase::setIsDetailInfoRequested, isDetailInfoRequested : %d, addressDetailInfoType : %d", isDetailInfoRequested, addressDetailInfoType));
   _isDetailInfoRequested = isDetailInfoRequested;
   _addressDetailInfoType = addressDetailInfoType;
}


void InfoStoreBase::setLightMode(enLightMode lightMode, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setLightMode(%d)", lightMode));
   if (_lightMode != lightMode
         || forceUpdate)
   {
      _lightMode = lightMode;
      if (!_writeLazyToInfoStore)
      {
         NaviDataPoolConfig::getInstance()->setDpLightMode(_lightMode);
      }
      notify(NAVI_LIGHT_MODE_CHANGED);
   }
}


void InfoStoreBase::setBTActiveDeviceAvailable(bool isAbleToCall, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setBTActiveDeviceAvailable(%d)", isAbleToCall));
   if (_isAbleToCall != isAbleToCall
         || forceUpdate)
   {
      _isAbleToCall = isAbleToCall;
      ETG_TRACE_COMP(("notify NAVI_BT_ACTIVE_DEVICE_CHANGED(%d)", _isAbleToCall));
      notify(NAVI_BT_ACTIVE_DEVICE_CHANGED);
   }
}


void InfoStoreBase::setBTConnectedDeviceAvailable(bool isDeviceConnected, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setBTConnectedDeviceAvailable(%d)", isDeviceConnected));
   if (_isBTDeviceConnected != isDeviceConnected
         || forceUpdate)
   {
      _isBTDeviceConnected = isDeviceConnected;
      ETG_TRACE_COMP(("notify NAVI_BT_CONNECTED_DEVICE_CHANGED(%d)", _isBTDeviceConnected));
      notify(NAVI_BT_CONNECTED_DEVICE_CHANGED);
   }
}


//void InfoStoreBase::setSortOrder(navmiddleware::LocationSortOrder sortOrder, bool forceUpdate)
//{
//   ETG_TRACE_COMP(("InfoStoreBase::setSortOrder(%d)", sortOrder));
//   if (_sortOrder != sortOrder
//         || forceUpdate)
//   {
//      _sortOrder = sortOrder;
//      ETG_TRACE_COMP(("notify NAVI_POI_FTS_SORT_ORDER_CHANGED(%d)", _sortOrder));
//      notify(NAVI_POI_FTS_SORT_ORDER_CHANGED);
//   }
//}


void InfoStoreBase::setGuidanceVolume(int value, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setGuidanceVolume(%d)", value));
   if (_guidanceVolume != value
         || forceUpdate)
   {
      _guidanceVolume = value;
      notify(NAVI_GUIDANCE_VOLUME_CHANGED);
   }
}


void InfoStoreBase::setSpeedLimitStatus(int speedLimit, int speedLimitDisplay, enSpeedUnit speedLimitUnit, bool freeSpeed, bool overSpeed, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setSpeedLimitStatus(speedLimit %d, speedLimitDisplay %d, speedLimitUnit %d, freeSpeed = %d, overSpeed = %d)", speedLimit, speedLimitDisplay, speedLimitUnit, freeSpeed, overSpeed));
   if ((speedLimit != _speedLimit)
         || (speedLimitDisplay != _speedLimitDisplay)
         || (speedLimitUnit != _speedLimitUnit)
         || (freeSpeed != _freeSpeed)
         || (overSpeed != _overSpeed)
         || forceUpdate)
   {
      _speedLimit = speedLimit;
      _speedLimitDisplay = speedLimitDisplay;
      _speedLimitUnit = speedLimitUnit;
      _freeSpeed = freeSpeed;
      _overSpeed = overSpeed;
      notify(NAVI_SPEED_LIMIT_STATUS_CHANGED);
   }
}


void InfoStoreBase::setWaypointIndex(unsigned int waypointIndex, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setSelectedWaypoint(%d)", waypointIndex));

   if (_waypointIndex != waypointIndex
         || forceUpdate)
   {
      _waypointIndex = waypointIndex;
//      sendActiveWaypointIndexDataBinding();
      notify(NAVI_WAYPOINT_INDEX_CHANGED);
   }
}


void InfoStoreBase::setWaypointListIdx(unsigned int waypointListIdx, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setSelectedListIdx(%d)", waypointListIdx));

   if (_waypointListIdx != waypointListIdx
         || forceUpdate)
   {
      _waypointListIdx = waypointListIdx;
   }
}


void InfoStoreBase::setWaypointListRequestNeeded(bool waypointListRequestNeeded, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setWaypointListRequestNeeded(%d)", waypointListRequestNeeded));

   if (_waypointListRequestNeeded != waypointListRequestNeeded
         || forceUpdate)
   {
      _waypointListRequestNeeded = waypointListRequestNeeded;
      notify(NAVI_WAYPOINT_LIST_REQUEST_CHANGED);
   }
}


void InfoStoreBase::setWaypointListSize(unsigned int waypointSize)
{
   ETG_TRACE_COMP(("InfoStoreBase::setSelectedWaypoint(%d)", waypointSize));

   if (_waypointListSize != waypointSize)
   {
      _waypointListSize = waypointSize;
   }
}


void InfoStoreBase::setIsWaypointListModified(bool isWaypointListModified)
{
   ETG_TRACE_COMP(("InfoStoreBase::setIsWaypointListModified(%d)", isWaypointListModified));

   if (_isWaypointListModified != isWaypointListModified)
   {
      _isWaypointListModified = isWaypointListModified;
   }
}


void InfoStoreBase::setTmId(navmiddleware::ValidValue<int> tmId, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setTmId(valid %d, value %d)", tmId.isValid(), tmId.getValue()));

   if (!(_tmIdToBeShownInMap == tmId)
         || forceUpdate)
   {
      _tmIdToBeShownInMap = tmId;
      notify(NAVI_TM_ID_CHANGED);
   }
}


void InfoStoreBase::sendRegionTypeDataBinding()
{
   ETG_TRACE_COMP(("InfoStoreBase::sendRegionDatabinding()"));
//
//   if ((*_regionTypeDataBinding).mKdsRegionType != (unsigned char)_regionType)
//   {
//      (*_regionTypeDataBinding).mKdsRegionType = (unsigned char)_regionType;
//      _regionTypeDataBinding.MarkItemModified(ItemKey::RegionType::KdsRegionTypeItem);
//      if (_regionTypeDataBinding.SendUpdate() == false)
//      {
//         ETG_TRACE_ERR(("_regionDatabinding update failed!"));
//      }
//   }
}


//void InfoStoreBase::sendDisplayOrientationStatusDataBinding(bool isPotrait)
//{
//   ETG_TRACE_COMP(("InfoStoreBase::sendDisplayOrientationStatusDataBinding()"));
//
//   (*_displayOrientation).mDisplayOrientationType = isPotrait;
//   _displayOrientation.MarkItemModified(ItemKey::DisplayOrientation::DisplayOrientationTypeItem);
//   _displayOrientation.SendUpdate();
//}
//
//
//void InfoStoreBase::sendActiveWaypointIndexDataBinding()
//{
//   ETG_TRACE_COMP(("InfoStoreBase::sendActiveWaypointIndexDataBinding()"));
//
//   if ((*_editRouteDataBinding).mActiveWaypointListIndex != (signed char)_waypointIndex)
//   {
//      (*_editRouteDataBinding).mActiveWaypointListIndex = (signed char)_waypointIndex;
//      _editRouteDataBinding.MarkItemModified(ItemKey::EditRoute::ActiveWaypointListIndexItem);
//
//      if (_editRouteDataBinding.SendUpdate() == false)
//      {
//         ETG_TRACE_ERR(("_editRouteDataBinding update failed!"));
//      }
//   }
//}


void InfoStoreBase::setDestinationMemoryStatus(const InfoStoreBase::DestinationMemoryStatus& destinationMemoryStatus, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setDestinationMemoryStatus()"));

   if (!(_destinationMemoryStatus == destinationMemoryStatus)
         || forceUpdate)
   {
      _destinationMemoryStatus = destinationMemoryStatus;
      notify(NAVI_DESTINATION_MEMORY_STATUS_CHANGED);
   }
}


void InfoStoreBase::setStoredLocationPosition(const uint64_t& storedLocationPosition, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setStoredLocationPosition(%d)", storedLocationPosition));

   if (!(_storedLocationPosition == storedLocationPosition)
         || forceUpdate)
   {
      _storedLocationPosition = storedLocationPosition;
      notify(NAVI_DESTINATION_MEMORY_ENTRY_POSITION_CHANGED);
   }
}


void InfoStoreBase::setStateEntry(bool stateEntry)
{
   ETG_TRACE_COMP(("InfoStoreBase::setStateEntry"));

   if (_isStateEntry != stateEntry)
   {
      _isStateEntry = stateEntry;
   }
}


void InfoStoreBase::setRefineSpellerStatus(bool refineSpellerStatus)
{
   ETG_TRACE_COMP(("InfoStoreBase::setRefineSpellerStatus"));

   if (_refineSpellerStatus != refineSpellerStatus)
   {
      _refineSpellerStatus = refineSpellerStatus;
   }
}


void InfoStoreBase::setIsGuidanceActive(const bool isGuidanceActive, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setIsGuidanceActive(%d)", isGuidanceActive));

   if (_isGuidanceActive != isGuidanceActive
         || forceUpdate)
   {
      _isGuidanceActive = isGuidanceActive;
      // If guidance is active, the active state is written into DP only on shutdown (in InfoStore::savePersistencyToDataPool())
      if ((FALSE == _isGuidanceActive) && !_writeLazyToInfoStore)
      {
         NaviDataPoolConfig::getInstance()->setDpIsGuidanceActive(_isGuidanceActive);
      }
      notify(NAVI_GUIDANCE_STATE_CHANGED);
   }
}


void InfoStoreBase::setIsHardRestrictionPresent(const bool isHardRestrictionPresent, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setIsHardRestrictionPresent(%d)", isHardRestrictionPresent));

   if (_isHardRestrictionPresent != isHardRestrictionPresent
         || forceUpdate)
   {
      _isHardRestrictionPresent = isHardRestrictionPresent;
      notify(NAVI_HARD_RESTRICTION_STATUS_CHANGED);
   }
}


void InfoStoreBase::setIsNavMiddlewareStarted(const bool isNavMiddlewareStarted)
{
   _isNavMiddlewareStarted = isNavMiddlewareStarted;
}


void InfoStoreBase::setDestinationAddress(const ::std::string name, bool forceUpdate)
{
   _destinationName = name;
}


void InfoStoreBase::setIsDisclaimerAccepted(bool isDisclaimerAccepted)
{
   ETG_TRACE_COMP(("InfoStoreBase::setIsDisclaimerAccepted(), Is disclaimer accepted : %d", isDisclaimerAccepted));
   if(_isDisclaimerAccepted != isDisclaimerAccepted)
   {
      _isDisclaimerAccepted = isDisclaimerAccepted;
      notify(NAVI_DISCLAIMER_STATUS_CHANGED);
   }
}


const InfoStoreBase::CoordinatesToBeShownInMap& InfoStoreBase::getCoordinatesToBeShownInMap() const
{
   return _coordinatesToBeShownInMap;
}


InfoStoreBase::CoordinatesToBeShownInMap& InfoStoreBase::getCoordinatesToBeShownInMap()
{
   return _coordinatesToBeShownInMap;
}


void InfoStoreBase::resetCoordinatesToBeShownInMap()
{
   ETG_TRACE_COMP(("InfoStoreBase::resetCoordinatesToBeShownInMap()"));
   _coordinatesToBeShownInMap = CoordinatesToBeShownInMap();
}


void InfoStoreBase::addSelectedTurnInfo(const int index)
{
   const int INVALID_TURN_INDEX = -1;
   ETG_TRACE_COMP(("InfoStoreBase::addSelectedTurnInfo()"));
   if (index != INVALID_TURN_INDEX)
   {
      _avoidTurnList.push_back(index);
      notify(NAVI_TURNLIST_AVOID_INDEX_CHANGED);
   }
   else
   {
      ETG_TRACE_COMP(("InfoStoreBase::Invalid Index"));
   }
}


void InfoStoreBase::clearSelectedTurnInfo()
{
   ETG_TRACE_COMP(("InfoStoreBase::clearSelectedTurnInfo()"));
   _avoidTurnList.clear();
   notify(NAVI_TURNLIST_CLEAR_AVOID_INDEX_CHANGED);
}


void InfoStoreBase::setCurrentlySelectedTurnIndex(const int selectedHmiTurnIndex, const int selectedMWTurnIndex)
{
   ETG_TRACE_COMP(("InfoStoreBase::setCurrentlySelectedTurnIndex()"));
   _selectedHmiTurnIndex = selectedHmiTurnIndex;
   _selectedMWTurnIndex = selectedMWTurnIndex;
   notify(NAVI_TURNLIST_SELECTED_INDEX_CHANGED);
}


void InfoStoreBase::setNearbyPoi(const navmiddleware::ValidValue< navmiddleware::RouteListInfos::RouteListElementInfo::ExitPOIInfo >& routePOIInfo)
{
   _routeExitPoiInfo = routePOIInfo;
}


navmiddleware::ValidValue< navmiddleware::RouteListInfos::RouteListElementInfo::ExitPOIInfo > InfoStoreBase::getNearbyPoi() const
{
   return _routeExitPoiInfo;
}


//Set Function to set details of individual element.
void InfoStoreBase::StoredOptionsListItemInfo::setOptionsListItemEnableInfo(const enOptionListElement enListElement, const bool enableStatusToSet)
{
   switch (enListElement)
   {
      case OPTION_LIST_FIRST_ELEMENT:
      case OPTION_LIST_SECOND_ELEMENT:
      case OPTION_LIST_THIRD_ELEMENT:
      case OPTION_LIST_FOURTH_ELEMENT:
      case OPTION_LIST_FIFTH_ELEMENT:
         _OptionsListItem[enListElement].OptionItemEnableState = enableStatusToSet;
         break;
      default:
         /* No Action */
         break;
   }

   ETG_TRACE_USR4(("StoredOptionsListItemInfo::setOptionsListItemEnableInfo Element:%d Value:%d", enListElement, enableStatusToSet));
}


//Set function that sets the Context ID information.
void InfoStoreBase::StoredOptionsListItemInfo::setOptionsLisItemContextIDInfo(const Courier::UInt32 optionsContextID)
{
   ETG_TRACE_USR4(("StoredOptionsListItemInfo::setOptionsLisItemContextIDInfo CntID:%d ", optionsContextID));
   _OptionContextID = optionsContextID;
}


//Get function to get individual element details.
bool InfoStoreBase::StoredOptionsListItemInfo::getOptionsListItemEnableInfo(const enOptionListElement enListElement) const
{
   bool retVal = false;
   if (enListElement < OPTION_LIST_MAX_ELEMENT)
   {
      retVal = _OptionsListItem[enListElement].OptionItemEnableState;
   }

   ETG_TRACE_USR4(("StoredOptionsListItemInfo::getOptionsListItemEnableInfo CntID:%d ", _OptionContextID));
   ETG_TRACE_USR4(("StoredOptionsListItemInfo::getOptionsListItemEnableInfo Element:%d Value:%d", enListElement, retVal));

   return retVal;
}


//Get function to get Context ID information.
Courier::UInt32 InfoStoreBase::StoredOptionsListItemInfo::getOptionsListItemContextInfo(void) const
{
   ETG_TRACE_USR4(("StoredOptionsListItemInfo::getOptionsListItemContextInfo CntID:%d ", _OptionContextID));
   return _OptionContextID;
}


void InfoStoreBase::triggerMapMainScreenStateChanged()
{
   notify(NAVI_MAP_MAIN_SCREEN_STATE_CHANGED);
}


void InfoStoreBase::setDestinationAttributesInfoName(const ::std::string& name, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setDestinationAttributesInfoName(%s)", name.c_str()));
   if (_destinationAttributesInfo.getName() != name
         || forceUpdate)
   {
      _destinationAttributesInfo.setName(name);
      notify(NAVI_DESTINATION_ATTRIBUTES_INFO_CHANGED);
   }
}


void InfoStoreBase::setDestinationAttributesInfoSoundId(const navmiddleware::DestinationMemoryEntryAttributes::Sound& soundId, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setDestinationAttributesInfoSoundId(%d)", soundId));
   if (_destinationAttributesInfo.getSound() != soundId
         || forceUpdate)
   {
      _destinationAttributesInfo.setSound(soundId);
      notify(NAVI_DESTINATION_ATTRIBUTES_INFO_CHANGED);
   }
}


void InfoStoreBase::setDestinationAttributesInfoVoiceTagId(const uint64_t& voiceTag, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setDestinationAttributesInfoVoiceTagId(%d)", voiceTag));
   if (_destinationAttributesInfo.getVoiceTag().getValue() != voiceTag
         || forceUpdate)
   {
      _destinationAttributesInfo.setVoiceTag(voiceTag);
      notify(NAVI_DESTINATION_ATTRIBUTE_VOICE_TAG_INFO_CHANGED);
      notify(NAVI_DESTINATION_ATTRIBUTES_INFO_CHANGED);
   }
}


void InfoStoreBase::setDestinationAttributesInfoInvalidateVoiceTagId()
{
   ETG_TRACE_COMP(("InfoStoreBase::setDestinationAttributesInfoInvalidateVoiceTagId()"));
   if (_destinationAttributesInfo.getVoiceTag().isValid())
   {
      _destinationAttributesInfo.invalidateVoiceTag();
      notify(NAVI_DESTINATION_ATTRIBUTE_VOICE_TAG_INFO_CHANGED);
      notify(NAVI_DESTINATION_ATTRIBUTES_INFO_CHANGED);
   }
}


void InfoStoreBase::setDestinationAttributesInfoIconId(const uint64_t& iconId, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setDestinationAttributesInfoIconId(%d)", iconId));
   /* if (_destinationAttributesInfo.getIconId() != iconId
          || forceUpdate)
    {
       _destinationAttributesInfo.setIconId(iconId);
       //notify(NAVI_DESTINATION_ATTRIBUTES_INFO_CHANGED);
    }*/
}


void InfoStoreBase::setDestinationAttributesInfoDistance(const navmiddleware::DestinationMemoryEntryAttributes::Distance& distance, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setDestinationAttributesInfoDistance(%d)", distance));
   if (_destinationAttributesInfo.getDistance() != distance
         || forceUpdate)
   {
      _destinationAttributesInfo.setDistance(distance);
      notify(NAVI_DESTINATION_ATTRIBUTES_INFO_CHANGED);
   }
}


void InfoStoreBase::setDestinationAttributesInfoDirection(const uint16_t& direction, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setDestinationAttributesInfoDirection(%d)", direction));
   if ((_destinationAttributesInfo.getDirection().getValue() != direction)
         || (!_destinationAttributesInfo.getDirection().isValid())
         || forceUpdate)
   {
      _destinationAttributesInfo.setDirection(direction);
      notify(NAVI_DESTINATION_ATTRIBUTES_INFO_CHANGED);
   }
}


void InfoStoreBase::setDestinationAttributesInfoInvalidateDirection()
{
   ETG_TRACE_COMP(("InfoStoreBase::setDestinationAttributesInfoInvalidateDirection()"));
   if (_destinationAttributesInfo.getDirection().isValid())
   {
      _destinationAttributesInfo.invalidateDirection();
      notify(NAVI_DESTINATION_ATTRIBUTES_INFO_CHANGED);
   }
}


void InfoStoreBase::setDestinationAttributesInfoShowOnMap(const bool& showOnMap, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setDestinationAttributesInfoShowOnMap(%d)", showOnMap));
   if (_destinationAttributesInfo.isShownOnMap() != showOnMap
         || forceUpdate)
   {
      _destinationAttributesInfo.setShowOnMap(showOnMap);
      notify(NAVI_DESTINATION_ATTRIBUTES_INFO_CHANGED);
   }
}


void InfoStoreBase::setDestinationAttributesInfoPhoneNumber(const ::std::vector< ::std::string >& phoneNumberList, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setDestinationAttributesInfoPhoneNumber()"));
   if (_destinationAttributesInfo.getPhoneNumber() != phoneNumberList
         || forceUpdate)
   {
      _destinationAttributesInfo.setPhoneNumber(phoneNumberList);
      notify(NAVI_DESTINATION_ATTRIBUTES_INFO_CHANGED);
   }
}


void InfoStoreBase::setDestinationAttributesInfoGroupList(const ::std::vector<navmiddleware::DestinationMemoryAttributesGroup>& groupList, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setDestinationAttributesInfoGroupList()"));
   if (_destinationAttributesInfo.getGroupList() != groupList
         || forceUpdate)
   {
      _destinationAttributesInfo.setGroupList(groupList);
      notify(NAVI_DESTINATION_ATTRIBUTES_INFO_CHANGED);
   }
}


void InfoStoreBase::setDestinationAttributesInfoInternalMapIconId(const uint64_t& internalMapIconId, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setDestinationAttributesInfoInternalMapIconId(%d)", internalMapIconId));
   /*if (_destinationAttributesInfo.getInternalMapIconId() != internalMapIconId
         || forceUpdate)
   {
      _destinationAttributesInfo.setInternalMapIconId(internalMapIconId);
      //notify(NAVI_DESTINATION_ATTRIBUTES_INFO_CHANGED);
   }*/
}


void InfoStoreBase::setDestinationAttributesInfo(const navmiddleware::DestinationMemoryEntryAttributes& destinationAttributesInfo)
{
   ETG_TRACE_COMP(("InfoStoreBase::setDestinationAttributesInfo()"));
   _destinationAttributesInfo = destinationAttributesInfo;
   notify(NAVI_DESTINATION_ATTRIBUTES_INFO_RESET);
}


void InfoStoreBase::setDestinationMemoryHomeStatus(const bool& status, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setDestinationMemoryHomeStatus(%d)", status));
   if (_destinationMemoryHomeInfo._status != status
         || forceUpdate)
   {
      _destinationMemoryHomeInfo._status = status;
      notify(NAVI_DESTINATION_MEMORY_HOME_CHANGED);
   }
}


void InfoStoreBase::setDestinationMemoryHomeId(const navmiddleware::DestinationMemoryEntry::Id& id, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setDestinationMemoryHomeId(%d)", id));
   if (_destinationMemoryHomeInfo._id != id
         || forceUpdate)
   {
      _destinationMemoryHomeInfo._id = id;
      notify(NAVI_DESTINATION_MEMORY_HOME_CHANGED);
   }
}


void InfoStoreBase::setDestinationHomeAttribute(const navmiddleware::DestinationMemoryEntryAttributes& destinationHomekAttributes)
{
   ETG_TRACE_COMP(("InfoStoreBase::setDestinationHomeAttribute()"));
   _destinationMemoryHomeInfo._destinationMemoryAttributesInfo = destinationHomekAttributes;
   notify(NAVI_DESTINATION_MEMORY_HOME_CHANGED);
}


void InfoStoreBase::setDestinationHomeCoordinates(const std::vector< PosWGS84<double> >& destinationHomeCoordinates, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setDestinationHomeCoordinates()"));

   if (_destinationMemoryHomeInfo._coordinates != destinationHomeCoordinates
         || forceUpdate)
   {
      _destinationMemoryHomeInfo._coordinates = destinationHomeCoordinates;
      notify(NAVI_DESTINATION_MEMORY_HOME_CHANGED);
   }
}


void InfoStoreBase::setMoveLocationFlag(const bool& moveLocationFlag, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setMoveLocationFlag(%d)", moveLocationFlag));
   if (_moveLocationFlag != moveLocationFlag
         || forceUpdate)
   {
      _moveLocationFlag = moveLocationFlag;
      notify(NAVI_MOVE_LOCATION_CHANGED);
   }
}


void InfoStoreBase::setNavDataUpdateRunning(bool isRunning)
{
   ETG_TRACE_COMP(("InfoStoreBase::setNavDataUpdateRunning(%d)", isRunning));
   _isNavDataUpdateRunning = isRunning;
}


void InfoStoreBase::setNavDataUpdateRecovery(bool isRecovery)
{
   ETG_TRACE_COMP(("InfoStoreBase::setNavDataUpdateRecovery(%d)", isRecovery));
   _isNavDataUpdateRecovery = isRecovery;
}


void InfoStoreBase::setTourInformation(navmiddleware::Tour& tour)
{
   ETG_TRACE_COMP(("InfoStoreBase::setTourInformation(%d)", tour.getTourElements().size()));
   _tour = tour;
   notify(NAVI_TOUR_ELEMENTS_CHANGED);
}


void InfoStoreBase::setAutoPlayCoordinates(std::vector<navmiddleware::GeoCoordinateDegree>& autoPlayCoordinates)
{
   ETG_TRACE_COMP(("InfoStoreBase::setAutoPlayCoordinates(%d)", autoPlayCoordinates.size()));
   _autoPlayCoordinates = autoPlayCoordinates;
   notify(NAVI_AUTO_PLAY_COORDINATES_CHANGED);
}


void InfoStoreBase::setTrafficReceptionState(navmiddleware::TrafficReceptionState trafficReceptionState)
{
   ETG_TRACE_COMP(("InfoStoreBase::setTrafficReceptionState(%d)", trafficReceptionState));
   _trafficReceptionState = trafficReceptionState;
   notify(NAVI_TRAFFIC_RECEPTION_STATUS_CHANGED);
}


void InfoStoreBase::setSXMSubscriptionStatus(bool status, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setSXMSubscriptionStatus(%d)", status));
   if (_sxmSubscriptionStatus != status
         || forceUpdate)
   {
      _sxmSubscriptionStatus = status;
      notify(NAVI_SXM_SUBSCRIPTION_STATUS);
   }
}


void InfoStoreBase::setRouteListElementIndexToBeDeleted(unsigned int routeListElementIndexToBeDeleted, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setRouteListElementIndexToBeDeleted()"));
   if (_routeListElementIndexToBeDeleted != routeListElementIndexToBeDeleted
         || forceUpdate)
   {
      _routeListElementIndexToBeDeleted = routeListElementIndexToBeDeleted;
      notify(NAVI_ROUTELISTELEMENT_INDEX_TO_BE_DELETED_CHANGED);
   }
}


void InfoStoreBase::setRouteToBeUsedForE2EGuidance(const InfoStoreBase::RouteToBeUsedForE2EGuidance& routeToBeUsedForE2EGuidance)
{
   ETG_TRACE_COMP(("InfoStoreBase::setRouteToBeUsedForE2EGuidance"));
   _routeToBeUsedForE2EGuidance = routeToBeUsedForE2EGuidance;
   if (_isE2EEnabled)
   {
      notify(NAVI_COORDINATES_TO_BE_USED_FOR_E2E_GUIDANCE_CHANGED); // @Todo: Clarify the message to be actually used
   }
}


void InfoStoreBase::setIsTourEnabled(const bool isTourActive)
{
   ETG_TRACE_COMP(("InfoStoreBase::setIsTourEnabled"));
   _isTourEnabled = isTourActive;
}


void InfoStoreBase::setIsAddDestinationSelected(const bool isAddDestinationSelected)
{
   ETG_TRACE_COMP(("InfoStoreBase::setIsTourEnabled"));
   _isAddDestinationSelected = isAddDestinationSelected;
}


unsigned int InfoStoreBase::getRouteListElementIndexToBeDeleted() const
{
   ETG_TRACE_COMP(("InfoStoreBase::getRouteListElementIndexToBeDeleted()"));
   return _routeListElementIndexToBeDeleted;
}


void InfoStoreBase::setDateFormat(navmiddleware::DateFormat dateFormat, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setDateFormat(%d)", dateFormat));
   if (_dateFormat != dateFormat
         || forceUpdate)
   {
      _dateFormat = dateFormat;
      if (!_writeLazyToInfoStore)
      {
         NaviDataPoolConfig::getInstance()->setDpDateFormat(static_cast<unsigned char>(_dateFormat));
      }
      notify(NAVI_DATE_FORMAT_CHANGED);
   }
}


void InfoStoreBase::setTimeFormat(enTimeFormat  timeFormat, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setTimeFormat(%d)", timeFormat));
   if (_timeFormat != timeFormat
         || forceUpdate)
   {
      _timeFormat = timeFormat;
      if (!_writeLazyToInfoStore)
      {
         NaviDataPoolConfig::getInstance()->setDpTimeFormat(_timeFormat);
      }
      notify(NAVI_TIME_FORMAT_CHANGED);
   }
}


void InfoStoreBase::setLocalTimeOffsetInfo(int32_t offsetValue, bool isClockAutoMode, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setLocalTimeOffsetInfo(%d, %d)", offsetValue, isClockAutoMode));
   if (forceUpdate || (offsetValue != _localTimeOffset) || (isClockAutoMode != _isClockAutoMode))
   {
      _localTimeOffset = offsetValue;
      _isClockAutoMode = isClockAutoMode;

      if (!_writeLazyToInfoStore)
      {
         NaviDataPoolConfig::getInstance()->setDpLocalTimeOffset(_localTimeOffset);
         NaviDataPoolConfig::getInstance()->setDpClockAutoModeActivated(_isClockAutoMode);
      }

      notify(NAVI_LOCAL_TIME_OFFSET_INFO_CHANGED);
   }
}


unsigned int InfoStoreBase::getNumberOfAddressBookEntries() const
{
   ETG_TRACE_COMP(("InfoStoreBase::getNumberOfAddressBookEntries()"));
   return _numberOfAddressBookEntries;
}


void InfoStoreBase::setNumberOfAddressBookEntries(unsigned int numberOfAddressBookEntries, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setNumberOfAddressBookEntries()"));
   if (_numberOfAddressBookEntries != numberOfAddressBookEntries
         || forceUpdate)
   {
      _numberOfAddressBookEntries = numberOfAddressBookEntries;
      notify(NAVI_NUMBER_OF_ADDRESSBOOK_ENTRIES_CHANGED);
   }
}


unsigned int InfoStoreBase::getNumberOfStoredRouteEntries() const
{
   ETG_TRACE_COMP(("InfoStoreBase::getNumberOfStoredRouteEntries()"));
   return _numberOfStoredRouteEntries;
}


void InfoStoreBase::setNumberOfStoredRouteEntries(unsigned int numberOfStoredRouteEntries, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setNumberOfStoredRouteEntries()"));
   if (_numberOfStoredRouteEntries != numberOfStoredRouteEntries
         || forceUpdate)
   {
      _numberOfStoredRouteEntries = numberOfStoredRouteEntries;
      notify(NAVI_NUMBER_OF_STORED_ROUTE_ENTRIES_CHANGED);
   }
}


void InfoStoreBase::setStoredRouteInfo(const InfoStoreBase::StoredRouteInfo& storedRouteInfo, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setStoredRouteInfo()"));
   if (!(_storedRouteInfo == storedRouteInfo)
         || forceUpdate)
   {
      _storedRouteInfo = storedRouteInfo;
      notify(NAVI_STORED_ROUTE_INFO_CHANGED);
   }
}


void InfoStoreBase::setAddressBookSortByCategory(navmiddleware::DestinationMemoryEntry::SortingAlgorithm addressBookSortByCategory, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setAddressBookSortByCategory()"));
   if (addressBookSortByCategory != _addressBookSortByCategory
         || forceUpdate)
   {
      _addressBookSortByCategory = addressBookSortByCategory;
      notify(NAVI_ADDRESSBOOK_SORT_BY_CATEGORY_CHANGED);
   }
}


navmiddleware::DestinationMemoryEntry::SortingAlgorithm InfoStoreBase::getAddressBookSortByCategory() const
{
   ETG_TRACE_COMP(("InfoStoreBase::getAddressBookSortByCategory()"));
   return _addressBookSortByCategory;
}


void InfoStoreBase::setAddressBookSortByGroupType(navmiddleware::DestinationMemoryAttributesGroup addressBookSortByGroupType, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setAddressBookSortByGroupType()"));
   if (addressBookSortByGroupType != _addressBookSortByGroupType
         || forceUpdate)
   {
      _addressBookSortByGroupType = addressBookSortByGroupType;
      notify(NAVI_ADDRESSBOOK_SORT_BY_CATEGORY_CHANGED);
   }
}


void InfoStoreBase::setAddressBookSortByGroupListStartIndex(unsigned int addressBookSortByGroupIndex, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setAddressBookSortByGroupListStartIndex()"));
   if (addressBookSortByGroupIndex != _addressBookSortByGroupIndex
         || forceUpdate)
   {
      _addressBookSortByGroupIndex = addressBookSortByGroupIndex;
      notify(NAVI_ADDRESSBOOK_SORT_BY_GROUP_START_INDEX_CHANGED);
   }
}


void InfoStoreBase::setRouteCalculationStatus(bool isRouteCalculated, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setRouteCalculationStatus()"));
   if (_routeCalculationStatus != isRouteCalculated
         || forceUpdate)
   {
      _routeCalculationStatus = isRouteCalculated;
      notify(NAVI_ROUTE_CALCULATION_STATUS_CHANGED);
   }
}


void InfoStoreBase::setGuidanceStatefromSDS(bool isStartGuidance, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setGuidanceStatefromSDS()"));
   if (_speechGuidanceStatus != isStartGuidance
         || forceUpdate)
   {
      _speechGuidanceStatus = isStartGuidance;
   }
}


void InfoStoreBase::setMultipleRouteCalculationStatus(bool isMulipleRouteCalculated, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setMultipleRouteCalculationStatus()"));
   if (_isMulipleRouteCalculated != isMulipleRouteCalculated
         || forceUpdate)
   {
      _isMulipleRouteCalculated = isMulipleRouteCalculated;
   }
}


void InfoStoreBase::setMapDownloadLockState(bool isSWUpdating, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setMapDownloadLockState(%d)", isSWUpdating));

   if ((isSWUpdating != _isSWUpdating) || forceUpdate)
   {
      _isSWUpdating = isSWUpdating;
   }
}


//void InfoStoreBase::setMediaDeviceConnectedStatus(enMediaDeviceConnected mediaDeviceConnected, bool forceUpdate)
//{
//   ETG_TRACE_COMP(("InfoStoreBase::setMediaDeviceConnectedStatus()"));
//   if (_mediaDeviceConnected != mediaDeviceConnected
//         || forceUpdate)
//   {
//      _mediaDeviceConnected = mediaDeviceConnected;
//      notify(NAVI_MEDIA_DEVICE_CONNECTED_STATUS_CHANGED);
//   }
//}


void InfoStoreBase::setGoStraightIndicationStatus(bool isGoStraightIndicationActive)
{
   ETG_TRACE_COMP(("InfoStoreBase::setGoStarightIndicationStatus()"));
   if (_isGoStraightIndicationActive != isGoStraightIndicationActive)
   {
      _isGoStraightIndicationActive = isGoStraightIndicationActive;
   }
}


void InfoStoreBase::setTcuConfig(unsigned int tcuConfig)
{
   ETG_TRACE_COMP(("InfoStoreBase::setTcuConfig(%d)", tcuConfig));
   if (_tcuConfig != tcuConfig)
   {
      _tcuConfig = tcuConfig;
   }
}


void InfoStoreBase::setUserSelectedRouteAlternative(int userSelectedRouteAlternative)
{
   ETG_TRACE_COMP(("InfoStoreBase::setUserSelectedRouteAlternative(%d)", userSelectedRouteAlternative));
   if (_userSelectedRouteAlternative != userSelectedRouteAlternative)
   {
      _userSelectedRouteAlternative = userSelectedRouteAlternative;
   }
   notify(NAVI_USER_SELECTED_ROUTE_ALTERNATIVE_CHANGED);
}


void InfoStoreBase::setDefaultNominalMapScale(int32_t defaultNominalMapScale)
{
   ETG_TRACE_COMP(("InfoStoreBase::setDefaultNominalMapScale(%d)", defaultNominalMapScale));
   if (_defaultNominalMapScale != defaultNominalMapScale)
   {
      _defaultNominalMapScale = defaultNominalMapScale;
   }
   notify(NAVI_DEFAULT_NOMINAL_MAP_SCALE_CHANGED);
}


void InfoStoreBase::setListFocusIndex(unsigned int focusIndex)
{
   ETG_TRACE_COMP(("InfoStoreBase::setListFocusIndex = %d" , focusIndex));
   _listFocusIndex = focusIndex;
}


void InfoStoreBase::setLongitude(double longitude)
{
   ETG_TRACE_COMP(("InfoStoreBase::setLongitude = %f", longitude));
   _longitude = longitude;
}


void InfoStoreBase::setLatitude(double latitude)
{
   ETG_TRACE_COMP(("InfoStoreBase::setLatitude = %f", latitude));
   _latitude = latitude;
}


void InfoStoreBase::setListStartIndex(unsigned int startIndex)
{
   ETG_TRACE_COMP(("InfoStoreBase::setListStartIndex = %d", startIndex));
   _listStartIndex = startIndex;
}


unsigned int InfoStoreBase::getListFocusIndex() const
{
   ETG_TRACE_COMP(("InfoStoreBase::getListFocusIndex = %d", _listFocusIndex));
   return _listFocusIndex;
}


unsigned int InfoStoreBase::getListStartIndex() const
{
   ETG_TRACE_COMP(("InfoStoreBase::getListStartIndex = %d" , _listStartIndex));
   return _listStartIndex;
}


double InfoStoreBase::getLongitude() const
{
   return _longitude;
}


double InfoStoreBase::getLatitude() const
{
   return _latitude;
}


void InfoStoreBase::setCurrentWeekDay(uint32_t weekDay, bool forceUpdate)
{
   if ((_weekDay != weekDay) || forceUpdate)
   {
      _weekDay = weekDay;
      notify(NAVI_WEEK_DAY_CHANGED);
   }
}


void InfoStoreBase::setCurrentHour(uint32_t clockHour, bool forceUpdate)
{
   if ((_clockHour != clockHour) || forceUpdate)
   {
      _clockHour = clockHour;
      notify(NAVI_CLOCK_HOUR_CHANGED);
   }
}


void InfoStoreBase::setCurrentMinute(uint32_t clockMinute, bool forceUpdate)
{
   if ((_clockMinutes != clockMinute) || forceUpdate)
   {
      _clockMinutes = clockMinute;
      notify(NAVI_CLOCK_MINUTE_CHANGED);
   }
}


void InfoStoreBase::setNaviServiceRequestedData(const bool isNaviServiceRequestedData, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setNaviServiceRequestedData(%d)", isNaviServiceRequestedData));

   if (_isNaviServiceRequestedData != isNaviServiceRequestedData
         || forceUpdate)
   {
      _isNaviServiceRequestedData = isNaviServiceRequestedData;
      notify(NAVI_NAVIGTION_REQUESTED_DATA_CHNAGED);
   }
}


InfoStoreBase::RouteToBeUsedForE2EGuidance& InfoStoreBase::getRouteToBeUsedForE2EGuidance()
{
   return _routeToBeUsedForE2EGuidance;
}


void InfoStoreBase::setSpmSystemState(enSpmSystemState systemState, bool forceUpdate)
{
   ETG_TRACE_COMP(("InfoStoreBase::setSystemState(%d)", systemState));
   if (forceUpdate || (systemState != _spmSystemState))
   {
      ETG_TRACE_COMP(("InfoStoreBase::setSystemState Changed"));
      _spmSystemState = systemState;
      notify(NAVI_SYSTEM_STATE_CHANGED);
   }
}


void InfoStoreBase::setHmiAppState(hmibase::hmiappstates hmiAppState)
{
   ETG_TRACE_COMP(("InfoStoreBase::setHmiAppState(%d)", hmiAppState));
   if (_hmiAppState != hmiAppState)
   {
      _hmiAppState = hmiAppState;
      notify(NAVI_HMI_APP_STATE_CHANGED);
   }
}


void InfoStoreBase::setSatelliteSource(uint8_t satelliteSource)
{
   ETG_TRACE_COMP(("InfoStoreBase::setSatelliteSource(%d)", satelliteSource));
   if (satelliteSource != _satelliteSource)
   {
      _satelliteSource = satelliteSource;
      NaviDataPoolConfig::getInstance()->setSatelliteSource(satelliteSource);
   }
}


void InfoStoreBase::setMapCameraViewMode(enMapCameraAndViewModeMode mapCameraViewMode)
{
   ETG_TRACE_COMP(("InfoStoreBase::setMapCameraViewMode(%d)", mapCameraViewMode));
   if (mapCameraViewMode != _mapCameraViewMode)
   {
      _mapCameraViewMode = mapCameraViewMode;
      notify(NAVI_MAP_CAMERA_VIEW_MODE_CHANGED);
   }
}


bool InfoStoreBase::isSecondaryMapStreamingActive()
{
   // Returns true if map streaming is active in any one of the cabin regions
   bool status = false;
   for (int regionId = 0; regionId < MAX_REG; regionId++)
   {
      if (true == _mapOutStatus[regionId].mapOutStatus)
      {
         status = true;
         break;
      }
   }
   return status;
}


void InfoStoreBase::setSplitScreenEnabled(bool splitScreenEnabled)
{
   if(_isSplitScreenEnabled != splitScreenEnabled)
   {
      ETG_TRACE_COMP(("InfoStoreBase::setSplitScreenEnabled(%d)", splitScreenEnabled));
      _isSplitScreenEnabled = splitScreenEnabled;
      notify(NAVI_SPLIT_MAP_STATUS_CHANGED);
   }
}


bool InfoStoreBase::isSplitScreenEnabled()
{
   return _isSplitScreenEnabled;
}


void InfoStoreBase::addMapOutRequest(const MapOutRegionStatus& mapOutRequestInfo)
{
   if (mapOutRequestInfo.regionId < MAX_REG)
   {
      _mapOutRequestQueue.push(mapOutRequestInfo);
   }
}


const InfoStoreBase::MapOutRegionStatus InfoStoreBase::getMapOutRequest() const
{
   InfoStoreBase::MapOutRegionStatus mapOutRegionStatus;
   if(0 < _mapOutRequestQueue.size())
   {
      mapOutRegionStatus = _mapOutRequestQueue.front();
   }
   return mapOutRegionStatus;
}


void InfoStoreBase::removeMapOutRequest()
{
   if(0 < _mapOutRequestQueue.size())
   {
      _mapOutRequestQueue.pop();
   }
}


bool InfoStoreBase::isMapOutRequestPending() const
{
   bool isPending = (0 == _mapOutRequestQueue.size()) ? false : true;
   return isPending;
}


void InfoStoreBase::setMapOutStatus(const REGION& regionId, const bool& mapOutStatus)
{
   if (regionId < MAX_REG)
   {
      _mapOutStatus[regionId].mapOutStatus = mapOutStatus;
   }
   notify(NAVI_MAP_STREAMING_STATUS_CHANGED);
}


bool InfoStoreBase::getMapOutStatus(const REGION& regionId) const
{
   bool status = (regionId < MAX_REG) ? _mapOutStatus[regionId].mapOutStatus : false;
   return status;
}
