/**
 * @file        :  CommonUnitsUtility.cpp
 * @addtogroup  :  AppHmi_App_Common
 * @brief       :  Contains all utility methods used for Distance and Speed calculations
 * @copyright   :  (c) 2018-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */
#include "asf/core/Types.h"
#include "CommonUnitsUtility.h"
#include "math.h"
#include <sstream>
#include <iomanip>
#include <stdlib.h>
#include <errno.h>

#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_KDS_FI
#include "dp_kds_if.h"
#endif

/*
 * @Constructor
 */
CommonUnitsUtility::CommonUnitsUtility()
{
   //Region types available in "\nincg3_GEN\ai_projects\generated\components\vd_vehicledata\Scope2\UnitList.h"
#ifdef DP_DATAPOOL_ID
   _uRegionType = ((DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "DestinationRegion1", &_uRegionType, 1)) ? _uRegionType : 0);
#else
   _uRegionType = 0;
#endif
}


/**
 * @Destructor
 */
CommonUnitsUtility::~CommonUnitsUtility()
{
}


/*--------Defining the macros, before usage in functions--------*/
/*If defined before namespace, it will be accessible to all*/
#define DIST_CONVERT_KM_100000  100000
#define MILE_TO_M_CONV  1609.34
#define DIST_CONVERT_KM_1000  1000
#define DIST_1_MILE  1.0
#define DIST_CONVERT_MILES_TO_FEET  5280.0
#define DIST_CONVERT_MILES_TO_YARDS  1760.0
#define DIST_THREE_FOURTH_MILE  0.75
#define DIST_HALF_MILE  0.5
#define DIST_ONE_FOURTH_MILE  0.25
#define DIST_1000_UNITS  1000.0
#define DIST_300_UNITS  300.0
#define DIST_100_UNITS  100.0
#define DIST_CONVERT_KM_TO_MILES  0.621371
#define MINUTES_PER_HOUR 60
#define SECONDS_PER_HOUR 3600
#define MIN_DIGIT_OF_COORDINATE_FORMAT 8
#define MIN_DIGIT_OF_DECIMAL_FORMAT 4

/*----------Define symbol for degree, minute and second-------------*/
/*-----------For Latitude & Longitude formatconversion--------------*/
static const std::string separatingSymbol = ".";
static const unsigned int numberOfDigit = 2; // LAT: 00.00.00 - LONG: 000.00.00
static const unsigned int minuteSymbolIndex = 5;
static const unsigned int secondSymbolIndex = 7;


/**
 * Helper function to get distance as per system units.
 * @param [in] : distanceInMiles
 * @param [in] : Type of unit
 * @return : formatted distance string
 */
std::string CommonUnitsUtility::convertDistance(float distance, eDistanceUnitType fromDistType, eDistanceUnitType toDistType)
{
   distance = (distance < 0.0) ? 0.0 : distance; //handle negative distance
   eDistanceUnitType distanceUnitType = toDistType;
   std::ostringstream sstreamDist;
   float distanceInMtrs = (EN_MILES == fromDistType && EN_KM == toDistType) ? (distance * MILE_TO_M_CONV) :
                          ((EN_KM == fromDistType && EN_KM == toDistType) ? (distance * DIST_1000_UNITS) : 0);
   distance = (EN_KM == fromDistType && EN_MILES == toDistType) ? (distance * DIST_CONVERT_KM_TO_MILES) :
              ((EN_MILES == fromDistType && EN_MILES == toDistType) ? distance : 0);

   switch (distanceUnitType)
   {
      case EN_MILES:
      {
         if (_uRegionType == static_cast<tU8>(EN_USA))
         {
            americanUnitHandler(sstreamDist, distance);//US MILES handling
         }
         else if (_uRegionType == static_cast<tU8>(EN_UK))
         {
            britishUnitHandler(sstreamDist, distance);//UK miles handling
         }
      }
      break;

      case EN_KM:
      {
         const char* kilomtr = " km";
         unsigned int uDistMts = static_cast <unsigned int>(distanceInMtrs);
         if (uDistMts >= DIST_CONVERT_KM_100000)  //Distance More than 100K == 100000 meters
         {
            sstreamDist << (uDistMts / DIST_CONVERT_KM_1000) << kilomtr;
         }
         else //between 1 Km and 100 km
         {
            sstreamDist << convertDistanceToString(distanceInMtrs / DIST_CONVERT_KM_1000) << kilomtr;
         }
      }
      break;

      default://unknown unit
      {
         sstreamDist << "";
      }
      break;
   }
   return sstreamDist.str();
}


/**
 * Helper function to format for American unit (feature ID 10783 - US standard unit)
 * @param [in] : streamBuffer
 * @param [in] : distanceInMiles
 */
void CommonUnitsUtility::americanUnitHandler(std::ostringstream& sstreamDist, float distanceInMiles)
{
   unsigned int u32distanceInMiles = static_cast <unsigned int>(distanceInMiles);
   const char* feet = " ft";
   const char* mile = " mi";
   float distInFT = (distanceInMiles * DIST_CONVERT_MILES_TO_FEET);
   unsigned int u32DistInFT = static_cast <unsigned int>(distInFT);

   if (distanceInMiles >= DIST_100_UNITS)
   {
      sstreamDist << u32distanceInMiles << mile;
   }
   else if ((distanceInMiles >= DIST_1_MILE) && (distanceInMiles < DIST_100_UNITS))
   {
      sstreamDist << convertDistanceToString(distanceInMiles) << mile;
   }
   else if ((distanceInMiles < DIST_1_MILE) && (distanceInMiles >= DIST_THREE_FOURTH_MILE))
   {
      sstreamDist << "3/4" << mile;
   }
   else if ((distanceInMiles < DIST_THREE_FOURTH_MILE) && (distanceInMiles >= DIST_HALF_MILE))
   {
      sstreamDist << "1/2" << mile;
   }
   else if ((distanceInMiles < DIST_HALF_MILE) && (distanceInMiles >= DIST_ONE_FOURTH_MILE))
   {
      sstreamDist << "1/4" << mile;
   }
   else if ((distanceInMiles <  DIST_ONE_FOURTH_MILE) && (distInFT >= DIST_1000_UNITS))
   {
      sstreamDist << "1000" << feet;
   }
   else if ((distInFT < DIST_1000_UNITS) && (distInFT >= DIST_300_UNITS))
   {
      sstreamDist << roundUpToInteger(u32DistInFT, 100) << feet;//increment in 100
   }
   else if (distInFT <  DIST_300_UNITS)
   {
      sstreamDist << roundUpToInteger(u32DistInFT, 50) << feet;//increment in 50
   }
}


/**
 * Helper function to format for British unit(feature ID 10781)
 * @param [in] : stream buffer
 * @param [in] : distanceInMiles
 */
void CommonUnitsUtility::britishUnitHandler(std::ostringstream& sstreamDist, float distanceInMiles)
{
   unsigned int u32distanceInMiles = static_cast <unsigned int>(distanceInMiles);
   float distInYards = (distanceInMiles * DIST_CONVERT_MILES_TO_YARDS);
   unsigned int u32DistInYd = static_cast <unsigned int>(distInYards);
   const char* mile = " mi";
   const char* yard = " yd";

   if (distanceInMiles >= DIST_100_UNITS)
   {
      sstreamDist << u32distanceInMiles << mile;
   }
   else if ((distanceInMiles >= DIST_1_MILE) && (distanceInMiles < DIST_100_UNITS))
   {
      sstreamDist << convertDistanceToString(distanceInMiles) << mile;
   }
   else if ((distanceInMiles < DIST_1_MILE) && (distanceInMiles >= DIST_THREE_FOURTH_MILE))
   {
      sstreamDist << "3/4" << mile;
   }
   else if ((distanceInMiles < DIST_THREE_FOURTH_MILE) && (distInYards >= DIST_1000_UNITS))
   {
      sstreamDist << "1000" << yard;
   }
   else if ((distInYards < DIST_1000_UNITS) && (distInYards >= DIST_300_UNITS))
   {
      sstreamDist << roundUpToInteger(u32DistInYd, 100) << yard;//increment in 100
   }
   else if (distInYards < DIST_300_UNITS)
   {
      sstreamDist << roundUpToInteger(u32DistInYd, 50) << yard;//increment in 50
   }
}


/*--------Undefining the macros, after usage in functions---------*/
#undef DIST_CONVERT_KM_100000
#undef MILE_TO_M_CONV
#undef DIST_CONVERT_KM_1000
#undef DIST_1_MILE
#undef DIST_CONVERT_MILES_TO_FEET
#undef DIST_CONVERT_MILES_TO_YARDS
#undef DIST_THREE_FOURTH_MILE
#undef DIST_HALF_MILE
#undef DIST_ONE_FOURTH_MILE
#undef DIST_1000_UNITS
#undef DIST_300_UNITS
#undef DIST_100_UNITS


/**
* roundUpToInteger - round the integer to given multiple
* @param[in] : number to round
* @param[in] : multiple to be rounded
* return : rounded value
*/
unsigned int CommonUnitsUtility::roundUpToInteger(unsigned int numToRound, unsigned int multiple)
{
   if (multiple == 0)
   {
      return numToRound;
   }
   int rem = numToRound % multiple;
   numToRound = (rem == 0) ? numToRound : (((numToRound + (multiple / 2)) / multiple) * multiple);
   return numToRound;
}


/**
* convertDistanceToString - Converts float value into std::string
* @param[in] : float value
* return : string value
*/
std::string CommonUnitsUtility::convertDistanceToString(float flValue)
{
   std::ostringstream buffer;
   flValue = (flValue < 0.0f) ? 0.0 : flValue; //Avoid negative numbers
   double doValue = static_cast <double>(flValue); // round & floor takes input as double
   doValue = round(doValue * 10) / 10;
   buffer << std::fixed << std::setprecision(1) << doValue;//one digit after decimal point
   return buffer.str();
}


#define TEMPCONV_CONST1  32
#define TEMPCONV_CONST2  1.8
#define MPH_TO_KPH_CONV_FACTOR  1.609344
#define KPH_TO_MPH_CONV_FACTOR  0.621371
#define MM_TO_INCH_CONV_FACTOR  25.4

/**
 * Helper Function to convert temperature from Fahrenheit to Centigrade
 * @param[in] : temperature
 * @param[in] : input temperature type
 * @param[in] : type to be converted
 * return : converted string
 */
//std::string CommonUnitsUtility::convertTemperature(int16 temp, eTempType fromTempType, eTempType toTempType)
//{
//   std::ostringstream buffer;
//   double tempValue = static_cast<double>(temp);
//   if (EN_FAHRENHEIT == fromTempType && EN_CELSIUS == toTempType)
//   {
//      tempValue = (tempValue - TEMPCONV_CONST1) / TEMPCONV_CONST2;//F to C conversion
//      tempValue = round(tempValue);
//   }
//   else if (EN_CELSIUS == fromTempType && EN_FAHRENHEIT == toTempType)
//   {
//      tempValue = (tempValue * TEMPCONV_CONST2) + TEMPCONV_CONST1;//C to F conversion
//      tempValue = round(tempValue);
//   }
//   buffer << tempValue;
//   return buffer.str();
//}
//

/**
 * Helper Function to convert wind speed from MPH to KPH.
 * param[in] : speed
 * param[in] : type of the input speed
 * param[in] : type to be converted
 * return : speed in string
 */
std::string CommonUnitsUtility::convertSpeed(unsigned int speed, eDistanceUnitType fromSpeedType, eDistanceUnitType toSpeedType)
{
   std::ostringstream buffer;
   double doValue = 0.0;

   if (EN_MILES == fromSpeedType && EN_KM == toSpeedType)
   {
      doValue = speed * MPH_TO_KPH_CONV_FACTOR;//MPH to KPH conversion
   }
   else if (EN_KM == fromSpeedType &&  EN_MILES == toSpeedType)
   {
      doValue = speed * KPH_TO_MPH_CONV_FACTOR;//KPH to MPH conversion
   }
   else if (fromSpeedType == toSpeedType)
   {
      doValue = static_cast<double>(speed);
   }
   doValue = round(doValue * 10) / 10;
   int decP = static_cast <int>((doValue - floor(doValue)) * 10);
   unsigned int precision = (decP == 0) ? 0 : 2; //digits after decimal point will be 0 / 2
   buffer << std::fixed << std::setprecision(precision) << doValue;
   return buffer.str();
}


/**
 * Helper Function to convert height
 * param[in] : height value
 * param[in] : type of the input height
 * param[in] : type to be converted
 * return : height converted
 */
//uint32 CommonUnitsUtility::convertHeight(unsigned int height, eHeightType fromHeightType, eHeightType toHeightType)
//{
//   uint32 uHeight = 0;
//   if (EN_MILLIMETER == fromHeightType && EN_INCHES == toHeightType)
//   {
//      uHeight = height / MM_TO_INCH_CONV_FACTOR;
//   }
//   return uHeight;
//}


/**
* Convert decimal display from Decimal format to Sexagesimal format
*/
CommonUnitsUtility::sCoordinateInfo CommonUnitsUtility::ConvertToSexagesimalFormat(const std::string& inputString, eCoordinatesType coordinatesDisplayType)
{
   double degreeValue = 0.0f;
   double minuteValue = 0.0f;
   double secondValue = 0.0f;
   double processValue = 0.0f;
   unsigned int offset = static_cast<unsigned int>(coordinatesDisplayType);
   sCoordinateInfo sCoordinateInfoParsed =
   {
      "00", //decimal
      "00", //fraction
      "00", //degree
      "00", //minute
      "00", //second
      NORTH
   };

   processValue = atof(inputString.c_str());

   degreeValue = static_cast<double>(static_cast<int>(processValue));
   minuteValue = static_cast<double>(static_cast<int>((processValue - degreeValue) * MINUTES_PER_HOUR));
   secondValue = static_cast<double>(static_cast<int>((processValue - degreeValue - minuteValue / MINUTES_PER_HOUR) * SECONDS_PER_HOUR));

   sCoordinateInfoParsed.degree = to_string(degreeValue);
   sCoordinateInfoParsed.minute = to_string(minuteValue);
   sCoordinateInfoParsed.second = to_string(secondValue);
   sCoordinateInfoParsed.decimal = getSubstring(inputString, 0, numberOfDigit + offset);
   sCoordinateInfoParsed.fraction = getSubstring(inputString, inputString.find(separatingSymbol) + 1, inputString.length() - inputString.find(separatingSymbol) - 1);

   return sCoordinateInfoParsed;
}


std::string CommonUnitsUtility::getSubstring(std::string stringToExtract, unsigned int fromPos, unsigned int numChar)
{
   std::string retString;
   if ((numChar > 0) && ((fromPos + numChar) <= stringToExtract.length()))
   {
      retString = stringToExtract.substr(fromPos, numChar);
   }
   return retString;
}


double CommonUnitsUtility::stringToDouble(std::string processString)
{
   errno = 0;
   double retVal = strtod(processString.c_str(), NULL);
   if (0 != errno)
   {
      retVal = 0.0;
   }
   return retVal;
}


std::string CommonUnitsUtility::to_string(double const& value)
{
   std::stringstream sstr;
   if (value < 10)
   {
      sstr << '0';
   }
   sstr << value;
   return sstr.str();
}


/**
* Convert decimal display from Sexagesimal format to decimal format
*/
CommonUnitsUtility::sCoordinateInfo CommonUnitsUtility::ConvertToDecimalFormat(const std::string& inputString, eCoordinatesType coordinatesDisplayType)
{
   double degreeValue = 0.0f;
   double minuteValue = 0.0f;
   double secondValue = 0.0f;
   double processValue = 0.0f;
   double fractionalValue = 0.0f;
   sCoordinateInfo sCoordinateInfoParsed =
   {
      "00", //decimal
      "00", //fraction
      "00", //degree
      "00", //minute
      "00", //second
      NORTH
   };
   unsigned int offset = static_cast<unsigned int>(coordinatesDisplayType);

   if (inputString.length() >= (MIN_DIGIT_OF_COORDINATE_FORMAT + offset))
   {
      sCoordinateInfoParsed.degree = getSubstring(inputString, 0, numberOfDigit + offset);
      sCoordinateInfoParsed.minute = getSubstring(inputString, (minuteSymbolIndex - numberOfDigit + offset), numberOfDigit);
      sCoordinateInfoParsed.second = getSubstring(inputString, (inputString.length() - numberOfDigit), numberOfDigit);

      degreeValue = stringToDouble(sCoordinateInfoParsed.degree);
      minuteValue = stringToDouble(sCoordinateInfoParsed.minute);
      secondValue = stringToDouble(sCoordinateInfoParsed.second);

      processValue = degreeValue + ((minuteValue / MINUTES_PER_HOUR) + (secondValue / SECONDS_PER_HOUR));
      fractionalValue = round((processValue - static_cast<int>(processValue)) * 100000); //to get 5 digits only

      sCoordinateInfoParsed.decimal = to_string(processValue);
      sCoordinateInfoParsed.fraction = to_string(fractionalValue);
   }
   return sCoordinateInfoParsed;
}


/**
* Helper Function to convert coordinates display format
* param[in] : string to convert
* param[in] : coordinate type (latitude or longitude)
* param[in] : display format to be converted
   Must be XX.XX.XX or XXX.XX.XX - if in Sexagesimal format
           XX.XXXXX or XXX.XXXXX - if in Decimal format
* return : string converted - return empty string if the inputString is not in correct format before converting
*/
CommonUnitsUtility::sCoordinateInfo CommonUnitsUtility::convertCoordinateDisplayFormat(const std::string& inputString, eCoordinatesType coordinatesDisplayType, eCoordinatesDisplayFormat coordinatesDisplayFormat)
{
   std::string processString = inputString;
   sCoordinateInfo sCoordinateInfoParsed =
   {
      "00", //decimal
      "00", //fraction
      "00", //degree
      "00", //minute
      "00", //second
      NORTH
   };
   bool isNegativeValue = false;
   if (inputString[0] == '-')
   {
      isNegativeValue = true;
      processString = getSubstring(inputString, 1, inputString.length() - 1); // ignore the sign
   }
   if (EN_DECIMAL == coordinatesDisplayFormat)
   {
      sCoordinateInfoParsed = ConvertToDecimalFormat(processString, coordinatesDisplayType);
   }
   else // (EN_SEXAGESIMAL == coordinatesDisplayFormat)
   {
      sCoordinateInfoParsed = ConvertToSexagesimalFormat(processString, coordinatesDisplayType);
   }
   if (coordinatesDisplayType == EN_LATITUDE)
   {
      if (isNegativeValue)
      {
         sCoordinateInfoParsed.direction = SOUTH;
      }
      else
      {
         sCoordinateInfoParsed.direction = NORTH;
      }
   }
   else
   {
      if (isNegativeValue)
      {
         sCoordinateInfoParsed.direction = WEST;
      }
      else
      {
         sCoordinateInfoParsed.direction = EAST;
      }
   }
   return sCoordinateInfoParsed;
}


#undef TEMPCONV_CONST1
#undef TEMPCONV_CONST2
#undef MPH_TO_KPH_CONV_FACTOR
#undef KPH_TO_MPH_CONV_FACTOR
#undef MM_TO_INCH_CONV_FACTOR
