/**************************************************************************************
* @file         : HMIModelNavigationService.cpp
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "HMIModelNavigationService.h"
#include "AppBase/ScreenBrokerClient/ScreenBrokerClient.h"
#include "ProjectBaseTypes.h"
#include "Common/Util/NavMiddlewareUtils.h"
#include "NavigationServiceUtils.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#include "trcGenProj/Header/HMIModelNavigationService.cpp.trc.h"
#endif

#if defined(HALL_TO_MDW_COM)
#if defined(HALL_NAVIGATION_SERVICE)

#define NEARBY_CATEGORY_SHORTCUT_MAX 4
static const int DEFAULT_ZOOM_SCALE_FOR_LOCATION_OVERVIEW = 200;
static const int MAX_ALLOWED_ADDRESS_BOOK_ENTRIES_COUNT = 200;
static const int MAX_ALLOWED_ADDRESS_LIST_COUNT = 20;
static const int WAYPOINT_DELETE_POPUP = 7;
static const int WAYPOINT_DELETEALL_POPUP = 8;

using namespace ::org::bosch::cm::navigation::NavigationService;
using namespace navmiddleware;

HMIModelNavigationService::HMIModelNavigationService(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , NavigationServiceStub("navigationPort")
   , _cityAddressRequestAct(0)
   , _sdsCheckAddressRequestAct(0)
   , _selectSdsRefinementAct(0)
   , _freeTextSearchResultAct(0)
   , _getAdressbookDestinationAct(0)
   , _getLastDestinationAct(0)
   , _getPersonalEPOIAct(0)
   , _getSdsGetRefinementListAct(0)
   , _getRequestCityNameAct(0)
   , _getNavDataCompatibilityAct(0)
   , _getHouseNumberPatternAct(0)
   , _getDeleteWaypointAct(0)
   , _storePOILocationToAddressBookAct(0)
   , _storeLocationToAddressBookAct(0)
   , _getRequestLocationAttributesAct(0)
   , _requestDetailsForSelectedCoordinateWithRadiusRequestAct(0)
   , _routeCriterion(RouteCriterion__ROUTE_CRITERION_ECONOMIC)
   , _deleteWaypointCount(0)
   , _deleteWaypointStartIndex(-1)
   , _isZoomStreetPending(false)
   , _zoomLevelSteps(0)
   , _zoomToStreetScaleLevel(0)
   , _isZoomOutPending(false)
   , _isZoomInPending(false)
   , _isAdjustCurrentLocationPending(false)
   , _showLocationSelectionAndGetAddressAct(0)
   , _ftsLocationType(FTSLocationType__FREETEXT_SEARCH_TYPE_POI)
   , _isCancelRouteGuidancePending(false)
   , _etcStatusCache(ETCStatus__INACTIVE)
   , _isVicsSourceDsrcConnected(true)
   , _getJamAheadListAct(0)
   , _getFixedPOICategorySearchAct(0)
   , _getSetLocationWithFixedPOICategoryInputAct(0)
   , _hmiModelNavigationServiceExtended(navMiddleware, infoStore, *this)
   , _getShowAndSetLocationWithDetailsAct(0)
   , _startTourCalculationAct(0)
{
}


HMIModelNavigationService::~HMIModelNavigationService()
{
}


void HMIModelNavigationService::initialize()
{
   setNavStatus(NavStatus__NAVSTATUS_IDLE);
   setRouteRecalculationReason(RouteRecalculationReason__NO_ROUTE_AVAILABLE);
   _navMiddleware.registerGuidancePropertyUpdateCallback(*this);
   _navMiddleware.registerPositionPropertyUpdateCallback(*this);
   _navMiddleware.registerRoutePropertyUpdateCallback(*this);
   _navMiddleware.registerLocationPropertyUpdateCallback(*this);
   _navMiddleware.registerMapPropertyUpdateCallback(*this);
   _navMiddleware.registerDestinationMemoryPropertyUpdateCallback(*this);
   _navMiddleware.registerTrafficMessagePropertyUpdateCallback(*this);
   _navMiddleware.registerNavDataUpdatePropertyUpdateCallback(*this);
   _navMiddleware.registerLocationBasedServicesPropertyUpdateCallback(*this);
   _navMiddleware.registerConfigurationPropertyUpdateCallback(*this);
   _infoStore.registerDataPropertyUpdateCallback(*this);

   _navMiddleware.requestNavigationDataVersionInfo();

   //Need to nitify Navigation Settings to AppHmi_Vehicle
   updateNavigationSettingsToOtherApps();
}


void HMIModelNavigationService::deinitialize()
{
   _navMiddleware.unregisterGuidancePropertyUpdateCallback(*this);
   _navMiddleware.unregisterPositionPropertyUpdateCallback(*this);
   _navMiddleware.unregisterRoutePropertyUpdateCallback(*this);
   _navMiddleware.unregisterLocationPropertyUpdateCallback(*this);
   _navMiddleware.unregisterMapPropertyUpdateCallback(*this);
   _navMiddleware.unregisterDestinationMemoryPropertyUpdateCallback(*this);
   _navMiddleware.unregisterTrafficMessagePropertyUpdateCallback(*this);
   _navMiddleware.unregisterNavDataUpdatePropertyUpdateCallback(*this);
   _navMiddleware.unregisterLocationBasedServicesPropertyUpdateCallback(*this);
   _navMiddleware.unregisterConfigurationPropertyUpdateCallback(*this);
   _infoStore.unregisterDataPropertyUpdateCallback(*this);
   setNavStatus(NavStatus__NAVSTATUS_UNKNOWN);
   setRouteRecalculationReason(RouteRecalculationReason__UNKNOWN);
}


void HMIModelNavigationService::onPropertyHomePositionStatusChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyHomePositionStatusChanged(%d)", _navMiddleware.getHomeStatus()));
   /* @ToDo: Implementation to be added if required
   POST_MSG((COURIER_MESSAGE_NEW(HomePositionStatusUpdMsg)(static_cast< Courier::UInt32 >(_navMiddleware.getHomeStatus())))); */
}


void HMIModelNavigationService::onPropertyEVBatteryStatusWarningSettingsChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyEVBatteryStatusWarningSettingsChanged(%d)", _navMiddleware.getEvBatteryStatusWarningSettings().isLowBatteryWarningActive()));
   /* @ToDo: Implementation to be added if required
   setLowBatteryWarningActive(_navMiddleware.getEvBatteryStatusWarningSettings().isLowBatteryWarningActive()); */
}


void HMIModelNavigationService::onPropertyLocationBasedServicesSettingsChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyLocationBasedServicesSettingsChanged(%d)", _navMiddleware.getLocationBasedServicesSettings().getWeatherInformationSource()));
}


bool HMIModelNavigationService::onPropertyUpdateGuidanceStatusChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateGuidanceStatusChanged(%d)", _navMiddleware.isGuidanceActive()));
   setNavStatus(_navMiddleware.isGuidanceActive() ? NavStatus__NAVSTATUS_GUIDANCE_ACTIVE : NavStatus__NAVSTATUS_IDLE);

   if (_isCancelRouteGuidancePending)
   {
      POST_MSG((COURIER_MESSAGE_NEW(RefreshMapOnBackGroundReqMsg)()));
   }
   _isCancelRouteGuidancePending = false;
   return true;
}


void HMIModelNavigationService::recoverNavData()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::recoverNavData()"));
   setNavStatus(NavStatus__NAVSTATUS_MAP_RECOVERY_MODE);
}


bool HMIModelNavigationService::onPropertyUpdateRouteCalculationStarted()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateRouteCalculationStarted()"));
   if (getNavStatus() != NavStatus__NAVSTATUS_GUIDANCE_ACTIVE)
   {
      setNavStatus(NavStatus__NAVSTATUS_CALCULATING_ROUTE);
   }

   if (_startTourCalculationAct != 0)
   {
      sendShowCalculateTourResponse();
      _infoStore.setGuidanceStatefromSDS(true);
      POST_MSG((COURIER_MESSAGE_NEW(StartRouteGuidanceFromSpeechReqMsg)())); //starts Route guidance
      showMapMain();
      _startTourCalculationAct = 0;
   }
   return true;
}


bool HMIModelNavigationService::onPropertyUpdateNavDataUpdateFinished()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateNavDataUpdateFinished()"));
   setNavStatus(NavStatus__NAVSTATUS_IDLE);
   return true;
}


bool HMIModelNavigationService::onPropertyUpdateNavDataUpdateStarted()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateNavDataUpdateStarted()"));
   setNavStatus(NavStatus__NAVSTATUS_NAVDATA_UPDATING);
   return true;
}


bool HMIModelNavigationService::onPropertyUpdateManeuverViewChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateManeuverViewChanged()"));

   const ManeuverViewInfo maneuverViewInfo = _navMiddleware.getManeuverViewInfo();
   DeformedImage deformedImage;

   deformedImage.setValid((_navMiddleware.getManeuverViewInfo().getManeuverViewType() == navmiddleware::MANEUVERVIEWTYPE_JUNCTION_VIEW));
   deformedImage.setDeformedImageRoadType(convertDeformedImageRoadType(maneuverViewInfo.getDeformedImageDescription().m_deformedImageRoadType));
   deformedImage.setDeformedImageArrowType(convertDeformedImageArrowType(maneuverViewInfo.getDeformedImageDescription().m_deformedImageArrowType));

   ManeuverView maneuverView;
   maneuverView.setManeuverViewType(convertManeuverViewType(maneuverViewInfo.getManeuverViewType()));
   maneuverView.setDeformedImage(deformedImage);
   maneuverView.setManeuverId(maneuverViewInfo.getManeuverId());

   setManeuverView(maneuverView);

   return true;
}


::org::bosch::cm::navigation::NavigationService::ManeuverViewType HMIModelNavigationService::convertManeuverViewType(navmiddleware::ManeuverViewType maneuverViewType)
{
   switch (maneuverViewType)
   {
      case MANEUVERVIEWTYPE_NONE:
         return ManeuverViewType__NONE;
      case MANEUVERVIEWTYPE_JUNCTION_VIEW:
         return ManeuverViewType__DEFORMED_IMAGE;
      case MANEUVERVIEWTYPE_INTERSECTION_MAP:
         return ManeuverViewType__INTERSECTION_MAP;
      default:
         ETG_TRACE_USR4(("Default case returning ManeuverViewType__NONE"));
         return ManeuverViewType__NONE;
   }
}


::org::bosch::cm::navigation::NavigationService::DeformedImageRoadType HMIModelNavigationService::convertDeformedImageRoadType(navmiddleware::DeformedImageRoadType deformedImageRoadType)
{
   switch (deformedImageRoadType)
   {
      case DEFORMEDIMAGEROADTYPE__NONE:
         return DeformedImageRoadType__NONE;
      case DEFORMEDIMAGEROADTYPE__FORK:
         return DeformedImageRoadType__FORK;
      case DEFORMEDIMAGEROADTYPE__EXIT_LEFT:
         return DeformedImageRoadType__EXIT_LEFT;
      case DEFORMEDIMAGEROADTYPE__EXIT_RIGHT:
         return DeformedImageRoadType__EXIT_RIGHT;
      default:
         ETG_TRACE_USR4(("Default case returning DeformedImageRoadType__NONE"));
         return DeformedImageRoadType__NONE;
   }
}


::org::bosch::cm::navigation::NavigationService::DeformedImageArrowType HMIModelNavigationService::convertDeformedImageArrowType(navmiddleware::DeformedImageArrowType deformedImageArrowType)
{
   switch (deformedImageArrowType)
   {
      case DEFORMEDIMAGEARROWTYPE__NONE:
         return DeformedImageArrowType__NONE;
      case DEFORMEDIMAGEARROWTYPE__STRAIGHT:
         return DeformedImageArrowType__STRAIGHT;
      case DEFORMEDIMAGEARROWTYPE__LEFT:
         return DeformedImageArrowType__LEFT;
      case DEFORMEDIMAGEARROWTYPE__RIGHT:
         return DeformedImageArrowType__RIGHT;
      default:
         ETG_TRACE_USR4(("Default case returning DeformedImageArrowType__NONE"));
         return DeformedImageArrowType__NONE;
   }
}


bool HMIModelNavigationService::onPropertyUpdateManeuverInfosChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateManeuverInfosChanged()"));

   const NextManeuverInfos& nextManeuverInfos = _navMiddleware.getNextManeuverInfos();
   // set current bar graph value
   BarGraphInfo barGraphInfo;
   barGraphInfo.setValid(nextManeuverInfos.isBarGraphValid());
   // calculate & send ManeuverBargraphValue update each BARGRAPH_STEP_SIZE amount changed
   const unsigned int oBarGraphValue = nextManeuverInfos.getBarGraphValue();
   const unsigned int BARGRAPH_STEP_SIZE = 100;
   const unsigned int BARGRAPH_STEP_MAX = 1000;
   unsigned int barGraphValue = (static_cast<unsigned int>(((static_cast<float>(oBarGraphValue) / BARGRAPH_STEP_SIZE)) + 0.5)) * BARGRAPH_STEP_SIZE;
   // set maneuver symbol
   unsigned int maneuverSymbol;
   navmiddleware::ManeuverSymbolType maneuverSymbolType = nextManeuverInfos.getNextManeuverSymbolInfo().m_maneuverSymbolType;
   navmiddleware::ManeuverSymbol guidacneSymbol = nextManeuverInfos.getNextManeuverSymbolInfo().m_symbolIndex;
   navmiddleware::DirectionDescription destnationDirectionSymbol = nextManeuverInfos.getDestinationDirection();
   navmiddleware::DirectionDescription roadEntryPointDirectionSymbol = nextManeuverInfos.getRoadEntryPointDirection().getValue();
   bool isValidEntryPointDirectionSymbol = nextManeuverInfos.getRoadEntryPointDirection().isValid();
   bool showManeuverInfo = false;

   barGraphValue = (barGraphValue > BARGRAPH_STEP_MAX) ? BARGRAPH_STEP_MAX : barGraphValue;
   if ((oBarGraphValue > 0) && (oBarGraphValue < BARGRAPH_STEP_SIZE))
   {
      barGraphValue = BARGRAPH_STEP_SIZE;
   }
   barGraphInfo.setValue(barGraphValue);
   barGraphInfo.setUnit(convertDistanceUnit(nextManeuverInfos.getDistanceToManeuverUnit()));
   barGraphInfo.setBarGraphMaxScale(nextManeuverInfos.getBargraphMaxScale());
   barGraphInfo.setBarGraphMinScale(nextManeuverInfos.getBargraphMinScale());
   setBarGraphInfo(barGraphInfo);

   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateManeuverInfosChanged() - BarGraphMaxScale: %s", nextManeuverInfos.getBargraphMaxScale().c_str()));
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateManeuverInfosChanged() - BarGraphMinScale: %s", nextManeuverInfos.getBargraphMinScale().c_str()));
   // set current maneuver symbol
   maneuverSymbol = mapManeuverOrDirectionSymbolIndex(maneuverSymbolType, guidacneSymbol, destnationDirectionSymbol, roadEntryPointDirectionSymbol, isValidEntryPointDirectionSymbol);
   setManeuverSymbol(static_cast <org::bosch::cm::navigation::NavigationService::ManeuverSymbol>(maneuverSymbol));

   // set RoundaboutExitNumber
   if (nextManeuverInfos.getNextManeuverSymbolInfo().m_roundaboutExitNumber.isValid())
   {
      setRoundaboutExitNumber(nextManeuverInfos.getNextManeuverSymbolInfo().m_roundaboutExitNumber.getValue());
   }
   else
   {
      setRoundaboutExitNumber(0);
   }

   // set current turn to street
   TurnToStreet turnToStreet(nextManeuverInfos.getManeuverId(), nextManeuverInfos.getTurnToStreet());
   setTurntoStreet(turnToStreet);

   NextManeuverDetails nextManeuverDetails;
   nextManeuverDetails.setDistanceToManeuver(nextManeuverInfos.getDistanceToManeuver());
   nextManeuverDetails.setDistanceToManeuverUnit(convertDistanceUnit(nextManeuverInfos.getDistanceToManeuverUnit()));
   nextManeuverDetails.setRecommendationType(convertManeuverRecommendationType(nextManeuverInfos.getRecommendationType().getValue()));
   nextManeuverDetails.setIsNewManeuver(nextManeuverInfos.isNewManeuver());
   nextManeuverDetails.setIsReRouted(false); // @ToDo: Currently hard coded to false, will be modified after MW interface is available
   setNextManeuverDetails(nextManeuverDetails);

   const std::vector<navmiddleware::LaneInfos::LaneInfo>& LaneList = _navMiddleware.getLaneInfos().getLaneInfos();
   ::std::vector< LaneRecommendation > laneRecommendations;

   for (std::vector<navmiddleware::LaneInfos::LaneInfo>::const_iterator itr = LaneList.begin(); itr != LaneList.end(); ++itr)
   {
      LaneRecommendation laneRecommendation;
      laneRecommendation.setLaneStatus(convertLaneStatus(itr->laneStatus));
      laneRecommendation.setCombinedFollowingLaneDirection(itr->combinedFollowingLaneDirection);
      laneRecommendation.setGuidedFollowingLaneDirection(itr->guidedFollowingLaneDirection);
      laneRecommendations.push_back(laneRecommendation);
   }
   setLaneRecommendationList(laneRecommendations);

   return true;
}


::org::bosch::cm::navigation::NavigationService::ManeuverRecommendationType HMIModelNavigationService::convertManeuverRecommendationType(navmiddleware::RecommendationType recommendationType)
{
   switch (recommendationType)
   {
      case RECOMMENDATION_TYPE_PREVIEW:
         return ManeuverRecommendationType__PREVIEW;
      case RECOMMENDATION_TYPE_PRELIMINARY:
         return ManeuverRecommendationType__PRELIMINARY;
      case RECOMMENDATION_TYPE_REGULAR:
         return ManeuverRecommendationType__REGULAR;
      case RECOMMENDATION_TYPE_REPEATED:
         return ManeuverRecommendationType__REPEATED;
      default:
         ETG_TRACE_USR4(("Default case returning ManeuverRecommendationType__PREVIEW"));
         return ManeuverRecommendationType__PREVIEW;
   }
}


::org::bosch::cm::navigation::NavigationService::LaneStatus HMIModelNavigationService::convertLaneStatus(navmiddleware::LaneStatus laneStatus)
{
   switch (laneStatus)
   {
      case navmiddleware::LaneStatus__UNKNOWN:
         return org::bosch::cm::navigation::NavigationService::LaneStatus__UNKNOWN;
      case navmiddleware::LaneStatus__FORBIDDEN:
         return org::bosch::cm::navigation::NavigationService::LaneStatus__FORBIDDEN;
      case navmiddleware::LaneStatus__NOT_RECOMMENDED:
         return org::bosch::cm::navigation::NavigationService::LaneStatus__NOT_RECOMMENDED;
      case navmiddleware::LaneStatus__RECOMMENDED:
         return org::bosch::cm::navigation::NavigationService::LaneStatus__RECOMMENDED;
      default:
         ETG_TRACE_USR4(("Default case returning LaneStatus__UNKNOWN"));
         return org::bosch::cm::navigation::NavigationService::LaneStatus__UNKNOWN;
   }
}


::org::bosch::cm::navigation::NavigationService::DistanceUnit HMIModelNavigationService::convertDistanceUnit(navmiddleware::DistanceUnit distanceUnit)
{
   switch (distanceUnit)
   {
      case DISTANCEUNIT_METERS:
         return DistanceUnit__METERS;
      case DISTANCEUNIT_KILOMETERS:
         return DistanceUnit__KILOMETERS;
      case DISTANCEUNIT_MILES:
         return DistanceUnit__MILES;
      case DISTANCEUNIT_FEET:
         return DistanceUnit__FEET;
      case DISTANCEUNIT_YARD:
         return DistanceUnit__YARD;
      default:
         ETG_TRACE_USR4(("Default case returning DistanceUnit__METERS"));
         return DistanceUnit__METERS;
   }
}


void HMIModelNavigationService::onGetVersionStringRequest(const ::boost::shared_ptr< GetVersionStringRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onGetVersionStringRequest()"));

   sendGetVersionStringResponse(_navMiddleware.getVersionString());
}


void HMIModelNavigationService::onShowCityInputAndGetCityCoordinatesRequest(const ::boost::shared_ptr< ShowCityInputAndGetCityCoordinatesRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onShowCityInputAndGetCityAddressRequest()"));
   _cityAddressRequestAct = request->getAct();
   /* @ToDo: Implementation to be added if required
   POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchOutReqMsg)(NAVIGATION_CONTEXT_SXM_CITY_SELECT, 0, APPID_APPHMI_NAVIGATION))); */
}


void HMIModelNavigationService::updateNavigationSettingsToOtherApps()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::updateNavigationSettingsToOtherApps()"));
   ETG_TRACE_USR4(("HMIModelNavigationService::updateNavigationSettingsToOtherApps(): TrafficSignInformation value = %d", _navMiddleware.getGuidanceSettings().getTrafficSignInformation()));
   setTrafficSignInfoActive(_navMiddleware.getGuidanceSettings().getTrafficSignInformation());
   sendTrafficSignInfoActiveUpdate();
   ETG_TRACE_USR4(("HMIModelNavigationService::updateNavigationSettingsToOtherApps(): isOverSpeedActive value = %d", _navMiddleware.getSafetyGuidanceSettings().isOverSpeedWarningActive()));
   setOverSpeedSoundAlertInfoActive(_navMiddleware.getSafetyGuidanceSettings().isOverSpeedWarningActive());
   sendOverSpeedSoundAlertInfoActiveUpdate();
   ETG_TRACE_USR4(("HMIModelNavigationService::updateNavigationSettingsToOtherApps(): isSafetyCameraIconActive value = %d", _navMiddleware.getSafetyCameraSettings().isSafetyCameraIconActive()));
   setRiskAreaDisplayInfoActive(_navMiddleware.getSafetyCameraSettings().isSafetyCameraIconActive());
   sendRiskAreaDisplayInfoActiveUpdate();
   ETG_TRACE_USR4(("HMIModelNavigationService::updateNavigationSettingsToOtherApps(): isSafetyCameraWarningActive value = %d", _navMiddleware.getSafetyCameraSettings().isSafetyCameraWarningActive()));
   setRiskAreaSoundAlertInfoActive(_navMiddleware.getSafetyCameraSettings().isSafetyCameraWarningActive());
   sendRiskAreaSoundAlertInfoActiveUpdate();

   //Set Low battery warning setting
   /* @ToDo: Implementation to be added if required
   setLowBatteryWarningActive(_navMiddleware.getEvBatteryStatusWarningSettings().isLowBatteryWarningActive()); */

   onPropertyUpdateTrafficMessageSettingChanged();
}


void HMIModelNavigationService::onStartGuidanceRequest(const ::boost::shared_ptr< StartGuidanceRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onStartGuidanceRequest()"));
   _infoStore.setGuidanceStatefromSDS(true);
   POST_MSG((COURIER_MESSAGE_NEW(StartRouteGuidanceFromSpeechReqMsg)()));
   showMapMain();
   sendStartGuidanceResponse();
}


void HMIModelNavigationService::onSetLocationWithSdsInputRequest(const ::boost::shared_ptr< SetLocationWithSdsInputRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSetLocationWithSdsInputRequest()"));
   _navMiddleware.setLocationWithSdsInput();
   sendSetLocationWithSdsInputResponse();
}


void HMIModelNavigationService::onStartGuidanceToPosWGS84Request(const ::boost::shared_ptr< StartGuidanceToPosWGS84Request >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onStartGuidanceToPosWGS84Request()"));
   startGuidanceToPosWGS84(_navMiddleware, PosWGS84<double>(request->getDestination().getLongitude(), request->getDestination().getLatitude()));
   showMapMain();
   sendStartGuidanceToPosWGS84Response();
}


void HMIModelNavigationService::onRequestLocationDetailsForCoordinatesRequest(const ::boost::shared_ptr< RequestLocationDetailsForCoordinatesRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onRequestLocationDetailsForCoordinatesRequest()"));
   _getRequestCityNameAct = request->getAct();
   _navMiddleware.requestAddressAttributesForCoordinate(navmiddleware::GeoCoordinateDegree(request->getCityCoordinate().getLatitude(), request->getCityCoordinate().getLongitude()));
}


void HMIModelNavigationService::onRequestDetailsForSelectedLocationRequest(const ::boost::shared_ptr< RequestDetailsForSelectedLocationRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onRequestDetailsForSelectedLocationRequest()"));
   _getRequestLocationAttributesAct = request->getAct();
   _navMiddleware.requestLocationAttributes();
}


void HMIModelNavigationService::onRequestDetailsForSelectedCoordinateWithRadiusRequest(const ::boost::shared_ptr< RequestDetailsForSelectedCoordinateWithRadiusRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::RequestDetailsForSelectedCoordinateWithRadiusRequest()"));
   _requestDetailsForSelectedCoordinateWithRadiusRequestAct = request->getAct();
   _navMiddleware.requestAddressesNearPosition(navmiddleware::GeoCoordinateDegree(request->getLocation().getLatitude(), request->getLocation().getLongitude()), request->getRadius(), MAX_ALLOWED_ADDRESS_LIST_COUNT);
}


void HMIModelNavigationService::onPropertyLocationAddressInfosOnPositionChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyLocationAddressInfosOnPositionChanged()"));
   bool isLocationUpdated = false;

   if (_requestDetailsForSelectedCoordinateWithRadiusRequestAct != 0)
   {
      ::std::vector< ::org::bosch::cm::navigation::NavigationService::LocationDetails > locationDetailsList;
      const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getAddressesNearPositionInfos();
      const std::vector<navmiddleware::LocationInfos::LocationInfo>& locationInfo = locationInfos.getLocationInfos();
      ::org::bosch::cm::navigation::NavigationService::LocationDetails locationDetail;
      locationDetailsList.reserve(locationInfo.size());
      PositionWGS84 locationCoordinate;

      for (std::vector<navmiddleware::LocationInfos::LocationInfo>::const_iterator itr = locationInfo.begin(); itr != locationInfo.end(); ++itr)
      {
         const navmiddleware::LocationAttributeInfos& locationAttrInfos = itr->m_attributes;
         // getAddressDetailInfos function internally take care of filtering the data based on ATTRIBUTE_TYPE__ADDRESS_DETAILS type
         const navmiddleware::LocationAttributeInfos::AddressDetailInfo* addressDetailInfo = locationAttrInfos.getAddressDetailInfos();

         isLocationUpdated = updateLocationDetailsFromMap(addressDetailInfo, locationDetail);

         if (isLocationUpdated)
         {
            if ((itr->m_coordinate).isValid())
            {
               locationCoordinate.setLongitude(itr->m_coordinate.getValue().getLongitude());
               locationCoordinate.setLatitude(itr->m_coordinate.getValue().getLatitude());
            }
            locationDetail.setCoordinate(locationCoordinate);

            locationDetailsList.push_back(locationDetail);
         }
      }
      sendRequestDetailsForSelectedCoordinateWithRadiusResponse(locationDetailsList, _requestDetailsForSelectedCoordinateWithRadiusRequestAct);
      _requestDetailsForSelectedCoordinateWithRadiusRequestAct = 0;
   }
}


bool HMIModelNavigationService::onPropertyUpdateLocationCoordinateChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateLocationCoordinateChanged()"));
   if (_getRequestCityNameAct != 0)
   {
      ::org::bosch::cm::navigation::NavigationService::LocationDetails locationDetails;
      updateLocationDetails(_navMiddleware.getAddressAttributesOfCoordinate(), locationDetails);
      sendRequestLocationDetailsForCoordinatesResponse(locationDetails, _getRequestCityNameAct);
      _getRequestCityNameAct = 0;
   }
   return true;
}


bool HMIModelNavigationService::onPropertyUpdateLocationAttributesAvailable()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateLocationAttributesAvailable()"));
   if (((_getRequestLocationAttributesAct != 0) || (_getShowAndSetLocationWithDetailsAct != 0)) && (_navMiddleware.getLocationAttributes().getCoordinate().isValid()))
   {
      double latitude = _navMiddleware.getLocationAttributes().getCoordinate().getValue().getLatitude();
      double longitude = _navMiddleware.getLocationAttributes().getCoordinate().getValue().getLongitude();
      std::vector<PosWGS84<double> > coordinatesVector;
      coordinatesVector.push_back(PosWGS84<double>(longitude, latitude));

      InfoStoreBase::CoordinatesToBeShownInMap& coordinatesToBeShownInMap = _infoStore.getCoordinatesToBeShownInMap();
      coordinatesToBeShownInMap._coordinates = coordinatesVector;
   }

   if (_getRequestLocationAttributesAct != 0)
   {
      ::org::bosch::cm::navigation::NavigationService::LocationDetails locationDetails;
      updateLocationDetails(_navMiddleware.getLocationAttributes(), locationDetails);
      sendRequestDetailsForSelectedLocationResponse(locationDetails, _getRequestLocationAttributesAct);

      _getRequestLocationAttributesAct = 0;
   }

   if (_getShowAndSetLocationWithDetailsAct != 0)
   {
      sendShowAndSetLocationWithDetailsResponse(_getShowAndSetLocationWithDetailsAct);
      _getShowAndSetLocationWithDetailsAct = 0;
   }
   return true;
}


void HMIModelNavigationService::updateLocationDetails(const navmiddleware::LocationAttributeInfos& locationAttributeInfos, ::org::bosch::cm::navigation::NavigationService::LocationDetails& locationDetails)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::updateLocationDetails()"));

   const ::std::vector<const LocationAttributeInfos::LocationAttributeInfo*>& infos = locationAttributeInfos.getAttributeInfos();
   ::std::vector<const LocationAttributeInfos::LocationAttributeInfo*>::const_iterator infosIterator = infos.begin();
   for (; infosIterator != infos.end(); ++infosIterator)
   {
      if (NULL != *infosIterator)
      {
         switch ((*infosIterator)->m_type)
         {
            case LocationAttributeInfos::LocationAttributeInfo::ATTRIBUTE_TYPE__ADDRESS_DETAILS:
            {
               const LocationAttributeInfos::AddressDetailInfo* addressDetailInfo = locationAttributeInfos.getAddressDetailInfos();
               if (NULL != addressDetailInfo)
               {
                  if (!locationDetails.hasCityName())
                  {
                     ::std::string cityName = addressDetailInfo->getAddressValue(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__CITY_NAME);
                     ETG_TRACE_USR4(("cityName %s ", cityName.c_str()));
                     locationDetails.setCityName(cityName);
                  }
                  if (!locationDetails.hasCountryName())
                  {
                     ::std::string countryName = addressDetailInfo->getAddressValue(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__COUNTRY_NAME);
                     ETG_TRACE_USR4(("countryName %s ", countryName.c_str()));
                     locationDetails.setCountryName(countryName);
                  }
                  if (!locationDetails.hasCountryCode())
                  {
                     ::std::string countryCode = addressDetailInfo->getAddressValue(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__COUNTRY_CODE);
                     ETG_TRACE_USR4(("countryCode %s ", countryCode.c_str()));
                     locationDetails.setCountryCode(countryCode);
                  }
                  if (!locationDetails.hasStateName())
                  {
                     ::std::string stateName = addressDetailInfo->getAddressValue(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__STATE_NAME);
                     ETG_TRACE_USR4(("stateName %s ", stateName.c_str()));
                     locationDetails.setStateName(stateName);
                  }
                  if (!locationDetails.hasStateCode())
                  {
                     ::std::string stateCode = addressDetailInfo->getAddressValue(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__STATE_CODE);
                     ETG_TRACE_USR4(("stateCode %s ", stateCode.c_str()));
                     locationDetails.setStateCode(stateCode);
                  }
                  if (!locationDetails.hasPoiAddress())
                  {
                     ::std::string fulladdress = addressDetailInfo->getAddressValue(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__FULL_ADDRESS);
                     ETG_TRACE_USR4(("fulladdress %s ", fulladdress.c_str()));
                     locationDetails.setPoiAddress(fulladdress);
                  }
                  if (!locationDetails.hasStreetName())
                  {
                     ::std::string streetName = addressDetailInfo->getAddressValue(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__STREET_NAME);
                     ETG_TRACE_USR4(("streetName %s ", streetName.c_str()));
                     locationDetails.setStreetName(streetName);
                  }
                  if (!locationDetails.hasZipCode())
                  {
                     ::std::string zipCode = addressDetailInfo->getAddressValue(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__ZIP_CODE);
                     ETG_TRACE_USR4(("zipCode %s ", zipCode.c_str()));
                     locationDetails.setZipCode(zipCode);
                  }
               }
            }
            break;

            case LocationAttributeInfos::LocationAttributeInfo::ATTRIBUTE_TYPE__PHONE_NUMBER:
            {
               const LocationAttributeInfos::StringAttributeInfo* phoneNumInfo = locationAttributeInfos.getPhoneNumberInfo();
               if (NULL != phoneNumInfo)
               {
                  if (!locationDetails.hasPhoneNumber())
                  {
                     std::string phoneNumber = phoneNumInfo->m_content;
                     ETG_TRACE_USR4(("phoneNumber %s ", phoneNumber.c_str()));
                     locationDetails.setPhoneNumber(phoneNumber);
                  }
               }
            }
            break;

            default:
            {
               // nothing to do
            }
         }     // end of switch case
      }        // end of if null pointer check
   }           // end of for loop
   if (!locationDetails.hasCoordinate())
   {
      double latitude = locationAttributeInfos.getCoordinate().getValue().getLatitude();
      double longitude = locationAttributeInfos.getCoordinate().getValue().getLongitude();

      org::bosch::cm::navigation::NavigationService::PositionWGS84 positionWGS84;
      positionWGS84.setLatitude(latitude);
      positionWGS84.setLongitude(longitude);

      ETG_TRACE_USR4(("Latitude %f, Longitude %f ", latitude, longitude));
      locationDetails.setCoordinate(positionWGS84);
   }
}


void HMIModelNavigationService::onCancelRouteGuidanceRequest(const ::boost::shared_ptr< CancelRouteGuidanceRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onCancelRouteGuidanceRequest()"));

   if (_navMiddleware.isGuidanceActive())
   {
      _navMiddleware.stopGuidance();
      sendCancelRouteGuidanceResponse(request->getAct());
      _isCancelRouteGuidancePending = true;
      EXT_bIsGuidanceActive = false;
      POST_MSG((COURIER_MESSAGE_NEW(CancelRouteGuidanceReqMsg)()));
      POST_MSG((COURIER_MESSAGE_NEW(SetMapCameraModeReqMsg)(MAP_CAMERA_AND_VIEW_MODE_MAP_MAIN_SCREEN)));
      POST_MSG((COURIER_MESSAGE_NEW(StopGuidanceMsg)()));
   }
   else
   {
      sendCancelRouteGuidanceError(DBUS_ERROR_FAILED, "Invalid request since guidance is not active!", request->getAct());
   }
}


void HMIModelNavigationService::onShowPOILocationFromTCURequest(const ::boost::shared_ptr< ShowPOILocationFromTCURequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onShowPOILocationFromTCURequest()"));
   std::vector<navmiddleware::GeoCoordinateDegree> optionalCoordinates;
   std::vector<navmiddleware::GeoCoordinateDegree> geoCoordinateDegreeVector;
   geoCoordinateDegreeVector.clear();
   for (unsigned int i = 0; i < request->getPoiLocations().size(); ++i)
   {
      geoCoordinateDegreeVector.push_back(navmiddleware::GeoCoordinateDegree(request->getPoiLocations()[i].getLocation().getPositionWGS84().getLatitude(), request->getPoiLocations()[i].getLocation().getPositionWGS84().getLongitude()));
      ETG_TRACE_USR4(("latitide %f, longitude %f", request->getPoiLocations()[i].getLocation().getPositionWGS84().getLatitude(), request->getPoiLocations()[i].getLocation().getPositionWGS84().getLongitude()));
   }
   _infoStore.setAutoPlayCoordinates(geoCoordinateDegreeVector);
   navmiddleware::ValidValue<int> scale(DEFAULT_ZOOM_SCALE_FOR_LOCATION_OVERVIEW);

   _navMiddleware.showLocationsInMap(MAP_VIEW_ID__PRIMARY, geoCoordinateDegreeVector, optionalCoordinates, MapIconInfo(), scale, navmiddleware::CameraAnimation__ADAPTIVE);
   sendShowPOILocationFromTCUResponse();
}


void HMIModelNavigationService::onShowDestinationOverviewFromTCURequest(const ::boost::shared_ptr< ShowDestinationOverviewFromTCURequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onShowDestinationOverviewFromTCURequest()"));
}


void HMIModelNavigationService::onShowCalculateTourRequest(const ::boost::shared_ptr< ShowCalculateTourRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onShowCalculateTourRequest()"));
   navmiddleware::Tour tour;
   ::std::vector<navmiddleware::Tour::TourElement >& navTourElements = tour.getTourElements();
   const ::std::vector< TourElement >& tourElements = request->getTour();
   navTourElements.clear();
   navmiddleware::ValidValue<navmiddleware::LocationDetails> validLocationDetails;
   for (::std::vector< TourElement >::const_iterator iter = tourElements.begin(); iter != tourElements.end(); ++iter)
   {
      navmiddleware::ValidValue<uint64_t> id; // because of not known use an invalid Id
      navmiddleware::LocationOrigin LocationOrigin(LOCATION_ORIGIN__UNKNOWN);
      ::std::string uniqueID;

      validLocationDetails.setValue(
         navmiddleware::LocationDetails(
            iter->getLocation().getPhoneNumber(),
            iter->getLocation().getAddress(),
            navmiddleware::Image(
               iter->getLocation().getIcon().getImage().getSize(),
               iter->getLocation().getIcon().getImage().getBytes(),
               id,
               static_cast< navmiddleware::Image::Type >(iter->getLocation().getIcon().getImageType())
            ),
            LocationOrigin,
            uniqueID
         )
      );

      ValidValue<navmiddleware::settings::RouteCriterion> routeCriterion;
      routeCriterion.setValue(convertRouteCriterion(iter->getRouteCriterion()));

      std::string name = iter->getLocation().getLocation().getName();
      double latitude = iter->getLocation().getLocation().getPositionWGS84().getLatitude();
      double longtitude = iter->getLocation().getLocation().getPositionWGS84().getLongitude();
      ETG_TRACE_USR4(("latitide %f, longitude %f", latitude, longtitude));
      ETG_TRACE_USR4(("location name %s", name.c_str()));
      navmiddleware::Tour::TourElement tourElement(
          navmiddleware::ExternalLocation(name, navmiddleware::GeoCoordinateDegree(latitude, longtitude), validLocationDetails),
          routeCriterion,
          navmiddleware::ValidValue< navmiddleware::WaypointType >(),
          navmiddleware::ValidValue< navmiddleware::ChargingStationAdditionType >(),
          navmiddleware::ValidValue< navmiddleware::Tour::TourElement::ChargingDetails >());
      navTourElements.push_back(tourElement);
   }
   _infoStore.setTourInformation(tour);
   /* @ToDo: Implementation to be added if required
   POST_MSG((COURIER_MESSAGE_NEW(WaypointCountUpdateReqMsg)(tourElements.size()))); */

   //screen transition is based on the enum value.
   switch (request->getScreenTransition())
   {
      case ScreenTransition__ROUTE_INFO_SCREEN:
         ETG_TRACE_USR4(("HMIModelNavigationService::onShowCalculateTourRequest(): context switch to RouteCalculationScreen message posted "));
         /* @ToDo: Implementation to be added if required
           POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchOutReqMsg)(NAVIGATION_CONTEXT_ROUTE_INFO, 0, APPID_APPHMI_NAVIGATION))); */ //switches to Route info screen
         sendShowCalculateTourResponse();
         break;

      case ScreenTransition__MAIN_MAP_SCREEN:
         ETG_TRACE_USR4(("HMIModelNavigationService::onShowCalculateTourRequest(): context switch to MapMainScreen message posted"));
         _startTourCalculationAct = request->getAct();
         _navMiddleware.startTourCalculation(tour); //starts tour calculation
         break;

      default:
         ETG_TRACE_ERR(("HMIModelNavigationService::onShowCalculateTourRequest(): Invalid screen transition received"));
         sendShowCalculateTourError(DBUS_ERROR_FAILED, "Invalid screen transition received", request->getAct());
         break;
   }
}


bool HMIModelNavigationService::onPropertyUpdateTrafficMessageSettingChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateTrafficMessageSettingChanged()"));
   const navmiddleware::settings::TrafficSettings trafficSettings = _navMiddleware.getTrafficSettings();
   setNonPremiumTMCStatus(trafficSettings.isNonPremiumTrafficEnabled());

   return true;
}


void HMIModelNavigationService::onStorePOILocationToAddressBookRequest(const ::boost::shared_ptr< StorePOILocationToAddressBookRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onStorePOILocationToAddressBookRequest()"));
   _storePOILocationToAddressBookAct = request->getAct();
   std::string name = request->getPoiLocation().getLocation().getName();
   double latitude = request->getPoiLocation().getLocation().getPositionWGS84().getLatitude();
   double longtitude = request->getPoiLocation().getLocation().getPositionWGS84().getLongitude();
   ETG_TRACE_USR4(("latitide %f, longitude %f", latitude, longtitude));
   navmiddleware::ValidValue<navmiddleware::LocationDetails> validLocationDetails;
   navmiddleware::ValidValue<uint64_t> id; // because of not known use an invalid Id
   navmiddleware::LocationOrigin LocationOrigin(LOCATION_ORIGIN__UNKNOWN);
   ::std::string uniqueID;
   validLocationDetails.setValue(
      navmiddleware::LocationDetails(
         request->getPoiLocation().getPhoneNumber(), request->getPoiLocation().getAddress(),
         navmiddleware::Image(
            request->getPoiLocation().getIcon().getImage().getSize(),
            request->getPoiLocation().getIcon().getImage().getBytes(),
            id,
            static_cast< navmiddleware::Image::Type >(request->getPoiLocation().getIcon().getImageType())
         ),
         LocationOrigin,
         uniqueID
      )
   );
   _navMiddleware.setLocationWithExternalLocation(navmiddleware::ExternalLocation(name, navmiddleware::GeoCoordinateDegree(latitude, longtitude), validLocationDetails));
   _infoStore.setDestinationMemoryStatus(InfoStoreBase::DestinationMemoryStatus(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK, UINT64_MAX, ""));
   _navMiddleware.storeEntry(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK);
   _savePoiDetails.poiName = request->getPoiLocation().getLocation().getName();
   _savePoiDetails.phoneNumber.push_back(request->getPoiLocation().getPhoneNumber());
}


void HMIModelNavigationService::onStoreLocationToAddressBookRequest(const ::boost::shared_ptr<org::bosch::cm::navigation::NavigationService::StoreLocationToAddressBookRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onStoreLocationToAddressBookRequest()"));
   _storeLocationToAddressBookAct = request->getAct();

   //set required location to middleware.
   _navMiddleware.setLocationWithCoordinates(request->getLocation().getPositionWGS84().getLatitude(), request->getLocation().getPositionWGS84().getLongitude());
   //set memory status in infoStore.
   _infoStore.setDestinationMemoryStatus(InfoStoreBase::DestinationMemoryStatus(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK, UINT64_MAX, ""));
   //call middleware to store location entry in navigation address book.
   _navMiddleware.storeEntry(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK);
}


bool HMIModelNavigationService::onCourierMessage(const AddressGadgetItemSelectedMsg& oMsg)
{
   uint64_t destinationMemoryId = static_cast<uint64>(oMsg.GetDestinationMemoryId());

   ETG_TRACE_USR4(("HMIModelNavigationService::AddressGadgetItemSelectedMsg(destinationMemoryId: %d)", destinationMemoryId));
   if (_showLocationSelectionAndGetAddressAct != 0)
   {
      showLocationSelectionAndGetAddressDetailsCallback();
      _showLocationSelectionAndGetAddressAct = 0;
   }
   return true;
}


void HMIModelNavigationService::showLocationSelectionAndGetAddressDetailsCallback()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::showLocationSelectionAndGetAddressDetailsCallback()"));
   const navmiddleware::DestinationMemoryDestination& destinationMemoryDestination = _navMiddleware.getEntryDetails().getDestination();
   const navmiddleware::LocationAttributeInfos& locationAttributes = destinationMemoryDestination.getAttributes();
   const navmiddleware::LocationAttributeInfos::AddressDetailInfo* addressDetailInfo = locationAttributes.getAddressDetailInfos();
   ::org::bosch::cm::navigation::NavigationService::DestinationMemoryEntry destinationMemoryEntry;
   ::org::bosch::cm::navigation::NavigationService::DestinationMemoryEntryDetails destinationMemoryEntryDetails;

   destinationMemoryEntry.setEntryID(_navMiddleware.getEntryDetails().getId());
   destinationMemoryEntry.setName(_navMiddleware.getEntryDetails().getAttributes().getValue().getName());
   destinationMemoryEntry.setVoiceTagId(0);

   std::stringstream stream;
   if (addressDetailInfo != NULL)
   {
      ::std::map < navmiddleware::LocationAttributeInfos::AddressDetailInfo::AddressInfoType, navmiddleware::LocationAttributeInfos::AddressDetailInfo::AddressInfoContent >::const_iterator itr;
      itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__COUNTRY_NAME);
      if (itr != addressDetailInfo->getAddressInfoMap().end())
      {
         stream << "Country=" << (itr->second).value << " ";
         destinationMemoryEntryDetails.setCountryName((itr->second).value);
      }
      itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__COUNTRY_CODE);
      if (itr != addressDetailInfo->getAddressInfoMap().end())
      {
         stream << "CountryCode=" << (itr->second).value << " ";
         destinationMemoryEntryDetails.setCountryCode((itr->second).value);
      }
      itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__STATE_NAME);
      if (itr != addressDetailInfo->getAddressInfoMap().end())
      {
         stream << "State= " << (itr->second).value << " ";
         destinationMemoryEntryDetails.setStateName((itr->second).value);
      }
      itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__CITY_NAME);
      if (itr != addressDetailInfo->getAddressInfoMap().end())
      {
         stream << "City= " << (itr->second).value << " ";
         destinationMemoryEntryDetails.setCityName((itr->second).value);
      }
      itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__STREET_NAME);
      if (itr != addressDetailInfo->getAddressInfoMap().end())
      {
         stream << "Street= " << (itr->second).value << " ";
         destinationMemoryEntryDetails.setStreetName((itr->second).value);
      }
      itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__HOUSE_NUMBER);
      if (itr != addressDetailInfo->getAddressInfoMap().end())
      {
         stream << "HouseNumber= " << (itr->second).value << " ";
         destinationMemoryEntryDetails.setHouseno((itr->second).value);
      }
      itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__CROSS_STREET_NAME);
      if (itr != addressDetailInfo->getAddressInfoMap().end())
      {
         stream << "CrossStreetName= " << (itr->second).value << std::endl;
         destinationMemoryEntryDetails.setJunctionName((itr->second).value);
      }
      itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__FULL_ADDRESS);
      if (itr != addressDetailInfo->getAddressInfoMap().end())
      {
         stream << "AddressDetail= " << (itr->second).value << std::endl;
         destinationMemoryEntryDetails.setAddressDetail((itr->second).value.c_str());
      }
   }
   if (locationAttributes.getLocationType() == navmiddleware::LocationAttributeInfos::LOCATIONTYPE_POI)
   {
      if (locationAttributes.getLocationNameInfo() != NULL)
      {
         destinationMemoryEntryDetails.setPoiName(locationAttributes.getLocationNameInfo()->m_content);
         stream << "[POI NAME]" << locationAttributes.getLocationNameInfo()->m_content << std::endl;
      }
      if (locationAttributes.getPoiCategoryInfo() != NULL)
      {
         destinationMemoryEntryDetails.setPoiCategoryName(locationAttributes.getPoiCategoryInfo()->m_categoryName);
         stream << "[POI CATEGORY]" << locationAttributes.getPoiCategoryInfo()->m_categoryName << std::endl;
      }
      if (locationAttributes.getBrandNameInfo() != NULL)
      {
         destinationMemoryEntryDetails.setPoiBrandName(locationAttributes.getBrandNameInfo()->m_content);
         stream << "[POI BRAND NAME]" << locationAttributes.getBrandNameInfo()->m_content << std::endl;
      }
   }

   ::org::bosch::cm::navigation::NavigationService::DestinationMemoryDetailListElement destinationMemoryDetailListElement;

   destinationMemoryDetailListElement.setDetails(destinationMemoryEntryDetails);
   destinationMemoryDetailListElement.setEntry(destinationMemoryEntry);

   sendShowLocationSelectionAndGetAddressDetailsResponse(destinationMemoryDetailListElement, _showLocationSelectionAndGetAddressAct);
}


bool HMIModelNavigationService::onPropertyUpdateDestinationMemoryStoreEntryStatusChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateDestinationMemoryStoreEntryStatusChanged()"));
   if (_storeLocationToAddressBookAct != 0)
   {
      navmiddleware::DestinationMemoryStoreEntryStatus destinationMemoryStoreEntryStatus = _navMiddleware.getStoreEntryStatus();

      if (destinationMemoryStoreEntryStatus.getStatus() == navmiddleware::DestinationMemoryStoreEntryStatus::STOREENTRYSTATUS__OK)
      {
         ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateDestinationMemoryStoreEntryStatusChanged() Done"));
      }
      else
      {
         ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateDestinationMemoryStoreEntryStatusChanged() Failed"));
      }
      sendStoreLocationToAddressBookResponse(_storeLocationToAddressBookAct);
      _storeLocationToAddressBookAct = 0;
   }

   return true;
}


bool HMIModelNavigationService::onPropertyUpdateDestinationMemoryLocationAddressStored()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateDestinationMemoryLocationAdressStored()"));
   if (_storePOILocationToAddressBookAct != 0)
   {
      navmiddleware::DestinationMemoryStoreEntryStatus destinationMemoryStoreEntryStatus = _navMiddleware.getStoreEntryStatus();

      // @ToDo: The responsible navigation HALL developer must clarify if doing no action is correct for every use case in case of StoreEntryStatus not equal to STOREENTRYSTATUS__OK
      if (destinationMemoryStoreEntryStatus.getStatus() == navmiddleware::DestinationMemoryStoreEntryStatus::STOREENTRYSTATUS__OK)
      {
         const navmiddleware::DestinationMemoryEntryAttributes& destinationAttributeDetails = _navMiddleware.getEntryDetails().getAttributes().getValue();
         _infoStore.setDestinationAttributesInfo(destinationAttributeDetails);
         _infoStore.setDestinationAttributesInfoName(_savePoiDetails.poiName);
         _infoStore.setDestinationAttributesInfoPhoneNumber(_savePoiDetails.phoneNumber);

         _navMiddleware.modifyEntryAttributes(destinationMemoryStoreEntryStatus.getStoredEntryId(), _infoStore.getDestinationAttributesInfo());
      }
      else
      {
         ETG_TRACE_USR4(("trigger was ignored because StoreEntryStatus was not equal to STOREENTRYSTATUS__OK"));
      }
      sendStorePOILocationToAddressBookResponse(_storePOILocationToAddressBookAct);
      _storePOILocationToAddressBookAct = 0;
   }

   return true;
}


navmiddleware::settings::RouteCriterion HMIModelNavigationService::convertRouteCriterion(::org::bosch::cm::navigation::NavigationService::RouteCriterion routeCriterion)
{
   switch (routeCriterion)
   {
      case RouteCriterion__ROUTE_CRITERION_ECONOMIC:
         return  navmiddleware::settings::ROUTE_CRITERION_ECONOMIC;
      case RouteCriterion__ROUTE_CRITERION_FAST:
         return navmiddleware::settings::ROUTE_CRITERION_FAST;
      case RouteCriterion__ROUTE_CRITERION_SHORT:
         return navmiddleware::settings::ROUTE_CRITERION_SHORT;
      default:
         ETG_TRACE_USR4(("Default case returning ROUTE_CRITERION_ECONOMIC"));
         return navmiddleware::settings::ROUTE_CRITERION_ECONOMIC;
   }
}


::org::bosch::cm::navigation::NavigationService::RouteCriterion HMIModelNavigationService::convertRouteCriterion(navmiddleware::settings::RouteCriterion routeCriterion)
{
   switch (routeCriterion)
   {
      case navmiddleware::settings::ROUTE_CRITERION_ECONOMIC:
         return RouteCriterion__ROUTE_CRITERION_ECONOMIC;
      case navmiddleware::settings::ROUTE_CRITERION_FAST:
         return RouteCriterion__ROUTE_CRITERION_FAST;
      case navmiddleware::settings::ROUTE_CRITERION_SHORT:
         return RouteCriterion__ROUTE_CRITERION_SHORT;
      default:
         ETG_TRACE_USR4(("Default case returning RouteCriterion__ROUTE_CRITERION_ECONOMIC"));
         return RouteCriterion__ROUTE_CRITERION_ECONOMIC;
   }
}


bool HMIModelNavigationService::onCourierMessage(const StartCalculateTourReqMsg& /*oMsg*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::StartCalculateTourReqMsg()"));
   _navMiddleware.startTourCalculation(_infoStore.getTourInformation());
   return true;
}


void HMIModelNavigationService::onShowOnMapRequest(const ::boost::shared_ptr< ShowOnMapRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onShowOnMapRequest(latitide %f, longitude %f)", request->getLocation().getLatitude(), request->getLocation().getLongitude()));
   navmiddleware::ValidValue<int> scale(DEFAULT_ZOOM_SCALE_FOR_LOCATION_OVERVIEW);
   _navMiddleware.centerMapToGeoPosition(MAP_VIEW_ID__PRIMARY, navmiddleware::GeoCoordinateDegree(request->getLocation().getLatitude(), request->getLocation().getLongitude()), scale);
   sendShowOnMapResponse();
}


void HMIModelNavigationService::onSetZoomOutStepRequest(const ::boost::shared_ptr< SetZoomOutStepRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSetZoomInStepRequest(%d)", request->getZoomOutStep()));
   _isZoomOutPending = true;
   _zoomLevelSteps = request->getZoomOutStep();
   showMapMain();
}


bool HMIModelNavigationService::onCourierMessage(const EnteredMapScreenReqMsg& /*oMsg*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::EnteredMapScreenReqMsg(isZoomOut %d, isZoomIn %d, _zoomLevel %d)", _isZoomOutPending, _isZoomInPending, _zoomLevelSteps));

   if (_isZoomOutPending == true)
   {
      _navMiddleware.zoomOutStep(MAP_VIEW_ID__PRIMARY, _zoomLevelSteps);
      _isZoomOutPending = false;
   }
   else if (_isZoomInPending == true)
   {
      _navMiddleware.zoomInStep(MAP_VIEW_ID__PRIMARY, _zoomLevelSteps);
      _isZoomInPending = false;
   }

   _zoomLevelSteps = 0;
   if (_isAdjustCurrentLocationPending == true)
   {
      _navMiddleware.showSetLocationInMap(MAP_VIEW_ID__PRIMARY, DEFAULT_ZOOM_SCALE_FOR_LOCATION_OVERVIEW);
      _isAdjustCurrentLocationPending = false;
   }
   return true;
}


void HMIModelNavigationService::onSetZoomInStepRequest(const ::boost::shared_ptr< SetZoomInStepRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSetZoomInStepRequest(%d)", request->getZoomInStep()));
   _isZoomInPending = true;
   _zoomLevelSteps = request->getZoomInStep();
   showMapMain();
}


void HMIModelNavigationService::showMapMain()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::showMapMain()"));

   /* @ToDo: Implementation to be added if required
   POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchOutReqMsg)(NAVIGATION_CONTEXT_MAP_MAIN, 0, APPID_APPHMI_NAVIGATION))); */
}


bool HMIModelNavigationService::onPropertyUpdateGuidanceArrivalInfosChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateGuidanceArrivalInfosChanged()"));
   updateGuidanceArrivalInformation();
   return true;
}


void HMIModelNavigationService::updateGuidanceArrivalInformation()
{
   if (_navMiddleware.isGuidanceActive())
   {
      const std::vector<navmiddleware::ArrivalInfos::ArrivalInfo>& arrivalInfo = _navMiddleware.getArrivalInfosGuidedRoute().getArrivalInfos();
      ::std::vector< ArrivalInformation > arrivalInformations;
      for (std::vector<navmiddleware::ArrivalInfos::ArrivalInfo>::const_iterator itr = arrivalInfo.begin(); itr != arrivalInfo.end(); ++itr)
      {
         ArrivalInformation arrivalInformation;
         //Distance to destination
         DistanceToDestination distanceToDestination;
         distanceToDestination.setDistanceToDestination(itr->m_distanceToDest);
         distanceToDestination.setDistanceToDestinationAsString(itr->m_distanceToDestination);
         distanceToDestination.setUnit(convertDistanceUnit(itr->m_distanceUnit));
         arrivalInformation.setDistanceToDestination(distanceToDestination);
         //Time to destination
         arrivalInformation.setTimeToDestinationAsString(itr->m_timeToDestination);
         arrivalInformation.setTimeToDestination(itr->m_timeToDest);
         //Estimated time of arrival
         arrivalInformation.setEstimatedTimeOfArrivalAsString(itr->m_distanceToDestination);
         arrivalInformation.setEstimatedTimeOfArrivalHour(itr->m_estimatedTimeOfArrivalHour);
         arrivalInformation.setEstimatedTimeOfArrivalMinute(itr->m_estimatedTimeOfArrivalMinute);
         //Estimated time type
         arrivalInformation.setEstimatedTimeType(static_cast<EstimatedTimeType>(_infoStore.getEstimatedTimeType()));
         //Time type
         arrivalInformation.setDisplayTimeType(static_cast<TimeType>(_infoStore.getEstimatedTimeMode()));
         //Time Format
         arrivalInformation.setDisplayTimeFormat(static_cast< ::org::bosch::cm::navigation::NavigationService::TimeFormat>(_infoStore.getTimeFormat()));
         arrivalInformation.setDestinationIndex(_navMiddleware.getArrivalInfos().getCurrentDestinationIndex());
         //Consumption values and type
         arrivalInformation.setConsumptionUnit(static_cast< ::org::bosch::cm::navigation::NavigationService::ConsumptionUnit>(itr->m_consumptionUnit));
         arrivalInformation.setConsumptionToDestinationAsInt(itr->m_consumptionToDestAsInt);
         //m_consumptionToDestinationAsString without unit information
         arrivalInformation.setConsumptionToDestinationAsString(itr->m_consumptionToDestinationAsString);

         arrivalInformations.push_back(arrivalInformation);
      }
      setArrivalInformationList(arrivalInformations);
   }
}


bool HMIModelNavigationService::onPropertyUpdateSettingsEstimatedTimeTypeChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateSettingsEstimatedTimeTypeChanged()"));
   updateGuidanceArrivalInformation();
   return true;
}


bool HMIModelNavigationService::onPropertyRouteDestinationChanged()
{
   ETG_TRACE_USR1(("HMIModelNavigationService::onPropertyRouteDestinationChanged()"));

   const navmiddleware::DestinationInfos& routeInfoInfos = _navMiddleware.getDestinationInfos();

   const std::vector< navmiddleware::DestinationInfos::DestinationInfo>& routeInfoVector = routeInfoInfos.getDestinationInfos();
   std::vector<navmiddleware::DestinationInfos::DestinationInfo>::const_iterator info = routeInfoVector.begin();

   ::std::vector< org::bosch::cm::navigation::NavigationService::DestinationInfoElement > destinationInformation;
   for (; (info != routeInfoVector.end()); ++info)
   {
      DestinationInfoElement waypointListElement;
      org::bosch::cm::navigation::NavigationService::PositionWGS84 positionWGS84;
      positionWGS84.setLatitude(info->m_latitude);
      positionWGS84.setLongitude(info->m_longitude);
      waypointListElement.setLocation(positionWGS84);
      waypointListElement.setCountryCode(info->m_countryCode);
      waypointListElement.setSubCountryCode(info->m_subCountryCode);
      waypointListElement.setName(info->m_locationString);
      destinationInformation.push_back(waypointListElement);

      ETG_TRACE_USR4(("WaypointListElement %d, latitude %f, longitude %f, countryCode: %s", destinationInformation.size(), info->m_latitude, info->m_longitude, info->m_countryCode.c_str()));
   }

   setDestinationInformation(destinationInformation);

   return true;
}


::org::bosch::cm::navigation::NavigationService::SDSAddressElementType HMIModelNavigationService::convertSDSAddressElementType(navmiddleware::SDSAddressElementType middlSDSAddressElementType)
{
   switch (middlSDSAddressElementType)
   {
      case SDSADDRESSELEMENTTYPE_COUNTRY:
         return SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_COUNTRY;
         break;
      case SDSADDRESSELEMENTTYPE_PLACE:
         return SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_PLACE;
         break;
      case SDSADDRESSELEMENTTYPE_ROAD:
         return SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_ROAD;
         break;
      case SDSADDRESSELEMENTTYPE_CROSSROAD:
         return SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_CROSSROAD;
         break;
      case SDSADDRESSELEMENTTYPE_HOUSENUMBER:
         return SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_HOUSENUMBER;
         break;
      case SDSADDRESSELEMENTTYPE_POSTALDISRICT:
         return SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_POSTALDISTRICT;
         break;
      case SDSADDRESSELEMENTTYPE_CITYDISTRICT:
         return SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_CITYDESTRICT;
         break;
      case SDSADDRESSELEMENTTYPE_PLACEREFINEMENT:
         return SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_PLACEREFINEMENT;
         break;
      case SDSADDRESSELEMENTTYPE_ROADREFINEMENT:
         return SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_ROADREFINEMENT;
         break;
      case SDSADDRESSELEMENTTYPE_STATE:
         return SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_STATE;
         break;
      default:
         ETG_TRACE_USR4(("Default case returning country"));
         return SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_COUNTRY;
         break;
   }
}


navmiddleware::SDSAddressElementType HMIModelNavigationService::convertSDSAddressElementType(::org::bosch::cm::navigation::NavigationService::SDSAddressElementType serviceSDSAddressElementType)
{
   switch (serviceSDSAddressElementType)
   {
      case SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_COUNTRY:
         return SDSADDRESSELEMENTTYPE_COUNTRY;
         break;
      case SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_PLACE:
         return SDSADDRESSELEMENTTYPE_PLACE;
         break;
      case SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_ROAD:
         return SDSADDRESSELEMENTTYPE_ROAD;
         break;
      case SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_CROSSROAD:
         return SDSADDRESSELEMENTTYPE_CROSSROAD;
         break;
      case SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_HOUSENUMBER:
         return SDSADDRESSELEMENTTYPE_HOUSENUMBER;
         break;
      case SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_POSTALDISTRICT:
         return SDSADDRESSELEMENTTYPE_POSTALDISRICT;
         break;
      case SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_CITYDESTRICT:
         return SDSADDRESSELEMENTTYPE_CITYDISTRICT;
         break;
      case SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_PLACEREFINEMENT:
         return SDSADDRESSELEMENTTYPE_PLACEREFINEMENT;
         break;
      case SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_ROADREFINEMENT:
         return SDSADDRESSELEMENTTYPE_ROADREFINEMENT;
         break;
      case SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_STATE:
         return SDSADDRESSELEMENTTYPE_STATE;
         break;
      default:
         ETG_TRACE_USR4(("Default case returning country"));
         return SDSADDRESSELEMENTTYPE_COUNTRY;
         break;
   }
}


void HMIModelNavigationService::onSdsCheckAddressRequest(const ::boost::shared_ptr< SdsCheckAddressRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSdsCheckAddressRequest()"));
   _sdsCheckAddressRequestAct = request->getAct();
   ::std::vector< ::org::bosch::cm::navigation::NavigationService::SDSAddressElement > requestedAddress = request->getRequestedAddress();
   ::std::vector< ::org::bosch::cm::navigation::NavigationService::SDSAddressElement >::const_iterator iter = requestedAddress.begin();

   navmiddleware::SDSAddress sdsAddress;
   ::std::vector< navmiddleware::SDSAddressElement >& addressElements = sdsAddress.getAddressElementsMutable();

   // copy request data for SDS address from DBUS structures to navmiddleware structures
   for (; iter != requestedAddress.end(); ++iter)
   {
      navmiddleware::SDSAddressElement addressElement;
      addressElement.setAddressElementType(convertSDSAddressElementType(iter->getAddressElementType()));
      ETG_TRACE_USR4(("addressElementType %d", addressElement.getAddressElementType()));
      addressElement.setData(iter->getData());

      navmiddleware::SDSElementOptions sdsElementOption;
      sdsElementOption.setIsAmbiguous(iter->getElementOptions().getIsAmbigious()); // @Todo: Fix spelling error "getIsAmbigious" in navigation DBUS service

      addressElement.setAddressElementOptions(sdsElementOption);

      addressElements.push_back(addressElement);
   }

   _navMiddleware.sdsCheckAddress(sdsAddress);
}


bool HMIModelNavigationService::onPropertySdsAddressWithOptionsChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertySdsAddressWithOptionsChanged()"));
   ::org::bosch::cm::navigation::NavigationService::SDSAddressWithOptions& sdsAddressWithOptions = getSdsAddressWithOptionsMutable();

   const navmiddleware::SDSAddressWithOptions& addressWithOptions = _navMiddleware.getCurrentSDSAddressWithOptions();
   const ::std::vector< navmiddleware::SDSAddressElement >& address = addressWithOptions.getAddress().getAddressElements();
   std::vector< navmiddleware::SDSAddressElement >::const_iterator info = address.begin();

   ::std::vector< ::org::bosch::cm::navigation::NavigationService::SDSAddressElement > addressElements = sdsAddressWithOptions.getAddressMutable();
   addressElements.clear();
   addressElements.resize(addressWithOptions.getAddress().getAddressElements().size());
   for (; (info != address.end()); ++info)
   {
      ::org::bosch::cm::navigation::NavigationService::SDSAddressElement addressElement;
      ::org::bosch::cm::navigation::NavigationService::SDSAddressElementType  addressElementType;

      addressElementType = convertSDSAddressElementType(info->getAddressElementType());
      ETG_TRACE_USR4(("HMIModelNavigationService::onPropertySdsAddressWithOptionsChanged(), addressElementType: %d, addressElementData %s"
                      , addressElementType, info->getData().c_str()));

      addressElement.setAddressElementType(addressElementType);
      addressElement.setData(info->getData());

      ::org::bosch::cm::navigation::NavigationService::SDSElementOptions sdsElementOptions;
      sdsElementOptions.setIsAmbigious(info->getAddressElementOptions().isAmbiguous()); // @Todo: Fix spelling error "getIsAmbigious" in navigation DBUS service
      addressElement.setElementOptions(sdsElementOptions);

      addressElements.push_back(addressElement);
   }

   sdsAddressWithOptions.setAddress(addressElements);

   ::org::bosch::cm::navigation::NavigationService::SDSAddressOptions addressOption = sdsAddressWithOptions.getAddressOptionsMutable();
   addressOption.setNavigable(addressWithOptions.getAddressOptions().isNavigable());
   addressOption.setAmbigious(addressWithOptions.getAddressOptions().isAmbiguous()); // @Todo: Fix spelling error "getIsAmbigious" in navigation DBUS service
   addressOption.setRemovedInvalidEntry(addressWithOptions.getAddressOptions().hasRemovedInvalidEntry());
   addressOption.setMoreEntriesPossible(addressWithOptions.getAddressOptions().areMoreEntriesPossible()); // @Todo: Fix spelling error "setMoreEntrysPossible" in navigation DBUS service
   addressOption.setHouseNumberAvailable(addressWithOptions.getAddressOptions().isHousenumberAvailable());
   addressOption.setHouseNumberValid(addressWithOptions.getAddressOptions().isHousenumberValid());

   sdsAddressWithOptions.setAddressOptions(addressOption);

   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertySdsAddressWithOptionsChanged(), isNavigable: %d, isAmbiguous: %d, areMoreEntriesPossible: %d, isHousenumberAvailable: %d, isHousenumberValid: %d"
                   , addressWithOptions.getAddressOptions().isNavigable()
                   , addressWithOptions.getAddressOptions().isAmbiguous()
                   , addressWithOptions.getAddressOptions().areMoreEntriesPossible()
                   , addressWithOptions.getAddressOptions().isHousenumberAvailable()
                   , addressWithOptions.getAddressOptions().isHousenumberValid()));

   const ::std::vector< navmiddleware::SDSAddressElementType >& addressElementTypes = addressWithOptions.getAddressElementTypes();
   std::vector< navmiddleware::SDSAddressElementType >::const_iterator iter = addressElementTypes.begin();

   ::std::vector< ::org::bosch::cm::navigation::NavigationService::SDSAddressElementType > serviceAddressElementTypes = sdsAddressWithOptions.getAddressElementTypeMutable();
   serviceAddressElementTypes.clear();
   serviceAddressElementTypes.resize(addressElementTypes.size());
   for (; (iter != addressElementTypes.end()); ++iter)
   {
      ::org::bosch::cm::navigation::NavigationService::SDSAddressElementType serviceAddressElement;
      serviceAddressElement = convertSDSAddressElementType(*iter);
      ETG_TRACE_USR4(("serviceAddressElement %d", *iter));
      serviceAddressElementTypes.push_back(serviceAddressElement);
   }

   sdsAddressWithOptions.setAddressElementType(serviceAddressElementTypes);

   sendSdsAddressWithOptionsUpdate();
   if (_sdsCheckAddressRequestAct != 0)
   {
      ETG_TRACE_USR4(("send SdsCheckAddressResponse"));
      sendSdsCheckAddressResponse(_sdsCheckAddressRequestAct);
      _sdsCheckAddressRequestAct = 0;
   }
   else if (_selectSdsRefinementAct != 0)
   {
      ETG_TRACE_USR4(("send SelectSdsRefinementResponse"));
      sendSelectSdsRefinementResponse(_selectSdsRefinementAct);
      _selectSdsRefinementAct = 0;
   }
   return true;
}


bool HMIModelNavigationService::onPropertySensorInfoChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertySensorInfoChanged()"));
   const GnssDataInfo& gnssDataInfo = _navMiddleware.getGnssDataInfo();
   SatelliteInfo& satelliteInfo = getSatelliteInfoMutable();

   uint8 visibleSatellites = gnssDataInfo.getNumberOfVisibleSatellites();
   satelliteInfo.setVisibleSatellites(visibleSatellites);

   uint8 receivedSatellites = gnssDataInfo.getNumberOfReceivedSatellites();
   satelliteInfo.setReceivedSatellites(receivedSatellites);

   uint8 usedSatellites = gnssDataInfo.getNumberOfUsedSatellites();
   satelliteInfo.setUsedSatellites(usedSatellites);

   uint16 gnssSpeed = static_cast<uint16>(gnssDataInfo.getSpeed());
   satelliteInfo.setGnssSpeed(gnssSpeed);

   uint8 gnssSpeedReliability = gnssDataInfo.getSpeedReliability();
   satelliteInfo.setGnssSpeedReliability(gnssSpeedReliability);

   navmiddleware::GeoCoordinateDegree positionWGS84 = gnssDataInfo.getGnssPosition();
   PositionWGS84 gnssPosition;
   gnssPosition.setLatitude(positionWGS84.getLatitude());
   gnssPosition.setLongitude(positionWGS84.getLongitude());
   satelliteInfo.setGnssPosition(gnssPosition);

   // tbd: How has to modifify the naviagtion service interface no longer to use GnssMode as string?
   navmiddleware::GnssMode gnssMode = gnssDataInfo.getGnssMode();
   ::std::string stringGnssMode;

   switch (gnssMode)
   {
      case navmiddleware::GNSS_MODE__3D_FIX:
         stringGnssMode = "3D_FIX";
         break;
      case navmiddleware::GNSS_MODE__2D_FIX:
         stringGnssMode = "2D_FIX";
         break;
      case navmiddleware::GNSS_MODE__NO_FIX:
         stringGnssMode = "NO_FIX";
         break;
      default:
         stringGnssMode = "UNKNOWN";
         break;
   }
   satelliteInfo.setGnssMode(stringGnssMode);

   ETG_TRACE_USR4(("visibleSatellites %d, receivedSatellites %d, usedSatellites %d ", visibleSatellites, receivedSatellites, usedSatellites));
   sendSatelliteInfoUpdate();
   updatePositionInformation();
   return true;
}


void HMIModelNavigationService::updatePositionInformation()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::updatePositionInformation()"));

   const PositionStatusInfo& posStatusInfo = _navMiddleware.getPositionStatusInfo();
   const GnssDataInfo& gnssDataInfo = _navMiddleware.getGnssDataInfo();
   const PositionInfo& positionInfo = _navMiddleware.getPositionInfo();

   PositionInformation& positionInformation = getPositionInformationMutable();

   // set vehicle speed
   positionInformation.setVehicleSpeed(posStatusInfo.getSpeedInfo());

   // set vehicle heading
   positionInformation.setVehicleHeading(posStatusInfo.getHeadingInfo());

   // set current position
   PositionWGS84 positionWGS84 = positionInformation.getPositionWGS84Mutable();
   positionWGS84.setLatitude(posStatusInfo.getLatitude());
   positionWGS84.setLongitude(posStatusInfo.getLongitude());
   positionInformation.setPositionWGS84(positionWGS84);

   // set map matching position accuracy
   positionInformation.setMapMatchingPositionAccuracy(posStatusInfo.getMapMatchedPositionAccuracy().getValueOr(0));

   // set country code and set state code (/*RTC-1202932, RTC-1202491*/)
   if ((positionInfo.getUniqueID().isValid()) && (posStatusInfo.getUniqueID().isValid())
         && (positionInfo.getUniqueID().getValue() != posStatusInfo.getUniqueID().getValue()))
   {
      positionInformation.clearCountryCode();
      positionInformation.clearStateCode();
   }
   else
   {
      positionInformation.setCountryCode(posStatusInfo.getCountryCode());
      positionInformation.setStateCode(posStatusInfo.getStateCode());
   }

   // set RoadClass
   positionInformation.setRoadClass(convertRoadClass(posStatusInfo.getRoadClass()));

   // set altitude
   positionInformation.setAltitude(gnssDataInfo.getAltitude());
   //PositionInfo
   positionInformation.setCountry(positionInfo.getCurrentCountry());
   positionInformation.setProvince(positionInfo.getCurrentState());
   positionInformation.setCity(positionInfo.getCurrentCity());
   positionInformation.setCityDistrict(positionInfo.getCurrentCityDistrict());
   positionInformation.setStreet(positionInfo.getCurrentStreet());
   positionInformation.setStreetNumber(positionInfo.getCurrentStreetNumber());

   sendPositionInformationUpdate();
}


bool HMIModelNavigationService::onPropertyUpdatePositionInfoChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdatePositionInfoChanged()"));
   updatePositionInformation();
   return true;
}


bool HMIModelNavigationService::onPropertyPositionStatusChanged()
{
   updatePositionInformation();
   return true;
}


::org::bosch::cm::navigation::NavigationService::RoadClass HMIModelNavigationService::convertRoadClass(navmiddleware::RoadClass roadClass)
{
   switch (roadClass)
   {
      case ROAD_CLASS__NONE:
         return RoadClass__ROAD_CLASS_NONE;
      case ROAD_CLASS__MOTORWAY:
         return RoadClass__ROAD_CLASS_MOTORWAY;
      case ROAD_CLASS__FREEWAY:
         return RoadClass__ROAD_CLASS_FREEWAY;
      case ROAD_CLASS__HIGHWAY:
         return RoadClass__ROAD_CLASS_HIGHWAY;
      case ROAD_CLASS__ARTERIAL:
         return RoadClass__ROAD_CLASS_ARTERIAL;
      case ROAD_CLASS__COLLECTOR:
         return RoadClass__ROAD_CLASS_COLLECTOR;
      case ROAD_CLASS__LOCAL:
         return RoadClass__ROAD_CLASS_LOCAL;
      case ROAD_CLASS__ADRESSABLESTREET:
         return RoadClass__ROAD_CLASS_ADDRESSABLESTREET;
      case ROAD_CLASS__TRAIL:
         return RoadClass__ROAD_CLASS_TRAIL;
      default:
         ETG_TRACE_USR4(("Default case returning RoadClass__ROAD_CLASS_NONE"));
         return RoadClass__ROAD_CLASS_NONE;
   }
}


bool HMIModelNavigationService::onCourierMessage(const ShowCityInputAndGetCityAddressResponseReqMsg& oMsg)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onCourierMessage(ShowCityInputAndGetCityAddressResponseReqMsg())"));
   _navMiddleware.setLocationWithCurrentAddressInput();
   const AddressInputInfo& addressInputInfo = _navMiddleware.getFormBasedAddressInputInfo();
   bool isValidAddress = oMsg.GetIsValidAddress();
   PositionWGS84 cityLocation;

   if (addressInputInfo.getCurrentCoordinates().isValid())
   {
      cityLocation.setLatitude(addressInputInfo.getCurrentCoordinates().getValue().getLatitude());
      cityLocation.setLongitude(addressInputInfo.getCurrentCoordinates().getValue().getLongitude());
   }
   else
   {
      ETG_TRACE_USR4(("invalid destination!"));
      isValidAddress = false;
   }
   if (_cityAddressRequestAct != 0)
   {
      sendShowCityInputAndGetCityCoordinatesResponse(cityLocation, isValidAddress, _cityAddressRequestAct);
      _cityAddressRequestAct = 0;
   }

   return true;
}


void HMIModelNavigationService::onStartGuidanceToHomeLocationRequest(const ::boost::shared_ptr< StartGuidanceToHomeLocationRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onStartGuidanceToHomeLocationRequest()"));
   if (_infoStore.getDestinationMemoryHomeInfo()._status)
   {
      _navMiddleware.setLocationWithDestinationMemoryId(_infoStore.getDestinationMemoryHomeInfo()._id);
      _infoStore.setGuidanceStatefromSDS(true);
      POST_MSG((COURIER_MESSAGE_NEW(StartRouteGuidanceFromSpeechReqMsg)()));
      showMapMain();
   }
   else
   {
      ETG_TRACE_USR4(("Invalid home location"));
   }
   sendStartGuidanceToHomeLocationResponse();
}


void HMIModelNavigationService::onStartGuidanceToWorkLocationRequest(const ::boost::shared_ptr< StartGuidanceToWorkLocationRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onStartGuidanceToWorkLocationRequest()"));
   if (_infoStore.getDestinationMemoryWorkInfo()._status)
   {
      _navMiddleware.setLocationWithDestinationMemoryId(_infoStore.getDestinationMemoryWorkInfo()._id);
      POST_MSG((COURIER_MESSAGE_NEW(StartRouteGuidanceFromSpeechReqMsg)()));
      showMapMain();
   }
   else
   {
      ETG_TRACE_USR4(("Invalid work location"));
   }
   sendStartGuidanceToWorkLocationResponse();
}


void HMIModelNavigationService::onShowWhereAmIScreenRequest(const ::boost::shared_ptr< ShowWhereAmIScreenRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onShowWhereAmIScreenRequest()"));
   /* @ToDo: Implementation to be added if required
   POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchOutReqMsg)(NAVIGATION_CONTEXT_WHERE_AM_I, 0, APPID_APPHMI_NAVIGATION))); */
   sendShowWhereAmIScreenResponse();
}


void HMIModelNavigationService::onShowAdjustCurrentLocationScreenRequest(const ::boost::shared_ptr< ShowAdjustCurrentLocationScreenRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onShowAdjustCurrentLocationScreenRequest()"));
   _isAdjustCurrentLocationPending = true;
   /* @ToDo: Implementation to be added if required
   POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchOutReqMsg)(NAVIGATION_CONTEXT_BY_MAP, 0, APPID_APPHMI_NAVIGATION))); */
   sendShowAdjustCurrentLocationScreenResponse();
}


void HMIModelNavigationService::onInsertWaypointRequest(const ::boost::shared_ptr< InsertWaypointRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onInsertWaypointRequest(%d)", request->getNewIndex()));
   if (_navMiddleware.isGuidanceActive())
   {
      _hmiModelNavigationServiceExtended.InsertWaypoint(request->getNewIndex());
      sendInsertWaypointResponse();
   }
   else
   {
      ETG_TRACE_ERR(("Invalid request since guidance is not active!"));
      sendInsertWaypointError(DBUS_ERROR_FAILED, "Invalid request since guidance is not active!");
   }
}


void HMIModelNavigationService::onInsertWaypointAtDefaultPositionRequest(const ::boost::shared_ptr< InsertWaypointAtDefaultPositionRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onInsertWaypointAtDefaultPositionRequest()"));
   if (_navMiddleware.isGuidanceActive())
   {
      _hmiModelNavigationServiceExtended.InsertWaypointAtDefaultPosition();
      sendInsertWaypointAtDefaultPositionResponse();
   }
   else
   {
      ETG_TRACE_ERR(("Invalid request since guidance is not active!"));
      sendInsertWaypointAtDefaultPositionError(DBUS_ERROR_FAILED, "Invalid request since guidance is not active!");
   }
}


bool HMIModelNavigationService::onPropertyUpdateRouteGuidancWaypointInfoChanged()
{
   const navmiddleware::WaypointInfos& wayPointInfos = _navMiddleware.getWaypointInfos();
   const std::vector< navmiddleware::WaypointInfos::WaypointInfo >& waypointInfoList = wayPointInfos.getWaypointInfos();
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateRouteGuidancWaypointInfoChanged(%d)", waypointInfoList.size()));
   if ((_navMiddleware.isGuidanceActive()) && (waypointInfoList.size() != 0))
   {
      ::std::vector< org::bosch::cm::navigation::NavigationService::WaypointListElement >& WaypointInfoListElements = getWaypointListMutable();
      WaypointInfoListElements.resize(waypointInfoList.size());

      for (unsigned int i = waypointInfoList.size(); i-- > 0;)
      {
         PositionWGS84 positionWGS84 = WaypointInfoListElements[i].getLocationMutable();
         positionWGS84.setLatitude(waypointInfoList[i].m_coordinate.getLatitude());
         positionWGS84.setLongitude(waypointInfoList[i].m_coordinate.getLongitude());
         WaypointInfoListElements[i].setName(waypointInfoList[i].m_locationString);
         WaypointInfoListElements[i].setRouteCriterion(convertRouteCriterion(waypointInfoList[i].m_routeCriterion));
         ETG_TRACE_USR4(("waypointInfoList size %d, latitude %f, longitude %f, string: %s", waypointInfoList.size(), waypointInfoList[i].m_coordinate.getLatitude(), waypointInfoList[i].m_coordinate.getLongitude(), waypointInfoList[i].m_locationString.c_str()));
      }
      sendWaypointListUpdate();
   }
   else
   {
      ETG_TRACE_ERR(("Invalid request since guidance is not active or destination not set!"));
   }
   //Issue 611781
   //NOTE: this code is blocked not removed as there might be a dependency with Speech accessing this information
   //The Problem was the code was causing a hang when destinations and waypoints were continuously added.
   //based on testing from speech we can delete this code
   /*if (waypointInfoList.size() != 0)
   {
      navmiddleware::WaypointInfos::WaypointInfo editRouteListItem = waypointInfoList.at(_infoStore.getWaypointListIdx());
      navmiddleware::WaypointInfos::WaypointInfo destinationItem = waypointInfoList.at(waypointInfoList.size() - 1);
      if (destinationItem == editRouteListItem)
      {
         EXT_bIsWaypointDestinationIndex = 1;
      }
      else
      {
         EXT_bIsWaypointDestinationIndex = 0;
      }
   }*/
   if (_getDeleteWaypointAct != 0)
   {
      if (_deleteWaypointCount > 0)
      {
         ETG_TRACE_ERR(("Delete waypoint till _deleteWaypointCount becomes 0"));
         _infoStore.setWaypointListIdx(_deleteWaypointStartIndex);
         POST_MSG((COURIER_MESSAGE_NEW(WaypointDeleteReqMsg)(WAYPOINT_DELETE_POPUP)));
         _deleteWaypointCount--;
      }
      else if (_deleteWaypointCount == 0)
      {
         sendDeleteWaypointsResponse(_getDeleteWaypointAct);
         _getDeleteWaypointAct = 0;
         _deleteWaypointStartIndex = -1;
      }
   }
   return true;
}


bool HMIModelNavigationService::onPropertyUpdateGuidanceWaypointSkipped()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateGuidanceWaypointSkipped()"));

   setDestinationReachedStatus(DestinationReachedStatus__WAYPOINT_SKIPPED);
   return true;
}


bool HMIModelNavigationService::onPropertyUpdateGuidanceStartedOnPlannedRoute()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateGuidanceStartedOnPlannedRoute()"));
   const navmiddleware::PlannedRouteHistoryInfos& plannedRouteHistoryInfos = _navMiddleware.getPlannedRouteHistoryInfos();
   ::std::vector< org::bosch::cm::navigation::NavigationService::WaypointOperationStatus > waypointOperationStatusList;
   const ::std::vector< navmiddleware::PlannedRouteHistoryInfos::WaypointOperation >& waypointOperations = plannedRouteHistoryInfos.getWaypointOperations();
   waypointOperationStatusList.reserve(waypointOperations.size());

   ::std::vector< navmiddleware::PlannedRouteHistoryInfos::WaypointOperation >::const_iterator itr = waypointOperations.begin();
   for (; itr != waypointOperations.end(); ++itr)
   {
      org::bosch::cm::navigation::NavigationService::WaypointOperationStatus waypointOperationStatus;
      switch (*itr)
      {
         case navmiddleware::PlannedRouteHistoryInfos::WAYPOINT_OPERATION__INSERTED:
            waypointOperationStatus = WaypointOperationStatus__INSERTED;
            break;
         case navmiddleware::PlannedRouteHistoryInfos::WAYPOINT_OPERATION__DELETED:
            waypointOperationStatus = WaypointOperationStatus__DELETED;
            break;
         case navmiddleware::PlannedRouteHistoryInfos::WAYPOINT_OPERATION__MOVED:
            waypointOperationStatus = WaypointOperationStatus__MOVED;
            break;
         case navmiddleware::PlannedRouteHistoryInfos::WAYPOINT_OPERATION__REPLACED:
            waypointOperationStatus = WaypointOperationStatus__REPLACED;
            break;
         default:
            ETG_TRACE_USR4(("Unknown waypoint operation type received"));
            break;
      }
      waypointOperationStatusList.push_back(waypointOperationStatus);
   }
   setWaypointOperationStatusList(waypointOperationStatusList);
   return true;
}


void HMIModelNavigationService::onRequestWaypointListRequest(const ::boost::shared_ptr< RequestWaypointListRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onRequestWaypointListRequest()"));
   if (_navMiddleware.isGuidanceActive())
   {
      _navMiddleware.requestWaypointList(navmiddleware::WAYPOINT_LIST_MODE__VIEW);
      sendRequestWaypointListResponse();
   }
   else
   {
      ETG_TRACE_ERR(("Invalid request since guidance is not active!"));
      sendRequestWaypointListError(DBUS_ERROR_FAILED, "Invalid request since guidance is not active!");
   }
}


void HMIModelNavigationService::onReplaceWaypointRequest(const ::boost::shared_ptr< ReplaceWaypointRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onReplaceWaypointRequest()"));
   if (_navMiddleware.isGuidanceActive())
   {
      _infoStore.setWaypointIndex(request->getIndex());
      /* @ToDo: Implementation to be added if required
         POST_MSG((COURIER_MESSAGE_NEW(ReplaceWaypointReqMsg)())); */
      sendReplaceWaypointResponse();
   }
   else
   {
      ETG_TRACE_ERR(("Invalid request since guidance is not active!"));
      sendReplaceWaypointError(DBUS_ERROR_FAILED, "Invalid request since guidance is not active!");
   }
}


void HMIModelNavigationService::onApplyWaypointListChangeRequest(const ::boost::shared_ptr< ApplyWaypointListChangeRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onApplyWaypointListChangeRequest()"));
   _navMiddleware.applyWaypointListChange();
   _navMiddleware.startGuidance();
   _infoStore.setRouteCalculationStatus(false);
   _infoStore.setWaypointListRequestNeeded(true);
   int waypointIndex = 0;
   _infoStore.setWaypointIndex(waypointIndex);
   showMapMain();
   sendApplyWaypointListChangeResponse();
}


void HMIModelNavigationService::onDeleteWaypointsRequest(const ::boost::shared_ptr< DeleteWaypointsRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onDeleteWaypointsRequest()"));
   _getDeleteWaypointAct = request->getAct();
   _deleteWaypointStartIndex = request->getFirstIndex();
   ETG_TRACE_USR4(("HMIModelNavigationService::onDeleteWaypointsRequest(%d %d)", request->getFirstIndex(), request->getLastIndex()));

   const navmiddleware::WaypointInfos& wayPointInfos = _navMiddleware.getWaypointInfos();
   const std::vector< navmiddleware::WaypointInfos::WaypointInfo >& waypointInfoList = wayPointInfos.getWaypointInfos();

   if (waypointInfoList.size() != 0)
   {
      navmiddleware::WaypointInfos::WaypointInfo editRouteListItem = waypointInfoList.at(_infoStore.getWaypointListIdx());
      navmiddleware::WaypointInfos::WaypointInfo destinationItem = waypointInfoList.at(waypointInfoList.size() - 1);

      if (destinationItem == editRouteListItem)
      {
         EXT_bIsWaypointDestinationIndex = 1;
      }
      else
      {
         EXT_bIsWaypointDestinationIndex = 0;
      }
   }
   if (_deleteWaypointCount == 0)
   {
      _deleteWaypointCount = (request->getLastIndex() - _deleteWaypointStartIndex + 1);
      if (_deleteWaypointCount > 0)
      {
         _infoStore.setWaypointListIdx(_deleteWaypointStartIndex);
         POST_MSG((COURIER_MESSAGE_NEW(WaypointDeleteReqMsg)(WAYPOINT_DELETE_POPUP)));
         _deleteWaypointCount--;
      }
   }
}


bool HMIModelNavigationService::onPropertyUpdateSdsElementsChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onpropertyUpdateSdsElementsChanged()"));
   if (_getSdsGetRefinementListAct != 0)
   {
      const navmiddleware::RefinementWithIdInfos& refinementWithIdInfos = _navMiddleware.getSdsRefinementListInfo();
      ::std::vector< ::org::bosch::cm::navigation::NavigationService::SDSAddressElement > addressElements;

      ::std::vector< org::bosch::cm::navigation::NavigationService::RefinementListElement > refinementListElements;
      std::vector<navmiddleware::RefinementWithIdInfos::RefinementWithId>::const_iterator info = refinementWithIdInfos.getElements().begin();
      for (; (info != refinementWithIdInfos.getElements().end()); ++info)
      {
         RefinementListElement refinementListElement;
         refinementListElement.setData(info->m_data);
         refinementListElement.setId(info->m_id);
         if (info->getDirection().isValid())
         {
            refinementListElement.setDirection(convertDirectionDescription(info->getDirection().getValue()));
         }
         else
         {
            refinementListElement.setDirection(DirectionDescription__INVALID);
         }
         refinementListElement.setDistance(info->getAirDistance());
         refinementListElements.push_back(refinementListElement);
      }
      sendSdsGetRefinementListResponse(refinementListElements, _getSdsGetRefinementListAct);
      _getSdsGetRefinementListAct = 0;
   }
   return true;
}


::org::bosch::cm::navigation::NavigationService::DirectionDescription HMIModelNavigationService::convertDirectionDescription(navmiddleware::DirectionDescription direction)
{
   switch (direction)
   {
      case navmiddleware::DIRECTIONDESCRIPTION_NORTH:
         return DirectionDescription__NORTH;
      case navmiddleware::DIRECTIONDESCRIPTION_NORTH_NORTH_EAST:
         return DirectionDescription__NORTH_NORTH_EAST;
      case navmiddleware::DIRECTIONDESCRIPTION_NORTH_EAST:
         return DirectionDescription__NORTH_EAST;
      case navmiddleware::DIRECTIONDESCRIPTION_EAST_NORTH_EAST:
         return DirectionDescription__EAST_NORTH_EAST;
      case navmiddleware::DIRECTIONDESCRIPTION_EAST:
         return DirectionDescription__EAST;
      case navmiddleware::DIRECTIONDESCRIPTION_EAST_SOUTH_EAST:
         return DirectionDescription__EAST_SOUTH_EAST;
      case navmiddleware::DIRECTIONDESCRIPTION_SOUTH_EAST:
         return DirectionDescription__SOUTH_EAST;
      case navmiddleware::DIRECTIONDESCRIPTION_SOUTH_SOUTH_EAST:
         return DirectionDescription__SOUTH_SOUTH_EAST;
      case navmiddleware::DIRECTIONDESCRIPTION_SOUTH:
         return DirectionDescription__SOUTH;
      case navmiddleware::DIRECTIONDESCRIPTION_SOUTH_SOUTH_WEST:
         return DirectionDescription__SOUTH_SOUTH_WEST;
      case navmiddleware::DIRECTIONDESCRIPTION_SOUTH_WEST:
         return DirectionDescription__SOUTH_WEST;
      case navmiddleware::DIRECTIONDESCRIPTION_WEST_SOUTH_WEST:
         return DirectionDescription__WEST_SOUTH_WEST;
      case navmiddleware::DIRECTIONDESCRIPTION_WEST:
         return DirectionDescription__WEST;
      case navmiddleware::DIRECTIONDESCRIPTION_WEST_NORTH_WEST:
         return DirectionDescription__WEST_NORTH_WEST;
      case navmiddleware::DIRECTIONDESCRIPTION_NORTH_WEST:
         return DirectionDescription__NORTH_WEST;
      case navmiddleware::DIRECTIONDESCRIPTION_NORTH_NORTH_WEST:
         return DirectionDescription__NORTH_NORTH_WEST;
      case navmiddleware::DIRECTIONDESCRIPTION_SIZE:
         return DirectionDescription__INVALID;
      default:
         ETG_TRACE_USR4(("Default case returning DirectionDescription__INVALID"));
         return DirectionDescription__INVALID;
   }
}


void HMIModelNavigationService::onSdsGetRefinementListRequest(const ::boost::shared_ptr< SdsGetRefinementListRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSdsGetRefinementListRequest(), Address element type : %d", request->getSdsAddressElementType()));

   _getSdsGetRefinementListAct = request->getAct();
   _navMiddleware.requestSdsRefinementList(convertSDSAddressElementType(request->getSdsAddressElementType()));
}


void HMIModelNavigationService::onSelectSdsRefinementRequest(const ::boost::shared_ptr< SelectSdsRefinementRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSelectSdsRefinementRequest()"));
   _selectSdsRefinementAct = request->getAct();
   _navMiddleware.selectSdsRefinement(request->getId());
}


void HMIModelNavigationService::onSetRouteCriterionRequest(const ::boost::shared_ptr< SetRouteCriterionRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSetRouteCriterionRequest(%d)", request->getRouteCriterion()));
   navmiddleware::settings::RouteCriteriaSettings routeCriteriaSettings;
   routeCriteriaSettings.setRouteCriterion(convertRouteCriterion(request->getRouteCriterion()));
   _navMiddleware.updateRouteCriteriaSettings(routeCriteriaSettings);
   _navMiddleware.applySettings();
   showMapMain();
   sendSetRouteCriterionResponse();
}


void HMIModelNavigationService::onSetLocationWithDestinationMemoryEntryRequest(const ::boost::shared_ptr< SetLocationWithDestinationMemoryEntryRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSetLocationWithDestinationMemoryEntryRequest(%d)", request->getId()));
   _navMiddleware.setLocationWithDestinationMemoryId(static_cast<uint64>(request->getId()));
   sendSetLocationWithDestinationMemoryEntryResponse();
}


void HMIModelNavigationService::updateToDestinationMemoryItemList(const navmiddleware::LocationAttributeInfos::AddressDetailInfo* addressDetailInfo,
      const navmiddleware::LocationAttributeInfos& locationAttributes,
      ::std::vector< DestinationMemoryEntryItem >& destinationMemoryEntryItemList)
{
   if (addressDetailInfo != NULL)
   {
      ETG_TRACE_USR4(("HMIModelNavigationService::updateDestinationMemoryItemList()"));
      navmiddleware::LocationAttributeInfos::AddressDetailInfo::T_AddressInfoMap::const_iterator itr = addressDetailInfo->getAddressInfoMap().begin();
      std::stringstream stream;
      for (itr; itr != addressDetailInfo->getAddressInfoMap().end(); itr++)
      {
         ETG_TRACE_USR4(("HMIModelNavigationService::updateDestinationMemoryItemList() - Updating Address info"));
         DestinationMemoryEntryItem item;
         switch (itr->first)
         {
            case navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__COUNTRY_NAME:
            case navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__COUNTRY_CODE:
            case navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__STATE_NAME:
            case navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__CITY_NAME:
            case navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__STREET_NAME:
            case navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__CROSS_STREET_NAME:
            case navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__FULL_ADDRESS:
               item.setAddressType(getDestinationMemoryAddressType(itr->first));
               item.setItemString((itr->second).value);
               item.setItemPhoneme((itr->second).valuePhoneme);
               destinationMemoryEntryItemList.push_back(item);
               stream << "[VALUE]: " << (itr->second).value << std::endl;
               break;
            default:
               ETG_TRACE_USR4(("HMIModelNavigationService::updateDestinationMemoryItemList() - address type not supported for SDS"));
               break;
         }
      }
      ETG_TRACE_USR4(("%s", stream.str()));
      stream.clear();
      if (locationAttributes.getLocationType() == navmiddleware::LocationAttributeInfos::LOCATIONTYPE_POI)
      {
         ETG_TRACE_USR4(("HMIModelNavigationService::updateDestinationMemoryItemList() - Updating POI info"));
         if (locationAttributes.getLocationNameInfo() != NULL)
         {
            DestinationMemoryEntryItem item;
            item.setAddressType(DestinationMemoryAddressType__DESTINATION_MEMORY_POI);
            item.setItemString(locationAttributes.getLocationNameInfo()->m_content);
            if (locationAttributes.getLocationNamePhonemeInfo() != NULL)
            {
               item.setItemPhoneme(locationAttributes.getLocationNamePhonemeInfo()->m_content);
            }
            stream << "[POI NAME]: " << locationAttributes.getLocationNameInfo()->m_content << std::endl;
            destinationMemoryEntryItemList.push_back(item);
         }
         if (locationAttributes.getPoiCategoryInfo() != NULL)
         {
            DestinationMemoryEntryItem item;
            item.setAddressType(DestinationMemoryAddressType__DESTINATION_MEMORY_POI_CATEGORY);
            item.setItemString(locationAttributes.getPoiCategoryInfo()->m_categoryName);
            if (locationAttributes.getLocationNamePhonemeInfo() != NULL)
            {
               item.setItemPhoneme(locationAttributes.getPoiCategoryInfo()->m_categoryNamePhoneme);
            }
            stream << "[POI CATEGORY]: " << locationAttributes.getPoiCategoryInfo()->m_categoryName << std::endl;
            destinationMemoryEntryItemList.push_back(item);
         }
         if (locationAttributes.getBrandNameInfo() != NULL)
         {
            DestinationMemoryEntryItem item;
            item.setAddressType(DestinationMemoryAddressType__DESTINATION_MEMORY_POI_BRAND);
            item.setItemString(locationAttributes.getBrandNameInfo()->m_content);
            item.setItemPhoneme("");
            stream << "[POI BRAND NAME]: " << locationAttributes.getBrandNameInfo()->m_content << std::endl;
            destinationMemoryEntryItemList.push_back(item);
         }
      }
      ETG_TRACE_USR4(("%s", stream.str()));
   }
}


bool HMIModelNavigationService::onPropertyUpdateDestinationMemoryLastDestListChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateDestinationMemoryLastDestListChanged()"));
   if (_getLastDestinationAct != 0)
   {
      const navmiddleware::DestinationMemoryItemList& destinationMemoryItemList = _navMiddleware.getLastDestinationList();
      const ::std::vector<navmiddleware::DestinationMemoryItem>& infos = destinationMemoryItemList.getItemList();
      ::std::vector<navmiddleware::DestinationMemoryItem>::const_iterator info = infos.begin();
      ::std::vector<DestinationMemoryDetailListElement> destinationMemoryDetailList;
      ETG_TRACE_USR4(("HMIModelNavigationService::updateDestinationMemoryEntries(%d)", infos.size()));
      for (; (info != infos.end()); ++info)
      {
         DestinationMemoryDetailListElement destinationMemoryDetailListElement;
         ::org::bosch::cm::navigation::NavigationService::DestinationMemoryEntry& destinationMemoryEntry = destinationMemoryDetailListElement.getEntryMutable();
         const navmiddleware::LocationAttributeInfos& locationAttributes = info->getEntry().getDestination().getAttributes();
         const navmiddleware::LocationAttributeInfos::AddressDetailInfo* addressDetailInfo = locationAttributes.getAddressDetailInfos();
         std::stringstream stream;
         destinationMemoryEntry.setEntryID(info->getEntry().getId());
         destinationMemoryEntry.setName(info->getEntry().getDestination().getName());
         if (info->getEntry().getAttributes().getValue().getVoiceTag().isValid())
         {
            destinationMemoryEntry.setVoiceTagId(static_cast<uint32>(info->getEntry().getAttributes().getValue().getVoiceTag().getValue()));
            ETG_TRACE_USR4(("voiceTagId %d", info->getEntry().getAttributes().getValue().getVoiceTag().getValue()));
         }
         else
         {
            destinationMemoryEntry.setVoiceTagId(0);
            ETG_TRACE_USR4(("Invalid voiceTagId "));
         }
         destinationMemoryDetailListElement.setEntry(destinationMemoryEntry);
         ETG_TRACE_USR4(("id %d", info->getEntry().getId()));
         ETG_TRACE_USR4(("Name %s", info->getEntry().getDestination().getName().c_str()));

         ::org::bosch::cm::navigation::NavigationService::DestinationMemoryEntryDetails& destinationMemoryEntryDetails = destinationMemoryDetailListElement.getDetailsMutable();
         updateToDestinationMemoryItemList(addressDetailInfo, locationAttributes, destinationMemoryEntryDetails.getSdsDestinationMemoryEntriesMutable());
         if (addressDetailInfo != NULL)
         {
            ::std::map < navmiddleware::LocationAttributeInfos::AddressDetailInfo::AddressInfoType, navmiddleware::LocationAttributeInfos::AddressDetailInfo::AddressInfoContent >::const_iterator itr;
            itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__COUNTRY_NAME);
            if (itr != addressDetailInfo->getAddressInfoMap().end())
            {
               stream << "Country=" << (itr->second).value << " ";
               destinationMemoryEntryDetails.setCountryName((itr->second).value);
            }
            itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__COUNTRY_CODE);
            if (itr != addressDetailInfo->getAddressInfoMap().end())
            {
               stream << "CountryCode=" << (itr->second).value << " ";
               destinationMemoryEntryDetails.setCountryCode((itr->second).value);
            }
            itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__STATE_NAME);
            if (itr != addressDetailInfo->getAddressInfoMap().end())
            {
               stream << "State= " << (itr->second).value << " ";
               destinationMemoryEntryDetails.setStateName((itr->second).value);
            }
            itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__CITY_NAME);
            if (itr != addressDetailInfo->getAddressInfoMap().end())
            {
               stream << "City= " << (itr->second).value << " ";
               destinationMemoryEntryDetails.setCityName((itr->second).value);
            }
            itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__STREET_NAME);
            if (itr != addressDetailInfo->getAddressInfoMap().end())
            {
               stream << "Street= " << (itr->second).value << " ";
               destinationMemoryEntryDetails.setStreetName((itr->second).value);
            }
            itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__HOUSE_NUMBER);
            if (itr != addressDetailInfo->getAddressInfoMap().end())
            {
               stream << "HouseNumber= " << (itr->second).value << " ";
               destinationMemoryEntryDetails.setHouseno((itr->second).value);
            }
            itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__CROSS_STREET_NAME);
            if (itr != addressDetailInfo->getAddressInfoMap().end())
            {
               stream << "CrossStreetName= " << (itr->second).value << std::endl;
               destinationMemoryEntryDetails.setJunctionName((itr->second).value);
            }
            itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__FULL_ADDRESS);
            if (itr != addressDetailInfo->getAddressInfoMap().end())
            {
               stream << "AddressDetail= " << (itr->second).value << std::endl;
               destinationMemoryEntryDetails.setAddressDetail((itr->second).value.c_str());
            }
         }
         if (locationAttributes.getLocationType() == navmiddleware::LocationAttributeInfos::LOCATIONTYPE_POI)
         {
            if (locationAttributes.getLocationNameInfo() != NULL)
            {
               destinationMemoryEntryDetails.setPoiName(locationAttributes.getLocationNameInfo()->m_content);
               stream << "[POI NAME]" << locationAttributes.getLocationNameInfo()->m_content << std::endl;
            }
            if (locationAttributes.getPoiCategoryInfo() != NULL)
            {
               destinationMemoryEntryDetails.setPoiCategoryName(locationAttributes.getPoiCategoryInfo()->m_categoryName);
               stream << "[POI CATEGORY]" << locationAttributes.getPoiCategoryInfo()->m_categoryName << std::endl;
            }
            if (locationAttributes.getBrandNameInfo() != NULL)
            {
               destinationMemoryEntryDetails.setPoiBrandName(locationAttributes.getBrandNameInfo()->m_content);
               stream << "[POI BRAND NAME]" << locationAttributes.getBrandNameInfo()->m_content << std::endl;
            }
         }

         destinationMemoryDetailListElement.setDetails(destinationMemoryEntryDetails);
         ::org::bosch::cm::navigation::NavigationService::PositionWGS84& coordinates = destinationMemoryDetailListElement.getCoordinatesMutable();
         coordinates.setLongitude(info->getEntry().getDestination().getLongitude());
         coordinates.setLatitude(info->getEntry().getDestination().getLatitude());
         destinationMemoryDetailList.push_back(destinationMemoryDetailListElement);
         ETG_TRACE_USR4(("%s", stream.str()));
      }
      sendRequestDestinationMemoryEntriesResponse(destinationMemoryDetailList, _getLastDestinationAct);
      setIsLastDestinationListChanged(false);
      _getLastDestinationAct = 0;
   }
   return true;
}


bool HMIModelNavigationService::onPropertyUpdateDestinationMemoryLastDestStatusChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateDestinationMemoryLastDestStatusChanged()"));
   setIsLastDestinationListChanged(true);
   return true;
}


bool HMIModelNavigationService::onPropertyDestinationMemoryAddBookStatusChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyDestinationMemoryAddBookStatusChanged()"));
   setIsAddressBookChanged(true);

   //property update for isMaxAllowedAddressBookEntriesStored.
   const navmiddleware::DestinationMemoryStatus& destinationMemoryStatus = _navMiddleware.getDestinationMemoryStatus();
   setIsMaxAllowedAddressBookEntriesStored(destinationMemoryStatus.getNumberOfAddressBookEntries() >= MAX_ALLOWED_ADDRESS_BOOK_ENTRIES_COUNT);
   return true;
}


bool HMIModelNavigationService::onPropertyUpdateDestinationMemoryAddBookListChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateDestinationMemoryAddBookListChanged()"));

   if (_getAdressbookDestinationAct != 0)
   {
      ::std::vector<DestinationMemoryDetailListElement> destinationMemoryDetailListElements;
      constructDestinationMemoryEntries(destinationMemoryDetailListElements);
      sendRequestDestinationMemoryEntriesResponse(destinationMemoryDetailListElements, _getAdressbookDestinationAct);
      setIsAddressBookChanged(false);
      _getAdressbookDestinationAct = 0;
   }
   return true;
}


bool HMIModelNavigationService::onPropertyUpdateDestinationMemoryPersonalEPOIListChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateDestinationMemoryPersonalEPOIListChanged()"));
   if (_getPersonalEPOIAct != 0)
   {
      ::std::vector<DestinationMemoryDetailListElement> destinationMemoryDetailListElements;
      constructDestinationMemoryEntries(destinationMemoryDetailListElements);
      sendRequestDestinationMemoryEntriesResponse(destinationMemoryDetailListElements, _getPersonalEPOIAct);
      _getPersonalEPOIAct = 0;
   }
   return true;
}


void HMIModelNavigationService::constructDestinationMemoryEntries(::std::vector<DestinationMemoryDetailListElement>& destinationMemoryDetailListElements)
{
   const navmiddleware::DestinationMemoryEntryList& destinationMemoryEntryList = _navMiddleware.getEntryList();
   const ::std::vector<navmiddleware::DestinationMemoryEntry>& infos = destinationMemoryEntryList.getEntryList();
   std::stringstream stream;
   ::std::vector<navmiddleware::DestinationMemoryEntry>::const_iterator info = infos.begin();
   destinationMemoryDetailListElements.clear();
   destinationMemoryDetailListElements.reserve(infos.size());
   ETG_TRACE_USR4(("HMIModelNavigationService::updateDestinationMemoryEntries(%d)", infos.size()));
   for (; (info != infos.end()); ++info)
   {
      DestinationMemoryDetailListElement destinationMemoryDetailListElement;
      ::org::bosch::cm::navigation::NavigationService::DestinationMemoryEntry& destinationMemoryEntry = destinationMemoryDetailListElement.getEntryMutable();
      const navmiddleware::LocationAttributeInfos& locationAttributes = info->getDestination().getAttributes();
      const navmiddleware::LocationAttributeInfos::AddressDetailInfo* addressDetailInfo = locationAttributes.getAddressDetailInfos();
      destinationMemoryEntry.setEntryID(info->getId());
      if (info->getAttributes().isValid())
      {
         destinationMemoryEntry.setName(info->getAttributes().getValue().getName());
         if (info->getAttributes().getValue().getVoiceTag().isValid())
         {
            destinationMemoryEntry.setVoiceTagId(static_cast<uint32>(info->getAttributes().getValue().getVoiceTag().getValue()));
            ETG_TRACE_USR4(("voiceTagId %d", info->getAttributes().getValue().getVoiceTag().getValue()));
         }
         else
         {
            destinationMemoryEntry.clearVoiceTagId();
            ETG_TRACE_USR4(("Invalid voiceTagId "));
         }
      }
      else
      {
         ETG_TRACE_USR4(("Invalid Atributes "));
         destinationMemoryEntry.clearName();
         destinationMemoryEntry.clearVoiceTagId();
      }

      ETG_TRACE_USR4(("id %d", info->getId()));
      ETG_TRACE_USR4(("Name %s", info->getAttributes().getValue().getName().c_str()));

      ::org::bosch::cm::navigation::NavigationService::DestinationMemoryEntryDetails& destinationMemoryEntryDetails = destinationMemoryDetailListElement.getDetailsMutable();
      std::stringstream stream;
      updateToDestinationMemoryItemList(addressDetailInfo, locationAttributes, destinationMemoryEntryDetails.getSdsDestinationMemoryEntriesMutable());
      if (addressDetailInfo != NULL)
      {
         ::std::map < navmiddleware::LocationAttributeInfos::AddressDetailInfo::AddressInfoType, navmiddleware::LocationAttributeInfos::AddressDetailInfo::AddressInfoContent >::const_iterator itr;
         itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__COUNTRY_NAME);
         if (itr != addressDetailInfo->getAddressInfoMap().end())
         {
            stream << "Country=" << (itr->second).value << " ";
            destinationMemoryEntryDetails.setCountryName((itr->second).value);
         }
         itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__COUNTRY_CODE);
         if (itr != addressDetailInfo->getAddressInfoMap().end())
         {
            stream << "CountryCode=" << (itr->second).value << " ";
            destinationMemoryEntryDetails.setCountryCode((itr->second).value);
         }
         itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__STATE_NAME);
         if (itr != addressDetailInfo->getAddressInfoMap().end())
         {
            stream << "State= " << (itr->second).value << " ";
            destinationMemoryEntryDetails.setStateName((itr->second).value);
         }
         itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__CITY_NAME);
         if (itr != addressDetailInfo->getAddressInfoMap().end())
         {
            stream << "City= " << (itr->second).value << " ";
            destinationMemoryEntryDetails.setCityName((itr->second).value);
         }
         itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__STREET_NAME);
         if (itr != addressDetailInfo->getAddressInfoMap().end())
         {
            stream << "Street= " << (itr->second).value << " ";
            destinationMemoryEntryDetails.setStreetName((itr->second).value);
         }
         itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__HOUSE_NUMBER);
         if (itr != addressDetailInfo->getAddressInfoMap().end())
         {
            stream << "HouseNumber= " << (itr->second).value << " ";
            destinationMemoryEntryDetails.setHouseno((itr->second).value);
         }
         itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__CROSS_STREET_NAME);
         if (itr != addressDetailInfo->getAddressInfoMap().end())
         {
            stream << "CrossStreetName= " << (itr->second).value << std::endl;
            destinationMemoryEntryDetails.setJunctionName((itr->second).value);
         }
         itr = addressDetailInfo->getAddressInfoMap().find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__FULL_ADDRESS);
         if (itr != addressDetailInfo->getAddressInfoMap().end())
         {
            stream << "AddressDetail= " << (itr->second).value << std::endl;
            destinationMemoryEntryDetails.setAddressDetail((itr->second).value.c_str());
         }
      }
      if (locationAttributes.getLocationType() == navmiddleware::LocationAttributeInfos::LOCATIONTYPE_POI)
      {
         if (locationAttributes.getLocationNameInfo() != NULL)
         {
            destinationMemoryEntryDetails.setPoiName(locationAttributes.getLocationNameInfo()->m_content);
            stream << "[POI NAME]" << locationAttributes.getLocationNameInfo()->m_content << std::endl;
         }
         if (locationAttributes.getPoiCategoryInfo() != NULL)
         {
            destinationMemoryEntryDetails.setPoiCategoryName(locationAttributes.getPoiCategoryInfo()->m_categoryName);
            stream << "[POI CATEGORY]" << locationAttributes.getPoiCategoryInfo()->m_categoryName << std::endl;
         }
         if (locationAttributes.getBrandNameInfo() != NULL)
         {
            destinationMemoryEntryDetails.setPoiBrandName(locationAttributes.getBrandNameInfo()->m_content);
            stream << "[POI BRAND NAME]" << locationAttributes.getBrandNameInfo()->m_content << std::endl;
         }
      }
      ::org::bosch::cm::navigation::NavigationService::PositionWGS84& coordinates = destinationMemoryDetailListElement.getCoordinatesMutable();
      coordinates.setLongitude(info->getDestination().getLongitude());
      coordinates.setLatitude(info->getDestination().getLatitude());

      //ETG_TRACE_USR4(("CountryName %s", info->getDestination().getDestinationMemoryDestinationDetails().getCountry().c_str()));
      //ETG_TRACE_USR4(("CityName %s", info->getDestination().getDestinationMemoryDestinationDetails().getCity().c_str()));
      //ETG_TRACE_USR4(("PoiName %s", info->getDestination().getDestinationMemoryDestinationDetails().getPoiName().c_str()));
      destinationMemoryDetailListElements.push_back(destinationMemoryDetailListElement);
   }
}


bool HMIModelNavigationService::onCourierMessage(const VoiceTagOptionsMsg& oMsg)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::VoiceTagOptionsMsg(%d)", oMsg.GetVoiceTagOptionsValue()));
   SDSVoiceTagId& SDSVoiceTagId = getSdsVoiceTagWithOptionsMutable();
   SDSVoiceTagId.setDestinationMemoryEntryId(_infoStore.getDestinationMemoryStatus()._id);
   SDSVoiceTagId.setDestinationMemoryEntryName(_infoStore.getDestinationAttributesInfo().getName());
   SDSVoiceTagId.setOptions(static_cast<SDSVoiceTagOptions>(oMsg.GetVoiceTagOptionsValue()));
   if ((oMsg.GetVoiceTagOptionsValue() == PLAY_VOICETAG) || (oMsg.GetVoiceTagOptionsValue() == DELETE_VOICETAG))
   {
      SDSVoiceTagId.setSdsVoiceTagId((int32)(_infoStore.getDestinationAttributesInfo().getVoiceTag().getValue()));
   }
   else if (oMsg.GetVoiceTagOptionsValue() == DELETE_ALL_VOICETAGS)
   {
      SDSVoiceTagId.setDestinationMemoryEntryId((long long)(0));
      SDSVoiceTagId.setDestinationMemoryEntryName("");
      SDSVoiceTagId.setSdsVoiceTagId(0);
   }
   else
   {
      SDSVoiceTagId.setSdsVoiceTagId(0);
   }
   sendSdsVoiceTagWithOptionsUpdate();
   return true;
}


void HMIModelNavigationService::onRequestDestinationMemoryEntriesRequest(const ::boost::shared_ptr< RequestDestinationMemoryEntriesRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onRequestDestinationMemoryEntriesRequest(%d, %d)", request->getCategoryType(), getNavStatus()));
   if (isInitialized())
   {
      if (DestinationMemoryCategoryType__ADDRESS_BOOK == request->getCategoryType())
      {
         _getAdressbookDestinationAct = request->getAct();
         navmiddleware::ValidValue<navmiddleware::DestinationMemoryAttributesGroup> groupFilter; // value still invalid means, there is no special filter group selected
         _navMiddleware.requestEntryList(navmiddleware::DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK, groupFilter, 0, 20,
                                         navmiddleware::DestinationMemoryEntry::SORTINGALGORITHM_ALPHABETICAL_BY_NAME_ASC);//@To Do: will be modified based on the MW generalized API.
      }
      else if (DestinationMemoryCategoryType__LAST_DESTINATIONS == request->getCategoryType())
      {
         _getLastDestinationAct = request->getAct();
         _navMiddleware.requestLastDestinationList(0, 20, navmiddleware::DestinationMemoryEntry::SORTINGALGORITHM_BY_START_GUIDANCE_TIME_NEWEST_FIRST,
               navmiddleware::DestinationMemoryItemList::FILTER_ENTRY_AND_ROUTE_WAYPOINT);//@To Do: will be modified based on the MW generalized API.
      }
      else
      {
         ETG_TRACE_USR4(("Invalid destination memory category"));
         sendRequestDestinationMemoryEntriesError(DBUS_ERROR_NO_REPLY, "Invalid destination memory category", request->getAct());
      }
   }
   else
   {
      //sendRequestDestinationMemoryEntriesResponse
      sendRequestDestinationMemoryEntriesError(DBUS_ERROR_SERVICE_UNKNOWN, "NavigationService not available", request->getAct());
      ETG_TRACE_USR4(("NavigationService not available"));
   }
}


void HMIModelNavigationService::onSdsGetHouseNumberPatternRequest(const ::boost::shared_ptr< SdsGetHouseNumberPatternRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSdsGetHouseNumberPatternRequest()"));
   _getHouseNumberPatternAct = request->getAct();
   _navMiddleware.requestSdsHouseNumberPatterns();
}


bool HMIModelNavigationService::onPropertyUpdateSdsHouseNumberPatternsChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateSdsHouseNumberPatternsChanged()"));
   if (_getHouseNumberPatternAct != 0)
   {
      const SdsHouseNumberPatternsInfos& sdsHouseNumberPatternsInfos = _navMiddleware.getSdsHouseNumberPatternsInfo();
      ::std::vector< ::std::string> houseNumberPatterns;
      for (::std::vector< ::std::string>::const_iterator iter = sdsHouseNumberPatternsInfos.getPatterns().begin(); iter != sdsHouseNumberPatternsInfos.getPatterns().end(); ++iter)
      {
         houseNumberPatterns.push_back(*iter);
      }
      sendSdsGetHouseNumberPatternResponse(sdsHouseNumberPatternsInfos.getMinimum(), sdsHouseNumberPatternsInfos.getMaximum(), houseNumberPatterns, _getHouseNumberPatternAct);
      _getHouseNumberPatternAct = 0;
   }
   return true;
}


void HMIModelNavigationService::onSdsUpdateVoiceTagRequest(const ::boost::shared_ptr< SdsUpdateVoiceTagRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSdsUpdateVoiceTagRequest()"));
   if (request->getVoiceTagID().getOptions() == SDSVoiceTagOptions__NEW_VOICETAG_CREATED)
   {
      _infoStore.setDestinationAttributesInfoVoiceTagId(static_cast<uint64_t>(request->getVoiceTagID().getSdsVoiceTagId())); //@todo: Adapt datatype in NavigationService.fidl
      _navMiddleware.modifyEntryAttributes(request->getVoiceTagID().getDestinationMemoryEntryId(), _infoStore.getDestinationAttributesInfo());
   }
   else if (request->getVoiceTagID().getOptions() == SDSVoiceTagOptions__VOICETAG_DELETED)
   {
      _infoStore.setDestinationAttributesInfoVoiceTagId(static_cast<uint64_t>(request->getVoiceTagID().getSdsVoiceTagId())); //@todo: Adapt datatype in NavigationService.fidl
      _navMiddleware.modifyEntryAttributes(request->getVoiceTagID().getDestinationMemoryEntryId(), _infoStore.getDestinationAttributesInfo());
   }
   else
   {
      ETG_TRACE_USR4(("Voice tag creation failed"));
   }
   sendSdsUpdateVoiceTagResponse();
}


void HMIModelNavigationService::onStartTripReplayRequest(const boost::shared_ptr<org::bosch::cm::navigation::NavigationService::StartTripReplayRequest>& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onStartTripReplayRequest()"));
   _navMiddleware.startTripReplay(request->getTripFileName());
   sendStartTripReplayResponse();
}


void HMIModelNavigationService::onStopTripReplayRequest(const boost::shared_ptr<org::bosch::cm::navigation::NavigationService::StopTripReplayRequest>& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onStopTripReplayRequest()"));
   _navMiddleware.stopTripReplay();
   sendStopTripReplayResponse();
}


void HMIModelNavigationService::onSeekTripReplayRequest(const boost::shared_ptr<org::bosch::cm::navigation::NavigationService::SeekTripReplayRequest>& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSeekTripReplayRequest()"));
   _navMiddleware.seekTripReplay(request->getSeekValue());
   sendSeekTripReplayResponse();
}


void HMIModelNavigationService::onSetTripReplayModeRequest(const boost::shared_ptr<org::bosch::cm::navigation::NavigationService::SetTripReplayModeRequest>& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSetTripReplayModeRequest()"));
   _navMiddleware.setTripReplayMode(request->getLoopModeActive(), request->getSpeed());
   sendSetTripReplayModeResponse();
}


bool HMIModelNavigationService::onPropertyUpdateTripReplayStatusChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateTripReplayStatusChanged()"));
   const navmiddleware::TripReplayStatusInfo& tripReplayStatusInfo = _navMiddleware.getTripReplayStatusInfo();
   org::bosch::cm::navigation::NavigationService::TripReplayStatusInfo& statusInfo = getTripReplayStatusInfoMutable();
   switch (tripReplayStatusInfo.getTripReplayStatus())
   {
      case navmiddleware::TripReplayStatusInfo::TRIP_REPLAY_STATUS__UNKNOWN:
         statusInfo.setStatus(TripReplayStatus__UNKNOWN);
         break;
      case navmiddleware::TripReplayStatusInfo::TRIP_REPLAY_STATUS__RESERVED:
         statusInfo.setStatus(TripReplayStatus__RESERVED);
         break;
      case navmiddleware::TripReplayStatusInfo::TRIP_REPLAY_STATUS__NOT_ACTIVE:
         statusInfo.setStatus(TripReplayStatus__NOT_ACTIVE);
         break;
      case navmiddleware::TripReplayStatusInfo::TRIP_REPLAY_STATUS__ACTIVE:
         statusInfo.setStatus(TripReplayStatus__ACTIVE);
         break;
      case navmiddleware::TripReplayStatusInfo::TRIP_REPLAY_STATUS__FILE_NOT_FOUND_ERROR:
         statusInfo.setStatus(TripReplayStatus__FILE_NOT_FOUND_ERROR);
         break;
      case navmiddleware::TripReplayStatusInfo::TRIP_REPLAY_STATUS__INTERNAL_ERROR:
         statusInfo.setStatus(TripReplayStatus__INTERNAL_ERROR);
         break;
      default:
         break;
   }
   statusInfo.setLoopModeActive(tripReplayStatusInfo.getLoopModeActive());
   statusInfo.setTripFileName(tripReplayStatusInfo.getTripFileName());
   statusInfo.setProgress(tripReplayStatusInfo.getProgress());
   statusInfo.setSyncSegment(tripReplayStatusInfo.getSyncSegment());
   statusInfo.setCountSyncSegments(tripReplayStatusInfo.getCountSyncSegments());
   statusInfo.setSpeed(tripReplayStatusInfo.getSpeed());
   sendTripReplayStatusInfoUpdate();

   return true;
}


void HMIModelNavigationService::onPropertyUpdateEtcInterruptOccured()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateEtcInterruptOccured()"));
   updateEtcStatusFromInterrupt();
}


void HMIModelNavigationService::onPropertyUpdateEtcUsageDataHistoryListStatusChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateEtcUsageDataHistoryListStatusChanged()"));
}


void HMIModelNavigationService::onPropertyUpdateEtcSettingsChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateEtcSettingsChanged()"));
}


void HMIModelNavigationService::updateEtcStatusFromInterrupt(bool forceInitialValue)
{
   /*to nothing*/
}


bool HMIModelNavigationService::updateLocationDetailsFromMap(const navmiddleware::LocationAttributeInfos::AddressDetailInfo* addressDetailInfos, ::org::bosch::cm::navigation::NavigationService::LocationDetails& locationDetail)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::updateLocationDetailsFromMap()"));
   if (!addressDetailInfos)
   {
      return false;
   }

   //TODO: addressInfoMap should be made const once naviMiddleware makes getAddressInfoMap() function const.
   ::navmiddleware::LocationAttributeInfos::AddressDetailInfo::T_AddressInfoMap addressInfoMap = const_cast<navmiddleware::LocationAttributeInfos::AddressDetailInfo*>(addressDetailInfos)->getAddressInfoMap();

   ::std::map < navmiddleware::LocationAttributeInfos::AddressDetailInfo::AddressInfoType, navmiddleware::LocationAttributeInfos::AddressDetailInfo::AddressInfoContent >::const_iterator mapItr;

   mapItr = addressInfoMap.find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__CITY_NAME);
   ::std::string cityName = (mapItr != addressInfoMap.end() ? (mapItr->second).value : "");
   locationDetail.setCityName(cityName);

   mapItr = addressInfoMap.find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__COUNTRY_NAME);
   ::std::string countryName = (mapItr != addressInfoMap.end() ? (mapItr->second).value : "");
   locationDetail.setCountryName(countryName);

   mapItr = addressInfoMap.find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__COUNTRY_CODE);
   ::std::string countryCode = (mapItr != addressInfoMap.end() ? (mapItr->second).value : "");
   locationDetail.setCountryCode(countryCode);

   mapItr = addressInfoMap.find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__STATE_NAME);
   ::std::string stateName = (mapItr != addressInfoMap.end() ? (mapItr->second).value : "");
   locationDetail.setStateName(stateName);

   mapItr = addressInfoMap.find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__STATE_CODE);
   ::std::string stateCode = (mapItr != addressInfoMap.end() ? (mapItr->second).value : "");
   locationDetail.setStateCode(stateCode);

   mapItr = addressInfoMap.find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__STREET_NAME);
   ::std::string streetName = (mapItr != addressInfoMap.end() ? (mapItr->second).value : "");
   locationDetail.setStreetName(streetName);

   mapItr = addressInfoMap.find(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__ZIP_CODE);
   ::std::string zipCode = (mapItr != addressInfoMap.end() ? (mapItr->second).value : "");
   locationDetail.setZipCode(zipCode);

   return true;
}


// @brief Set EtcStatus only when it has changed from the cached value and displaying the ETC icon is enabled in settings
void HMIModelNavigationService::setEtcStatusWithCheck(org::bosch::cm::navigation::NavigationService::ETCStatus status, bool isEtcIconDisplayEnabled)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::setEtcStatusWithCheck(%d)", status));

   if (_etcStatusCache != status)
   {
      _etcStatusCache = status;

      // Set status only if it is allowed and DSRC is connected
      if (isEtcIconDisplayEnabled && _isVicsSourceDsrcConnected)
      {
         setEtcStatus(_etcStatusCache);
      }
   }
}


void HMIModelNavigationService::onSetMapRepresentationRequest(const ::boost::shared_ptr< SetMapRepresentationRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSetMapRepresentationRequest(%d)", request->getRepresentation()));
   _navMiddleware.setMapRepresentation(MAP_VIEW_ID__PRIMARY, convertMapRepresentation(request->getRepresentation()));
   showMapMain();
   sendSetMapRepresentationResponse();
}


navmiddleware::MapRepresentation HMIModelNavigationService::convertMapRepresentation(::org::bosch::cm::navigation::NavigationService::MapRepresentation mapRepresentation)
{
   navmiddleware::MapRepresentation mapRepresentationConverted;

   switch (mapRepresentation)
   {
      case MapRepresentation__MAP_REPRESENTATION_2D_NORTH_UP:
      {
         // 2D_NORTH_UP
         mapRepresentationConverted.m_mapPitch = navmiddleware::MAP_PITCH_2D;
         mapRepresentationConverted.m_mapOrientation = navmiddleware::settings::MAP_ORIENTATION_NORTH_UP;
      }
      break;
      case MapRepresentation__MAP_REPRESENTATION_3D_CAR_HEADING:
      {
         // 3D_CAR_HEADING
         mapRepresentationConverted.m_mapPitch = navmiddleware::MAP_PITCH_3D;
         mapRepresentationConverted.m_mapOrientation = navmiddleware::settings::MAP_ORIENTATION_HEAD_UP;
      }
      break;
      case MapRepresentation__MAP_REPRESENTATION_2D_CAR_HEADING:
      {
         // 2D_CAR_HEADING
         mapRepresentationConverted.m_mapPitch = navmiddleware::MAP_PITCH_2D;
         mapRepresentationConverted.m_mapOrientation = navmiddleware::settings::MAP_ORIENTATION_HEAD_UP;
      }
      break;
      case MapRepresentation__MAP_REPRESENTATION_NOT_SET:
      default:
      {
         ETG_TRACE_USR4(("Default case returning MAP_PITCH_2D and MAP_ORIENTATION_HEAD_UP"));
         mapRepresentationConverted.m_mapPitch = navmiddleware::MAP_PITCH_2D;
         mapRepresentationConverted.m_mapOrientation = navmiddleware::settings::MAP_ORIENTATION_HEAD_UP;
      }
      break;
   }

   return mapRepresentationConverted;
}


::org::bosch::cm::navigation::NavigationService::MapRepresentation  HMIModelNavigationService::convertMapRepresentation(navmiddleware::MapRepresentation mapRepresentation)
{
   if (mapRepresentation.m_mapPitch == navmiddleware::MAP_PITCH_2D && mapRepresentation.m_mapOrientation == navmiddleware::settings::MAP_ORIENTATION_NORTH_UP)
   {
      // 2D_NORTH_UP
      return MapRepresentation__MAP_REPRESENTATION_2D_NORTH_UP;
   }
   else if (mapRepresentation.m_mapPitch == navmiddleware::MAP_PITCH_3D && mapRepresentation.m_mapOrientation == navmiddleware::settings::MAP_ORIENTATION_HEAD_UP)
   {
      // 3D_CAR_HEADING
      return MapRepresentation__MAP_REPRESENTATION_3D_CAR_HEADING;
   }
   else if (mapRepresentation.m_mapPitch == navmiddleware::MAP_PITCH_2D && mapRepresentation.m_mapOrientation == navmiddleware::settings::MAP_ORIENTATION_HEAD_UP)
   {
      // 2D_CAR_HEADING
      return MapRepresentation__MAP_REPRESENTATION_2D_CAR_HEADING;
   }
   else
   {
      ETG_TRACE_USR4(("Default case returning MapRepresentation__MAP_REPRESENTATION_NOT_SET"));
      return MapRepresentation__MAP_REPRESENTATION_NOT_SET;
   }
}


bool HMIModelNavigationService::onPropertyMapPresentatioanModeChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyMapPresentatioanModeChanged()"));
   setMapRepresentation(convertMapRepresentation(_navMiddleware.getMapRepresentationInfo().getMapRepresentation()));
   return true;
}


void HMIModelNavigationService::onRetriggerAcousticOutputRequest(const ::boost::shared_ptr< RetriggerAcousticOutputRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onRetriggerAcousticOutputRequest()"));
   _navMiddleware.retriggerAcousticOutput();
   sendRetriggerAcousticOutputResponse();
}


void HMIModelNavigationService::onSpeakSamplePhraseRequest(const ::boost::shared_ptr< SpeakSamplePhraseRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSpeakSamplePhraseRequest()"));
   SamplePhrase samplePhrase = SAMPLE_PHRASE__VOICE_OUTPUT;
   switch (request->getSampleVoicePhrase())
   {
      case SampleVoicePhrase__SAMPLE_PHRASE__GUIDANCE_WILL_BE_AT_THIS_VOLUME:
         samplePhrase = SAMPLE_PHRASE__GUIDANCE_WILL_BE_AT_THIS_VOLUME;
         break;
      case SampleVoicePhrase__SAMPLE_PHRASE__LOUDER:
         samplePhrase = SAMPLE_PHRASE__LOUDER;
         break;
      case SampleVoicePhrase__SAMPLE_PHRASE__QUIETER:
         samplePhrase = SAMPLE_PHRASE__QUIETER;
         break;
      case SampleVoicePhrase__SAMPLE_PHRASE__VOICE_OUTPUT:
         samplePhrase = SAMPLE_PHRASE__VOICE_OUTPUT;
         break;
      default:
         ETG_TRACE_USR4(("Default sample phrase SAMPLE_PHRASE__VOICE_OUTPUT"));
         break;
   }
   _navMiddleware.speakSamplePhrase(samplePhrase);
   sendSpeakSamplePhraseResponse();
}


void HMIModelNavigationService::onSetVoiceGuidanceRequest(const ::boost::shared_ptr< SetVoiceGuidanceRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSetVoiceGuidanceRequest()"));
   navmiddleware::settings::GuidanceSettings guidanceSettings = _navMiddleware.getGuidanceSettings();
   guidanceSettings.setVoiceGuidanceMode(request->getActive());
   _navMiddleware.updateGuidanceSettings(guidanceSettings);
   _navMiddleware.applySettings();
   sendSetVoiceGuidanceResponse();
}


bool HMIModelNavigationService::onPropertyUpdateGuidanceSettingsChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateGuidanceSettingsChanged(%d)", _navMiddleware.getGuidanceSettings().getVoiceGuidanceMode()));
   setIsVoiceGuidanceActive(_navMiddleware.getGuidanceSettings().getVoiceGuidanceMode());
   sendIsVoiceGuidanceActiveUpdate();
   return true;
}


void HMIModelNavigationService::onRequestSetFreeTextSearchTypeRequest(const ::boost::shared_ptr< RequestSetFreeTextSearchTypeRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onRequestSetFreeTextSearchTypeRequest(%d)", request->getLocationType()));
   _ftsLocationType = request->getLocationType();
   sendRequestSetFreeTextSearchTypeResponse();
}


void HMIModelNavigationService::onShowDestinationFreeTextSearchInputRequest(const ::boost::shared_ptr< ShowDestinationFreeTextSearchInputRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onShowNaviDestinationFreeTextSearchRequest()"));
   _navMiddleware.startFreeTextSearchInput(convertFreeTextSearchLocationType(request->getLocationType()), LocationSearchScope(convertFreeTextSearchScope(request->getLocationSearchScope())));
   _navMiddleware.setFreeTextSearchString(request->getSearchString());
   /* @ToDo: Implementation to be added if required
   POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchOutReqMsg)(NAVIGATION_CONTEXT_DEST_FREETEXT_SEARCH, 0, APPID_APPHMI_NAVIGATION))); */
   sendShowDestinationFreeTextSearchInputResponse();
}


void HMIModelNavigationService::onRequestFreeTextSearchResultsRequest(const ::boost::shared_ptr< RequestFreeTextSearchResultsRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onRequestFreeTextSearchResultsRequest(%d)", request->getDestinationIndex()));
   _freeTextSearchResultAct = request->getAct();
   _navMiddleware.startFreeTextSearchInput(convertFreeTextSearchLocationType(_ftsLocationType), LocationSearchScope(convertFreeTextSearchScope(request->getSearchScope())),
                                           LOCATIONSEARCHSOURCE__OFFLINE, LocationContext(navmiddleware::LocationContext::HMICONTEXT__SDS, APPLICATIONID__NAVIGATION));
   _navMiddleware.setFreeTextSearchString(request->getSearchString());
   _navMiddleware.requestFreeTextSearchResults(request->getStartIndex(), request->getNumElements(), convertFreeTextSearchLocationType(_ftsLocationType));
   _ftsLocationType = FTSLocationType__FREETEXT_SEARCH_TYPE_POI;
}


navmiddleware::FreeTextSearchType HMIModelNavigationService::convertFreeTextSearchLocationType(org::bosch::cm::navigation::NavigationService::FTSLocationType locationType)
{
   switch (locationType)
   {
      case FTSLocationType__FREETEXT_SEARCH_TYPE_COMBINED:
         return FREETEXTSEARCHTYPE__COMBINED;
      case FTSLocationType__FREETEXT_SEARCH_TYPE_ADDRESS:
         return FREETEXTSEARCHTYPE__ADDRESS;
      case FTSLocationType__FREETEXT_SEARCH_TYPE_POI:
         return FREETEXTSEARCHTYPE__POI;
      default:
         ETG_TRACE_USR4(("Default case returning FREETEXTSEARCHTYPE__POI"));
         return FREETEXTSEARCHTYPE__POI;
   }
}


void HMIModelNavigationService::onUndoFreeTextSearchElementSelectionRequest(const ::boost::shared_ptr< UndoFreeTextSearchElementSelectionRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onUndoFreeTextSearchElementSelectionRequest()"));
   _navMiddleware.undoFreeTextSearchElementSelection();
   sendUndoFreeTextSearchElementSelectionResponse();
}


navmiddleware::LocationSearchScope::Type HMIModelNavigationService::convertFreeTextSearchScope(org::bosch::cm::navigation::NavigationService::FreeTextSearchScope freeTextSearchScope)
{
   switch (freeTextSearchScope)
   {
      case FreeTextSearchScope__FREETEXT_SEARCH_SCOPE_POI_AT_CURRENT_POSITION:
         return LocationSearchScope::TYPE__AROUND_CURRENT_VEHICLE_POSITION;
      case FreeTextSearchScope__FREETEXT_SEARCH_SCOPE_POI_AT_DESTINATION_POSITION:
         return LocationSearchScope::TYPE__AROUND_WAYPOINT_POSITION;
      case FreeTextSearchScope__FREETEXT_SEARCH_SCOPE_POI_AT_LOCATION:
         return LocationSearchScope::TYPE__AROUND_CURRENT_LOCATION_POSITION;
      case FreeTextSearchScope__FREETEXT_SEARCH_SCOPE_POI_ALONG_ROUTE:
         return LocationSearchScope::TYPE__ALONG_ROUTE;
      case FreeTextSearchScope__FREETEXT_SEARCH_SCOPE_ON_WHOLE_DATASET:
         return LocationSearchScope::TYPE__ON_WHOLE_DATASET;
      case FreeTextSearchScope__FREETEXT_SEARCH_SCOPE_POI_WHOLE_DATA:
      case FreeTextSearchScope__FREETEXT_SEARCH_SCOPE_ADDRESS_WHOLE_DATA:
      case FreeTextSearchScope__FREETEXT_SEARCH_SCOPE_COMBINED_WHOLE_DATA:
      default:
         ETG_TRACE_USR4(("Default case returning TYPE__AROUND_CURRENT_VEHICLE_POSITION"));
         return LocationSearchScope::TYPE__AROUND_CURRENT_VEHICLE_POSITION;
   }
}


bool HMIModelNavigationService::onPropertyUpdateLocationFreeTextSearchResultsChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateLocationFreeTextSearchResultsChanged()"));
   updateLocationSearchResult(_navMiddleware.getFreeTextSearchResultInfos());
   return true;
}


void HMIModelNavigationService::updateLocationSearchResult(const navmiddleware::LocationInfos& locationInfos)
{
   ETG_TRACE_USR4(("(freeTextSearch = %d,getFixedPOI = %d, areResultsOutstanding = %d)", _freeTextSearchResultAct, _getFixedPOICategorySearchAct, locationInfos.areResultsOutstanding()));
   if (((_freeTextSearchResultAct != 0) || (_getFixedPOICategorySearchAct != 0)) && (locationInfos.areResultsOutstanding() == false))
   {
      const ::std::vector<navmiddleware::LocationInfos::LocationInfo>& infos = locationInfos.getLocationInfos();
      ::std::vector<navmiddleware::LocationInfos::LocationInfo>::const_iterator info = infos.begin();
      ::std::vector< org::bosch::cm::navigation::NavigationService::AddressListElement > AddressListElements;
      for (; (info != infos.end()); ++info)
      {
         AddressListElement addressListElement;
         addressListElement.setData(info->m_elementName);
         addressListElement.setId(static_cast<uint64>(info->m_index));
         if (info->m_direction.isValid())
         {
            addressListElement.setDirection(convertDirectionDescription(info->m_direction.getValue()));
         }
         else
         {
            addressListElement.setDirection(DirectionDescription__INVALID);
         }
         addressListElement.setDistance(info->m_distance);
         addressListElement.setAddress(info->m_description);
         PositionWGS84 addressPosition;
         if (info->m_coordinate.isValid())
         {
            addressPosition.setLongitude(info->m_coordinate.getValue().getLongitude());
            addressPosition.setLatitude(info->m_coordinate.getValue().getLatitude());
         }
         addressListElement.setCoordinates(addressPosition);

         GlobalUpdateRegionId globalUpdateRegionId;
         globalUpdateRegionId.setProductId(info->m_sdsGlobalNamedObjectIdInfo.getValue().getSdsGlobalUpdateRegionInfo().getProductId());
         globalUpdateRegionId.setSupplierId(info->m_sdsGlobalNamedObjectIdInfo.getValue().getSdsGlobalUpdateRegionInfo().getSupplierId());
         globalUpdateRegionId.setUpdateRegionId(info->m_sdsGlobalNamedObjectIdInfo.getValue().getSdsGlobalUpdateRegionInfo().getUpdateRegionId());

         GlobalNamedObjectId globalNameObjectIds;
         globalNameObjectIds.setGlobalUpdateRegionId(globalUpdateRegionId);
         globalNameObjectIds.setNamedObjectId(info->m_sdsGlobalNamedObjectIdInfo.getValue().getNamedObjectId());
         globalNameObjectIds.setIsValid(info->m_sdsGlobalNamedObjectIdInfo.isValid());

         addressListElement.setSdsNameObjectId(globalNameObjectIds);

         AddressListElements.push_back(addressListElement);
         ETG_TRACE_USR4(("Idx %d, name %s", info->m_index, info->m_elementName.c_str()));
      }
      _infoStore.setNaviServiceRequestedData(false);
      if (_freeTextSearchResultAct)
      {
         sendRequestFreeTextSearchResultsResponse(AddressListElements, _freeTextSearchResultAct);
         _freeTextSearchResultAct = 0;
      }
      if (_getFixedPOICategorySearchAct)
      {
         sendRequestFixedPOICategorySearchResultsResponse(AddressListElements, _getFixedPOICategorySearchAct);
         _getFixedPOICategorySearchAct = 0;
      }
   }
}


void HMIModelNavigationService::onSetLocationWithFreeTextSearchInputRequest(const ::boost::shared_ptr< SetLocationWithFreeTextSearchInputRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onRequestFreeTextSearchResultsRequest(%d)", request->getId()));
   const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getFreeTextSearchResultInfos();
   const ::std::vector<navmiddleware::LocationInfos::LocationInfo>& locationInfoVector = locationInfos.getLocationInfos();

   if (request->getId() < unsigned(locationInfoVector.size()))
   {
      navmiddleware::LocationInfos::LocationInfo locationInfo = locationInfoVector[request->getId()];
      _navMiddleware.setLocationWithFreeTextSearchResult(locationInfo);
   }
   sendSetLocationWithFreeTextSearchInputResponse();
}


bool HMIModelNavigationService::onPropertyUpdateInterSectionMapVectorDataChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateInterSectionMapVectorDataChanged()"));
   IntersectionMap& intersectionMap = getIntersectionMapMutable();
   const navmiddleware::IntersectionMapVectorDataInfos::IntersectionMapVectorDataInfo& intersectionMapVectorDataInfo = _navMiddleware.getIntersectionMapVectorDataInfos().getInfo();

   ::std::vector< org::bosch::cm::navigation::NavigationService::IntersectionMapRoadLine >& intersectionMapRoadLinesList = intersectionMap.getRoadMutable();
   intersectionMapRoadLinesList.resize(intersectionMapVectorDataInfo.getRoads().size());
   const navmiddleware::IntersectionMapVectorDataInfos::IntersectionMapVectorDataInfo::IntersectionMapRoadLineList& mw_IntersectionMapRoadLineList = intersectionMapVectorDataInfo.getRoads();
   for (unsigned int roads = 0; roads < mw_IntersectionMapRoadLineList.size(); ++roads)
   {
      ::std::vector< org::bosch::cm::navigation::NavigationService::ScreenCoordinate >& screenCoordinate = intersectionMapRoadLinesList[roads].getCoordinatesMutable();
      screenCoordinate.resize(mw_IntersectionMapRoadLineList[roads].m_coordinates.size());
      for (unsigned int road = 0; road < screenCoordinate.size(); ++road)
      {
         screenCoordinate[road].setXPos(mw_IntersectionMapRoadLineList[roads].m_coordinates[road].m_xPos);
         screenCoordinate[road].setYPos(mw_IntersectionMapRoadLineList[roads].m_coordinates[road].m_yPos);
      }
      intersectionMapRoadLinesList[roads].setRoadClass(static_cast<org::bosch::cm::navigation::NavigationService::AdminRoadClass>(mw_IntersectionMapRoadLineList[roads].m_roadclass));
   }

   ::std::vector< org::bosch::cm::navigation::NavigationService::IntersectionMapRoadLine >& intersectionMapRouteLinesList = intersectionMap.getRouteMutable();
   intersectionMapRouteLinesList.resize(intersectionMapVectorDataInfo.getRoute().size());
   const navmiddleware::IntersectionMapVectorDataInfos::IntersectionMapVectorDataInfo::IntersectionMapRoadLineList& mw_IntersectionMapRoutsLineList = intersectionMapVectorDataInfo.getRoute();
   for (unsigned int routes = 0; routes < mw_IntersectionMapRoutsLineList.size(); ++routes)
   {
      ::std::vector< org::bosch::cm::navigation::NavigationService::ScreenCoordinate >& screenCoordinate = intersectionMapRouteLinesList[routes].getCoordinatesMutable();
      screenCoordinate.resize(mw_IntersectionMapRoutsLineList[routes].m_coordinates.size());
      for (unsigned int route = 0; route < screenCoordinate.size(); ++route)
      {
         screenCoordinate[route].setXPos(mw_IntersectionMapRoutsLineList[routes].m_coordinates[route].m_xPos);
         screenCoordinate[route].setYPos(mw_IntersectionMapRoutsLineList[routes].m_coordinates[route].m_yPos);
      }
      intersectionMapRouteLinesList[routes].setRoadClass(static_cast<org::bosch::cm::navigation::NavigationService::AdminRoadClass>(mw_IntersectionMapRoutsLineList[routes].m_roadclass));
   }

   ::std::vector< org::bosch::cm::navigation::NavigationService::ScreenCoordinate >& setNoEntryIcons = intersectionMap.getNoEntryIconsMutable();
   setNoEntryIcons.resize(intersectionMapVectorDataInfo.getNoEntryIconsPositions().size());
   for (unsigned int noEntryIcons = 0; noEntryIcons < setNoEntryIcons.size(); ++noEntryIcons)
   {
      setNoEntryIcons[noEntryIcons].setXPos(intersectionMapVectorDataInfo.getNoEntryIconsPositions()[noEntryIcons].m_xPos);
      setNoEntryIcons[noEntryIcons].setYPos(intersectionMapVectorDataInfo.getNoEntryIconsPositions()[noEntryIcons].m_yPos);
   }
   intersectionMap.setManeuverId(intersectionMapVectorDataInfo.getManeuverId());
   // Update intersection view data to Fc_meter
   sendIntersectionMapUpdate();
   return true;
}


void HMIModelNavigationService::onShowMapScreenWithMapViewModeRequest(const ::boost::shared_ptr< ShowMapScreenWithMapViewModeRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onShowMapScreenWithMapViewModeRequest(%d)", request->getMapViewMode()));
   navmiddleware::settings::MapView mapView = _navMiddleware.getChangeMapViewSettings().getMapView();
   navmiddleware::settings::ChangeMapViewSettings mapViewSettings = _navMiddleware.getChangeMapViewSettings();
   if ((request->getMapViewMode() == MapViewMode__MAP_VIEW_MODE_SPLIT_MAP) || (request->getMapViewMode() == MapViewMode__MAP_VIEW_MODE_FULL_MAP))
   {
      showMapMain();
      if ((mapView == navmiddleware::settings::MAPVIEW_2D) || (mapView == navmiddleware::settings::MAPVIEW_2D_SPLIT))
      {
         if (request->getMapViewMode() == MapViewMode__MAP_VIEW_MODE_SPLIT_MAP)
         {
            mapViewSettings.setMapView(navmiddleware::settings::MAPVIEW_2D_SPLIT);
            _infoStore.triggerMapMainScreenStateChanged();
         }
         else
         {
            mapViewSettings.setMapView(navmiddleware::settings::MAPVIEW_2D);
            _infoStore.triggerMapMainScreenStateChanged();
         }
      }
      else if ((mapView == navmiddleware::settings::MAPVIEW_3D) || (mapView == navmiddleware::settings::MAPVIEW_3D_SPLIT))
      {
         if (request->getMapViewMode() == MapViewMode__MAP_VIEW_MODE_SPLIT_MAP)
         {
            mapViewSettings.setMapView(navmiddleware::settings::MAPVIEW_3D_SPLIT);
            _infoStore.triggerMapMainScreenStateChanged();
         }
         else
         {
            mapViewSettings.setMapView(navmiddleware::settings::MAPVIEW_3D);
            _infoStore.triggerMapMainScreenStateChanged();
         }
      }
   }
   else
   {
      ETG_TRACE_USR4(("invalid input!"));
   }
}


void HMIModelNavigationService::onShowMapScreenCarsorLockModeRequest(const ::boost::shared_ptr< ShowMapScreenCarsorLockModeRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onShowMapScreenCarsorLockModeRequest()"));
   showMapMain();
   /* @ToDo: Implementation to be added if required
   POST_MSG((COURIER_MESSAGE_NEW(SetMapCameraModeReqMsg)(MAP_CAMERA_AND_VIEW_MODE_CARSOR_FULLSCREEN))); */
}


void HMIModelNavigationService::onShowDeleteWaypointListScreenRequest(const ::boost::shared_ptr< ShowDeleteWaypointListScreenRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onShowDeleteWaypointListScreenRequest()"));
   /* @ToDo: Implementation to be added if required
   POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchOutReqMsg)(NAVIGATION_CONTEXT_CHANGE_EDIT_ROUTE, 0, APPID_APPHMI_NAVIGATION))); */
   sendShowDeleteWaypointListScreenResponse();
}


void HMIModelNavigationService::onShowCarsorLocationOnMapRequest(const ::boost::shared_ptr< ShowCarsorLocationOnMapRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onShowCarsorLocationOnMapRequest()"));
   navmiddleware::ValidValue<int> scale(DEFAULT_ZOOM_SCALE_FOR_LOCATION_OVERVIEW);
   _navMiddleware.centerMapToGeoPosition(MAP_VIEW_ID__PRIMARY, navmiddleware::GeoCoordinateDegree(getPositionInformation().getPositionWGS84().getLatitude(), getPositionInformation().getPositionWGS84().getLongitude()), scale);
   sendShowCarsorLocationOnMapResponse();
}


void HMIModelNavigationService::onSetZoomLevelRequest(const ::boost::shared_ptr< SetZoomLevelRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSetZoomLevelRequest(%d)", request->getZoomLevel()));
   _zoomToStreetScaleLevel = convertZoomLevel(request->getZoomLevel());
   _isZoomStreetPending = true;
   sendSetZoomLevelResponse();
}


bool HMIModelNavigationService::onPropertyUpdateMapCameraAndViewModesDoneChanged()
{
   if (_isZoomStreetPending == true)
   {
      ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateMapCameraAndViewModesDoneChanged(_isZoomStreetPending %d, _zoomToStreetScaleLevel %d)", _isZoomStreetPending, _zoomToStreetScaleLevel));
      // setMapScale() has to be called after setMapCameraAndViewModes() and switchToLinkedMode()
      _navMiddleware.setMapScale(navmiddleware::MAP_VIEW_ID__PRIMARY, _zoomToStreetScaleLevel, navmiddleware::CameraAnimation__IMMEDIATE);
      _isZoomStreetPending = false;
      _zoomToStreetScaleLevel = 0;
   }
   return true;
}


void HMIModelNavigationService::onCheckNavDataCompatibilityRequest(const ::boost::shared_ptr< CheckNavDataCompatibilityRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onCheckNavDataCompatibilityRequest()"));
   _getNavDataCompatibilityAct = request->getAct();
   if (isInitialized() || _infoStore.isNavDataUpdateRunning()) // send COMPATIBLE/NOT_COMPATIBLE
   {
      _navMiddleware.checkNavDataCompatibility(request->getNavDataVersion(), request->getCompatibilityInfoMutable());
   }
   else // MDW stopped, MapUpdate started, MapUpdate stopped
   {
      sendCheckNavDataCompatibilityResponse(NavDataCompatibility__NAVDATA_COMPATIBILITY_UNKNOWN, _getNavDataCompatibilityAct);
      _getNavDataCompatibilityAct = 0;
   }
}


bool HMIModelNavigationService::onPropertyUpdateNavDataUpdateCompatibilityChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateNavDataUpdateCompatibilityChanged()"));

   if (_getNavDataCompatibilityAct != 0)
   {
      sendCheckNavDataCompatibilityResponse(convertNavDataCompatibilityType(_navMiddleware.getNavDataCompatibilityInfo().getCompatibilityState()), _getNavDataCompatibilityAct);
   }
   else
   {
      ETG_TRACE_ERR(("onPropertyUpdateNavDataUpdateCompatibilityChanged Error(%d)", _getNavDataCompatibilityAct));
   }
   return true;
}


::org::bosch::cm::navigation::NavigationService::NavDataCompatibility HMIModelNavigationService::convertNavDataCompatibilityType(navmiddleware::NavDataCompatibilityInfo::CompatibilityState compatibilityState)
{
   switch (compatibilityState)
   {
      case navmiddleware::NavDataCompatibilityInfo::COMPATIBILITYSTATE__COMPATIBLE:
         return NavDataCompatibility__NAVDATA_COMPATIBLE;
      case navmiddleware::NavDataCompatibilityInfo::COMPATIBILITYSTATE__INCOMPATIBLE:
         return NavDataCompatibility__NAVDATA_INCOMPATIBLE;
      case navmiddleware::NavDataCompatibilityInfo::COMPATIBILITYSTATE__UNKNOWN:
         return NavDataCompatibility__NAVDATA_COMPATIBILITY_UNKNOWN;
      default:
         ETG_TRACE_USR4(("Default case retrning NavDataCompatibility__NAVDATA_COMPATIBILITY_UNKNOWN"));
         return NavDataCompatibility__NAVDATA_COMPATIBILITY_UNKNOWN;
   }
}


bool  HMIModelNavigationService::onPropertyUpdateDestinationMemoryHomeLocationListChanged()
{
   const navmiddleware::DestinationMemoryEntryList& destinationMemoryEntryList = _navMiddleware.getEntryList();
   const ::std::vector<navmiddleware::DestinationMemoryEntry>& infos = destinationMemoryEntryList.getEntryList();
   Location& location = getHomeLocationMutable();
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateDestinationMemoryHomeLocationListChanged(%d)", infos.empty()));
   if (!infos.empty())
   {
      location.setValid(true);
      location.setName(infos.front().getDestination().getName());
      PositionWGS84& position = location.getPositionWGS84Mutable();;
      position.setLatitude(infos.front().getDestination().getLatitude());
      position.setLongitude(infos.front().getDestination().getLongitude());
      location.setPositionWGS84(position);
      ETG_TRACE_USR4(("location: %s", location.getName().c_str()));
   }
   else
   {
      ETG_TRACE_USR4(("Invalid home location list"));
      location.clearName();
      location.clearValid();
      location.clearPositionWGS84();
   }
   sendHomeLocationUpdate();
   return true;
}


bool HMIModelNavigationService::onPropertyUpdateHomeLocationDeleted()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateHomeLocationDeleted"));
   Location& location = getHomeLocationMutable();
   location.clearName();
   location.clearValid();
   location.clearPositionWGS84();
   sendHomeLocationUpdate();
   return true;
}


bool  HMIModelNavigationService::onPropertyUpdateDestinationMemoryWorkLocationListChanged()
{
   const navmiddleware::DestinationMemoryEntryList& destinationMemoryEntryList = _navMiddleware.getEntryList();
   const ::std::vector<navmiddleware::DestinationMemoryEntry>& infos = destinationMemoryEntryList.getEntryList();
   Location& location = getWorkLocationMutable();
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateDestinationMemoryWorkLocationListChanged(%d)", infos.empty()));
   if (!infos.empty())
   {
      location.setValid(true);
      location.setName(infos.front().getDestination().getName());
      PositionWGS84& position = location.getPositionWGS84Mutable();
      position.setLatitude(infos.front().getDestination().getLatitude());
      position.setLongitude(infos.front().getDestination().getLongitude());
      location.setPositionWGS84(position);
   }
   else
   {
      ETG_TRACE_USR4(("Invalid work location list"));
      location.clearName();
      location.clearValid();
      location.clearPositionWGS84();
   }
   sendWorkLocationUpdate();
   return true;
}


bool HMIModelNavigationService::onPropertyUpdateWorkLocationDeleted()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateWorkLocationDeleted"));
   Location& location = getWorkLocationMutable();
   location.clearName();
   location.clearValid();
   location.clearPositionWGS84();
   sendWorkLocationUpdate();
   return true;
}


bool HMIModelNavigationService::onPropertyUpdateSpeedLimitChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateSpeedStatusChanged()"));
   const SpeedInfo& speedInfo = _navMiddleware.getSpeedInfo();
   ::org::bosch::cm::navigation::NavigationService::SpeedLimit speedLimit;
   speedLimit.setPresent(speedInfo.isSpeedLimitPresent());
   speedLimit.setCountryCode(speedInfo.getCountryCode());
   speedLimit.setValue(speedInfo.getSpeedLimit());
   speedLimit.setSpeedUnit(convertSpeedUnit(speedInfo.getSpeedLimitUnit()));

   setSpeedLimit(speedLimit);
   return true;
}


::org::bosch::cm::navigation::NavigationService::SpeedUnit HMIModelNavigationService::convertSpeedUnit(navmiddleware::SpeedUnit speedUnit)
{
   switch (speedUnit)
   {
      case navmiddleware::KM_PER_HOUR:
         return SpeedUnit__KM_PER_HOUR;
      case navmiddleware::MI_PER_HOUR:
         return SpeedUnit__MI_PER_HOUR;
      default:
         ETG_TRACE_USR4(("Default case returning SpeedUnit__KM_PER_HOUR"));
         return SpeedUnit__KM_PER_HOUR;
   }
}


bool HMIModelNavigationService::onPropertyUpdateLocationPoiShortcutInformationChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateLocationPoiShortcutInformationChanged()"));
   updateLocationPoiShortcutInformation(navmiddleware::LocationContext::HMICONTEXT__SDS);
   updateLocationPoiShortcutInformation(navmiddleware::LocationContext::HMICONTEXT__GUI);
   return true;
}


void HMIModelNavigationService::updateLocationPoiShortcutInformation(navmiddleware::LocationContext::HmiContext hmiContext)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::updateLocationPoiShortcutInformation(%d)", hmiContext));
   const navmiddleware::LocationCategoryInfos& locationCategoryInfos = _navMiddleware.getPoiShortcutCategoryInfos(LocationContext(
            hmiContext, navmiddleware::APPLICATIONID__NAVIGATION));
   const ::std::vector<navmiddleware::LocationCategoryInfos::LocationCategoryInfo>& infos = locationCategoryInfos.getLocationCategoryInfos();
   ::std::vector< PoiCategoryShortcut > poiCategoryShortcuts;

   if (!infos.empty())
   {
      std::ostringstream traceOut;
      traceOut << "{ iconSize blobLength } = { ";
      for (unsigned int shortcutIndex = 0; shortcutIndex < infos.size(); ++shortcutIndex)
      {
         PoiCategoryShortcut poiCategoryShortcut;
         poiCategoryShortcut.setCategoryName(infos[shortcutIndex].m_categoryName);
         poiCategoryShortcut.setPoiCategoryId(infos[shortcutIndex].m_poiCategoryId);
         ::org::bosch::cm::navigation::NavigationService::Image icon;
         icon.setImage(::asf::core::Blob(infos[shortcutIndex].m_imageBlob, infos[shortcutIndex].m_imageBlobLength));
         traceOut << icon.getImage().getSize() << " " << infos[shortcutIndex].m_imageBlobLength << ", ";
         icon.setImageType(ImageType__PNG);
         poiCategoryShortcut.setIcon(icon);
         poiCategoryShortcut.setShortcutIndex(shortcutIndex);
         poiCategoryShortcuts.push_back(poiCategoryShortcut);
      }
      traceOut << " }";
      ETG_TRACE_USR4(("%s", traceOut.str().c_str()));
   }
   else
   {
      ETG_TRACE_USR4(("locationCategoryInfos is empty!"));
   }

   if (navmiddleware::LocationContext::HMICONTEXT__SDS == hmiContext)
   {
      setSdsPOICategoryShortcuts(poiCategoryShortcuts);
   }
   else if (navmiddleware::LocationContext::HMICONTEXT__GUI == hmiContext)
   {
      setMenuPoiCategoryShortcuts(poiCategoryShortcuts);
   }
}


void HMIModelNavigationService::onSelectPoiCategoryShortcutRequest(const ::boost::shared_ptr< SelectPoiCategoryShortcutRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSelectPoiCategoryShortcutRequest()"));
   POST_MSG((COURIER_MESSAGE_NEW(InitPOINearbyInputReqMsg)(request->getShortcutIndex(), 0)));
   /* @ToDo: Implementation to be added if required
   POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchOutReqMsg)(NAVIGATION_CONTEXT_NEARBY_POI_SHORTCUTS, 0, APPID_APPHMI_NAVIGATION))); */
   sendSelectPoiCategoryShortcutResponse();
}


void HMIModelNavigationService::onGetPoiCategoryShortcutsRequest(const ::boost::shared_ptr< GetPoiCategoryShortcutsRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onGetPoiCategoryShortcutsRequest()"));
   const navmiddleware::LocationCategoryInfos& locationCategoryInfos = _navMiddleware.getPoiShortcutCategoryInfos(LocationContext(
            navmiddleware::LocationContext::HMICONTEXT__SDS, navmiddleware::APPLICATIONID__NAVIGATION));
   const ::std::vector<navmiddleware::LocationCategoryInfos::LocationCategoryInfo>& infos = locationCategoryInfos.getLocationCategoryInfos();
   ::std::vector< PoiCategoryShortcut > poiCategoryShortcuts;
   if (infos.size() != 0)
   {
      for (unsigned int shortcutIndex = 0; ((shortcutIndex < infos.size()) && (shortcutIndex < NEARBY_CATEGORY_SHORTCUT_MAX)); ++shortcutIndex)
      {
         PoiCategoryShortcut poiCategoryShortcut;
         poiCategoryShortcut.setCategoryName(infos[shortcutIndex].m_categoryName);
         ::org::bosch::cm::navigation::NavigationService::Image icon;
         icon.setImage(::asf::core::Blob(infos[shortcutIndex].m_imageBlob, infos[shortcutIndex].m_imageBlobLength));
         icon.setImageType(ImageType__PNG);
         poiCategoryShortcut.setShortcutIndex(shortcutIndex);
         ETG_TRACE_USR4(("categoryName: %s", infos[shortcutIndex].m_categoryName.c_str()));
         poiCategoryShortcuts.push_back(poiCategoryShortcut);
      }
      sendGetPoiCategoryShortcutsResponse(poiCategoryShortcuts, request->getAct());
   }
   else
   {
      sendGetPoiCategoryShortcutsError(DBUS_ERROR_FAILED, "list is empty", request->getAct());
   }
}


bool HMIModelNavigationService::onPropertyUpdateRouteCalculationInfoChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateRouteCalculationInfoChanged"));
   const ::navmiddleware::RouteCalculationInfo& routeCalculationInfo = _navMiddleware.getRouteCalculationInfo();
   org::bosch::cm::navigation::NavigationService::RouteRecalculationReason routeRecalculationReason;
   const ValidValue<unsigned int>& indexActiveAternative = routeCalculationInfo.getIndexActiveAlternative();

   if ((routeCalculationInfo.getStatus() != ::navmiddleware::RouteCalculationInfo::STATUS__NOT_STARTED) && (indexActiveAternative.isValid()))
   {
      const ::std::vector< ::navmiddleware::RouteCalculationInfo::RouteCalculationInfoAlternative >& routeCalculationInfoAlternatives = routeCalculationInfo.getRouteCalculationInfoAlternatives();

      if (routeCalculationInfoAlternatives.size() > indexActiveAternative.getValue())
      {
         switch ((routeCalculationInfoAlternatives.at(indexActiveAternative.getValue())).m_reason)
         {
            case ::navmiddleware::RouteCalculationInfo::RouteCalculationInfoAlternative::REASON__UNKNOWN:
               routeRecalculationReason = RouteRecalculationReason__UNKNOWN;
               break;
            case ::navmiddleware::RouteCalculationInfo::RouteCalculationInfoAlternative::REASON__NO_CALCULATION:
               routeRecalculationReason = RouteRecalculationReason__NO_CALCULATION;
               break;
            case ::navmiddleware::RouteCalculationInfo::RouteCalculationInfoAlternative::REASON__FIRST_CALCULATION:
               routeRecalculationReason = RouteRecalculationReason__FIRST_CALCULATION;
               break;
            case ::navmiddleware::RouteCalculationInfo::RouteCalculationInfoAlternative::REASON__OPTIMIZE_ROUTE:
               routeRecalculationReason = RouteRecalculationReason__OPTIMIZE_ROUTE;
               break;
            case ::navmiddleware::RouteCalculationInfo::RouteCalculationInfoAlternative::REASON__REGARDING_TRAFFIC_MESSAGE:
               routeRecalculationReason = RouteRecalculationReason__REGARDING_TRAFFIC_MESSAGE;
               break;
            case ::navmiddleware::RouteCalculationInfo::RouteCalculationInfoAlternative::REASON__DISREGARD_ROUTE:
               routeRecalculationReason = RouteRecalculationReason__DISREGARD_ROUTE;
               break;
            case ::navmiddleware::RouteCalculationInfo::RouteCalculationInfoAlternative::REASON__REGARDING_USER_TRAFFIC_MESSAGE:
               routeRecalculationReason = RouteRecalculationReason__REGARDING_USER_TRAFFIC_MESSAGE;
               break;
            case ::navmiddleware::RouteCalculationInfo::RouteCalculationInfoAlternative::REASON__ROUTE_OPTION_CHANGED:
               routeRecalculationReason = RouteRecalculationReason__ROUTE_OPTION_CHANGED;
               break;
            case ::navmiddleware::RouteCalculationInfo::RouteCalculationInfoAlternative::REASON__ROUTE_CRITERION_CHANGED:
               routeRecalculationReason = RouteRecalculationReason__ROUTE_CRITERION_CHANGED;
               break;
            case ::navmiddleware::RouteCalculationInfo::RouteCalculationInfoAlternative::REASON__ROUTE_OPTION_CRITERION_CHANGED:
               routeRecalculationReason = RouteRecalculationReason__ROUTE_OPTION_CRITERION_CHANGED;
               break;
            case ::navmiddleware::RouteCalculationInfo::RouteCalculationInfoAlternative::REASON__ENTRY_POINT_EVALUATION:
               routeRecalculationReason = RouteRecalculationReason__ENTRY_POINT_EVALUATION;
               break;
            default:
               routeRecalculationReason = RouteRecalculationReason__NO_ROUTE_AVAILABLE;
               break;
         }
         setRouteRecalculationReason(routeRecalculationReason);
      }
      else
      {
         ETG_TRACE_USR4(("Index of alternative route is out of bound"));
         setRouteRecalculationReason(RouteRecalculationReason__NO_ROUTE_AVAILABLE);
      }
   }
   else
   {
      ETG_TRACE_USR4(("No route is available"));
      setRouteRecalculationReason(RouteRecalculationReason__NO_ROUTE_AVAILABLE);
   }
   return true;
}


bool HMIModelNavigationService::onPropertyRouteStatusChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyRouteStatusChanged"));
   updateRouteCriteria(_navMiddleware.getRouteCriteriaSettings().getRouteCriterion());
   return true;
}


void HMIModelNavigationService::updateRouteCriteria(navmiddleware::settings::RouteCriterion routeCriterion)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::updateRouteCriteria(%d)", routeCriterion));
   RouteCalculationDuration& routeCalculationDuration = getRouteCalculationDurationMutable();
   switch (routeCriterion)
   {
      case navmiddleware::settings::ROUTE_CRITERION_FAST:
         routeCalculationDuration.setRouteCalDurationToFinalDest(_navMiddleware.getRouteCalcDurationInfos().getRouteCalDurationFastToFinalDest().getValue());
         routeCalculationDuration.setRouteCalDurationToNextDest(_navMiddleware.getRouteCalcDurationInfos().getRouteCalDurationFastToNextDest().getValue());
         break;
      case navmiddleware::settings::ROUTE_CRITERION_ECONOMIC:
         routeCalculationDuration.setRouteCalDurationToFinalDest(_navMiddleware.getRouteCalcDurationInfos().getRouteCalDurationEcoToFinalDest().getValue());
         routeCalculationDuration.setRouteCalDurationToNextDest(_navMiddleware.getRouteCalcDurationInfos().getRouteCalDurationEcoToNextDest().getValue());
         break;
      case navmiddleware::settings::ROUTE_CRITERION_SHORT:
         routeCalculationDuration.setRouteCalDurationToFinalDest(_navMiddleware.getRouteCalcDurationInfos().getRouteCalDurationShortToFinalDest().getValue());
         routeCalculationDuration.setRouteCalDurationToNextDest(_navMiddleware.getRouteCalcDurationInfos().getRouteCalDurationShortToNextDest().getValue());
         break;
      default:
         ETG_TRACE_USR4(("Invalid case"));
         break;
   }
   sendRouteCalculationDurationUpdate();
}


bool HMIModelNavigationService::onCourierMessage(const UpdateRouteCriteriaReqMsg& oMsg)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onCourierMessage(const UpdateRouteCriteriaReqMsg(%d))", oMsg.GetCriteriaType()));
   updateRouteCriteria(static_cast<navmiddleware::settings::RouteCriterion>(oMsg.GetCriteriaType()));
   return true;
}


void HMIModelNavigationService::onSetLocationWithCoordinatesRequest(const ::boost::shared_ptr< SetLocationWithCoordinatesRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSetLocationWithCoordinatesRequest()"));
   _navMiddleware.setLocationWithCoordinates(request->getLocation().getLatitude(), request->getLocation().getLongitude());
   sendSetLocationWithCoordinatesResponse();
}


void HMIModelNavigationService::setLocationDetails(const org::bosch::cm::navigation::NavigationService::POILocation& poiLocation)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::setLocationDetails()"));
   std::string name = poiLocation.getLocation().getName();
   double latitude = poiLocation.getLocation().getPositionWGS84().getLatitude();
   double longtitude = poiLocation.getLocation().getPositionWGS84().getLongitude();
   ETG_TRACE_USR4(("latitude %f, longitude %f", latitude, longtitude));
   navmiddleware::ValidValue< navmiddleware::LocationDetails > validLocationDetails;
   navmiddleware::ValidValue<uint64_t> id; // because of not known use an invalid Id
   navmiddleware::LocationOrigin LocationOrigin(LOCATION_ORIGIN__UNKNOWN);
   ::std::string uniqueID;
   validLocationDetails.setValue(
      navmiddleware::LocationDetails(
         poiLocation.getPhoneNumber(),
         poiLocation.getAddress(),
         navmiddleware::Image(
            poiLocation.getIcon().getImage().getSize(),
            poiLocation.getIcon().getImage().getBytes(),
            id,
            static_cast< navmiddleware::Image::Type >(poiLocation.getIcon().getImageType())
         ),
         LocationOrigin,
         uniqueID
      )
   );
   _navMiddleware.setLocationWithExternalLocation(navmiddleware::ExternalLocation(name, navmiddleware::GeoCoordinateDegree(latitude, longtitude), validLocationDetails));
}


void HMIModelNavigationService::onSetLocationWithDetailsRequest(const ::boost::shared_ptr< SetLocationWithDetailsRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSetLocationWithDetailsRequest()"));
   setLocationDetails(request->getLocation());

   sendSetLocationWithDetailsResponse();
}


bool HMIModelNavigationService::onPropertyUpdateTrafficMessageInfoListChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateTrafficMessageInfoListChanged()"));
   setTrafficMessagesCount(_navMiddleware.getCountOfAllTrafficMessages());
   return true;
}


bool HMIModelNavigationService::onPropertyUpdateTrafficMessagesReceptionStatusChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateTrafficMessagesReceptionStatusChanged(%d)", _navMiddleware.getCountOfAllTrafficMessages()));
   ::org::bosch::cm::navigation::NavigationService::TrafficServiceReceptionStatus trafficServiceReceptionStatus = getTrafficServiceReceptionStatus();
   if (_navMiddleware.getTrafficServiceReceptionStatusInfo().isValid())
   {
      trafficServiceReceptionStatus.setServiceName(_navMiddleware.getTrafficServiceReceptionStatusInfo().getValue().getTrafficServiceName());
      trafficServiceReceptionStatus.setTrafficReceptionState(convertTrafficReceptionState(_navMiddleware.getTrafficServiceReceptionStatusInfo().getValue().getTrafficReceptionState()));
      ETG_TRACE_USR4(("TrafficServiceReceptionStatus %d", _navMiddleware.getTrafficServiceReceptionStatusInfo().getValue().getTrafficReceptionState()));
      trafficServiceReceptionStatus.setTrafficServiceReceptionStatus(_navMiddleware.getTrafficServiceReceptionStatusInfo().getValue().getTrafficServiceReceptionStatus());
      trafficServiceReceptionStatus.setTrafficMessageCount(_navMiddleware.getCountOfAllTrafficMessages());
      trafficServiceReceptionStatus.setTrafficMessageSource(convertTrafficMessageSource(_navMiddleware.getTrafficServiceReceptionStatusInfo().getValue().getTrafficMessageSource()));
   }
   else
   {
      trafficServiceReceptionStatus.setServiceName("");
      trafficServiceReceptionStatus.setTrafficReceptionState(TrafficReceptionState__UNKNOWN);
      trafficServiceReceptionStatus.setTrafficServiceReceptionStatus(false);
      trafficServiceReceptionStatus.setTrafficMessageCount(0);
      trafficServiceReceptionStatus.setTrafficMessageSource(TrafficMessageSource__UNKOWN);
   }
   setTrafficServiceReceptionStatus(trafficServiceReceptionStatus);
   return true;
}


::org::bosch::cm::navigation::NavigationService::TrafficMessageSource HMIModelNavigationService::convertTrafficMessageSource(navmiddleware::TrafficMessageSource trafficMessageSource)
{
   switch (trafficMessageSource)
   {
      case navmiddleware::TRAFFIC_MESSAGE_SOURCE__CONNECT_TRAFFIC:
         return TrafficMessageSource__CONNECT_TRAFFIC;
      case navmiddleware::TRAFFIC_MESSAGE_SOURCE__TMC_TUNER:
         return    TrafficMessageSource__TMC_TUNER;
      default:
         ETG_TRACE_USR4(("Default case returning TrafficMessageSource__UNKOWN"));
         return TrafficMessageSource__UNKOWN;
   }
}


void HMIModelNavigationService::onSetSXMTravelLinkSubscriptionStatusRequest(const ::boost::shared_ptr< SetSXMTravelLinkSubscriptionStatusRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSetSXMTravelLinkSubscriptionStatusRequest(%d)", request->getStatus()));
   _infoStore.setSXMSubscriptionStatus(request->getStatus());
}


::org::bosch::cm::navigation::NavigationService::TrafficReceptionState HMIModelNavigationService::convertTrafficReceptionState(navmiddleware::TrafficReceptionState trafficReceptionState)
{
   switch (trafficReceptionState)
   {
      case navmiddleware::TRAFFICRECEPTIONSTATE__UNKNOWN:
         return TrafficReceptionState__UNKNOWN;
      case navmiddleware::TRAFFICRECEPTIONSTATE__OK:
         return TrafficReceptionState__OK;
      case navmiddleware::TRAFFICRECEPTIONSTATE__NOT_SUBSCRIBED:
         return TrafficReceptionState__NOT_SUBSCRIBED;
      case navmiddleware::TRAFFICRECEPTIONSTATE__NOT_AVAILABLE:
         return TrafficReceptionState__NOT_AVAILABLE;
      case navmiddleware::TRAFFICRECEPTIONSTATE__NO_ANTENNA:
         return TrafficReceptionState__NO_ANTENNA;
      case navmiddleware::TRAFFICRECEPTIONSTATE__NO_SIGNAL:
         return TrafficReceptionState__NO_SIGNAL;
      case navmiddleware::TRAFFICRECEPTIONSTATE__TEMPORARILY_NOT_AVAILABLE:
         return TrafficReceptionState__TEMPORARILY_NOT_AVAILABLE;
      case navmiddleware::TRAFFICRECEPTIONSTATE__INACTIVE:
         return TrafficReceptionState__INACTIVE;
      case navmiddleware::TRAFFICRECEPTIONSTATE__NO_INTERNET_CONNECTION:
         return TrafficReceptionState__NO_INTERNET_CONNECTION;
      case navmiddleware::TRAFFICRECEPTIONSTATE__AUTHENTICATION_INVALID:
         return TrafficReceptionState__AUTHENTICATION_INVALID;
      case navmiddleware::TRAFFICRECEPTIONSTATE__CONNECTION_LOST:
         return TrafficReceptionState__CONNECTION_LOST;
      default:
         ETG_TRACE_USR4(("Default case returning TrafficReceptionState__UNKNOWN"));
         return TrafficReceptionState__UNKNOWN;
   }
}


void HMIModelNavigationService::onShowTrafficInfoScreenRequest(const ::boost::shared_ptr< ShowTrafficInfoScreenRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onShowTrafficInfoScreenRequest()"));
   /* @ToDo: Implementation to be added if required
   POST_MSG((COURIER_MESSAGE_NEW(TrafficInfoScreenReqMsg)())); */
}


bool HMIModelNavigationService::onPropertyUpdateNavDataVersionInfoChanged()
{
   const ::std::string& currentVersion = _navMiddleware.getNavigationDataVersionInfo().getCurrentVersion();
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateNavDataVersionInfoChanged(%s)", currentVersion.c_str()));
   setDatasetVersion(currentVersion);
   sendDatasetVersionUpdate();
   return true;
}


bool HMIModelNavigationService::onPropertyUpdateMapIntersectionMapCarsorPositionChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateMapIntersectionMapCarsorPositionChanged()"));
   ::org::bosch::cm::navigation::NavigationService::IntersectionMapCarsorPosition& intersectionMapCarsorPosition = getIntersectionMapCarsorPositionMutable();
   intersectionMapCarsorPosition.setCarsorAngle(_navMiddleware.getIntersectionMapCarsorPosition().getCarsorAngle());
   ::org::bosch::cm::navigation::NavigationService::ScreenCoordinate screenCoordinate = intersectionMapCarsorPosition.getCarsorPositionMutable();
   screenCoordinate.setXPos(_navMiddleware.getIntersectionMapCarsorPosition().getCarsorPosition().xPos);
   screenCoordinate.setYPos(_navMiddleware.getIntersectionMapCarsorPosition().getCarsorPosition().yPos);
   intersectionMapCarsorPosition.setCarsorPosition(screenCoordinate);
   intersectionMapCarsorPosition.setCarsorState(convertMapCarsorState(_navMiddleware.getIntersectionMapCarsorPosition().getCarsorState()));
   intersectionMapCarsorPosition.setManeuverId(_navMiddleware.getIntersectionMapCarsorPosition().getManeuverId());
   setIntersectionMapCarsorPosition(intersectionMapCarsorPosition);
   sendIntersectionMapCarsorPositionUpdate();

   return true;
}


::org::bosch::cm::navigation::NavigationService::IntersectionMapCarsorState HMIModelNavigationService::convertMapCarsorState(navmiddleware::IntersectionMapCarsorPosition::CarsorState carsorState)
{
   switch (carsorState)
   {
      case navmiddleware::IntersectionMapCarsorPosition::OFF:
         return IntersectionMapCarsorState__OFF;
      case navmiddleware::IntersectionMapCarsorPosition::ON:
         return IntersectionMapCarsorState__ON;
      case navmiddleware::IntersectionMapCarsorPosition::STANDBY:
         return IntersectionMapCarsorState__STANDBY;
      default:
         ETG_TRACE_USR4(("Default case returning IntersectionMapCarsorState__OFF"));
         return IntersectionMapCarsorState__OFF;
   }
}


void HMIModelNavigationService::onSetMapLightModeRequest(const ::boost::shared_ptr< SetMapLightModeRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSetMapLightModeRequest(%d)", request->getActive()));
   bool lightMode = request->getActive();
   if (false == lightMode)
   {
      _navMiddleware.configureLightMode(navmiddleware::ConfigurationInfos::LIGHT_MODE_NIGHT_PERMANENT);
   }
   else
   {
      _navMiddleware.configureLightMode(navmiddleware::ConfigurationInfos::LIGHT_MODE_DAY_PERMANENT);
   }
}


void HMIModelNavigationService::onPropertyVicsSourcesChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyVicsSourcesChanged"));

   updateEtcStatusFromVicsSourcesChanged();
}


void HMIModelNavigationService::updateEtcStatusFromVicsSourcesChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::updateEtcStatusFromVicsSourcesChanged"));
}


void HMIModelNavigationService::onRequestJamAheadListRequest(const ::boost::shared_ptr< RequestJamAheadListRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onRequestJamAheadListRequest"));
   _getJamAheadListAct = request->getAct();
   _navMiddleware.requestJamAheadList();
}


void HMIModelNavigationService::onPropertyJamAheadInfoChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyJamAheadInfoChanged"));
   if (_getJamAheadListAct != 0)
   {
      JamAheadList jamAheadList = getJamAheadListMutable();
      const JamAheadListInfo& jamAheadListInfo = _navMiddleware.getJamAheadListInfo();
      jamAheadList.setJamAheadActivityStatus(getJamAheadActivityStatus(jamAheadListInfo.getJamAheadActivityStatus()));
      ::std::vector< JamAheadListElement >& jamAheadListElements = jamAheadList.getJamAheadListElementsMutable();
      jamAheadListElements.resize(jamAheadListInfo.getJamAheadList().size());
      for (::std::vector<navmiddleware::JamAheadListInfo::JamAheadListElement>::const_iterator itr = jamAheadListInfo.getJamAheadList().begin(); itr != jamAheadListInfo.getJamAheadList().end(); ++itr)
      {
         JamAheadListElement jamAheadListElement;
         jamAheadListElement.setDistance(itr->m_distance);
         jamAheadListElement.setIsEnabled(itr->m_enabled);
         jamAheadListElements.push_back(jamAheadListElement);
      }
      jamAheadList.setJamAheadListElements(jamAheadListElements);
      setJamAheadList(jamAheadList);
      sendRequestJamAheadListResponse(_getJamAheadListAct);
      _getJamAheadListAct = 0;
   }
}


void HMIModelNavigationService::onSelectJamAheadListEntryRequest(const ::boost::shared_ptr< SelectJamAheadListEntryRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSelectJamAheadListEntryRequest"));
   _navMiddleware.selectJamAheadListEntry(request->getIndex());
}


void HMIModelNavigationService::onReleaseJamAheadListRequest(const ::boost::shared_ptr< ReleaseJamAheadListRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onReleaseJamAheadListRequest"));
   _navMiddleware.releaseJamAheadList();
}


void HMIModelNavigationService::onCancelJamAheadRequest(const ::boost::shared_ptr< CancelJamAheadRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onCancelJamAheadRequest"));
   _navMiddleware.cancelJamAhead();
}


void HMIModelNavigationService::onRequestFixedPOICategorySearchResultsRequest(const ::boost::shared_ptr< RequestFixedPOICategorySearchResultsRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onRequestFixedPOICategorySearchResultsRequest"));
   _navMiddleware.startPoiInputWithPredefinedCategory(getFixedPOICategory(request->getFixedPOICategory()), LocationSearchScope(convertFreeTextSearchScope(request->getLocationSearchScope())),
         LOCATIONSEARCHSOURCE__OFFLINE, POI_CATEGORY_FILTER__NO_FILTER, LocationContext(navmiddleware::LocationContext::HMICONTEXT__SDS, APPLICATIONID__NAVIGATION));
   _infoStore.setNaviServiceRequestedData(true);
   _navMiddleware.requestPoiEntries(request->getStartIndex(), request->getNumElements());
   _getFixedPOICategorySearchAct = request->getAct();
}


void HMIModelNavigationService::onPropertyPOIEntriesChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyPOIEntriesChanged"));
   updateLocationSearchResult(_navMiddleware.getPoiEntries());
}


void HMIModelNavigationService::onUndoSetLocationWithFixedPOICategoryInputRequest(const ::boost::shared_ptr< UndoSetLocationWithFixedPOICategoryInputRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onUndoSetLocationWithFixedPOICategoryInputRequest"));
   _navMiddleware.undoLastPoiEntrySelection();
}


void HMIModelNavigationService::onPropertyPOIInputInfoChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyPOIInputInfoChanged"));
   if (_getSetLocationWithFixedPOICategoryInputAct != 0)
   {
      _navMiddleware.setLocationWithCurrentPoiInput();
      sendSetLocationWithFixedPOICategoryInputResponse(_getSetLocationWithFixedPOICategoryInputAct);
      _getSetLocationWithFixedPOICategoryInputAct = 0;
   }
}


void HMIModelNavigationService::applyNonPremiumTMCStatus(bool nonPremiumTMCStatus)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::applyNonPremiumTMCStatus(%s)", nonPremiumTMCStatus ? "true" : "false"));
   navmiddleware::settings::TrafficSettings trafficSettings = _navMiddleware.getTrafficSettings();

   if (trafficSettings.isNonPremiumTrafficEnabled() != nonPremiumTMCStatus)
   {
      trafficSettings.setNonPremiumTraffic(nonPremiumTMCStatus);
      _navMiddleware.updateTrafficSettings(trafficSettings);
      _navMiddleware.applySettings();
   }
}


void HMIModelNavigationService::onActivateNonPremiumTMCRequest(const boost::shared_ptr<org::bosch::cm::navigation::NavigationService::ActivateNonPremiumTMCRequest>& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onActivateNonPremiumTMCRequest"));
   applyNonPremiumTMCStatus(true);
   sendActivateNonPremiumTMCResponse();
}


void HMIModelNavigationService::onDeactivateNonPremiumTMCRequest(const boost::shared_ptr<org::bosch::cm::navigation::NavigationService::DeactivateNonPremiumTMCRequest>& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onDeactivateNonPremiumTMCRequest"));
   applyNonPremiumTMCStatus(false);
   sendDeactivateNonPremiumTMCResponse();
}


void HMIModelNavigationService::onSetLocationWithFixedPOICategoryInputRequest(const ::boost::shared_ptr< SetLocationWithFixedPOICategoryInputRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onSetLocationWithFixedPOICategoryInputRequest"));
   const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getPoiEntries();
   const ::std::vector<navmiddleware::LocationInfos::LocationInfo>& locationInfoVector = locationInfos.getLocationInfos();
   if (request->getIndex() < locationInfoVector.size())
   {
      _getSetLocationWithFixedPOICategoryInputAct = request->getAct();
      _navMiddleware.selectPoiEntry(request->getIndex());
   }
}


void HMIModelNavigationService::onShowLocationSelectionAndGetAddressDetailsRequest(const boost::shared_ptr<org::bosch::cm::navigation::NavigationService::ShowLocationSelectionAndGetAddressDetailsRequest>& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onShowLocationSelectionAndGetAddressDetailsRequest"));
   _showLocationSelectionAndGetAddressAct = request->getAct();

   /* @ToDo: Implementation to be added if required
   POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchOutReqMsg)(NAVIGATION_CONTEXT_NAV_ADD_ADDRESS_TO_HOME_MENU, 0, APPID_APPHMI_NAVIGATION))); */
}


void HMIModelNavigationService::onShowRouteCalculationToDestinationRequest(const ::boost::shared_ptr< ShowRouteCalculationToDestinationRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onShowRouteCalculationToDestinationRequest()"));

   /* @ToDo: Implementation to be added if required
   POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchOutReqMsg)(NAVIGATION_CONTEXT_ROUTE_INFO, 0, APPID_APPHMI_NAVIGATION))); */
}


void HMIModelNavigationService::onShowAndSetLocationWithDetailsRequest(const boost::shared_ptr<org::bosch::cm::navigation::NavigationService::ShowAndSetLocationWithDetailsRequest>& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onShowAndSetLocationWithDetailsRequest"));
   setLocationDetails(request->getLocation());

   _getShowAndSetLocationWithDetailsAct = request->getAct();
   _hmiModelNavigationServiceExtended.showLocationDetails();
}


void HMIModelNavigationService::onPropertyGuidanceDestinationReached()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyGuidanceDestinationReached()"));

   const ::navmiddleware::DestinationReachedInfo& destinationReachedInfo = _navMiddleware.getDestinationReachedInfo();
   DestinationReachedStatus destinationReachedStatus;
   switch (destinationReachedInfo.getReachedDestinationType())
   {
      case REACHED_DESTINATION_TYPE__WAYPOINT:
         destinationReachedStatus = DestinationReachedStatus__WAYPOINT;
         break;
      case REACHED_DESTINATION_TYPE__FINAL_DESTINATION:
         destinationReachedStatus = DestinationReachedStatus__FINAL_DESTINATION;
         break;
      default:
         ETG_TRACE_ERR(("HMIModelNavigationService::reachedDesinationType(),No DestinationReachedStatus mapping available, default value set to UNKNOWN."));
         destinationReachedStatus = DestinationReachedStatus__UNKNOWN;
         break;
   }
   setDestinationReachedStatus(destinationReachedStatus);
}


bool HMIModelNavigationService::onPropertyUpdateMapCameraInfosChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationService::onPropertyUpdateMapCameraInfosChanged"));
   navmiddleware::MapCameraInfos mapCameraInfos = _navMiddleware.getMapCameraInfos();
   NAVSERVICE::MapCameraData mapCameraData;

   ETG_TRACE_USR4(("scaleBarWidth = %d", mapCameraInfos.getScaleBarWidthInPixel()));
   mapCameraData.setScaleBarWidth(mapCameraInfos.getScaleBarWidthInPixel());
   ETG_TRACE_USR4(("mapScale = %s", mapCameraInfos.getFormattedMapScale().c_str()));
   mapCameraData.setMapScale(mapCameraInfos.getFormattedMapScale().c_str());
   ETG_TRACE_USR4(("cameraHeading = %f", mapCameraInfos.getCameraHeading()));
   mapCameraData.setCameraHeading(mapCameraInfos.getCameraHeading());

   setMapCameraData(mapCameraData);
   sendMapCameraDataUpdate();
   return true;
}


#endif // HALL_NAVIGATION_SERVICE
#endif // HALL_TO_MDW_COM
