/**************************************************************************************
* @file         : HMIModelNavigationSDSService.cpp
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "HMIModelNavigationSDSService.h"
#include "NavigationServiceUtils.h"
#include "AppBase/ScreenBrokerClient/ScreenBrokerClient.h"
#include "ProjectBaseTypes.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#include "trcGenProj/Header/HMIModelNavigationSDSService.cpp.trc.h"
#endif

#if defined(HALL_TO_MDW_COM)
#if defined(HALL_NAVIGATION_SDS_SERVICE)

using namespace ::org::bosch::cm::navigation::NavigationSDSService;
using namespace navmiddleware;

static const unsigned int E_GLO_HANDOVER_DATA_NONE = 0;
static const unsigned int E_GLO_HANDOVER_DATA_SINGLE = 1;
static const unsigned int E_GLO_HANDOVER_DATA_LIST = 2;
static const unsigned int E_GLO_HANDOVER_DATA_MULTIPLE = 5;

HMIModelNavigationSDSService::HMIModelNavigationSDSService(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
   , NavigationSDSServiceStub("navigationSDSPort")
   , _naviSdsHandOverSreenId(SDS_HANDOVER_NAVI_INACTIVE_SCREEN)
   , _currentRGFlow(SDS_HANDOVER_NAVI_INACTIVE_SCREEN)
   , _getGlobalDataForSdsAct(0)
   , _getIdListOfCountryAndSubCountryAct(0)
   , _getCityHasStreetsAct(0)
   , _getBaseAddressFTSAct(0)
   , _getBaseAddressPOIAct(0)
   , _getBaseAddressStartRouteAct(0)
   , _getFTSConfirmPOIorAddressAct(0)
   , _sdsCheckAddressRequestAct(0)
   , _selectSdsRefinementAct(0)
   , _getHouseNumberPatternAct(0)
   , _getSdsGetRefinementListAct(0)
   , _getAddressFormatForAvailableCountriesAct(0)
   , _getRegisterDatasetUserAct(0)
   , _getDeregisterDatasetUserAct(0)
{
}


HMIModelNavigationSDSService::~HMIModelNavigationSDSService()
{
}


void HMIModelNavigationSDSService::initialize()
{
   _navMiddleware.registerLocationPropertyUpdateCallback(*this);
   _navMiddleware.registerPositionPropertyUpdateCallback(*this);
   _navMiddleware.registerMapPropertyUpdateCallback(*this);
   _navMiddleware.registerNavDataUpdatePropertyUpdateCallback(*this);
}


void HMIModelNavigationSDSService::deinitialize()
{
   _navMiddleware.unregisterLocationPropertyUpdateCallback(*this);
   _navMiddleware.unregisterPositionPropertyUpdateCallback(*this);
   _navMiddleware.unregisterMapPropertyUpdateCallback(*this);
   _navMiddleware.unregisterNavDataUpdatePropertyUpdateCallback(*this);
}


void HMIModelNavigationSDSService::onGetGlobalDataForSdsRequest(const ::boost::shared_ptr< GetGlobalDataForSdsRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onGetGlobalDataForSdsRequest()"));
   _getGlobalDataForSdsAct = request->getAct();
   _navMiddleware.requestGlobalSdsData();
}


void HMIModelNavigationSDSService::onPropertyUpdateGlobalDataForSdsChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onPropertyUpdateGlobalDataForSdsChanged()"));
   if (_getGlobalDataForSdsAct != 0)
   {
      const SdsGlobalDataInfo& sdsGlobalDataInfo = _navMiddleware.getSdsGlobalDataInfo();
      std::string deviceName = sdsGlobalDataInfo.getDeviceName();
      uint32_t datasetId = sdsGlobalDataInfo.getDatasetId();
      std::string persistentDatasetId = sdsGlobalDataInfo.getPersistentDatasetId();
      std::vector<SdsProducts> sdsProductsList;
      std::vector<UpdateRegionSpecifierRelations> updateRegionSpecifierRelationsList;
      ETG_TRACE_USR4(("MDW return to Navi %s}", sdsGlobalDataInfo.toString().c_str()));

      const std::vector<SdsProductInfo>& sdsProductInfos = sdsGlobalDataInfo.getSdsProductInfos();
      sdsProductsList.reserve(sdsProductInfos.size());
      std::vector< SdsProductInfo>:: const_iterator iterProductInfo = sdsProductInfos.begin();
      for (; iterProductInfo != sdsProductInfos.end(); ++iterProductInfo)
      {
         SdsProducts sdsProducts;
         sdsProducts.setSupplierId(iterProductInfo->getSupplierId());
         sdsProducts.setProductId(iterProductInfo->getProductId());
         constructSDSUpdateRegionsList(iterProductInfo->getSdsUpdateRegionInfos(), sdsProducts.getSdsUpdateRegionsListMutable());
         constructSDSLanguages(iterProductInfo->getSdsLanguageInfos(), sdsProducts.getSdsLanguagesListMutable());
         sdsProductsList.push_back(sdsProducts);
      }

      const std::vector<UpdateRegionSpecifierRelationInfo>& updateRegionSpecifierRelationInfo = sdsGlobalDataInfo.getUpdateRegionSpecifierRelationInfo();
      updateRegionSpecifierRelationsList.reserve(updateRegionSpecifierRelationInfo.size());
      std::vector<UpdateRegionSpecifierRelationInfo>::const_iterator iterRegionSpecifier = updateRegionSpecifierRelationInfo.begin();
      for (; iterRegionSpecifier != updateRegionSpecifierRelationInfo.end(); ++iterRegionSpecifier)
      {
         UpdateRegionSpecifierRelations updateRegionSpecifierRelations;
         updateRegionSpecifierRelations.setUpdateRegionSpecifier(iterRegionSpecifier->getUpdateRegionSpecifier());
         updateRegionSpecifierRelations.setSupplierId(iterRegionSpecifier->getSupplierId());
         updateRegionSpecifierRelations.setProductId(iterRegionSpecifier->getProductId());
         updateRegionSpecifierRelations.setUpdateRegionId(iterRegionSpecifier->getUpdateRegionId());
         updateRegionSpecifierRelationsList.push_back(updateRegionSpecifierRelations);
      }

      sendGetGlobalDataForSdsResponse(deviceName, datasetId, persistentDatasetId, sdsProductsList, updateRegionSpecifierRelationsList, _getGlobalDataForSdsAct);
      _getGlobalDataForSdsAct = 0;
   }
}


void HMIModelNavigationSDSService::constructSDSUpdateRegionsList(const ::std::vector<SdsUpdateRegionInfo>& sdsUpdateRegionInfos, std::vector<SdsUpdateRegions>& sdsUpdateRegionNeedToConstruct)
{
   std::vector<SdsUpdateRegionInfo>:: const_iterator iterRegion = sdsUpdateRegionInfos.begin();
   sdsUpdateRegionNeedToConstruct.clear();
   sdsUpdateRegionNeedToConstruct.reserve(sdsUpdateRegionInfos.size());
   for (; iterRegion != sdsUpdateRegionInfos.end(); ++iterRegion)
   {
      SdsUpdateRegions sdsUpdateRegions;
      sdsUpdateRegions.setUpdateRegionId(iterRegion->getUpdateRegionId());
      constructSDSBuildingBlocks(iterRegion->getSdsBuildingBlockInfos(), sdsUpdateRegions.getSdsBuildingBlocksListMutable());
      constructSDSRegionMetadata(iterRegion->getSdsRegionMetaDataInfos(), sdsUpdateRegions.getSdsRegionMetadataListMutable());
      sdsUpdateRegionNeedToConstruct.push_back(sdsUpdateRegions);
   }
}


void HMIModelNavigationSDSService::constructSDSLanguages(const ::std::vector<SdsLanguageInfo>& SdsLanguageInfo, std::vector<SdsLanguages>& sdsLanguagesList)
{
   std::vector<navmiddleware::SdsLanguageInfo>:: const_iterator iterLanguagesList = SdsLanguageInfo.begin();
   sdsLanguagesList.clear();
   sdsLanguagesList.reserve(SdsLanguageInfo.size());
   for (; iterLanguagesList != SdsLanguageInfo.end(); ++iterLanguagesList)
   {
      SdsLanguages sdsLanguages;
      sdsLanguages.setLanguageCode(iterLanguagesList->getLanguageCode());
      sdsLanguages.setIsoLanguageCode(iterLanguagesList->getIsoLanguageCode());
      sdsLanguages.setIsoCountryCode(iterLanguagesList->getIsoCountryCode());
      sdsLanguages.setIsoScriptCode(iterLanguagesList->getIsoScriptCode());
      sdsLanguages.setIsTransliterationOf(iterLanguagesList->getIsTransliterationOf());
      sdsLanguages.setIsDiacriticTransliterationOf(iterLanguagesList->getIsDiacriticTransliterationOf());

      LanguageEntry languageEntry;
      std::string iSOLanguageCode = iterLanguagesList->getLanguageEntryInfo().getIsoLanguageCode();
      std::string iSOCountryCode = iterLanguagesList->getLanguageEntryInfo().getIsoCountryCode();
      std::string transcriptCode = iterLanguagesList->getLanguageEntryInfo().getIsoScriptCode();
      languageEntry.setISOLanguageCode(iSOLanguageCode);
      languageEntry.setISOCountryCode(iSOCountryCode);
      languageEntry.setTranscriptCode(transcriptCode);
      sdsLanguages.setLanguageEntry(languageEntry);
      sdsLanguagesList.push_back(sdsLanguages);
   }
}


void HMIModelNavigationSDSService::constructSDSRegionMetadata(const std::vector<SdsRegionMetaDataInfo>& sdsRegionMetaDataInfo, std::vector<SdsRegionMetadata>& sdsRegionMetadataList)
{
   sdsRegionMetadataList.clear();
   sdsRegionMetadataList.reserve(sdsRegionMetaDataInfo.size());
   std::vector<SdsRegionMetaDataInfo>:: const_iterator iterRegionMeta = sdsRegionMetaDataInfo.begin();
   for (; iterRegionMeta != sdsRegionMetaDataInfo.end(); ++iterRegionMeta)
   {
      SdsRegionMetadata sdsRegionMetadata;
      sdsRegionMetadata.setUpdateRegionId(iterRegionMeta->getUpdateRegionId());
      sdsRegionMetadata.setIsoCountryCodeText(iterRegionMeta->getIsoCountryCode());
      sdsRegionMetadata.setIsoSubCountryCodeText(iterRegionMeta->getIsoSubCountryCode());
      sdsRegionMetadataList.push_back(sdsRegionMetadata);
   }
}


void HMIModelNavigationSDSService::constructSDSBuildingBlocks(const ::std::vector<SdsBuildingBlockInfo>& sdsBuildingBlockInfos, std::vector<SdsBuildingBlocks>& sdsBuildingBlocksList)
{
   sdsBuildingBlocksList.clear();
   sdsBuildingBlocksList.reserve(sdsBuildingBlockInfos.size());
   std::vector<SdsBuildingBlockInfo>:: const_iterator iterBuildingBlock = sdsBuildingBlockInfos.begin();
   for (; iterBuildingBlock != sdsBuildingBlockInfos.end(); ++iterBuildingBlock)
   {
      SdsBuildingBlocks sdsBuildingBlocks;
      sdsBuildingBlocks.setBuildingBlockId(iterBuildingBlock->getBuildingBlockId());
      SdsBuildingBlockType blockType = iterBuildingBlock->getSdsBuildingBlockType();
      sdsBuildingBlocks.setNdsBuildingBlockType((blockType == SDS_BUILDING_BLOCK_TYPE__SPEECH) ? NDSBuildingBlockType__SPEECH : NDSBuildingBlockType__EXTENSION);
      sdsBuildingBlocks.setUri(iterBuildingBlock->getURI());
      sdsBuildingBlocks.setEncryptionKeyId(iterBuildingBlock->getEncryptionId());
      sdsBuildingBlocks.setTypeNdsDbSupplierId(iterBuildingBlock->getTypeNdsDbSupplierID());
      sdsBuildingBlocks.setTypeExtId(iterBuildingBlock->getTypeExtId());

      SpeechMetadata speechMetadata;
      speechMetadata.setHasPhonetic(iterBuildingBlock->getSdsSpeechMetaDataInfo().hasPhonetic());
      speechMetadata.setHasPrerecorded(iterBuildingBlock->getSdsSpeechMetaDataInfo().hasPreRecorded());

      PhoneticTraMetadata phoneticTraMetadata;
      SdsPhoneticTranscriptionFormat phoneticTranscriptionFormat = iterBuildingBlock->getSdsSpeechMetaDataInfo().getSdsPhoneticTraMetaDataInfo().getSdsPhoneticTranscriptionFormat();
      phoneticTraMetadata.setPhoneticTranscriptionFormat(convertPhoneticTranscriptionFormat(phoneticTranscriptionFormat));

      std::vector<LanguageVersion> phonetic_LanguageVersionList;
      const std::vector<SdsLanguageVersionInfo>& sdsLanguageVersionInfos = iterBuildingBlock->getSdsSpeechMetaDataInfo().getSdsPhoneticTraMetaDataInfo().getSdsLanguageVersionInfos();
      phonetic_LanguageVersionList.reserve(sdsLanguageVersionInfos.size());
      std::vector<SdsLanguageVersionInfo>:: const_iterator iteranguageVersion = sdsLanguageVersionInfos.begin();
      for (; iteranguageVersion != sdsLanguageVersionInfos.end(); ++iteranguageVersion)
      {
         LanguageVersion languageVersion;
         languageVersion.setLanguageCode(iteranguageVersion->getLanguageCode());
         languageVersion.setSpecificationVersion(iteranguageVersion->getSpecificationVersion());
         phonetic_LanguageVersionList.push_back(languageVersion);
      }
      phoneticTraMetadata.setLanguageVersionList(phonetic_LanguageVersionList);
      speechMetadata.setPhoneticTraMetadata(phoneticTraMetadata);

      PrerecordedVMetadata prerecordedVMetadata;
      SdsPreRecordedVoiceFormat sdsPreRecordedVoiceFormat = iterBuildingBlock->getSdsSpeechMetaDataInfo().getSdsPreRecordedVMetaDataInfo().getSdsPreRecordedVoiceFormat();
      prerecordedVMetadata.setPrerecordedVoiceFormat((sdsPreRecordedVoiceFormat == SDS_PRE_RECORDED_VOICE_FORMAT__ADPCM) ? PrerecordedVoiceFormat__ADPCM : PrerecordedVoiceFormat__MP3);
      std::vector<LanguageVersion> preRecordedLanguageVersionList;
      const std::vector<SdsLanguageVersionInfo>& preRecordedLanguageVersionInfos = iterBuildingBlock->getSdsSpeechMetaDataInfo().getSdsPreRecordedVMetaDataInfo().getSdsLanguageVersionInfos();
      preRecordedLanguageVersionList.reserve(preRecordedLanguageVersionInfos.size());
      std::vector<SdsLanguageVersionInfo>:: const_iterator iterPreRecordedlanguageVersion = preRecordedLanguageVersionInfos.begin();
      for (; iterPreRecordedlanguageVersion != preRecordedLanguageVersionInfos.end(); ++iterPreRecordedlanguageVersion)
      {
         LanguageVersion languageVersion;
         languageVersion.setLanguageCode(iterPreRecordedlanguageVersion->getLanguageCode());
         languageVersion.setSpecificationVersion(iterPreRecordedlanguageVersion->getSpecificationVersion());
         preRecordedLanguageVersionList.push_back(languageVersion);
      }
      prerecordedVMetadata.setLanguageVersionList(preRecordedLanguageVersionList);
      speechMetadata.setPrerecordedVMetadata(prerecordedVMetadata);

      sdsBuildingBlocks.setSpeechMetadata(speechMetadata);
      sdsBuildingBlocksList.push_back(sdsBuildingBlocks);
   }
}


void HMIModelNavigationSDSService::onGetIdListOfCountryAndSubCountryRequest(const ::boost::shared_ptr< GetIdListOfCountryAndSubCountryRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onGetIdListOfCountryAndSubCountryRequest()"));
   _getIdListOfCountryAndSubCountryAct = request->getAct();
   _navMiddleware.requestIdListOfCountryAndSubCountry();
}


void HMIModelNavigationSDSService::onPropertyUpdateCountryAndSubCountryChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onPropertyUpdateCountryAndSubCountryChanged()"));
   if (_getIdListOfCountryAndSubCountryAct)
   {
      const SdsCountryAndSubCountryInfos& sdsCountryAndSubCountryInfos = _navMiddleware.getSdsCountryAndSubCountryInfos();
      ETG_TRACE_USR4(("onPropertyUpdateCountryAndSubCountryChanged %s", sdsCountryAndSubCountryInfos.toString().c_str()));
      std::vector<GlobalNamedObjectAndRegionId> countryIdList;
      const std::vector<SdsCountryInfo>& sdsCountryInfos = sdsCountryAndSubCountryInfos.getSdsCountryInfos();
      std::vector<SdsCountryInfo>::const_iterator iterCountryInfos = sdsCountryInfos.begin();
      countryIdList.reserve(sdsCountryInfos.size());
      for (; iterCountryInfos != sdsCountryInfos.end(); ++iterCountryInfos)
      {
         GlobalUpdateRegionId globalUpdateRegionId;
         constructSDSGlobalUpdateRegionId(iterCountryInfos->getSdsGlobalNamedObjectIdWithPoiAndXrefIdInfo().getSdsGlobalNamedObjectIdInfo().getSdsGlobalUpdateRegionInfo(), globalUpdateRegionId);
         GlobalNamedObjectAndRegionId globalNamedObjectAndRegionId;
         globalNamedObjectAndRegionId.setGlobalUpdateRegionId(globalUpdateRegionId);
         globalNamedObjectAndRegionId.setNamedObjectId(iterCountryInfos->getSdsGlobalNamedObjectIdWithPoiAndXrefIdInfo().getSdsGlobalNamedObjectIdInfo().getNamedObjectId());
         globalNamedObjectAndRegionId.setPoiRegionId(iterCountryInfos->getSdsGlobalNamedObjectIdWithPoiAndXrefIdInfo().getPoiRegionId());
         globalNamedObjectAndRegionId.setGlobalXrefID(iterCountryInfos->getSdsGlobalNamedObjectIdWithPoiAndXrefIdInfo().getGlobalXrefId());
         countryIdList.push_back(globalNamedObjectAndRegionId);
      }
      std::vector<SubCountryIds> subCountryIdList;
      const std::vector<SdsSubCountryInfos>& sdsSubCountryInfos = sdsCountryAndSubCountryInfos.getSdsSubCountryInfos();
      std::vector<SdsSubCountryInfos>::const_iterator iterSubCountry = sdsSubCountryInfos.begin();
      subCountryIdList.reserve(sdsSubCountryInfos.size());
      for (; iterSubCountry != sdsSubCountryInfos.end(); ++iterSubCountry)
      {
         GlobalUpdateRegionId globalUpdateRegionId;
         constructSDSGlobalUpdateRegionId(iterSubCountry->getSdsCountryInfo().getSdsGlobalUpdateRegionInfo(), globalUpdateRegionId);
         GlobalNamedObjectId countryId;
         countryId.setGlobalUpdateRegionId(globalUpdateRegionId);
         countryId.setNamedObjectId(iterSubCountry->getSdsCountryInfo().getNamedObjectId());
         SubCountryIds subCountryId;
         subCountryId.setCountryId(countryId);
         // this is for subCountryId.setSubCountryIdList()
         constructSDSSubCountryIdList(iterSubCountry->getSdsSubCountryInfos(), subCountryId.getSubCountryIdListMutable());
         subCountryIdList.push_back(subCountryId);
      }
      sendGetIdListOfCountryAndSubCountryResponse(countryIdList, subCountryIdList, _getIdListOfCountryAndSubCountryAct);
      _getIdListOfCountryAndSubCountryAct = 0;
   }
}


void HMIModelNavigationSDSService::constructSDSGlobalUpdateRegionId(const SdsGlobalUpdateRegionInfo& sdsGlobalUpdateRegionInfo, GlobalUpdateRegionId& globalUpdateRegionId)
{
   globalUpdateRegionId.setProductId(sdsGlobalUpdateRegionInfo.getProductId());
   globalUpdateRegionId.setSupplierId(sdsGlobalUpdateRegionInfo.getSupplierId());
   globalUpdateRegionId.setUpdateRegionId(sdsGlobalUpdateRegionInfo.getUpdateRegionId());
}


void HMIModelNavigationSDSService::constructSDSSubCountryIdList(const std::vector<SdsGlobalNamedObjectIdWithPoiAndXrefIdInfo>& sdsGlobalNamedObjectIdInfos, std::vector<GlobalNamedObjectAndRegionId>& subCountryIds)
{
   std::vector<SdsGlobalNamedObjectIdWithPoiAndXrefIdInfo>::const_iterator iterSdsSubCountry = sdsGlobalNamedObjectIdInfos.begin();
   subCountryIds.clear();
   subCountryIds.reserve(sdsGlobalNamedObjectIdInfos.size());
   for (; iterSdsSubCountry != sdsGlobalNamedObjectIdInfos.end(); ++iterSdsSubCountry)
   {
      GlobalUpdateRegionId subGlobalUpdateRegionId;
      constructSDSGlobalUpdateRegionId(iterSdsSubCountry->getSdsGlobalNamedObjectIdInfo().getSdsGlobalUpdateRegionInfo(), subGlobalUpdateRegionId);
      GlobalNamedObjectAndRegionId globalNamedObjectAndRegionId;
      globalNamedObjectAndRegionId.setGlobalUpdateRegionId(subGlobalUpdateRegionId);
      globalNamedObjectAndRegionId.setNamedObjectId(iterSdsSubCountry->getSdsGlobalNamedObjectIdInfo().getNamedObjectId());
      globalNamedObjectAndRegionId.setPoiRegionId(iterSdsSubCountry->getPoiRegionId());
      globalNamedObjectAndRegionId.setGlobalXrefID(iterSdsSubCountry->getGlobalXrefId());
      subCountryIds.push_back(globalNamedObjectAndRegionId);
   }
}


void HMIModelNavigationSDSService::onGetNamesForNDSIdsRequest(const ::boost::shared_ptr< GetNamesForNDSIdsRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onGetNamesForNDSIdsRequest()"));
   _getNamesForNDSIdsAct.push(request->getAct());
   std::vector<navmiddleware::SdsGlobalNamedObjectIdInfo> sdsGlobalNamedObjectIdInfos;
   constructMWGlobalNamedObjectIdList(request->getGlobalNamedObjectIdList(), sdsGlobalNamedObjectIdInfos);
   std::vector<navmiddleware::LanguageEntryInfo> languageEntryInfos;
   constructMWLanguageEntryList(request->getLanguageEntryList(), languageEntryInfos);
   _navMiddleware.requestNamesForNdsIds(sdsGlobalNamedObjectIdInfos, languageEntryInfos, convertNDSFeatureType(request->getNdsFeatureType()), request->getStopOnFirstMatch());
}


void HMIModelNavigationSDSService::constructSDSNdsIds(const std::vector<SdsNdsIdStringInfo>& sdsNdsIdStringInfos, std::vector<NamedObjectIdStrings>& namedObjectIdStringsList)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::constructSDSNdsIds(), sdsNdsIdStringInfos size %d", sdsNdsIdStringInfos.size()));
   namedObjectIdStringsList.clear();
   namedObjectIdStringsList.reserve(sdsNdsIdStringInfos.size());
   std::vector<SdsNdsIdStringInfo>::const_iterator iter = sdsNdsIdStringInfos.begin();

   for (; iter != sdsNdsIdStringInfos.end(); ++iter)
   {
      ETG_TRACE_USR4(("HMIModelNavigationSDSService::constructSDSNdsIds(), sdsNdsIdStringInfo %s", iter->toString().c_str()));

      NamedObjectIdStrings namedObjectIdStrings;
      //GlobalNamedObjectId
      namedObjectIdStrings.getGlobalNamedObjectIdMutable().getGlobalUpdateRegionIdMutable().setUpdateRegionId(iter->getSdsGlobalNamedObjectIdInfo().getSdsGlobalUpdateRegionInfo().getUpdateRegionId());
      namedObjectIdStrings.getGlobalNamedObjectIdMutable().getGlobalUpdateRegionIdMutable().setProductId(iter->getSdsGlobalNamedObjectIdInfo().getSdsGlobalUpdateRegionInfo().getProductId());
      namedObjectIdStrings.getGlobalNamedObjectIdMutable().getGlobalUpdateRegionIdMutable().setSupplierId(iter->getSdsGlobalNamedObjectIdInfo().getSdsGlobalUpdateRegionInfo().getSupplierId());
      namedObjectIdStrings.getGlobalNamedObjectIdMutable().setNamedObjectId(iter->getSdsGlobalNamedObjectIdInfo().getNamedObjectId());

      //LanguageEntry
      namedObjectIdStrings.getLanguageEntryMutable().setISOLanguageCode(iter->getLanguageEntryInfo().getIsoLanguageCode());
      namedObjectIdStrings.getLanguageEntryMutable().setISOCountryCode(iter->getLanguageEntryInfo().getIsoCountryCode());
      namedObjectIdStrings.getLanguageEntryMutable().setTranscriptCode(iter->getLanguageEntryInfo().getIsoScriptCode());
      ETG_TRACE_USR4(("HMIModelNavigationSDSService::constructSDSNdsIds(), Language ISO code %s", iter->getLanguageEntryInfo().getIsoLanguageCode().c_str()));

      //AddressElementType
      ETG_TRACE_USR4(("HMIModelNavigationSDSService::constructSDSNdsIds(), Address element %d", convertAddressElementType(iter->getSDSAdressElementType().getValue())));
      namedObjectIdStrings.setAddressElementType(convertAddressElementType(iter->getSDSAdressElementType().getValue()));

      //SdsNamedStringList
      const std::vector<std::string>& namedStringsList = iter->getSdsNamedString();
      namedObjectIdStrings.getSdsNamedStringsListMutable().clear();
      namedObjectIdStrings.getSdsNamedStringsListMutable().reserve(namedStringsList.size());
      for (std::vector< std::string>:: const_iterator iterString = namedStringsList.begin(); iterString != namedStringsList.end(); ++iterString)
      {
         ETG_TRACE_USR4(("HMIModelNavigationSDSService::constructSDSNdsIds(), SDSNamedString %s", (*iterString).c_str()));
         namedObjectIdStrings.getSdsNamedStringsListMutable().push_back(*iterString);
      }
      namedObjectIdStringsList.push_back(namedObjectIdStrings);
   }
}


void HMIModelNavigationSDSService::onPropertyUpdateNamesForNdsIdsChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onPropertyUpdateNamesForNdsIdsChanged()"));
   if (!_getNamesForNDSIdsAct.empty())
   {
      std::vector<NamedObjectIdStrings> namedObjectIdStringsList;
      const SdsNdsIdStringInfos& sdsNdsIdStringInfos = _navMiddleware.getSdsNdsIdNamedStringInfos();
      const ::std::vector<SdsNdsIdStringInfo>& sdsNdsIdStringInfo = sdsNdsIdStringInfos.getSdsNdsIdStringInfos();
      constructSDSNdsIds(sdsNdsIdStringInfo, namedObjectIdStringsList);
      sendGetNamesForNDSIdsResponse(namedObjectIdStringsList, _getNamesForNDSIdsAct.pop());
   }
}


void HMIModelNavigationSDSService::constructMWGlobalNamedObjectIdList(const std::vector<org::bosch::cm::navigation::NavigationSDSService::GlobalNamedObjectId>& globalNamedObjectIds, std::vector<navmiddleware::SdsGlobalNamedObjectIdInfo>& sdsGlobalNamedObjectIdInfos)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::constructMWGlobalNamedObjectIdList()"));
   sdsGlobalNamedObjectIdInfos.reserve(globalNamedObjectIds.size());
   std::vector<GlobalNamedObjectId>::const_iterator iterNameObject = globalNamedObjectIds.begin();
   for (; iterNameObject != globalNamedObjectIds.end(); ++iterNameObject)
   {
      SdsGlobalNamedObjectIdInfo sdsGlobalNamedObjectIdInfo;
      SdsGlobalUpdateRegionInfo sdsGlobalUpdateRegionInfo;

      GlobalUpdateRegionId globalUpdateRegionId = iterNameObject->getGlobalUpdateRegionId();
      uint16_t updateRegionId = iterNameObject->getGlobalUpdateRegionId().getUpdateRegionId();
      uint32_t productId = iterNameObject->getGlobalUpdateRegionId().getProductId();
      uint8_t supplierId = iterNameObject->getGlobalUpdateRegionId().getSupplierId();
      sdsGlobalUpdateRegionInfo.setUpdateRegionId(updateRegionId);
      sdsGlobalUpdateRegionInfo.setProductId(productId);
      sdsGlobalUpdateRegionInfo.setSupplierId(supplierId);
      ETG_TRACE_USR4(("SDS request to Navi GlobalUpdateRegionId{updateRegionId %d, productId %d, supplierId %d}", updateRegionId, productId, supplierId));
      sdsGlobalNamedObjectIdInfo.setSdsGlobalUpdateRegionInfo(sdsGlobalUpdateRegionInfo);
      sdsGlobalNamedObjectIdInfo.setNamedObjectId(iterNameObject->getNamedObjectId());
      ETG_TRACE_USR4(("SDS request to Navi HMI GlobalNamedObjectId{namedObjectId %d,}", iterNameObject->getNamedObjectId()));
      sdsGlobalNamedObjectIdInfos.push_back(sdsGlobalNamedObjectIdInfo);
   }
}


void HMIModelNavigationSDSService::constructMWLanguageEntryList(const std::vector<org::bosch::cm::navigation::NavigationSDSService::LanguageEntry>& languageEntryList, std::vector<navmiddleware::LanguageEntryInfo>& languageEntryInfos)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::constructMWLanguageEntryList()"));
   /* get LanguageEntryList */
   std::stringstream traceOut;
   languageEntryInfos.reserve(languageEntryList.size());
   std::vector<LanguageEntry>::const_iterator iterLanguageEntry = languageEntryList.begin();
   for (; iterLanguageEntry != languageEntryList.end(); ++iterLanguageEntry)
   {
      LanguageEntryInfo languageEntryInfo;
      languageEntryInfo.setIsoLanguageCode(iterLanguageEntry->getISOLanguageCode());
      languageEntryInfo.setIsoCountryCode(iterLanguageEntry->getISOCountryCode());
      languageEntryInfo.setIsoScriptCode(iterLanguageEntry->getTranscriptCode());

      // use a stream string helper because of missing target support for printing out two separate strings within the ETG trace
      traceOut << "{ language " << LanguageCode() << ", "
               << iterLanguageEntry->getISOLanguageCode() << ", "
               << iterLanguageEntry->getISOCountryCode() << ", "
               << iterLanguageEntry->getTranscriptCode() << " }" << std::endl;
      ETG_TRACE_USR4(("HMIModelNavigationSDSService::constructMWLanguageEntryList(), Language details: %s", traceOut.str().c_str()));

      languageEntryInfos.push_back(languageEntryInfo);
   }
}


void HMIModelNavigationSDSService::onGetPhonemesForNDSIdsRequest(const ::boost::shared_ptr< GetPhonemesForNDSIdsRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onGetPhonemesForNDSIdsRequest()"));
   _getPhonemesForNDSIdsAct.push(request->getAct());
   std::vector<navmiddleware::SdsGlobalNamedObjectIdInfo> sdsGlobalNamedObjectIdInfos;
   constructMWGlobalNamedObjectIdList(request->getGlobalNamedObjectIdList(), sdsGlobalNamedObjectIdInfos);
   std::vector<navmiddleware::LanguageEntryInfo> languageEntryInfos;
   constructMWLanguageEntryList(request->getLanguageEntryList(), languageEntryInfos);
   _navMiddleware.requestPhonemesForNdsIds(sdsGlobalNamedObjectIdInfos, languageEntryInfos, convertNDSFeatureType(request->getNdsFeatureType()), request->getStopOnFirstMatch());
}


void HMIModelNavigationSDSService::onPropertyUpdateLocationSDSPhonemesForNDSIdsChanged()
{
   if (!_getPhonemesForNDSIdsAct.empty())
   {
      ETG_TRACE_USR4(("HMIModelNavigationSDSService::onPropertyUpdateLocationSDSPhonemesForNDSIdsChanged()"));
      std::vector<NamedObjectIdStrings> namedObjectIdStringsList;
      const SdsNdsIdStringInfos& sdsNdsIdStringInfos = _navMiddleware.getSdsNdsIdPhonemeStringInfos();
      const ::std::vector<SdsNdsIdStringInfo>& sdsNdsIdStringInfo = sdsNdsIdStringInfos.getSdsNdsIdStringInfos();
      constructSDSNdsIds(sdsNdsIdStringInfo, namedObjectIdStringsList);
      sendGetPhonemesForNDSIdsResponse(namedObjectIdStringsList, _getPhonemesForNDSIdsAct.pop());
   }
}


void HMIModelNavigationSDSService::onGetCityHasStreetsRequest(const ::boost::shared_ptr< GetCityHasStreetsRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onGetCityHasStreetsRequest()"));
   _getCityHasStreetsAct = request->getAct();
   std::vector<GlobalNamedObjectId> globalNamedObjectIds = request->getGlobalNamedObjectIdList();
   std::vector<SdsGlobalNamedObjectIdInfo> sdsGlobalNamedObjectIdInfos;
   sdsGlobalNamedObjectIdInfos.reserve(globalNamedObjectIds.size());
   std::vector<GlobalNamedObjectId>::const_iterator iterNameObject = globalNamedObjectIds.begin();
   for (; iterNameObject != globalNamedObjectIds.end(); ++iterNameObject)
   {
      SdsGlobalNamedObjectIdInfo sdsGlobalNamedObjectIdInfo;
      SdsGlobalUpdateRegionInfo sdsGlobalUpdateRegionInfo;

      GlobalUpdateRegionId globalUpdateRegionId = iterNameObject->getGlobalUpdateRegionId();
      uint16_t updateRegionId = iterNameObject->getGlobalUpdateRegionId().getUpdateRegionId();
      uint32_t productId = iterNameObject->getGlobalUpdateRegionId().getProductId();
      uint8_t supplierId = iterNameObject->getGlobalUpdateRegionId().getSupplierId();
      sdsGlobalUpdateRegionInfo.setUpdateRegionId(updateRegionId);
      sdsGlobalUpdateRegionInfo.setProductId(productId);
      sdsGlobalUpdateRegionInfo.setSupplierId(supplierId);
      ETG_TRACE_USR4(("SDS request to Navi GlobalUpdateRegionId{updateRegionId %d, productId %d, supplierId %d}", updateRegionId, productId, supplierId));
      sdsGlobalNamedObjectIdInfo.setSdsGlobalUpdateRegionInfo(sdsGlobalUpdateRegionInfo);
      sdsGlobalNamedObjectIdInfo.setNamedObjectId(iterNameObject->getNamedObjectId());
      ETG_TRACE_USR4(("SDS request to Navi HMI GlobalNamedObjectId{namedObjectId %d,}", iterNameObject->getNamedObjectId()));
      sdsGlobalNamedObjectIdInfos.push_back(sdsGlobalNamedObjectIdInfo);
   }

   _navMiddleware.requestCityHasStreets(sdsGlobalNamedObjectIdInfos);
}


void HMIModelNavigationSDSService::onPropertyUpdateCityHasStreetsChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onPropertyUpdateCityHasStreetsChanged()"));
   if (_getCityHasStreetsAct)
   {
      std::vector< CityNamedObjectIdHasStreets > cityNamedObjectIdHasStreetsList;
      const std::vector<SdsCityNamedObjectHasStreet>&  sdsCityNamedObjectHasStreets = _navMiddleware.getSdsCityNamedObjectHasStreetInfos();
      std::vector<SdsCityNamedObjectHasStreet>::const_iterator iterCityNamed = sdsCityNamedObjectHasStreets.begin();
      cityNamedObjectIdHasStreetsList.reserve(sdsCityNamedObjectHasStreets.size());
      for (; iterCityNamed != sdsCityNamedObjectHasStreets.end(); ++iterCityNamed)
      {
         GlobalUpdateRegionId globalUpdateRegionId;
         constructSDSGlobalUpdateRegionId(iterCityNamed->getSdsGlobalNamedObjectIdInfo().getSdsGlobalUpdateRegionInfo(), globalUpdateRegionId);
         GlobalNamedObjectId globalNamedObjectId;
         globalNamedObjectId.setGlobalUpdateRegionId(globalUpdateRegionId);
         globalNamedObjectId.setNamedObjectId(iterCityNamed->getSdsGlobalNamedObjectIdInfo().getNamedObjectId());
         CityNamedObjectIdHasStreets cityNamedObjectIdHasStreet;
         cityNamedObjectIdHasStreet.setGlobalNamedObjectId(globalNamedObjectId);
         cityNamedObjectIdHasStreet.setHasStreets(iterCityNamed->getHasStreets());
         cityNamedObjectIdHasStreetsList.push_back(cityNamedObjectIdHasStreet);
      }
      sendGetCityHasStreetsResponse(cityNamedObjectIdHasStreetsList, _getCityHasStreetsAct);
      _getCityHasStreetsAct = 0;
   }
}


void HMIModelNavigationSDSService::onNdsDatasetReleasedRequest(const ::boost::shared_ptr< NdsDatasetReleasedRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onNdsDatasetReleasedRequest()"));
}


void HMIModelNavigationSDSService::onPropertyNavDataSetInfoChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onPropertyNavDataSetInfoChanged()"));
   NavDataSetInfo& ndsDatasetInfo = getNavDatasetInfoMutable();
   ndsDatasetInfo.setDatasetId(_navMiddleware.getNavDatasetInfo().getDatasetId());
   ndsDatasetInfo.setPersistentDatasetId(_navMiddleware.getNavDatasetInfo().getPersistentDatasetId());
   ndsDatasetInfo.getSharedDataAccessAttributesMutable().setSharedDataAccess(convertSharedDataAccess(_navMiddleware.getNavDatasetInfo().getSharedDataAccessAttributes().m_sharedDataAccess));
   ndsDatasetInfo.getSharedDataAccessAttributesMutable().setJournalingMode(convertJournalingMode(_navMiddleware.getNavDatasetInfo().getSharedDataAccessAttributes().m_journalingMode));
   ndsDatasetInfo.getSharedDataAccessAttributesMutable().setLockType(convertLockType(_navMiddleware.getNavDatasetInfo().getSharedDataAccessAttributes().m_lockType));
   sendNavDatasetInfoUpdate();
}


void HMIModelNavigationSDSService::onPropertyNavDataUserRegChanged()
{
   //ETG_TRACE_USR4(("HMIModelNavigationSDSService::onPropertyNavDataUserRegChanged(%d)", _navMiddleware.getDatasetUserRegisterStatus()));
   RegistrationAndDeRegistrationStatus status = RegistrationAndDeRegistrationStatus__STATUS_UNKNOWN;
   if (_getRegisterDatasetUserAct)
   {
      if (_navMiddleware.getDatasetUserRegisterStatus().getRegisterDatasetUserStatus() == navmiddleware::REGISTER_DATASET_USER_STATUS__REG_OK)
      {
         status = RegistrationAndDeRegistrationStatus__STATUS_OK;
      }
      else if (_navMiddleware.getDatasetUserRegisterStatus().getRegisterDatasetUserStatus() == navmiddleware::REGISTER_DATASET_USER_STATUS__REG_FAILED)
      {
         status = RegistrationAndDeRegistrationStatus__STATUS_FAILED;
      }
      sendSdsDatasetUserRegistrationResponse(status, _getRegisterDatasetUserAct);
      _getRegisterDatasetUserAct = 0;
   }
   else if (_getDeregisterDatasetUserAct)
   {
      if (_navMiddleware.getDatasetUserRegisterStatus().getRegisterDatasetUserStatus() == navmiddleware::REGISTER_DATASET_USER_STATUS__DEREG_OK)
      {
         status = RegistrationAndDeRegistrationStatus__STATUS_OK;
      }
      else if (_navMiddleware.getDatasetUserRegisterStatus().getRegisterDatasetUserStatus() == navmiddleware::REGISTER_DATASET_USER_STATUS__DEREG_FAILED)
      {
         status = RegistrationAndDeRegistrationStatus__STATUS_FAILED;
      }
      sendSdsDatasetUserDeregistrationResponse(status, _getDeregisterDatasetUserAct);
      _getDeregisterDatasetUserAct = 0;
   }
}


void HMIModelNavigationSDSService::onSdsDatasetUserRegistrationRequest(const boost::shared_ptr<SdsDatasetUserRegistrationRequest>& request)
{
   _getRegisterDatasetUserAct = request->getAct();
   _navMiddleware.registerDatasetUser(convertNDSDataUser(request->getNdsDataSetUser()), request->getDatasetId());
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onSdsDatasetUserRegistrationRequest(%d, %d)", request->getNdsDataSetUser(), request->getDatasetId()));
}


void HMIModelNavigationSDSService::onSdsDatasetUserDeregistrationRequest(const boost::shared_ptr<SdsDatasetUserDeregistrationRequest>& request)
{
   _getDeregisterDatasetUserAct = request->getAct();
   _navMiddleware.deregisterDatasetUser(convertNDSDataUser(request->getNdsDataSetUser()), request->getDatasetId());
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onSdsDatasetUserDeregistrationRequest(%d, %d)", request->getNdsDataSetUser(), request->getDatasetId()));
}


void HMIModelNavigationSDSService::onDatasetAccessModeChangedIndicationRequest(const boost::shared_ptr<org::bosch::cm::navigation::NavigationSDSService::DatasetAccessModeChangedIndicationRequest>& request)
{
   _navMiddleware.datasetUserAccess(convertNDSDataUser(request->getNdsDataSetUser()), request->getDatasetId(), convertNDSDataAccessType(request->getNdsDataAccessType()));
   sendDatasetAccessModeChangedIndicationResponse();
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onDatasetAccessModeChangedIndicationRequest(%d, %d , %d)", request->getNdsDataSetUser(), request->getDatasetId(), request->getNdsDataAccessType()));
}


void HMIModelNavigationSDSService::onGetDynamicPoiIconCategoriesInMapRequest(const boost::shared_ptr<GetDynamicPoiIconCategoriesInMapRequest>& request)
{
   navmiddleware::settings::MapDynamicIconSettings mapDynamicIconSettings = _navMiddleware.getMapDynamicIconSettings();
   const std::vector<navmiddleware::settings::MapDynamicIconSettings::PoiCategoryInfo>& mapDynamicPOICategory = mapDynamicIconSettings.getPoiCategoryInfoList();
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onGetDynamicPoiIconCategoriesInMapRequest(POI list size %d)", mapDynamicPOICategory.size()));
   std::vector< navmiddleware::settings::MapDynamicIconSettings::PoiCategoryInfo > ::const_iterator iter = mapDynamicPOICategory.begin();
   std::vector< DynamicMapPoiCategory > dynamicMapPOIIconCategories;
   dynamicMapPOIIconCategories.reserve(mapDynamicPOICategory.size());
   for (; iter != mapDynamicPOICategory.end(); ++iter)
   {
      std::stringstream traceOut;
      DynamicMapPoiCategory mapPOIIconCategory;
      mapPOIIconCategory.setPoiCategoryId(iter->m_poiCategoryId);
      mapPOIIconCategory.setPoiCategoryName(iter->m_poiCategoryName);
      mapPOIIconCategory.setPoiCategoryVisibilityOnMap(iter->m_poiCategoryVisibilityOnMap);
      dynamicMapPOIIconCategories.push_back(mapPOIIconCategory);
      traceOut << " { " << iter->m_poiCategoryName << ",   " << iter->m_poiCategoryId << ",   " << iter->m_poiCategoryVisibilityOnMap << " } " << std::endl;
      ETG_TRACE_USR4(("%s", traceOut.str().c_str()));
   }
   sendGetDynamicPoiIconCategoriesInMapResponse(dynamicMapPOIIconCategories);
}


void HMIModelNavigationSDSService::onSetDynamicPoiIconCategoriesInMapRequest(const boost::shared_ptr<SetDynamicPoiIconCategoriesInMapRequest>& request)
{
   navmiddleware::settings::MapDynamicIconSettings mapDynamicIconSettings = _navMiddleware.getMapDynamicIconSettings();
   std::vector<navmiddleware::settings::MapDynamicIconSettings::PoiCategoryInfo>& mapPoiCategoryInfo = mapDynamicIconSettings.getPoiCategoryInfoList();
   std::vector< DynamicMapPoiCategory > mapDynamicPOIIconCategories = request->getDynamicMapPoiCategory();
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onSetDynamicPoiIconCategoriesInMapRequest(list POI icon size %d)", mapDynamicPOIIconCategories.size()));
   std::vector< DynamicMapPoiCategory >::const_iterator iter = mapDynamicPOIIconCategories.begin();
   mapPoiCategoryInfo.resize(mapDynamicPOIIconCategories.size());
   for (int i = 0; iter != mapDynamicPOIIconCategories.end(); ++iter, ++i)
   {
      std::stringstream traceOut;
      mapPoiCategoryInfo[i].m_poiCategoryId = iter->getPoiCategoryId();
      mapPoiCategoryInfo[i].m_poiCategoryName = iter->getPoiCategoryName();
      mapPoiCategoryInfo[i].m_poiCategoryVisibilityOnMap = iter->getPoiCategoryVisibilityOnMap();
      traceOut << " { " << iter->getPoiCategoryName() << ",   " << iter->getPoiCategoryId() << ",   " << iter->getPoiCategoryVisibilityOnMap() << " } " << std::endl;
      ETG_TRACE_USR4(("%s", traceOut.str().c_str()));
   }
   _navMiddleware.updateMapDynamicIconSettings(mapDynamicIconSettings);
   _navMiddleware.applySettings();
   /* @TODO: Implementation to be added if required
   POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchOutReqMsg)(NAVIGATION_CONTEXT_MAP_MAIN, 0, APPID_APPHMI_NAVIGATION))); */
   sendSetDynamicPoiIconCategoriesInMapResponse();
}


void HMIModelNavigationSDSService::onGetHapticHandOverDataRequest(const ::boost::shared_ptr< GetHapticHandOverDataRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onGetHapticHandOverDataRequest(NaviSDSHandoverScreenID %d)", _naviSdsHandOverSreenId));
   switch (_naviSdsHandOverSreenId)
   {
      case SDS_HANDOVER_NAVI_FREE_TEXT_SEARCH_ADDRESS:
      case SDS_HANDOVER_NAVI_FREE_TEXT_SEARCH_POI:
         if (_navMiddleware.getFormBasedAddressInputInfo().getCurrentCountry().empty())
         {
            _navMiddleware.startFormBasedAddressInput(navmiddleware::LocationContext(navmiddleware::LocationContext::HMICONTEXT__GUI, navmiddleware::APPLICATIONID__NAVIGATION), BOOL_TRUE);
            _getBaseAddressFTSAct = request->getAct();
         }
         else
         {
            _getBaseAddressFTSAct = request->getAct();
            updateFreeTextSearchAddressHandOver();
         }
         break;
      case SDS_HANDOVER_NAVI_FREE_TEXT_SEARCH_CANDIDATE_LIST:
         updateFreeTextSearchCandidateListHandOver();
         _getFTSConfirmPOIorAddressAct = request->getAct();
         break;
      case SDS_HANDOVER_NAVI_ENTER_COUNTRY:
      case SDS_HANDOVER_NAVI_INTERSECTION_COUNTRY:
         updateCountryHandOver();
         break;
      case SDS_HANDOVER_NAVI_ENTER_STATE:
      case SDS_HANDOVER_NAVI_ENTER_STATE_MAIN:
      case SDS_HANDOVER_NAVI_CITYCENTER_STATE:
      case SDS_HANDOVER_NAVI_INTERSECTION_STATE:
         updateStreetAddressEnterStateHandOver();
         break;
      case SDS_HANDOVER_NAVI_HNOFIRST_HNO:
         updateStreetAddressEnterStateHandOver();
         break;
      case SDS_HANDOVER_NAVI_HNOFIRST_STREET:
         updateStreetAddressHouseNumberStreetHandOver();
         break;
      case SDS_HANDOVER_NAVI_HNOFIRST_STREET_CITY:
         updateStreetAddressHouseNumberStreetCityHandOver();
         break;
      case SDS_HANDOVER_NAVI_STREET_ADDRESS_WITHOUT_STATE_MAIN:
         updateStreetAddressMainWithoutStateHandOver();
         break;
      case SDS_HANDOVER_NAVI_CITYCENTER_CITY:
      case SDS_HANDOVER_NAVI_CITYFIRST_CITY:
      case SDS_HANDOVER_NAVI_INTERSECTION_CITY:
         updateStreetAddressCityFirstCityHandOver();
         break;
      case SDS_HANDOVER_NAVI_CITYFIRST_HNO:
         updateStreetAddressCityFirstHouseNumberHandOver();
         break;
      case SDS_HANDOVER_NAVI_CITYFIRST_HNO_STREET:
         updateStreetAddressCityFirstHouseNumberStreetHandOver();
         break;
      case SDS_HANDOVER_NAVI_AMBIG_ADDRESS:
         updateStreetAddressAmbiguityHandOver();
         break;
      case SDS_HANDOVER_NAVI_AMBIG_CITYCENTER:
         updateCityCenterAmbiguityHandOver();
         break;
      case SDS_HANDOVER_NAVI_INTERSECTION_STREET1:
      case SDS_HANDOVER_NAVI_INTERSECTION_STREET1_CANDIDATE_LIST:
      case SDS_HANDOVER_NAVI_INTERSECTION_STREET1_NORMAL_LIST:
      case SDS_HANDOVER_NAVI_INTERSECTION_STREET2:
      case SDS_HANDOVER_NAVI_INTERSECTION_STREET2_CANDIDATE_LIST:
         updateInterSectionStreetFirstHandOver();
         break;
      case SDS_HANDOVER_NAVI_INTERSECTION_LIST:
         updateInterSectionAmbigutyHandOver();
         break;
      case SDS_HANDOVER_NAVI_POI:
      case SDS_HANDOVER_NAVI_POI_CATEGORY:
         updatePOICategoryHandOver();
         break;
      case SDS_HANDOVER_NAVI_POI_LIST:
         if (_navMiddleware.getFormBasedAddressInputInfo().getCurrentCountry().empty())
         {
            _navMiddleware.startFormBasedAddressInput(navmiddleware::LocationContext(navmiddleware::LocationContext::HMICONTEXT__GUI, navmiddleware::APPLICATIONID__NAVIGATION), BOOL_TRUE);
            _getBaseAddressPOIAct = request->getAct();
         }
         else
         {
            _getBaseAddressPOIAct = request->getAct();
            updatePOIListHandOver();
         }
         break;
      case SDS_HANDOVER_NAVI_POI_INCITY:
      case SDS_HANDOVER_NAVI_POI_INCITY_CANDIDATE_LIST:
         updatePOIInCityHandOver();
         break;
      case SDS_ENTRY_NAVI_POI_ENTER_STATE:
      case SDS_ENTRY_NAVI_POI_ENTER_STATE_CANDIDATE_LIST:
         updatePOIStateHandOver();
         break;
      case SDS_ENTRY_NAVI_POI_ENTER_COUNTRY:
         updatePOICountryHandover();
         break;
      case SDS_HANDOVER_NAVI_SETDEST_RG_START:
      case SDS_HANDOVER_NAVI_SETDEST_RG_REPLACE:
         if (_navMiddleware.getFormBasedAddressInputInfo().getCurrentCountry().empty())
         {
            _navMiddleware.startFormBasedAddressInput(navmiddleware::LocationContext(navmiddleware::LocationContext::HMICONTEXT__GUI, navmiddleware::APPLICATIONID__NAVIGATION), BOOL_TRUE);
            _getBaseAddressStartRouteAct = request->getAct();
         }
         else
         {
            _getBaseAddressStartRouteAct = request->getAct();
            updateSetDestStartRGHandover();
         }
         break;
      default:
         break;
   }
}


void HMIModelNavigationSDSService::onSelectHapticHandOverListRequest(const ::boost::shared_ptr< SelectHapticHandOverListRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onSelectHapticHandOverListRequest()"));
   /* Method will be implemented soon*/
}


// Incoming courier messages from HMI
bool HMIModelNavigationSDSService::onCourierMessage(const SetNaviSDSHandoverScreenIDReqMsg& oMsg)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::SetNaviSDSHandoverScreenIDReqMsg(%d)", oMsg.GetNaviSDSHandoverScreenID()));
   _naviSdsHandOverSreenId = oMsg.GetNaviSDSHandoverScreenID();
   HandOverSessionStatus handOverSessionactiveStatus;
   handOverSessionactiveStatus.setIsHandOverScreenActive(_naviSdsHandOverSreenId != SDS_HANDOVER_NAVI_INACTIVE_SCREEN);
   // below screens do not have speedlock behavior
   if (_naviSdsHandOverSreenId == SDS_HANDOVER_NAVI_ENTER_COUNTRY || _naviSdsHandOverSreenId == SDS_HANDOVER_NAVI_INTERSECTION_COUNTRY)
   {
      // Always set Lockout status as FALSE in screens which do not have speedlock behavior
      handOverSessionactiveStatus.setIsLockOutModeActive(FALSE);
   }
   else
   {
      handOverSessionactiveStatus.setIsLockOutModeActive(_infoStore.getSpeedLockState());
   }
   // Update Route guidance flow
   switch (_naviSdsHandOverSreenId)
   {
      case SDS_HANDOVER_NAVI_HNOFIRST_STREET_CITY:
      case SDS_HANDOVER_NAVI_CITYFIRST_HNO_STREET:
      case SDS_HANDOVER_NAVI_AMBIG_ADDRESS:
      case SDS_HANDOVER_NAVI_STREET_ADDRESS_WITHOUT_STATE_MAIN:
         // Street address flow
         _currentRGFlow = SDS_HANDOVER_NAVI_SETDEST_ADDRESS;
         break;
      case SDS_HANDOVER_NAVI_CITYCENTER_CITY:
      case SDS_HANDOVER_NAVI_AMBIG_CITYCENTER:
         // City center flow
         _currentRGFlow = SDS_HANDOVER_NAVI_SETDEST_CITYCENTER;
         break;
      case SDS_HANDOVER_NAVI_INTERSECTION_STREET1:
      case SDS_HANDOVER_NAVI_INTERSECTION_LIST:
         // Street Intersection
         _currentRGFlow = SDS_HANDOVER_NAVI_SETDEST_INTERSECTION;
         break;
      case SDS_HANDOVER_NAVI_POI:
      case SDS_HANDOVER_NAVI_POI_LIST:
         // POI flow
         _currentRGFlow = SDS_HANDOVER_NAVI_SETDEST_POI;
         break;
      case SDS_HANDOVER_NAVI_SETDEST_RG_REPLACE:
      case SDS_HANDOVER_NAVI_SETDEST_RG_START:
      case SDS_HANDOVER_NAVI_INACTIVE_SCREEN:
         break;
      default:
         _currentRGFlow = SDS_HANDOVER_NAVI_INACTIVE_SCREEN;
         break;
   }

   setHandOverSessionStatus(handOverSessionactiveStatus);
   return true;
}


bool HMIModelNavigationSDSService::onCourierMessage(const SetNaviSdsFTSOptionReqMsg& oMsg)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::SetNaviSdsFTSOptionReqMsg(%d)", oMsg.GetIsPOISelected()));
   SDSHandOverScreenId sdsHandOverScreenId = SDSHandOverScreenId__NAVI_INACTIVE_SCREEN;
   if (oMsg.GetIsPOISelected())
   {
      sdsHandOverScreenId = SDSHandOverScreenId__NAVI_FREE_TEXT_SEARCH_POI;
   }
   else
   {
      sdsHandOverScreenId = SDSHandOverScreenId__NAVI_FREE_TEXT_SEARCH_ADDRESS;
   }
   ::std::vector< HapticData > dataList; // not required
   ::std::vector< SelectedDataItem > selectedDataItems;// not required
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = LocationType__DEFAULT;
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > addressListElements;
   if (_getFTSConfirmPOIorAddressAct != 0)
   {
      sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, addressListElements, _getFTSConfirmPOIorAddressAct);
      _getFTSConfirmPOIorAddressAct = 0;
   }
   return true;
}


HMIModelNavigationSDSService::Hand_Over_Data_Type HMIModelNavigationSDSService::checkHandOverDataType(unsigned int listSize)
{
   Hand_Over_Data_Type ret = HAND_OVER_NONE;
   if (listSize == E_GLO_HANDOVER_DATA_NONE)
   {
      ret = HAND_OVER_NONE;
   }
   else if (listSize == E_GLO_HANDOVER_DATA_SINGLE)
   {
      ret = HAND_OVER_SINGLE;
   }
   else if (listSize >= E_GLO_HANDOVER_DATA_LIST && listSize <= E_GLO_HANDOVER_DATA_MULTIPLE)
   {
      ret = HAND_OVER_LIST;
   }
   else if (listSize > E_GLO_HANDOVER_DATA_MULTIPLE)
   {
      ret = HAND_OVER_MULTIPLE;
   }
   return ret;
}


HMIModelNavigationSDSService::Hand_Over_Data_Type HMIModelNavigationSDSService::checkHandOverDataTypeCandidateList(unsigned int listSize, bool candidateListVisible)
{
   Hand_Over_Data_Type ret = HAND_OVER_NONE;
   if (candidateListVisible)
   {
      ret = checkHandOverDataType(listSize);
   }
   else
   {
      // user does not enter anything to search
      ret = HAND_OVER_NONE;
   }
   return ret;
}


void HMIModelNavigationSDSService::updateDataListHandOver(const Hand_Over_Data_Type handOverType, const navmiddleware::LocationInfos& locationInfos, std::vector< HapticData >& dataList)
{
   const ::std::vector<navmiddleware::LocationInfos::LocationInfo>& infos = locationInfos.getLocationInfos();
   const unsigned int totalItems = locationInfos.getCountOfAllLocations();
   HapticData hapticData;
   dataList.clear();
   switch (handOverType)
   {
      case HAND_OVER_NONE:
         break;
      case HAND_OVER_SINGLE:
      {
         std::stringstream traceOut;
         traceOut << " HAND_OVER_SINGLE: dataList size = " << totalItems;
         dataList.reserve(E_GLO_HANDOVER_DATA_SINGLE);
         traceOut << " List item: ";
         if (!infos.empty())
         {
            hapticData.setSdsNameObjectId(constructSDSGlobalNameID(infos.front().m_sdsGlobalNamedObjectIdInfo));
            hapticData.setData(infos.front().m_elementNameAsInNavigationData.empty() ? infos.front().m_elementName.c_str() : infos.front().m_elementNameAsInNavigationData.c_str());
            dataList.push_back(hapticData);
            traceOut << infos.front().m_elementNameAsInNavigationData.empty() ? infos.front().m_elementName.c_str() : infos.front().m_elementNameAsInNavigationData.c_str();
         }
         ETG_TRACE_USR4((" %s", traceOut.str().c_str()));
         break;
      }
      case HAND_OVER_LIST:
      {
         std::stringstream traceOut;
         traceOut << "HAND_OVER_LIST: dataList size = " << totalItems;
         std::vector<navmiddleware::LocationInfos::LocationInfo>::const_iterator info = infos.begin();
         dataList.reserve(infos.size());
         traceOut << " List item: ";
         for (; info != infos.end(); ++info)
         {
            hapticData.setSdsNameObjectId(constructSDSGlobalNameID(info->m_sdsGlobalNamedObjectIdInfo));
            hapticData.setData(info->m_elementNameAsInNavigationData.empty() ? info->m_elementName.c_str() : info->m_elementNameAsInNavigationData.c_str()) ;
            dataList.push_back(hapticData);
            traceOut << (info->m_elementNameAsInNavigationData.empty() ? info->m_elementName.c_str() : info->m_elementNameAsInNavigationData.c_str()) << " , ";
         }
         ETG_TRACE_USR4(("%s", traceOut.str().c_str()));
         break;
      }
      case HAND_OVER_MULTIPLE:
      {
         std::stringstream traceOut;
         traceOut << " HAND_OVER_MULTIPLE: dataList size = " << totalItems;
         std::vector<navmiddleware::LocationInfos::LocationInfo>::const_iterator info = infos.begin();
         dataList.reserve(E_GLO_HANDOVER_DATA_MULTIPLE + 1); // Send list size more than 5 entries (maxium upto 6)
         traceOut << " List item: ";
         for (unsigned int i = 0; (info != infos.end() && (i <= E_GLO_HANDOVER_DATA_MULTIPLE)); ++info, ++i)
         {
            hapticData.setSdsNameObjectId(constructSDSGlobalNameID(info->m_sdsGlobalNamedObjectIdInfo));
            hapticData.setData(info->m_elementNameAsInNavigationData.empty() ? info->m_elementName.c_str() : info->m_elementNameAsInNavigationData.c_str()) ;
            dataList.push_back(hapticData);
            traceOut << (info->m_elementNameAsInNavigationData.empty() ? info->m_elementName.c_str() : info->m_elementNameAsInNavigationData.c_str()) << " , ";
         }
         ETG_TRACE_USR4((" %s", traceOut.str().c_str()));
         break;
      }
      default:
         break;
   }
}


GlobalNamedObjectId HMIModelNavigationSDSService::constructSDSGlobalNameID(const navmiddleware::ValidValue<SdsGlobalNamedObjectIdInfo>& SdsGlobalNamedObjectIdInfo)
{
   GlobalUpdateRegionId globalUpdateRegionId;
   constructSDSGlobalUpdateRegionId(SdsGlobalNamedObjectIdInfo.getValue().getSdsGlobalUpdateRegionInfo(), globalUpdateRegionId);
   GlobalNamedObjectId globalNameObjectIds;
   globalNameObjectIds.setGlobalUpdateRegionId(globalUpdateRegionId);
   globalNameObjectIds.setNamedObjectId(SdsGlobalNamedObjectIdInfo.getValue().getNamedObjectId());
   globalNameObjectIds.setIsValid(SdsGlobalNamedObjectIdInfo.isValid());
   return globalNameObjectIds;
}


GlobalNamedObjectId HMIModelNavigationSDSService::constructSDSGlobalNameID(const navmiddleware::SdsGlobalNamedObjectIdInfo& SdsGlobalNamedObjectIdInfo)
{
   GlobalUpdateRegionId globalUpdateRegionId;
   constructSDSGlobalUpdateRegionId(SdsGlobalNamedObjectIdInfo.getSdsGlobalUpdateRegionInfo(), globalUpdateRegionId);
   GlobalNamedObjectId globalNameObjectIds;
   globalNameObjectIds.setGlobalUpdateRegionId(globalUpdateRegionId);
   globalNameObjectIds.setNamedObjectId(SdsGlobalNamedObjectIdInfo.getNamedObjectId());
   globalNameObjectIds.setIsValid(true);
   return globalNameObjectIds;
}


void HMIModelNavigationSDSService::updateSelectedDataItem(bool addCountry, bool addState, bool addCity, bool addHouseNo, bool addStreet, bool addCrosStreet, std::vector<SelectedDataItem>& selectedDataItems)
{
   const navmiddleware::AddressInputInfo& addresInputInfo = _navMiddleware.getFormBasedAddressInputInfo();
   std::stringstream traceOut;
   if (addCountry)
   {
      SelectedDataItem selectedDataItemCountry;
      selectedDataItemCountry.setHmiSelectedAddressElementType(HMISelectedAddressElementType__COUNTRY);
      selectedDataItemCountry.setData(addresInputInfo.getCurrentCountry());
      selectedDataItems.push_back(selectedDataItemCountry);
      traceOut << " Country: " << addresInputInfo.getCurrentCountry().c_str();
   }
   if (addState)
   {
      SelectedDataItem selectedDataItemState;
      selectedDataItemState.setHmiSelectedAddressElementType(HMISelectedAddressElementType__STATE);
      selectedDataItemState.setData(addresInputInfo.getCurrentState());
      selectedDataItems.push_back(selectedDataItemState);
      traceOut << " State: " << addresInputInfo.getCurrentState().c_str();
   }
   if (addCity)
   {
      SelectedDataItem selectedDataItemCity;
      selectedDataItemCity.setHmiSelectedAddressElementType(HMISelectedAddressElementType__CITY);
      selectedDataItemCity.setData(addresInputInfo.getCurrentCity());
      selectedDataItems.push_back(selectedDataItemCity);
      traceOut << " City: " << addresInputInfo.getCurrentCity().c_str();
   }
   if (addHouseNo)
   {
      SelectedDataItem selectedDataItemHouseNo;
      selectedDataItemHouseNo.setHmiSelectedAddressElementType(HMISelectedAddressElementType__HOUSENUMBER);
      selectedDataItemHouseNo.setData(addresInputInfo.getCurrentHouseNumber());
      selectedDataItems.push_back(selectedDataItemHouseNo);
      traceOut << " HouseNo: " << addresInputInfo.getCurrentHouseNumber().c_str();
   }
   if (addStreet)
   {
      SelectedDataItem selectedDataItemStreet;
      selectedDataItemStreet.setHmiSelectedAddressElementType(HMISelectedAddressElementType__STREET);
      selectedDataItemStreet.setData(addresInputInfo.getCurrentStreet());
      selectedDataItems.push_back(selectedDataItemStreet);
      traceOut << " Street: " << addresInputInfo.getCurrentStreet().c_str();
   }
   if (addCrosStreet)
   {
      SelectedDataItem selectedDataItemCrossStreet;
      selectedDataItemCrossStreet.setHmiSelectedAddressElementType(HMISelectedAddressElementType__STREET);
      selectedDataItemCrossStreet.setData(addresInputInfo.getCurrentCrossStreet());
      selectedDataItems.push_back(selectedDataItemCrossStreet);
      traceOut << " CrossStreet: " << addresInputInfo.getCurrentCrossStreet().c_str();
   }
   ETG_TRACE_USR4((" %s", traceOut.str().c_str()));
}


void HMIModelNavigationSDSService::updateFreeTextSearchAddressHandOver()
{
   const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getFreeTextSearchResultInfos();
   const navmiddleware::AddressInputInfo& addresInputInfo = _navMiddleware.getFormBasedAddressInputInfo();
   const ::std::vector<navmiddleware::LocationInfos::LocationInfo>& infos = locationInfos.getLocationInfos();
   const unsigned int actDestListSize = locationInfos.getCountOfAllLocations();

   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_naviSdsHandOverSreenId);
   ::std::vector< HapticData > dataList; // not required
   ::std::vector< SelectedDataItem > selectedDataItems;
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = convertNaviSdsHandOverLocationType(locationInfos.getCurrentSearchScope().m_type);
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > addressListElements;

   Hand_Over_Data_Type handOverType = checkHandOverDataType(actDestListSize);
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updateFreeTextSearchAddressHandOver(handOverType %d, actDestListSize%d)", handOverType, actDestListSize));

   switch (handOverType)
   {
      case HAND_OVER_NONE:
         break;
      case HAND_OVER_SINGLE:
      {
         std::stringstream traceOut;
         SelectedDataItem selectedDataItem_SearchResult;
         SelectedDataItem selectedDataItem_Country;
         HapticData listItem;
         ::std::string addressString;
         AddressListElement addressListElement;
         traceOut << " HAND_OVER_SINGLE " << " Country : " << addresInputInfo.getCurrentCountry().c_str();
         selectedDataItem_SearchResult.setHmiSelectedAddressElementType(HMISelectedAddressElementType__POI_NAME);
         selectedDataItem_Country.setHmiSelectedAddressElementType(HMISelectedAddressElementType__COUNTRY);
         if (!infos.empty())
         {
            selectedDataItem_SearchResult.setData(infos.front().m_elementNameAsInNavigationData.empty() ? infos.front().m_elementName : infos.front().m_elementNameAsInNavigationData);
            addressString = infos.front().m_elementNameAsInNavigationData.empty() ? infos.front().m_elementName : infos.front().m_elementNameAsInNavigationData;
            addressListElement.setData(infos.front().m_elementNameAsInNavigationData.empty() ? infos.front().m_elementName : infos.front().m_elementNameAsInNavigationData);
            addressListElement.setId(static_cast<uint64>(infos.front().m_index));
            if (infos.front().m_direction.isValid())
            {
               addressListElement.setDirection(convertNaviSDSDirectionDescription(infos.front().m_direction.getValue()));
            }
            else
            {
               addressListElement.setDirection(DirectionDescription__INVALID);
            }
            addressListElement.setDistance(infos.front().m_distance);
            addressListElement.setAddress(infos.front().m_description);
            addressListElements.reserve(1);// Only one list element is used in case HAND_OVER_SINGLE
            addressListElements.push_back(addressListElement);
            traceOut << " POI name : " << addressString.c_str() << infos.front().m_distance << infos.front().m_description;

            listItem.setData(addressString);
            listItem.setSdsNameObjectId(constructSDSGlobalNameID(infos.front().m_sdsGlobalNamedObjectIdInfo));
         }
         selectedDataItem_Country.setData(addresInputInfo.getCurrentCountry());
         selectedDataItems.reserve(2);// Selected item vector has 2 elements(Country and one search result) in case HAND_OVER_SINGLE
         selectedDataItems.push_back(selectedDataItem_Country);
         selectedDataItems.push_back(selectedDataItem_SearchResult);
         dataList.push_back(listItem);
         ETG_TRACE_USR4((" %s", traceOut.str().c_str()));
         break;
      }
      case HAND_OVER_LIST:
      {
         std::stringstream traceOut;
         traceOut << " HAND_OVER_SINGLE " << " Country : " << addresInputInfo.getCurrentCountry().c_str();
         SelectedDataItem selectedDataItem_Country;
         selectedDataItem_Country.setHmiSelectedAddressElementType(HMISelectedAddressElementType__COUNTRY);
         selectedDataItem_Country.setData(addresInputInfo.getCurrentCountry());
         selectedDataItems.reserve(1);// Selected Item vector has one element (Country) in case HAND_OVER_LIST
         selectedDataItems.push_back(selectedDataItem_Country);
         ETG_TRACE_USR4((" %s", traceOut.str().c_str()));
         break;
      }
      case HAND_OVER_MULTIPLE:
         break;
      default:
         break;
   }
   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, addressListElements, _getBaseAddressFTSAct);
   _getBaseAddressFTSAct = 0;
}


void HMIModelNavigationSDSService::updateFreeTextSearchCandidateListHandOver()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updateFreeTextSearchCandidateListHandOver()"));
}


void HMIModelNavigationSDSService::updateCountryHandOver()
{
   const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getAddressElements();
   const unsigned int totalItems = locationInfos.getCountOfAllLocations();
   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_naviSdsHandOverSreenId);
   ::std::vector< HapticData > dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > AddressListElements; // not require in this flow
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = convertNaviSdsHandOverLocationType(locationInfos.getCurrentSearchScope().m_type);

   Hand_Over_Data_Type handOverType = checkHandOverDataType(totalItems);
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updateCountryHandOver(handOverType:%d)", handOverType));
   // Update Country list
   updateDataListHandOver(handOverType, locationInfos, dataList);
   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, AddressListElements);
}


void HMIModelNavigationSDSService::updateStreetAddressEnterStateHandOver()
{
   Hand_Over_Data_Type handOverType = HAND_OVER_NONE;
   const navmiddleware::AddressInputInfo& addresInputInfo = _navMiddleware.getFormBasedAddressInputInfo();
   const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getAddressElements();
   const navmiddleware::ElementSpellerInfo&  elementSpellerInfo = _navMiddleware.getAddressSpellerInfo();
   const bool candidateListVisible = !(elementSpellerInfo.getCurrentSpellerMatch().empty());
   const unsigned int totalItems = locationInfos.getCountOfAllLocations();

   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_naviSdsHandOverSreenId);
   ::std::vector< HapticData > dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = convertNaviSdsHandOverLocationType(locationInfos.getCurrentSearchScope().m_type);
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > AddressListElements; // not require in this flow

   if (_naviSdsHandOverSreenId == SDS_HANDOVER_NAVI_ENTER_STATE_MAIN || _naviSdsHandOverSreenId == SDS_HANDOVER_NAVI_HNOFIRST_HNO)
   {
      handOverType = HAND_OVER_SINGLE;
   }
   else
   {
      handOverType = checkHandOverDataTypeCandidateList(totalItems, candidateListVisible);
   }
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updateStreetAddressEnterStateHandOver(handOverType:%d ; candidateListVisible:%d)", handOverType, candidateListVisible));
   switch (handOverType)
   {
      case HAND_OVER_NONE:
         break;
      case HAND_OVER_SINGLE:
      {
         std::stringstream traceOut;
         // update selected items State
         updateSelectedDataItem(false, true, false, false, false, false, selectedDataItems);
         traceOut << "HAND_OVER_SINGLE: " << " SelectedDataItem State: " << addresInputInfo.getCurrentState().c_str();
         ETG_TRACE_USR4((" %s", traceOut.str().c_str()));
         break;
      }
      case HAND_OVER_LIST:
         break;
      case HAND_OVER_MULTIPLE:
         break;
      default:
         break;
   }
   // Update State list
   updateDataListHandOver(handOverType, locationInfos, dataList);
   // update selected items Country
   updateSelectedDataItem(true, false, false, false, false, false, selectedDataItems);
   ETG_TRACE_USR4(("Country: %s )", addresInputInfo.getCurrentCountry()));
   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, AddressListElements);
}


void HMIModelNavigationSDSService::updateStreetAddressHouseNumberStreetHandOver()
{
   Hand_Over_Data_Type handOverType = HAND_OVER_NONE;
   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_naviSdsHandOverSreenId);
   ::std::vector< HapticData > dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getAddressElements();
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = convertNaviSdsHandOverLocationType(locationInfos.getCurrentSearchScope().m_type);
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > AddressListElements; // not req

   const navmiddleware::ElementSpellerInfo&  elementSpellerInfo = _navMiddleware.getAddressSpellerInfo();
   const bool candidateListVisible = !(elementSpellerInfo.getCurrentSpellerMatch().empty());
   const unsigned int totalItems = locationInfos.getCountOfAllLocations();
   // if user does not enter anything to search, send default country and state
   handOverType = checkHandOverDataTypeCandidateList(totalItems, candidateListVisible);
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updateStreetAddressHouseNumberStreetHandOver(handOverType : %d; candidateListVisible: %d)", handOverType, candidateListVisible));
   // Update City list
   updateDataListHandOver(handOverType, locationInfos, dataList);
   selectedDataItems.reserve(3);
   // update selected items Country, State and HouseNo
   updateSelectedDataItem(true, true, false, true, false, false, selectedDataItems);
   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, AddressListElements);
}


void HMIModelNavigationSDSService::updateStreetAddressHouseNumberStreetCityHandOver()
{
   Hand_Over_Data_Type handOverType = HAND_OVER_NONE;
   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_naviSdsHandOverSreenId);
   ::std::vector< HapticData > dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getAddressElements();
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = convertNaviSdsHandOverLocationType(locationInfos.getCurrentSearchScope().m_type);
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > AddressListElements; // not req

   const navmiddleware::ElementSpellerInfo&  elementSpellerInfo = _navMiddleware.getAddressSpellerInfo();
   const bool candidateListVisible = !(elementSpellerInfo.getCurrentSpellerMatch().empty());
   const unsigned int totalItems = locationInfos.getCountOfAllLocations();
   handOverType = checkHandOverDataTypeCandidateList(totalItems, candidateListVisible);
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updateStreetAddressHouseNumberStreetCityHandOver(handOverType : %d; candidateListVisible: %d)", handOverType, candidateListVisible));
   // Update City lit
   updateDataListHandOver(handOverType, locationInfos, dataList);
   // Update selected items Country, State, HouseNo, Street
   selectedDataItems.reserve(4);
   updateSelectedDataItem(true, true, false, true, true, false, selectedDataItems);
   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, AddressListElements);
}


void HMIModelNavigationSDSService::updateStreetAddressMainWithoutStateHandOver()
{
   Hand_Over_Data_Type handOverType = HAND_OVER_NONE;
   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_naviSdsHandOverSreenId);
   ::std::vector< HapticData> dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getAddressElements();
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = convertNaviSdsHandOverLocationType(locationInfos.getCurrentSearchScope().m_type);
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > AddressListElements; // not req

   const navmiddleware::ElementSpellerInfo&  elementSpellerInfo = _navMiddleware.getAddressSpellerInfo();
   const bool candidateListVisible = !(elementSpellerInfo.getCurrentSpellerMatch().empty());
   const unsigned int totalItems = locationInfos.getCountOfAllLocations();
   handOverType = checkHandOverDataTypeCandidateList(totalItems, candidateListVisible);
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updateStreetAddressMainWithoutStateHandOver(handOverType : %d; candidateListVisible: %d)", handOverType, candidateListVisible));
   // Update City lit
   updateDataListHandOver(handOverType, locationInfos, dataList);
   // Update selected items Country, State, HouseNo, Street
   selectedDataItems.reserve(5);
   updateSelectedDataItem(true, false, true, true, true, true, selectedDataItems);
   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, AddressListElements);
}


void HMIModelNavigationSDSService::updateStreetAddressCityFirstCityHandOver()
{
   const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getAddressElements();
   Hand_Over_Data_Type handOverType = HAND_OVER_NONE;
   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_naviSdsHandOverSreenId);
   ::std::vector< HapticData > dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = convertNaviSdsHandOverLocationType(locationInfos.getCurrentSearchScope().m_type);
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > AddressListElements; // not req

   const navmiddleware::ElementSpellerInfo&  elementSpellerInfo = _navMiddleware.getAddressSpellerInfo();
   const bool candidateListVisible = !(elementSpellerInfo.getCurrentSpellerMatch().empty());
   const unsigned int totalItems = locationInfos.getCountOfAllLocations();
   handOverType = checkHandOverDataTypeCandidateList(totalItems, candidateListVisible);
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updateStreetAddressCityFirstCityHandOver(handOverType : %d; candidateListVisible: %d)", handOverType, candidateListVisible));
   // Update City list
   updateDataListHandOver(handOverType, locationInfos, dataList);
   selectedDataItems.reserve(2); // need info of State and Country
   updateSelectedDataItem(true, true, false, false, false, false, selectedDataItems);

   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, AddressListElements);
}


void HMIModelNavigationSDSService::updateStreetAddressCityFirstHouseNumberHandOver()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updateStreetAddressCityFirstHouseNumberHandOver()"));
   const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getAddressElements();
   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_naviSdsHandOverSreenId);
   ::std::vector< HapticData > dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = convertNaviSdsHandOverLocationType(locationInfos.getCurrentSearchScope().m_type);
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > AddressListElements; // not req
   selectedDataItems.reserve(3); // need info of State, Country and City
   updateSelectedDataItem(true, true, true, false, false, false, selectedDataItems);
   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, AddressListElements);
}


void HMIModelNavigationSDSService::updateStreetAddressCityFirstHouseNumberStreetHandOver()
{
   const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getAddressElements();
   Hand_Over_Data_Type handOverType = HAND_OVER_NONE;
   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_naviSdsHandOverSreenId);
   ::std::vector< HapticData> dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = convertNaviSdsHandOverLocationType(locationInfos.getCurrentSearchScope().m_type);
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > AddressListElements; // not req
   const navmiddleware::ElementSpellerInfo&  elementSpellerInfo = _navMiddleware.getAddressSpellerInfo();
   const bool candidateListVisible = !(elementSpellerInfo.getCurrentSpellerMatch().empty());
   const unsigned int totalItems = locationInfos.getCountOfAllLocations();
   //handOverType = checkHandOverDataTypeCandidateList(totalItems, candidateListVisible);
   handOverType = checkHandOverDataType(totalItems);
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updateStreetAddressCityFirstHouseNumberStreetHandOver(handOverType : %d; candidateListVisible: %d)", handOverType, candidateListVisible));
   // Update Street list
   updateDataListHandOver(handOverType, locationInfos, dataList);
   selectedDataItems.reserve(4); // need info of State, Country, City and House Number
   updateSelectedDataItem(true, true, true, true, false, false, selectedDataItems);

   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, AddressListElements);
}


void HMIModelNavigationSDSService::updateStreetAddressAmbiguityHandOver()
{
   const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getAddressElements();
   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_naviSdsHandOverSreenId);
   ::std::vector< HapticData > dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = convertNaviSdsHandOverLocationType(locationInfos.getCurrentSearchScope().m_type);
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > AddressListElements; // not req
   const unsigned int totalItems = locationInfos.getCountOfAllLocations();
   Hand_Over_Data_Type handOverType = checkHandOverDataType(totalItems);
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updateStreetAddressAmbiguityHandOver(handOverType %d, dataSzie %d)", handOverType, totalItems));
   // Update ambiguity list
   updateDataListHandOver(handOverType, locationInfos, dataList);

   SDSHandOverOptions sdsHandOverOption;
   sdsHandOverOption.setIsAmbigious(TRUE);
   sdsHandOverOptions.push_back(sdsHandOverOption);
   selectedDataItems.reserve(4); // need info of State, Country, City and House Number
   updateSelectedDataItem(true, true, true, true, false, false, selectedDataItems);

   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, AddressListElements);
}


void HMIModelNavigationSDSService::updateCityCenterAmbiguityHandOver()
{
   const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getAddressElements();
   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_naviSdsHandOverSreenId);
   ::std::vector< HapticData > dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = convertNaviSdsHandOverLocationType(locationInfos.getCurrentSearchScope().m_type);
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > AddressListElements; // not req
   const unsigned int totalItems = locationInfos.getCountOfAllLocations();
   Hand_Over_Data_Type handOverType = checkHandOverDataType(totalItems);
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updateCityCenterAmbiguityHandOver(handOverType %d, dataSzie %d)", handOverType, totalItems));
   // Update ambiguity list
   updateDataListHandOver(handOverType, locationInfos, dataList);

   SDSHandOverOptions sdsHandOverOption;
   sdsHandOverOption.setIsAmbigious(TRUE);
   sdsHandOverOptions.push_back(sdsHandOverOption);
   selectedDataItems.reserve(3); // need info of State, Country and City
   updateSelectedDataItem(true, true, true, false, false, false, selectedDataItems);

   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, AddressListElements);
}


void HMIModelNavigationSDSService::updateInterSectionStreetFirstHandOver()
{
   const navmiddleware::AddressInputInfo& addresInputInfo = _navMiddleware.getFormBasedAddressInputInfo();
   Hand_Over_Data_Type handOverType = HAND_OVER_NONE;
   bool isCitySelected = !(addresInputInfo.getCurrentCity().empty() && addresInputInfo.isStreetSelectable());
   bool isStreetSelected = !(addresInputInfo.getCurrentStreet().empty());
   bool isCrossStreetSelected = !(addresInputInfo.getCurrentCrossStreet().empty());

   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_naviSdsHandOverSreenId);
   ::std::vector< HapticData > dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = convertNaviSdsHandOverLocationType(_navMiddleware.getAddressElements().getCurrentSearchScope().m_type);
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > AddressListElements; // not req
   std::stringstream traceOut;

   switch (_naviSdsHandOverSreenId)
   {
      case SDS_HANDOVER_NAVI_INTERSECTION_STREET1:
      {
         handOverType = HAND_OVER_NONE;
         if (isStreetSelected)
         {
            // Update selected items Street
            updateSelectedDataItem(false, false, false, false, true, false, selectedDataItems);
            traceOut << " INTERSECTION_STREET1::Selected Street 1: " << addresInputInfo.getCurrentStreet().c_str();
         }
         if (isCrossStreetSelected)
         {
            // Update selected items Cross-street
            updateSelectedDataItem(false, false, false, false, false, true, selectedDataItems);
            traceOut << " INTERSECTION_STREET1:: Selected Street 2: " << addresInputInfo.getCurrentCrossStreet().c_str();
         }
         break;
      }
      case SDS_HANDOVER_NAVI_INTERSECTION_STREET1_CANDIDATE_LIST:
      {
         const bool candidateListVisible = !(_navMiddleware.getAddressSpellerInfo().getCurrentSpellerMatch().empty());
         handOverType = checkHandOverDataTypeCandidateList(_navMiddleware.getAddressElements().getCountOfAllLocations(), candidateListVisible);
         break;
      }
      case SDS_HANDOVER_NAVI_INTERSECTION_STREET1_NORMAL_LIST:
      {
         handOverType = checkHandOverDataType(_navMiddleware.getAddressElements().getCountOfAllLocations());
         break;
      }
      case SDS_HANDOVER_NAVI_INTERSECTION_STREET2:
      {
         handOverType = checkHandOverDataType(_navMiddleware.getAddressElements().getCountOfAllLocations());
         // Update selected items Street
         updateSelectedDataItem(false, false, false, false, true, false, selectedDataItems);
         traceOut << " INTERSECTION_STREET2::Selected Street 1: " << addresInputInfo.getCurrentStreet().c_str();
         break;
      }
      case SDS_HANDOVER_NAVI_INTERSECTION_STREET2_CANDIDATE_LIST:
      {
         const bool candidateListVisible = !(_navMiddleware.getAddressSpellerInfo().getCurrentSpellerMatch().empty());
         handOverType = checkHandOverDataTypeCandidateList(_navMiddleware.getAddressElements().getCountOfAllLocations(), candidateListVisible);
         // Update selected items Street
         updateSelectedDataItem(false, false, false, false, true, false, selectedDataItems);
         traceOut << " INTERSECTION_STREET2_CANDIDATE::Selected Street 1: " << addresInputInfo.getCurrentStreet().c_str();
         break;
      }
      default:
         break;
   }
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updateInterSectionStreetFirstHandOver(handOverType %d, isCitySelected %d, isStreetSelected %d, isCrossStreetSelected %d )"
                   , handOverType, isCitySelected, isStreetSelected, isCrossStreetSelected));
   updateDataListHandOver(handOverType, _navMiddleware.getAddressElements(), dataList);
   // Update selected items Country, State
   updateSelectedDataItem(true, true, false, false, false, false, selectedDataItems);
   if (isCitySelected)
   {
      updateSelectedDataItem(false, false, true, false, false, false, selectedDataItems);
   }
   ETG_TRACE_USR4(("%s", traceOut.str().c_str()));
   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, AddressListElements);
}


void HMIModelNavigationSDSService::updateInterSectionAmbigutyHandOver()
{
   const navmiddleware::AddressInputInfo& addresInputInfo = _navMiddleware.getFormBasedAddressInputInfo();
   bool isCitySelected = !(addresInputInfo.getCurrentCity().empty() && addresInputInfo.isStreetSelectable());
   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_naviSdsHandOverSreenId);
   ::std::vector< HapticData > dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = convertNaviSdsHandOverLocationType(_navMiddleware.getAddressElements().getCurrentSearchScope().m_type);
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > AddressListElements; // not req
   const unsigned int totalItems = _navMiddleware.getAddressElements().getCountOfAllLocations();
   Hand_Over_Data_Type handOverType = checkHandOverDataType(totalItems);
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updateInterSectionAmbigutyHandOver(totalItems %d, handOverType %d)", totalItems, handOverType));
   // Update list of Intersection Ambiguity list
   updateDataListHandOver(handOverType, _navMiddleware.getAddressElements(), dataList);
   // Update selected items Country, State, Street and Cross-street
   updateSelectedDataItem(true, true, false, false, true, true, selectedDataItems);
   if (isCitySelected)
   {
      // Update selected items City
      updateSelectedDataItem(false, false, true, false, false, false, selectedDataItems);
   }
   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, AddressListElements);
}


void HMIModelNavigationSDSService::updatePOICategoryHandOver()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updatePOICategoryHandOver()"));
   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_naviSdsHandOverSreenId);
   ::std::vector< HapticData > dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > addressListElements; // not req
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = convertNaviSdsHandOverLocationType(_navMiddleware.getAddressElements().getCurrentSearchScope().m_type);
   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, addressListElements);
}


void HMIModelNavigationSDSService::updatePOIListHandOver()
{
   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_naviSdsHandOverSreenId);
   ::std::vector< HapticData > dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > addressListElements;
   const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getPoiEntries();
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = convertNaviSdsHandOverLocationType(locationInfos.getCurrentSearchScope().m_type);
   const ::std::vector<navmiddleware::LocationInfos::LocationInfo>& infos = locationInfos.getLocationInfos();
   ::std::vector<navmiddleware::LocationInfos::LocationInfo>::const_iterator info = infos.begin();
   const unsigned int actDestListSize = locationInfos.getCountOfAllLocations();
   Hand_Over_Data_Type handOverType = checkHandOverDataType(actDestListSize);
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updatePOIListHandOver(ListSize %d, handOverType %d, SearchScope %d)", actDestListSize, handOverType, locationType));
   updateDataListHandOver(handOverType, locationInfos, dataList);
   std::stringstream traceOut;
   //Update POI category
   SelectedDataItem selectedDataItem_POICategory;
   // Update selected items Country and State and Category
   selectedDataItems.reserve(3);
   updateSelectedDataItem(true, true, false, false, false, false, selectedDataItems);
   selectedDataItem_POICategory.setHmiSelectedAddressElementType(HMISelectedAddressElementType__POI_CATEGORY);
   // Selected POI category will be updated later
   selectedDataItem_POICategory.setData("");
   selectedDataItems.push_back(selectedDataItem_POICategory);
   // Update Description, Distance, Direction for POI items.
   for (unsigned int i = 0; (info != infos.end() && (i <= E_GLO_HANDOVER_DATA_MULTIPLE)); ++info, ++i)
   {
      AddressListElement addressListElement;
      traceOut << " POI detail :";

      addressListElement.setData(info->m_elementNameAsInNavigationData.empty() ? info->m_elementName : info->m_elementNameAsInNavigationData);
      traceOut << " POI Name :" << info->m_elementNameAsInNavigationData.empty() ? info->m_elementName.c_str() : info->m_elementNameAsInNavigationData.c_str();
      addressListElement.setId(static_cast<uint64>(info->m_index));
      if (info->m_direction.isValid())
      {
         addressListElement.setDirection(convertNaviSDSDirectionDescription(info->m_direction.getValue()));
         traceOut << " Dir :" << info->m_direction.getValue();
      }
      else
      {
         addressListElement.setDirection(DirectionDescription__INVALID);
         traceOut << " Dir : INVALID";
      }
      addressListElement.setDistance(info->m_distance);
      addressListElement.setAddress(info->m_description);
      traceOut << " m_distance " << info->m_distance.c_str();
      addressListElements.push_back(addressListElement);
   }
   ETG_TRACE_USR4(("%s", traceOut.str().c_str()));
   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, addressListElements, _getBaseAddressPOIAct);
   _getBaseAddressPOIAct = 0;
}


void HMIModelNavigationSDSService::updatePOIInCityHandOver()
{
   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_naviSdsHandOverSreenId);
   ::std::vector< HapticData > dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > addressListElements; // not req
   Hand_Over_Data_Type handOverType = HAND_OVER_NONE;
   const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getAddressElements();
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = convertNaviSdsHandOverLocationType(locationInfos.getCurrentSearchScope().m_type);
   const navmiddleware::AddressInputInfo& addresInputInfo = _navMiddleware.getFormBasedAddressInputInfo();
   const unsigned int actDestListSize = locationInfos.getCountOfAllLocations();
   if (_naviSdsHandOverSreenId == SDS_HANDOVER_NAVI_POI_INCITY)
   {
      handOverType = checkHandOverDataType(actDestListSize);
      ETG_TRACE_USR4(("HMIModelNavigationSDSService::updatePOIInCityHandOver(ListSize %d, handOverType %d)", actDestListSize, handOverType));
   }
   else if (_naviSdsHandOverSreenId == SDS_HANDOVER_NAVI_POI_INCITY_CANDIDATE_LIST)
   {
      const bool candidateListVisible = !(_navMiddleware.getAddressSpellerInfo().getCurrentSpellerMatch().empty());
      handOverType = checkHandOverDataTypeCandidateList(actDestListSize, candidateListVisible);
      ETG_TRACE_USR4(("HMIModelNavigationSDSService::updatePOIInCityHandOver(ListSize %d, handOverType %d, candidateListVisible %d)", actDestListSize, handOverType, candidateListVisible));
   }
   // Update City list
   updateDataListHandOver(handOverType, locationInfos, dataList);
   selectedDataItems.reserve(3); // need info of Country, State and Category
   updateSelectedDataItem(true, true, false, false, false, false, selectedDataItems);
   SelectedDataItem selectedDataItem_POICategory;
   selectedDataItem_POICategory.setHmiSelectedAddressElementType(HMISelectedAddressElementType__POI_CATEGORY);
   // Selected POI category will be updated later
   selectedDataItem_POICategory.setData("");
   selectedDataItems.push_back(selectedDataItem_POICategory);
   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, addressListElements);
}


void HMIModelNavigationSDSService::updatePOIStateHandOver()
{
   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_naviSdsHandOverSreenId);
   ::std::vector< HapticData > dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > addressListElements; // not req
   Hand_Over_Data_Type handOverType = HAND_OVER_NONE;
   const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getAddressElements();
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = convertNaviSdsHandOverLocationType(locationInfos.getCurrentSearchScope().m_type);
   const navmiddleware::AddressInputInfo& addresInputInfo = _navMiddleware.getFormBasedAddressInputInfo();
   const unsigned int actDestListSize = locationInfos.getCountOfAllLocations();
   if (_naviSdsHandOverSreenId == SDS_ENTRY_NAVI_POI_ENTER_STATE)
   {
      handOverType = checkHandOverDataType(actDestListSize);
      ETG_TRACE_USR4(("HMIModelNavigationSDSService::updatePOIStateHandOver(ListSize %d, handOverType %d)", actDestListSize, handOverType));
   }
   else if (_naviSdsHandOverSreenId == SDS_ENTRY_NAVI_POI_ENTER_STATE_CANDIDATE_LIST)
   {
      const bool candidateListVisible = !(_navMiddleware.getAddressSpellerInfo().getCurrentSpellerMatch().empty());
      handOverType = checkHandOverDataTypeCandidateList(actDestListSize, candidateListVisible);
      ETG_TRACE_USR4(("HMIModelNavigationSDSService::updatePOIStateHandOver(ListSize %d, handOverType %d, candidateListVisible %d)", actDestListSize, handOverType, candidateListVisible));
   }
   // Update State list
   updateDataListHandOver(handOverType, locationInfos, dataList);
   SelectedDataItem selectedDataItem_POICategory;
   selectedDataItems.reserve(2); // need info of Country and Category
   updateSelectedDataItem(true, false, false, false, false, false, selectedDataItems);
   selectedDataItem_POICategory.setHmiSelectedAddressElementType(HMISelectedAddressElementType__POI_CATEGORY);
   // Selected POI category will be updated later
   selectedDataItem_POICategory.setData("");
   selectedDataItems.push_back(selectedDataItem_POICategory);
   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, addressListElements);
}


void HMIModelNavigationSDSService::updatePOICountryHandover()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updatePOICountryHandover()"));
   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_naviSdsHandOverSreenId);
   ::std::vector< HapticData > dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > addressListElements; // not req
   const navmiddleware::LocationInfos& locationInfos = _navMiddleware.getAddressElements();
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = convertNaviSdsHandOverLocationType(locationInfos.getCurrentSearchScope().m_type);
   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, addressListElements);
}


void HMIModelNavigationSDSService::updateSetDestStartRGHandover()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updateSetDestStartRGHandover(_currentRGFlow %d)", _currentRGFlow));
   switch (_currentRGFlow)
   {
      case SDS_HANDOVER_NAVI_SETDEST_ADDRESS:
         updateSetDestAddressHandover();
         break;
      case SDS_HANDOVER_NAVI_SETDEST_CITYCENTER:
         updateSetDestCityCenterHandover();
         break;
      case SDS_HANDOVER_NAVI_SETDEST_INTERSECTION:
         updateSetDestIntersectionHandover();
         break;
      case SDS_HANDOVER_NAVI_SETDEST_POI:
         updateSetDestPOIHandover();
         break;
      default:
         break;
   }
}


void HMIModelNavigationSDSService::updateSetDestAddressHandover()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updateSetDestAddressHandover()"));
   const navmiddleware::AddressInputInfo& addresInputInfo = _navMiddleware.getFormBasedAddressInputInfo();
   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_currentRGFlow);
   ::std::vector< HapticData > dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > addressListElements; // not req
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = LocationType__DEFAULT;
   if (_infoStore.getRegionType() == USA || _infoStore.getRegionType() == THI || _infoStore.getRegionType() == CAN || _infoStore.getRegionType() == MEX)
   {
      // Update selected items Country, State, City, HouseNo and Street
      selectedDataItems.reserve(5);
      updateSelectedDataItem(true, true, true, true, true, false, selectedDataItems);
   }
   else
   {
      // Update selected items Country, City, HouseNo and Street
      selectedDataItems.reserve(4); // need info of State, Country, City, House Number and Street
      updateSelectedDataItem(true, false, true, true, true, false, selectedDataItems);
   }

   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, addressListElements, _getBaseAddressStartRouteAct);
   _getBaseAddressStartRouteAct = 0;
}


void HMIModelNavigationSDSService::updateSetDestCityCenterHandover()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updateSetDestCityCenterHandover()"));
   const navmiddleware::AddressInputInfo& addresInputInfo = _navMiddleware.getFormBasedAddressInputInfo();
   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_currentRGFlow);
   ::std::vector< HapticData> dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > addressListElements; // not req
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = LocationType__DEFAULT;
   // Update selected items Country, State and City
   selectedDataItems.reserve(3);
   updateSelectedDataItem(true, true, true, false, false, false, selectedDataItems);
   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, addressListElements, _getBaseAddressStartRouteAct);
   _getBaseAddressStartRouteAct = 0;
}


void HMIModelNavigationSDSService::updateSetDestIntersectionHandover()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updateSetDestIntersectionHandover()"));
   const navmiddleware::AddressInputInfo& addresInputInfo = _navMiddleware.getFormBasedAddressInputInfo();
   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_currentRGFlow);
   ::std::vector< HapticData > dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > addressListElements; // not req
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = LocationType__DEFAULT;
   // Update selected items Country, State and Street and Cross-street
   updateSelectedDataItem(true, true, true, false, true, true, selectedDataItems);
   if (addresInputInfo.getCurrentCity().empty())
   {
      // Update selected items City
      updateSelectedDataItem(true, true, true, false, true, true, selectedDataItems);
   }
   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, addressListElements, _getBaseAddressStartRouteAct);
   _getBaseAddressStartRouteAct = 0;
}


void HMIModelNavigationSDSService::updateSetDestPOIHandover()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::updateSetDestPOIHandover()"));
   const navmiddleware::AddressInputInfo& addresInputInfo = _navMiddleware.getFormBasedAddressInputInfo();
   SDSHandOverScreenId sdsHandOverScreenId = convertNaviSdsHandOverScreenId(_currentRGFlow);
   ::std::vector< HapticData > dataList;
   ::std::vector< SelectedDataItem > selectedDataItems;
   ::std::vector< SDSHandOverOptions > sdsHandOverOptions;// not required
   ::std::vector< AddressListElement > addressListElements; // not req
   org::bosch::cm::navigation::NavigationSDSService::LocationType locationType = LocationType__DEFAULT;

   SelectedDataItem selectedDataItem_POICategory;
   selectedDataItem_POICategory.setHmiSelectedAddressElementType(HMISelectedAddressElementType__POI_CATEGORY);
   // Selected POI category will be updated later
   selectedDataItem_POICategory.setData("");
   selectedDataItems.reserve(3);
   selectedDataItems.push_back(selectedDataItem_POICategory);
   updateSelectedDataItem(true, true, false, false, false, false, selectedDataItems);
   sendGetHapticHandOverDataResponse(sdsHandOverScreenId, dataList, selectedDataItems, locationType, sdsHandOverOptions, addressListElements, _getBaseAddressStartRouteAct);
   _getBaseAddressStartRouteAct = 0;
}


void HMIModelNavigationSDSService::onPropertyUpdateAddressInputInfoChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onPropertyUpdateAddressInputInfoChanged()"));
   if (_getBaseAddressPOIAct != 0)
   {
      updatePOIListHandOver();
      _navMiddleware.leaveFormBasedAddressInput();
   }
   else if (_getBaseAddressFTSAct != 0)
   {
      updateFreeTextSearchAddressHandOver();
      _navMiddleware.leaveFormBasedAddressInput();
   }
   else if (_getBaseAddressStartRouteAct != 0)
   {
      updateSetDestStartRGHandover();
      _navMiddleware.leaveFormBasedAddressInput();
   }
}


void HMIModelNavigationSDSService::onPropertyUpdatePositionInfoChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onPropertyUpdatePositionInfoChanged()"));
   const navmiddleware::PositionInfo& positionInfo = _navMiddleware.getPositionInfo();
   org::bosch::cm::navigation::NavigationSDSService::PositionInfo& positionInfoService = getPositionInfoMutable();
   positionInfoService.setCountry(positionInfo.getCurrentCountry());
   positionInfoService.setProvince(positionInfo.getCurrentState());
   positionInfoService.setCity(positionInfo.getCurrentCity());
   positionInfoService.setCityDistrict(positionInfo.getCurrentCityDistrict());
   positionInfoService.setStreet(positionInfo.getCurrentStreet());
   positionInfoService.setStreetNumber(positionInfo.getCurrentStreetNumber());
   /*RTC-1202932, RTC-1202491*/
   positionInfoService.setIsUniqueIdValid(positionInfo.getUniqueID().isValid());
   positionInfoService.setUniqueId(positionInfo.getUniqueID().getValue());

   const navmiddleware::SdsPositionInfo& sdsPositionInfo = _navMiddleware.getSdsPositionInfo();
   positionInfoService.getCurrentCountryMutable().setObjectName(sdsPositionInfo.getCurrentCountry().m_objectName);
   positionInfoService.getCurrentCountryMutable().setNdsNamedObjectId(sdsPositionInfo.getCurrentCountry().m_namedObjectId);
   positionInfoService.getCurrentCountryMutable().setNdsUpdateRegionSpecifier(sdsPositionInfo.getCurrentCountry().m_updateRegionSpecifier);

   positionInfoService.getCurrentStateMutable().setObjectName(sdsPositionInfo.getCurrentState().m_objectName);
   positionInfoService.getCurrentStateMutable().setNdsNamedObjectId(sdsPositionInfo.getCurrentState().m_namedObjectId);
   positionInfoService.getCurrentStateMutable().setNdsUpdateRegionSpecifier(sdsPositionInfo.getCurrentState().m_updateRegionSpecifier);

   positionInfoService.getCurrentCityMutable().setObjectName(sdsPositionInfo.getCurrentCity().m_objectName);
   positionInfoService.getCurrentCityMutable().setNdsNamedObjectId(sdsPositionInfo.getCurrentCity().m_namedObjectId);
   positionInfoService.getCurrentCityMutable().setNdsUpdateRegionSpecifier(sdsPositionInfo.getCurrentCity().m_updateRegionSpecifier);

   positionInfoService.getCurrentCityDistrictMutable().setObjectName(sdsPositionInfo.getCurrentCityDistrict().m_objectName);
   positionInfoService.getCurrentCityDistrictMutable().setNdsNamedObjectId(sdsPositionInfo.getCurrentCityDistrict().m_namedObjectId);
   positionInfoService.getCurrentCityDistrictMutable().setNdsUpdateRegionSpecifier(sdsPositionInfo.getCurrentCityDistrict().m_updateRegionSpecifier);

   positionInfoService.getCurrentStreetMutable().setObjectName(sdsPositionInfo.getCurrentStreet().m_objectName);
   positionInfoService.getCurrentStreetMutable().setNdsNamedObjectId(sdsPositionInfo.getCurrentStreet().m_namedObjectId);
   positionInfoService.getCurrentStreetMutable().setNdsUpdateRegionSpecifier(sdsPositionInfo.getCurrentStreet().m_updateRegionSpecifier);

   positionInfoService.getCurrentIntersectionMutable().setObjectName(sdsPositionInfo.getCurrentIntersection().m_objectName);
   positionInfoService.getCurrentIntersectionMutable().setNdsNamedObjectId(sdsPositionInfo.getCurrentIntersection().m_namedObjectId);
   positionInfoService.getCurrentIntersectionMutable().setNdsUpdateRegionSpecifier(sdsPositionInfo.getCurrentIntersection().m_updateRegionSpecifier);

   positionInfoService.getCurrentZipCodeMutable().setObjectName(sdsPositionInfo.getCurrentZipCode().m_objectName);
   positionInfoService.getCurrentZipCodeMutable().setNdsNamedObjectId(sdsPositionInfo.getCurrentZipCode().m_namedObjectId);
   positionInfoService.getCurrentZipCodeMutable().setNdsUpdateRegionSpecifier(sdsPositionInfo.getCurrentZipCode().m_updateRegionSpecifier);

   sendPositionInfoUpdate();
}


void HMIModelNavigationSDSService::onPropertyPositionStatusChanged()
{
   const navmiddleware::PositionStatusInfo& posStatusInfo = _navMiddleware.getPositionStatusInfo();
   org::bosch::cm::navigation::NavigationSDSService::PositionStatusInfo& positionStatusInfoService = getPositionStatusInfoMutable();
   positionStatusInfoService.setVehicleSpeed(posStatusInfo.getSpeedInfo());
   positionStatusInfoService.setVehicleHeading(posStatusInfo.getHeadingInfo());
   PositionWGS84& positionWGS84 = positionStatusInfoService.getPositionWGS84Mutable();
   positionWGS84.setLatitude(posStatusInfo.getLatitude());
   positionWGS84.setLongitude(posStatusInfo.getLongitude());
   /*RTC-1202932, RTC-1202491*/
   positionStatusInfoService.setIsUniqueIdValid(posStatusInfo.getUniqueID().isValid());
   positionStatusInfoService.setUniqueId(posStatusInfo.getUniqueID().getValue());

   positionStatusInfoService.setCountryCode(posStatusInfo.getCountryCode());
   positionStatusInfoService.setStateCode(posStatusInfo.getStateCode());
   positionStatusInfoService.setRoadClass(convertRoadClass(posStatusInfo.getRoadClass()));
   sendPositionStatusInfoUpdate();
}


void HMIModelNavigationSDSService::onSdsCheckAddressRequest(const ::boost::shared_ptr< SdsCheckAddressRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onSdsCheckAddressRequest()"));
   _sdsCheckAddressRequestAct = request->getAct();
   ::std::vector< ::org::bosch::cm::navigation::NavigationSDSService::SDSAddressElement > requestedAddress = request->getRequestedAddress();
   ::std::vector< ::org::bosch::cm::navigation::NavigationSDSService::SDSAddressElement >::const_iterator iter = requestedAddress.begin();

   navmiddleware::SDSAddress sdsAddress;
   ::std::vector< navmiddleware::SDSAddressElement >& addressElements = sdsAddress.getAddressElementsMutable();

   // copy request data for SDS address from DBUS structures to navmiddleware structures
   for (; iter != requestedAddress.end(); ++iter)
   {
      navmiddleware::SDSAddressElement addressElement;
      addressElement.setAddressElementType(convertSDSAddressElementType(iter->getAddressElementType()));
      ETG_TRACE_USR4(("addressElementType %d", addressElement.getAddressElementType()));
      addressElement.setData(iter->getData());

      navmiddleware::SDSElementOptions sdsElementOption;
      sdsElementOption.setIsAmbiguous(iter->getElementOptions().getIsAmbigious()); // @Todo: Fix spelling error "getIsAmbigious" in navigation DBUS service

      addressElement.setAddressElementOptions(sdsElementOption);

      addressElements.push_back(addressElement);
   }

   _navMiddleware.sdsCheckAddress(sdsAddress);
}


void HMIModelNavigationSDSService::onSdsGetRefinementListRequest(const ::boost::shared_ptr< SdsGetRefinementListRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onSdsGetRefinementListRequest()"));
   _getSdsGetRefinementListAct = request->getAct();
   _navMiddleware.requestSdsRefinementList();
}


bool HMIModelNavigationSDSService::onPropertyUpdateSdsElementsChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onpropertyUpdateSdsElementsChanged()"));
   if (_getSdsGetRefinementListAct != 0)
   {
      const navmiddleware::RefinementWithIdInfos& refinementWithIdInfos = _navMiddleware.getSdsRefinementListInfo();

      ::std::vector< org::bosch::cm::navigation::NavigationSDSService::RefinementListElement > refinementListElements;
      std::vector<navmiddleware::RefinementWithIdInfos::RefinementWithId>::const_iterator info = refinementWithIdInfos.getElements().begin();
      for (; (info != refinementWithIdInfos.getElements().end()); ++info)
      {
         RefinementListElement refinementListElement;
         refinementListElement.setData(info->m_data);
         refinementListElement.setId(info->m_id);
         if (info->getDirection().isValid())
         {
            refinementListElement.setDirection(convertNaviSDSDirectionDescription(info->getDirection().getValue()));
         }
         else
         {
            refinementListElement.setDirection(DirectionDescription__INVALID);
         }
         refinementListElement.setDistance(info->getAirDistance());
         refinementListElements.push_back(refinementListElement);
      }
      sendSdsGetRefinementListResponse(refinementListElements, _getSdsGetRefinementListAct);
      _getSdsGetRefinementListAct = 0;
   }
   return true;
}


void HMIModelNavigationSDSService::onSelectSdsRefinementRequest(const ::boost::shared_ptr< SelectSdsRefinementRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onSelectSdsRefinementRequest()"));
   _selectSdsRefinementAct = request->getAct();
   _navMiddleware.selectSdsRefinement(request->getId());
}


void HMIModelNavigationSDSService::onSetLocationWithSdsInputRequest(const ::boost::shared_ptr< SetLocationWithSdsInputRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onSetLocationWithSdsInputRequest()"));
   _navMiddleware.setLocationWithSdsInput();
   sendSetLocationWithSdsInputResponse();
}


bool HMIModelNavigationSDSService::onPropertySdsAddressWithOptionsChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onPropertySdsAddressWithOptionsChanged()"));
   ::org::bosch::cm::navigation::NavigationSDSService::SDSAddressWithOptions sdsAddressWithOptions;

   const navmiddleware::SDSAddressWithOptions& addressWithOptions = _navMiddleware.getCurrentSDSAddressWithOptions();
   const ::std::vector< navmiddleware::SDSAddressElement >& address = addressWithOptions.getAddress().getAddressElements();
   std::vector< navmiddleware::SDSAddressElement >::const_iterator info = address.begin();

   ::std::vector< ::org::bosch::cm::navigation::NavigationSDSService::SDSAddressElement > addressElements;
   for (; (info != address.end()); ++info)
   {
      ::org::bosch::cm::navigation::NavigationSDSService::SDSAddressElement addressElement;
      ::org::bosch::cm::navigation::NavigationSDSService::SDSAddressElementType  addressElementType;

      addressElementType = convertSDSAddressElementType(info->getAddressElementType());
      ETG_TRACE_USR4(("addressElementType %d", addressElementType));
      addressElement.setAddressElementType(addressElementType);

      addressElement.setData(info->getData());

      ::org::bosch::cm::navigation::NavigationSDSService::SDSElementOptions sdsElementOptions;
      sdsElementOptions.setIsAmbigious(info->getAddressElementOptions().isAmbiguous()); // @Todo: Fix spelling error "getIsAmbigious" in navigation DBUS service
      addressElement.setElementOptions(sdsElementOptions);

      addressElements.push_back(addressElement);
   }

   sdsAddressWithOptions.setAddress(addressElements);

   ::org::bosch::cm::navigation::NavigationSDSService::SDSAddressOptions addressOption;
   addressOption.setNavigable(addressWithOptions.getAddressOptions().isNavigable());
   addressOption.setAmbigious(addressWithOptions.getAddressOptions().isAmbiguous()); // @Todo: Fix spelling error "getIsAmbigious" in navigation DBUS service
   addressOption.setRemovedInvalidEntry(addressWithOptions.getAddressOptions().hasRemovedInvalidEntry());
   addressOption.setMoreEntriesPossible(addressWithOptions.getAddressOptions().areMoreEntriesPossible()); // @Todo: Fix spelling error "setMoreEntrysPossible" in navigation DBUS service
   addressOption.setHouseNumberAvailable(addressWithOptions.getAddressOptions().isHousenumberAvailable());
   addressOption.setHouseNumberValid(addressWithOptions.getAddressOptions().isHousenumberValid());

   sdsAddressWithOptions.setAddressOptions(addressOption);

   const ::std::vector< navmiddleware::SDSAddressElementType >& addressElementTypes = addressWithOptions.getAddressElementTypes();
   std::vector< navmiddleware::SDSAddressElementType >::const_iterator iter = addressElementTypes.begin();

   ::std::vector< ::org::bosch::cm::navigation::NavigationSDSService::SDSAddressElementType > serviceAddressElementTypes;
   for (; (iter != addressElementTypes.end()); ++iter)
   {
      ::org::bosch::cm::navigation::NavigationSDSService::SDSAddressElementType serviceAddressElement;
      serviceAddressElement = convertSDSAddressElementType(*iter);
      ETG_TRACE_USR4(("serviceAddressElement %d", *iter));
      serviceAddressElementTypes.push_back(serviceAddressElement);
   }

   sdsAddressWithOptions.setAddressElementType(serviceAddressElementTypes);

   setSdsAddressWithOptions(sdsAddressWithOptions);
   if (_sdsCheckAddressRequestAct != 0)
   {
      ETG_TRACE_USR4(("send SdsCheckAddressResponse"));
      sendSdsCheckAddressResponse(_sdsCheckAddressRequestAct);
      _sdsCheckAddressRequestAct = 0;
   }
   else if (_selectSdsRefinementAct != 0)
   {
      ETG_TRACE_USR4(("send SelectSdsRefinementResponse"));
      sendSelectSdsRefinementResponse(_selectSdsRefinementAct);
      _selectSdsRefinementAct = 0;
   }
   return true;
}


void HMIModelNavigationSDSService::onSdsGetHouseNumberPatternRequest(const ::boost::shared_ptr< SdsGetHouseNumberPatternRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onSdsGetHouseNumberPatternRequest()"));
   _getHouseNumberPatternAct = request->getAct();
   _navMiddleware.requestSdsHouseNumberPatterns();
}


bool HMIModelNavigationSDSService::onPropertyUpdateSdsHouseNumberPatternsChanged()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onPropertyUpdateSdsHouseNumberPatternsChanged()"));
   if (_getHouseNumberPatternAct != 0)
   {
      const navmiddleware::SdsHouseNumberPatternsInfos& sdsHouseNumberPatternsInfos = _navMiddleware.getSdsHouseNumberPatternsInfo();
      ::std::vector< ::std::string> houseNumberPatterns;
      for (::std::vector< ::std::string>:: const_iterator iter = sdsHouseNumberPatternsInfos.getPatterns().begin(); iter != sdsHouseNumberPatternsInfos.getPatterns().end(); ++iter)
      {
         houseNumberPatterns.push_back(*iter);
      }
      sendSdsGetHouseNumberPatternResponse(sdsHouseNumberPatternsInfos.getMinimum(), sdsHouseNumberPatternsInfos.getMaximum(), houseNumberPatterns, _getHouseNumberPatternAct);
      _getHouseNumberPatternAct = 0;
   }
   return true;
}


void HMIModelNavigationSDSService::onSdsUpdateVoiceTagRequest(const ::boost::shared_ptr< SdsUpdateVoiceTagRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onSdsUpdateVoiceTagRequest()"));
   if (request->getVoiceTagID().getOptions() == SDSVoiceTagOptions__NEW_VOICETAG_CREATED)
   {
      _infoStore.setDestinationAttributesInfoVoiceTagId(static_cast<uint64_t>(request->getVoiceTagID().getSdsVoiceTagId())); //@todo: Adapt datatype in NavigationSDSService.fidl
      _navMiddleware.modifyEntryAttributes(request->getVoiceTagID().getDestinationMemoryEntryId(), _infoStore.getDestinationAttributesInfo());
   }
   else if (request->getVoiceTagID().getOptions() == SDSVoiceTagOptions__VOICETAG_DELETED)
   {
      _infoStore.setDestinationAttributesInfoVoiceTagId(static_cast<uint64_t>(request->getVoiceTagID().getSdsVoiceTagId())); //@todo: Adapt datatype in NavigationSDSService.fidl
      _navMiddleware.modifyEntryAttributes(request->getVoiceTagID().getDestinationMemoryEntryId(), _infoStore.getDestinationAttributesInfo());
   }
   else
   {
      ETG_TRACE_USR4(("Voice tag creation failed"));
   }
   sendSdsUpdateVoiceTagResponse();
}


void HMIModelNavigationSDSService::onSdsShowReachableAreaMapRequest(const ::boost::shared_ptr< SdsShowReachableAreaMapRequest >& /*request*/)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onSdsShowReachableAreaMapRequest"));
   //bool isReachableAreaAvailable = true; // Interface for checking Reachable area is not available
   //intput sourceContextId = NAVIGATION_CONTEXT_MAP_MAIN to goback to Main map from Reachable Area map
   /*if (_infoStore.isPivi())
   {
      POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchOutReqMsg)(NAVIGATION_CONTEXT_ACTIVE_MAP, NAVIGATION_CONTEXT_PIVI_MAP_UPPER_DISPLAY, APPID_APPHMI_NAVIGATION)));
   }
   else
   {
      POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchOutReqMsg)(NAVIGATION_CONTEXT_ACTIVE_MAP, NAVIGATION_CONTEXT_MAP_MAIN, APPID_APPHMI_NAVIGATION)));
   }*/
   sendSdsShowReachableAreaMapResponse();
}


void HMIModelNavigationSDSService::onRequestAddressFormatForAvailableCountriesRequest(const ::boost::shared_ptr< RequestAddressFormatForAvailableCountriesRequest >& request)
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onRequestAddressFormatForAvailableCountriesRequest"));
   _getAddressFormatForAvailableCountriesAct = request->getAct();
   _navMiddleware.requestAddressFormatForAvailableCountries();
}


void HMIModelNavigationSDSService::onPropertyUpdateAddressFormatForAvailableCountries()
{
   ETG_TRACE_USR4(("HMIModelNavigationSDSService::onPropertyUpdateAddressFormatForAvailableCountries"));
   if (_getAddressFormatForAvailableCountriesAct != 0)
   {
      const ::std::vector<AddressFormatInfos::AddressFormatInfo>& addressFormatInfos = _navMiddleware.getAddressFormatInfos().getAddressFormatInfos();
      std::vector<AddressFormatInfo> addressFormatInfosConstructed;
      addressFormatInfosConstructed.reserve(addressFormatInfos.size());
      ETG_TRACE_USR4(("HMIModelNavigationSDSService::AddressFormat size =%d", addressFormatInfos.size()));
      std::vector<AddressFormatInfos::AddressFormatInfo> ::const_iterator iterAddressFormat = addressFormatInfos.begin();
      for (; iterAddressFormat != addressFormatInfos.end(); ++iterAddressFormat)
      {
         AddressFormatInfo addressFormat;
         addressFormat.setIsoCountryCode(iterAddressFormat->m_isoCountryCode);
         addressFormat.setAddressFormat(iterAddressFormat->m_addressFormat);
         addressFormatInfosConstructed.push_back(addressFormat);
      }
      sendRequestAddressFormatForAvailableCountriesResponse(addressFormatInfosConstructed, _getAddressFormatForAvailableCountriesAct);
      _getAddressFormatForAvailableCountriesAct = 0;
   }
}


#endif // HALL_NAVIGATION_SDS_SERVICE
#endif // HALL_TO_MDW_COM
